#-------------------------------------------------------------------------------
#  a2w/core/dm/MiningEngine.pm
#
#  Perl module to process database and to generate reports
#
#  Author   : Panneer, AFP2web Team
#
#  V100   2014-02-14    Initial Release
#
#  V101   2015-05-20    Fixed minor bug in positioning starting anchor objects
#                       when block span over pages  (AFP-224)
#
#  V102   2015-07-20    Fixed minor bug in evaluating output file name when page
#                       output is turned on
#
#  V103   2015-10-15    Extended to pass in custom block info like HTML style (AFP-298)
#
#  V104   2015-11-17    Extended to sort line objects based on X position along with sequence id  (AFP-320)
#
#  V105   2017-12-14    Extended to support json formatted (as occur in AFP TLE value) document block value (AFP-624)
#
#  V106   2018-01-19    Extended to process container objects
#
#  V107   2018-01-24    a. Extended to update block attributes (AFP-456)
#                       b. Extended to force blocks as filled to have anchors less block definition (AFP-456)
#                       c. Extended to mark page's first and last objects for page begin/end marking (AFP-456)
#
#  V108   2018-09-11    - OXS-8490: Extended with parser anchors to preprocess objects and to invoke callback
#                         when anchor matching object is found
#
#  V109   2018-10-03    a. AFP-743: Extended with composite block processing
#                       b. Improved logging object position
#                       c. Extended to handle chained blocks that has been split by other blocks or by page
#
#  V110   2018-10-26    a. AFP-756: Preprocessed page content to have lines with Y tolerance
#                       b. AFP-756: Extended with data mining APIs to find and fetch eyecatcher/value
#
#  V111   2019-01-28    a. OXS-8853: Extended to get list of page written blocks
#
#  V112   2020-03-04    a. AFP-929: Fixed minor bug when processing POM of empty page
#                       b. AFP-929: Extended to set/get visitor instance
#                       c. AFP-929: Extended to get/set output language
#
#  V113   2020-09-29    AFP-974: Extended to handle Annotation object
#
#  V114   2020-12-11    OXS-11587: Extended to get page blocks (using getPageBlocks() API)
#
#  V115   2021-03-18    AFP-1028: Extended with output configuration to control HTML generation
#
#  V116   2021-06-01    AFP-1046: Extended with output config to have block specific config, content to be embedded etc in output
#                       AFP-1046: Extended to sort page blocks (as needed by output visitor)
#
#  V117   2021-07-27    OXS-12591: Extended to specify properties for SF generated outputs
#
#  V118   2021-09-21    a. AFP-1068: Extended with FindAndPrepend and FindAndAppend content processing objects
#                       b. AFP-1068: Extended with default parser rule handler function (this default is used
#                          only when parser action is defined in block definition)
#
#-------------------------------------------------------------------------------
package a2w::core::dm::MiningEngine;

#-----------------------------------------------------------------------
# Include required modules
#-----------------------------------------------------------------------
use a2w::TypeConstants;
use a2w::core::log::Logger;

use a2w::core::dm::Constants;
use a2w::core::dm::Constraint;
use a2w::core::dm::Block;

use a2w::core::dm::Anchor; # V108 Change
use a2w::core::dm::Database;
use a2w::core::dm::ContentParser;

use a2w::core::dm::ConfigParser;
use a2w::core::dm::Dumper;

use a2w::core::dm::MiningUtils; # V110 Change

#-----------------------------------------------------------------------
# Constructor
#-----------------------------------------------------------------------
sub new{
    my $proto = shift;
    my $class = ref( $proto ) || $proto;

    #---- Define boolean values
    $TRUE  = $a2w::TypeConstants::TRUE;     # TRUE  boolean value # V118 Change
    $FALSE = $a2w::TypeConstants::FALSE;    # FALSE boolean value # V118 Change

    my $this = {
          'Config'               => undef   # Data Mining Framework Configuration (blocks definition)
        , 'ContentParser'        => undef   # Content parser
        , 'SpoolFilename'        => undef   # Spool filename
        , 'OutputPath'           => undef   # Output path
        , 'OutputRes'            => undef   # Output resolution
        , 'DOM'                  => undef   # Hash of POMs (where key is Page ID and value is POM)
        , 'DocBlockIDs'          => undef   # List of current document processable block ids
        , 'DocBlocks'            => undef   # List of current document processable blocks
        , 'DocBlocksTable'       => undef   # List of current document processable blocks # V107 Change
        , 'PageBlocks'           => undef   # Page contained (both not configured and configured) blocks
        , 'PreDefAC'             => undef   # Header/Body/Footer additional contents of predefined blocks
        , 'Visitor'              => undef   # Visitor to render content
        , 'Title'                => undef   # Title of document
        , 'a2wDocument'          => undef   # AFP2web document reference
        , 'DocumentCount'        => 0       # Document count
        , 'PageCount'            => 0       # Document page count
        , 'ParserRules'          => []      # Array of anchors which is used to preprocess page objects and callback is invoked when any anchor matching object is found # V108 Change
        , 'ParserRulesIndex'     => -1      # Index to add anchors to above array # V108 Change
        , 'ParserActions'        => undef   # Parser actions # V118b Change
        , 'PendingBlocks'        => undef   # Blocks started on earlier pages but where pending as they were not filled yet (ex: head block of chained blocks) # V109 Change
        , 'PageWrittenBlocks'    => undef   # Blocks written on current page # V111 Change
        , 'OutputLanguage'       => undef   # Output language # V112c Change
        , 'OutputConfig'         => undef   # Output configuration # V115 Change
        , 'Initialized'          => $FALSE  # Flag indicating this object is initialized or not # V118 Change
    };

    bless( $this, $class );

    #---- Get logger
    our $theLogger = a2w::core::log::Logger->getSingleton();
    our $bLog = $theLogger->isRegistered( __PACKAGE__ );

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "new()" );
    #}

    #---- Initialize
    $this->initialize();

    return $this;
}

#-----------------------------------------------------------------------
# Destructor
#-----------------------------------------------------------------------
sub DESTROY{
    my $this = shift;

    #---- Finalize
    $this->finalize();
}

#-----------------------------------------------------------------------
# Mutators
#-----------------------------------------------------------------------
sub setSpoolFilename{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setSpoolFilename" );
    #}

    $this->{ 'SpoolFilename' } = shift;
}

sub setOutputPath{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setOutputPath()" );
    #}

    $this->{ 'OutputPath' } = shift;
}

sub setOutputRes{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setOutputRes" );
    #}

    $this->{ 'OutputRes' } = shift;
}

# V111 Begin
sub setPageWrittenBlocks{
    my $this = shift;

    #---- Get parameter
    # 1. Page id
    # 2. Page blocks (array reference)
    #
    my $iPageIdPar = shift;
    my $arefBlocksPar = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "setPageWrittenBlocks( $iPageIdPar, $arefBlocksPar )" ); }

    $this->{ 'PageWrittenBlocks' }{ $iPageIdPar } = $arefBlocksPar;
}
# V111 End

# V112b Begin
sub setVisitor{
    my $this = shift;

    #---- Get parameter
    # 1. Visitor (instance)
    #
    my $objVisitorPar = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "setVisitor( $objVisitorPar )" ); }

    $this->{ 'Visitor' } = $objVisitorPar;
}
# V112b End

# V112c Begin
sub setOutputLanguage{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setOutputLanguage" );
    #}

    $this->{ 'OutputLanguage' } = shift;
}
# V112c End

# V115 Begin
sub setOutputConfig{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setOutputConfig" );
    #}

    $this->{ 'OutputConfig' } = shift;
}
# V115 End

# V118b Begin
sub setParserActions{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setParserActions" );
    #}

    $this->{ 'ParserActions' } = shift;
}
# V118b End

#-----------------------------------------------------------------------
# Accessors
#-----------------------------------------------------------------------
sub getSpoolFilename{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getSpoolFilename" );
    #}

    return $this->{ 'SpoolFilename' };
}

sub getOutputPath{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getOutputPath" );
    #}

    return $this->{ 'OutputPath' };
}

sub getOutputRes{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getOutputRes" );
    #}

    return $this->{ 'OutputRes' };
}

# V111 Begin
sub getPageWrittenBlocks{
    my $this = shift;

    #---- Get parameter
    # 1. Page id
    #
    my $iPageIdPar = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "getPageWrittenBlocks( $iPageIdPar )" ); }

    return $this->{ 'PageWrittenBlocks' }{ $iPageIdPar };
}
# V111 End

# V112b Begin
sub getVisitor{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "getVisitor()" ); }

    return $this->{ 'Visitor' };
}
# V112b End

# V112c Begin
sub getOutputLanguage{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getOutputLanguage" );
    #}

    return $this->{ 'OutputLanguage' };
}
# V112c End

# V115 Begin
sub getOutputConfig{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getOutputConfig" );
    #}

    return $this->{ 'OutputConfig' };
}
# V115 End

# V118b Begin
sub getParserActions{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getParserActions" );
    #}

    return $this->{ 'ParserActions' };
}
# V118b End

#-----------------------------------------------------------------------
# Workers
#-----------------------------------------------------------------------
#-----------------------------------------------------------------------
# Initialize
#-----------------------------------------------------------------------
sub initialize{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "initialize" );
    }

    # Skip initialization, if already initialized
    if ( $this->{ 'Initialized' } == $TRUE ){ return; } # V118 Change

    #---- Initialize attributes
    $this->{ 'Config' }            = undef;
    $this->{ 'ContentParser' }     = new a2w::core::dm::ContentParser();
    $this->{ 'SpoolFilename' }     = undef;
    $this->{ 'OutputPath' }        = undef;
    $this->{ 'OutputRes' }         = undef;
    $this->{ 'DOM' }               = undef;
    $this->{ 'DocBlockIDs' }       = undef;
    $this->{ 'DocBlocks' }         = undef;
    $this->{ 'DocBlocksTable' }    = undef; # V107 Change
    $this->{ 'PageBlocks' }        = undef;
    $this->{ 'PreDefAC' }          = undef;
    $this->{ 'Visitor' }           = undef;
    $this->{ 'Title' }             = undef;
    $this->{ 'a2wDocument' }       = undef;
    $this->{ 'DocumentCount' }     = 0;     # V118 Change
    $this->{ 'PageCount' }         = 0;
    $this->{ 'ParserRules' }       = [];    # V118 Change
    $this->{ 'ParserRulesIndex' }  = -1;    # V118 Change
    $this->{ 'ParserActions' }     = undef; # V118 Change
    $this->{ 'PendingBlocks' }     = undef; # V118 Change
    $this->{ 'PageWrittenBlocks' } = undef; # V111 Change
    $this->{ 'OutputLanguage' }    = undef; # V118 Change
    $this->{ 'OutputConfig' }      = undef; # V118 Change

    # Set initialized flag
    $this->{ 'Initialized' }       = $TRUE; # V118 Change

    return 0;
}

#-----------------------------------------------------------------------
# Finalize
#-----------------------------------------------------------------------
sub finalize{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "finalize" );
    }

    #---- Cleanup
    $this->{ 'Config' }            = undef;
    $this->{ 'ContentParser' }     = undef;
    $this->{ 'SpoolFilename' }     = undef;
    $this->{ 'OutputPath' }        = undef;
    $this->{ 'OutputRes' }         = undef;
    $this->{ 'DOM' }               = undef;
    $this->{ 'DocBlockIDs' }       = undef;
    $this->{ 'DocBlocks' }         = undef;
    $this->{ 'DocBlocksTable' }    = undef;  # V107 Change
    $this->{ 'PageBlocks' }        = undef;
    $this->{ 'PreDefAC' }          = undef;
    $this->{ 'Visitor' }           = undef;
    $this->{ 'Title' }             = undef;
    $this->{ 'a2wDocument' }       = undef;
    $this->{ 'DocumentCount' }     = 0;      # V118 Change
    $this->{ 'PageCount' }         = 0;
    $this->{ 'ParserRules' }       = [];     # V118 Change
    $this->{ 'ParserRulesIndex' }  = -1;     # V118 Change
    $this->{ 'ParserActions' }     = undef;  # V118 Change
    $this->{ 'PendingBlocks' }     = undef;  # V118 Change
    $this->{ 'PageWrittenBlocks' } = undef;  # V111 Change
    $this->{ 'OutputLanguage' }    = undef;  # V118 Change
    $this->{ 'OutputConfig' }      = undef;  # V118 Change

    $this->{ 'Initialized' }       = $FALSE; # V118 Change

    return 0;
}

#-----------------------------------------------------------------------
# Initialize document
#
# Parameter
# 1. Visitor module name (concrete visitor module name)
# 2. Document instance (of type a2w::Document)
# 3. Title
# 4. Page output, TRUE means each page as one output, FALSE means each document as one output (default is FALSE)
#
# Returns
# >=0 in case of success
# < 0 in case of error
# 
#-----------------------------------------------------------------------
sub initializeDoc{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "initializeDoc" );
    }

    #---- Get parameter
    #
    # 1. Visitor module name
    # 2. Document instance (of type a2w::Document)
    # 3. Title
    # 4. Page output
    #
    my $sVisitorModNamePar = shift;
    my $a2wDocumentPar = shift;
    my $sTitlePar = shift;
    my $bPageOutputPar = $FALSE;
    if ( @_ > 0 ){
        $bPageOutputPar = shift;
    }

    #---- Set process factors
    $this->{ 'Title' }          = $sTitlePar;
    $this->{ 'a2wDocument' }    = $a2wDocumentPar;
    $this->{ 'PageCount' }      = 0;
    $this->{ 'DocumentCount' } += 1;

    #---- Create visitor
    $this->{ 'Visitor' } = $sVisitorModNamePar->new();

    #---- Fill in visitor with processing info
    $this->{ 'Visitor' }->setUnitBase( "pixel" );
    $this->{ 'Visitor' }->setOutputRes( $this->{ 'OutputRes' } );
    $this->{ 'Visitor' }->setDocumentId( $this->{ 'DocumentCount' } );
    if ( $bPageOutputPar == $TRUE ){
        $this->{ 'Visitor' }->setPageOutput( $TRUE );
    }
    $this->{ 'Visitor' }->setSpoolFilename( $this->getSpoolFilename() ); # V108 Change
    $this->{ 'Visitor' }->setOutputLanguage( $this->getOutputLanguage() ); # V112c Change

    # V115 Begin
    $this->{ 'Visitor' }->setOutputConfig( $this->getOutputConfig() );
    $this->{ 'Visitor' }->updateConfig();
    # V115 End

    #---- Evaluate output simple file name
    my $sOutputFilenameTmp = $a2wDocumentPar->getOutputFilename();
    my $sSimpleNameTmp = "";
    if ( $sOutputFilenameTmp =~ /(.*)\..{3,4}$/ ){
        $sSimpleNameTmp = $1;
    }

    #---- Initialize visitor
    $this->{ 'Visitor' }->initialize( # Output path
                                        $this->{ 'OutputPath' }
                                      # Simple name
                                      , $sSimpleNameTmp
                                    );

    return 0;
}

#-----------------------------------------------------------------------
# Finalize document
#-----------------------------------------------------------------------
sub finalizeDoc{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "finalizeDoc" );
    }

    # Get A2W_DMF_PAGE_BODY predefined block's additional contents included
    # at after position
    #
    my $sAfterBodyAddContTmp = $this->{ 'PreDefAC' }{ 'BdyAfter' };

    # Get A2W_DMF_PAGE_FOOTER predefined block's additional contents
    my $sFtContTmp = $this->{ 'PreDefAC' }{ 'Footer' };

    #---- Finalize visitor
    $this->{ 'Visitor' }->finalize(   $sAfterBodyAddContTmp
                                    , $sFtContTmp
                                  );

    #---- Cleanup ----#
    my @arrBlkListTmp  = @{ $this->{ 'DocBlocks' } };
    foreach my $blkTmp ( @arrBlkListTmp ){
        if ( $blkTmp == undef ){
            next;
        }
        $blkTmp->reset();
    }
    $this->{ 'DOM' }               = undef;
    $this->{ 'DocBlockIDs' }       = undef;
    $this->{ 'DocBlocks' }         = undef;
    $this->{ 'DocBlocksTable' }    = undef; # V107 Change
    $this->{ 'PageBlocks' }        = undef;
    $this->{ 'Visitor' }           = undef;
    $this->{ 'PageCount' }         = 0;
    $this->{ 'PageWrittenBlocks' } = undef; # V11 Change

    return 0;
}

# V117 Begin
#-----------------------------------------------------------------------
# Set document properties
#-----------------------------------------------------------------------
sub setDocumentProperties{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "setDocumentProperties" );
    }

    #---- Get parameter
    #
    # 1. Hash reference of document properties
    #
    my $hrefPropertiesPar = shift;

    #---- Pass on the properties to output visitor
    $this->{ 'Visitor' }->setProperties( $hrefPropertiesPar );

    return 0;
}

#-----------------------------------------------------------------------
# Get document properties
#-----------------------------------------------------------------------
sub getDocumentProperties{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "getDocumentProperties" );
    }

    #---- Get properties from output visitor
    return $this->{ 'Visitor' }->getProperties();
}
# V117 End

#-----------------------------------------------------------------------
# Load config
#-----------------------------------------------------------------------
sub loadConfig{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "loadConfig" );
    }

    #---- Create config parser
    my $parserConfigTmp = new a2w::core::dm::ConfigParser();

    #---- Load config from hash
    $this->{ 'Config' } = $parserConfigTmp->parseHash( @_ );

    #---- Cleanup
    $parserConfigTmp = undef;

    #---- Resolve next chain links
    my $iRetTmp = $this->{ 'Config' }->resolveNextChain();

    #---- Process predefined blocks ----#
    $this->{ 'PreDefAC' } = $this->_getAdditionalContentsOfPredefinedBlocks();

    return $iRetTmp;
}

#-----------------------------------------------------------------------
# Process page
#
# Process page objects and add them to appropriate block they belong to
#
# Parameters:
# 1. Page Instance
# 2. Process overlays for AFP page type (optional)
#
# Returns 0 in case of success, <0 in case of error
#-----------------------------------------------------------------------
sub processPage{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "processPage" );
    }

    #---- Increment page count
    $this->{ 'PageCount' }++;

    #---- Parse page and create POM
    my $dmContentParserTmp = $this->{ 'ContentParser' };
    # V108 Begin
    #---- Set parser rules
    if ( $this->{ 'PageCount' } == 1 && $this->{ 'ParserRulesIndex' } >= 0 ){ $dmContentParserTmp->setRules( $this->{ 'ParserRules' } ); }
    # V108 End
    my $dmPOMTmp = $dmContentParserTmp->parse( @_ );

    # V102 Begin
    if ( $_[ 0 ] != undef ){
        my $sPageFilenameTmp = $_[ 0 ]->getOutputFilename();
        if ( $sPageFilenameTmp ne "" ){
            $dmPOMTmp->setPageFilename( $sPageFilenameTmp );
        }
    }
    # V102 End

    #---- Store POM on DOM
    my $iPageIdTmp = $dmPOMTmp->getPageID();
    $this->{ 'DOM' }{ $iPageIdTmp } = $dmPOMTmp;

    #---- Process POM objects and add them to appropriate blocks ----#
    my $iObjCountTmp = $dmPOMTmp->getObjectCount();

    #---- Process POM ----#
    return $this->_processPOM( $dmPOMTmp );
}

#-----------------------------------------------------------------------
# Initialize page
#
# Returns 0 in case of success, <0 in case of error
#-----------------------------------------------------------------------
sub initializePage{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "initializePage" );
    }

    require a2w::Page;    # Include page module

    #---- Get parameter ( Par: Page instance )
    my $a2wPagePar = shift;

    #---- Increment page count
    $this->{ 'PageCount' }++;

    #---- Initiate page parse
    my $dmContentParserTmp = $this->{ 'ContentParser' };
    # V108 Begin
    #---- Set parser rules
    if ( $this->{ 'PageCount' } == 1 ){ $dmContentParserTmp->setRules( $this->{ 'ParserRules' } ); }
    # V108 End
    # V102 Begin
    #$dmContentParserTmp->initializePage( $a2wPagePar );
    my $dmPOMTmp = $dmContentParserTmp->initializePage( $a2wPagePar );

    if ( $a2wPagePar != undef ){
        my $sPageFilenameTmp = $a2wPagePar->getOutputFilename();
        if ( $sPageFilenameTmp ne "" ){
            $dmPOMTmp->setPageFilename( $sPageFilenameTmp );
        }
    }
    # V102 End

    return 0;
}

#-----------------------------------------------------------------------
# Add object
#
# Adds given object to active page database (POM)
#
# Returns 0 in case of success, <0 in case of error
#-----------------------------------------------------------------------
sub addObject{
    my $this = shift;

    #---- Get parameter
    #
    # 1. Object
    #
    my $a2wObjPar = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "addObject" );
    }

    #---- Add object to page
    my $dmContentParserTmp = $this->{ 'ContentParser' };
    $dmContentParserTmp->addObject( $a2wObjPar );
    return 0;
}

#-----------------------------------------------------------------------
# Finalize page
#
# Returns 0 in case of success, <0 in case of error
#-----------------------------------------------------------------------
sub finalizePage{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "finalizePage" );
    }

    #---- Finalize page parse
    my $dmContentParserTmp = $this->{ 'ContentParser' };
    my $dmPOMTmp = $dmContentParserTmp->finalizePage();

    #---- Store POM on DOM
    my $iPageIdTmp = $dmPOMTmp->getPageID();
    $this->{ 'DOM' }{ $iPageIdTmp } = $dmPOMTmp;

    #---- Process POM ----#
    return $this->_processPOM( $dmPOMTmp );
}

# V103 Begin
#-----------------------------------------------------------------------
# Add document possible block
#-----------------------------------------------------------------------
sub addDocumentBlock{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "addDocumentBlock" );
    }

    #---- Get parameter
    #
    # 1. Block id
    # 2. Block info
    #
    my $sBlockIdPar   = shift;
    # V105 Begin
    my $sBlockInfoPar = undef;
    if ( @_ > 0 ){ $sBlockInfoPar = shift; }
    #if ( $bLog == $TRUE ){ $theLogger->logMessage( "Block: Id:>" . $sBlockIdPar . "< Info:>" . $sBlockInfoPar . "<" ); }

    #---- Add block id to list
    $this->{ 'DocBlockIDs' }[ @{ $this->{ 'DocBlockIDs' } } ] = $sBlockIdPar;


    #---- Get block from configured list
    my $confDMFTmp = $this->{ 'Config' };
    my $blkAppliedTmp = $confDMFTmp->getBlock( $sBlockIdPar );

    #---- Assert block
    if ( $blkAppliedTmp == undef ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Error! Block (" . $sBlockIdPar . ") not defined in block definition" ); }
        return ( -1, "Error! Block (" . $sBlockIdPar . ") not defined in block definition" );
    }

    #---- Add block to table (id => block)
    $this->{ 'DocBlocksTable' }{ $sBlockIdPar } = $blkAppliedTmp; # V107 Change

    #---- Add block to document list
    $this->{ 'DocBlocks' }[ @{ $this->{ 'DocBlocks' } } ] = $blkAppliedTmp;
    if ( $bLog == $TRUE ){ $theLogger->logMessage( "Block (" . $sBlockIdPar . ") added, total blocks on list is " . @{ $this->{ 'DocBlocks' } } ); }

    if ( $sBlockInfoPar eq "" ){ return 0; }   # V105 Change
    $blkAppliedTmp->setInfo( $sBlockInfoPar ); # V105 Change

    return 0;
}

#-----------------------------------------------------------------------
# Add repetitive document possible block (used for repetitive blocks adding)
#-----------------------------------------------------------------------
sub addRepetitiveDocumentBlock{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "addRepetitiveDocumentBlock" );
    }

    #---- Get parameter
    #
    # 1. Block reference
    #
    my $blkDocPar = shift;

    #---- Add block id to list
    $this->{ 'DocBlockIDs' }[ @{ $this->{ 'DocBlockIDs' } } ] = $blkDocPar->getId();

    #---- Add block to table (id => block)
    $this->{ 'DocBlocksTable' }{ $blkDocPar->getId() } = $blkDocPar; # V107 Change

    #---- Add block to list
    $this->{ 'DocBlocks' }[ @{ $this->{ 'DocBlocks' } } ] = $blkDocPar;
    if ( $bLog == $TRUE ){
        $theLogger->logMessage( "Repetitive block (" . $blkDocPar->getId() . ") added, total blocks on list is " . @{ $this->{ 'DocBlocks' } } );
    }
}
# V103 End

# V107 Begin
#-----------------------------------------------------------------------
# Update document block
#
# Updates given id specific document block attributes with given attributes
#
# Parameters:
# 1. Block Id
# 2. Hash of attributes
#
# Returns none
#
#-----------------------------------------------------------------------
sub updateDocumentBlock{
    my $this = shift;
    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "updateDocumentBlock" ); }

    #---- Get parameter
    #
    # 1. Block Id
    # 2. Hash of attributes
    #
    my $blkIdPar = shift;
    my $hrefAttrsPar = shift;

    #---- Assert parameter
    if ( $blkIdPar eq "" ){ return; }
    #if ( $bLog == $TRUE ){ 
    #    $theLogger->logMessage( "Block ($blkIdPar):" );
    #    $theLogger->logHashMessage( $hrefAttrsPar );
    #}

    #---- Get block
    my $blkToUpdateTmp = $this->{ 'DocBlocksTable' }{ $blkIdPar };
    if ( $blkToUpdateTmp == undef ){ if ( $bLog == $TRUE ){ $theLogger->logMessage( "Warning! Invalid block ($blkIdPar) to update attributes" ); } return; }

    #---- Iterate through attributes and update them on block
    my @arrAttrsTmp = sort keys( %{ $hrefAttrsPar } );
    foreach my $sAttrTmp ( @arrAttrsTmp ){ $blkToUpdateTmp->setAttribute( $sAttrTmp, $hrefAttrsPar->{ $sAttrTmp } ); }
}
# V107 End

# V114 Begin
#-----------------------------------------------------------------------
# Get page blocks
#
# Returns the array (reference) of blocks that has to be written for given page id
#-----------------------------------------------------------------------
sub getPageBlocks{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "getPageBlocks" );
    }

    #---- Get parameter
    #
    # 1. Page Id
    #
    my $iPageIdPar = shift;

    #---- Get pending blocks
    my @arrPendingBlocksTmp = @{ $this->{ 'PendingBlocks' } }; # V109 Change

    #---- Get page blocks
    my $arefBlocksTmp = $this->{ 'PageBlocks' }{ $iPageIdPar };
    my @arrBlkListTmp = @{ $arefBlocksTmp };
    # V109 Begin
    # Prepend pending blocks of previous pages
    if ( @arrPendingBlocksTmp > 0 ){
        my @arrTmp = @arrPendingBlocksTmp;
        my %hshPending1Tmp = map { $_->getId(), $TRUE } @arrPendingBlocksTmp;
        my %hshPending2Tmp = map { $_->getId(), $_ } @arrPendingBlocksTmp;

        # Ensure to avoid duplicates when adding pending blocks and current page blocks
        foreach my $blk ( @arrBlkListTmp ){
            unless (    $hshPending1Tmp{ $blk->getId() } == $TRUE
                     && $hshPending2Tmp{ $blk->getId() }->getUniqueId() eq $blk->getUniqueId()
                   ){ push( @arrTmp, $blk ); }
        }
        @arrBlkListTmp = @arrTmp;
    }
    my @arrPageBlocksTmp = @arrBlkListTmp;
    # V109 End

    #---- Filter out page blocks ----#
    #---- Filter blocks that are started on current page
    #@arrBlkListTmp = grep { $_->doesStartedOn( $iPageIdPar ) == $TRUE } @arrBlkListTmp;
    #if ( $bLog == $TRUE ){
    #    my $sBlkListTmp = "";
    #    foreach my $blkTmp ( @arrBlkListTmp ){
    #        $sBlkListTmp .= " " . $blkTmp->getId();
    #    }
    #    $theLogger->logMessage( "Page $iPageIdPar blocks:$sBlkListTmp" );
    #}

    #---- Filter blocks that are filled
    @arrBlkListTmp = grep { $_->isFilled() == $TRUE } @arrBlkListTmp;
    if ( $bLog == $TRUE ){
        my $sBlkListTmp = "";
        foreach my $blkTmp ( @arrBlkListTmp ){
            $sBlkListTmp .= " " . $blkTmp->getId();
        }
        $theLogger->logMessage( "Page $iPageIdPar filled blocks:$sBlkListTmp" );
    }

    #---- Filter blocks that are chained
    my $hrefChainedIDsTmp = $this->{ 'Config' }->getChainedIDs();
    @arrBlkListTmp = grep { $hrefChainedIDsTmp->{ $_->getId() } != $TRUE } @arrBlkListTmp;
    #if ( $bLog == $TRUE ){
    #    my $sBlkListTmp = "";
    #    foreach my $blkTmp ( @arrBlkListTmp ){
    #        $sBlkListTmp .= " " . $blkTmp->getId();
    #    }
    #    $theLogger->logMessage( "Page $iPageIdPar filled blocks:$sBlkListTmp" );
    #}

    #---- Filter blocks that are not flushed already
    @arrBlkListTmp = grep { $_->isFlushed() == $FALSE } @arrBlkListTmp;
    #if ( $bLog == $TRUE ){
    #    my $sBlkListTmp = "";
    #    foreach my $blkTmp ( @arrBlkListTmp ){
    #        $sBlkListTmp .= " " . $blkTmp->getId();
    #    }
    #    $theLogger->logMessage( "Page $iPageIdPar not flushed blocks:$sBlkListTmp" );
    #}

    return \@arrBlkListTmp;
}
# V114 End

#-----------------------------------------------------------------------
# Write page
#-----------------------------------------------------------------------
sub writePage{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writePage" );
    }

    #---- Get parameter
    #
    # 1. Page Id
    #
    my $iPageIdPar = shift;

    #---- Get page blocks
    my $arefPgBlksTmp = $this->getPageBlocks( $iPageIdPar ); # V116 Change
    $arefPgBlksTmp = $this->{ 'Visitor' }->sortPageBlocks( $arefPgBlksTmp ); # V116 Change
    my @arrBlkListTmp = @{ $arefPgBlksTmp }; # V114 Change

    #---- Get block additional contents for header/body/footer
    my $hrefAddContentsTmp = $this->_getAdditionalContents( \@arrBlkListTmp );

    #---- Get page POM ----#
    my $dbPagePOMTmp = $this->{ 'DOM' }{ $iPageIdPar };

    # Get A2W_DMF_PAGE_BODY predefined block's additional contents included
    # at before/after positions
    #
    my $sBeforeBodyAddContTmp = $this->{ 'PreDefAC' }{ 'BdyBefore' };
    my $sAfterBodyAddContTmp  = $this->{ 'PreDefAC' }{ 'BdyAfter' };

    #---- Initialize page
    my $sHdrContTmp = $this->{ 'PreDefAC' }{ 'Header' };
    if ( $hrefAddContentsTmp != undef ){
        $sHdrContTmp .= $hrefAddContentsTmp->{ 'Header' };
    }
    $this->{ 'Visitor' }->initializePage(   $dbPagePOMTmp
                                          , $this->{ 'Title' }
                                          , $sHdrContTmp
                                          , $sBeforeBodyAddContTmp
                                         );

    #---- Flush contents
    foreach my $blkToWriteTmp ( @arrBlkListTmp ){
        #---- List might have duplicates due to repetitive blocks, so assert flushed or not before writing
        #if ( $blkToWriteTmp->isFlushed() == $TRUE ){
        #    next;
        #}

        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "Block (" . $blkToWriteTmp->getId() . ") writing started" );
        }

        #---- Write block
        $blkToWriteTmp->write( $this->{ 'Visitor' } );

        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "Block (" . $blkToWriteTmp->getId() . ") writing ended" );
        }
    }

    #---- Finalize page
    my $sFtContTmp = $this->{ 'PreDefAC' }{ 'Footer' };
    if ( $hrefAddContentsTmp != undef ){
        $sFtContTmp .= $hrefAddContentsTmp->{ 'Footer' };
    }
    $this->{ 'Visitor' }->finalizePage(   $sAfterBodyAddContTmp
                                        , $sFtContTmp
                                      );

    # V109 Begin
    # Filter out page blocks that were not written and retain them as pending blocks to continue writing on next page
    #
    #'PendingBlocks'
    my @arrBlocksTmp = @arrPageBlocksTmp;
    @arrBlocksTmp = grep { $_->isFilled() == $FALSE } @arrBlocksTmp; # filter blocks that are NOT filled
    @arrBlocksTmp = grep { $hrefChainedIDsTmp->{ $_->getId() } != $TRUE } @arrBlocksTmp; # filter blocks that are chained
    @arrBlocksTmp = grep { $_->isFlushed() == $FALSE } @arrBlocksTmp; # filter blocks that are not flushed already
    $this->{ 'PendingBlocks' } = \@arrBlocksTmp; # Store pending blocks to use when writing next page
    # V109 End

    #---- Set page written blocks
    $this->setPageWrittenBlocks( $iPageIdPar, \@arrBlkListTmp ); # V111 Change

    #---- Clean up
    $this->{ 'PageBlocks' }{ $iPageIdPar } = undef;
}

#-----------------------------------------------------------------------
# Dump the DOM to a file
#-----------------------------------------------------------------------
sub dumpDOM{
    my $this = shift;

    #---- Get parameters
    #
    # 1. Output path
    # 2. Flag to sort or not the contents
    #
    my $sOutputFilePathPar = shift;
    my $bSortedPar = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "dumpDOM(" . $sOutputFilePathPar . ", " . (($bSortedPar)? "TRUE" : "FALSE") . ")" );
    }

    #---- Iterate and dump all the POMs from DOM
    my $dmDumperTmp = undef;
    my $domObjTmp = $this->{ 'DOM' };
    my @arrPageIdsTmp = sort keys( %{ $domObjTmp } );
    foreach my $sPageIdTmp ( @arrPageIdsTmp ){
        #---- Create dumper
        $dmDumperTmp = new a2w::core::dm::Dumper();

        #---- Dump POM
        $dmDumperTmp->dumpDOM( $sOutputFilePathPar, $this->{ 'DOM' }{ $sPageIdTmp }, $bSortedPar );

        #---- Release the Dumper instance
        $dmDumperTmp = undef;
    }
}

#-----------------------------------------------------------------------
# Register data mining classes for logging
#-----------------------------------------------------------------------
sub registerClassesForLog{
    my $this = shift;

    #---- Register data mining specific classes for logging
    #$theLogger->registerAdditionalClass( "a2w::Container" );
    #$theLogger->registerAdditionalClass( "a2w::Image" );
    #$theLogger->registerAdditionalClass( "a2w::Line" );
    #$theLogger->registerAdditionalClass( "a2w::Text" );
    #$theLogger->registerAdditionalClass( "a2w::Vector" );

    #$theLogger->registerAdditionalClass( "a2w::core::dm::AddContent" );
    #$theLogger->registerAdditionalClass( "a2w::core::dm::Anchor" );
    #$theLogger->registerAdditionalClass( "a2w::core::dm::Block" );
    #$theLogger->registerAdditionalClass( "a2w::core::dm::BlockFormatter" );
    #$theLogger->registerAdditionalClass( "a2w::core::dm::CompositeBlockFormatter" );
    #$theLogger->registerAdditionalClass( "a2w::core::dm::CompositeContentDef" );
    #$theLogger->registerAdditionalClass( "a2w::core::dm::Config" );
    #$theLogger->registerAdditionalClass( "a2w::core::dm::ConfigParser" );
    #$theLogger->registerAdditionalClass( "a2w::core::dm::Constraint" );
    #$theLogger->registerAdditionalClass( "a2w::core::dm::Constraints" );
    #$theLogger->registerAdditionalClass( "a2w::core::dm::ContentDef" );
    #$theLogger->registerAdditionalClass( "a2w::core::dm::ContentParser" );
    #$theLogger->registerAdditionalClass( "a2w::core::dm::ContentProcessor" );
    #$theLogger->registerAdditionalClass( "a2w::core::dm::Database" );
    #$theLogger->registerAdditionalClass( "a2w::core::dm::Dumper" );
    #$theLogger->registerAdditionalClass( "a2w::core::dm::FindAndAppend" );   # V118a Change
    #$theLogger->registerAdditionalClass( "a2w::core::dm::FindAndPrepend" );  # V118a Change
    #$theLogger->registerAdditionalClass( "a2w::core::dm::FindAndReplace" );
    #$theLogger->registerAdditionalClass( "a2w::core::dm::Index" );
    #$theLogger->registerAdditionalClass( "a2w::core::dm::ListBlockFormatter" );
    #$theLogger->registerAdditionalClass( "a2w::core::dm::ListContentDef" );
    #$theLogger->registerAdditionalClass( "a2w::core::dm::MiningUtils" );
    #$theLogger->registerAdditionalClass( "a2w::core::dm::ParagraphBlockFormatter" );
    #$theLogger->registerAdditionalClass( "a2w::core::dm::ParagraphContentDef" );
    #$theLogger->registerAdditionalClass( "a2w::core::dm::RawBlockFormatter" );
    #$theLogger->registerAdditionalClass( "a2w::core::dm::RawContentDef" );
    #$theLogger->registerAdditionalClass( "a2w::core::dm::SkipBlockFormatter" );
    #$theLogger->registerAdditionalClass( "a2w::core::dm::SkipContentDef" );
    #$theLogger->registerAdditionalClass( "a2w::core::dm::TableBlockFormatter" );
    #$theLogger->registerAdditionalClass( "a2w::core::dm::TableContentDef" );
    #$theLogger->registerAdditionalClass( "a2w::core::dm::IllustrationBlockFormatter" );
    #$theLogger->registerAdditionalClass( "a2w::core::dm::IllustrationContentDef" );
    #$theLogger->registerAdditionalClass( "a2w::core::visitor::Visitor" );
    #$theLogger->registerAdditionalClass( "a2w::core::visitor::HTMLVisitor" );
}

# V108 Begin
#-----------------------------------------------------------------------
# Add parser rule
#
# Add parser rule (anchor) based on which content parser will preprocess each
# object against parser rules and will invoke callback when object match any one
# of parser rule
#
# Parameters:
# hrefRulePar    Hash reference    Anchor object from block definition
# fnrefCBPar     Function          Callback to be invoked when object match any one of rule
#
# Returns 0 in case of success, <0 in case of error
#
#-----------------------------------------------------------------------
sub addParserRule{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "addParserRule" );
    }

    #---- Get parameter
    #
    # hrefRulePar    Hash reference    Anchor object from block definition
    # fnrefCBPar     Function          Callback to be invoked when object match any one of rule
    #
    my $hrefRulePar = shift;
    my $fnrefCBPar  = shift;

    #---- Assert parameters
    if ( $hrefRulePar == undef ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Skipped invalid parser rule" ); }
        return -1;
    }
    if ( $fnrefCBPar == undef ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Skipped parser rule due to missing callback function handle" ); }
        return -1;
    }

    #---- Create anchor from definition
    my $ancRuleTmp = a2w::core::dm::Block::createAnchor( undef, $hrefRulePar );

    #---- Add in to parser rules array
    $this->{ 'ParserRulesIndex' }++;
    $this->{ 'ParserRules' }[ $this->{ 'ParserRulesIndex' } ] = {
          'ANCHOR'   => $ancRuleTmp
        , 'CALLBACK' => $fnrefCBPar
    };
    if ( $bLog == $TRUE ){
        $theLogger->logMessage( "Added parser rule at " . $this->{ 'ParserRulesIndex' } . ":" );
        $theLogger->logMessage( " Anchor:>" );
        $theLogger->logHashMessage( $hrefRulePar );
        $theLogger->logMessage( " <" );
        $theLogger->logMessage( " Callback:>$fnrefCBPar<" );
    }

    return 0;
}
# V108 End

#-----------------------------------------------------------------------
# Process POM
#
# Process page objects and add them to appropriate block they belong to
#
# Parameters:
# 1. POM
# 2. Flag to indicate whether objects can be removed or skipped (0=>Skip, 1=>Removable)
#
# Returns 0 in case of success, <0 in case of error
#-----------------------------------------------------------------------
sub _processPOM{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "_processPOM" );
    }

    #---- Get parameter
    #
    # 1. POM
    # 2. Flag to indicate whether objects can be removed or skipped (0=>Skip, 1=>Removable)
    #
    my $dmPOMPar   = shift;
    my $bRemovePar = $FALSE;
    if ( @_ > 0 ){
        $bRemovePar = shift;
    }

    if ( $dmPOMPar->getObjectCount() <= 0 ){ return 0; } # V112a Change

    #---- Preprocess page ----#
    if ( $dmPOMPar->preProcess() < 0 ){ return -1; }

    #---- Process page content and classify them as blocks ----#
    my $iPageIdTmp   = $dmPOMPar->getPageID();
    my @arrLineTmp   = ();
    my @arrLinesTmp  = @{ $dmPOMPar->getLines() };
    my $hrefObjectsTmp = $dmPOMPar->getHashObj();
    my $hrefObjectTmp  = undef;
    my $a2wObjectTmp   = undef;

    my $blkObjectTmp   = undef;
    my $blkPageDefTmp  = undef;
    my $bObjAddedTmp   = $FALSE;
    my $bObjIsAncTmp   = $FALSE;
    my $bSkipObjTmp    = $FALSE;

    my @arrBlkListTmp = ();
    my @arrBlkLst2Tmp = ();

    my $hrefPageBlocksTmp   = {};
    my $arefPageBlocksTmp   = [];
    my $iPageDefBlkCountTmp = 0;

    #---- Iterate through page lines
    foreach my $arefLineTmp ( @arrLinesTmp ){
        @arrLineTmp = @{ $arefLineTmp }; # get line objects

        #---- Iterate through line objects
        foreach $hrefObjectTmp ( @arrLineTmp ){
            #---- Get document blocks
            @arrBlkListTmp = @{ $this->{ 'DocBlocks' } };

            #---- Filter out unfilled blocks
            @arrBlkListTmp = grep { $_ != undef && $_->isFilled() == $FALSE } @arrBlkListTmp;

            #---- Get kernel object
            $a2wObjectTmp = $hrefObjectsTmp->{ $hrefObjectTmp->{ $a2w::core::dm::Constants::AT_OBJSEQID } }{ 'OBJECT' };
            if ( $bLog == $TRUE ){ $this->_logObject( $a2wObjectTmp, $hrefObjectTmp ); }

            #---- Detect all the blocks the object falls in and add to them ----#
            $bObjAddedTmp  = $FALSE;
            $bSkipObjTmp   = $FALSE;
            @arrBlkLst2Tmp = @arrBlkListTmp;
            do {
                $bObjIsAncTmp = $FALSE;
                $blkObjectTmp = $this->_fetchObjectBlock( # POM page
                                                            $dmPOMPar
                                                          # POM object hash
                                                          , $hrefObjectTmp
                                                          # Object
                                                          , $a2wObjectTmp
                                                          # Block list
                                                          , \@arrBlkLst2Tmp
                                                          # Anchor or not flag return value
                                                          , \$bObjIsAncTmp
                                                        );

                #---- Add object to the configured block
                if ( $blkObjectTmp != undef ){
                    if ( $blkPageDefTmp != undef ){
                        $blkPageDefTmp->setAsFilled();
                        $blkPageDefTmp = undef;
                    }
                    my $sBlkIdTmp = $blkObjectTmp->getUniqueId();
                    if ( $hrefPageBlocksTmp->{ $sBlkIdTmp } == undef ){
                        if ( $bLog == $TRUE ){
                            $theLogger->logMessage( "Block " . $blkObjectTmp->getId() . " added to page blocks list" );
                        }
                        $hrefPageBlocksTmp->{ $sBlkIdTmp } = $blkObjectTmp;
                        $arefPageBlocksTmp->[ @{ $arefPageBlocksTmp } ] = $blkObjectTmp;
                    }

                    if ( $bObjIsAncTmp == $FALSE ){
                        $blkObjectTmp->addObject( $a2wObjectTmp, $hrefObjectTmp );
                    }
                    if ( $bSkipObjTmp == $FALSE ){
                        $bSkipObjTmp = $blkObjectTmp->hasToBeSkipped();
                    }
                    $bObjAddedTmp  = $TRUE;

                    #---- Filter blocks other than current block to find object belong to them too
                    my $sBlkIdTmp  = $blkObjectTmp->getId();
                    @arrBlkLst2Tmp = grep { $_ != undef && $_->getId() ne $sBlkIdTmp } @arrBlkLst2Tmp;
                }
            } while ( $blkObjectTmp != undef );

            #---- Add object to default block
            if ( $bObjAddedTmp == $FALSE ){
                #---- Add block to page default block
                if ( $blkPageDefTmp == undef ){
                    $iPageDefBlkCountTmp++;
                    my $sBlkIdTmp = $a2w::core::dm::Constants::BLK_PAGE_DEF_PREFIX . $iPageIdTmp . "_" . $iPageDefBlkCountTmp;
                    $blkPageDefTmp = $this->{ 'Config' }->createDefaultBlock( $sBlkIdTmp );
                    $sBlkIdTmp = $blkPageDefTmp->getUniqueId();
                    if ( $hrefPageBlocksTmp->{ $sBlkIdTmp } == undef ){
                        if ( $bLog == $TRUE ){
                            $theLogger->logMessage( "Block " . $blkPageDefTmp->getId() . " added to page blocks list" );
                        }
                        $hrefPageBlocksTmp->{ $sBlkIdTmp } = $blkPageDefTmp;
                        $arefPageBlocksTmp->[ @{ $arefPageBlocksTmp } ] = $blkPageDefTmp;
                    }

                    #---- Fill in info on default block
                    $blkPageDefTmp->setWidth( $dmPOMPar->getPageWidth() );
                    $blkPageDefTmp->setStartedOn( $iPageIdTmp );
                    # V116 Change: Updated boundary in addObject itself
                }
                $blkPageDefTmp->addObject( $a2wObjectTmp, $hrefObjectTmp );
                # V116 Change: Updated boundary in addObject itself
            }

            #---- Assert and remove object from kernel presentation
            if ( $bRemovePar == $TRUE ){
                my $hrefObjTmp = {
                      'A2WOBJ' => $a2wObjectTmp
                    , 'POMOBJ' => $hrefObjectTmp
                };
                if (    $this->{ 'Visitor' } != undef
                     && $this->{ 'Visitor' }->isWritable( $hrefObjTmp ) == $TRUE
                   ){
                    #---- Remove object from kernel presentation
                    $a2wObjectTmp->remove();
                    if ( $bLog == $TRUE ){
                        $theLogger->logMessage( "object removed from presentation" );
                    }
                }
                elsif ( $bSkipObjTmp == $TRUE ){
                    #---- Remove object from kernel presentation
                    $a2wObjectTmp->remove();
                    if ( $bLog == $TRUE ){
                        $theLogger->logMessage( "object removed from presentation" );
                    }
                }
            }
        } # foreach $hrefObjectTmp ( @arrLineTmp )
    } # foreach my $arefLineTmp ( @arrLinesTmp )

    #---- Set last page default block as filled
    if ( $blkPageDefTmp != undef ){
        $blkPageDefTmp->setAsFilled();
        $blkPageDefTmp = undef;
    }

    # V107 Begin
    #---- Store the page blocks
    $this->{ 'PageBlocks' }{ $iPageIdTmp } = $arefPageBlocksTmp;

    #---- Iterate and fill region like blocks
    my $sBlkListTmp = "";
    my @arrBlkListTmp = @{ $arefPageBlocksTmp };
    foreach my $blkTmp ( @arrBlkListTmp ){
        $sBlkListTmp .= " " . $blkTmp->getId();

        #---- Mark block as filled if it is region block
        if ( $blkTmp->isRegion() == $TRUE ){
            $blkTmp->setAsFilled();

            # V109 Begin
            #---- Mark on which page block started
            if ( $blkTmp->getStartedOn() <= 0 ){ $blkTmp->setStartedOn( $iPageIdTmp ); }
            # V109 End
        }

        if ( $blkTmp->isFilled() ){
            #---- Assert and create repetitive block ----#
            if (    $blkTmp->isRepetitive() == $TRUE
                 && $blkTmp->hasMoreRepeatation() == $TRUE
               ){
                #---- Create repetitive block
                my $blkRepetitiveTmp = $blkTmp->createRepetitiveBlock();

                #---- Assert and set info (if missing in config but exists in repetitive block)
                if (    $blkTmp->getInfo() ne ""            # Info exists on repetitive block (might be specified in TLE)
                     && $blkRepetitiveTmp->getInfo() eq ""  # Info missing on config
                   ){
                    $blkRepetitiveTmp->setInfo( $blkTmp->getInfo() );
                }

                #---- Add to document block list
                $this->addRepetitiveDocumentBlock( $blkRepetitiveTmp );
            }

            #---- Update block
            $blkTmp->update();
        } # if ( $blkTmp->isFilled() )
    }
    if ( $bLog == $TRUE ){ $theLogger->logMessage( "Page $iPageIdTmp blocks:$sBlkListTmp" ); }

    #---- Post process blocks ----#
    # 1. Reset block start anchor, if not completed on current page
    #    in order to continue content collection on following page
    #
    @arrBlkListTmp = @{ $this->{ 'DocBlocks' } };
    for ( my $i = 0; $i < @arrBlkListTmp; $i++ ){
        if ( @arrBlkListTmp[ $i ] != undef ){
            # 1. Reset block start anchor
            if (    @arrBlkListTmp[ $i ]->hasStarted() == $TRUE
                 && @arrBlkListTmp[ $i ]->hasEnded() == $FALSE
               ){
                @arrBlkListTmp[ $i ]->resetStartAnchor();
            }
        }
    }

    return 0;
}

#-----------------------------------------------------------------------
# Fetch object block
#-----------------------------------------------------------------------
sub _fetchObjectBlock{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "_fetchObjectBlock()" );
    #}

    #---- Get parameter
    #
    # 1. POM page
    # 2. Object hash
    # 3. Object
    # 4. Block array reference
    # 5. Reference to return flag saying whether object is start/end anchor
    #
    my $pomPagePar   = shift;
    my $hrefObjPar   = shift;
    my $a2wObjPar    = shift;
    my $blkListPar   = shift;
    my $bObjIsAncPar = shift;

    #---- Iterate through active blocks list and identify on which block the object falls ----#
    my $blkObjFallInTmp = undef;
    my $bObjFallInTmp   = $FALSE;

    my $bObjIsStartAnchorTmp = $FALSE;
    my $bObjIsEndAnchorTmp   = $FALSE;
    my @arrDocBlocksTmp = @{ $blkListPar };
    for ( my $i = 0; $i < @arrDocBlocksTmp; $i++ ){
        $blkObjFallInTmp = @arrDocBlocksTmp[ $i ];
        if ( $blkObjFallInTmp == undef ){
            next;
        }

        #---- Assert whether object is start anchor of a block
        if ( $blkObjFallInTmp->isStartAnchor( $pomPagePar, $a2wObjPar, $hrefObjPar ) == $TRUE ){
            $$bObjIsAncPar = $TRUE;
            $bObjFallInTmp = $TRUE;
            $bObjIsStartAnchorTmp = $TRUE;

            # V101 Begin
            #---- Mark on which page block started
            if ( $blkObjFallInTmp->getStartedOn() <= 0 ){
                $blkObjFallInTmp->setStartedOn( $hrefObjPar->{ $a2w::core::dm::Constants::AT_PAGE_ID } );

                #---- Add start anchor to block only when detected first, for following pages add it as regular object
                $blkObjFallInTmp->addStartAnchor( $a2wObjPar, $hrefObjPar );
            }
            else {
                my $ancStTmp = $blkObjFallInTmp->getStartAnchor();
                if ( $ancStTmp != undef && $ancStTmp->isSkipped() == $FALSE ){
                    # Let object added as regular instead of starting anchor
                    $$bObjIsAncPar = $FALSE;
                }
                else {
                    #---- Add start anchor to block
                    $blkObjFallInTmp->addStartAnchor( $a2wObjPar, $hrefObjPar );
                }
            }
            # V101 End
        }

        #---- Assert whether object fall in block
        # V101 Begin
        #if ( $blkObjFallInTmp->doesObjectFallIn( $a2wObjPar ) == $TRUE ){
        if ( $blkObjFallInTmp->doesObjectFallIn( $pomPagePar, $a2wObjPar, $hrefObjPar ) == $TRUE ){
        # V101 End
            $bObjFallInTmp = $TRUE;
        }

        #---- Assert whether object is end anchor of a block
        if ( $blkObjFallInTmp->isEndAnchor( $pomPagePar, $a2wObjPar, $hrefObjPar ) == $TRUE ){
            $$bObjIsAncPar = $TRUE;
            $bObjFallInTmp = $TRUE;
            $bObjIsEndAnchorTmp = $TRUE;

            #---- Add end anchor to block
            if ( $bObjIsStartAnchorTmp == $FALSE ){
                #---- Do not add to block, if same object is start/end anchor
                #     to avoid anchor being written twice
                $blkObjFallInTmp->addEndAnchor( $a2wObjPar, $hrefObjPar );
            }

            if ( $blkObjFallInTmp->isFilled() ){
                #---- Assert and create repetitive block ----#
                if (    $blkObjFallInTmp->isRepetitive() == $TRUE
                     && $blkObjFallInTmp->hasMoreRepeatation() == $TRUE
                   ){
                    #---- Create repetitive block
                    my $blkRepetitiveTmp = $blkObjFallInTmp->createRepetitiveBlock();

                    # V105 Begin
                    #---- Assert and set info (if missing in config but exists in repetitive block)
                    if (    $blkObjFallInTmp->getInfo() ne ""
                         && $blkRepetitiveTmp->getInfo() eq ""
                       ){
                        $blkRepetitiveTmp->setInfo( $blkObjFallInTmp->getInfo() );
                    }
                    # V105 End

                    #---- Add to document block list
                    # V103 Begin
                    #$this->addDocumentBlock( $blkRepetitiveTmp );
                    $this->addRepetitiveDocumentBlock( $blkRepetitiveTmp );
                    # V103 End
                }

                #---- Update block
                # V119 Change
                #$blkObjFallInTmp->update();
            }
        }

        #---- Break if object found to fall in a block
        if ( $bObjFallInTmp == $TRUE ){
            last;
        }
    }
    if ( $bObjFallInTmp == $FALSE ){ return undef; }

    return $blkObjFallInTmp;
}

#-----------------------------------------------------------------------
# Get additional contents of predefined blocks
#
# Analyze the blocks, detect header/body/footer/before/after additional
# contents if specified and collect them with out duplicate (using
# additional content id).
#
# Returns
# - undef in case of error
# - Hash reference having following structure, in case of success
#   {
#       'Header'    => <Additional content that belong to header>
#       'HdrBefore' => <Additional content that belong right after header start tag>
#       'HdrAfter'  => <Additional content that belong right above header end tag>
#       'Body'      => <Additional content that belong to body end>
#       'BdyBefore' => <Additional content that belong right after body start tag>
#       'BdyAfter'  => <Additional content that belong right above body end tag>
#       'Footer'    => <Additional content that belong to footer>
#       'FtrBefore' => <Additional content that belong right after footer start tag>
#       'FtrAfter'  => <Additional content that belong right above footer end tag>
#   }
#
#-----------------------------------------------------------------------
sub _getAdditionalContentsOfPredefinedBlocks{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "_getAdditionalContentsOfPredefinedBlocks()" );
    }

    #---- Predefined block names list
    my @arrPDHdrBlkNamesTmp = (
          $a2w::core::dm::Constants::BLK_DOC_HEADER
        , $a2w::core::dm::Constants::BLK_PAGE_HEADER
    );
    my @arrPDFtrBlkNamesTmp = (
          $a2w::core::dm::Constants::BLK_PAGE_FOOTER
        , $a2w::core::dm::Constants::BLK_DOC_FOOTER
    );
    my @arrPDBlkNamesTmp = (
          @arrPDHdrBlkNamesTmp
        , $a2w::core::dm::Constants::BLK_PAGE_BODY
        , @arrPDFtrBlkNamesTmp
    );

    #---- Filter configuration defined blocks
    my @arrPDBlkListTmp = ();
    my $hrefPDBlksTmp   = {};
    my $blkRefTmp = undef;
    foreach $id ( @arrPDBlkNamesTmp ){
        $blkRefTmp = $this->{ 'Config' }->getBlock( $id );
        if ( $blkRefTmp != undef ){
            if ( $bLog == $TRUE ){
                $theLogger->logMessage( "Block:>" . $id . "<" );
            }
            @arrPDBlkListTmp[ ( $#arrPDBlkListTmp + 1 ) ] = $blkRefTmp;
            $hrefPDBlksTmp->{ $id } = $blkRefTmp;
        }
    }

    #---- Get additional contents
    my $hrefPDACTmp = $this->_getAdditionalContents( \@arrPDBlkListTmp );

    #---- Additional processing ----#
    # For A2W_DMF_DOC_HEADER/A2W_DMF_PAGE_HEADER predefined block, additional contents included at
    # 'before'/'after' are treated same as 'header' position
    #
    my $hrefBeforeTmp = {};
    my $hrefAfterTmp  = {};
    my $contDefTmp    = undef;
    my $hrefACTmp     = undef;
    my @arrACKeysTmp  = ();
    # V116 Begin: Optimized and fixed minor bugs
    my $sPositionTmp  = '';

    foreach my $sHKeyTmp ( @arrPDHdrBlkNamesTmp ){
        if ( $hrefPDBlksTmp->{ $sHKeyTmp } == undef ){ next; }
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Block:>" . $sHKeyTmp . "<" ); }

        #---- Get content defition
        $contDefTmp = $hrefPDBlksTmp->{ $sHKeyTmp }->getContentDef();
        if ( $contDefTmp == undef ){ next; }

        $hrefBeforeTmp = {};
        $hrefAfterTmp  = {};
        $hrefACTmp = $contDefTmp->getAddContent();
        @arrACKeysTmp = sort keys( %{ $hrefACTmp } );
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "ContentDef exists: @arrACKeysTmp" ); }

        foreach my $sIdTmp ( @arrACKeysTmp ){
            $sPositionTmp = lc( $hrefACTmp->{ $sIdTmp }->getInclude() ); # V116 Change
            if ( $sPositionTmp eq "before" ){ # V116 Change
                $hrefBeforeTmp->{ $sIdTmp } = $hrefACTmp->{ $sIdTmp }->getContent();
                if ( $bLog == $TRUE ){ $theLogger->logMessage( "Before content of " . $sIdTmp . ": " . $hrefBeforeTmp->{ $sIdTmp } ); }
            }
            elsif ( $sPositionTmp eq "after" ){ # V116 Change
                $hrefAfterTmp->{ $sIdTmp } = $hrefACTmp->{ $sIdTmp }->getContent();
                if ( $bLog == $TRUE ){ $theLogger->logMessage( "After content of " . $sIdTmp . ": " . $hrefAfterTmp->{ $sIdTmp } ); }
            }
        }

        my $sBeforeTmp = '';
        my $sAfterTmp  = '';
        my @arrHdrContTmp = sort keys( %{ $hrefBeforeTmp } );
        foreach my $b ( @arrHdrContTmp ){
            $sBeforeTmp .= $hrefBeforeTmp->{ $b };
        }
        @arrHdrContTmp = sort keys( %{ $hrefAfterTmp } );
        foreach my $a ( @arrHdrContTmp ){
            $sAfterTmp .= $hrefAfterTmp->{ $a }; # V116 Change
        }
        $hrefPDACTmp->{ 'HdrBefore' } .= $sBeforeTmp;
        $hrefPDACTmp->{ 'HdrAfter' }  .= $sAfterTmp;
        $hrefPDACTmp->{ 'Header' }    .= $sBeforeTmp . $sAfterTmp;
    }
    # V116 End

    # For A2W_DMF_PAGE_BODY predefined block, additional contents included at
    # 'after' are treated same as 'body' position
    #
    if ( $hrefPDBlksTmp->{ $a2w::core::dm::Constants::BLK_PAGE_BODY } != undef ){
        $hrefBeforeTmp = {};
        $hrefAfterTmp  = {};
        $contDefTmp    = undef;
        $hrefACTmp     = undef;
        @arrACKeysTmp  = ();

        #---- Get content defition
        $contDefTmp = $hrefPDBlksTmp->{ $a2w::core::dm::Constants::BLK_PAGE_BODY }->getContentDef();
        if ( $contDefTmp != undef ){
            $hrefACTmp = $contDefTmp->getAddContent();
            @arrACKeysTmp = sort keys( %{ $hrefACTmp } );
            $sPositionTmp = ''; # V116 Change

            foreach my $sIdTmp ( @arrACKeysTmp ){
                $sPositionTmp = lc( $hrefACTmp->{ $sIdTmp }->getInclude() ); # V116 Change
                if ( $sPositionTmp eq "before" ){ # V116 Change
                    $hrefBeforeTmp->{ $sIdTmp } = $hrefACTmp->{ $sIdTmp }->getContent();
                }
                elsif ( $sPositionTmp eq "after" ){ # V116 Change
                    $hrefAfterTmp->{ $sIdTmp } = $hrefACTmp->{ $sIdTmp }->getContent();
                }
            }

            my $sBeforeTmp = '';
            my $sAfterTmp  = '';
            my @arrBodyContTmp = sort keys( %{ $hrefBeforeTmp } );
            foreach my $b ( @arrBodyContTmp ){
                $sBeforeTmp .= $hrefBeforeTmp->{ $b };
            }
            @arrBodyContTmp = sort keys( %{ $hrefAfterTmp } );
            foreach my $a ( @arrBodyContTmp ){ # V116 Change
                $sAfterTmp .= $hrefAfterTmp->{ $a }; # V116 Change
            }
            $hrefPDACTmp->{ 'BdyBefore' } .= $sBeforeTmp;
            $hrefPDACTmp->{ 'BdyAfter' }  .= $sAfterTmp;
            $hrefPDACTmp->{ 'Body' }      .= $sBeforeTmp . $sAfterTmp;
        }
    }

    # For A2W_DMF_DOC_FOOTER/A2W_DMF_PAGE_FOOTER predefined block, additional contents included at
    # 'before'/'after' are treated same as 'footer' position
    #
    $contDefTmp    = undef;
    $hrefACTmp     = undef;
    @arrACKeysTmp  = ();
    foreach my $sFKeyTmp ( @arrPDFtrBlkNamesTmp ){
        if ( $hrefPDBlksTmp->{ $sFKeyTmp } == undef ){
            next;
        }

        #---- Get content defition
        $hrefBeforeTmp = {};
        $hrefAfterTmp  = {};
        $contDefTmp = $hrefPDBlksTmp->{ $sFKeyTmp }->getContentDef();
        if ( $contDefTmp != undef ){
            $hrefACTmp = $contDefTmp->getAddContent();
            @arrACKeysTmp = sort keys( %{ $hrefACTmp } );
            $sPositionTmp = ''; # V116 Change

            foreach my $sIdTmp ( @arrACKeysTmp ){
                $sPositionTmp = lc( $hrefACTmp->{ $sIdTmp }->getInclude() ); # V116 Change
                if ( $sPositionTmp eq "before" ){ # V116 Change
                    $hrefBeforeTmp->{ $sIdTmp } = $hrefACTmp->{ $sIdTmp }->getContent();
                }
                elsif ( $sPositionTmp eq "after" ){ # V116 Change
                    $hrefAfterTmp->{ $sIdTmp } = $hrefACTmp->{ $sIdTmp }->getContent();
                }
            }

            my $sBeforeTmp = '';
            my $sAfterTmp  = '';
            my @arrFtContTmp = sort keys( %{ $hrefBeforeTmp } );
            foreach my $b ( @arrFtContTmp ){
                $sBeforeTmp .= $hrefBeforeTmp->{ $b };
            }
            @arrFtContTmp = sort keys( %{ $hrefAfterTmp } );
            foreach my $a ( @arrFtContTmp ){
                $sAfterTmp .= $hrefAfterTmp->{ $a };
            }
            $hrefPDACTmp->{ 'FtrBefore' } .= $sBeforeTmp;
            $hrefPDACTmp->{ 'FtrAfter' }  .= $sAfterTmp;
            $hrefPDACTmp->{ 'Footer' }    .= $sBeforeTmp . $sAfterTmp;
        }
    }

    if ( $bLog == $TRUE ){
        $theLogger->logMessage( "Predefined block additional contents:" );
        $theLogger->logMessage( "Header:>" . $hrefPDACTmp->{ 'Header' } . "<" );
        $theLogger->logMessage( "Header Before:>" . $hrefPDACTmp->{ 'HdrBefore' } . "<" );
        $theLogger->logMessage( "Header After:>" . $hrefPDACTmp->{ 'HdrAfter' } . "<" );
        $theLogger->logMessage( "Body:>" . $hrefPDACTmp->{ 'Body' } . "<" );
        $theLogger->logMessage( "Body Before:>" . $hrefPDACTmp->{ 'BdyBefore' } . "<" );
        $theLogger->logMessage( "Body After:>" . $hrefPDACTmp->{ 'BdyAfter' } . "<" );
        $theLogger->logMessage( "Footer:>" . $hrefPDACTmp->{ 'Footer' } . "<" );
        $theLogger->logMessage( "Footer Before:>" . $hrefPDACTmp->{ 'FtrBefore' } . "<" );
        $theLogger->logMessage( "Footer After:>" . $hrefPDACTmp->{ 'FtrAfter' } . "<" );
    }
    return $hrefPDACTmp;
}

#-----------------------------------------------------------------------
# Get additional contents of blocks
#
# Analyze the blocks, detect header/body/footer additional contents if
# specified and collect them with out duplicate (using additional content
# id).
#
# Parameter
# 1. Array of blocks
#
# Returns
# - undef in case of error
# - Hash reference having following structure, in case of success
#   {
#       'Header' => <Additional content that belong to header>
#       'Body'   => <Additional content that belong to body end>
#       'Footer' => <Additional content that belong to footer>
#   }
#
#-----------------------------------------------------------------------
sub _getAdditionalContents{
    my $this = shift;

    #---- Get parameter
    #
    # 1. Array of blocks
    #
    my $arefBlocksPar = shift;

    #---- Get blocks
    my @arrBlkListTmp = @{ $arefBlocksPar };

    #---- Collect additional header/body/footer contents, if block has any ----#
    my $hrefHeaderTmp = {};
    my $hrefBodyTmp   = {};
    my $hrefFooterTmp = {};
    my $contDefTmp    = undef;
    my $hrefACTmp     = undef;
    my @arrACKeysTmp  = ();
    my $sPositionTmp  = ''; # V116 Change

    foreach my $blkTmp ( ( @arrBlkListTmp, @arrPDBlkListTmp ) ){
        #---- Get content defition
        $contDefTmp = $blkTmp->getContentDef();
        if ( $contDefTmp == undef ){ next; }

        $hrefACTmp = $contDefTmp->getAddContent();
        @arrACKeysTmp = sort keys( %{ $hrefACTmp } );

        foreach my $sIdTmp ( @arrACKeysTmp ){
            $sPositionTmp = lc( $hrefACTmp->{ $sIdTmp }->getInclude() ); # V116 Change
            if ( $sPositionTmp eq "header" ){ # V116 Change
                $hrefHeaderTmp->{ $sIdTmp } = $hrefACTmp->{ $sIdTmp }->getContent();
            }
            elsif ( $sPositionTmp eq "body" ){ # V116 Change
                $hrefBodyTmp->{ $sIdTmp } = $hrefACTmp->{ $sIdTmp }->getContent();
            }
            elsif ( $sPositionTmp eq "footer" ){ # V116 Change
                $hrefFooterTmp->{ $sIdTmp } = $hrefACTmp->{ $sIdTmp }->getContent();
            }
        }
    }

    #---- Prepare return value
    my $hrefAddContTmp = {
          'Header'    => ''
        , 'HdrBefore' => ''
        , 'HdrAfter'  => ''
        , 'Body'      => ''
        , 'BdyBefore' => ''
        , 'BdyAfter'  => ''
        , 'Footer'    => ''
        , 'FtrBefore' => ''
        , 'FtrAfter'  => ''
    };
    my @arrHdrContTmp = sort keys( %{ $hrefHeaderTmp } );
    foreach my $h ( @arrHdrContTmp ){
        $hrefAddContTmp->{ 'Header' } .= $hrefHeaderTmp->{ $h };
    }
    my @arrBdContTmp = sort keys( %{ $hrefBodyTmp } );
    foreach my $b ( @arrBdContTmp ){
        $hrefAddContTmp->{ 'Body' } .= $hrefBodyTmp->{ $b };
    }
    my @arrFtContTmp = sort keys( %{ $hrefFooterTmp } );
    foreach my $f ( @arrFtContTmp ){
        $hrefAddContTmp->{ 'Footer' } .= $hrefFooterTmp->{ $f };
    }

    return $hrefAddContTmp;
}

#-----------------------------------------------------------------------
# Log object
#-----------------------------------------------------------------------
sub _logObject{
    my $this = shift;

    #---- Get parameter
    #
    # 1. Object
    # 2. POM object wrapper (optional)
    #
    my $a2wObjectPar = shift;
    my $pomObjectPar = undef;
    if ( @_ > 0 ){
        $pomObjectPar = shift;
    }

    #---- Build log message ----#
    my $sObjTypeTmp = $a2wObjectPar->_getType();
    my $sLogMsgTmp  = "";
    # V109 Begin
    my $sLogObjTmp  = "";
    if ( $pomObjectPar->{ $a2w::core::dm::Constants::AT_PAGEFIRST } == $TRUE ){ $sLogObjTmp  = " Page first object"; }
    if ( $pomObjectPar->{ $a2w::core::dm::Constants::AT_PAGELAST } == $TRUE ){
        if ( $sLogObjTmp ne "" ){ $sLogObjTmp = " Page first and last object"; }
        else { $sLogObjTmp = " Page last object"; }
    }
    # V109 End

    if ( $sObjTypeTmp eq "image" ){
        if ( $pomObjectPar == undef ){
            $sLogMsgTmp = "img>" . $a2wObjectPar->getName() . "<@(" . $a2wObjectPar->getXPos() . "," . $a2wObjectPar->getYPos() . ") W=>" . $a2wObjectPar->getWidth() . "< H=>" . $a2wObjectPar->getHeight() . "<$sLogObjTmp";
        }
        else {
            $sLogMsgTmp = "img>" . $a2wObjectPar->getName() . "<@(" . $pomObjectPar->{ $a2w::core::dm::Constants::AT_XPOS } . "," . $pomObjectPar->{ $a2w::core::dm::Constants::AT_YPOS } . ") W=>" . $a2wObjectPar->getWidth() . "< H=>" . $a2wObjectPar->getHeight() . "<$sLogObjTmp";
        }
    }
    # V106 Begin
    if ( $sObjTypeTmp eq "container" ){
        if ( $pomObjectPar == undef ){
            $sLogMsgTmp = "obj>" . $a2wObjectPar->getName() . "<@(" . $a2wObjectPar->getXPos() . "," . $a2wObjectPar->getYPos() . ") W=>" . $a2wObjectPar->getWidth() . "< H=>" . $a2wObjectPar->getHeight() . "<$sLogObjTmp";
        }
        else {
            $sLogMsgTmp = "obj>" . $a2wObjectPar->getName() . "<@(" . $pomObjectPar->{ $a2w::core::dm::Constants::AT_XPOS } . "," . $pomObjectPar->{ $a2w::core::dm::Constants::AT_YPOS } . ") W=>" . $a2wObjectPar->getWidth() . "< H=>" . $a2wObjectPar->getHeight() . "<$sLogObjTmp";
        }
    }
    # V106 End
    # V109 Begin
    elsif ( $sObjTypeTmp eq "line" ){
        if ( $pomObjectPar == undef ){
            $sLogMsgTmp = "lin><@(" . $a2wObjectPar->getXPos() . "," . $a2wObjectPar->getYPos() . ") W=>" . $a2wObjectPar->getWidth() . "< L=>" . $a2wObjectPar->getLength() . "<$sLogObjTmp";
        }
        else {
            $sLogMsgTmp = "lin><@(" . $pomObjectPar->{ $a2w::core::dm::Constants::AT_XPOS } . "," . $pomObjectPar->{ $a2w::core::dm::Constants::AT_YPOS } . ") W=>" . $a2wObjectPar->getWidth() . "< L=>" . $a2wObjectPar->getLength() . "<$sLogObjTmp";
        }
    }
    elsif ( $sObjTypeTmp eq "text" ){
        if ( $pomObjectPar == undef ){
            $sLogMsgTmp = "txt>" . $a2wObjectPar->getText() . "<@(" . $a2wObjectPar->getXPos() . "," . $a2wObjectPar->getYPos() . ")$sLogObjTmp";
        }
        else {
            $sLogMsgTmp = "txt>" . $a2wObjectPar->getText() . "<@(" . $pomObjectPar->{ $a2w::core::dm::Constants::AT_XPOS } . "," . $pomObjectPar->{ $a2w::core::dm::Constants::AT_YPOS } . ")$sLogObjTmp";
        }
    }
    elsif ( $sObjTypeTmp eq "vector" ){
        if ( $pomObjectPar == undef ){
            $sLogMsgTmp = "vec><@(" . $a2wObjectPar->getXPos() . "," . $a2wObjectPar->getYPos() . ")$sLogObjTmp";
        }
        else {
            $sLogMsgTmp = "vec><@(" . $pomObjectPar->{ $a2w::core::dm::Constants::AT_XPOS } . "," . $pomObjectPar->{ $a2w::core::dm::Constants::AT_YPOS } . ")$sLogObjTmp";
        }
    }
    # V109 End
    # V113 Begin
    elsif ( $sObjTypeTmp eq "annotation" ){
        if ( $pomObjectPar == undef ){
            $sLogMsgTmp = "ant>" . $a2wObjectPar->getURL() . "<@(" . $a2wObjectPar->getXPos() . "," . $a2wObjectPar->getYPos() . ")$sLogObjTmp";
        }
        else {
            $sLogMsgTmp = "ant>" . $a2wObjectPar->getURL() . "<@(" . $pomObjectPar->{ $a2w::core::dm::Constants::AT_XPOS } . "," . $pomObjectPar->{ $a2w::core::dm::Constants::AT_YPOS } . ")$sLogObjTmp";
        }
    }
    # V113 End
    $theLogger->logMessage( $sLogMsgTmp );
}

# V110 Begin
#-----------------------------------------------------------------------
# findEyecatcher
#
# Find eyecatcher on given page content
#
# Parameters:
# pageid      Page id. Used to get page content for searching eyecatcher.
# options     options is an object, it should be defined as follow
#             "options":{
#               'xRange':        {'from':0,'to':100000}, // where xRange, from and to are optional
#               'yRange':        {'from':0,'to':100000}, // where yRange, from and to are optional
#               'reEC':          <regular expression>,   // eyecatcher regexpr
#               'start':         OPTIONAL: specify from which array index the search should start from, default is 0
#               'end':           OPTIONAL: specify up to which array index the search should go, default is maximum of content array
#               'objectType':    Type of eyecatcher object (text, line, vector, image, container). Default is text
#             }
#                  
# Prototypes:
#  ecIndexName = findEyecatcher(content, options)   where options is as defined above
# 
# Example:
#  ecIndexName = findEyecatcher(content,{'reEC' => qr/^Datum.*$/i})
#-----------------------------------------------------------------------
sub findEyecatcher{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "findEyecatcher()" ); }

    # Get page id and assert value
    my $iPageIdPar = shift;
    if ( $iPageIdPar <= 0 ){ return undef; }

    #---- Get page object
    my $pageTmp = $this->{ 'DOM' }{ $iPageIdPar }; # get page
    if ( $pageTmp == undef ){ return undef; }

    return $pageTmp->findEyecatcher( $arefObjsTmp, @_ );
}

#-----------------------------------------------------------------------
# findEyecatcherValue
#
# Find eyecatcher value on given page content
#
# Parameters:
# pageid      Page id. Used to get page content for searching eyecatcher/value.
# options     options is an object, it should be defined as follow
#             "options":{
#               'xRange':        {'from':0,'to':100000}, // where xRange, from and to are optional
#               'yRange':        {'from':0,'to':100000}, // where yRange, from and to are optional
#               'reEC':          <regular expression>,   // eyecatcher regexpr
#               'reECValue':     <regular expression>,   // eyecatcher value regexpr
#               'start':         OPTIONAL: specify from which array index the search should start from, default is 0
#               'end':           OPTIONAL: specify up to which array index the search should go, default is maximum of content array
#               'objectType':    Type of eyecatcher object (text, line, vector, image, container). Default is text
#               'direction':     Search direction for the eyecatcher value (right, left, top and bottom). Default is right
#               'xTolerance':    OPTIONAL: Specify adjustment in X range to search value. Default is 1
#               'yTolerance':    OPTIONAL: Specify adjustment in Y range to search value. Default is 1
#             }
#                  
# Prototypes:
#  ecIndexValue = findEyecatcherValue(content, options)   where options is as defined above
# 
# Example:
#  ecIndexValue = findEyecatcherValue(content,{'reEC' => qr/^Datum.*$/i, 'reECValue' => qr/^(\d{2}.\d{2}.\d{4})$/i})
#-----------------------------------------------------------------------
sub findEyecatcherValue{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "findEyecatcherValue()" ); }

    # Get page id and assert value
    my $iPageIdPar = shift;
    if ( $iPageIdPar <= 0 ){ return undef; }

    #---- Get page object
    my $pageTmp = $this->{ 'DOM' }{ $iPageIdPar }; # get page
    if ( $pageTmp == undef ){ return undef; }

    return $pageTmp->findEyecatcherValue( $arefObjsTmp, @_ );
}
# V110 End

# V118b Begin
#-----------------------------------------------------------------------
# Load block definition
#
# Loads given block definition module and returns the hash structure
# Returns <0 and message in case of error
#
#-----------------------------------------------------------------------
sub loadBlockDefinition{
    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "_loadBlockDefinition()" ); }

    #---- Get parameter
    #
    # 1. Module name
    # 2. Module filename (optional)
    # 3. Parser ruler handler (optional, function reference)
    #
    my $sBlkDefModPar = shift;
    my $sBlkDefFilePar = "";
    my $funcParserRulerHandlerPar = undef;
    if ( @_ > 0 ){ $sBlkDefFilePar = shift; }
    if ( @_ > 0 ){ $funcParserRulerHandlerPar = shift; }

    #---- Assert parameter
    if ( $sBlkDefFilePar eq "" && $sBlkDefModPar eq "" ){ return ( -1, "Empty block definition module name" ); } # Return if name is empty
    if ( $sBlkDefFilePar eq "" && $sBlkDefModPar !~ /^.*\:\:.*$/ ){ return ( -1, "Invalid block definition module name ($sBlkDefModPar)" ); } # Return if name is not like module name

    #---- Parse and load block definition module
    if ( $sBlkDefFilePar eq "" ){
        eval "require $sBlkDefModPar";
        if ( $@ ){ return ( -2, "Unable to parse block definition module $sBlkDefModPar. reason=" . $@ ); }
    }
    else {
        require $sBlkDefFilePar;
    }

    #---- Fill in block definition
    $hrefBlockDef = {
          'Blocks'        => \%{ $sBlkDefModPar . '::Blocks' }        # Block definitions
        , 'ParserRules'   => \@{ $sBlkDefModPar . '::ParserRules' }   # Parser rules
        , 'ParserActions' => \%{ $sBlkDefModPar . '::ParserActions' } # Parser actions
    };
    my $hrefBlocksTmp = $hrefBlockDef->{ 'Blocks' };
    my @arrRulesTmp = @{ $hrefBlockDef->{ 'ParserRules' } };
    if ( $bLog == $TRUE && @arrRulesTmp > 0 ){ $theLogger->logMessage( "$sBlkDefModPar Defined Parser Rules:>" . @arrRulesTmp . "<" ); }

    #---- Load all blocks
    $this->loadConfig( $hrefBlocksTmp );

    #---- Fetch all config module defined blocks
    @arrBlocks = sort keys( %{ $hrefBlocksTmp } );
    if ( $bLog == $TRUE ){ $theLogger->logMessage( "$sBlkDefModPar Defined Blocks:>@arrBlocks<" ); }

    #---- Iterate through blocks list and add them as document block
    my $blkCurrentTmp = undef;
    foreach my $sBlkNameTmp ( @arrBlocks ){
        #---- Get block
        $blkCurrentDefTmp = $hrefBlocksTmp->{ $sBlkNameTmp };
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Name:>" . $sBlkNameTmp . "< Id:>" . $blkCurrentDefTmp->{ 'Id' } . "< Info:>" . $blkCurrentDefTmp->{ 'Info' } . "<" ); }

        #---- Add block to document
        $this->addDocumentBlock( $blkCurrentDefTmp->{ 'Id' }, $blkCurrentDefTmp->{ 'Info' } );
    }

    #---- Set parser actions
    if ( $hrefBlockDef->{ 'ParserActions' } != undef ){
		$this->setParserActions( $hrefBlockDef->{ 'ParserActions' } );

		#---- Assert and assign default parser ruler handler
		if (    $funcParserRulerHandlerPar == undef
             && @arrRulesTmp > 0
           ){
			$funcParserRulerHandlerPar = sub { $this->preProcessParserObject( @_ ) };
		}
    }

    #---- Iterate through array of rules and add them
    if (    @arrRulesTmp > 0
         && $funcParserRulerHandlerPar == undef
       ){
		return ( -3, "Invalid/Undefined parser ruler handler" );
    }
    foreach my $pr ( @arrRulesTmp ){
        $this->addParserRule( $pr, $funcParserRulerHandlerPar );
    }

    return $hrefBlocksTmp;
}

#-----------------------------------------------------------------------
# Create parser action defintion object
#
#-----------------------------------------------------------------------
sub _createActionObject{
    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "_createActionObject()" ); }

    #---- Get parameter
    #
    # 1. Parser action
    # 2. Position reference object
    # 3. Page reference
    #
    my $prActionPar = shift;
    my $a2wRefObjPar = shift;
    my $a2wPageRefPar = shift;

    #---- Assert parameters
    if (    $prActionPar == undef
         || $a2wRefObjPar == undef
         || $a2wPageRefPar == undef
       ){
       return undef;
    }

    my $iRefXPosTmp = $a2wRefObjPar->getXPos();
    my $iRefYPosTmp = $a2wRefObjPar->getYPos();

    my $a2wObjTmp = undef;
    my $iXPosTmp = $prActionPar->{ 'XPOS' };
    my $iYPosTmp = $prActionPar->{ 'YPOS' };

    if ( $iXPosTmp == undef && defined( $prActionPar->{ 'REL_XPOS' } ) ){
        $iXPosTmp = $iRefXPosTmp + $prActionPar->{ 'REL_XPOS' };
    }
    if ( $iYPosTmp == undef && defined( $prActionPar->{ 'REL_YPOS' } ) ){
        $iYPosTmp = $iRefYPosTmp + $prActionPar->{ 'REL_YPOS' };
    }

    my $sObjTypeTmp = lc( $prActionPar->{ 'OBJ_TYPE' } );
    if ( $sObjTypeTmp eq "line" ){
        $a2wObjTmp = new a2w::Line();
        if ( $a2wObjTmp == undef ){ return undef; }

        my $iWidthTmp = ( $prActionPar->{ 'WIDTH' } == undef ) ? 1 : $prActionPar->{ 'WIDTH' };
        my $iLenghtTmp = ( $prActionPar->{ 'LENGTH' } == undef ) ? 1 : $prActionPar->{ 'LENGTH' };

        $a2wObjTmp->setXPos( $iXPosTmp );
        $a2wObjTmp->setYPos( $iYPosTmp );
        $a2wObjTmp->setWidth( $iWidthTmp );
        $a2wObjTmp->setLength( $iLenghtTmp );
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "line\@($iXPosTmp, $iYPosTmp) W=$iWidthTmp L=$iLenghtTmp" ); }
    }
    #elsif ( $sObjTypeTmp eq "text" ){
    #TODO
    #}
    $a2wObjTmp->remove(); # Don't present it on output
    $a2wPageRefPar->addLine( $a2wObjTmp ); # Add it to page, so it gets deleted later in core

    return $a2wObjTmp;
}

#-----------------------------------------------------------------------
# Preprocess object callback implementation
#
# prototype:
# <Array of objects> <function name>( a2w::core::dm::Anchor, a2w::Page, a2w::<Container|Image|Line|Text|Vector>Object )
#
# Callback must preprocess the object and return array of objects to add on the page
#
# NOTE: Object passed in must also be included on returned array of objects if it
#       has to be added on page otherwise the object will be skipped
#
#-----------------------------------------------------------------------
sub preProcessParserObject{
    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "preProcessParserObject()" ); }

    #---- Get parameter
    #
    # 1. Anchor
    # 2. Page reference
    # 3. Object
    #
    my $ancMatchPar   = shift;
    my $a2wPageRefPar = shift;
    my $a2wObjectPar  = shift;

    #---- Assert parameters
    if (    $ancMatchPar == undef
         || $a2wPageRefPar == undef
         || $a2wObjectPar == undef
       ){
        return [ $a2wObjectPar ];
    }

    $iParserRulesCount++;

    #---- Do actions for matched rule
    my $iIdxTmp = 1;
    my @arrObjsTmp = ( $a2wObjectPar );
    my $sRuleIdTmp = lc( $ancMatchPar->getId() );
    if ( $bLog == $TRUE ){ $theLogger->logMessage( "Rule Id:>$sRuleIdTmp<" ); }

    my $hrefActionsTmp = $this->getParserActions();
    my @arrActionsTmp = @{ $hrefActionsTmp->{ $sRuleIdTmp } };
    if ( @arrActionsTmp <= 0 ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Rule ($sRuleIdTmp) has no actions defined, rule skipped." ); }
        return [ $a2wObjectPar ];
    }

    #---- Iterate and perform the actions
    foreach my $a ( @arrActionsTmp ){
        if ( $a == undef ){ next; }

        #---- Perform 'add' action
        if ( $a->{ 'ACTION' } eq 'add' ){
            #---- Create object to be added based on type
            my $a2wObjTmp = $this->_createActionObject( $a, $a2wObjectPar, $a2wPageRefPar );
            if ( $a2wObjTmp != undef ){ @arrObjsTmp[ $iIdxTmp++ ] = $a2wObjTmp; }
        }
    }

    return \@arrObjsTmp;
}
# V118b End

#-----------------------------------------------------------------------
# Don't remove the following lines !!!
#-----------------------------------------------------------------------
1;
__END__
