#-------------------------------------------------------------------------------
#  a2w::core::dm::SkipBlockFormatter.pm:
#  Data Mining Framework Skip Block Formatter
#
#  Author  : AFP2web Team, Maas Holding GmbH
#
#  $V100   2014-05-08    Initial Release
#
#  $V101   2018-10-26    a. Optimized preparing list of block objects
#                        b. Extended to handle multi type chained blocks
#                        c. Fixed minor bug in processing empty chained blocks
#
#-------------------------------------------------------------------------------
package a2w::core::dm::SkipBlockFormatter;

#-----------------------------------------------------------------------
# Include required modules
#-----------------------------------------------------------------------
use a2w::TypeConstants;
use a2w::core::log::Logger;
use a2w::core::dm::Constants;
use a2w::core::dm::BlockFormatter;

#-----------------------------------------------------------------------
# Inherit from base class
#-----------------------------------------------------------------------
our @ISA = qw( a2w::core::dm::BlockFormatter );

#-----------------------------------------------------------------------
# Constructor
#-----------------------------------------------------------------------
sub new{
    my $proto = shift;
    my $class = ref( $proto ) || $proto;

    #---- Define boolean values
    $TRUE  = $a2w::TypeConstants::TRUE;     # TRUE  boolean value
    $FALSE = $a2w::TypeConstants::FALSE;    # FALSE boolean value

    #---- Instantiate from base class to inherit base class attributes
    my $this = a2w::core::dm::BlockFormatter->new( @_ );

    #---- Add this derived class specific attributes
    #$this->{ 'someattribute' } = 0;

    bless( $this, $class );

    #---- Get logger
    our $theLogger = a2w::core::log::Logger->getSingleton();
    our $bLog = $theLogger->isRegistered( __PACKAGE__ );

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "new()" );
    #}
    return $this;
}

#-----------------------------------------------------------------------
# Destructor
#-----------------------------------------------------------------------
sub DESTROY{
    my $this = shift;
}

#-----------------------------------------------------------------------
# Mutators
#-----------------------------------------------------------------------

#-----------------------------------------------------------------------
# Accessors
#-----------------------------------------------------------------------

#-----------------------------------------------------------------------
# Workers
#-----------------------------------------------------------------------
#-----------------------------------------------------------------------
# writeFormattedBlock
#
# Asserts whether object fall in given block and returns following value
#
#  0, in case of success
# <0, in case of error
#
#-----------------------------------------------------------------------
sub writeFormattedBlock{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeFormattedBlock()" );
    }

    #---- Parameter
    #
    # 1. Block
    # 2. Visitor
    #
    my $blkCurrentPar = shift;
    my $outVisitorPar = shift;

    #---- Fetch content definition of block
    my $contDefTmp = $blkCurrentPar->getContentDef();
    if ( lc( $contDefTmp->getType() ) ne "skip" ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Error! Skip block formatter can not process other type (" . $contDefTmp->getType() . ") of blocks<" ); }
        return -1;
    }

    # V101 Begin
    #---- Assert block has content or not ----#
    my $arefObjsListTmp = $blkCurrentPar->getObjects();
    my @arrObjsListTmp = @{ $arefObjsListTmp };

    #---- Get next block in chain
    my $blkNextTmp = $blkCurrentPar->getNextRef();
    if ( @arrObjsListTmp <= 0 ){
        # When current block (or chained block) is empty, ensure to continue with next block in
        # chain (if any exists) else return no error (so that formatting completed properly)
        $blkCurrentPar->setFlushed( $TRUE );
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Warning! Skipped writing empty block (" . $blkCurrentPar->getId() . ")" ); }

        #---- Write chained block contents ----#
        if ( $blkNextTmp != undef ){
            #---- Check and create formatter
            my $fmtNextTmp = $blkNextTmp->getFormatter();
            if ( $fmtNextTmp == undef ){
                $blkNextTmp->_createContentFormatter();
                $fmtNextTmp = $blkNextTmp->getFormatter();
            }

            #---- Write next block
            $iRetTmp = $fmtNextTmp->writeFormattedContent( # Block
                                                             $blkNextTmp
                                                           # Visitor
                                                           , $outVisitorPar
                                                           # Line count
                                                           , 0
                                                         );
            if ( $iRetTmp < 0 ){ return $iRetTmp; }
        }
        return 0;
    }
    # V101 End

    #---- Begin skip block
    my $iRetTmp = $outVisitorPar->beginSkip( $blkCurrentPar );
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    #---- Write formatted content of block ----#
    $iRetTmp = $this->_writeFormattedContent( # Block
                                               $blkCurrentPar
                                              # Visitor
                                              , $outVisitorPar
                                              # Line count
                                              , 0
                                            );
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    #---- End skip block
    if ( $this->isFinalized() == $FALSE ){ $iRetTmp = $outVisitorPar->endSkip( $blkCurrentPar ); } # V101 Change
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    return 0;
}

#-----------------------------------------------------------------------
# _writeFormattedContent
#
# Writes block content formatted as lines (low level implementation)
#
# >=0, in case of successful writing
# < 0, in case of error
#
#-----------------------------------------------------------------------
sub _writeFormattedContent{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "_writeFormattedContent()" );
    }

    #---- Parameter
    #
    # 1. Block
    # 2. Visitor
    # 3. Line count
    #
    my $blkCurrentPar = shift;
    my $outVisitorPar = shift;
    my $iLineCountPar = shift;
    if ( $bLog == $TRUE ){ $theLogger->logMessage( "Block: Id=>" . $blkCurrentPar->getId() . "<" ); }

    #---- Set flushed
    $blkCurrentPar->setFlushed( $TRUE );

    #---- Fetch content definition of block
    my $contDefTmp = $blkCurrentPar->getContentDef();
    if ( lc( $contDefTmp->getType() ) ne "skip" ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Error! Skip block formatter can not process other type (" . $contDefTmp->getType() . ") of blocks<" ); }
        return -1;
    }

    # V101 Begin
    #---- Assert block has content or not ----#
    my $arefObjsListTmp = $blkCurrentPar->getObjects();
    my @arrObjsListTmp = @{ $arefObjsListTmp };

    #---- Get next block in chain
    my $blkNextTmp = $blkCurrentPar->getNextRef();
    if ( @arrObjsListTmp <= 0 ){
        # When current block (or chained block) is empty, ensure to continue with next block in
        # chain (if any exists) else return no error (so that formatting completed properly)
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Warning! Skipped writing empty block (" . $blkCurrentPar->getId() . ")" ); }

        #---- Write chained block contents ----#
        if ( $blkNextTmp != undef ){
            #---- Check and create formatter
            my $fmtNextTmp = $blkNextTmp->getFormatter();
            if ( $fmtNextTmp == undef ){
                $blkNextTmp->_createContentFormatter();
                $fmtNextTmp = $blkNextTmp->getFormatter();
            }

            #---- Write next block
            $iRetTmp = $fmtNextTmp->_writeFormattedContent( # Block
                                                              $blkNextTmp
                                                            # Visitor
                                                            , $outVisitorPar
                                                            # Line count
                                                            , $iLineCountPar
                                                          );
            if ( $iRetTmp < 0 ){ return $iRetTmp; }
        }
        return 0;
    }
    # V101 End

    #---- Detect first row Y position
    my $iCurrentYTmp = 0;
    foreach my $elemFirstTmp ( @arrObjsListTmp ){
        if (    $elemFirstTmp != undef
             && $outVisitorPar->isWritable( $elemFirstTmp ) == $TRUE
           ){
            $iCurrentYTmp = $elemFirstTmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_YPOS };
            last;
        }
    }

    my $iRetTmp   = -1;
    my $a2wObjTmp = undef;
    my $pomObjTmp = undef;
    my $iLineCountTmp  = $iLineCountPar;
    my $iLineGapTmp    = 0;
    my @arrLineObjsTmp = ();
    foreach my $elemObjTmp ( @arrObjsListTmp ){
        $a2wObjTmp = $elemObjTmp->{ 'A2WOBJ' };
        $pomObjTmp = $elemObjTmp->{ 'POMOBJ' };

        #---- Assert whether visitor supports this object type ----#
        if ( $outVisitorPar->isWritable( $elemObjTmp ) == $FALSE ){
            next;    # go to next object
        }

        #---- Write line, if collected ----#
        if ( $iCurrentYTmp != $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS } ){
            #---- Update row position
            $iLineGapTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS } - $iCurrentYTmp;
            $iCurrentYTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS };
            $iLineCountTmp++;

            #---- Iterate through objects and write them ----#
            foreach my $objTmp ( @arrLineObjsTmp ){
                $outVisitorPar->writeObject( $objTmp );
            }

            @arrLineObjsTmp = ();
        } # if ( $iCurrentYTmp != $a2wObjTmp->getYPos() )

        #---- Collect objects on current line
        @arrLineObjsTmp[ ( $#arrLineObjsTmp + 1 ) ] = $elemObjTmp;
    } # foreach my $a2wObjTmp ( @arrObjsListTmp )

    #---- Write last line, if exists ----#
    if ( @arrLineObjsTmp > 0 ){
        #---- Update row position
        $iLineGapTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS } - $iCurrentYTmp;
        $iCurrentYTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS };
        $iLineCountTmp++;

        #---- Iterate through objects and write them ----#
        foreach my $objTmp ( @arrLineObjsTmp ){
            $outVisitorPar->writeObject( $objTmp );
        }
    }

    #---- Write chained block contents ----#
    if ( $blkNextTmp != undef ){
        #---- Check and create formatter
        my $fmtNextTmp = $blkNextTmp->getFormatter();
        if ( $fmtNextTmp == undef ){
            $blkNextTmp->_createContentFormatter();
            $fmtNextTmp = $blkNextTmp->getFormatter();
        }

        # V101 Begin
        #---- Assert whether next block is of same type
        # a. If it is same, continue current block state on visitor
        # b. If it is NOT same, finalize current block state on visitor and let next block started properly
        if ( lc( $blkNextTmp->getContentDefType() ) eq "skip" ){
            #---- Continue next block writing
            $iRetTmp = $fmtNextTmp->_writeFormattedContent( # Block
                                                              $blkNextTmp
                                                            # Visitor
                                                            , $outVisitorPar
                                                            # Line count
                                                            , $iLineCountTmp
                                                          );
            $this->setFinalized( $fmtNextTmp->isFinalized() ); # Current block is also finalized, if block is finalized on next block
        }
        else {
            #---- Finalize current block
            $this->setFinalized( $TRUE );
            $iRetTmp = $outVisitorPar->endSkip( $blkCurrentPar );
            if ( $iRetTmp < 0 ){ return $iRetTmp; }

            #---- Write next block
            $iRetTmp = $blkNextTmp->write( $outVisitorPar );
        }
        # V101 End
        if ( $iRetTmp < 0 ){ return $iRetTmp; }
    }

    return 0;
}

#-----------------------------------------------------------------------
# Don't remove the following lines !!!
#-----------------------------------------------------------------------
1;
__END__
