#-------------------------------------------------------------------------------
#  a2w::core::dm::TableBlockFormatter.pm:
#  Data Mining Framework Table Block Formatter
#
#  Author  : AFP2web Team, Maas Holding GmbH
#
#  $V100   2014-05-08    Initial Release
#
#  $V101   2015-08-26    Issue:
#                        Table have gap in between certain rows (in the ‘CHECKS CLEARED AT A GLANCE’ section
#                        in the html, there are 2 blank rows between Check# 5377 and 5350, and also a blank
#                        row between Check#5378 and 5379)
#
#                        Reason:
#                        Table has empty rows, which visually look like a gap in between rows. Empty rows occur
#                        due to lines (start/end anchor or actual content of block).
#
#                        Fix:
#                        Ensured to skip lines when evaluating table row elements
#
#                        JiRa:
#                        AFP-279
#
#                        Fixed by, Date:
#                        Panneer, 26/08/2015
#
#  $V102   2015-10-07    Issue:
#                        Table cell having multi line value occur now in multiple rows
#
#                        Reason:
#                        Table rows are identified based on change in Y position, so each line of
#                        same cell is treated as different row.
#
#                        Fix:
#                        Take first column as primary and mark beginning of a new row whenever the
#                        first column value on line is not empty. i.e, Following lines of a cell must
#                        have empty first column value on line
#
#                        JiRa:
#                        AFP-224
#
#                        Fixed by, Date:
#                        Panneer, 07/10/2015
#
#  $V103   2015-11-06    Fixed minor bug in writing begin table body '<tbody>' element
#
#                        JiRa:
#                        AFP-320
#
#                        Fixed by, Date:
#                        Panneer, 06/11/2015
#
#  $V104   2015-12-11    Fixed minor bug in writing begin table body '<tbody>' element
#                        when table block have less than or equal to two rows
#
#                        JiRa:
#                        AFP-320
#
#                        Fixed by, Date:
#                        Panneer, 11/12/2015
#
#  $V105   2017-10-30    Extended with 'PrimaryColumns' attribute to determine multiline
#                        rows. i.e, when primary column cells are empty then row is considered
#                        as following line of current row
#
#  $V106   2018-08-10    Extended to write tagging info at document level instead of page level
#
#  $V107   2018-10-25    Fixed minor bug in processing empty chained blocks
#
#  $V108   2018-10-26    a. Optimized preparing list of block objects
#                        b. Extended to handle multi type chained blocks
#                        c. Extended with table caption support
#                        d. Fixed minor bug in processing last line of a table row, when table is chained
#                        e. AFP-756: Preprocessed page content to have lines with Y tolerance
#
#-------------------------------------------------------------------------------
package a2w::core::dm::TableBlockFormatter;

#-----------------------------------------------------------------------
# Include required modules
#-----------------------------------------------------------------------
use a2w::TypeConstants;
use a2w::core::log::Logger;
use a2w::core::dm::Constants;
use a2w::core::dm::BlockFormatter;

#-----------------------------------------------------------------------
# Inherit from base class
#-----------------------------------------------------------------------
our @ISA = qw( a2w::core::dm::BlockFormatter );

#-----------------------------------------------------------------------
# Constructor
#-----------------------------------------------------------------------
sub new{
    my $proto = shift;
    my $class = ref( $proto ) || $proto;

    #---- Define boolean values
    $TRUE  = $a2w::TypeConstants::TRUE;     # TRUE  boolean value
    $FALSE = $a2w::TypeConstants::FALSE;    # FALSE boolean value

    #---- Instantiate from base class to inherit base class attributes
    my $this = a2w::core::dm::BlockFormatter->new( @_ );

    #---- Last row of page (that is continued over pages)
    $this->{ 'PageLastRow' } = undef; # $V105 Change

    #---- Add this derived class specific attributes
    #$this->{ 'someattribute' } = 0;

    bless( $this, $class );

    #---- Get logger
    our $theLogger = a2w::core::log::Logger->getSingleton();
    our $bLog = $theLogger->isRegistered( __PACKAGE__ );

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "new()" );
    #}
    return $this;
}

#-----------------------------------------------------------------------
# Destructor
#-----------------------------------------------------------------------
sub DESTROY{
    my $this = shift;
}

#-----------------------------------------------------------------------
# Mutators
#-----------------------------------------------------------------------

#-----------------------------------------------------------------------
# Accessors
#-----------------------------------------------------------------------

#-----------------------------------------------------------------------
# Workers
#-----------------------------------------------------------------------
#-----------------------------------------------------------------------
# writeFormattedBlock
#
# Writes block content formatted as table rows/columns
#
# >=0, in case of successful writing
# < 0, in case of error
#
#-----------------------------------------------------------------------
sub writeFormattedBlock{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeFormattedBlock()" );
    }

    #---- Parameter
    #
    # 1. Block
    # 2. Visitor
    #
    my $blkCurrentPar = shift;
    my $outVisitorPar = shift;

    #---- Fetch content definition of block
    my $contDefTmp = $blkCurrentPar->getContentDef();
    if ( lc( $contDefTmp->getType() ) ne "table" ){
        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "Warning! Table block formatter can not process other type (" . $contDefTmp->getType() . ") of blocks<" );
        }
        return -1;
    }

    # V108 Begin
    #---- Assert block has content or not ----#
    my $arefObjsListTmp = $blkCurrentPar->getObjects();
    my @arrObjsListTmp = @{ $arefObjsListTmp };
    # V108 End

    #---- Get next block in chain
    my $blkNextTmp = $blkCurrentPar->getNextRef(); # V107 Change
    if ( @arrObjsListTmp <= 0 ){
        # V107 Begin
        # When current block (or chained block) is empty, ensure to continue with next block in
        # chain (if any exists) else return no error (so that formatting completed properly)
        $blkCurrentPar->setFlushed( $TRUE );
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Warning! Skipped writing empty block (" . $blkCurrentPar->getId() . ")" ); }

        #---- Write chained block contents ----#
        if ( $blkNextTmp != undef ){
            #---- Check and create formatter
            my $fmtNextTmp = $blkNextTmp->getFormatter();
            if ( $fmtNextTmp == undef ){
                $blkNextTmp->_createContentFormatter();
                $fmtNextTmp = $blkNextTmp->getFormatter();
            }

            #---- Write next block
            $iRetTmp = $fmtNextTmp->writeFormattedContent( # Block
                                                             $blkNextTmp
                                                           # Visitor
                                                           , $outVisitorPar
                                                           # Row count
                                                           , 0
                                                         );
            if ( $iRetTmp < 0 ){ return $iRetTmp; }
        }
        return 0;
        # V107 End
    }

    #---- Fetch columns X positions
    my $hrefColsTmp = $contDefTmp->getColumns();
    my @arrColsTmp = sort { $hrefColsTmp->{ $a } <=> $hrefColsTmp->{ $b } } keys( %{ $hrefColsTmp } );
    my @arrColXPosTmp = map { $hrefColsTmp->{ $_ } } @arrColsTmp;

    #---- Detect first row Y position
    my $iCurrentYTmp = 0;
    my $iFontHeightTmp = 0;
    foreach my $a2wFirstCellTmp ( @arrObjsListTmp ){
        if (    $a2wFirstCellTmp != undef
             && $outVisitorPar->isWritable( $a2wFirstCellTmp ) == $TRUE
             # $V101 Begin
             && $this->_isSupported( $a2wFirstCellTmp ) == $TRUE
             # $V101 End
           ){
            $iCurrentYTmp = $a2wFirstCellTmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_ADJ_YPOS }; # V108 Change
            my $a2wFontTmp = $a2wFirstCellTmp->{ 'A2WOBJ' }->getFont();
            if ( $a2wFontTmp != undef ){
                $iFontHeightTmp = $a2wFontTmp->getHeight();
                $iFontHeightTmp *= $a2wFirstCellTmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_RESOLUTION } / 72;
            }
            last;
        }
    }

    #---- Begin Table
    my $iRetTmp = $outVisitorPar->beginTable( $blkCurrentPar, \@arrColXPosTmp, ( $iCurrentYTmp - $iFontHeightTmp ) );
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    #---- Write header row, if defined in config
    if ( $contDefTmp->isHeaderPredefined() == $TRUE ){
        #---- Start header
        $iRetTmp = $outVisitorPar->beginTableHeader();
        if ( $iRetTmp < 0 ){ return $iRetTmp; }

        #---- Write a row ----#
        $iRetTmp = $outVisitorPar->beginRow( # Row count
                                               0
                                             # Page id where row started
                                             , $blkCurrentPar->getStartedOn() # V106 Change
                                           );
        if ( $iRetTmp < 0 ){ return $iRetTmp; }

        #---- Write columns
        my @arrHeaderTmp = @{ $contDefTmp->getHeader() };
        my @arrCellTmp = ();
        for ( my $i = 0; $i < @arrHeaderTmp; $i++ ){
            @arrCellTmp[ 0 ] = @arrHeaderTmp[ $i ];
            $iRetTmp = $outVisitorPar->writeCell( \@arrCellTmp, @arrHeaderTmp[ $i ], @arrColsTmp );
            if ( $iRetTmp < 0 ){ return $iRetTmp; }
        }

        $iRetTmp = $outVisitorPar->endRow( 0 );    # Row count
        if ( $iRetTmp < 0 ){ return $iRetTmp; }

        #---- End header
        $iRetTmp = $outVisitorPar->endTableHeader();
        if ( $iRetTmp < 0 ){ return $iRetTmp; }
    }

    #---- Write formatted content of block ----#
    $iRetTmp = $this->_writeFormattedContent( # Block
                                                $blkCurrentPar
                                              # Visitor
                                              , $outVisitorPar
                                              # Row count
                                              , 0
                                            );
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    #---- End body
    $iRetTmp = $outVisitorPar->endTableBody();
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    #---- Begin footer
    #$iRetTmp = $outVisitorPar->beginTableFooter();
    #if ( $iRetTmp < 0 ){ return $iRetTmp; }
    #....
    #---- End footer
    #$iRetTmp = $outVisitorPar->endTableFooter();
    #if ( $iRetTmp < 0 ){ return $iRetTmp; }

    #---- End Table
    if ( $this->isFinalized() == $FALSE ){ $iRetTmp = $outVisitorPar->endTable( $blkCurrentPar ); } # V108 Change
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    return 0;
}

#-----------------------------------------------------------------------
# _writeFormattedContent
#
# Writes block content formatted as table rows/columns (low level implementation)
#
# >=0, in case of successful writing
# < 0, in case of error
#
#-----------------------------------------------------------------------
sub _writeFormattedContent{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "_writeFormattedContent()" );
    }

    #---- Parameter
    #
    # 1. Block
    # 2. Visitor
    # 3. Row count
    #
    my $blkCurrentPar = shift;
    my $outVisitorPar = shift;
    my $iRowCountPar  = shift;
    if ( $bLog == $TRUE ){
        $theLogger->logMessage( "Block: Id=>" . $blkCurrentPar->getId() . "<" );
    }

    #---- Set flushed
    $blkCurrentPar->setFlushed( $TRUE );

    #---- Fetch content definition of block
    my $contDefTmp = $blkCurrentPar->getContentDef();
    if ( lc( $contDefTmp->getType() ) ne "table" ){
        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "Warning! Table block formatter can not process other type (" . $contDefTmp->getType() . ") of blocks" );
        }
        return -1;
    }

    #---- Fetch columns definition
    my $hrefColsTmp = $contDefTmp->getColumns();
    my @arrColsTmp = sort { $hrefColsTmp->{ $a } <=> $hrefColsTmp->{ $b } } keys( %{ $hrefColsTmp } );

    # V108 Begin
    #---- Assert block has content or not ----#
    my $arefObjsListTmp = $blkCurrentPar->getObjects();
    my @arrObjsListTmp = @{ $arefObjsListTmp };
    # V108 End

    #---- Get next block in chain
    my $blkNextTmp = $blkCurrentPar->getNextRef(); # V107 Change
    if ( @arrObjsListTmp <= 0 ){
        # V107 Begin
        # When current block (or chained block) is empty, ensure to continue with next block in
        # chain (if any exists) else return no error (so that formatting completed properly)
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Warning! Skipped writing empty block (" . $blkCurrentPar->getId() . ")" ); }

        #---- Write chained block contents ----#
        if ( $blkNextTmp != undef ){
            #---- Check and create formatter
            my $fmtNextTmp = $blkNextTmp->getFormatter();
            if ( $fmtNextTmp == undef ){
                $blkNextTmp->_createContentFormatter();
                $fmtNextTmp = $blkNextTmp->getFormatter();
            }

            #---- Write next block
            $iRetTmp = $fmtNextTmp->_writeFormattedContent( # Block
                                                              $blkNextTmp
                                                            # Visitor
                                                            , $outVisitorPar
                                                            # Row count
                                                            , $iRowCountPar
                                                          );
            if ( $iRetTmp < 0 ){ return $iRetTmp; }
        }
        return 0;
        # V107 End
    }

    #---- Detect first row Y position
    my $iCurrentYTmp = 0;
    my $iObjResolutionTmp = 0; # V108 Change
    foreach my $a2wFirstCellTmp ( @arrObjsListTmp ){
        if (    $a2wFirstCellTmp != undef
             && $outVisitorPar->isWritable( $a2wFirstCellTmp ) == $TRUE
             # $V101 Begin
             && $this->_isSupported( $a2wFirstCellTmp ) == $TRUE
             # $V101 End
           ){
            $iCurrentYTmp = $a2wFirstCellTmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_ADJ_YPOS }; # V108 Change
            $iObjResolutionTmp = $a2wFirstCellTmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_RESOLUTION }; # V108 Change
            last;
        }
    }

    # $V105 Begin
    my $iRetTmp = -1;
    my $a2wObjTmp = undef;
    my $pomObjTmp = undef;
    my $elemObjTmp = undef;

    my %hshLineTmp = map { $_, undef } @arrColsTmp;
    my $hrefLineTmp = \%hshLineTmp;

    my $iLineCountTmp = 0;
    my $iObjCountTmp = @arrObjsListTmp;
    my @arrTableLinesTmp = ();

    my $iLastObjectIndexTmp = -1;
    my $bLastLineProcessedTmp = $FALSE;

    #---- Process and collect table lines ----#
    for ( my $o = 0; $o < $iObjCountTmp; $o++ ){
        $elemObjTmp = $arrObjsListTmp[ $o ]; # Get object from list

        #---- Assert whether visitor supports this object type ----#
        if (    $outVisitorPar->isWritable( $elemObjTmp ) == $FALSE
             || $this->_isSupported( $elemObjTmp ) == $FALSE
           ){
            #---- Trick to process last line (assert for last object and extend the iteration to run exactly once)
            if ( $o == ( $iObjCountTmp - 1 ) && $bLastLineProcessedTmp == $FALSE ){
                $bLastLineProcessedTmp = $TRUE;
                $iCurrentYTmp = 0;
                $o = $iLastObjectIndexTmp - 1;
            }

            next; # go to next object
        }

        $iLastObjectIndexTmp = $o;
        $a2wObjTmp = $elemObjTmp->{ 'A2WOBJ' };
        $pomObjTmp = $elemObjTmp->{ 'POMOBJ' };
        #if ( $bLog == $TRUE && $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJTYPE } == $a2w::core::dm::Constants::OT_TEXT && $bLastLineProcessedTmp == $FALSE ){ $theLogger->logMessage( "Object $o:>" . $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_VALUE } . "<" ); }

        if ( $iCurrentYTmp != $pomObjTmp->{ $a2w::core::dm::Constants::AT_ADJ_YPOS } ){ # V108 Change
            $arrTableLinesTmp[ $iLineCountTmp ] = $hrefLineTmp;
            $iLineCountTmp++;
            $iCurrentYTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_ADJ_YPOS }; # V108 Change

            if ( $bLog == $TRUE ){
                my $sLineTmp = "|";
                foreach my $colId ( @arrColsTmp ){
                    my $sSeparatorTmp = '';
                    my @arrElementsTmp = @{ $hrefLineTmp->{ $colId } };
                    foreach my $elem ( @arrElementsTmp ){
                        if ( $elem->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJTYPE } != $a2w::core::dm::Constants::OT_TEXT ){ next; }
                        $sLineTmp .= $sSeparatorTmp . $elem->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_VALUE };
                        $sSeparatorTmp = ' ';
                    }
                    $sLineTmp .= "|";
                }
                $theLogger->logMessage( "Detected Line $iLineCountTmp: $sLineTmp" );
            }

            #---- Revert object list array index to start over collection of line objects from current object again
            if ( $bLastLineProcessedTmp == $FALSE ){ $o--; }

            #---- Reinitialize line
            my %hshLineTmp = map { $_, undef } @arrColsTmp;
            $hrefLineTmp = \%hshLineTmp;

            next;
        }

        #---- Detect to which column the current object belongs to ----#
        for ( my $i = @arrColsTmp - 1; $i >= 0; $i-- ){
            if ( $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS } >= $hrefColsTmp->{ @arrColsTmp[ $i ] } ){
                $hrefLineTmp->{ @arrColsTmp[ $i ] }[ @{ $hrefLineTmp->{ @arrColsTmp[ $i ] } } ] = $elemObjTmp;
                last; # break this loop
            }
        } # for ( my $i = @arrColsTmp - 1; $i >= 0; $i-- ){

        #---- Trick to process last line (assert for last object and extend the iteration to run exactly once)
        if ( $o == ( $iObjCountTmp - 1 ) && $bLastLineProcessedTmp == $FALSE ){
            $bLastLineProcessedTmp = $TRUE;
            $iCurrentYTmp = 0;
            $o = $iLastObjectIndexTmp - 1;
        }
    }

    #---- Process table lines and make rows out of them appropriately using following rules ----#
    # Rules:
    # 0. If first line is caption, then write first line as caption and continue with further lines as usual # V108 Change
    # 1. If first row is header, then collect header line count given lines for first row and write header row
    # 2. A line having valid value for all primary column cells means beginning of row
    # 3. Lines having custom rules on given cells will also be considered as separate row
    # 4. If current line is last line of table, then write current row
    # 5. A line having no value for all primary column cells means line is part of current row (row has multi line cell value)
    #
    my $tl = 0;
    my $tblLineTmp = undef;
    my $tblLineNrTmp = 0;
    my $tblLineCountTmp = @arrTableLinesTmp;

    # V108 Begin
    # 0. If first line is caption, then write first line as caption and continue with further lines as usual
    my $tlc = 0;
    my $iRowCountTmp = $iRowCountPar;
    if ( $contDefTmp->hasCaption() == $TRUE ){
        my $captionRowTmp = $arrTableLinesTmp[ $tl++ ]; # fetch first header line
        $tlc++;

        #---- Collect caption line objects
        my @arrCaptionObjsTmp = ();
        foreach my $colId ( @arrColsTmp ){
            my @arrElementsTmp = @{ $captionRowTmp->{ $colId } };
            push( @arrCaptionObjsTmp, @arrElementsTmp );
        } # foreach my $colId ( @arrColsTmp )

        #---- Write first line as caption
        if ( @arrCaptionObjsTmp > 0 ){ $outVisitorPar->writeTableCaption( \@arrCaptionObjsTmp ); }
    }
    # V108 End

    # 1. If first row is header, then collect header line count given lines for first row and write header row
    my $bFirstRowIsHeaderTmp = $contDefTmp->isFirstRowHeader();
    my @arrHeaderTmp = ( $contDefTmp->isHeaderPredefined() ) ? @{ $contDefTmp->getHeader() } : ();
    if ( $bFirstRowIsHeaderTmp == $TRUE && $iRowCountTmp == 0 && @arrHeaderTmp <= 0 ){
        $tblLineNrTmp = $tl + 1;
        my $hdrRowTmp = $arrTableLinesTmp[ $tl++ ]; # Fetch first header line
        $tlc++; # V108 Change
        my $iHeaderLineCountTmp = $contDefTmp->getHeaderLineCount();

        #---- Merge header lines
        if ( $iHeaderLineCountTmp > 1 ){
            if ( $bLog == $TRUE ){ $theLogger->logMessage( "Merging $iHeaderLineCountTmp lines for header row" ); }
            for ( $tl = $tlc; $tl < $iHeaderLineCountTmp; $tl++, $tlc++ ){ # V108 Change
                $tblLineTmp = $arrTableLinesTmp[ $tl ];
                foreach my $colId ( @arrColsTmp ){
                    my @arrElementsTmp = @{ $tblLineTmp->{ $colId } };
                    foreach my $elem ( @arrElementsTmp ){
                        $hdrRowTmp->{ $colId }[ @{ $hdrRowTmp->{ $colId } } ] = $elem;
                    }
                } # foreach my $colId ( @arrColsTmp )
            } # for ( $tl = 1; $tl < $iHeaderLineCountTmp; $tl++ )
        } # if ( $iHeaderLineCountTmp > 1 )

        #---- Fill in header array
        my $h = 0;
        my $sHdrLineTmp = "|";
        foreach my $cell ( @arrColsTmp ){
            my $sSeparatorTmp = '';
            my @arrCellObjsTmp = @{ $hdrRowTmp->{ $cell } };
            foreach my $a2wObjTmp ( @arrCellObjsTmp ){
                if ( $a2wObjTmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJTYPE } != $a2w::core::dm::Constants::OT_TEXT ){ next; }
                @arrHeaderTmp[ $h ] .= $sSeparatorTmp . $a2wObjTmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_VALUE };
                $sHdrLineTmp .= $sSeparatorTmp . $a2wObjTmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_VALUE };
                $sSeparatorTmp = ' ';
            }
            $h++;
            $sHdrLineTmp .= "|";
        }
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Header Line: $sHdrLineTmp" ); }

        #---- Write header row
        $iRetTmp = $outVisitorPar->beginTableHeader();
        if ( $iRetTmp < 0 ){ return $iRetTmp; }

        $iRetTmp = $this->_writeRow( # Visitor
                                       $outVisitorPar
                                     # Row
                                     , $hdrRowTmp
                                     # Row number
                                     , $iRowCountTmp
                                     # Column names
                                     , \@arrColsTmp
                                     # Column header
                                     , \@arrHeaderTmp
                                     # Column positions
                                     , $hrefColsTmp
                                   );
        if ( $iRetTmp < 0 ){ return $iRetTmp; }

        $iRetTmp = $outVisitorPar->endTableHeader();
        if ( $iRetTmp < 0 ){ return $iRetTmp; }
    } # if ( $bFirstRowIsHeaderTmp == $TRUE && $iRowCountTmp == 0 && @arrHeaderTmp <= 0 )

    #---- Get next block in chain
    my $blkNextTmp = $blkCurrentPar->getNextRef();

    #---- Declare current row and current line (hash of cells)
    my $hrefCurrentRowTmp = $this->{ 'PageLastRow' };
    $this->{ 'PageLastRow' } = undef; # Reset page last row

    my @arrLinePrimaryColsTmp = ();
    my @arrPrimaryColsTmp = @{ $contDefTmp->getPrimaryColumns() };

    #---- Process (header lines excluded) table lines
    for ( ; $tl < $tblLineCountTmp; $tl++, $tlc++ ){
        $tblLineNrTmp = $tl + 1;
        $tblLineTmp = $arrTableLinesTmp[ $tl ];

        if ( $bLog == $TRUE ){
            my $sLineTmp = "|";
            foreach my $colId ( @arrColsTmp ){
                my $sSeparatorTmp = '';
                my @arrElementsTmp = @{ $tblLineTmp->{ $colId } };
                foreach my $elem ( @arrElementsTmp ){
                    if ( $elem->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJTYPE } != $a2w::core::dm::Constants::OT_TEXT ){ next; }
                    $sLineTmp .= $sSeparatorTmp . $elem->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_VALUE };
                    $sSeparatorTmp = ' ';
                }
                $sLineTmp .= "|";
            }
            $theLogger->logMessage( "Line $tblLineNrTmp: $sLineTmp" );
        }

        # 2. A line having valid value for all primary column cells means beginning of row
        @arrLinePrimaryColsTmp = grep { $tblLineTmp->{ $_ } == undef } @arrPrimaryColsTmp;
        if ( @arrLinePrimaryColsTmp <= 0 ){ # line has all primary column cells
            if ( $hrefCurrentRowTmp == undef ){
                if ( $bLog == $TRUE ){ $theLogger->logMessage( "Line $tblLineNrTmp is new row but held from writing to collect following lines (if any exist)" ); }

                #---- Assign current line as new row
                $hrefCurrentRowTmp = $tblLineTmp;
            }
            else {
                if ( $bLog == $TRUE ){ $theLogger->logMessage( "Line $tblLineNrTmp is new row." ); }

                #---- Write row (collected so far)
                $iRowCountTmp++;
                if ( $bLog == $TRUE ){ $theLogger->logMessage( "Writing row $iRowCountTmp" ); }
                $iRetTmp = $this->_writeRow( # Visitor
                                               $outVisitorPar
                                             # Row
                                             , $hrefCurrentRowTmp
                                             # Row number
                                             , $iRowCountTmp
                                             # Column names
                                             , \@arrColsTmp
                                             # Column header
                                             , \@arrHeaderTmp
                                             # Column positions
                                             , $hrefColsTmp
                                           );
                if ( $iRetTmp < 0 ){ return $iRetTmp; }

                #---- Assign current line as new row
                $hrefCurrentRowTmp = $tblLineTmp;
            }
        }
        # 3. Lines having custom rules on given cells will also be considered as separate row
        elsif ( $contDefTmp->isNewRow( $tblLineTmp ) == $TRUE ){
            # V106 Begin
            if ( $bLog == $TRUE ){ $theLogger->logMessage( "Line $tblLineNrTmp is new row based on row additional rules but held from writing to collect following lines (if any exist)" ); }
            if ( $hrefCurrentRowTmp == undef ){
                #---- Assign current line as new row
                $hrefCurrentRowTmp = $tblLineTmp;
            }
            else {
                #---- Write row (collected so far)
                $iRowCountTmp++;
                if ( $bLog == $TRUE ){ $theLogger->logMessage( "Writing row $iRowCountTmp" ); }
                $iRetTmp = $this->_writeRow( # Visitor
                                               $outVisitorPar
                                             # Row
                                             , $hrefCurrentRowTmp
                                             # Row number
                                             , $iRowCountTmp
                                             # Column names
                                             , \@arrColsTmp
                                             # Column header
                                             , \@arrHeaderTmp
                                             # Column positions
                                             , $hrefColsTmp
                                           );
                if ( $iRetTmp < 0 ){ return $iRetTmp; }

                #---- Assign current line as new row
                $hrefCurrentRowTmp = $tblLineTmp;
            }
            # V106 End
        }
        # 4. If current line is last line of table, then write current row (after merging current line with current row)
        elsif ( $tblLineNrTmp == $tblLineCountTmp ){
            if ( $blkNextTmp == undef ){ # Ensures the line last of table
                if ( $bLog == $TRUE ){ $theLogger->logMessage( "Line $tblLineNrTmp is last line of current row due to page end" ); }

                #---- Merge last line with current row
                foreach my $colId ( @arrColsTmp ){
                    my @arrElementsTmp = @{ $tblLineTmp->{ $colId } };
                    foreach my $elem ( @arrElementsTmp ){
                        $hrefCurrentRowTmp->{ $colId }[ @{ $hrefCurrentRowTmp->{ $colId } } ] = $elem;
                    }
                }

                #---- Write row (collected so far)
                $iRowCountTmp++;
                if ( $bLog == $TRUE ){ $theLogger->logMessage( "Writing row $iRowCountTmp" ); }
                $iRetTmp = $this->_writeRow( # Visitor
                                               $outVisitorPar
                                             # Row
                                             , $hrefCurrentRowTmp
                                             # Row number
                                             , $iRowCountTmp
                                             # Column names
                                             , \@arrColsTmp
                                             # Column header
                                             , \@arrHeaderTmp
                                             # Column positions
                                             , $hrefColsTmp
                                           );
                if ( $iRetTmp < 0 ){ return $iRetTmp; }

                #---- Clean up current row
                $hrefCurrentRowTmp = undef;
            }
            else {
                # V108 Begin
                #---- Merge last line with current row
                foreach my $colId ( @arrColsTmp ){
                    my @arrElementsTmp = @{ $tblLineTmp->{ $colId } };
                    foreach my $elem ( @arrElementsTmp ){
                        $hrefCurrentRowTmp->{ $colId }[ @{ $hrefCurrentRowTmp->{ $colId } } ] = $elem;
                    }
                }
                # V108 End

                # Retain the current row of table until all lines are collected (where lines may run across chained blocks or across pages)
                $this->{ 'PageLastRow' } = $hrefCurrentRowTmp;
            }
        }
        # 5. A line having no value for all primary column cells means line is part of current row (row has multi line cell value)
        else {
            if ( $bLog == $TRUE ){ $theLogger->logMessage( "Line $tblLineNrTmp is following line of row " . ( $iRowCountTmp + 1 ) . ", appending line to row." ); }

            #---- Merge last line with current row
            foreach my $colId ( @arrColsTmp ){
                my @arrElementsTmp = @{ $tblLineTmp->{ $colId } };
                foreach my $elem ( @arrElementsTmp ){
                    $hrefCurrentRowTmp->{ $colId }[ @{ $hrefCurrentRowTmp->{ $colId } } ] = $elem;
                }
            }
        }
    }

    #---- Write row (collected so far)
    if ( $hrefCurrentRowTmp != undef ){
        $iRowCountTmp++;
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Writing row $iRowCountTmp" ); }
        $iRetTmp = $this->_writeRow( # Visitor
                                       $outVisitorPar
                                     # Row
                                     , $hrefCurrentRowTmp
                                     # Row number
                                     , $iRowCountTmp
                                     # Column names
                                     , \@arrColsTmp
                                     # Column header
                                     , \@arrHeaderTmp
                                     # Column positions
                                     , $hrefColsTmp
                                   );
        if ( $iRetTmp < 0 ){ return $iRetTmp; }

        #---- Clean up current row
        $hrefCurrentRowTmp = undef;
    }
    # $V105 End

    #---- Write chained block contents ----#
    if ( $blkNextTmp != undef ){
        #---- Check and create formatter
        my $fmtNextTmp = $blkNextTmp->getFormatter();
        if ( $fmtNextTmp == undef ){
            $blkNextTmp->_createContentFormatter();
            $fmtNextTmp = $blkNextTmp->getFormatter();
        }

        # V108 Begin
        #---- Assert whether next block is of same type
        # a. If it is same, continue current block state on visitor
        # b. If it is NOT same, finalize current block state on visitor and let next block started properly
        if ( lc( $blkNextTmp->getContentDefType() ) eq "table" ){
            #---- Continue next block writing
            $iRetTmp = $fmtNextTmp->_writeFormattedContent( # Block
                                                              $blkNextTmp
                                                            # Visitor
                                                            , $outVisitorPar
                                                            # Row count
                                                            , $iRowCountTmp
                                                          );
            $this->setFinalized( $fmtNextTmp->isFinalized() ); # Current block is also finalized, if block is finalized on next block
        }
        else {
            #---- Finalize current block
            $this->setFinalized( $TRUE );
            $iRetTmp = $outVisitorPar->endTable( $blkCurrentPar );
            if ( $iRetTmp < 0 ){ return $iRetTmp; }

            #---- Write next block
            $iRetTmp = $blkNextTmp->write( $outVisitorPar );
        }
        # V108 End
        if ( $iRetTmp < 0 ){ return $iRetTmp; }
    }

    return 0;
}

#-----------------------------------------------------------------------
# _getCellContentWidth
#
# Evaluates cell width based on contents and returns the same
#
# >=0, in case of successful evaluation
# < 0, in case of error
#
#-----------------------------------------------------------------------
sub _getCellContentWidth{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "_getCellContentWidth()" );
    }

    #---- Parameter
    #
    # 1. Cell start X
    # 2. Cell objects (Array reference)
    #
    my $iStartXPar = shift;
    my $arefObjectsPar = shift;

    #---- Fetch objects list
    my @arrObjsTmp = @{ $arefObjectsPar };

    #TODO: Evaluate cell width to determine column span of objects

    return 0;
}

# $V101 Begin
#-----------------------------------------------------------------------
# _isSupported
#
# Asserts whether given object is supported to be table content
#
# Returns
# TRUE means object can be table content
# FALSE means object can not be table content
#
#-----------------------------------------------------------------------
sub _isSupported{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "_isSupported()" );
    #}

    #---- Fetch parameter(s)
    #
    # 1. Object hash (having both kernel object and pom object)
    #
    my $hrefObjPar = shift;

    #---- Assert object
    my $a2wObjTmp = $hrefObjPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefObjPar->{ 'POMOBJ' };
    my $bRetTmp   = $TRUE;

    my $iObjTypeTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJTYPE };
    unless ( $iObjTypeTmp == $a2w::core::dm::Constants::OT_TEXT ){
        $bRetTmp = $FALSE;
    }

    return $bRetTmp;
}
# $V101 End

# $V102 Begin
#-----------------------------------------------------------------------
# _writeRow
#
# write row of table
#
# Returns
# >=0 in case of success
# <0 in case of error
#
#-----------------------------------------------------------------------
sub _writeRow{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "_writeRow()" );
    #}

    #---- Fetch parameter(s)
    #
    # 1. Visitor (a2w::core::visitor::Visitor object)
    # 2. Row cells (Hash Reference)
    # 3. Row number (integer)
    # 4. Column names (Array reference)
    # 4. Column header (Array reference)
    # 4. Column positions (Hash reference)
    #
    my $outVisitorPar   = shift;
    my $hrefRowPar      = shift;
    my $iRowNrPar       = shift;
    my $arefColNamesPar = shift;
    my $arefColHdrPar   = shift;
    my $hrefColPosPar   = shift;

    #---- Begin table body before first row
    my $iRetTmp = 0;
    if ( $iRowNrPar == 1 ){
        $iRetTmp = $outVisitorPar->beginTableBody();
        if ( $iRetTmp < 0 ){ return $iRetTmp; }
    }

    # V106 Begin
    #---- Determine the page where the row starts
    my @arrColsTmp = @{ $arefColNamesPar };
    my @arrColHdrTmp = @{ $arefColHdrPar };

    my @arrCellObjsTmp = ();
    my $arefCellObjsTmp = undef;
    my $iRowStartPageIdTmp = -1;
    for ( my $c = 0; $c < @arrColsTmp; $c++ ){
        $arefCellObjsTmp = $hrefRowPar->{ @arrColsTmp[ $c ] };
        if ( $arefCellObjsTmp == undef ){ next; }

        @arrCellObjsTmp = @{ $arefCellObjsTmp };
        if ( @arrCellObjsTmp <= 0 ){ next; }

        $iRowStartPageIdTmp = @arrCellObjsTmp[ 0 ]->{ 'POMOBJ' }{ $a2w::core::dm::Constants::AT_PAGE_ID };
        last; # break the loop after identifying the row start page id
    }
    # V106 End

    #---- Write a row ----#
    $iRetTmp = $outVisitorPar->beginRow( # Row number
                                           $iRowNrPar
                                         # Page id where row started
                                         , $iRowStartPageIdTmp # V106 Change
                                       );
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    #---- Write cells
    my $sColTmp = '';
    my $iCellWidthTmp = 0;
    for ( my $i = 0; $i < @arrColsTmp; $i++ ){
        $sColTmp = @arrColsTmp[ $i ];

        #---- Evaluate cell width
        #$iCellWidthTmp = $this->_getCellContentWidth( $hrefColPosPar->{ $sColTmp }, $hrefRowPar->{ $sColTmp } );

        #---- Write cell
        $iRetTmp = $outVisitorPar->writeCell( $hrefRowPar->{ $sColTmp }, @arrColHdrTmp[ $i ], $sColTmp );
        if ( $iRetTmp < 0 ){ return $iRetTmp; }

        $hrefRowPar->{ $sColTmp } = undef; # Reset cell value
    }

    #---- End a row
    $iRetTmp = $outVisitorPar->endRow( $iRowNrPar );
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    return 0;
}
# $V102 End

#-----------------------------------------------------------------------
# Don't remove the following lines !!!
#-----------------------------------------------------------------------
1;
__END__
