#-------------------------------------------------------------------------------
#  a2w::core::dm::TableContentDef.pm:
#  Data Mining Framework Table Content Definition
#
#  Author  : AFP2web Team, Maas Holding GmbH
#
#  $V100   2014-05-08    Initial Release
#
#  $V101   2017-10-30    Extended with 'PrimaryColumns' attribute to determine multiline
#                        rows. i.e, when primary column cells are empty then row is considered
#                        as following line of current row
#
#  $V102   2018-10-03    a. Extended with table caption support
#
#-------------------------------------------------------------------------------
package a2w::core::dm::TableContentDef;

#-----------------------------------------------------------------------
# Include required modules
#-----------------------------------------------------------------------
use a2w::TypeConstants;
use a2w::core::log::Logger;
use a2w::core::dm::Constants;
use a2w::core::dm::ContentDef;

#-----------------------------------------------------------------------
# Inherit from base class
#-----------------------------------------------------------------------
our @ISA = qw( a2w::core::dm::ContentDef );

#-----------------------------------------------------------------------
# Constructor
#-----------------------------------------------------------------------
sub new{
    my $proto = shift;
    my $class = ref( $proto ) || $proto;

    #---- Instantiate from base class to inherit base class attributes
    my $this = a2w::core::dm::ContentDef->new( @_ );

    #---- Add this derived class specific attributes
    # Header is complex value used appropriate to define the header of table and possible values are explained below
    #
    # <FALSE | 1stRow | 1stColumn> | <Array reference to column names>
    # Default is "1stRow"
    #
    $this->{ 'Header' } = undef;

    # Columns is hash of column id and column start X position, which in turn used to split up the cell values of table row
    $this->{ 'Columns' } = undef;

    # $V101 Begin
    # Header row line count
    #
    # Default is one line
    #
    $this->{ 'HeaderLineCount' } = 1;

    # PrimaryColumns is array reference having column ids. When primary column cells are empty then row is considered
    # as following line of current row
    #
    # Default: First column is primary
    #
    $this->{ 'PrimaryColumns' } = undef;

    # Row additional rules to determine whether current line of table is new row or not.
    #
    # NOTE: By default, new row is identified when all primary column cells have value
    #       These additional rules are applied along with above default rule.
    #
    # Even if one of the rule match, then the line is considered as new row
    #
    $this->{ 'RowAdditionalRules' } = undef;
    # $V101 End

    # $V102 Begin
    # Table caption flag
    #
    # When turned on, table formatting will use the first line of content as table caption
    # Default is off
    #
    $this->{ 'Caption' } = $FALSE;
    # $V102 End

    bless( $this, $class );

    #---- Set base class details
    $this->{ 'Type' } = 'table';

    #---- Define boolean values
    $TRUE  = $a2w::TypeConstants::TRUE;     # TRUE  boolean value
    $FALSE = $a2w::TypeConstants::FALSE;    # FALSE boolean value

    #---- Get logger
    our $theLogger = a2w::core::log::Logger->getSingleton();
    our $bLog = $theLogger->isRegistered( __PACKAGE__ );

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "new" );
    #}

    return $this;
}

#-----------------------------------------------------------------------
# Destructor
#-----------------------------------------------------------------------
sub DESTROY{
    my $this = shift;
}

#-----------------------------------------------------------------------
# Mutators
#-----------------------------------------------------------------------
sub setHeader{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setHeader()" );
    #}

    $this->{ 'Header' } = shift;
}

sub setColumns{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setColumns()" );
    #}

    $this->{ 'Columns' } = shift;
}

# $V101 Begin
sub setHeaderLineCount{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setHeaderLineCount()" );
    #}

    $this->{ 'HeaderLineCount' } = shift;
}

sub setPrimaryColumns{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setPrimaryColumns()" );
    #}

    $this->{ 'PrimaryColumns' } = shift;
}

sub setRowAdditionalRules{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setRowAdditionalRules()" );
    #}

    $this->{ 'RowAdditionalRules' } = shift;
}
# $V101 End

# $V102 Begin
sub setCaption{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setCaption()" );
    #}

    $this->{ 'Caption' } = shift;
}
# $V102 End

#-----------------------------------------------------------------------
# Accessors
#-----------------------------------------------------------------------
sub getHeader{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getHeader()" );
    #}

    return $this->{ 'Header' };
}

sub getColumns{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getColumns()" );
    #}

    return $this->{ 'Columns' };
}

# $V101 Begin
sub getHeaderLineCount{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getHeaderLineCount()" );
    #}

    return $this->{ 'HeaderLineCount' };
}

sub getPrimaryColumns{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getPrimaryColumns()" );
    #}

    return $this->{ 'PrimaryColumns' };
}

sub getRowAdditionalRules{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getRowAdditionalRules()" );
    #}

    return $this->{ 'RowAdditionalRules' };
}
# $V101 End

# $V102 Begin
sub hasCaption{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "hasCaption()" );
    #}

    return $this->{ 'Caption' };
}
# $V102 End

#-----------------------------------------------------------------------
# Workers
#-----------------------------------------------------------------------
sub isHeaderPredefined(){
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "isHeaderPredefined()" );
    #}

    return ( lc( ref( $this->{ 'Header' } ) ) eq "array" ) ? $TRUE : $FALSE;
}

sub isFirstRowHeader(){
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "isFirstRowHeader()" );
    #}

    return ( lc( $this->{ 'Header' } ) eq "1strow" ) ? $TRUE : $FALSE;
}

sub isFirstColumnHeader(){
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "isFirstColumnHeader()" );
    #}

    return ( lc( $this->{ 'Header' } ) eq "1stcolumn" ) ? $TRUE : $FALSE;
}

# $V101 Begin
# Evaluate default values for content definition attributes
sub evaluateDefaults(){
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "evaluateDefaults()" );
    #}

    #---- Evaluate header default
    if ( $this->{ 'Header' } == undef ){
        $this->{ 'Header' } = "1stRow";
    }
    
    #---- Evaluate primary columns default
    if ( $this->{ 'PrimaryColumns' } == undef ){
        #---- Get columns
        my $hrefColsTmp = $this->{ 'Columns' };

        #---- Sort columns based X position to find the first column
        my @arrColsTmp = sort { $hrefColsTmp->{ $a } <=> $hrefColsTmp->{ $b } } keys( %{ $hrefColsTmp } );

        #---- Set first column id as default for primary columns
        if ( @arrColsTmp > 0 ){
            $this->{ 'PrimaryColumns' } = [ $arrColsTmp[ 0 ] ];
        }
    }

    return 0;
}

# Assert whether current line of table is new row or not (based on additional rules)
sub isNewRow(){
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "isNewRow()" );
    #}

    #---- Get parameter
    # 1. Line of table (hash of table cells, where each cell will have list of objects)
    my $hrefTableLinePar = shift;

    #---- Assert rules
    if ( $this->{ 'RowAdditionalRules' } == undef ){ return $FALSE; }

    #---- Get rules
    my @arrRowAddRulesTmp = @{ $this->{ 'RowAdditionalRules' } };

    #---- Build cell values (to assert against rules)
    my $hrefCellsTmp = {};
    my @arrColsTmp = sort keys( %{ $hrefTableLinePar } );
    foreach my $cell ( @arrColsTmp ){
        my @arrCellObjsTmp = @{ $hrefTableLinePar->{ $cell } };
        foreach my $objCellTmp ( @arrCellObjsTmp ){
            if ( $objCellTmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJTYPE } != $a2w::core::dm::Constants::OT_TEXT ){ next; }
            $hrefCellsTmp->{ $cell } .= $objCellTmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_VALUE };
        }
    }

    #---- Assert cell values against additional rules
    my $reRuleTmp = undef;
    my $sCellValTmp = "";
    my $bNewRowTmp = $FALSE;
    my $bRuleResultTmp = $TRUE;
    my @arrRuleKeysTmp = ();
    foreach my $rule ( @arrRowAddRulesTmp ){
        $bRuleResultTmp = $TRUE;
        @arrRuleKeysTmp = sort keys( %{ $rule } );
        foreach my $colId ( @arrRuleKeysTmp ){
            $reRuleTmp = $rule->{ $colId };
            $sCellValTmp = $hrefCellsTmp->{ $colId };
            if ( $sCellValTmp =~ $reRuleTmp ){
                #if ( $bLog == $TRUE ){ $theLogger->logMessage( "$colId:>$sCellValTmp< Rule:>$reRuleTmp< Result:>Matched<" ); }
            }
            else {
                #if ( $bLog == $TRUE ){ $theLogger->logMessage( "$colId:>$sCellValTmp< Rule:>$reRuleTmp< Result:>Not Matched<" ); }
                $bRuleResultTmp = $FALSE;
                last;
            }
        }

        # If atleast once rule succeeds, return true
        $bNewRowTmp ||= $bRuleResultTmp;
        if ( $bNewRowTmp == $TRUE ){ return $TRUE; }
    }

    return $FALSE;
}
# $V101 End

#-----------------------------------------------------------------------
# Don't remove the following lines !!!
#-----------------------------------------------------------------------
1;
__END__
