#-------------------------------------------------------------------------------
#  a2w/core/ext/Document.pm:
#  Extension module for core's a2w::Document to transform document pages as AFP Overlay
#
#  Author  : Fa. Maas
#  Date    : 2018-04-20
#
#  $V100   2012-08-21    Initial Release
#
#  $V101   2012-08-24   a. Extended BMO/EMO name to have syntax "<Spool simple name><Page ID>"
#                       b. Extended to remove padded bytes to BMO structured field
#
#  $V102   2018-04-20    Updated to be compatible with latest scripting facility framework
#-------------------------------------------------------------------------------

#
# NOTE:
# Though the filename is a2w/core/ext/Document.pm, in actual it is 'a2w::Document' module extension
#

package a2w::Document;

use a2w::TypeConstants;

use a2w::afp::Reader;
use a2w::afp::OverlayWriter;

use a2w::core::log::Logger; # $V102 Change
use a2w::core::file::Utils;

#---- Define boolean values
$TRUE  = $a2w::TypeConstants::TRUE;    # TRUE  boolean value
$FALSE = $a2w::TypeConstants::FALSE;   # FALSE boolean value

#-----------------------------------------------------------------------
# Transform to AFP Overlay
#
# Parameters
# 1. Output file path
# 2. Output file extension
# 3. Overlay file extension
#
# Returns
# >= 0 in case of successful transformation to AFP Overlay
# <  0 in case of error
#
#-----------------------------------------------------------------------
sub _transformToAFPOverlay{
    $this = shift;

    #---- Get parameter
    #
    # 1. Output file path
    # 2. Output file extension
    # 3. Overlay file extension
    #
    my $sOutputFilePathPar = shift;
    my $sOutputExtensionPar = shift;
    my $sOverlayExtensionPar = shift;

    #---- Get logger
    $theLogger = a2w::core::log::Logger::getSingleton(); # $V102
    my $bLogTmp  = $theLogger->isRegistered( ref( $this ) );

    if ( $bLogTmp ){
        $theLogger->logFunctionName( ref( $this ), "_transformToAFPOverlay" );
        $theLogger->logMessage( "Output file path: " . $sOutputFilePathPar );
    }

    #---- Get output filename
    my $sOutputFilenameTmp = $sOutputFilePathPar . $this->getOutputFilename();
    if ( $bLogTmp ){ $theLogger->logMessage( "Output file name: " . $sOutputFilenameTmp ); }

    #---- Evaluate Overlay filename
    my $sOverlayFilenameTmp = $sOutputFilenameTmp;
    $sOverlayFilenameTmp =~ s/$sOutputExtensionPar$/$sOverlayExtensionPar/;
    if ( $bLogTmp ){ $theLogger->logMessage( "Overlay file name: " . $sOverlayFilenameTmp ); }

    #---- Create AFP overlay writer
    my $afpWriterTmp = new a2w::afp::OverlayWriter( $sOverlayFilenameTmp );

    # V101 Begin
    #---- Set document details on overlay writer (to name BMO/EMO SFIs)
    $afpWriterTmp->setSpoolName( $this->getName() );
    $afpWriterTmp->setPageID( $this->getId() );
    # V101 End

    #---- Initialize AFP overlay writer
    my ( $iRcTmp, $sMsgTmp ) = $afpWriterTmp->initialize();
    if ( $iRcTmp < 0 ){
        return ( $iRcTmp, $sMsgTmp );
    }

    #---- Create AFP reader
    my $afpReaderTmp = new a2w::afp::Reader( $sOutputFilenameTmp );

    #---- Initialize AFP reader
    ( $iRcTmp, $sMsgTmp ) = $afpReaderTmp->initialize();
    if ( $iRcTmp < 0 ){
        #---- Finalize AFP overlay writer
        $afpWriterTmp->finalize();

        return ( $iRcTmp, $sMsgTmp );
    }

    #---- Set writer reference on reader
    $afpReaderTmp->setWriter( $afpWriterTmp );

    #---- Set whether input structured fields have 5A startup or not
    $afpWriterTmp->set5A( $afpReaderTmp->is5A() );

    #---- Process AFP
    ( $iRcTmp, $sMsgTmp ) = $afpReaderTmp->read();
    if ( $iRcTmp < 0 ){
        #---- Finalize AFP overlay writer
        $afpWriterTmp->finalize();

        return ( $iRcTmp, $sMsgTmp );
    }

    #---- Finalize AFP overlay writer
    ( $iRcTmp, $sMsgTmp ) = $afpWriterTmp->finalize();
    if ( $iRcTmp < 0 ){
        #---- Finalize AFP reader
        $afpReaderTmp->finalize();

        return ( $iRcTmp, $sMsgTmp );
    }

    #---- Finalize AFP reader
    ( $iRcTmp, $sMsgTmp ) = $afpReaderTmp->finalize();
    if ( $iRcTmp < 0 ){
        return ( $iRcTmp, $sMsgTmp );
    }

    #---- Remove AFP output
    ( $iRcTmp, $sMsgTmp ) = a2w::core::file::Utils::deleteFile( $sOutputFilenameTmp );
    if ( $iRcTmp < 0 ){
        return ( $iRcTmp, $sMsgTmp );
    }

    return 0;
}

1;
__END__
