#-------------------------------------------------------------------------------
#  a2w/core/ext/Page.pm:
#  Extension module with additional processing functionalities
#
#  Author  : Fa. Maas
#
#  V102   2021-07-30    OXS-12636: Extended to get region content
#  V101   2021-04-22    OXS-12071: Extended with "isNonCoded" API
#  V100   2020-04-10    Initial Release
#
#-------------------------------------------------------------------------------

#
# NOTE:
# Though the filename is a2w/core/ext/Page.pm, in actual it is 'a2w::Page' module extension
#

package a2w::Page;

use a2w::Line;
use a2w::Page;
use a2w::ConfigConstants;
use a2w::TypeConstants;
use a2w::core::log::Logger;

#---- Declare constants
$TRUE  = $a2w::TypeConstants::TRUE;    # TRUE  boolean value
$FALSE = $a2w::TypeConstants::FALSE;   # FALSE boolean value

#-----------------------------------------------------------------------
# Workers
#-----------------------------------------------------------------------

# V101 Begin
#-----------------------------------------------------------------------
# isNonCoded
#
# Determines whether page is non coded or not (based on content)
#
# Rule: If page has only image content, then page is non coded
#
# NOTE: Works only for the PDF input
#
# Returns
# true, if page is non coded
# false, if page is coded
#-----------------------------------------------------------------------
sub isNonCoded{
    $this = shift;

    #---- Get logger
    $theLoggerLocal = a2w::core::log::Logger->getSingleton();
    my $bLogTmp = $theLoggerLocal->isRegistered( __PACKAGE__ );
    if ( $bLogTmp ){ $theLoggerLocal->logFunctionName( __PACKAGE__, "isNonCoded" ); }

    my $bNonCodedPageTmp = $FALSE;

    #---- Analyze page contents ----#
    #
    # Rule: If page has only one image, then page is non coded
    #
    #NOTE: Overlay/PSEG are not checked since input is PDF always
    if (    $this->getFirstLine()   == undef
         && $this->getFirstText()   == undef
         && $this->getFirstVector() == undef
         && $this->getFirstImage()  != undef
       ){
        $bNonCodedPageTmp = $TRUE;
    }

    if ( $bLog == $TRUE ){ $theLogger->logMessage( "Page " . $this->getParseId() . " is " . ( ( $bNonCodedPageTmp == $TRUE ) ? "non coded" : "coded" ) ); }

    return $bNonCodedPageTmp;

}
# V101 End

# V102 Begin
#-----------------------------------------------------------------------
# Get texts of a region
#-----------------------------------------------------------------------
sub getRegionTexts{
    $this = shift;

    #---- Get logger
    $theLoggerLocal = a2w::core::log::Logger->getSingleton();
    my $bLogTmp = $theLoggerLocal->isRegistered( __PACKAGE__ );
    if ( $bLogTmp ){ $theLoggerLocal->logFunctionName( __PACKAGE__, "getRegionTexts" ); }

    #---- Parameter
    # 1. Box (hash reference: {x, y, w, h})
    # 2. Page texts (array of objects)
    #
    my $hrefBoxPar = shift;
    my $arefPageTextsPar = undef;
    if ( @_ > 0 ){ $arefPageTextsPar = shift; }

    #---- Get page text content
    if ( $arefPageTextsPar == undef ){ $arefPageTextsPar = $this->_getContent( { 'text' => $TRUE } ); }
    my @arrPageTextsTmp = @{ $arefPageTextsPar };

    #---- Sort the page texts
    @arrPageTextsTmp = sort _sortObjs @arrPageTextsTmp;

    #---- Filter content that fall in given box
	my $xMinTmp = $hrefBoxPar->{ 'x' };
	my $yMinTmp = $hrefBoxPar->{ 'y' };
	my $xMaxTmp = $xMinTmp + $hrefBoxPar->{ 'w' };
	my $yMaxTmp = $yMinTmp + $hrefBoxPar->{ 'h' };
    my @arrRegionTextsTmp = grep {
		   $_->{ 'XPOS' } >= $xMinTmp
		&& $_->{ 'XPOS' } <= $xMaxTmp
		&& $_->{ 'YPOS' } >= $yMinTmp
		&& $_->{ 'YPOS' } <= $yMaxTmp
	} @arrPageTextsTmp;

    return ( @arrRegionTextsTmp > 0 ) ? \@arrRegionTextsTmp : undef;
}
# V102 End

#-----------------------------------------------------------------------
# Get page texts
#-----------------------------------------------------------------------
sub getTexts{
    $this = shift;
    
    #---- Get logger
    $theLoggerLocal = a2w::core::log::Logger->getSingleton();
    my $bLogTmp = $theLoggerLocal->isRegistered( __PACKAGE__ );
    if ( $bLogTmp ){ $theLoggerLocal->logFunctionName( __PACKAGE__, "getTexts" ); }

    my $arefPageTextsTmp = $this->_getContent( { 'text' => $TRUE } );
    if ( $bLogTmp ){ my $hrefLogTmp = { 'PageTexts' => $arefPageTextsTmp }; $theLoggerLocal->logHashMessage( $hrefLogTmp ); }
    return $arefPageTextsTmp;
}

#-----------------------------------------------------------------------
# Sort objects
#
# 1. At first sort the objects based on Y position
# 2. Then sort the objects based on X position
#
# Sorted object will have the content in natural reading order (Left to right and top to bottom)
#
#-----------------------------------------------------------------------
sub _sortObjs{
    $a->{ 'YPOS' } <=> $b->{ 'YPOS' } || $a->{ 'XPOS' } <=> $b->{ 'XPOS' }
}

#-----------------------------------------------------------------------
# Get page content as array of objects
#
# IMPORTANT NOTES:
# 1. Page objects are assumed to have absolute position always
#-----------------------------------------------------------------------
sub _getContent{
    $this = shift;
    
    #---- Get logger
    #$theLoggerLocal = a2w::core::log::Logger->getSingleton();
    #my $bLogTmp = $theLoggerLocal->isRegistered( __PACKAGE__ );
    #if ( $bLogTmp ){ $theLoggerLocal->logFunctionName( __PACKAGE__, "_getContent" ); }

    #---- Parameter
    #
    # 1. Hash specifying which objects should be collected
    #    {
    #        'text'      => $TRUE or $FALSE
    #        'line'      => $TRUE or $FALSE
    #        'image'     => $TRUE or $FALSE
    #        'container' => $TRUE or $FALSE
    #        'vector'    => $TRUE or $FALSE
    #    }
    my $hrefObjTypesPar = shift;

    my @arrPageObjsTmp = ();

    #---- Collect page texts
    if ( $hrefObjTypesPar->{ 'text' } == $TRUE ){
        #---- Fetch first text
        my $a2wTextTmp = $this->getFirstText();

        #---- Loop thru all the text objects
        while ( $a2wTextTmp != 0 ){
            my %hshBBoxTmp = %{ $a2wTextTmp->getBoundaryBox() };
            $arrPageObjsTmp[ @arrPageObjsTmp ] = {
                  'OBJ'     => $a2wTextTmp
                , 'TYPE'    => 'text'
                , 'XPOS'    => $a2wTextTmp->getXPos()
                , 'YPOS'    => $a2wTextTmp->getYPos()
                , 'TEXT'    => $a2wTextTmp->getText()
                , 'BBOX'    => \%hshBBoxTmp
            };
            $a2wTextTmp = $this->getNextText(); # get the next text
        } # while ( $a2wTextTmp != 0 )
    }

    #---- Collect page lines
    if ( $hrefObjTypesPar->{ 'line' } == $TRUE ){
        #---- Fetch first line
        my $a2wLineTmp = $this->getFirstLine();

        #---- Loop thru all the line objects
        while ( $a2wLineTmp != 0 ){
            my %hshBBoxTmp = %{ $a2wLineTmp->getBoundaryBox() };
            $arrPageObjsTmp[ @arrPageObjsTmp ] = {
                  'OBJ'  => $a2wLineTmp
                , 'TYPE' => 'line'
                , 'XPOS' => $a2wLineTmp->getXPos()
                , 'YPOS' => $a2wLineTmp->getYPos()
                , 'BBOX'    => \%hshBBoxTmp
            };
            $a2wLineTmp = $this->getNextLine(); # get the next line
        } # while ( $a2wLineTmp != 0 )
    }

    #---- Collect page images
    if ( $hrefObjTypesPar->{ 'image' } == $TRUE ){
        #---- Fetch first image
        my $a2wImageTmp = $this->getFirstImage();

        #---- Loop thru all the image objects
        while ( $a2wImageTmp != 0 ){
            my %hshBBoxTmp = %{ $a2wLineTmp->getBoundaryBox() };
            $arrPageObjsTmp[ @arrPageObjsTmp ] = {
                  'OBJ'  => $a2wImageTmp
                , 'TYPE' => 'image'
                , 'XPOS' => $a2wImageTmp->getXPos()
                , 'YPOS' => $a2wImageTmp->getYPos()
                , 'BBOX'    => \%hshBBoxTmp
            };
            $a2wImageTmp = $this->getNextImage(); # get the next image
        } # while ( $a2wImageTmp != 0 )
    }

    #---- Collect page container
    if ( $hrefObjTypesPar->{ 'container' } == $TRUE ){
        #---- Fetch first container
        my $a2wContainerTmp = $this->getFirstContainer();

        #---- Loop thru all the container objects
        while ( $a2wContainerTmp != 0 ){
            my %hshBBoxTmp = %{ $a2wLineTmp->getBoundaryBox() };
            $arrPageObjsTmp[ @arrPageObjsTmp ] = {
                  'OBJ'  => $a2wContainerTmp
                , 'TYPE' => 'container'
                , 'XPOS' => $a2wContainerTmp->getXPos()
                , 'YPOS' => $a2wContainerTmp->getYPos()
                , 'BBOX'    => \%hshBBoxTmp
            };
            $a2wContainerTmp = $this->getNextContainer(); # get the next container
        } # while ( $a2wContainerTmp != 0 )
    }

    #---- Collect page vector
    if ( $hrefObjTypesPar->{ 'vector' } == $TRUE ){
        #---- Fetch first vector
        my $a2wVectorTmp = $this->getFirstVector();

        #---- Loop thru all the vector objects
        while ( $a2wVectorTmp != 0 ){
            my %hshBBoxTmp = %{ $a2wLineTmp->getBoundaryBox() };
            $arrPageObjsTmp[ @arrPageObjsTmp ] = {
                  'OBJ'  => $a2wVectorTmp
                , 'TYPE' => 'vector'
                , 'XPOS' => $a2wVectorTmp->getXPos()
                , 'YPOS' => $a2wVectorTmp->getYPos()
                , 'BBOX'    => \%hshBBoxTmp
            };
            $a2wVectorTmp = $this->getNextVector(); # get the next vector
        } # while ( $a2wVectorTmp != 0 )
    }

    #if ( $bLogTmp ){ $theLoggerLocal->logHashMessage( { 'PageObjects' => @arrPageObjsTmp } ); }
    return \@arrPageObjsTmp;
}

#-----------------------------------------------------------------------
# Highlight texts
#-----------------------------------------------------------------------
sub highlightTexts{
    $this = shift;
    
    #---- Get logger
    $theLoggerLocal = a2w::core::log::Logger->getSingleton();
    my $bLogTmp = $theLoggerLocal->isRegistered( __PACKAGE__ );
    if ( $bLogTmp ){ $theLoggerLocal->logFunctionName( __PACKAGE__, "highlightTexts" ); }

    #---- Get list of texts to be highlighted
    #
    # 1. Config handle
    # 2. Array of texts
    #
    my $a2wConfigPar = shift;
    my $arefHLTextsPar = shift;
    my @arrHLTextsTmp = @{ $arefHLTextsPar };
    if ( @arrHLTextsTmp <= 0 ){
        if ( $bLogTmp ){ $theLoggerLocal->logMessage( "Warning! Empty text list to highlight in page: " . $this->getParseId() ); }
        return 0;
    }

    #---- Get text content of page
    my $arefTextsTmp = $this->getTexts();
    my @arrTextsTmp = @{ $arefTextsTmp };
    if ( @arrTextsTmp <= 0 ){
        if ( $bLogTmp ){ $theLoggerLocal->logMessage( "Warning! No text to highlight in page: " . $this->getParseId() ); }
        return 0;
    }

    #---- Sort the texts
    @arrTextsTmp = sort _sortObjs @arrTextsTmp;

    #---- Iterate through each line of texts and look for text to be highlighted
    my $iLnNrTmp = 0;
    my $iPgResTmp = $this->getResolution();
    my $fFactorTmp = $iPgResTmp / 2880;
    my $fTpAdjTmp = 1.3;
    my $iYTmp = $arrTextsTmp[ 0 ]->{ 'YPOS' };
    my $sTxtLine1Tmp = "";      # texts grouped with space
    my $sTxtLine2Tmp = "";      # texts grouped without space
    my $sTextTmp  = "";         # text
    my $hrefHLBBoxTmp = undef;  # highlight text boundary box
    my $sHLTextTmp = "";        # highlight text
    my @arrLineTmp = ();
    my $bPDFOutTmp = $FALSE;

    my $sOutFmtTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::OUTPUTFORMAT );
    if ( lc( $sOutFmtTmp ) =~ /^pdf/ ){
        $bPDFOutTmp = $TRUE;
        $fTpAdjTmp = 1.1;
    }
    for my $t ( @arrTextsTmp ){
        # If all texts are highlighted, then skip further processing
        if ( @arrHLTextsTmp <= 0 ){ last; }

        if ( $iYTmp != $t->{ 'YPOS' } ){
            $iLnNrTmp++;
            $sTxtLine1Tmp =~ s/\s+$//;

            # next line found, process current line text against the texts to be highlighted
            if ( $bLogTmp ){ $theLoggerLocal->logMessage( "Line ($iLnNrTmp) text:>$sTxtLine1Tmp<" ); }
            for ( my $i = 0; $i < @arrHLTextsTmp; $i++ ){
                $sHLTextTmp = $arrHLTextsTmp[ $i ];

                # Case 1: Highlight text is made of multiple text objects
                if ( $sTxtLine1Tmp =~ /$sHLTextTmp/ ){
                    # Evaluate text boundary box
                    $hrefHLBBoxTmp = $this->_evaluateMultiTextBBox( $sHLTextTmp, $sTxtLine1Tmp, $bPDFOutTmp, \@arrLineTmp );
                }
                # Case 2: Highlight text is made of single text object or multiple char objects or texts are split uneven
                elsif ( $sTxtLine2Tmp =~ /$sHLTextTmp/ ){
                    # Evaluate text boundary box
                    $hrefHLBBoxTmp = $this->_evaluateTextBBox( $sHLTextTmp, $sTxtLine2Tmp, $bPDFOutTmp, \@arrLineTmp );
                }
                # Case 3: Highlight text not found in current line
                else {
                    $sHLTextTmp = "";
                }

                # Line with highlight text found, ensure to highlight the part of line
                if ( $sHLTextTmp ne "" ){
                    if ( $bLogTmp ){ $theLoggerLocal->logMessage( "Highlighting text:>$sHLTextTmp<" ); }
                    if ( $bLogTmp ){ $theLoggerLocal->logMessage( "output format:>" . $sOutFmtTmp . "<" ); }
                    $arrHLTextsTmp[ $i ] = undef; # remove from list

                    my $boxWidthTmp     = ( $hrefHLBBoxTmp->{ 'right' } - $hrefHLBBoxTmp->{ 'left' } ) * $fFactorTmp;
                    my $boxHeightTmp    = ( $hrefHLBBoxTmp->{ 'bottom' } - $hrefHLBBoxTmp->{ 'top' } ) * $fFactorTmp;
                    my $boxXTmp         = $hrefHLBBoxTmp->{ 'left' };
                    my $boxYTmp         = $hrefHLBBoxTmp->{ 'top' } + ( $boxHeightTmp * $fTpAdjTmp );

                    # Draw box
                    my $boxHighlightTmp = new a2w::Line();
                    $boxHighlightTmp->setXPos( $boxXTmp );
                    $boxHighlightTmp->setYPos( $boxYTmp );
                    $boxHighlightTmp->setWidth( $boxWidthTmp );
                    $boxHighlightTmp->setLength( $boxHeightTmp );
                    $boxHighlightTmp->setColor( 0xFFFF00 );    # Yellow
                    $this->addLine( $boxHighlightTmp );
                    if ( $bLogTmp == $TRUE ){ $theLoggerLocal->logMessage( "box><\@(" . $boxXTmp . "," . $boxYTmp . "), W=" . $boxWidthTmp . ", H=" . $boxHeightTmp . ", C=0xFFFF00" ); }

                }
            } # for ( my $i = 0; $i < @arrHLTextsTmp; $i++ )
            @arrHLTextsTmp = grep { defined( $_ ) } @arrHLTextsTmp;

            # reset line info
            $iYTmp = $t->{ 'YPOS' };
            $sTextTmp = $t->{ 'OBJ' }->getText();
            $sTxtLine1Tmp = $sTextTmp . ' ';
            $sTxtLine2Tmp = $sTextTmp;
            @arrLineTmp = ( $t );
            next;
        } # if ( $iYTmp != $t->{ 'YPOS' } )

        #---- Collect line
        $arrLineTmp[ @arrLineTmp ] = $t;
        $sTextTmp = $t->{ 'OBJ' }->getText();
        $sTxtLine1Tmp .= $sTextTmp . ' ';
        $sTxtLine2Tmp .= $sTextTmp;
    } # for my $t ( @arrTextsTmp )

    # last line found, process last line text against the texts to be highlighted
    if ( @arrHLTextsTmp > 0 && @arrLineTmp > 0 ){
        $iLnNrTmp++;
        $sTxtLine1Tmp =~ s/\s+$//;
        if ( $bLogTmp ){ $theLoggerLocal->logMessage( "Line ($iLnNrTmp) text:>$sTxtLine1Tmp<" ); }
        for ( my $i = 0; $i < @arrHLTextsTmp; $i++ ){
            $sHLTextTmp = $arrHLTextsTmp[ $i ];

            # Case 1: Highlight text is made of multiple text objects
            if ( $sTxtLine1Tmp =~ /$sHLTextTmp/ ){
                # Evaluate text boundary box
            }
            # Case 2: Highlight text is made of single text object or multiple char objects or texts are split uneven
            elsif ( $sTxtLine2Tmp =~ /$sHLTextTmp/ ){
                # Evaluate text boundary box
            }
            # Case 3: Highlight text not found in current line
            else {
                $sHLTextTmp = undef;
            }

            # Line with highlight text found, ensure to highlight the part of line
            if ( $sHLTextTmp != undef ){
                if ( $bLogTmp ){ $theLoggerLocal->logMessage( "Highlighting text:>$sHLTextTmp<" ); }
                $arrHLTextsTmp[ $i ] = undef; # remove from list
            }
        }
        @arrHLTextsTmp = grep { defined( $_ ) } @arrHLTextsTmp;
    }

    return 0;
}

#-----------------------------------------------------------------------
# _evaluateMultiTextBBox
#
# Evaluate boundary box for highlight text (made of multiple texts)
#-----------------------------------------------------------------------
sub _evaluateMultiTextBBox{
    $this = shift;
    
    #---- Get logger
    $theLoggerLocal = a2w::core::log::Logger->getSingleton();
    my $bLogTmp = $theLoggerLocal->isRegistered( __PACKAGE__ );
    if ( $bLogTmp ){ $theLoggerLocal->logFunctionName( __PACKAGE__, "_evaluateMultiTextBBox" ); }

    #---- Get parameter(s)
    #
    # 1. Highlight text
    # 2. Line text
    # 3. Flag indicating whether output is PDF or not
    # 4. Line texts (array of objects)
    my $sHLTextPar = shift;
    my $sLineTextPar = shift;
    my $bPDFOutPar = shift;
    my $arefLineTextsPar = shift;

    #---- Get starting and ending index of the highlight text
    my $iStartIdxTmp = index( $sLineTextPar, $sHLTextPar );
    my $iEndIdxTmp = $iStartIdxTmp + length( $sHLTextPar );

    #---- Evaluate boundary box for the highlight text
    my %hshBBoxTmp = ();
    my $iTxtIdxTmp = 0;
    my $iSepCntTmp = 0;
    foreach my $t ( @{ $arefLineTextsPar } ){
        $iTxtIdxTmp += length( $t->{ 'TEXT' } );

        # Skip up to highlight text start
        if ( $iTxtIdxTmp < $iStartIdxTmp ){ next; }

        # Stop after processing highlight text
        if ( $iTxtIdxTmp > $iEndIdxTmp ){ last; }

        # Calculate the separator count
        $iSepCntTmp++;

        # Initialize boundary box
        if ( !defined( $hshBBoxTmp{ 'top' } ) ){ $hshBBoxTmp{ 'top' } = $t->{ 'BBOX' }{ 'top' }; }
        if ( !defined( $hshBBoxTmp{ 'left' } ) ){ $hshBBoxTmp{ 'left' } = $t->{ 'BBOX' }{ 'left' }; }
        if ( !defined( $hshBBoxTmp{ 'right' } ) ){ $hshBBoxTmp{ 'right' } = $t->{ 'BBOX' }{ 'right' }; }
        if ( !defined( $hshBBoxTmp{ 'bottom' } ) ){ $hshBBoxTmp{ 'bottom' } = $t->{ 'BBOX' }{ 'bottom' }; }

        # Update boundary box
        if ( $hshBBoxTmp{ 'top' } > $t->{ 'BBOX' }{ 'top' } ){ $hshBBoxTmp{ 'top' } = $t->{ 'BBOX' }{ 'top' }; }
        if ( $hshBBoxTmp{ 'left' } > $t->{ 'BBOX' }{ 'left' } ){ $hshBBoxTmp{ 'left' } = $t->{ 'BBOX' }{ 'left' }; }
        if ( $hshBBoxTmp{ 'right' } < $t->{ 'BBOX' }{ 'right' } ){ $hshBBoxTmp{ 'right' } = $t->{ 'BBOX' }{ 'right' }; }
        if ( $hshBBoxTmp{ 'bottom' } > $t->{ 'BBOX' }{ 'bottom' } ){ $hshBBoxTmp{ 'bottom' } = $t->{ 'BBOX' }{ 'bottom' }; }
    }
    $iSepCntTmp--;

    if ( $bLogTmp ){ $theLoggerLocal->logMessage( "HLText:>$sHLTextPar< SepCnt:>$iSepCntTmp<" ); }

    #---- Adjust width based on separator count
    my $iWidthTmp = $hshBBoxTmp{ 'right' } - $hshBBoxTmp{ 'left' };
    if ( $iSepCntTmp > 1 ){
        my $iTxtLenTmp = length( $sHLTextPar ) - $iSepCntTmp;
        my $iCharWidthTmp = $iWidthTmp / $iTxtLenTmp;
        $iWidthTmp += int( $iCharWidthTmp * ( $iSepCntTmp + 1 ) );
        $hshBBoxTmp{ 'right' } = $hshBBoxTmp{ 'left' } + $iWidthTmp;
    }
    elsif ( $iSepCntTmp == 0 ){
        my $iTxtLenTmp = length( $sHLTextPar );
        my $iCharWidthTmp = $iWidthTmp / $iTxtLenTmp;
        if ( $bPDFOutPar == $FALSE ){ $iWidthTmp -= int( $iCharWidthTmp ); } # less 1 char width
        $hshBBoxTmp{ 'right' } = $hshBBoxTmp{ 'left' } + $iWidthTmp;
    }

    if ( $bLogTmp ){ $theLoggerLocal->logHashMessage( \%hshBBoxTmp ); }

    return \%hshBBoxTmp;
}

#-----------------------------------------------------------------------
# _evaluateTextBBox
#
# Evaluate boundary box for highlight text (made of single text)
#-----------------------------------------------------------------------
sub _evaluateTextBBox{
    $this = shift;
    
    #---- Get logger
    $theLoggerLocal = a2w::core::log::Logger->getSingleton();
    my $bLogTmp = $theLoggerLocal->isRegistered( __PACKAGE__ );
    if ( $bLogTmp ){ $theLoggerLocal->logFunctionName( __PACKAGE__, "_evaluateTextBBox" ); }

    #---- Get parameter(s)
    #
    # 1. Highlight text
    # 2. Line text
    # 3. Flag indicating whether output is PDF or not
    # 4. Line texts (array of objects)
    my $sHLTextPar = shift;
    my $sLineTextPar = shift;
    my $bPDFOutPar = shift;
    my $arefLineTextsPar = shift;

    #---- Get starting and ending index of the highlight text
    my $iStartIdxTmp = index( $sLineTextPar, $sHLTextPar );
    my $iEndIdxTmp = $iStartIdxTmp + length( $sHLTextPar );

    #---- Evaluate boundary box for the highlight text
    my %hshBBoxTmp = ();
    my $iTxtIdxTmp = 0;
    foreach my $t ( @{ $arefLineTextsPar } ){
        $iTxtIdxTmp += length( $t->{ 'TEXT' } );

        # Skip up to highlight text start
        if ( $iTxtIdxTmp < $iStartIdxTmp ){ next; }

        # Stop after processing highlight text
        if ( $iTxtIdxTmp > $iEndIdxTmp ){ last; }

        # Initialize boundary box
        if ( !defined( $hshBBoxTmp{ 'top' } ) ){ $hshBBoxTmp{ 'top' } = $t->{ 'BBOX' }{ 'top' }; }
        if ( !defined( $hshBBoxTmp{ 'left' } ) ){ $hshBBoxTmp{ 'left' } = $t->{ 'BBOX' }{ 'left' }; }
        if ( !defined( $hshBBoxTmp{ 'right' } ) ){ $hshBBoxTmp{ 'right' } = $t->{ 'BBOX' }{ 'right' }; }
        if ( !defined( $hshBBoxTmp{ 'bottom' } ) ){ $hshBBoxTmp{ 'bottom' } = $t->{ 'BBOX' }{ 'bottom' }; }

        # Update boundary box
        if ( $hshBBoxTmp{ 'top' } > $t->{ 'BBOX' }{ 'top' } ){ $hshBBoxTmp{ 'top' } = $t->{ 'BBOX' }{ 'top' }; }
        if ( $hshBBoxTmp{ 'left' } > $t->{ 'BBOX' }{ 'left' } ){ $hshBBoxTmp{ 'left' } = $t->{ 'BBOX' }{ 'left' }; }
        if ( $hshBBoxTmp{ 'right' } < $t->{ 'BBOX' }{ 'right' } ){ $hshBBoxTmp{ 'right' } = $t->{ 'BBOX' }{ 'right' }; }
        if ( $hshBBoxTmp{ 'bottom' } > $t->{ 'BBOX' }{ 'bottom' } ){ $hshBBoxTmp{ 'bottom' } = $t->{ 'BBOX' }{ 'bottom' }; }
    }

    if ( $bLogTmp ){ $theLoggerLocal->logHashMessage( \%hshBBoxTmp ); }

    return \%hshBBoxTmp;
}

1;
__END__
