#-------------------------------------------------------------------------------
#  a2w::core::visitor::AccessibilityVisitor.pm:
#  PERL module to evaluate unique tag id for block contents and to classify block contents
#
#  Author  : AFP2web Team, Maas Holding GmbH
#
#  $V100   2017-10-12    Initial Release
#
#  $V101   2018-01-19    - Extended with "Illustration" block functions
#                          a. beginIllustration
#                          b. writeIllustrationLine
#                          c. endIllustration
#                          d. writeContainer
#                        - Extended with alternate text processing
# 
#  $V102   2018-01-26    - Extended to handle abbreviation/expansion of texts
#
#  $V103   2018-02-20    - Fixed minor bug in evaluating tag id for paragraph line objects
#
#  $V104   2018-04-05    - Fixed minor bug in find and replacing abbreviation with expansion
#                          when abbreviation string has regex quantifier characters
#
#  $V105   2018-05-17    - Fixed minor bug in grouping paragraph texts when they do not have
#                          sequential tag id (AFP-706)
#
#  $V106   2018-07-30    AFP-716: Extended with header level (Block->Info->headerLevel) processing
#
#  $V107   2018-08-09    AFP-735: Extended to handle tagging for reusable objects (with different tag info
#                        at each presentation of same object)
#
#  $V108   2018-08-10    Extended to write tagging info at document level instead of page level
#
#  $V109   2018-10-03    a. Extended to handle multi type chained blocks
#                        b. Extended with composite block
#
#  $V110   2018-11-29    a. AFP-771: Extended to group/tag the content of a Paragraph block under one <P> tag
#                        b. AFP-771: Extended with "List" block write functions
#                           - beginList
#                           - beginListItem
#                           - writeListLabel
#                           - writeListBody
#                           - endListItem
#                           - endList
#
#  $V111   2018-12-10    AFP-772: Extended to tag line objects under one <P> tag based on "TagLineAsParagraph" flag
#                        Added following functions
#                        - beginParagraphLine
#                        - endParagraphLine
#
#  $V112   2018-12-12    AFP-773: Fixed minor bug in evaluating tag id for paragraph objects
#
#  $V113   2019-01-12    AFP-780: Fixed minor bug in evaluating tag id for address block objects
#
#-------------------------------------------------------------------------------
package a2w::core::visitor::AccessibilityVisitor;

#-----------------------------------------------------------------------
# Include required modules
#-----------------------------------------------------------------------
use a2w::Font;
use a2w::Text;
use a2w::Line;
use a2w::Image;
use a2w::Vector;
use a2w::ContentBlock;
use a2w::ContentBlockConstants;

use a2w::TypeConstants;
use a2w::core::log::Logger;
use a2w::core::dm::Constants;
use a2w::core::visitor::Visitor;

#-----------------------------------------------------------------------
# Inherit from base class
#-----------------------------------------------------------------------
our @ISA = qw( a2w::core::visitor::Visitor );

#-----------------------------------------------------------------------
# Constructor
#-----------------------------------------------------------------------
sub new{
    my $proto = shift;
    my $class = ref( $proto ) || $proto;

    #---- Define boolean values
    $TRUE  = $a2w::TypeConstants::TRUE;     # TRUE  boolean value
    $FALSE = $a2w::TypeConstants::FALSE;    # FALSE boolean value

    #---- Instantiate from base class to inherit base class attributes
    my $this = a2w::core::visitor::Visitor->new( @_ );

    #---- Add this derived class specific attributes
    $this->{ 'TagId' }          = 0;      # Unique tag id (aka marked content id)
    $this->{ 'ObjectId' }       = 0;      # Input object sequence id
    $this->{ 'InParagraph' }    = $FALSE; # Flag indicating writing phase is in paragraph writing
    $this->{ 'InIllustration' } = $FALSE; # Flag indicating writing phase is in illustration writing # $V101 Change
    $this->{ 'InTableHeader' }  = $FALSE; # Flag indicating writing phase is in table header writing
    $this->{ 'InTableBody' }    = $FALSE; # Flag indicating writing phase is in table body writing
    $this->{ 'InTableFooter' }  = $FALSE; # Flag indicating writing phase is in table footer writing
    $this->{ 'InTableRow' }     = $FALSE; # Flag indicating writing phase is in table row writing
    $this->{ 'InTableCell' }    = $FALSE; # Flag indicating writing phase is in table cell writing
    $this->{ 'RowLastObject' }  = undef;  # Last object of current row (i.e, last object of last cell of a row)
	
    $this->{ 'BlockObjects' }      = undef; # Block objects array
    $this->{ 'BlockObjectsCount' } = 0;     # Block objects count
    $this->{ 'TableObjectTCId' }   = 0;     # Tag classification id of table object

    $this->{ 'ActualText' }     = '';       # Actual text for current line/cell of text # V102 Change
    $this->{ 'HeaderLevel' }    = 'H1';     # Active header level # V106 Change

    $this->{ 'RowPageId' }      = 0;        # Active row starts on given page id # V108 Change
    $this->{ 'ActiveTagPage' }  = 0;        # Actively tagged page               # V108 Change
    $this->{ 'DefFont' }        = 0;        # Default font                       # V109 Change

    $this->{ 'ActiveBlockIsEmpty' } = $FALSE; # Flag indicating whether active block is empty or not # V109 Change

    # V110 Begin
    # Paragraph context to keep tag classification info.
    # Hash reference filled with following fields
    # {
    #     'TagClassification' => {
    #           'Begin' => <Begin classification id>
    #         , 'Element' => <Element classification id>
    #         , 'End' => <End classification id>
    #     }
    # }
    $this->{ 'ParagraphContext' }   = undef;

    $this->{ 'ListBeginTCIds' }  = 0;       # Tag classification ids for begin list and begin list item tags
    $this->{ 'ListLastObject' }  = undef;   # Last object of current list item (could be the last object of label when body is empty or the last object of the body)
    # V110 End

    bless( $this, $class );

    #---- Initialize constants ----#

    #---- Get logger
    our $theLogger = a2w::core::log::Logger->getSingleton();
    our $bLog = $theLogger->isRegistered( __PACKAGE__ );

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "new()" ); }
    return $this;
}

#-----------------------------------------------------------------------
# Destructor
#-----------------------------------------------------------------------
sub DESTROY{
    my $this = shift;
}

#-----------------------------------------------------------------------
# Mutator
#-----------------------------------------------------------------------
sub setBackgroundFilename{
    my $this = shift;

    $this->{ 'BackgroundFilename' } = shift;
}

sub setUnitBase{
    my $this = shift;

    $this->{ 'UnitBase' } = shift;
    if ( lc( $this->{ 'UnitBase' } ) eq "pixel" ){
        $this->{ 'UnitBase' } = "px";
    }
    elsif ( lc( $this->{ 'UnitBase' } ) eq "mm" ){
        $this->{ 'UnitBase' } = "mm";
    }
    else {
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Warning! Invalid unit base (" . $this->{ 'UnitBase' } . ") using default px" ); }
        $this->{ 'UnitBase' } = "px";
    }
}

#-----------------------------------------------------------------------
# Accessor
#-----------------------------------------------------------------------
sub getBackgroundFilename{
    my $this = shift;

    return $this->{ 'BackgroundFilename' };
}

sub getUnitBase{
    my $this = shift;

    return $this->{ 'UnitBase' };
}

#-----------------------------------------------------------------------
# Workers
#-----------------------------------------------------------------------
#-----------------------------------------------------------------------
# Initialize
#-----------------------------------------------------------------------
sub initialize{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "initialize()" ); }

    #---- Force page output
    $this->{ 'PageOutput' } = $TRUE; # Turn on to get page level process dumping
    $this->{ 'SkipOutput' } = $TRUE; # Turn on to stop creating page level process dumping

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::initialize( @_ );
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    #---- Initialize attributes
    $this->{ 'TagId' } = 0;
    $this->{ 'ObjectId' } = 0;

    $this->{ 'ActiveTagPage' } = 0; # V108 Change
    $this->{ 'ActiveBlockIsEmpty' } = $FALSE; # V109 Change

    return 0;
}

#-----------------------------------------------------------------------
# Finalize
#-----------------------------------------------------------------------
sub finalize{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "finalize()" ); }

    #---- Fetch parameters
    #
    # 1. Additional content for body end (optional)
    # 2. Additional content for footer (optional)
    #
    my $sBodyAddContentPar   = "";
    my $sAddFooterContentPar = "";
    if ( @_ > 0 ){
        $sBodyAddContentPar = shift;
    }
    if ( @_ > 0 ){
        $sAddFooterContentPar = shift;
    }

    #---- Finalize attributes
    $this->{ 'TagId' } = 0;
    $this->{ 'ObjectId' } = 0;

    $this->{ 'ActiveTagPage' } = 0; # V108 Change
    $this->{ 'ActiveBlockIsEmpty' } = $FALSE; # V109 Change

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::finalize( @_ );
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    return 0;
}

#-----------------------------------------------------------------------
# Write header
#-----------------------------------------------------------------------
sub writeHeader{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "writeHeader()" ); }

    #---- Parameter(s)
    #
    # 1. Title
    # 2. Additional content
    #
    my $sTitlePar      = shift;
    my $sAddContentPar = shift;

    return 0;
}

#-----------------------------------------------------------------------
# Write footer
#-----------------------------------------------------------------------
sub writeFooter{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "writeFooter()" ); }

    #---- Parameter(s)
    #
    # 1. Additional content
    #
    my $sAddContentPar = shift;
    if ( $sAddContentPar eq "" ){ return 0; }

    return 0;
}

#-----------------------------------------------------------------------
# Initialize Page
#-----------------------------------------------------------------------
sub initializePage{
    my $this = shift;

    #---- Fetch parameters
    #
    # 1. Database
    # 2. Title
    # 3. Additional content for header (optional)
    # 4. Additional content for before body (optional)
    #
    my $dbPagePar = shift;
    my $sTitlePar = shift;
    my $sAddHeaderContentPar = "";
    my $sAddBodyContentPar   = "";
    if ( @_ > 0 ){
        $sAddHeaderContentPar = shift;
    }
    if ( @_ > 0 ){
        $sAddBodyContentPar = shift;
    }
    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "initializePage(" . $dbPagePar->getPageID() . ")" ); }

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::initializePage( $dbPagePar, $sAddHeaderContentPar );
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    $this->_printMessage( sprintf( "Begin page %04d", $dbPagePar->getPageID() ) );

    #---- Initialize attributes
    $this->{ 'TagId' } = 0;
    $this->{ 'ObjectId' }  = 0;
    $this->{ 'InParagraph' }    = $FALSE;
    $this->{ 'InIllustration' } = $FALSE; # $V101 Change
    $this->{ 'InTableHeader' }  = $FALSE;
    $this->{ 'InTableBody' }    = $FALSE;
    $this->{ 'InTableFooter' }  = $FALSE;
    $this->{ 'InTableRow' }     = $FALSE;
    $this->{ 'InTableCell' }    = $FALSE;
    $this->{ 'RowLastObject' }  = undef;

    $this->{ 'BlockObjects' }      = undef;
    $this->{ 'BlockObjectsCount' } = 0;
    $this->{ 'TableObjectTCId' }   = 0;

    $this->{ 'ActualText' }     = ''; # $V102 Change
    $this->{ 'HeaderLevel' }    = 'H1'; # V106 Change

    # V108 Begin
    $this->{ 'RowPageId' } = 0;
    $this->{ 'ActiveTagPage' } = $dbPagePar->getPageID();

    #---- Initialize active tag id in page, which is used to continue tagging objects
    #---- when the block start on current page and ends on different page
    $this->{ 'ActivePage' }{ 'TagId' } = 0;
    # V108 End

    $this->{ 'ActiveBlockIsEmpty' } = $FALSE; # V109 Change

    # V110 Begin
    $this->{ 'ParagraphContext' } = undef;
    $this->{ 'ListBeginTCIds' }   = 0;
    $this->{ 'ListLastObject' }   = undef;
    # V110 End

    return 0;
}

#-----------------------------------------------------------------------
# Finalize page
#-----------------------------------------------------------------------
sub finalizePage{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "finalizePage()" ); }

    #---- Fetch parameters
    #
    # 1. Additional content for body end (optional)
    # 2. Additional content for footer (optional)
    #
    my $sBodyAddContentPar   = "";
    my $sAddFooterContentPar = "";
    if ( @_ > 0 ){
        $sBodyAddContentPar = shift;
    }
    if ( @_ > 0 ){
        $sAddFooterContentPar = shift;
    }
    #if ( $bLog == $TRUE ){ $theLogger->logMessage( "Additional Content: Body=>" . $sBodyAddContentPar . "< Footer=>" . $sAddFooterContentPar . "<" ); }

    # ... processing statements ...
    $this->_printMessage( "End page" );

    #---- Finalize attributes
    $this->{ 'TagId' } = 0;
    $this->{ 'ObjectId' }  = 0;
    $this->{ 'InParagraph' }    = $FALSE;
    $this->{ 'InIllustration' } = $FALSE; # $V101 Change
    $this->{ 'InTableHeader' }  = $FALSE;
    $this->{ 'InTableBody' }    = $FALSE;
    $this->{ 'InTableFooter' }  = $FALSE;
    $this->{ 'InTableRow' }     = $FALSE;
    $this->{ 'InTableCell' }    = $FALSE;
    $this->{ 'RowLastObject' }  = undef;

    $this->{ 'BlockObjects' }      = undef;
	$this->{ 'BlockObjectsCount' } = 0;
    $this->{ 'TableObjectTCId' }   = 0;

    $this->{ 'ActualText' }     = ''; # $V102 Change
    $this->{ 'HeaderLevel' }    = 'H1'; # V106 Change

    $this->{ 'RowPageId' }     = 0; # V108 Change
    $this->{ 'ActiveTagPage' } = 0; # V108 Change

    $this->{ 'ActiveBlockIsEmpty' } = $FALSE; # V109 Change

    # V110 Begin
    $this->{ 'ParagraphContext' } = undef;
    $this->{ 'ListBeginTCIds' }   = 0;
    $this->{ 'ListLastObject' }   = undef;
    # V110 End

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::finalizePage( $sAddFooterContentPar );
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    return 0;
}

#-----------------------------------------------------------------------
# Get File Extension
#
# Returns output file extension based on visitor type. Base returns empty
# string and concrete visitors MUST return appropriate extension
#
#-----------------------------------------------------------------------
sub getFileExtension{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "getFileExtension()" ); }

    return ".txt";
}

#-----------------------------------------------------------------------
# Update resource table
#-----------------------------------------------------------------------
sub updateResourceTable{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "updateResourceTable()" ); }

    #---- Fetch parameters
    #
    # 1. Unique resource id
    # 2. Resource reference
    # 3. Object hash (having both kernel object and pom object)
    #
    my $iUniqueResIdPar = shift;
    my $a2wResourcePar  = shift;
    my $hrefObjectPar   = shift;

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::updateResourceTable( $iUniqueResIdPar, $a2wResourcePar, $hrefObjectPar );
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    return 0;
}

#-----------------------------------------------------------------------
# isWritable
#
# Asserts whether given object is writeable by visitor
#
# Returns
# TRUE is visitor can write the object
# FALSE is visitor can not write the object
#
#-----------------------------------------------------------------------
sub isWritable{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "isWritable()" ); }

    #---- Fetch parameter(s)
    #
    # 1. Object hash (having both kernel object and pom object)
    #
    my $hrefObjPar = shift;

    #---- Assert object
    my $a2wObjTmp = $hrefObjPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefObjPar->{ 'POMOBJ' };
    my $bRetTmp   = $TRUE;

    my $iObjTypeTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJTYPE };
    unless (    $iObjTypeTmp == $a2w::core::dm::Constants::OT_TEXT
             || $iObjTypeTmp == $a2w::core::dm::Constants::OT_IMAGE
             || $iObjTypeTmp == $a2w::core::dm::Constants::OT_CONTAINER # $V101 Change
           ){
        $bRetTmp = $FALSE;
    }

    return $bRetTmp;
}

#-----------------------------------------------------------------------
# Write raw content
#-----------------------------------------------------------------------
sub writeRawContent{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "writeRawContent()" ); }

    #---- Fetch parameter(s)
    #
    # 1. Raw content
    #
    my $sRawContentPar = shift;

    $this->{ 'Content' } .= "\n" . $sRawContentPar;

    return 0;
}

# Modified to write grouped text alone and separated writing table cell content
#
#-----------------------------------------------------------------------
# Write grouped text
#-----------------------------------------------------------------------
sub writeGroupedText{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "writeGroupedText()" ); }

    #---- Fetch parameter(s)
    #
    # 1. First Object hash (having both kernel object and pom object)
    # 2. Grouped Text
    # 3. Line gap (vertical difference between previous and current line, zero for first line)
    # 4. Array reference of grouped objects
    #
    my $hrefFirstObjPar       = shift;
    my $sGroupedTextPar       = shift;
    my $iLineGapPar           = shift;
    my $arefGroupedObjectsPar = shift;

    return 0;
}

#-----------------------------------------------------------------------
# Write image
#-----------------------------------------------------------------------
sub writeImage{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "writeImage()" ); }

    #---- Fetch parameter(s)
    #
    # 1. Image object hash (having both kernel object and pom object)
    #
    my $hrefImageObjPar = shift;

    #---- Get actual objects
    my $a2wObjTmp = $hrefImageObjPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefImageObjPar->{ 'POMOBJ' };

    #---- Skip, if skip mode is on
    if ( $this->{ 'SkipMode' } == $TRUE ){ return 0; }

    #---- Collect block objects
    $this->{ 'BlockObjects' }[ $this->{ 'BlockObjectsCount' }++ ] = $hrefImageObjPar;

    #---- Get image sequence id
    my $iObjIdTmp = $a2wObjTmp->getSequenceId();

    #---- Evaluate tag id
    if ( $this->{ 'ObjectId' } == 0 ){ # First object of block
        $this->{ 'ObjectId' } = $iObjIdTmp;
    }
    elsif ( $iObjIdTmp == ( $this->{ 'ObjectId' } + 1 ) ){ # Following object of block
        $this->{ 'ObjectId' } = $iObjIdTmp;
    }
    else { # Following object (with different sequence id) of block
        $this->{ 'ObjectId' } = $iObjIdTmp;
        #$this->{ 'TagId' } += 1; # V108 Change
        $this->_incrementTagId( $hrefImageObjPar ); # V108 Change
    }

    #---- Set tag id
    $a2wObjTmp->setTagId( $this->{ 'TagId' } );

    return 0;
}

# $V101 Begin
#-----------------------------------------------------------------------
# Write container
#-----------------------------------------------------------------------
sub writeContainer{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "writeContainer()" ); }

    #---- Fetch parameter(s)
    #
    # 1. Container object hash (having both kernel object and pom object)
    #
    my $hrefContainerObjPar = shift;

    #---- Get actual objects
    my $a2wObjTmp = $hrefContainerObjPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefContainerObjPar->{ 'POMOBJ' };

    #---- Skip, if skip mode is on
    if ( $this->{ 'SkipMode' } == $TRUE ){ return 0; }

    #---- Collect block objects
    $this->{ 'BlockObjects' }[ $this->{ 'BlockObjectsCount' }++ ] = $hrefContainerObjPar;

    #---- Get container sequence id
    my $iObjIdTmp = $a2wObjTmp->getSequenceId();

    #---- Evaluate tag id
    if ( $this->{ 'ObjectId' } == 0 ){ # First object of block
        $this->{ 'ObjectId' } = $iObjIdTmp;
    }
    elsif ( $iObjIdTmp == ( $this->{ 'ObjectId' } + 1 ) ){ # Following object of block
        $this->{ 'ObjectId' } = $iObjIdTmp;
    }
    else { # Following object (with different sequence id) of block
        $this->{ 'ObjectId' } = $iObjIdTmp;
        #$this->{ 'TagId' } += 1; # V108 Change
        $this->_incrementTagId( $hrefContainerObjPar ); # V108 Change
    }

    #---- Set tag id
    $a2wObjTmp->setTagId( $this->{ 'TagId' } );

    return 0;
}
# $V101 End

#-----------------------------------------------------------------------
# Write line
#-----------------------------------------------------------------------
sub writeLine{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "writeLine()" ); }

    #---- Fetch parameter(s)
    #
    # 1. Line object hash (having both kernel object and pom object)
    #
    my $hrefLineObjPar = shift;

    #---- Get actual objects
    my $a2wObjTmp = $hrefLineObjPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefLineObjPar->{ 'POMOBJ' };

    #---- Skip, if skip mode is on
    if ( $this->{ 'SkipMode' } == $TRUE ){ return 0; }

    #---- Collect block objects
    $this->{ 'BlockObjects' }[ $this->{ 'BlockObjectsCount' }++ ] = $hrefLineObjPar;

    return 0;
}

#-----------------------------------------------------------------------
# Write text
#-----------------------------------------------------------------------
sub writeText{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "writeText()" ); }

    #---- Fetch parameter(s)
    #
    # 1. Text object hash (having both kernel object and pom object)
    #
    my $hrefTextObjPar = shift;

    #---- Get actual objects
    my $a2wObjTmp = $hrefTextObjPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefTextObjPar->{ 'POMOBJ' };

    #---- Skip, if skip mode is on
    if ( $this->{ 'SkipMode' } == $TRUE ){ return 0; }

    #---- Collect texts of current line/cell
    my $sTextValTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_VALUE }; # V108 Change
    $this->{ 'ActualText' } .= $sTextValTmp . ' '; # V102 Change

    #---- Collect block objects
    $this->{ 'BlockObjects' }[ $this->{ 'BlockObjectsCount' }++ ] = $hrefTextObjPar;

    #---- Get text sequence id
    my $iObjIdTmp = $a2wObjTmp->getSequenceId();

    #---- Evaluate tag id
    if ( $this->{ 'ObjectId' } == 0 ){ # First object of block
        $this->{ 'ObjectId' } = $iObjIdTmp;
    }
    elsif ( $iObjIdTmp == ( $this->{ 'ObjectId' } + 1 ) ){ # Following object of block
        $this->{ 'ObjectId' } = $iObjIdTmp;
    }
    else { # Following object (with different sequence id) of block
        $this->{ 'ObjectId' } = $iObjIdTmp;
        #$this->{ 'TagId' } += 1; # V108 Change
        $this->_incrementTagId( $hrefTextObjPar ); # V108 Change
    }

    #---- Set tag id
    $a2wObjTmp->setTagId( $this->{ 'TagId' } );

    return 0;
}

#-----------------------------------------------------------------------
# Write vector
#-----------------------------------------------------------------------
sub writeVector{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "writeVector()" ); }

    #---- Fetch parameter(s)
    #
    # 1. Vector object hash (having both kernel object and pom object)
    #
    my $hrefVectorObjPar = shift;

    #---- Get actual objects
    my $a2wObjTmp = $hrefVectorObjPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefVectorObjPar->{ 'POMOBJ' };

    #---- Skip, if skip mode is on
    if ( $this->{ 'SkipMode' } == $TRUE ){ return 0; }

    #---- Collect block objects
    $this->{ 'BlockObjects' }[ $this->{ 'BlockObjectsCount' }++ ] = $hrefVectorObjPar;

    return 0;
}

#---- Block related writing methods ----#
#-----------------------------------------------------------------------
# Begin block
#-----------------------------------------------------------------------
sub beginBlock{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "beginBlock()" ); }

    #---- Fetch parameter(s)
    #
    # 1. Block
    # 2. Skip flag
    #
    my $blkToBeginPar = shift;
    my $bSkipPar = shift;

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::beginBlock( $blkToBeginPar, $bSkipPar );
    if ( $iRetTmp < 0 ){ return $iRetTmp; }
    if ( $bSkipPar == $TRUE ){ return 0; }

    #---- Skip page default blocks (that are not configured)
    my $sBlkIdTmp   = $blkToBeginPar->getId();
    my $reDefPfxTmp = qr/$a2w::core::dm::Constants::BLK_PAGE_DEF_PREFIX/;
    if ( $sBlkIdTmp =~ $reDefPfxTmp ){ $this->{ 'SkipMode' } = $TRUE; return 0; }

    $this->_printMessage( sprintf( "Begin block %s", $blkToBeginPar->getId() ) );

    #---- Initialize attributes
    $this->{ 'ObjectId' }  = 0;
    $this->{ 'InParagraph' }    = $FALSE;
    $this->{ 'InIllustration' } = $FALSE; # $V101 Change
    $this->{ 'InTableHeader' }  = $FALSE;
    $this->{ 'InTableBody' }    = $FALSE;
    $this->{ 'InTableFooter' }  = $FALSE;
    $this->{ 'InTableRow' }     = $FALSE;
    $this->{ 'InTableCell' }    = $FALSE;
    $this->{ 'RowLastObject' }  = undef;

    $this->{ 'BlockObjects' }      = undef;
	$this->{ 'BlockObjectsCount' } = 0;
    $this->{ 'TableObjectTCId' }   = 0;

    $this->{ 'ActualText' }     = ''; # $V102 Change

    $this->{ 'ActiveTagPage' }  = $this->{ 'ActivePage' }{ 'ID' }; # V108 Change
    $this->{ 'RowPageId' }      = 0;  # V108 Change

    # V110 Begin
    $this->{ 'ParagraphContext' } = undef;
    $this->{ 'ListBeginTCIds' }   = 0;
    $this->{ 'ListLastObject' }   = undef;
    # V110 End

    # V109 Begin
    $this->{ 'ActiveBlockIsEmpty' } = $FALSE;
    #---- Process empty sub block and add invisible text when prefix is set
    if ( $blkToBeginPar->isEmpty() == $TRUE ){
        $this->{ 'ActiveBlockIsEmpty' } = $TRUE;

        #---- Check whether block has prefix set on it
        my $sPrefixTmp = undef;
        my $sDefContentTmp = undef;
        my $blockInfoTmp = $blockInfoTmp = $blkToBeginPar->getJSONInfo();
        if ( $blockInfoTmp != undef ){
            if ( defined( $blockInfoTmp->{ 'Prefix' } ) ){ $sPrefixTmp = $blockInfoTmp->{ 'Prefix' }; }
            if ( defined( $blockInfoTmp->{ 'DefaultContent' } ) ){ $sDefContentTmp = $blockInfoTmp->{ 'DefaultContent' }; }
        }

        if (    $sDefContentTmp ne ""
             || $sPrefixTmp ne ""
           ){
            $this->_printMessage( sprintf( "Block %s is empty, but set with default content (%s) or prefix (%s). Extending block with empty text", $blkToBeginPar->getId(), $sDefContentTmp, $sPrefixTmp ) );

            #---- Get page handle (specific to current block start)
            my $dbPageTmp = $this->{ 'Pages' }{ $blkToBeginPar->getStartedOn() }{ 'POM' };
            my $a2wPageTmp = $dbPageTmp->getPageReference();
            if ( $a2wPageTmp != undef ){
                #---- Create invisible text object (whose text=prefix) and add it to page and block
                my $a2wTextTmp = new a2w::Text();
                if ( $a2wTextTmp != undef ){
                    my $iPageResTmp = $dbPageTmp->getPageRes();
                    my $iXPosTmp = ( $blkToBeginPar->getStartX() / $iPageResTmp ) * 240;
                    my $iYPosTmp = ( $blkToBeginPar->getStartY() / $iPageResTmp ) * 240;
                    my $iColorTmp = 0xFFFFFF; # Color value format: RRGGBB, 0xFFFFFF means white
                    $a2wTextTmp->setXPos( $iXPosTmp );
                    $a2wTextTmp->setYPos( $iYPosTmp );
                    $a2wTextTmp->setText( " " );
                    $a2wTextTmp->setColor( $iColorTmp );
                    if ( $this->{ 'DefFont' } == undef ){
                        $this->{ 'DefFont' } = new a2w::Font( $a2w::FontConstants::TYPE_TYPE1, "Arial" );
                        $this->{ 'DefFont' }->setHeight( 6 ); # 6 points
                    }
                    $a2wTextTmp->setFont( $this->{ 'DefFont' } );

                    #---- Add text to page
                    $a2wPageTmp->addText( $a2wTextTmp );
                    my $iObjSeqIdTmp = $dbPageTmp->addObject(   $a2w::core::dm::Constants::OT_TEXT
					                                          , $iXPosTmp
					                                          , $iYPosTmp
					                                          , 0 # angle
					                                          , $iColorTmp
					                                          , $a2wTextTmp
					                                        );

                    #---- Add text to block
                    my $pomObjTmp = $dbPageTmp->getPOMObject( $iObjSeqIdTmp );
                    $pomObjTmp->{ $a2w::core::dm::Constants::AT_PAGE_ID } = $dbPageTmp->getPageID(); # set page id on object
                    $blkToBeginPar->addObject( $a2wTextTmp, $pomObjTmp );
                } # if ( $a2wTextTmp != undef )
            } # if ( $a2wPageTmp != undef )
        } # if ( $sDefContentTmp ne "" || $sPrefixTmp ne "" )
    } # if ( $blkToBeginPar->isEmpty() == $TRUE )
    # V109 End

    return 0;
}

#-----------------------------------------------------------------------
# End block
#-----------------------------------------------------------------------
sub endBlock{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "endBlock()" ); }

    #---- Fetch parameter(s)
    #
    # 2. Skip flag
    #
    my $blkToEndPar = shift;
    my $bSkipPar = shift;

    #---- End skipping page default blocks (that are not configured)
    my $sBlkIdTmp   = $blkToEndPar->getId();
    my $reDefPfxTmp = qr/$a2w::core::dm::Constants::BLK_PAGE_DEF_PREFIX/;
    if ( $sBlkIdTmp =~ $reDefPfxTmp && $this->{ 'SkipMode' } == $TRUE ){ $this->{ 'SkipMode' } = $FALSE; return 0; }

    $this->_printMessage( sprintf( "End block %s", $blkToEndPar->getId() ) );

    #---- Finalize attributes
    $this->{ 'ObjectId' }  = 0;
    $this->{ 'InParagraph' }    = $FALSE;
    $this->{ 'InIllustration' } = $FALSE; # $V101 Change
    $this->{ 'InTableHeader' }  = $FALSE;
    $this->{ 'InTableBody' }    = $FALSE;
    $this->{ 'InTableFooter' }  = $FALSE;
    $this->{ 'InTableRow' }     = $FALSE;
    $this->{ 'InTableCell' }    = $FALSE;
    $this->{ 'RowLastObject' }  = undef;

    $this->{ 'BlockObjects' }      = undef;
	$this->{ 'BlockObjectsCount' } = 0;
    $this->{ 'TableObjectTCId' }   = 0;

    $this->{ 'ActualText' }     = ''; # $V102 Change

    $this->{ 'ActiveTagPage' }  = 0; # V108 Change
    $this->{ 'RowPageId' }      = 0; # V108 Change

    $this->{ 'ActiveBlockIsEmpty' } = $FALSE; # V109 Change

    # V110 Begin
    $this->{ 'ParagraphContext' } = undef;
    $this->{ 'ListBeginTCIds' }   = 0;
    $this->{ 'ListLastObject' }   = undef;
    # V110 End

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::endBlock( $blkToEndPar, $bSkipPar );

    return $iRetTmp;
}

#---- Paragraph related writing methods ----#
#-----------------------------------------------------------------------
# Begin paragraph
#-----------------------------------------------------------------------
sub beginParagraph{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "beginParagraph()" ); }

    #---- Fetch parameter(s)
    #
    # 1. Block
    #
    my $blkParagraphPar = shift;

    $this->{ 'InParagraph' } = $TRUE;
    $this->{ 'BlockObjects' } = [];

    # V110 Begin
    # Tag the content under one <P> tag
    #
    #---- Skip, if skip mode is on
    if ( $this->{ 'SkipMode' } == $TRUE ){ return 0; }

    #---- Evaluate tag classification id for paragraph objects
    my $sParaTypeTmp = "P";
    if (    $blkParagraphPar != undef
         && lc( $blkParagraphPar->getId() ) =~ /_label$/
       ){
        # $V106 Begin
        #$sParaTypeTmp = 'H1';
        $sParaTypeTmp = $this->{ 'HeaderLevel' };
        my $hrefBlockInfoTmp = $blkParagraphPar->getJSONInfo();
        if (    $hrefBlockInfoTmp != undef
             && $hrefBlockInfoTmp->{ 'headerLevel' } ne ""
           ){
            $sParaTypeTmp = $hrefBlockInfoTmp->{ 'headerLevel' };
            $this->{ 'HeaderLevel' } = $sParaTypeTmp;
        }
        # $V106 End
    }

    # Fill in paragraph context with tag classification info
    # {
    #     'TagClassification' => {
    #           'Begin' => <Begin classification id>
    #         , 'Element' => <Element classification id>
    #         , 'End' => <End classification id>
    #     }
    # }
    $this->{ 'ParagraphContext' } = {
        'TagClassification' => {
              'Begin' => $a2w::ContentBlockConstants::TAG_CLASSIFICATION_PARA_ELEMENT_P_BEGIN
            , 'Element' => $a2w::ContentBlockConstants::BLOCK_TYPE_PARAGRAPH | $a2w::ContentBlockConstants::TAG_CLASSIFICATION_PARA_ELEMENT_P
            , 'End' => $a2w::ContentBlockConstants::TAG_CLASSIFICATION_PARA_ELEMENT_P_END
        }
	};

    # $V106 Begin
    # Evaluate tag classification info based on configured header level
    if ( $sParaTypeTmp =~ /H([1-6])?/ ){
        my $sVarNameTmp = "TAG_CLASSIFICATION_PARA_ELEMENT_H" . $1 . "_BEGIN";
        $this->{ 'ParagraphContext' }{ 'TagClassification' }{ 'Begin' } = ${$a2w::ContentBlockConstants::{$sVarNameTmp}};

        $sVarNameTmp = "TAG_CLASSIFICATION_PARA_ELEMENT_H" . $1;
        $this->{ 'ParagraphContext' }{ 'TagClassification' }{ 'Element' } = $a2w::ContentBlockConstants::BLOCK_TYPE_PARAGRAPH | ${$a2w::ContentBlockConstants::{$sVarNameTmp}};

        $sVarNameTmp = "TAG_CLASSIFICATION_PARA_ELEMENT_H" . $1 . "_END";
        $this->{ 'ParagraphContext' }{ 'TagClassification' }{ 'End' } = ${$a2w::ContentBlockConstants::{$sVarNameTmp}};
    }
    # $V106 End

    # V109 Begin
    #---- Fetch and use block prefix as actual text start
    my $blockInfoTmp = undef;
    if ( $blkParagraphPar != undef ){ $blockInfoTmp = $blkParagraphPar->getJSONInfo(); }
    if ( $blockInfoTmp != undef ){
        # Block is empty
        if ( $this->{ 'ActiveBlockIsEmpty' } == $TRUE ){
            if ( defined( $blockInfoTmp->{ 'DefaultContent' } ) ){
                $this->{ 'ActualText' } = $blockInfoTmp->{ 'DefaultContent' };
            }
            elsif ( defined( $blockInfoTmp->{ 'Prefix' } ) ){
                $this->{ 'ActualText' } = $blockInfoTmp->{ 'Prefix' };
            }
        }
        # Block is non empty
        else {
            if ( defined( $blockInfoTmp->{ 'Prefix' } ) ){
                $this->{ 'ActualText' } = $blockInfoTmp->{ 'Prefix' } . ' ';
            }
        }
    }
    # V109 End
    # V110 End

    return 0;
}

# V111 Begin
#-----------------------------------------------------------------------
# Begin paragraph line
#-----------------------------------------------------------------------
sub beginParagraphLine{
    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "beginParagraphLine()" ); }

    #---- Fetch parameter(s)
    #
    # 1. Flag indicating whether line should be tagged as paragraph or not
    #
    my $bTagLineAsParagraphPar = shift;

    $this->{ 'ParagraphContext' }{ 'TagLineAsParagraph' } = $bTagLineAsParagraphPar;

    return 0;
}
# V111 End

#-----------------------------------------------------------------------
# Write pargraph line
#-----------------------------------------------------------------------
sub writeParagraphLine{
    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "writeParagraphLine()" ); }

    #---- Fetch parameter(s)
    #
    # 1. Array of line objects
    # 2. Line count
    # 3. Line gap (vertical difference between previous and current line, zero for first line)
    #
    my $arefObjectsPar = shift;
    my $iLineCountPar  = shift;
    my $iLineGapPar    = shift;

    #---- Skip, if skip mode is on
    if ( $this->{ 'SkipMode' } == $TRUE ){ return 0; }

    my @arrObjsTmp = @{ $arefObjectsPar };
    my $iObjsCountTmp = @arrObjsTmp;
    if ( $iObjsCountTmp <= 0 ){ return -1; }

    #---- Get page handle
    my $pgHandleTmp = $this->{ 'ActivePage' };

    #---- Get active block
    my $blkActiveTmp = $pgHandleTmp->{ 'ActiveBlock' }; # V109 Change

    # V110 Begin
    # Tag the content under one <P> tag
    #
    my $iDefTCIDFlagsTmp = $this->{ 'ParagraphContext' }{ 'TagClassification' }{ 'Element' };
    my @arrTCIdFlagsTmp = ($iDefTCIDFlagsTmp) x $iObjsCountTmp; # fill with paragraph default flags
    # V110 End

    # V111 Begin
    # Tag lines as paragraph
    if (    $this->{ 'ParagraphContext' }{ 'TagLineAsParagraph' } == $TRUE
	     && $this->{ 'ParagraphContext' }{ 'TagClassification' }{ 'Begin' } == $a2w::ContentBlockConstants::TAG_CLASSIFICATION_PARA_ELEMENT_P_BEGIN
       ){
        $arrTCIdFlagsTmp[ 0 ] = $this->{ 'ParagraphContext' }{ 'TagClassification' }{ 'Begin' };
        $arrTCIdFlagsTmp[ $#arrTCIdFlagsTmp ] |= $this->{ 'ParagraphContext' }{ 'TagClassification' }{ 'End' };
    }
    # V111 End

    #---- Write paragraph line objects
    my $iObjTCIdTmp = 0;
    my $a2wObjTmp   = undef;
    my $elemObjTmp  = undef;
    $this->{ 'ObjectId' } = 0; # V113 Change
    # V110 Change: Removed initializing actual text

    for ( my $plo = 0; $plo < $iObjsCountTmp; $plo++ ){
        $elemObjTmp = $arrObjsTmp[ $plo ];
        $a2wObjTmp = $elemObjTmp->{ 'A2WOBJ' };

        #---- Update tag classification id
        $iObjTCIdTmp = $a2wObjTmp->getTagClassificationId(); # Get existing classification id
        $iObjTCIdTmp |= $arrTCIdFlagsTmp[ $plo ];            # Update existing classification id
        $a2wObjTmp->setTagClassificationId( $iObjTCIdTmp );  # Set back updated classification id

        #---- Write object
        $this->writeObject( $elemObjTmp );
    }

    # V111 Begin
    # Tag lines as paragraph
    if (    $this->{ 'ParagraphContext' }{ 'TagLineAsParagraph' } == $TRUE
	     && $this->{ 'ParagraphContext' }{ 'TagClassification' }{ 'Begin' } == $a2w::ContentBlockConstants::TAG_CLASSIFICATION_PARA_ELEMENT_P_BEGIN
       ){
        $this->_incrementTagId( $arrObjsTmp[ $#arrObjsTmp ] );
    }
    # V111 End

    # V110 Change: Moved actual text/prefix/suffix processing to endParagraph function

    return 0;
}

# V111 Begin
#-----------------------------------------------------------------------
# End paragraph line
#-----------------------------------------------------------------------
sub endParagraphLine{
    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "endParagraphLine()" ); }

    $this->{ 'ParagraphContext' }{ 'TagLineAsParagraph' } = $FALSE;

    return 0;
}
# V111 End

#-----------------------------------------------------------------------
# End paragraph
#-----------------------------------------------------------------------
sub endParagraph{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "endParagraph()" ); }

    #---- Fetch parameter(s)
    #
    # 1. Block
    #
    my $blkParagraphPar = shift;

    # V110 Begin
    # Tag the content under one <P> tag
    #
    my @arrObjsTmp = @{ $this->{ 'BlockObjects' } };
    if ( @arrObjsTmp > 0 ){
        # V111 Begin
        # Update block's first object with proper begin tag
        my $iTCIdTmp = $arrObjsTmp[ 0 ]->{ 'A2WOBJ' }->getTagClassificationId();
        $iTCIdTmp |= $this->{ 'ParagraphContext' }{ 'TagClassification' }{ 'Begin' };
        $arrObjsTmp[ 0 ]->{ 'A2WOBJ' }->setTagClassificationId( $iTCIdTmp );
        # V111 End

        # Update block's last object with proper end tag
        $iTCIdTmp = $arrObjsTmp[ $#arrObjsTmp ]->{ 'A2WOBJ' }->getTagClassificationId();
        $iTCIdTmp |= $this->{ 'ParagraphContext' }{ 'TagClassification' }{ 'End' };
        $arrObjsTmp[ $#arrObjsTmp ]->{ 'A2WOBJ' }->setTagClassificationId( $iTCIdTmp );

        # V112 Begin
        # Increment tag only if the paragraph is not tagged per line
        my $paraContDefTmp = $blkParagraphPar->getContentDef();
        if (    $paraContDefTmp != undef
		     && $paraContDefTmp->getTagLineAsParagraph() == $FALSE
           ){
            $this->_incrementTagId( $arrObjsTmp[ $#arrObjsTmp ] );
        }
        # V112 End

        # V102 Begin
        #---- Find and replace all block abbreviation with expansion
        #
        $this->{ 'ActualText' } =~ s/\s$//; # Truncate space at the end

        # V109 Begin
        #---- Fetch and use block prefix as actual text start
        my $bUpdateActualTextTmp = $FALSE;
        my $blockInfoTmp = undef;
        if ( $blkParagraphPar != undef ){ $blockInfoTmp = $blkParagraphPar->getJSONInfo(); }
        if ( $blockInfoTmp != undef ){
            # Block is empty
            if ( $this->{ 'ActiveBlockIsEmpty' } == $TRUE ){
                if ( defined( $blockInfoTmp->{ 'DefaultContent' } ) ){
                    $bUpdateActualTextTmp = $TRUE;
                }
                elsif ( defined( $blockInfoTmp->{ 'Prefix' } ) ){
                    $bUpdateActualTextTmp = $TRUE;
                }
            }
            # Block is non empty
            else {
                if ( defined( $blockInfoTmp->{ 'Prefix' } ) ){
                    $bUpdateActualTextTmp = $TRUE;
                }
            }
        }
        # V109 End

        if (    $this->{ 'ActualText' } ne ""
             && $blkParagraphPar != undef
             && $arrObjsTmp[ 0 ] != undef
           ){
            #---- Get block json info
            if (    $blockInfoTmp != undef
                 && $blockInfoTmp->{ 'abbreviations' } != undef
               ){
                my $sExpnTmp = "";
                my $sActTextTmp = $this->{ 'ActualText' };
                my $hrefAbbrExpnTmp = $blockInfoTmp->{ 'abbreviations' };
                my @arrAbbrTmp = keys( %{ $hrefAbbrExpnTmp } );

                foreach my $sAbbrTmp ( @arrAbbrTmp ){
                    $sExpnTmp = $hrefAbbrExpnTmp->{ $sAbbrTmp };
                    $this->{ 'ActualText' } =~ s/\Q$sAbbrTmp\E/$sExpnTmp/g; # $V104 Change
                }

                #---- Set actual text on first object
                if ( $this->{ 'ActualText' } ne $sActTextTmp ){
                    $bUpdateActualTextTmp = $TRUE; # V109 Change
                }
            } # if ( $blockInfoTmp != undef && $blockInfoTmp->{ 'abbreviations' } != undef )
        } # if ( $blkActiveTmp != undef )

        # V109 Begin
        if ( $blockInfoTmp != undef ){
            # Check and process block suffix
            if ( defined( $blockInfoTmp->{ 'Suffix' } ) ){
                $this->{ 'ActualText' } .= ' ' . $blockInfoTmp->{ 'Suffix' };
                $bUpdateActualTextTmp = $TRUE;
            }
        }

        #---- Set actual text on first object
        if ( $bUpdateActualTextTmp == $TRUE ){ $arrObjsTmp[ 0 ]->{ 'A2WOBJ' }->setActualText( $this->{ 'ActualText' } ); }
        # V109 End

        $this->{ 'ActualText' } = ''; # Reset actual text
        # V102 End
    }
    # V110 End

    #---- Iterate through objects and display them
    $this->_logParagraph( $this->{ 'BlockObjects' } );

    #---- Process block objects and add them to appropriate pages
    if ( $this->{ 'BlockObjectsCount' } > 0 ){ $this->_processBlockObjects(); } # V109 Change

    $this->{ 'InParagraph' } = $FALSE;
    $this->{ 'ParagraphContext' } = undef; # V110 Change

    return 0;
}

#---- Table related writing methods ----#
#-----------------------------------------------------------------------
# Begin table
#-----------------------------------------------------------------------
sub beginTable{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "beginTable()" ); }

    #---- Fetch parameter(s)
    #
    # 1. Block
    # 2. Columns X positions array
    # 3. Starting Y position
    #
    my $blkTablePar        = shift;
    my $arefColumnsXPosPar = shift;
    my $iStartYPar         = shift;

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::beginTable( $blkTablePar, $arefColumnsXPosPar, $iStartYPar );
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    $this->{ 'TableObjectTCId' } = $a2w::ContentBlockConstants::TAG_CLASSIFICATION_TABLE_ELEMENT_TABLE_BEGIN;

    return 0;
}

#-----------------------------------------------------------------------
# Begin table header
#-----------------------------------------------------------------------
sub beginTableHeader{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "beginTableHeader()" ); }

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::beginTableHeader();
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    $this->{ 'InTableHeader' } = $TRUE;
    $this->{ 'TableObjectTCId' } |= $a2w::ContentBlockConstants::TAG_CLASSIFICATION_TABLE_ELEMENT_THEAD_BEGIN;

    return 0;
}

#-----------------------------------------------------------------------
# End table header
#-----------------------------------------------------------------------
sub endTableHeader{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "endTableHeader()" ); }

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::endTableHeader();
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    $this->{ 'InTableHeader' } = $FALSE;

    #---- Process last object and assign proper tag classification id
    if ( $this->{ 'RowLastObject' } != undef ){
        my $a2wObjTmp = $this->{ 'RowLastObject' }->{ 'A2WOBJ' };

	    #---- Get existing object tag classification id
	    my $iLastObjTCIdTmp = $a2wObjTmp->getTagClassificationId();
        $iLastObjTCIdTmp |= $a2w::ContentBlockConstants::TAG_CLASSIFICATION_TABLE_ELEMENT_THEAD_END;

        #---- set modified object tag classification id
        $a2wObjTmp->setTagClassificationId( $iLastObjTCIdTmp );
    }

    return 0;
}

#-----------------------------------------------------------------------
# Begin table body
#-----------------------------------------------------------------------
sub beginTableBody{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "beginTableBody()" ); }

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::beginTableBody();
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    $this->{ 'InTableBody' } = $TRUE;
    $this->{ 'TableObjectTCId' } |= $a2w::ContentBlockConstants::TAG_CLASSIFICATION_TABLE_ELEMENT_TBODY_BEGIN;

    return 0;
}

#-----------------------------------------------------------------------
# End table body
#-----------------------------------------------------------------------
sub endTableBody{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "endTableBody()" ); }

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::endTableBody();
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    $this->{ 'InTableBody' } = $FALSE;
    $this->{ 'TableObjectTCId' } |= $a2w::ContentBlockConstants::TAG_CLASSIFICATION_TABLE_ELEMENT_TBODY_END;

    return 0;
}

#-----------------------------------------------------------------------
# Begin table footer
#-----------------------------------------------------------------------
sub beginTableFooter{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "beginTableFooter()" ); }

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::beginTableFooter();
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    $this->{ 'InTableFooter' } = $TRUE;

    return 0;
}

#-----------------------------------------------------------------------
# End table footer
#-----------------------------------------------------------------------
sub endTableFooter{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "endTableFooter()" ); }

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::endTableFooter();
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    $this->{ 'InTableFooter' } = $FALSE;

    return 0;
}

#-----------------------------------------------------------------------
# Begin row
#-----------------------------------------------------------------------
sub beginRow{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "beginRow()" ); }

    #---- Fetch parameter(s)
    #
    # 1. Row number
    # 2. Page id (where row starts on) # V108 Change
    #
    my $iRowNrPar  = shift;
    my $iPageIdPar = shift; # V108 Change
    #if ( $bLog == $TRUE ){ $theLogger->logMessage( "Row Nr:>" . $iRowNrPar . "< RowStartPageId:>" . $iPageIdPar . "<" ); }

    $this->{ 'RowPageId' } = $iPageIdPar; # V108 Change
    $this->{ 'InTableRow' } = $TRUE;
    $this->{ 'RowLastObject' } = undef;

    if ( $iRowNrPar == 0 ){ # means TH
        $this->{ 'TableObjectTCId' } |= $a2w::ContentBlockConstants::TAG_CLASSIFICATION_TABLE_ELEMENT_TH_BEGIN;
    }
    else { # means TR
        $this->{ 'TableObjectTCId' } |= $a2w::ContentBlockConstants::TAG_CLASSIFICATION_TABLE_ELEMENT_TR_BEGIN;
    }

    return 0;
}

#-----------------------------------------------------------------------
# Write cell
#-----------------------------------------------------------------------
sub writeCell{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "writeCell()" ); }

    #---- Fetch parameter(s)
    #
    # 1. Array of cell objects
    # 2. Column header name
    # 3. Column id (as in on block definition)
    #
    my $arefObjectsPar = shift;
    my $sColHdrNamePar = shift;
    my $sColIdPar      = shift;
    #if ( $bLog == $TRUE ){ $theLogger->logMessage( "Header Name:>" . $sColHdrNamePar . "< ColumnId:>" . $sColIdPar . "<" ); }

    #---- Fetch objects list
    my @arrObjsTmp = @{ $arefObjectsPar };

    #---- Handle empty cell (create a dummy line object and use it to pass in empty cell info to core)
    my $bEmptyCellTmp = $FALSE;
    if ( @arrObjsTmp <= 0 ){
        $bEmptyCellTmp = $TRUE;

        #---- Get page handle (specific to current row start)
        my $dbPageTmp = $this->{ 'Pages' }{ $this->{ 'RowPageId' } }{ 'POM' }; # V108 Change
        my $a2wPageTmp = $dbPageTmp->getPageReference();

        #---- Create line
        my $a2wLineTmp = new a2w::Line();
        $a2wLineTmp->setXPos( 0 );
        $a2wLineTmp->setYPos( 0 );
        $a2wLineTmp->setWidth( 1 );
        $a2wLineTmp->setLength( 1 );
        $a2wLineTmp->setColor( 0xFFFFFF );   # White
        $a2wLineTmp->remove();               # Don't present it on output
        $a2wPageTmp->addLine( $a2wLineTmp ); # Add it to page, so it gets deleted later in core

        #---- Fill in cell objects array with dummy line
        @arrObjsTmp = ( {
              'A2WOBJ' => $a2wLineTmp
            , 'POMOBJ' => {
                  $a2w::core::dm::Constants::AT_OBJTYPE => $a2w::core::dm::Constants::OT_LINE
                , $a2w::core::dm::Constants::AT_PAGE_ID => $dbPageTmp->getPageID() # V108 Change
              }
        } );
    }
    my $iObjsCountTmp = @arrObjsTmp;

    $this->{ 'ObjectId' } = 0;
    $this->{ 'InTableCell' } = $TRUE;
    $this->{ 'ActualText' } = ''; # V102 Change

    my $bIncrTagIdOfCurrentPageTmp = $FALSE; # V108 Change

    #---- Assert and Write cell objects
    if ( $iObjsCountTmp > 0 ){
        #---- Initialize tag id based on the first object's page
        $this->_setTagId( $arrObjsTmp[ 0 ] ); # V108 change

        #---- Fill in tag classification id for cell objects
        my $iDefTCIDFlagsTmp = $a2w::ContentBlockConstants::BLOCK_TYPE_TABLE;
        if ( ( $this->{ 'TableObjectTCId' } & $a2w::ContentBlockConstants::TAG_CLASSIFICATION_TABLE_ELEMENT_TH_BEGIN ) != 0 ){
            $iDefTCIDFlagsTmp |= $a2w::ContentBlockConstants::TAG_CLASSIFICATION_TABLE_ELEMENT_TH;
        }
        else {
            $iDefTCIDFlagsTmp |= $a2w::ContentBlockConstants::TAG_CLASSIFICATION_TABLE_ELEMENT_TD;
        }
        my @arrTCIdFlagsTmp = ($iDefTCIDFlagsTmp) x $iObjsCountTmp; # fill with paragraph default flags
        if ( $this->{ 'TableObjectTCId' } > 0 ){ $arrTCIdFlagsTmp[ 0 ] = $this->{ 'TableObjectTCId' }; }
        $arrTCIdFlagsTmp[ 0 ] |= $a2w::ContentBlockConstants::TAG_CLASSIFICATION_TABLE_ELEMENT_TD_BEGIN;
        $arrTCIdFlagsTmp[ ( $iObjsCountTmp - 1 ) ] |= $a2w::ContentBlockConstants::TAG_CLASSIFICATION_TABLE_ELEMENT_TD_END;

        #---- Get last object of row
        $this->{ 'RowLastObject' } = $arrObjsTmp[ ( $iObjsCountTmp - 1 ) ];

        #---- Write cell objects
        my $iObjTCIdTmp = 0;
        my $a2wObjTmp   = undef;
        my $pomObjTmp   = undef; # V108 Change
        my $prvElemTmp  = undef; # V108 Change
        my $elemObjTmp  = undef;
        for ( my $tco = 0; $tco < $iObjsCountTmp; $tco++ ){
            $elemObjTmp = $arrObjsTmp[ $tco ];
            $a2wObjTmp  = $elemObjTmp->{ 'A2WOBJ' };
            $pomObjTmp  = $elemObjTmp->{ 'POMOBJ' };

            # V108 Begin
            # Reset tag id if object does not belong to active page
			#
			# Do only when object page is different from active page
            if ( $prvElemTmp != undef && $pomObjTmp->{ $a2w::core::dm::Constants::AT_PAGE_ID } != $prvElemTmp->{ 'POMOBJ' }{ $a2w::core::dm::Constants::AT_PAGE_ID } ){
                $bIncrTagIdOfCurrentPageTmp = $TRUE;
                $this->{ 'ObjectId' } = 0;
                $this->_setTagId( $elemObjTmp );
            }
            # V108 End

            #---- Update tag classification id
            $iObjTCIdTmp = $a2wObjTmp->getTagClassificationId(); # Get existing classification id
            $iObjTCIdTmp |= $arrTCIdFlagsTmp[ $tco ];            # Update existing classification id
            $a2wObjTmp->setTagClassificationId( $iObjTCIdTmp );  # Set back updated classification id

            #---- Write object
            $this->writeObject( $elemObjTmp );
            $prvElemTmp = $elemObjTmp; # V108 Change
        }
    }

    # V102 Begin
    #---- Find and replace all block abbreviation with expansion
    #
    $this->{ 'ActualText' } =~ s/\s$//; # Truncate space at the end
    if ( $bLog == $TRUE ){ $theLogger->logMessage( "writeCell:>" . $this->{ 'ActualText' } . "<" ); }

    #---- Get active page context
    my $pgHandleTmp = $this->{ 'ActivePage' };

    #---- Get active block
    my $blkActiveTmp = $pgHandleTmp->{ 'ActiveBlock' };
    if (    $this->{ 'ActualText' } ne ""
         && $blkActiveTmp != undef
         && $arrObjsTmp[ 0 ] != undef
       ){
        #---- Get block json info
        my $hrefJSONInfoTmp = $blkActiveTmp->getJSONInfo();
        if (    $hrefJSONInfoTmp != undef
             && $hrefJSONInfoTmp->{ 'abbreviations' } != undef
           ){
            my $sExpnTmp = "";
            my $sActTextTmp = $this->{ 'ActualText' };
            my $hrefAbbrExpnTmp = $hrefJSONInfoTmp->{ 'abbreviations' };
            my @arrAbbrTmp = keys( %{ $hrefAbbrExpnTmp } );
            foreach my $sAbbrTmp ( @arrAbbrTmp ){
                $sExpnTmp = $hrefAbbrExpnTmp->{ $sAbbrTmp };
                $this->{ 'ActualText' } =~ s/\Q$sAbbrTmp\E/$sExpnTmp/g; # $V104 Change
            }

            #---- Set actual text on first object
            if ( $this->{ 'ActualText' } ne $sActTextTmp ){ $arrObjsTmp[ 0 ]->{ 'A2WOBJ' }->setActualText( $this->{ 'ActualText' } ); }
        } # if ( $hrefJSONInfoTmp != undef && $hrefJSONInfoTmp->{ 'abbreviations' } != undef )
    } # if ( $blkActiveTmp != undef )

    $this->{ 'ActualText' } = ''; # Reset actual text
    # V102 End

    $this->{ 'InTableCell' } = $FALSE;
    $this->{ 'TableObjectTCId' } = 0;
    if ( $bEmptyCellTmp == $FALSE ){
        # V108 Begin
        #$this->{ 'TagId' } += 1;
        $this->_incrementTagId( $arrObjsTmp[ 0 ] );
        if ( $bIncrTagIdOfCurrentPageTmp == $TRUE ){ $this->_incrementTagId( $this->{ 'RowLastObject' } ); }
        # V108 End
    }

    return 0;
}

#-----------------------------------------------------------------------
# End row
#-----------------------------------------------------------------------
sub endRow{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "endRow()" ); }

    #---- Fetch parameter(s)
    #
    # 1. Row number
    #
    my $iRowNrPar = shift;

    $this->{ 'InTableRow' } = $FALSE;

    #---- Process last object and assign proper tag classification id
    if ( $this->{ 'RowLastObject' } != undef ){
        my $a2wObjTmp = $this->{ 'RowLastObject' }->{ 'A2WOBJ' };

	    #---- Get existing object tag classification id
	    my $iLastObjTCIdTmp = $a2wObjTmp->getTagClassificationId();
        if ( $iRowNrPar == 0 ){ # means TH
            $iLastObjTCIdTmp |= $a2w::ContentBlockConstants::TAG_CLASSIFICATION_TABLE_ELEMENT_TH_END;
        }
        else { # means TR
            $iLastObjTCIdTmp |= $a2w::ContentBlockConstants::TAG_CLASSIFICATION_TABLE_ELEMENT_TR_END;
        }

        #---- set modified object tag classification id
        $a2wObjTmp->setTagClassificationId( $iLastObjTCIdTmp );
    }

    return 0;
}

#-----------------------------------------------------------------------
# End table
#-----------------------------------------------------------------------
sub endTable{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "endTable()" ); }

    #---- Fetch parameter(s)
    #
    # 1. Block
    #
    my $blkTablePar = shift;

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::endTable( $blkTablePar );
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    #---- Process last object and assign proper tag classification id
    my $elemLastObjTmp = $this->{ 'BlockObjects' }[ $this->{ 'BlockObjectsCount' } - 1 ];
    if ( $elemLastObjTmp != undef ){
        my $a2wObjTmp = $elemLastObjTmp->{ 'A2WOBJ' };

	    #---- Get existing object tag classification id
	    my $iLastObjTCIdTmp = $a2wObjTmp->getTagClassificationId();
	    if ( $this->{ 'TableObjectTCId' } > 0 ){ $iLastObjTCIdTmp |= $this->{ 'TableObjectTCId' }; }
        $iLastObjTCIdTmp |= $a2w::ContentBlockConstants::TAG_CLASSIFICATION_TABLE_ELEMENT_TABLE_END;

        #---- set modified object tag classification id
        $a2wObjTmp->setTagClassificationId( $iLastObjTCIdTmp );
    }

    #---- Iterate through objects and display them
    $this->_logTable( $this->{ 'BlockObjects' } );

    #---- Process block objects and add them to appropriate pages
    if ( $this->{ 'BlockObjectsCount' } > 0 ){ $this->_processBlockObjects(); } # V109 Change

    return 0;
}

# $V101 Begin
#---- Illustration related writing methods ----#
#-----------------------------------------------------------------------
# Begin illustration
#-----------------------------------------------------------------------
sub beginIllustration{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "beginIllustration()" ); }

    #---- Fetch parameter(s)
    #
    # 1. Block
    #
    my $blkIllustrationPar = shift;

    $this->{ 'InIllustration' } = $TRUE;
    $this->{ 'BlockObjects' } = [];

    return 0;
}

#-----------------------------------------------------------------------
# Write illustration line
#-----------------------------------------------------------------------
sub writeIllustrationLine{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "writeIllustrationLine()" ); }

    #---- Fetch parameter(s)
    #
    # 1. Array of line objects
    # 2. Line count
    # 3. Line gap (vertical difference between previous and current line, zero for first line)
    #
    my $arefObjectsPar = shift;
    my $iLineCountPar  = shift;
    my $iLineGapPar    = shift;

    #---- Skip, if skip mode is on
    if ( $this->{ 'SkipMode' } == $TRUE ){ return 0; }

    my @arrObjsTmp = @{ $arefObjectsPar };
    my $iObjsCountTmp = @arrObjsTmp;
    if ( $iObjsCountTmp <= 0 ){ return -1; }

    #---- Fill in tag classification id for line objects
    my $iDefTCIDFlagsTmp = ( $a2w::ContentBlockConstants::BLOCK_TYPE_ILLUSTRATION | $a2w::ContentBlockConstants::TAG_CLASSIFICATION_ILLUSTRATION_ELEMENT_FIGURE );
    my @arrTCIdFlagsTmp = ($iDefTCIDFlagsTmp) x $iObjsCountTmp; # fill with paragraph default flags
    $arrTCIdFlagsTmp[ 0 ] = $a2w::ContentBlockConstants::TAG_CLASSIFICATION_ILLUSTRATION_ELEMENT_FIGURE_BEGIN;
    $arrTCIdFlagsTmp[ ( $iObjsCountTmp - 1 ) ] |= $a2w::ContentBlockConstants::TAG_CLASSIFICATION_ILLUSTRATION_ELEMENT_FIGURE_END;

    #---- Write illustration line objects
    my $iObjTCIdTmp = 0;
    my $a2wObjTmp   = undef;
    my $elemObjTmp  = undef;
    $this->{ 'ActualText' } = ''; # V102 Change
    $this->{ 'ObjectId' } = 0;    # V109 Change
	
    for ( my $plo = 0; $plo < $iObjsCountTmp; $plo++ ){
        $elemObjTmp = $arrObjsTmp[ $plo ];
        $a2wObjTmp = $elemObjTmp->{ 'A2WOBJ' };

        #---- Update tag classification id
        $iObjTCIdTmp = $a2wObjTmp->getTagClassificationId(); # Get existing classification id
        $iObjTCIdTmp |= $arrTCIdFlagsTmp[ $plo ];            # Update existing classification id
        $a2wObjTmp->setTagClassificationId( $iObjTCIdTmp );  # Set back updated classification id

        #---- Write object
        $this->writeObject( $elemObjTmp );
    }
    #$this->{ 'TagId' } += 1; # V108 Change
    $this->_incrementTagId( $arrObjsTmp[ 0 ] ); # V108 Change

    # V102 Begin
    #---- Find and replace all block abbreviation with expansion
    #
    #---- Get active page context
    $this->{ 'ActualText' } =~ s/\s$//; # Truncate space at the end

    #---- Get page handle
    my $pgHandleTmp = $this->{ 'ActivePage' };

    #---- Get active block
    my $blkActiveTmp = $pgHandleTmp->{ 'ActiveBlock' };
    if (    $this->{ 'ActualText' } ne ""
         && $blkActiveTmp != undef
         && $arrObjsTmp[ 0 ] != undef
       ){
        #---- Get block json info
        my $hrefJSONInfoTmp = $blkActiveTmp->getJSONInfo();
        if (    $hrefJSONInfoTmp != undef
             && $hrefJSONInfoTmp->{ 'abbreviations' } != undef
           ){
            my $sExpnTmp = "";
            my $sActTextTmp = $this->{ 'ActualText' };
            my $hrefAbbrExpnTmp = $hrefJSONInfoTmp->{ 'abbreviations' };
            my @arrAbbrTmp = keys( %{ $hrefAbbrExpnTmp } );
            foreach my $sAbbrTmp ( @arrAbbrTmp ){
                $sExpnTmp = $hrefAbbrExpnTmp->{ $sAbbrTmp };
                $this->{ 'ActualText' } =~ s/$sAbbrTmp/$sExpnTmp/g;
            }

            #---- Set actual text on first object
            if ( $this->{ 'ActualText' } ne $sActTextTmp ){ $arrObjsTmp[ 0 ]->{ 'A2WOBJ' }->setActualText( $this->{ 'ActualText' } ); }
        } # if ( $hrefJSONInfoTmp != undef && $hrefJSONInfoTmp->{ 'abbreviations' } != undef )
    } # if ( $blkActiveTmp != undef )

    $this->{ 'ActualText' } = ''; # Reset actual text
    # V102 End

    return 0;
}

#-----------------------------------------------------------------------
# End illustration
#-----------------------------------------------------------------------
sub endIllustration{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "endIllustration()" ); }

    #---- Fetch parameter(s)
    #
    # 1. Block
    #
    my $blkIllustrationPar = shift;

    #---- Iterate through objects and display them
    $this->_logIllustration( $this->{ 'BlockObjects' } );

    #---- Process block objects and add them to appropriate pages
    if ( $this->{ 'BlockObjectsCount' } > 0 ){ $this->_processBlockObjects(); } # V109 Change

    $this->{ 'InIllustration' } = $FALSE;

    return 0;
}
# $V101 End

# V110 Begin
#---- List related writing methods ----#
#-----------------------------------------------------------------------
# Begin list
#-----------------------------------------------------------------------
sub beginList{
    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "beginList()" ); }

    #---- Fetch parameter(s)
    #
    # 1. Block
    # 2. Columns X positions array
    # 3. Starting Y position
    #
    my $blkListPar = shift;
    my $arefColumnsXPosPar = shift;
    my $iStartYPar         = shift;
    if ( @{ $arefColumnsXPosPar } <= 0 ){ return -1; }

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::beginList( $blkListPar, $arefColumnsXPosPar, $iStartYPar );
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    $this->{ 'BlockObjects' } = [];
    $this->{ 'BlockObjectsCount' } = 0;

    $this->{ 'ListBeginTCIds' } = $a2w::ContentBlockConstants::TAG_CLASSIFICATION_LIST_ELEMENT_L_BEGIN;
    $this->{ 'ListLastObject' } = undef;

    return 0;
}

#-----------------------------------------------------------------------
# Begin list item
#-----------------------------------------------------------------------
sub beginListItem{
    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "beginListItem()" ); }

    #---- Fetch parameter(s)
    #
    # 1. List item number
    # 2. Page id (where list item starts on)
    #
    my $iListItemNrPar = shift;
    my $iPageIdPar     = shift;

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::beginListItem( $iListItemNrPar, $iPageIdPar );
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    $this->{ 'ListBeginTCIds' } |= $a2w::ContentBlockConstants::TAG_CLASSIFICATION_LIST_ELEMENT_LI_BEGIN;
    $this->{ 'ListLastObject' } = undef;

    return 0;
}

#-----------------------------------------------------------------------
# Write list label
#-----------------------------------------------------------------------
sub writeListLabel{
    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "writeListLabel()" ); }

    #---- Fetch parameter(s)
    #
    # 1. Array of label objects
    #
    my $arefObjectsPar = shift;

    #---- Skip, if skip mode is on
    if ( $this->{ 'SkipMode' } == $TRUE ){ return 0; }

    #---- Set in item label flag
    $this->{ 'InList' } |= (1 << 3);

    my @arrObjsTmp = @{ $arefObjectsPar };
    my $iObjsCountTmp = @arrObjsTmp;
    if ( $iObjsCountTmp <= 0 ){ return 0; }

    #---- Fill in tag classification id for list item label objects
    my $iDefTCIDFlagsTmp = ( $a2w::ContentBlockConstants::BLOCK_TYPE_LIST | $a2w::ContentBlockConstants::TAG_CLASSIFICATION_LIST_ELEMENT_LBL );
    my @arrTCIdFlagsTmp = ($iDefTCIDFlagsTmp) x $iObjsCountTmp; # fill with label default flags
    $arrTCIdFlagsTmp[ 0 ] = $a2w::ContentBlockConstants::TAG_CLASSIFICATION_LIST_ELEMENT_LBL_BEGIN;
    if ( $this->{ 'ListBeginTCIds' } > 0 ){
        $arrTCIdFlagsTmp[ 0 ] |= $this->{ 'ListBeginTCIds' };
        $this->{ 'ListBeginTCIds' } = 0;
    }
    $arrTCIdFlagsTmp[ ( $iObjsCountTmp - 1 ) ] |= $a2w::ContentBlockConstants::TAG_CLASSIFICATION_LIST_ELEMENT_LBL_END;

    $this->{ 'ListLastObject' } = $arrObjsTmp[ $#arrObjsTmp ]; # Get last object of label

    #---- Write list item label objects
    my $iObjTCIdTmp = 0;
    my $a2wObjTmp   = undef;
    my $elemObjTmp  = undef;
    $this->{ 'ActualText' } = '';
    $this->{ 'ObjectId' } = 0;
	
    for ( my $plo = 0; $plo < $iObjsCountTmp; $plo++ ){
        $elemObjTmp = $arrObjsTmp[ $plo ];
        $a2wObjTmp = $elemObjTmp->{ 'A2WOBJ' };

        #---- Update tag classification id
        $iObjTCIdTmp = $a2wObjTmp->getTagClassificationId(); # Get existing classification id
        $iObjTCIdTmp |= $arrTCIdFlagsTmp[ $plo ];            # Update existing classification id
        $a2wObjTmp->setTagClassificationId( $iObjTCIdTmp );  # Set back updated classification id

        #---- Write object
        $this->writeObject( $elemObjTmp );
    }
    $this->_incrementTagId( $arrObjsTmp[ 0 ] );

    #---- Find and replace all block abbreviation with expansion
    #
    #---- Get active page context
    $this->{ 'ActualText' } =~ s/\s$//; # Truncate space at the end

    #---- Get page handle
    my $pgHandleTmp = $this->{ 'ActivePage' };

    #---- Get active block
    my $blkActiveTmp = $pgHandleTmp->{ 'ActiveBlock' };
    if (    $this->{ 'ActualText' } ne ""
         && $blkActiveTmp != undef
         && $arrObjsTmp[ 0 ] != undef
       ){
        #---- Get block json info
        my $hrefJSONInfoTmp = $blkActiveTmp->getJSONInfo();
        if (    $hrefJSONInfoTmp != undef
             && $hrefJSONInfoTmp->{ 'abbreviations' } != undef
           ){
            my $sExpnTmp = "";
            my $sActTextTmp = $this->{ 'ActualText' };
            my $hrefAbbrExpnTmp = $hrefJSONInfoTmp->{ 'abbreviations' };
            my @arrAbbrTmp = keys( %{ $hrefAbbrExpnTmp } );
            foreach my $sAbbrTmp ( @arrAbbrTmp ){
                $sExpnTmp = $hrefAbbrExpnTmp->{ $sAbbrTmp };
                $this->{ 'ActualText' } =~ s/$sAbbrTmp/$sExpnTmp/g;
            }

            #---- Set actual text on first object
            if ( $this->{ 'ActualText' } ne $sActTextTmp ){ $arrObjsTmp[ 0 ]->{ 'A2WOBJ' }->setActualText( $this->{ 'ActualText' } ); }
        } # if ( $hrefJSONInfoTmp != undef && $hrefJSONInfoTmp->{ 'abbreviations' } != undef )
    } # if ( $blkActiveTmp != undef )

    $this->{ 'ActualText' } = ''; # Reset actual text

    #---- Reset in item label flag
    $this->{ 'InList' } ^= (1 << 3);

    return 0;
}

#-----------------------------------------------------------------------
# Write list body
#-----------------------------------------------------------------------
sub writeListBody{
    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "writeListBody()" ); }

    #---- Fetch parameter(s)
    #
    # 1. Array of body objects
    #
    my $arefObjectsPar = shift;

    #---- Skip, if skip mode is on
    if ( $this->{ 'SkipMode' } == $TRUE ){ return 0; }

    #---- Set in item body flag
    $this->{ 'InList' } |= (1 << 4);

    my @arrObjsTmp = @{ $arefObjectsPar };
    my $iObjsCountTmp = @arrObjsTmp;
    if ( $iObjsCountTmp <= 0 ){ return 0; }

    #---- Fill in tag classification id for list item body objects
    my $iDefTCIDFlagsTmp = ( $a2w::ContentBlockConstants::BLOCK_TYPE_LIST | $a2w::ContentBlockConstants::TAG_CLASSIFICATION_LIST_ELEMENT_LBODY );
    my @arrTCIdFlagsTmp = ($iDefTCIDFlagsTmp) x $iObjsCountTmp; # fill with body default flags
    $arrTCIdFlagsTmp[ 0 ] = $a2w::ContentBlockConstants::TAG_CLASSIFICATION_LIST_ELEMENT_LBODY_BEGIN;
    if ( $this->{ 'ListBeginTCIds' } > 0 ){
        $arrTCIdFlagsTmp[ 0 ] |= $this->{ 'ListBeginTCIds' };
        $this->{ 'ListBeginTCIds' } = 0;
    }
    $arrTCIdFlagsTmp[ ( $iObjsCountTmp - 1 ) ] |= $a2w::ContentBlockConstants::TAG_CLASSIFICATION_LIST_ELEMENT_LBODY_END;

    $this->{ 'ListLastObject' } = $arrObjsTmp[ $#arrObjsTmp ]; # Get last object of body

    #---- Write list item body objects
    my $iObjTCIdTmp = 0;
    my $a2wObjTmp   = undef;
    my $elemObjTmp  = undef;
    $this->{ 'ActualText' } = '';
    $this->{ 'ObjectId' } = 0;
	
    for ( my $plo = 0; $plo < $iObjsCountTmp; $plo++ ){
        $elemObjTmp = $arrObjsTmp[ $plo ];
        $a2wObjTmp = $elemObjTmp->{ 'A2WOBJ' };

        #---- Update tag classification id
        $iObjTCIdTmp = $a2wObjTmp->getTagClassificationId(); # Get existing classification id
        $iObjTCIdTmp |= $arrTCIdFlagsTmp[ $plo ];            # Update existing classification id
        $a2wObjTmp->setTagClassificationId( $iObjTCIdTmp );  # Set back updated classification id

        #---- Write object
        $this->writeObject( $elemObjTmp );
    }
    $this->_incrementTagId( $arrObjsTmp[ 0 ] );

    #---- Find and replace all block abbreviation with expansion
    #
    #---- Get active page context
    $this->{ 'ActualText' } =~ s/\s$//; # Truncate space at the end

    #---- Get page handle
    my $pgHandleTmp = $this->{ 'ActivePage' };

    #---- Get active block
    my $blkActiveTmp = $pgHandleTmp->{ 'ActiveBlock' };
    if (    $this->{ 'ActualText' } ne ""
         && $blkActiveTmp != undef
         && $arrObjsTmp[ 0 ] != undef
       ){
        #---- Get block json info
        my $hrefJSONInfoTmp = $blkActiveTmp->getJSONInfo();
        if (    $hrefJSONInfoTmp != undef
             && $hrefJSONInfoTmp->{ 'abbreviations' } != undef
           ){
            my $sExpnTmp = "";
            my $sActTextTmp = $this->{ 'ActualText' };
            my $hrefAbbrExpnTmp = $hrefJSONInfoTmp->{ 'abbreviations' };
            my @arrAbbrTmp = keys( %{ $hrefAbbrExpnTmp } );
            foreach my $sAbbrTmp ( @arrAbbrTmp ){
                $sExpnTmp = $hrefAbbrExpnTmp->{ $sAbbrTmp };
                $this->{ 'ActualText' } =~ s/$sAbbrTmp/$sExpnTmp/g;
            }

            #---- Set actual text on first object
            if ( $this->{ 'ActualText' } ne $sActTextTmp ){ $arrObjsTmp[ 0 ]->{ 'A2WOBJ' }->setActualText( $this->{ 'ActualText' } ); }
        } # if ( $hrefJSONInfoTmp != undef && $hrefJSONInfoTmp->{ 'abbreviations' } != undef )
    } # if ( $blkActiveTmp != undef )

    $this->{ 'ActualText' } = ''; # Reset actual text

    #---- Reset in item body flag
    $this->{ 'InList' } ^= (1 << 4);

    return 0;
}

#-----------------------------------------------------------------------
# End list item
#-----------------------------------------------------------------------
sub endListItem{
    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "endListItem()" ); }

    #---- Fetch parameter(s)
    #
    # 1. List item number
    #
    my $iListItemNrPar = shift;

    #---- Process last object and assign proper tag classification id
    if ( $this->{ 'ListLastObject' } != undef ){
        my $a2wObjTmp = $this->{ 'ListLastObject' }->{ 'A2WOBJ' };

	    #---- Get existing object tag classification id
	    my $iLastObjTCIdTmp = $a2wObjTmp->getTagClassificationId();
        $iLastObjTCIdTmp |= $a2w::ContentBlockConstants::TAG_CLASSIFICATION_LIST_ELEMENT_LI_END;

        #---- set modified object tag classification id
        $a2wObjTmp->setTagClassificationId( $iLastObjTCIdTmp );
    }
    $this->{ 'ListLastObject' } = undef;

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::endListItem( $iListItemNrPar );
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    return 0;
}

#-----------------------------------------------------------------------
# End list
#-----------------------------------------------------------------------
sub endList{
    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "endList()" ); }

    #---- Fetch parameter(s)
    #
    # 1. Block
    #
    my $blkListPar = shift;

    #---- Process last object and assign proper tag classification id
    my $elemLastObjTmp = $this->{ 'BlockObjects' }[ $this->{ 'BlockObjectsCount' } - 1 ];
    if ( $elemLastObjTmp != undef ){
        my $a2wObjTmp = $elemLastObjTmp->{ 'A2WOBJ' };

	    #---- Get existing object tag classification id
	    my $iLastObjTCIdTmp = $a2wObjTmp->getTagClassificationId();
	    if ( $this->{ 'TableObjectTCId' } > 0 ){ $iLastObjTCIdTmp |= $this->{ 'TableObjectTCId' }; }
        $iLastObjTCIdTmp |= $a2w::ContentBlockConstants::TAG_CLASSIFICATION_LIST_ELEMENT_L_END;

        #---- set modified object tag classification id
        $a2wObjTmp->setTagClassificationId( $iLastObjTCIdTmp );
    }

    #---- Iterate through objects and display them
    $this->_logList( $this->{ 'BlockObjects' } );

    #---- Process block objects and add them to appropriate pages
    if ( $this->{ 'BlockObjectsCount' } > 0 ){ $this->_processBlockObjects(); }

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::endListItem( $blkListPar );
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    return 0;
}
# V110 End

#-----------------------------------------------------------------------
# Log paragraph
#-----------------------------------------------------------------------
sub _logParagraph{
    my $this = shift;

    #---- Get parameter
    #
    # 1. Array of objects hash (having both kernel object and pom object)
    #
    my $arefParagraphObjsPar = shift;
    my @arrParagraphObjsTmp  = @{ $arefParagraphObjsPar };

    my $iTCIdTmp = 0;
    my $iTagIdTmp = 0;
    my $iTagElementTmp = 0;
    my $iTagCategoryTmp = 0;
    my $iTagBeginEndTmp = 0;
    my $sTagCategoryTmp = "";

    my $sIndentTmp = "";
    my $sObjTypeTmp = "";
    my $handleFileTmp = $this->{ 'fHandle' };

    #---- Iterate objects and log them
    my $a2wObjTmp = undef;
    my $pomObjTmp = undef;
    foreach my $elem ( @arrParagraphObjsTmp ){
        #---- Get object details
        $a2wObjTmp = $elem->{ 'A2WOBJ' };
        $pomObjTmp = $elem->{ 'POMOBJ' };

        $sObjTypeTmp = $a2wObjTmp->_getType();
        $iTagIdTmp = $a2wObjTmp->getTagId();
        $iTCIdTmp = $a2wObjTmp->getTagClassificationId();

        $iTagElementTmp  = $iTCIdTmp & 0x000000FF;
        $iTagCategoryTmp = $iTCIdTmp & 0xFF000000;
        $iTagBeginEndTmp = $iTCIdTmp & 0x00FFFF00;

        #---- Write beginning of element
        if ( $iTagCategoryTmp == $a2w::ContentBlockConstants::BLOCK_TYPE_PARAGRAPH ){
            if ( ( $iTagElementTmp & 0x01 ) != 0 ){    # Paragraph
                $sTagCategoryTmp = "P";
                if ( ( $iTagBeginEndTmp & 0x000100 ) != 0 ){ $this->_printMessage( "$sTagCategoryTmp>" ); }
            }
            elsif ( ( $iTagElementTmp & 0x02 ) != 0 ){ # Heading
                $sTagCategoryTmp = "H";
                if ( ( $iTagBeginEndTmp & 0x000400 ) != 0 ){ $this->_printMessage( "$sTagCategoryTmp>" ); }
            }
            elsif ( ( $iTagElementTmp & 0x04 ) != 0 ){ # Heading 1
                $sTagCategoryTmp = "H1";
                if ( ( $iTagBeginEndTmp & 0x001000 ) != 0 ){ $this->_printMessage( "$sTagCategoryTmp>" ); }
            }
            elsif ( ( $iTagElementTmp & 0x08 ) != 0 ){ # Heading 2
                $sTagCategoryTmp = "H2";
                if ( ( $iTagBeginEndTmp & 0x004000 ) != 0 ){ $this->_printMessage( "$sTagCategoryTmp>" ); }
            }
            elsif ( ( $iTagElementTmp & 0x10 ) != 0 ){ # Heading 3
                $sTagCategoryTmp = "H3";
                if ( ( $iTagBeginEndTmp & 0x010000 ) != 0 ){ $this->_printMessage( "$sTagCategoryTmp>" ); }
            }
            elsif ( ( $iTagElementTmp & 0x20 ) != 0 ){ # Heading 4
                $sTagCategoryTmp = "H4";
                if ( ( $iTagBeginEndTmp & 0x040000 ) != 0 ){ $this->_printMessage( "$sTagCategoryTmp>" ); }
            }
            elsif ( ( $iTagElementTmp & 0x40 ) != 0 ){ # Heading 5
                $sTagCategoryTmp = "H5";
                if ( ( $iTagBeginEndTmp & 0x100000 ) != 0 ){ $this->_printMessage( "$sTagCategoryTmp>" ); }
            }
            elsif ( ( $iTagElementTmp & 0x80 ) != 0 ){ # Heading 6
                $sTagCategoryTmp = "H6";
                if ( ( $iTagBeginEndTmp & 0x400000 ) != 0 ){ $this->_printMessage( "$sTagCategoryTmp>" ); }
            }
            $sIndentTmp = " ";
        }

        #---- Log object
        if ( $sObjTypeTmp eq "text" ){
            my $sTextTmp = $a2wObjTmp->getActualText();
            if ( $sTextTmp eq "" ){ $sTextTmp = $a2wObjTmp->getText(); }
            $this->_printMessage( sprintf( $sIndentTmp . "txt>%s<@(%d,%d) ISeqId=%d, TagId=%d, TCId=%08X", $sTextTmp, $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS }, $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS }, $a2wObjTmp->getSequenceId(), $a2wObjTmp->getTagId(), $iTCIdTmp ) );
        }
        elsif ( $sObjTypeTmp eq "image" ){
            $this->_printMessage( sprintf( $sIndentTmp . "img>%s<@(%d,%d) W=%d, H=%d, ISeqId=%d, TagId=%d, TCId=%08X", $a2wObjTmp->getName(), $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS }, $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS }, $a2wObjTmp->getWidth(), $a2wObjTmp->getHeight(), $a2wObjTmp->getSequenceId(), $a2wObjTmp->getTagId(), $iTCIdTmp ) );
        }
        # $V101 Begin
        elsif ( $sObjTypeTmp eq "container" ){
            $this->_printMessage( sprintf( $sIndentTmp . "obj>%s<@(%d,%d) W=%d, H=%d, ISeqId=%d, TagId=%d, TCId=%08X", $a2wObjTmp->getName(), $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS }, $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS }, $a2wObjTmp->getWidth(), $a2wObjTmp->getHeight(), $a2wObjTmp->getSequenceId(), $a2wObjTmp->getTagId(), $iTCIdTmp ) );
        }
        # $V101 End
        elsif ( $sObjTypeTmp eq "line" ){
            $this->_printMessage( sprintf( $sIndentTmp . "lin><@(%d,%d) W=%d, L=%d, ISeqId=%d, TagId=%d, TCId=%08X", $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS }, $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS }, $a2wObjTmp->getWidth(), $a2wObjTmp->getLength(), $a2wObjTmp->getSequenceId(), $a2wObjTmp->getTagId(), $iTCIdTmp ) );
        }

        #---- Write ending of element
        if (    ( $iTagBeginEndTmp & 0x000200 ) != 0 # Paragraph
		     || ( $iTagBeginEndTmp & 0x000800 ) != 0 # Heading
		     || ( $iTagBeginEndTmp & 0x002000 ) != 0 # Heading 1
		     || ( $iTagBeginEndTmp & 0x008000 ) != 0 # Heading 2
		     || ( $iTagBeginEndTmp & 0x020000 ) != 0 # Heading 3
		     || ( $iTagBeginEndTmp & 0x080000 ) != 0 # Heading 4
		     || ( $iTagBeginEndTmp & 0x200000 ) != 0 # Heading 5
		     || ( $iTagBeginEndTmp & 0x800000 ) != 0 # Heading 6
		   ){
            $this->_printMessage( "<$sTagCategoryTmp" );
            $sIndentTmp = "";
        }
    }
}

#-----------------------------------------------------------------------
# Log table
#-----------------------------------------------------------------------
sub _logTable{
    my $this = shift;

    #---- Get parameter
    #
    # 1. Array of objects hash (having both kernel object and pom object)
    #
    my $arefTableObjsPar = shift;
    my @arrTableObjsTmp  = @{ $arefTableObjsPar };

    my $iTCIdTmp = 0;
    my $iTagIdTmp = 0;
    my $iTagMarkTmp = 0;
    my $iTagElemTmp = 0;
    my $iTagCategoryTmp = 0;

    my $sIndentTmp = "";
    my $sObjTypeTmp = "";
    my $handleFileTmp = $this->{ 'fHandle' };

    #---- Iterate objects and log them
    my $a2wObjTmp = undef;
    my $pomObjTmp = undef;
    foreach my $elem ( @arrTableObjsTmp ){
        #---- Get object details
        $a2wObjTmp = $elem->{ 'A2WOBJ' };
        $pomObjTmp = $elem->{ 'POMOBJ' };

        $sObjTypeTmp = $a2wObjTmp->_getType();
        $iTagIdTmp = $a2wObjTmp->getTagId();
        $iTCIdTmp = $a2wObjTmp->getTagClassificationId();

        $iTagMarkTmp = $iTCIdTmp & 0x00FFFF00;
        $iTagElemTmp = $iTCIdTmp & 0x000000FF;
        $iTagCategoryTmp = $iTCIdTmp & 0xFF000000;

        #---- Write beginning of element
        if ( $iTagCategoryTmp == $a2w::ContentBlockConstants::BLOCK_TYPE_TABLE ){
            if ( ( $iTagMarkTmp & 0x000100 ) != 0 && ( $iTagElemTmp & 0x01 ) != 0 ){ # Table
                $this->_printMessage( "tbl>" );
            }
            if ( ( $iTagMarkTmp & 0x010000 ) != 0 && ( $iTagElemTmp & 0x10 ) != 0 ){ # Table header
                $sIndentTmp = " " x 1;
                $this->_printMessage( $sIndentTmp . "thead>" );
            }
            if ( ( $iTagMarkTmp & 0x040000 ) != 0 && ( $iTagElemTmp & 0x20 ) != 0 ){ # Table body
                $sIndentTmp = " " x 1;
                $this->_printMessage( $sIndentTmp . "tbody>" );
            }
            if ( ( $iTagMarkTmp & 0x100000 ) != 0 && ( $iTagElemTmp & 0x40 ) != 0 ){ # Table footer
                $sIndentTmp = " " x 1;
                $this->_printMessage( $sIndentTmp . "tfoot>" );
            }
            if ( ( $iTagMarkTmp & 0x000400 ) != 0 && ( $iTagElemTmp & 0x02 ) != 0 ){ # Table row
                $sIndentTmp = " " x 2;
                $this->_printMessage( $sIndentTmp . "tr>" );
            }
            if ( ( $iTagMarkTmp & 0x001000 ) != 0 && ( $iTagElemTmp & 0x04 ) != 0 ){ # Table header row
                $sIndentTmp = " " x 2;
                $this->_printMessage( $sIndentTmp . "th>" );
            }
            if ( ( $iTagMarkTmp & 0x004000 ) != 0 && ( $iTagElemTmp & 0x08 ) != 0 ){ # Table cell
                $sIndentTmp = " " x 3;
                $this->_printMessage( $sIndentTmp . "td>" );
            }
        } # if ( $iTagCategoryTmp == $a2w::ContentBlockConstants::BLOCK_TYPE_TABLE )

        #---- Log object
        if ( $sObjTypeTmp eq "text" ){
            my $sTextTmp = $a2wObjTmp->getActualText();
            if ( $sTextTmp eq "" ){ $sTextTmp = $a2wObjTmp->getText(); }
            $this->_printMessage( sprintf( $sIndentTmp . "txt>%s<@(%d,%d) ISeqId=%d, TagId=%d, TCId=%08X", $sTextTmp, $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS }, $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS }, $a2wObjTmp->getSequenceId(), $a2wObjTmp->getTagId(), $iTCIdTmp ) );
        }
        elsif ( $sObjTypeTmp eq "image" ){
            $this->_printMessage( sprintf( $sIndentTmp . "img>%s<@(%d,%d) W=%d, H=%d, ISeqId=%d, TagId=%d, TCId=%08X", $a2wObjTmp->getName(), $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS }, $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS }, $a2wObjTmp->getWidth(), $a2wObjTmp->getHeight(), $a2wObjTmp->getSequenceId(), $a2wObjTmp->getTagId(), $iTCIdTmp ) );
        }
        # $V101 Begin
        elsif ( $sObjTypeTmp eq "container" ){
            $this->_printMessage( sprintf( $sIndentTmp . "obj>%s<@(%d,%d) W=%d, H=%d, ISeqId=%d, TagId=%d, TCId=%08X", $a2wObjTmp->getName(), $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS }, $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS }, $a2wObjTmp->getWidth(), $a2wObjTmp->getHeight(), $a2wObjTmp->getSequenceId(), $a2wObjTmp->getTagId(), $iTCIdTmp ) );
        }
        # $V101 End
        elsif ( $sObjTypeTmp eq "line" ){
            $this->_printMessage( sprintf( $sIndentTmp . "lin><@(%d,%d) W=%d, L=%d, ISeqId=%d, TagId=%d, TCId=%08X", $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS }, $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS }, $a2wObjTmp->getWidth(), $a2wObjTmp->getLength(), $a2wObjTmp->getSequenceId(), $a2wObjTmp->getTagId(), $iTCIdTmp ) );
        }

        #---- Write ending of element
        if ( $iTagCategoryTmp == 0x03000000 ){ # table
            if ( ( $iTagMarkTmp & 0x008000 ) != 0 && ( $iTagElemTmp & 0x08 ) != 0 ){ # Table cell
                $sIndentTmp = " " x 3;
                $this->_printMessage( $sIndentTmp . "<td" );
            }
            if ( ( $iTagMarkTmp & 0x002000 ) != 0 && ( $iTagElemTmp & 0x04 ) != 0 ){ # Table header row
                $sIndentTmp = " " x 2;
                $this->_printMessage( $sIndentTmp . "<th" );
            }
            if ( ( $iTagMarkTmp & 0x000800 ) != 0 && ( $iTagElemTmp & 0x02 ) != 0 ){ # Table row
                $sIndentTmp = " " x 2;
                $this->_printMessage( $sIndentTmp . "<tr" );
            }
            if ( ( $iTagMarkTmp & 0x020000 ) != 0 && ( $iTagElemTmp & 0x10 ) != 0 ){ # Table header
                $sIndentTmp = " " x 1;
                $this->_printMessage( $sIndentTmp . "<thead" );
            }
            if ( ( $iTagMarkTmp & 0x080000 ) != 0 && ( $iTagElemTmp & 0x20 ) != 0 ){ # Table body
                $sIndentTmp = " " x 1;
                $this->_printMessage( $sIndentTmp . "<tbody" );
            }
            if ( ( $iTagMarkTmp & 0x200000 ) != 0 && ( $iTagElemTmp & 0x40 ) != 0 ){ # Table footer
                $sIndentTmp = " " x 1;
                $this->_printMessage( $sIndentTmp . "<tfoot" );
            }
            if ( ( $iTagMarkTmp & 0x000200 ) != 0 && ( $iTagElemTmp & 0x01 ) != 0 ){ # Table
                $this->_printMessage( $sIndentTmp . "<tbl" );
            }
        } # if ( $iTagCategoryTmp == $a2w::ContentBlockConstants::BLOCK_TYPE_TABLE )
    } # foreach my $elem ( @arrTableObjsTmp )
}

# $V101 Begin
#-----------------------------------------------------------------------
# Log illustration
#-----------------------------------------------------------------------
sub _logIllustration{
    my $this = shift;

    #---- Get parameter
    #
    # 1. Array of objects hash (having both kernel object and pom object)
    #
    my $arefIllustrationObjsPar = shift;
    my @arrIllustrationObjsTmp  = @{ $arefIllustrationObjsPar };

    my $iTCIdTmp = 0;
    my $iTagIdTmp = 0;
    my $iTagElementTmp = 0;
    my $iTagCategoryTmp = 0;
    my $iTagBeginEndTmp = 0;
    my $sTagCategoryTmp = "";

    my $sIndentTmp = "";
    my $sObjTypeTmp = "";
    my $handleFileTmp = $this->{ 'fHandle' };

    #---- Iterate objects and log them
    my $a2wObjTmp = undef;
    my $pomObjTmp = undef;
    foreach my $elem ( @arrIllustrationObjsTmp ){
        #---- Get object details
        $a2wObjTmp = $elem->{ 'A2WOBJ' };
        $pomObjTmp = $elem->{ 'POMOBJ' };

        $sObjTypeTmp = $a2wObjTmp->_getType();
        $iTagIdTmp = $a2wObjTmp->getTagId();
        $iTCIdTmp = $a2wObjTmp->getTagClassificationId();

        $iTagElementTmp  = $iTCIdTmp & 0x000000FF;
        $iTagCategoryTmp = $iTCIdTmp & 0xFF000000;
        $iTagBeginEndTmp = $iTCIdTmp & 0x00FFFF00;

        #---- Write beginning of element
        if ( $iTagCategoryTmp == $a2w::ContentBlockConstants::BLOCK_TYPE_ILLUSTRATION ){
            if ( ( $iTagElementTmp & 0x01 ) != 0 ){    # Figure
                $sTagCategoryTmp = "Figure";
                if ( ( $iTagBeginEndTmp & 0x000100 ) != 0 ){ $this->_printMessage( "$sTagCategoryTmp>" ); }
            }
            $sIndentTmp = " ";
        }

        #---- Log object
        if ( $sObjTypeTmp eq "text" ){
            my $sTextTmp = $a2wObjTmp->getActualText();
            if ( $sTextTmp eq "" ){ $sTextTmp = $a2wObjTmp->getText(); }
            $this->_printMessage( sprintf( $sIndentTmp . "txt>%s<@(%d,%d) ISeqId=%d, TagId=%d, TCId=%08X", $sTextTmp, $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS }, $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS }, $a2wObjTmp->getSequenceId(), $a2wObjTmp->getTagId(), $iTCIdTmp ) );
        }
        elsif ( $sObjTypeTmp eq "image" ){
            $this->_printMessage( sprintf( $sIndentTmp . "img>%s<@(%d,%d) W=%d, H=%d, ISeqId=%d, TagId=%d, TCId=%08X", $a2wObjTmp->getName(), $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS }, $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS }, $a2wObjTmp->getWidth(), $a2wObjTmp->getHeight(), $a2wObjTmp->getSequenceId(), $a2wObjTmp->getTagId(), $iTCIdTmp ) );
        }
        elsif ( $sObjTypeTmp eq "container" ){
            $this->_printMessage( sprintf( $sIndentTmp . "obj>%s<@(%d,%d) W=%d, H=%d, ISeqId=%d, TagId=%d, TCId=%08X", $a2wObjTmp->getName(), $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS }, $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS }, $a2wObjTmp->getWidth(), $a2wObjTmp->getHeight(), $a2wObjTmp->getSequenceId(), $a2wObjTmp->getTagId(), $iTCIdTmp ) );
        }
        elsif ( $sObjTypeTmp eq "line" ){
            $this->_printMessage( sprintf( $sIndentTmp . "lin><@(%d,%d) W=%d, L=%d, ISeqId=%d, TagId=%d, TCId=%08X", $a2wObjTmp->getXPos(), $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS }, $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS }, $a2wObjTmp->getLength(), $a2wObjTmp->getSequenceId(), $a2wObjTmp->getTagId(), $iTCIdTmp ) );
        }

        #---- Write ending of element
        if (    ( $iTagBeginEndTmp & 0x000200 ) != 0 # Figure
		   ){
            $this->_printMessage( "<$sTagCategoryTmp" );
            $sIndentTmp = "";
        }
    }
}
# $V101 End

# $V110 Begin
#-----------------------------------------------------------------------
# Log list
#-----------------------------------------------------------------------
sub _logList{
    my $this = shift;

    #---- Get parameter
    #
    # 1. Array of objects hash (having both kernel object and pom object)
    #
    my $arefListObjsPar = shift;
    my @arrListObjsTmp  = @{ $arefListObjsPar };

    my $iTCIdTmp = 0;
    my $iTagIdTmp = 0;
    my $iTagMarkTmp = 0;
    my $iTagElemTmp = 0;
    my $iTagCategoryTmp = 0;

    my $sIndentTmp = "";
    my $sObjTypeTmp = "";
    my $handleFileTmp = $this->{ 'fHandle' };

    #---- Iterate objects and log them
    my $a2wObjTmp = undef;
    my $pomObjTmp = undef;
    foreach my $elem ( @arrListObjsTmp ){
        #---- Get object details
        $a2wObjTmp = $elem->{ 'A2WOBJ' };
        $pomObjTmp = $elem->{ 'POMOBJ' };

        $sObjTypeTmp = $a2wObjTmp->_getType();
        $iTagIdTmp = $a2wObjTmp->getTagId();
        $iTCIdTmp = $a2wObjTmp->getTagClassificationId();

        $iTagMarkTmp = $iTCIdTmp & 0x00FFFF00;
        $iTagElemTmp = $iTCIdTmp & 0x000000FF;
        $iTagCategoryTmp = $iTCIdTmp & 0xFF000000;

        #---- Write beginning of element
        if ( $iTagCategoryTmp == $a2w::ContentBlockConstants::BLOCK_TYPE_LIST ){
            if ( ( $iTagMarkTmp & 0x000100 ) != 0 && ( $iTagElemTmp & 0x01 ) != 0 ){ # List
                $this->_printMessage( "list>" );
            }
            if ( ( $iTagMarkTmp & 0x000400 ) != 0 && ( $iTagElemTmp & 0x02 ) != 0 ){ # List item
                $sIndentTmp = " " x 1;
                $this->_printMessage( $sIndentTmp . "item>" );
            }
            if ( ( $iTagMarkTmp & 0x001000 ) != 0 && ( $iTagElemTmp & 0x04 ) != 0 ){ # Item label
                $sIndentTmp = " " x 2;
                $this->_printMessage( $sIndentTmp . "lbl>" );
                $sIndentTmp = " " x 3;
            }
            if ( ( $iTagMarkTmp & 0x004000 ) != 0 && ( $iTagElemTmp & 0x08 ) != 0 ){ # Item body
                $sIndentTmp = " " x 2;
                $this->_printMessage( $sIndentTmp . "body>" );
                $sIndentTmp = " " x 3;
            }
        } # if ( $iTagCategoryTmp == $a2w::ContentBlockConstants::BLOCK_TYPE_LIST )

        #---- Log object
        if ( $sObjTypeTmp eq "text" ){
            my $sTextTmp = $a2wObjTmp->getActualText();
            if ( $sTextTmp eq "" ){ $sTextTmp = $a2wObjTmp->getText(); }
            $this->_printMessage( sprintf( $sIndentTmp . "txt>%s<@(%d,%d) ISeqId=%d, TagId=%d, TCId=%08X", $sTextTmp, $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS }, $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS }, $a2wObjTmp->getSequenceId(), $a2wObjTmp->getTagId(), $iTCIdTmp ) );
        }
        elsif ( $sObjTypeTmp eq "image" ){
            $this->_printMessage( sprintf( $sIndentTmp . "img>%s<@(%d,%d) W=%d, H=%d, ISeqId=%d, TagId=%d, TCId=%08X", $a2wObjTmp->getName(), $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS }, $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS }, $a2wObjTmp->getWidth(), $a2wObjTmp->getHeight(), $a2wObjTmp->getSequenceId(), $a2wObjTmp->getTagId(), $iTCIdTmp ) );
        }
        elsif ( $sObjTypeTmp eq "container" ){
            $this->_printMessage( sprintf( $sIndentTmp . "obj>%s<@(%d,%d) W=%d, H=%d, ISeqId=%d, TagId=%d, TCId=%08X", $a2wObjTmp->getName(), $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS }, $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS }, $a2wObjTmp->getWidth(), $a2wObjTmp->getHeight(), $a2wObjTmp->getSequenceId(), $a2wObjTmp->getTagId(), $iTCIdTmp ) );
        }
        elsif ( $sObjTypeTmp eq "line" ){
            $this->_printMessage( sprintf( $sIndentTmp . "lin><@(%d,%d) W=%d, L=%d, ISeqId=%d, TagId=%d, TCId=%08X", $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS }, $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS }, $a2wObjTmp->getWidth(), $a2wObjTmp->getLength(), $a2wObjTmp->getSequenceId(), $a2wObjTmp->getTagId(), $iTCIdTmp ) );
        }

        #---- Write ending of element
        if ( $iTagCategoryTmp == $a2w::ContentBlockConstants::BLOCK_TYPE_LIST ){
            if ( ( $iTagMarkTmp & 0x008000 ) != 0 && ( $iTagElemTmp & 0x08 ) != 0 ){ # Item body
                $sIndentTmp = " " x 2;
                $this->_printMessage( $sIndentTmp . "<body" );
            }
            if ( ( $iTagMarkTmp & 0x002000 ) != 0 && ( $iTagElemTmp & 0x04 ) != 0 ){ # Item label
                $sIndentTmp = " " x 2;
                $this->_printMessage( $sIndentTmp . "<lbl" );
            }
            if ( ( $iTagMarkTmp & 0x000800 ) != 0 && ( $iTagElemTmp & 0x02 ) != 0 ){ # List item
                $sIndentTmp = " " x 1;
                $this->_printMessage( $sIndentTmp . "<item" );
            }
            if ( ( $iTagMarkTmp & 0x000200 ) != 0 && ( $iTagElemTmp & 0x01 ) != 0 ){ # List
                $this->_printMessage( "<list" );
            }
        } # if ( $iTagCategoryTmp == $a2w::ContentBlockConstants::BLOCK_TYPE_LIST )
    } # foreach my $elem ( @arrTableObjsTmp )
}
# $V110 End

#-----------------------------------------------------------------------
# Print to output dump
#-----------------------------------------------------------------------
sub _printMessage{
    my $this = shift;

    #---- Get parameter
    #
    # 1. Message to be printed
    #
    my $sMsgPar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logMessage( $sMsgPar ); }

    #---- Get output file handle
    my $handleFileTmp = $this->{ 'fHandle' };
    if ( $handleFileTmp == undef ){ return; } # do nothing, output is configured to be skipped (refer 'SkipOutput' flag of visitor)

    #---- Print message to output
    printf $handleFileTmp ( $sMsgPar . "\n" );
}

# V108 Begin
#-----------------------------------------------------------------------
# Create content block
#
#-----------------------------------------------------------------------
sub _createContentBlock{
    my $this = shift;

    #---- Fetch parameter(s)
    #
    # 1. Block reference
    #
    my $blkActivePar = shift;

    if ( $blkActivePar == undef ){ return undef; }
	
    my $a2wContentBlockTmp = new a2w::ContentBlock();
    if ( $a2wContentBlockTmp == undef ){ return undef; }

    #---- Fill in content block info
    $a2wContentBlockTmp->setId( $blkActivePar->getId() );
    my $sBlkTypeTmp = $blkActivePar->getContentDefType();
    my $eBlkTypeTmp = 0;
    if ( lc( $sBlkTypeTmp ) eq "paragraph" ){
        $eBlkTypeTmp = $a2w::ContentBlockConstants::BLOCK_TYPE_PARAGRAPH >> 24;
        $a2wContentBlockTmp->setType( $eBlkTypeTmp );
    }
    elsif ( lc( $sBlkTypeTmp ) eq "table" ){
        $eBlkTypeTmp = $a2w::ContentBlockConstants::BLOCK_TYPE_TABLE >> 24;
        $a2wContentBlockTmp->setType( $eBlkTypeTmp );
    }
    elsif ( lc( $sBlkTypeTmp ) eq "list" ){
        $eBlkTypeTmp = $a2w::ContentBlockConstants::BLOCK_TYPE_LIST >> 24;
        $a2wContentBlockTmp->setType( $eBlkTypeTmp );
    }
    elsif ( lc( $sBlkTypeTmp ) eq "illustration" ){
        $eBlkTypeTmp = $a2w::ContentBlockConstants::BLOCK_TYPE_ILLUSTRATION >> 24;
        $a2wContentBlockTmp->setType( $eBlkTypeTmp );
    }

    # $V101 Begin
    #---- Get and add alternate text property to block (if exists)
    my $hrefBlockInfoTmp = $blkActivePar->getJSONInfo();
    # $V106 Begin
    if ( $hrefBlockInfoTmp != undef ){
        if ( $hrefBlockInfoTmp->{ 'alt' } ne "" ){
            my $sAltTextTmp = $hrefBlockInfoTmp->{ 'alt' };
            $a2wContentBlockTmp->addProperty( $a2w::ContentBlockConstants::BLOCK_PROP_ALTERNATE_TEXT, $sAltTextTmp );
            if ( $bLog == $TRUE ){ $theLogger->logMessage( "Added property \"" . $a2w::ContentBlockConstants::BLOCK_PROP_ALTERNATE_TEXT . "=" . $sAltTextTmp . "\" to block (" . $blkActivePar->getId() . ")" ); }
        }
        if ( $hrefBlockInfoTmp->{ 'headerLevel' } ne "" ){
            my $sHLTextTmp = $hrefBlockInfoTmp->{ 'headerLevel' };
            $a2wContentBlockTmp->addProperty( $a2w::ContentBlockConstants::BLOCK_PROP_HEADER_LEVEL, $sHLTextTmp );
            if ( $bLog == $TRUE ){ $theLogger->logMessage( "Added property \"" . $a2w::ContentBlockConstants::BLOCK_PROP_HEADER_LEVEL . "=" . $sHLTextTmp . "\" to block (" . $blkActivePar->getId() . ")" ); }
        }
    # $V106 End
    }
    # $V101 End

    return $a2wContentBlockTmp;
}

#-----------------------------------------------------------------------
# Set tag id
#
# Set current tag id based on object's page id
#
#-----------------------------------------------------------------------
sub _setTagId{
    my $this = shift;

    #---- Fetch parameter(s)
    #
    # 1. Object hash (having both kernel object and pom object)
    #
    my $hrefObjPar = shift;

    #---- Get actual objects
    my $a2wObjTmp = $hrefObjPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefObjPar->{ 'POMOBJ' };

    #---- Get page id of object
    my $iObjPageIdTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_PAGE_ID };
    if ( $this->{ 'ActiveTagPage' } != $iObjPageIdTmp ){ $this->{ 'ActiveTagPage' } = $iObjPageIdTmp; }

    #---- Get last tag id of object page and set it active tag id on visitor
    $this->{ 'TagId' } = $this->{ 'Pages' }{ $iObjPageIdTmp }{ 'TagId' };
}

#-----------------------------------------------------------------------
# Increment tag id
#
# Increment current tag id based on object's page id
#
#-----------------------------------------------------------------------
sub _incrementTagId{
    my $this = shift;

    #---- Fetch parameter(s)
    #
    # 1. Object hash (having both kernel object and pom object)
    #
    my $hrefObjPar = shift;

    #---- Get actual objects
    my $a2wObjTmp = $hrefObjPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefObjPar->{ 'POMOBJ' };

    #---- Get page id of object
    my $iObjPageIdTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_PAGE_ID };

    #---- Increment the tag id of object page
    $this->{ 'Pages' }{ $iObjPageIdTmp }{ 'TagId' } += 1;

    #---- Get last tag id of object page and set it active tag id on visitor
    $this->{ 'TagId' } = $this->{ 'Pages' }{ $iObjPageIdTmp }{ 'TagId' };
}
# V108 End

# V109 Begin
#-----------------------------------------------------------------------
# _processBlockObjects
#
# Process block objects and add them to core
#
#-----------------------------------------------------------------------
sub _processBlockObjects{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "_processBlockObjects()" ); }

    #---- Assert required parameters
    if ( $this->{ 'BlockObjectsCount' } <= 0 ){ return 0; }

    # V108 Begin
    #---- Process and add objects to core ----#
    #---- Get active page context
    my $pgHandleTmp = $this->{ 'ActivePage' };

    #---- Get active block
    my $blkActiveTmp = $pgHandleTmp->{ 'ActiveBlock' };

    #---- Get page handle
    my $iObjPageIdTmp = 0;
    my $dbPageTmp = undef;
    my $a2wCntBlkTmp = undef;
    my $hrefPageCntBlkTmp = {}; # hash of <page id> => <a2w::ContentBlock instance>

    #---- Add objects to content block
    my $a2wObjTmp = undef;
    my @arrBlkObjsTmp = @{ $this->{ 'BlockObjects' } };
    foreach my $o ( @arrBlkObjsTmp ){
        #---- Get core object
        $a2wObjTmp = $o->{ 'A2WOBJ' };
        if ( $a2wObjTmp == undef ){ next; }

        #---- Get page handle based on object's page id
        $iObjPageIdTmp = $o->{ 'POMOBJ' }{ $a2w::core::dm::Constants::AT_PAGE_ID };
        $dbPageTmp = $this->{ 'Pages' }{ $iObjPageIdTmp }{ 'POM' };

        #---- Check and create content block of object's page
        $a2wCntBlkTmp = $hrefPageCntBlkTmp->{ $iObjPageIdTmp };
        if ( $a2wCntBlkTmp == undef ){
            $a2wCntBlkTmp = $this->_createContentBlock( $blkActiveTmp );
            $hrefPageCntBlkTmp->{ $iObjPageIdTmp } = $a2wCntBlkTmp;
        }

        # V107 Begin
        #---- Add object to content block
        # NOTE:
        # Adding object to content ensures the tag info (id, classification id) is stored in content block
        #
        $a2wCntBlkTmp->addObject(   $a2wObjTmp
                                  , $o->{ 'POMOBJ' }{ $a2w::core::dm::Constants::AT_ID }
                                );

        #---- Reset tag id and classification info
        $a2wObjTmp->setTagId( -1 );
        $a2wObjTmp->setTagClassificationId( 0 );
        # V107 End
    }

    #---- Process and add content blocks to appropriate page
    $dbPageTmp = undef;
    $a2wCntBlkTmp = undef;
    my $a2wPageTmp = undef;
    my @arrPageIdsTmp = sort keys( %{ $hrefPageCntBlkTmp } );
    foreach my $iPgIdTmp ( @arrPageIdsTmp ){
        #---- Assert content block is created or not
        $a2wCntBlkTmp = $hrefPageCntBlkTmp->{ $iPgIdTmp };
        if ( $a2wCntBlkTmp == undef ){ next; }

        #---- Fetch page handle
        $dbPageTmp = $this->{ 'Pages' }{ $iPgIdTmp }{ 'POM' };
        if ( $dbPageTmp == undef ){ next; }

        #---- Add content block to page
        $a2wPageTmp = $dbPageTmp->getPageReference();
        $a2wPageTmp->addContentBlock( $a2wCntBlkTmp );
    }
    # V108 End

    return 0;
}
# V109 End

#-----------------------------------------------------------------------
# Don't remove the following lines !!!
#-----------------------------------------------------------------------
1;
__END__
