#-------------------------------------------------------------------------------
#  a2w::core::visitor::HTMLVisitor.pm:
#  PERL module to write the objects as HTML
#
#  Author  : AFP2web Team, Maas Holding GmbH
#
#  V100   2013-10-11    Initial Release
#
#  V101   2015-05-20    Fixed line overlapping image (within block) issue (AFP-224)
#
#  V102   2015-07-17    Extended to inline CSS and image data (AFP-263)
#
#  V103   2015-09-28    - Extended to group texts based on font (identifier) (AFP-297)
#                       - Skipped writing empty <div> for page, when page has no content
#                       - Modified following page's top margin as 5mm (from 10mm)
#
#  V104   2015-10-07    Issue:
#                       Table cell having multi line value occur now in multiple rows
#
#                       Reason:
#                       Table rows are identified based on change in Y position, so each line of
#                       same cell is treated as different row.
#
#                       Fix:
#                       Take first column as primary and mark beginning of a new row whenever the
#                       first column value on line is not empty. i.e, Following lines of a cell must
#                       have empty first column value on line
#
#                       JiRa:
#                       AFP-224
#
#                       Fixed by, Date:
#                       Panneer, 07/10/2015
#
#  V105   2015-10-15    Extended to pass in custom block info like HTML style (AFP-298)
#
#  V106   2015-11-04    - Extended to position content blocks absolute
#                       - Removed line breaks
#                       - Extended with "getTableHeight" function
#
#  V107   2015-11-16    - Fixed minor disposition (due to default margin) issue with lines. To fix, Forced line margin as zero. (AFP-320)
#                       - Fixed minor disposition issue with texts. To fix, Adjusted position based on text applied font height. (AFP-320)
#
#  V108   2015-12-08    - Forced table first column vertical alignment as "top" (AFP-320)
#
#  V109   2015-12-09    - Fixed minor disposition (due to missing page margins). (AFP-320)
#                       - Fixed minor disposition (due to skipped block height not considered for following block top). (AFP-320)
#
#  V110   2015-12-11    - Modified not to escape space (' ') character as HTML entity (&nbsp;) (AFP-320)
#
#  V111   2016-01-18    Customized to wrap block images with <a> tag when block info (TLE value) has "href=<url>" (AFP-342)
#
#  V112   2016-09-09    Extended to scale down images (by 1/4 of size) to optimize output size
#
#  V113   2016-09-19    Extended to scale down images (by presentation size) to optimize output size
#
#  V114   2018-06-28    Fixed minor bug in positioning block that follows skipped block
#
#  V115   2018-08-10    Extended to write tagging info at document level instead of page level
#
#  V116   2020-03-05    a. AFP-929: Extended to write container objects
#                       b. AFP-929: Extended to render solid rectangle when AFP IM Raster/IOCA image data is undefined
#                       c. AFP-929: Fixed minor disposition issue with texts due to missing baseline adjustment
#                       d. AFP-929: Extended to map font names using configuration
#                       e. AFP-929: Extended to generate HTML5 compatible output
#                       f. AFP-929: Extended to scale the page view automatically
#                       g. AFP-929: Fixed minor bug in object writing sequence as defined in input
#                       h. AFP-929: Fixed minor bug in evaluating page top for single page outputs
#                       i. AFP-929: Minimized html content (i.e, page elements are NOT separated using CRLF, removed unnecessary spaces)
#
#  V117   2020-03-30    AFP-929: Fixed minor bug in "fit to window" JS customization
#
#  V118   2020-04-15    AFP-941: Extended to get additional font attributes from afpcp/html.fontmap.json
#
#  V119   2020-05-25    AFP-954: Fixed minor bug in "fit to window" JS customization for MS Edge browser compatibility
#
#  V120   2020-07-01    a. AFP-951: Extended with write annotation object
#                       b. AFP-951: Modified writing formatted HTML string to file
#                          i.e, Used 'print' instead 'printf' to avoid error when
#                          HTML content has printf formatter like values (ex: %20)
#                          Such value occur when any URL parameter is MIME encoded
#                       c. AFP-951: Extended to handle the foreground/background order
#                          of SF added objects
#
#  V121   2021-01-28    AFP-1012: Extended to not scale down AFP container images
#
#  V122   2021-02-25    AFP-1023: Extended to set default body margin and padding to avoid
#                       page end content overlapped by following page start
#
#  V123   2021-03-18    a. AFP-1028: Extended to leave gap between page contents based on configurable value
#                       b. AFP-1028: Extended to configure image compression for HTML output to gain performance
#
#  V124   2021-06-01    a. AFP-1046: Extended with output config to have block specific config, content to be embedded etc in output
#                       b. AFP-1046: Extended to sort page blocks (as needed by output visitor)
#                       c. AFP-1046: Extended to add 'id' attribute to text SPAN elements within block "div"
#                                    Turned on using "<Block>->html->generateIDs" option
#                       d. AFP-1046: Extended to generate responsive html content
#
#  V125   2025-07-30    AFP-1255: Info Image Inc: Extended to handle image rotation
#
#-------------------------------------------------------------------------------
package a2w::core::visitor::HTMLVisitor;

# V116e Begin
#-----------------------------------------------------------------------
# Define required constants
#-----------------------------------------------------------------------
# Constant for HTML version, possible values are 0 or 5
$HTML_VERSION_0 = 0;
$HTML_VERSION_5 = 5;
# V116e End

#-----------------------------------------------------------------------
# Include required modules
#-----------------------------------------------------------------------
# V116a Begin
use a2w::Container;
use a2w::ContainerConstants;
# V116a End
use a2w::Font;
use a2w::Text;
use a2w::Line;
use a2w::Image;
use a2w::ImageConstants; # V116b Change
use a2w::Vector;

use a2w::TypeConstants;
use a2w::core::log::Logger;
use a2w::core::dm::Constants;
use a2w::core::visitor::Visitor;
# $V102 Begin
use a2w::core::utils::Base64;
# $V102 End

#-----------------------------------------------------------------------
# Inherit from base class
#-----------------------------------------------------------------------
our @ISA = qw( a2w::core::visitor::Visitor );

#-----------------------------------------------------------------------
# Constructor
#-----------------------------------------------------------------------
sub new{
    my $proto = shift;
    my $class = ref( $proto ) || $proto;

    #---- Define boolean values
    $TRUE  = $a2w::TypeConstants::TRUE;     # TRUE  boolean value
    $FALSE = $a2w::TypeConstants::FALSE;    # FALSE boolean value

    #---- Instantiate from base class to inherit base class attributes
    my $this = a2w::core::visitor::Visitor->new( @_ );

    #---- Add this derived class specific attributes
    $this->{ 'BackgroundFilename' } = "";        # Background filename
    $this->{ 'FontTable' }          = undef;     # Font table
    $this->{ 'BODY' }               = undef;     # Body of html
    $this->{ 'BPG_TAG' }            = undef;     # Page begin tag # $V103 Change
    $this->{ 'Content' }            = undef;     # Content of page
    $this->{ 'FontScale' }          = 0;         # Font scale factor
    $this->{ 'UnitBase' }           = "";        # Units base (possible values: pixel, mm)
    $this->{ 'PrevFontId' }         = 0;         # Previous font local id
    $this->{ 'CSS' }                = undef;     # Cascaded style sheet for current document (Hash having css file name, handle, style classes etc)
    $this->{ 'ResPaths' }           = undef;     # Hash of resource paths for image, css etc
    # $V102 Begin
    $this->{ 'bInlineImage' }       = $FALSE;    # Flag to inline image or not
    $this->{ 'bInlineCSS' }         = $FALSE;    # Flag to inline CSS or not
    # $V102 End
    # $V106 Begin
    $this->{ 'TableHdrHeight' }     = 0;         # Active Table header height
    $this->{ 'TableRowHeight' }     = 0;         # Active Table row height
    $this->{ 'TableRowSpan' }       = 0;         # Active Table row span count (multi line cell)
    $this->{ 'TableCellPadding' }   = 2;         # Table cell padding
    $this->{ 'BlockTop' }           = 0;         # Current block top
    $this->{ 'BlockTopDiff' }       = 0;         # Block top difference
    $this->{ 'PrevBlock' }          = undef;     # Previous block
    $this->{ 'FirstBlock' }         = undef;     # First block of page
    # $V106 End
    $this->{ 'HDR' }                = undef;     # Additional header tags to be added in <head> element # V116d Change
    $this->{ 'bInlineHDR' }         = $FALSE;    # Flag to inline additional header or not              # V116d Change
    $this->{ 'FontLocations' }      = {};        # Font locations                                       # V116d Change
    $this->{ 'HTMLVersion' }        = $HTML_VERSION_5; # HTML version                                   # V116e Change
    $this->{ 'AutoPageScale' }      = $FALSE;    # Automatically scale the page view or not             # V116f Change
    $this->{ 'Minimize' }           = $FALSE;    # Minimize page content                                # V116i Change
    $this->{ 'ImageFormat' }        = 'png';     # Image format                                         # V123b Change

    $this->{ 'BlockTextId' }        = 0;         # Id of text from current block being written          # V124c Change
    $this->{ 'ResponsiveContent' }  = '';        # Responsive content of html                           # V124d Change
    $this->{ 'ResCache' }           = undef;     # Cache of document resources                          # V124d Change

    bless( $this, $class );

    #---- Initialize constants ----#
    $this->{ 'ResPaths' } = {
          'CSS'   => 'css/'
        , 'IMAGE' => 'images/'
        , 'VIDEO' => 'videos/'
        , 'HDR'   => 'hdr/' # V116d Change
    };

    # $V102 Begin
    #--- Force inlining image, css, header and auto page scaling
    $this->{ 'bInlineImage' }  = $TRUE;
    $this->{ 'bInlineCSS' }    = $TRUE;
    # $V102 End
    $this->{ 'bInlineHDR' }    = $TRUE; # V116d Change
    $this->{ 'AutoPageScale' } = $TRUE; # V116f Change
    $this->{ 'Minimize' }      = $TRUE; # V116i Change

    #---- Get logger
    our $theLogger = a2w::core::log::Logger->getSingleton();
    our $bLog = $theLogger->isRegistered( __PACKAGE__ );

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "new()" );
    #}
    return $this;
}

#-----------------------------------------------------------------------
# Destructor
#-----------------------------------------------------------------------
sub DESTROY{
    my $this = shift;
}

#-----------------------------------------------------------------------
# Mutator
#-----------------------------------------------------------------------
sub setBackgroundFilename{
    my $this = shift;

    $this->{ 'BackgroundFilename' } = shift;
}

sub setUnitBase{
    my $this = shift;

    $this->{ 'UnitBase' } = shift;
    if ( lc( $this->{ 'UnitBase' } ) eq "pixel" ){
        $this->{ 'UnitBase' } = "px";
    }
    elsif ( lc( $this->{ 'UnitBase' } ) eq "mm" ){
        $this->{ 'UnitBase' } = "mm";
    }
    else {
        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "Warning! Invalid unit base (" . $this->{ 'UnitBase' } . ") using default px" );
        }
        $this->{ 'UnitBase' } = "px";
    }
}

#-----------------------------------------------------------------------
# Accessor
#-----------------------------------------------------------------------
sub getBackgroundFilename{
    my $this = shift;

    return $this->{ 'BackgroundFilename' };
}

sub getUnitBase{
    my $this = shift;

    return $this->{ 'UnitBase' };
}

#-----------------------------------------------------------------------
# Workers
#-----------------------------------------------------------------------
#-----------------------------------------------------------------------
# Initialize
#-----------------------------------------------------------------------
sub initialize{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "initialize()" );
    }

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::initialize( @_ );
    if ( $iRetTmp < 0 ){
        return $iRetTmp;
    }

    #---- Initialize CSS object
    $this->{ 'CSS' } = {
          'Filename' => ''
        , 'fHandle'  => undef
        , 'Styles'   => undef
        , 'FontId'   => undef
    };

    # V116d Begin
    $this->{ 'HDR' } = {
          'Filename' => ''
        , 'fHandle'  => undef
    };
    # V116d End

    # V123b Begin
    #---- Check and set image format
    my $hrefOutConfTmp = $this->getOutputConfig();
    if ( $hrefOutConfTmp != undef ){
        if (    $hrefOutConfTmp->{ 'imageFormat' } ne ""
             && lc( $hrefOutConfTmp->{ 'imageFormat' } ) ne 'png'
           ){
            $this->{ 'ImageFormat' } = lc( $hrefOutConfTmp->{ 'imageFormat' } );
            if ( $bLog == $TRUE ){ $theLogger->logMessage( "Image format set to " . $this->{ 'ImageFormat' } ); }
        }
    }
    # V123b End

    if ( $this->{ 'PageOutput' } == $FALSE ){
        $this->_initializeHTML();
    }

    return 0;
}

#-----------------------------------------------------------------------
# Finalize
#-----------------------------------------------------------------------
sub finalize{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "finalize()" );
    }

    #---- Fetch parameters
    #
    # 1. Additional content for body end (optional)
    # 2. Additional content for footer (optional)
    #
    my $sBodyAddContentPar   = "";
    my $sAddFooterContentPar = "";
    if ( @_ > 0 ){
        $sBodyAddContentPar = shift;
    }
    if ( @_ > 0 ){
        $sAddFooterContentPar = shift;
    }

    if ( $this->{ 'PageOutput' } == $FALSE ){
        #---- End body
        my $handleFileTmp = $this->{ 'fHandle' };
        if ( $sBodyAddContentPar ne "" ){
            print $handleFileTmp ( "\n" . $sBodyAddContentPar ); # V120b Change
        }
        # $V106 Begin
        printf $handleFileTmp ( "\n" . '</div>' ); # desktop_container closure
        # $V106 End

        # V124d Begin
        # Minimize page content elements separation (i.e, elements not separated by CRLF)
        if ( $this->{ 'Minimize' } == $TRUE ){
            $this->{ 'ResponsiveContent' } =~ s/[\r\n]+//g;
        }

        # Add responsive container
        print $handleFileTmp (   "\n" . '<div id="responsive_container">'
                               . "\n" . $this->{ 'ResponsiveContent' }
                               . "\n" . '</div>'
                             );
        # V124d End

        printf $handleFileTmp ( "\n" . '</body>' );

        #---- Write footer
        $this->writeFooter( # Additional content
                            $sAddFooterContentPar
                          );

        $this->_finalizeHTML();
    }

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::finalize( @_ );
    if ( $iRetTmp < 0 ){
        return $iRetTmp;
    }

    #---- Clean up image/css sub directories
    if ( $this->{ 'bInlineImage' } == $TRUE ){
        my $sImgOutputPathTmp = $this->{ 'OutputPath' } . $this->{ 'ResPaths' }{ 'IMAGE' };
        rmdir( $sImgOutputPathTmp );
    }
    # V116d Begin
    if ( $this->{ 'bInlineHDR' } == $TRUE ){
        my $sHDROutputPathTmp = $this->{ 'OutputPath' } . $this->{ 'ResPaths' }{ 'HDR' };
        rmdir( $sHDROutputPathTmp );
    }
    # V116d End
    if ( $this->{ 'bInlineCSS' } == $TRUE ){
        my $sCssOutputPathTmp = $this->{ 'OutputPath' } . $this->{ 'ResPaths' }{ 'CSS' };
        rmdir( $sCssOutputPathTmp );
    }
    
    return 0;
}

#-----------------------------------------------------------------------
# Write header
#-----------------------------------------------------------------------
sub writeHeader{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeHeader()" );
    }

    #---- Parameter(s)
    #
    # 1. Title
    # 2. Additional content
    #
    my $sTitlePar      = shift;
    my $sAddContentPar = shift;

    #---- Write HTML header
    my $handleFileTmp = $this->{ 'fHandle' };
    printf $handleFileTmp (   "\n" . '<head>'
                            . "\n" . '<title>' . $sTitlePar . '</title>'
                            . "\n" . '<meta http-equiv="Content-Type" content="text/html; charset=' . (( $this->{ 'HTMLVersion' } == $HTML_VERSION_5 ) ? 'utf-8' : 'Windows-1252') . '">' # V116e Change
                            . "\n" # V124a Change
                          );

    # V116f Begin
    my $iRetTmp = 0;
    if ( $this->{ 'AutoPageScale' } == $TRUE ){
        $iRetTmp = $this->_includeAutoPageScale();
        if ( $iRetTmp < 0 ){ return $iRetTmp; }
    }
    # V116f End

    # V124d Begin
    $iRetTmp = $this->_includeResponsiveJS();
    if ( $iRetTmp < 0 ){ return $iRetTmp; }
    # V124d End

    # V116d Begin
    #---- Include additional header place holder ----#
    if ( $this->{ 'HDR' }{ 'fHandle' } != undef ){
        $this->{ 'HDR' }{ 'IncludeStatement' } = '<!-- PLACEHOLDER FOR ADDITIONAL HEADER CONTENT -->';
        print $handleFileTmp ( $this->{ 'HDR' }{ 'IncludeStatement' } ); # V120b Change
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Added header place holder:>" . $this->{ 'HDR' }{ 'IncludeStatement' } . "<" ); }
    }
    # V116d End

    #---- Create CSS ----#
    if ( $this->{ 'CSS' }{ 'fHandle' } != undef ){
        $iRetTmp = $this->_writeCSSHeader();
        my $CSSFilenameTmp = $this->{ 'CSS' }{ 'Filename' };
        if ( $CSSFilenameTmp =~ /.*\/(css\/.*)/ ){
            $CSSFilenameTmp = $1;
        }
        printf $handleFileTmp ( "\n" . '<link rel="stylesheet" type="text/css" href="' . $CSSFilenameTmp . '" />' );
        # $V102 Begin
        $this->{ 'CSS' }{ 'IncludeStatement' } = '<link rel="stylesheet" type="text/css" href="' . $CSSFilenameTmp . '" />';
        # $V102 End
    }

    if ( $sAddContentPar ne "" ){
        print $handleFileTmp ( "\n" . $sAddContentPar ); # V120b Change
    }
    printf $handleFileTmp ( "\n" . '</head>' );

    return 0;
}

#-----------------------------------------------------------------------
# Write footer
#-----------------------------------------------------------------------
sub writeFooter{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeFooter()" );
    }

    #---- Parameter(s)
    #
    # 1. Additional content
    #
    my $sAddContentPar = shift;
    if ( $sAddContentPar eq "" ){
        return 0;
    }

    #---- Write HTML footer
    my $handleFileTmp = $this->{ 'fHandle' };
    printf $handleFileTmp ( "\n" . '<footer>' );
    if ( $sAddContentPar ne "" ){
        print $handleFileTmp ( "\n" . $sAddContentPar ); # V120b Change
    }
    printf $handleFileTmp ( "\n" . '</footer>' );

    return 0;
}

# V124b Begin
#-----------------------------------------------------------------------
# Sort page blocks
#-----------------------------------------------------------------------
sub sortPageBlocks{
    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "sortPageBlocks()" ); }

    #---- Get parameter
    #
    # 1. Page blocks (array reference)
    #
    my $arefPageBlocksPar = shift;

    #---- Iterate through blocks and assert whether 'Order' is specified on output config
    my @arrPgBlksTmp = @{ $arefPageBlocksPar };
    my @arrBlkOrderTmp = ();
    my $hrefOutConfTmp = undef;
    my $bOrderMissingTmp = $FALSE;
    my $hrefOrderedBlksTmp = {};
    foreach my $b ( @arrPgBlksTmp ){
        $hrefOutConfTmp = $b->getOutputConfig( 'html' );
        if ( $hrefOutConfTmp != undef && defined( $hrefOutConfTmp->{ 'Order' } ) ){
            push( @arrBlkOrderTmp, $hrefOutConfTmp->{ 'Order' } );
            $hrefOrderedBlksTmp->{ $hrefOutConfTmp->{ 'Order' } } = $b;
        }
        else {
            $bOrderMissingTmp = $TRUE;
            push( @arrBlkOrderTmp, -1 );
        }
    }

    #---- Ignore sorting, if the order is missing for any block
    if ( $bOrderMissingTmp == $TRUE ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Order not found for all the page blocks, Sorting ignored." ); }
        return $arefPageBlocksPar;
    }

    #---- Sort blocks
    @arrBlkOrderTmp = sort( @arrBlkOrderTmp );
    my @arrOrderedBlocksTmp = ();
    foreach my $o ( @arrBlkOrderTmp ){
        push( @arrOrderedBlocksTmp, $hrefOrderedBlksTmp->{ $o } );
    }

    return \@arrOrderedBlocksTmp;
}
# V124b End

#-----------------------------------------------------------------------
# Initialize Page
#-----------------------------------------------------------------------
sub initializePage{
    my $this = shift;

    #---- Fetch parameters
    #
    # 1. Database
    # 2. Title
    # 3. Additional content for header (optional)
    # 4. Additional content for before body (optional)
    #
    my $dbPagePar = shift;
    my $sTitlePar = shift;
    my $sAddHeaderContentPar = "";
    my $sAddBodyContentPar   = "";
    if ( @_ > 0 ){
        $sAddHeaderContentPar = shift;
    }
    if ( @_ > 0 ){
        $sAddBodyContentPar = shift;
    }
    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "initializePage(" . $dbPagePar->getPageID() . ")" );
    }

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::initializePage( $dbPagePar, $sAddHeaderContentPar );
    if ( $iRetTmp < 0 ){
        return $iRetTmp;
    }

    #---- Evaluate page width/height ----#
    #---- Evaluate scale factors
    my $iOutputResTmp         = $this->getOutputRes();
    $this->{ 'ContentScale' } = $iOutputResTmp / $dbPagePar->getPageRes();
    $this->{ 'FontScale' }    = $iOutputResTmp / 72;
    if ( $bLog == $TRUE ){
        $theLogger->logMessage( "Scale: Content=" . $this->{ 'ContentScale' } . " Font=" . $this->{ 'FontScale' } );
    }

    #---- Fetch page info
    my $iPageIdTmp     = $dbPagePar->getPageID();
    my $iPageWidthTmp  = $dbPagePar->getPageWidth();
    my $iPageHeightTmp = $dbPagePar->getPageHeight();
    if ( $this->{ 'UnitBase' } eq "px" ){
        $iPageWidthTmp  *= $this->{ 'ContentScale' };
        $iPageHeightTmp *= $this->{ 'ContentScale' };
    }

    #---- Activate current page
    $this->activatePage( $iPageIdTmp );

    #---- Store the page info on page context
    my $pgHandleTmp = $this->{ 'ActivePage' };
    $this->{ 'PrevFontId' } = 0;

    $pgHandleTmp->{ 'ContentScale' } = $this->{ 'ContentScale' };
    $pgHandleTmp->{ 'Width' }        = int( $iPageWidthTmp  + .5 * ( $iPageWidthTmp  <=> 0 ) );
    $pgHandleTmp->{ 'Height' }       = int( $iPageHeightTmp + .5 * ( $iPageHeightTmp <=> 0 ) );

    my $sPageBackgroundTmp = $dbPagePar->getPageBackground();
    if ( $sPageBackgroundTmp ne "" ){
        $this->setBackgroundFilename( $sPageBackgroundTmp );
    }

    if ( $this->{ 'PageOutput' } == $TRUE ){
        #---- Write html start
        $this->_initializeHTML();

        #---- Write header
        $this->writeHeader( # Title
                              $sTitlePar
                            # Additional content
                            , $sAddHeaderContentPar
                          );

        $this->{ 'ResponsiveContent' } = ''; # V124d Change
        $this->{ 'BODY' } =   "\n"
                            . '<body>'
                            # $V106 Begin
                            . "\n"
                            . '<div id="desktop_container">'
                            # $V106 End
                            ;
        if ( $sAddBodyContentPar ne "" ){
            $this->{ 'BODY' } .= "\n" . $sAddBodyContentPar;
        }
    }
    else {
        if ( $this->{ 'PageCount' } == 1 ){
            #---- Write header
            $this->writeHeader( # Title
                                  $sTitlePar
                                # Additional content
                                , $sAddHeaderContentPar
                              );

            #---- Start body
            my $handleFileTmp = $this->{ 'fHandle' };
            printf $handleFileTmp (   "\n"
                                    . '<body>'
                                    # $V106 Begin
                                    . "\n"
                                    . '<div id="desktop_container">'
                                    # $V106 End
                                  );

            if ( $sAddBodyContentPar ne "" ){
                print $handleFileTmp ( "\n" . $sAddBodyContentPar ); # V120b Change
            }
        }
        $this->{ 'BODY' } = "";
    }

    # $V106 Begin
    # V116h Begin
    #---- Get page top
    my $iTopTmp = 0;
    if ( $this->{ 'PageOutput' } == $FALSE ){
        $iTopTmp = $this->_getPageTop( $iPageIdTmp ); # V123b Change
    }
    if ( $iTopTmp > 0 && $this->{ 'UnitBase' } eq "px" ){ $iTopTmp *= $this->{ 'ContentScale' }; }

    # V123b Begin
    #---- Apply page gap (ensured to apply after scaling)
    if ( $this->{ 'FollowingPageTopMargin' } > 0 ){
        my $iPageGapTmp = ( $this->{ 'FollowingPageTopMargin' } / 25.4 ) * $iOutputResTmp;
        $iTopTmp += $iPageGapTmp;
    }
    # V123b End

    $iTopTmp = int( $iTopTmp + .5 * ( $iTopTmp <=> 0 ) );
    # V116h End

    $this->{ 'BlockTop' }     = 0;
    $this->{ 'BlockTopDiff' } = 0;
    $this->{ 'PrevBlock' }    = undef;
    $this->{ 'FirstBlock' }   = undef;
    # $V106 End

    #---- Start page div ----#
    $this->{ 'BPG_TAG' } =  "\n"
                            . '<div'
                            . ' id="page' . $this->{ 'PageCount' } . '"'
                            # $V106 Begin
                            . ' class="' . $this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' } . '"'
                            # $V106 End
                            . ' style="background-color:#' . sprintf( "%0X", $this->getDefBackColor() ) . ';'
                            # $V106 Begin
                            . (( $this->{ 'PageCount' } > 1 && $iTopTmp > 0 ) ? 'top:' . $iTopTmp . $this->{ 'UnitBase' } . ';' : "")
                            #. 'position:relative;'
                            # $V106 End
                            . 'width:' . $pgHandleTmp->{ 'Width' } . $this->{ 'UnitBase' } . ';'
                            #. 'height:' . $pgHandleTmp->{ 'Height' } . $this->{ 'UnitBase' } . ';'
                            . '">';
    # $V103 End

    $this->{ 'Content' } = "";
    $this->{ 'FontTable' } = undef;

    return 0;
}

#-----------------------------------------------------------------------
# Finalize page
#-----------------------------------------------------------------------
sub finalizePage{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "finalizePage()" );
    }

    #---- Fetch parameters
    #
    # 1. Additional content for body end (optional)
    # 2. Additional content for footer (optional)
    #
    my $sBodyAddContentPar   = "";
    my $sAddFooterContentPar = "";
    if ( @_ > 0 ){
        $sBodyAddContentPar = shift;
    }
    if ( @_ > 0 ){
        $sAddFooterContentPar = shift;
    }
    if ( $bLog == $TRUE ){
        $theLogger->logMessage( "Additional Content: Body=>" . $sBodyAddContentPar . "< Footer=>" . $sAddFooterContentPar . "<" );
    }

    #---- Build HTML body ----#
    #---- Get active page context
    my $pgHandleTmp = $this->{ 'ActivePage' };

    #---- Get active page
    my $dbPageTmp = $pgHandleTmp->{ 'POM' };

    # $V103 Begin
    if ( $this->{ 'Content' } ne "" ){
        $this->{ 'BODY' } .= $this->{ 'BPG_TAG' };
        $this->{ 'BPG_TAG' } = "";    # Reset value
    }
    # $V103 End

    #---- Add font table ----#
    #---- Created font locations list
    if ( !defined( $this->{ 'FontTable' }{ 'LOCATIONS' } ) ){
        if ( $this->{ 'PageOutput' } == $TRUE ){
            $this->{ 'FontTable' }{ 'LOCATIONS' } = {};
        }
        else {
            $this->{ 'FontTable' }{ 'LOCATIONS' } = $this->{ 'FontLocations' };
        }
    }

    #---- Build font table list
    my @arrFontTableListTmp = sort keys( %{ $this->{ 'FontTable' } } );
    if ( @arrFontTableListTmp > 0 ){
        # V116d Begin
        #---- Add used font links
        if ( $this->{ 'HDR' }{ 'fHandle' } != undef ){
            my $fAddHdrHandleTmp = $this->{ 'HDR' }{ 'fHandle' };
            my $sSeparatorTmp = "";
            my $hrefFontRefTmp = undef;
            my $hrefUniqueFontLinksTmp = $this->{ 'FontTable' }{ 'LOCATIONS' };

            #---- Write font links
            for ( my $i = 0; $i < @arrFontTableListTmp; $i++ ){
                $hrefFontRefTmp = $this->{ 'FontTable' }{ $arrFontTableListTmp[ $i ] };
                if (    defined( $hrefFontRefTmp->{ 'LOCATION' } )
                     && !defined( $hrefUniqueFontLinksTmp->{ $hrefFontRefTmp->{ 'LOCATION' } } )
                   ){
                    $hrefUniqueFontLinksTmp->{ $hrefFontRefTmp->{ 'LOCATION' } } = $TRUE;
                    if ( $bLog == $TRUE ){ $theLogger->logMessage( "Font link entry:>" . '<link href="' . $hrefFontRefTmp->{ 'LOCATION' } . '" rel="stylesheet">' . "<" ); }
                    printf $fAddHdrHandleTmp ( $sSeparatorTmp . '<link href="' . $hrefFontRefTmp->{ 'LOCATION' } . '" rel="stylesheet">' );
                    $sSeparatorTmp = "\n";
                }
            }
        }
        else {
            #---- Add font links
            my $sFontLinksTmp = "";
            my $sSeparatorTmp = "";
            my $hrefFontRefTmp = undef;
            my $hrefUniqueFontLinksTmp = $this->{ 'FontTable' }{ 'LOCATIONS' };

            for ( my $i = 0; $i < @arrFontTableListTmp; $i++ ){
                $hrefFontRefTmp = $this->{ 'FontTable' }{ $arrFontTableListTmp[ $i ] };
                if (    defined( $hrefFontRefTmp->{ 'LOCATION' } )
                     && !defined( $hrefUniqueFontLinksTmp->{ $hrefFontRefTmp->{ 'LOCATION' } } )
                   ){
                    $hrefUniqueFontLinksTmp->{ $hrefFontRefTmp->{ 'LOCATION' } } = $TRUE;
                    if ( $bLog == $TRUE ){ $theLogger->logMessage( "Font link entry:>" . '<link href="' . $hrefFontRefTmp->{ 'LOCATION' } . '" rel="stylesheet">' . "<" ); }
                    $sFontLinksTmp .= $sSeparatorTmp . '<link href="' . $hrefFontRefTmp->{ 'LOCATION' } . '" rel="stylesheet">';
                    $sSeparatorTmp = "\n";
                }
            }

            $this->{ 'BODY' } .= $sFontLinksTmp;
        }
        # V116d End

        if ( $this->{ 'CSS' }{ 'fHandle' } != undef ){
            my $hrefCSSTmp = $this->{ 'CSS' };
            my $fCSSHandleTmp = $this->{ 'CSS' }{ 'fHandle' };

            #---- Write font table entry
            printf $fCSSHandleTmp ( "\n/* Style for page (" . $dbPageTmp->getPageID() . ") */" ); # V116i Change
            for ( my $i = 0; $i < @arrFontTableListTmp; $i++ ){
                if ( $hrefCSSTmp->{ 'FontId' }{ $this->{ 'FontTable' }{ $arrFontTableListTmp[ $i ] }{ 'STYLENAME' } } == undef ){
                    $hrefCSSTmp->{ 'FontId' }{ $this->{ 'FontTable' }{ $arrFontTableListTmp[ $i ] }{ 'STYLENAME' } } = $TRUE;
                    if ( $this->{ 'FontTable' }{ $arrFontTableListTmp[ $i ] }{ 'CSS' } ne "" ){ # V116i Change
                        printf $fCSSHandleTmp ( "\n" . $this->{ 'FontTable' }{ $arrFontTableListTmp[ $i ] }{ 'CSS' } );
                    }
                }
            }
        }
        else {
            #---- Begin font table style
            my $sFontTableTmp =   "\n" . '<style type="text/css" scoped>'
                                . "\n" . "<!--";
    
            #---- Write font table entry
            for ( my $i = 0; $i < @arrFontTableListTmp; $i++ ){
                $sFontTableTmp .= "\n" . $this->{ 'FontTable' }{ $arrFontTableListTmp[ $i ] }{ 'CSS' };
            }
    
            #---- End font table style
            $sFontTableTmp .=   "\n" . '-->'
                              . "\n" . '</style>';

            $this->{ 'BODY' } .= $sFontTableTmp;
        }
    }

    # $V103 Begin
    if ( $this->{ 'Content' } ne "" ){
        #---- Add background image
        if ( $this->{ 'BackgroundFilename' } ne "" ){
            $this->{ 'BODY' } .= "\n" . '<img width="' . $pgHandleTmp->{ 'Width' } . $this->{ 'UnitBase' }
                                      . '" height="' . $pgHandleTmp->{ 'Height' } . $this->{ 'UnitBase' }
                                      . '" src="' . $this->{ 'BackgroundFilename' }
                                      . '" alt="background image">';
        }

        # V116i Begin
        # Minimize page content elements separation (i.e, elements not separated by CRLF)
        if ( $this->{ 'Minimize' } == $TRUE ){
            $this->{ 'Content' } =~ s/[\r\n]+//g;
        }
        # V116i End

        #---- Add page content
        $this->{ 'BODY' } .= $this->{ 'Content' };

        #---- End page div
        $this->{ 'BODY' } .= "\n" . '</div>';
    }
    # $V103 End

    #---- End body
    if ( $this->{ 'PageOutput' } == $TRUE ){
        if ( $sBodyAddContentPar ne "" ){
            $this->{ 'BODY' } .= "\n" . $sBodyAddContentPar;
        }
        # $V106 Begin
        $this->{ 'BODY' } .= "\n" . '</div>'; # desktop_container div closure
        # $V106 End

        # V124d Begin
        # Minimize page content elements separation (i.e, elements not separated by CRLF)
        if ( $this->{ 'Minimize' } == $TRUE ){
            $this->{ 'ResponsiveContent' } =~ s/[\r\n]+//g;
        }

        # Add responsive container
        $this->{ 'BODY' } .=     "\n" . '<div id="responsive_container">'
                               . "\n" . $this->{ 'ResponsiveContent' }
                               . "\n" . '</div>'
                             ;
        # V124d End

        $this->{ 'BODY' } .= "\n" . '</body>';
    }

    #---- Write body
    my $handleFileTmp = $this->{ 'fHandle' };
    print $handleFileTmp ( $this->{ 'BODY' } ); # V120b Change

    if ( $this->{ 'PageOutput' } == $TRUE ){
        #---- Write footer
        $this->writeFooter( # Additional content
                            $sAddFooterContentPar
                          );

        #---- Write html end
        $this->_finalizeHTML();
    }

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::finalizePage( $sAddFooterContentPar );
    if ( $iRetTmp < 0 ){
        return $iRetTmp;
    }

    return 0;
}

#-----------------------------------------------------------------------
# Get File Extension
#
# Returns output file extension based on visitor type. Base returns empty
# string and concrete visitors MUST return appropriate extension
#
#-----------------------------------------------------------------------
sub getFileExtension{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "getFileExtension()" );
    }

    return ".html";
}

#-----------------------------------------------------------------------
# Update resource table
#-----------------------------------------------------------------------
sub updateResourceTable{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "updateResourceTable()" );
    }

    #---- Fetch parameters
    #
    # 1. Unique resource id
    # 2. Resource reference
    # 3. Object hash (having both kernel object and pom object)
    #
    my $iUniqueResIdPar = shift;
    my $a2wResourcePar  = shift;
    my $hrefObjectPar   = shift;

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::updateResourceTable( $iUniqueResIdPar, $a2wResourcePar, $hrefObjectPar );
    # V124d Begin
    if ( $iRetTmp < 0 ){
        return ( $iRetTmp, 'Could not update resource in base visitor' );
    }

    # Update resource table based on the type of resource and object
    if ( $this->isFontResource( $a2wResourcePar ) == $TRUE ){
        return $this->_updateResourceTableForFont( $iUniqueResIdPar, $a2wResourcePar, $hrefObjectPar );
    }

    #---- Get actual object
    my $a2wObjTmp = $hrefObjectPar->{ 'A2WOBJ' };

    #---- Process image and container objects
    my $sTypeTmp = $a2wObjTmp->_getType();
    if ( $sTypeTmp eq "image" ){
        return $this->_updateResourceTableForImage( $iUniqueResIdPar, $a2wResourcePar, $hrefObjectPar );
    }
    elsif ( $sTypeTmp eq "container" ){
        return $this->_updateResourceTableForContainer( $iUniqueResIdPar, $a2wResourcePar, $hrefObjectPar );
    }

    return 0;
    # V124d End
}

#-----------------------------------------------------------------------
# isWritable
#
# Asserts whether given object is writeable by visitor
#
# Returns
# TRUE is visitor can write the object
# FALSE is visitor can not write the object
#
#-----------------------------------------------------------------------
sub isWritable{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "isWritable()" );
    #}

    #---- Fetch parameter(s)
    #
    # 1. Object hash (having both kernel object and pom object)
    #
    my $hrefObjPar = shift;

    #---- Assert object
    my $a2wObjTmp = $hrefObjPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefObjPar->{ 'POMOBJ' };
    my $bRetTmp   = $TRUE;

    my $iObjTypeTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJTYPE };
    unless (    $iObjTypeTmp == $a2w::core::dm::Constants::OT_LINE
             || $iObjTypeTmp == $a2w::core::dm::Constants::OT_TEXT
             || $iObjTypeTmp == $a2w::core::dm::Constants::OT_IMAGE
             || $iObjTypeTmp == $a2w::core::dm::Constants::OT_CONTAINER # V116a Change
             || $iObjTypeTmp == $a2w::core::dm::Constants::OT_ANNOTATION # V120a Change
           ){
        $bRetTmp = $FALSE;
    }

    return $bRetTmp;
}

#-----------------------------------------------------------------------
# Write raw content
#-----------------------------------------------------------------------
sub writeRawContent{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeRawContent()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Raw content
    #
    my $sRawContentPar = shift;

    $this->{ 'Content' } .= "\n" . $sRawContentPar;
    $this->{ 'ResponsiveContent' } .= "\n" . $sRawContentPar; # V124d Change

    return 0;
}

# $V106 Begin
# Modified to write grouped text alone and separated writing table cell content
#
#-----------------------------------------------------------------------
# Write grouped text
#-----------------------------------------------------------------------
sub writeGroupedText{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeGroupedText()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. First Object hash (having both kernel object and pom object)
    # 2. Grouped Text
    # 3. Line gap (vertical difference between previous and current line, zero for first line)
    #
    my $hrefFirstObjPar = shift;
    my $sGroupedTextPar = shift;
    my $iLineGapPar     = shift;

    #---- Get actual objects
    my $a2wObjTmp = $hrefFirstObjPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefFirstObjPar->{ 'POMOBJ' };

    #---- Fetch text font
    my $a2wFontTmp = $a2wObjTmp->getFont();

    #---- Update font table, if font is added to it already
    my $iFontLocalIdTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_FONTID };

    #---- Update resource table with text used font
    if ( not defined( $this->{ 'FontTable' }{ $iFontLocalIdTmp } ) ){
        $this->updateResourceTable( $iFontLocalIdTmp, $a2wFontTmp, $hrefFirstObjPar );
    }
    if ( $this->{ 'PrevFontId' } != $iFontLocalIdTmp ){
        $this->{ 'PrevFontId' } = $iFontLocalIdTmp;

        if ( $bLog == $TRUE ){
            $theLogger->logMessage(   "-->UsedFont="
                                    . $this->{ 'FontTable' }{ $iFontLocalIdTmp }{ 'FAMILY' }
                                    . ", W=" . ( ( $this->{ 'FontTable' }{ $iFontLocalIdTmp }{ 'BOLD' } == $TRUE ) ? "B" : "M" )
                                    . ", S=" . $a2wFontTmp->getHeight()
                                  );
        }
    }

    #---- Add text to body
    #---- Get active page context
    my $pgHandleTmp = $this->{ 'ActivePage' };

    #---- Get active block
    my $blkActiveTmp = $pgHandleTmp->{ 'ActiveBlock' };

    #---- Evaluate object left within block
    my $iBlkLeftTmp = $blkActiveTmp->getStartX();
    my $iRelXPosTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS } - $iBlkLeftTmp;
    if ( $iRelXPosTmp != 0 ){
        if ( $this->{ 'UnitBase' } eq "px" ){
            $iRelXPosTmp *= $this->{ 'ContentScale' };
        }
        $iRelXPosTmp = int( $iRelXPosTmp + .5 * ( $iRelXPosTmp <=> 0 ) );
    }

    
    #---- Evaluate object top within block
    my $iBlkTopTmp = $blkActiveTmp->getStartY();
    my $iRelYPosTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS } - $iBlkTopTmp;
    $iRelYPosTmp -= ( $a2wFontTmp->getHeight() / 72 ) * $pgHandleTmp->{ 'ContentRes' };
    if ( $iRelYPosTmp != 0 ){
        if ( $this->{ 'UnitBase' } eq "px" ){
            $iRelYPosTmp *= $this->{ 'ContentScale' };
        }
        $iRelYPosTmp = int( $iRelYPosTmp + .5 * ( $iRelYPosTmp <=> 0 ) );
    }

    # $V103 Begin
    #---- Apply relative position style to text when text is grouped based on font in paragraph blocks
    #
    my $bForceRelPosStyleTmp = $FALSE;
    my $contDefTmp = $blkActiveTmp->getContentDef();
    if (    lc( $contDefTmp->getType() ) eq "paragraph"
         && $contDefTmp->getGroup() == $TRUE
         && $contDefTmp->getGroupBy() eq "font"
       ){
        $bForceRelPosStyleTmp = $TRUE;
    }
    # $V103 End

    # V124d Change
    #---- Escape text for special characters
    my $sEscapedTextTmp = $this->_escapeText( $sGroupedTextPar );
    my $sTextTmp = $this->findAndReplaceText( 'html', $hrefFirstObjPar, $sEscapedTextTmp, $hrefTableInfoPar ); # Find and replace text, if configured
    # V124d Change

    if ( $bLog == $TRUE && $sEscapedTextTmp ne $sTextTmp ){
        $theLogger->logMessage(   "Text: $sEscapedTextTmp replaced with value: $sTextTmp" );
    }

    #--- Add the text
    my @arrClassTmp = ();
    my $sClassTmp   = '';
    my $sStyleTmp   = '';
    @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'FontTable' }->{ $iFontLocalIdTmp }{ 'STYLENAME' };
    # $V103 Begin
    if ( $bForceRelPosStyleTmp == $TRUE ){
        @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'CSS' }{ 'Styles' }{ 'RELPOS' };
        $sStyleTmp .= 'left:' . $iRelXPosTmp . $this->{ 'UnitBase' } . ';';
    }
    elsif ( $iRelXPosTmp != 0 ){
        @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' };
        $sStyleTmp .= 'left:' . $iRelXPosTmp . $this->{ 'UnitBase' } . ';';
    }
    # $V103 End
    # $V106 Begin
    if ( $bForceRelPosStyleTmp == $TRUE ){
        if ( "@arrClassTmp" !~ /$this->{ 'CSS' }{ 'Styles' }{ 'RELPOS' }/ ){ @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'CSS' }{ 'Styles' }{ 'RELPOS' }; }
        $sStyleTmp .= 'top:' . $iRelYPosTmp . $this->{ 'UnitBase' } . ';';
    }
    elsif ( $iRelYPosTmp != 0 ){
        if ( "@arrClassTmp" !~ /$this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' }/ ){ @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' }; }
        $sStyleTmp .= 'top:' . $iRelYPosTmp . $this->{ 'UnitBase' } . ';';
    }
    # $V106 End
    if ( $this->{ 'CSS' }{ 'Styles' }{ 'ROT' . $pomObjTmp->{ $a2w::core::dm::Constants::AT_ANGLE } } ne "" ){
        @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'CSS' }{ 'Styles' }{ 'ROT' . $pomObjTmp->{ $a2w::core::dm::Constants::AT_ANGLE } };
    }
    if ( @arrClassTmp > 0 ){
        $sClassTmp = ' class="' . "@arrClassTmp" . '"';
    }

    my $iColorTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_COLOR };
    if ( $iColorTmp != $this->getDefForeColor() ){
        $sStyleTmp .= 'color:#' . sprintf( "%06X", $iColorTmp ) . ';';
    }
    if ( $sStyleTmp ne "" ){
        $sStyleTmp = ' style="' . $sStyleTmp . '"';
    }

    # V124d Begin
    my $sElementTmp =  "\n"
                      . '<span' . $sClassTmp . $sStyleTmp . '>'
                      . $sTextTmp
                      . '</span>'
                      ;
    $this->{ 'Content' } .= $sElementTmp;

    # Check whether find and replace applied to the text
    if ( $sEscapedTextTmp ne $sTextTmp ){
        # Check whether any href points to div id, if yes, update div for responsive html
        if ( $sTextTmp =~ /href="\#/ ){ $sTextTmp =~ s/href="\#/href="\#r-/; }
        $sElementTmp =  "\n"
                       . '<span' . $sClassTmp . $sStyleTmp . '>'
                       . $sTextTmp
                       . '</span>'
                       ;
    }
    $this->{ 'ResponsiveContent' } .= $sElementTmp;
    # V124d End

    if ( $bLog == $TRUE ){
        #---- Fetch text position
        my $fXPosTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS };
        my $fYPosTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS };
        if ( $this->{ 'UnitBase' } eq "px" ){
            $fXPosTmp *= $this->{ 'ContentScale' };
            $fYPosTmp *= $this->{ 'ContentScale' };
        }
        $fXPosTmp = int( $fXPosTmp + .5 * ( $fXPosTmp <=> 0 ) );
        $fYPosTmp = int( $fYPosTmp + .5 * ( $fYPosTmp <=> 0 ) );

        $theLogger->logMessage(   "@("
                                . $fXPosTmp
                                . ",". $fYPosTmp
                                . ")>" . $sTextTmp . "<"
                              );
    }
    return 0;
}
# $V106 End

#-----------------------------------------------------------------------
# Write image
#-----------------------------------------------------------------------
sub writeImage{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeImage()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Image object hash (having both kernel object and pom object)
    #
    my $hrefImageObjPar = shift;

    #---- Get actual objects
    my $a2wObjTmp = $hrefImageObjPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefImageObjPar->{ 'POMOBJ' };

    # V124d Begin
    my $sUniqueNameTmp = $a2wObjTmp->_getAttribute( 'uniquename' );
    if ( not defined $this->{ 'ResCache' }{ $sUniqueNameTmp } ){
        my @arrResTmp = keys( %{ $this->{ 'ResCache' } } );
        my ($iURTRetTmp, $sURTMsgTmp) = $this->updateResourceTable( (@arrResTmp + 1), undef, $hrefImageObjPar );
        if ( $iURTRetTmp < 0 ){ return ($iURTRetTmp, $sURTMsgTmp); }
    }

    my $hrefResTableTmp = $this->{ 'ResCache' };
    my $fCSSHandleTmp = $this->{ 'CSS' }{ 'fHandle' };
    #---- Add CSS class for this image with base64 image data
    if (    $this->{ 'CSS' }{ 'CLASSES' }{ $sUniqueNameTmp } == undef
         && $fCSSHandleTmp != undef
       ){
        my $sImgFormatTmp = $this->{ 'ImageFormat' };
        if ( $sImgFormatTmp eq 'jpeg' ){ $sImgFormatTmp = 'jpg'; }

        my $b64ImageDataTmp = $hrefResTableTmp->{ $sUniqueNameTmp }{ 'B64DATA' };
        my $sImgCSSTmp =   "\n" . '.' . $sUniqueNameTmp . '{'
                         . "\n" . '  content:url("data:image/' . $sImgFormatTmp . ';base64,' . $b64ImageDataTmp . '")'
                         . "\n" . '}'
                         ;
        print $fCSSHandleTmp ( $sImgCSSTmp );
		$this->{ 'CSS' }{ 'CLASSES' }{ $sUniqueNameTmp } = $TRUE;
    }
    # V124d End

    #---- Get active page context
    my $pgHandleTmp = $this->{ 'ActivePage' };

    #---- Get active page
    my $dbPageTmp = $pgHandleTmp->{ 'POM' };

    #---- Get active block
    my $blkActiveTmp = $pgHandleTmp->{ 'ActiveBlock' };

    #---- Evaluate object left within block
    my $iBlkLeftTmp = $blkActiveTmp->getStartX();
    my $iRelXPosTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS } - $iBlkLeftTmp;
    if ( $iRelXPosTmp != 0 ){
        if ( $this->{ 'UnitBase' } eq "px" ){
            $iRelXPosTmp *= $this->{ 'ContentScale' };
        }
        $iRelXPosTmp = int( $iRelXPosTmp + .5 * ( $iRelXPosTmp <=> 0 ) );
    }

    # $V106 Begin
    #---- Evaluate object top within block
    my $iBlkTopTmp = $blkActiveTmp->getStartY();
    my $iRelYPosTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS } - $iBlkTopTmp;
    if ( $iRelYPosTmp != 0 ){
        if ( $this->{ 'UnitBase' } eq "px" ){
            $iRelYPosTmp *= $this->{ 'ContentScale' };
        }
        #$iRelYPosTmp = int( $iRelYPosTmp + .5 * ( $iRelYPosTmp <=> 0 ) );
    }
    # $V106 End

    #---- Fetch image attributes
    my $fWidthTmp     = $a2wObjTmp->getPresentationWidth();
    my $fHeightTmp    = $a2wObjTmp->getPresentationHeight();
    my $iResoluionTmp = $a2wObjTmp->getResolution();

    #---- Extend block end Y (add presentation size to with image position)
    my $fYPosTmp    = $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS };
    my $fImgEndYTmp = $fYPosTmp
                      + ( $fHeightTmp * ( $dbPageTmp->getPageRes() / $iResoluionTmp ) );
    if ( $pgHandleTmp->{ 'PrevBlkEndY' } < $fImgEndYTmp ){
        $pgHandleTmp->{ 'PrevBlkEndY' } = $fImgEndYTmp;
    }

    #---- Skip, if skip mode is on
    if ( $this->{ 'SkipMode' } == $TRUE ){
        return 0;
    }

    # $V113 Begin
    #---- Evaluate image presentation width/height
    if ( $this->{ 'UnitBase' } eq "px" ){
        $fWidthTmp  *= $this->getOutputRes() / $iResoluionTmp;
        $fHeightTmp *= $this->getOutputRes() / $iResoluionTmp;
    }
    $fWidthTmp  = int( $fWidthTmp + .5 * ( $fWidthTmp <=> 0 ) );
    $fHeightTmp = int( $fHeightTmp + .5 * ( $fHeightTmp <=> 0 ) );

    #---- Add image to body
    my $sStyleTmp = ' width='  . $fWidthTmp;
    $sStyleTmp   .= ' height='  . $fHeightTmp;

    # V125 Begin
    #---- Get image rotation
    my $iRotTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_ANGLE };

    # Adjust the position based on the rotation
    if ( $iRotTmp == 90 || $iRotTmp == 270 ){
        $iRelYPosTmp -= $fHeightTmp;
    }
    # V125 End

    my @arrClassTmp = ();
    my $sClassTmp   = '';
    my $sPosStyleTmp = '';
    if ( $iRelXPosTmp != 0 ){
        @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' };
        $sPosStyleTmp .= 'left:' . $iRelXPosTmp . $this->{ 'UnitBase' } . ';';
    }
    # $V106 Begin
    else {
        @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' };
        $sPosStyleTmp .= 'left:0' . $this->{ 'UnitBase' } . ';';
    }
    if ( $iRelYPosTmp != 0 ){
        if ( "@arrClassTmp" !~ /$this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' }/ ){ @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' }; }
        $sPosStyleTmp .= 'top:' . $iRelYPosTmp . $this->{ 'UnitBase' } . ';';
    }
    else {
        if ( "@arrClassTmp" !~ /$this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' }/ ){ @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' }; }
        $sPosStyleTmp .= 'top:0' . $this->{ 'UnitBase' } . ';';
    }
    # $V106 End

    # V124d Begin
    if (    $this->{ 'ResCache' }{ $sUniqueNameTmp }{ 'SOLIDRECT' } == $FALSE
         && $this->{ 'bInlineImage' } == $TRUE
       ){
        @arrClassTmp[ $#arrClassTmp + 1 ] = $sUniqueNameTmp;
    }
    # V124d End

    # V125 Begin
    if (    $iRotTmp > 0
	     && $this->{ 'CSS' }{ 'Styles' }{ 'ROT' . $iRotTmp } ne ""
	   ){
        @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'CSS' }{ 'Styles' }{ 'ROT' . $iRotTmp };
    }
    # V125 End

    if ( @arrClassTmp > 0 ){
        $sClassTmp = ' class="' . "@arrClassTmp" . '"';
    }

    # $V111 Begin
    #---- Assert and wrap image with <a> tag (when block info has "href=<url>")
    #
    # Fetch block info
    my $sBlkInfoTmp = $blkActiveTmp->getInfo();
    my $bWrappedWithAnchorTmp = $FALSE;
    if ( $sBlkInfoTmp =~ /\s+href\=\"(.*)\"\s*/ ){
        $bWrappedWithAnchorTmp = $TRUE;
        my $sURLTmp = $1;
        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "Block info has href URL (" . $sURLTmp . "), wrapping image with <a> tag" );
        }
        # V124d Begin
        my $sElementTmp =   "\n"
                          . '<a href="' . $sURLTmp . '">'
                          ;
        $this->{ 'Content' } .= $sElementTmp;
        $this->{ 'ResponsiveContent' } .= $sElementTmp;
        # V124d End
    }
    # $V111 End

    # V116b Begin
    # V124d Begin
    my $sImgInlinedTmp = (defined $this->{ 'ResCache' }{ $sUniqueNameTmp }{ 'B64DATA' }) ? "Yes" : "No";
    if ( $this->{ 'ResCache' }{ $sUniqueNameTmp }{ 'SOLIDRECT' } == $TRUE ){
        my $sElementTmp = "\n" . '<!-- Image ' . $a2wObjTmp->getName() . ' written as solid rectangle -->';
        $this->{ 'Content' } .= $sElementTmp;
        $this->{ 'ResponsiveContent' } .= $sElementTmp;

        $sElementTmp =    "\n"
                       . '<hr'
                       . $sClassTmp
                       . ' style="'
                       . $sPosStyleTmp
                       . 'width:' . int( $fWidthTmp ) . 'px;'
                       . 'height:' . int( $fHeightTmp ) . 'px;'
                       . 'background-color:#' . sprintf( "%06X", $pomObjTmp->{ $a2w::core::dm::Constants::AT_COLOR } ) . ';'
                       . 'border:none;'
                       . '"'
                       . '/>'
                       ;
        $this->{ 'Content' } .= $sElementTmp;
        $this->{ 'ResponsiveContent' } .= $sElementTmp;
    }
    else {
        if ( $sPosStyleTmp ne "" ){ $sPosStyleTmp = ' style="' . $sPosStyleTmp . '"'; }

        # $V102 Begin
        # V124d Begin
        #---- Write image in html ----#
        if ( $this->{ 'bInlineImage' } == $TRUE ){
            my $sElementTmp =   "\n"
                              . '<img' . $sClassTmp . $sPosStyleTmp . $sStyleTmp . ' alt="' . $a2wObjTmp->getName() . '"/>'
                              ;
            $this->{ 'Content' } .= $sElementTmp;
            $this->{ 'ResponsiveContent' } .= $sElementTmp;
        }
        else {
            my $sElementTmp =   "\n"
                              . '<img' . $sClassTmp . $sPosStyleTmp . $sStyleTmp . ' src="' . $this->{ 'ResCache' }{ $sUniqueNameTmp }{ 'SIMPLEFILENAME' } . '" alt="' . $a2wObjTmp->getName() . '"/>' # V101 Change
                              ;
            $this->{ 'Content' } .= $sElementTmp;
            $this->{ 'ResponsiveContent' } .= $sElementTmp;
        }
        # V124d End
        # $V102 End
    }
    # V116b End

    # $V111 Begin
    if ( $bWrappedWithAnchorTmp == $TRUE ){
        # V124d Begin
        my $sElementTmp =   "\n"
                          . '</a>'
                          ;
        $this->{ 'Content' } .= $sElementTmp;
        $this->{ 'ResponsiveContent' } .= $sElementTmp;
        # V124d End
    }
    # $V111 End

    if ( $bLog == $TRUE ){
        #---- Fetch image position
        my $fXPosTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS };
        if ( $this->{ 'UnitBase' } eq "px" ){
            $fXPosTmp *= $this->{ 'ContentScale' };
            $fYPosTmp *= $this->{ 'ContentScale' };
        }
        $fXPosTmp = int( $fXPosTmp + .5 * ( $fXPosTmp <=> 0 ) );
        $fYPosTmp = int( $fYPosTmp + .5 * ( $fYPosTmp <=> 0 ) );

        $theLogger->logMessage(   "-->Image="
                                . $a2wObjTmp->getName()
                                . ", ImgSize=(W:" . $a2wObjTmp->getWidth() . ", H:" . $a2wObjTmp->getHeight() . ")"
                                . ", BPP=" . $a2wObjTmp->getBitsPerPixel()
                                . ", Compression=" . $a2wObjTmp->getCompression()
                                . ", Pos=(X:" . $fXPosTmp . ", Y:" . $fYPosTmp . ")"
                                . ", PresentationSize=(W:" . $fWidthTmp . ", H:" . $fHeightTmp . ")"
                                # $V102 Begin
                                . ", Inlined=" . $sImgInlinedTmp
                                # $V102 End
                              );
    }

    return 0;
}

# V116a Begin
#-----------------------------------------------------------------------
# Write container
#-----------------------------------------------------------------------
sub writeContainer{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeContainer()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Container object hash (having both kernel object and pom object)
    #
    my $hrefContainerObjPar = shift;

    #---- Get actual objects
    my $a2wObjTmp = $hrefContainerObjPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefContainerObjPar->{ 'POMOBJ' };

    # V124d Begin
    my $sUniqueNameTmp = $a2wObjTmp->_getAttribute( 'uniquename' );
    $sUniqueNameTmp =~ s/[\.\#]/-/;
    if ( not defined $this->{ 'ResCache' }{ $sUniqueNameTmp } ){
        my @arrResTmp = keys( %{ $this->{ 'ResCache' } } );
        my ($iURTRetTmp, $sURTMsgTmp) = $this->updateResourceTable( (@arrResTmp + 1), undef, $hrefContainerObjPar );
        if ( $iURTRetTmp < 0 ){ return ($iURTRetTmp, $sURTMsgTmp); }
    }

    my $hrefResTableTmp = $this->{ 'ResCache' };
    my $fCSSHandleTmp = $this->{ 'CSS' }{ 'fHandle' };
    #---- Add CSS class for this image with base64 image data
    if (    $this->{ 'CSS' }{ 'CLASSES' }{ $sUniqueNameTmp } == undef
         && $fCSSHandleTmp != undef
       ){
        my $sImgFormatTmp = $this->{ 'ImageFormat' };
        if ( $sImgFormatTmp eq 'jpeg' ){ $sImgFormatTmp = 'jpg'; }

        my $b64ImageDataTmp = $hrefResTableTmp->{ $sUniqueNameTmp }{ 'B64DATA' };
        my $sImgCSSTmp =   "\n" . '.' . $sUniqueNameTmp . '{'
                         . "\n" . '  content:url("data:image/' . $sImgFormatTmp . ';base64,' . $b64ImageDataTmp . '")'
                         . "\n" . '}'
                         ;
        print $fCSSHandleTmp ( $sImgCSSTmp );
		$this->{ 'CSS' }{ 'CLASSES' }{ $sUniqueNameTmp } = $TRUE;
    }
    # V124d End

    #---- Get container type
    my $contTypeTmp = $a2wObjTmp->getObjectType();

    #---- Get active page context
    my $pgHandleTmp = $this->{ 'ActivePage' };

    #---- Get active page
    my $dbPageTmp = $pgHandleTmp->{ 'POM' };

    #---- Get active block
    my $blkActiveTmp = $pgHandleTmp->{ 'ActiveBlock' };

    #---- Evaluate object left within block
    my $iBlkLeftTmp = $blkActiveTmp->getStartX();
    my $iRelXPosTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS } - $iBlkLeftTmp;
    if ( $iRelXPosTmp != 0 ){
        if ( $this->{ 'UnitBase' } eq "px" ){
            $iRelXPosTmp *= $this->{ 'ContentScale' };
        }
        $iRelXPosTmp = int( $iRelXPosTmp + .5 * ( $iRelXPosTmp <=> 0 ) );
    }

    #---- Evaluate object top within block
    my $iBlkTopTmp = $blkActiveTmp->getStartY();
    my $iRelYPosTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS } - $iBlkTopTmp;
    if ( $iRelYPosTmp != 0 ){
        if ( $this->{ 'UnitBase' } eq "px" ){
            $iRelYPosTmp *= $this->{ 'ContentScale' };
        }
        #$iRelYPosTmp = int( $iRelYPosTmp + .5 * ( $iRelYPosTmp <=> 0 ) );
    }

    #---- Fetch image attributes
    my $fWidthTmp     = $a2wObjTmp->getPresentationWidth();  # V124d Change
    my $fHeightTmp    = $a2wObjTmp->getPresentationHeight(); # V124d Change
    my $iResoluionTmp = $a2wObjTmp->getResolution();
    my $iPageResoluionTmp = $dbPageTmp->getPageRes();
    my $fFactorTmp    = ( $iPageResoluionTmp / $iResoluionTmp );

    #---- Extend block end Y (add presentation size to with image position)
    my $fYPosTmp    = $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS };
    my $fImgEndYTmp = $fYPosTmp
                      + ( $fHeightTmp * $fFactorTmp );
    if ( $pgHandleTmp->{ 'PrevBlkEndY' } < $fImgEndYTmp ){
        $pgHandleTmp->{ 'PrevBlkEndY' } = $fImgEndYTmp;
    }

    #---- Skip, if skip mode is on
    if ( $this->{ 'SkipMode' } == $TRUE ){
        return 0;
    }

    #---- Skip writing container, if object type is unsupported
    if ( $this->_assertObjectType( $contTypeTmp ) == $FALSE ){
        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "Unsupported object type (" . $a2wObjTmp->getObjectTypeName() . "), container skipped from output presentation" ); # V120a change
        }
        return 0;
    }

    #---- Evaluate image presentation width/height
    if ( $this->{ 'UnitBase' } eq "px" ){
        $fWidthTmp  *= $this->getOutputRes() / $iResoluionTmp;
        $fHeightTmp *= $this->getOutputRes() / $iResoluionTmp;
    }
    $fWidthTmp  = int( $fWidthTmp + .5 * ( $fWidthTmp <=> 0 ) );
    $fHeightTmp = int( $fHeightTmp + .5 * ( $fHeightTmp <=> 0 ) );

    #---- Add image to body
    my $sStyleTmp = ' width='  . $fWidthTmp;
    $sStyleTmp   .= ' height='  . $fHeightTmp;

    my @arrClassTmp = ();
    my $sClassTmp   = '';
    my $sPosStyleTmp = '';
    if ( $iRelXPosTmp != 0 ){
        @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' };
        $sPosStyleTmp .= 'left:' . $iRelXPosTmp . $this->{ 'UnitBase' } . ';';
    }
    else {
        @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' };
        $sPosStyleTmp .= 'left:0' . $this->{ 'UnitBase' } . ';';
    }
    if ( $iRelYPosTmp != 0 ){
        if ( "@arrClassTmp" !~ /$this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' }/ ){ @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' }; }
        $sPosStyleTmp .= 'top:' . $iRelYPosTmp . $this->{ 'UnitBase' } . ';';
    }
    else {
        if ( "@arrClassTmp" !~ /$this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' }/ ){ @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' }; }
        $sPosStyleTmp .= 'top:0' . $this->{ 'UnitBase' } . ';';
    }

    # V124d Begin
    if ( $this->{ 'bInlineImage' } == $TRUE ){
        @arrClassTmp[ $#arrClassTmp + 1 ] = $sUniqueNameTmp;
    }
    # V124d End

    # V120c Begin
    # Determine the z-index for the SF added container image
    if ( $pomObjTmp->{ $a2w::core::dm::Constants::AT_SF_OBJECT } == $TRUE ){
        my $iMaxZIdxTmp = 999;
        if ( defined( $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJSEQID_BG } ) ){
            #---- Background object
            my $zIndexTmp = (($iMaxZIdxTmp + 1) * -1) + $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJSEQID_BG };
            $sPosStyleTmp .= 'z-index:' . $zIndexTmp . ';';
        }
        elsif ( defined( $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJSEQID_FG } ) ){
            #---- Foreground object
            my $zIndexTmp = ($iMaxZIdxTmp + 1) - $dbPageTmp->getForegroundObjectsCount();
            $sPosStyleTmp .= 'z-index:' . $zIndexTmp . ';';
        }
    }
    # V120c End

    if ( @arrClassTmp > 0 ){
        $sClassTmp = ' class="' . "@arrClassTmp" . '"';
    }
    if ( $sPosStyleTmp ne "" ){
        $sPosStyleTmp = ' style="' . $sPosStyleTmp . '"';
    }

    #---- Assert and wrap image with <a> tag (when block info has "href=<url>")
    #
    # Fetch block info
    my $sBlkInfoTmp = $blkActiveTmp->getInfo();
    my $bWrappedWithAnchorTmp = $FALSE;
    if ( $sBlkInfoTmp =~ /\s+href\=\"(.*)\"\s*/ ){
        $bWrappedWithAnchorTmp = $TRUE;
        my $sURLTmp = $1;
        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "Block info has href URL (" . $sURLTmp . "), wrapping image with <a> tag" );
        }
        # V124d Begin
        my $sElementTmp =   "\n"
                          . '<a href="' . $sURLTmp . '">'
                          ;
        $this->{ 'Content' } .= $sElementTmp;
        $this->{ 'ResponsiveContent' } .= $sElementTmp;
        # V124d End
    }

    # V124d Begin
    #---- Write image in html ----#
    my $sImgInlinedTmp = (defined $this->{ 'ResCache' }{ $sUniqueNameTmp }{ 'B64DATA' }) ? "Yes" : "No";
    if ( $this->{ 'bInlineImage' } == $TRUE ){
        my $sElementTmp =   "\n"
                          . '<img' . $sClassTmp . $sPosStyleTmp . $sStyleTmp . ' alt="' . $a2wObjTmp->getName() . '"/>'
                          ;
        $this->{ 'Content' } .= $sElementTmp;
        $this->{ 'ResponsiveContent' } .= $sElementTmp;
    }
    else {
        my $sElementTmp =   "\n"
                          . '<img' . $sClassTmp . $sPosStyleTmp . $sStyleTmp . ' src="' . $this->{ 'ResCache' }{ $sUniqueNameTmp }{ 'SIMPLEFILENAME' } . '" alt="' . $a2wObjTmp->getName() . '"/>' # V101 Change
                          ;
        $this->{ 'Content' } .= $sElementTmp;
        $this->{ 'ResponsiveContent' } .= $sElementTmp;
    }
    # V124d End

    if ( $bWrappedWithAnchorTmp == $TRUE ){
        # V124d Begin
        my $sElementTmp =   "\n"
                          . '</a>'
                          ;
        $this->{ 'Content' } .= $sElementTmp;
        $this->{ 'ResponsiveContent' } .= $sElementTmp;
        # V124d End
    }

    if ( $bLog == $TRUE ){
        #---- Fetch container position
        my $fXPosTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS };
        if ( $this->{ 'UnitBase' } eq "px" ){
            $fXPosTmp *= $this->{ 'ContentScale' };
            $fYPosTmp *= $this->{ 'ContentScale' };
        }
        $fXPosTmp = int( $fXPosTmp + .5 * ( $fXPosTmp <=> 0 ) );
        $fYPosTmp = int( $fYPosTmp + .5 * ( $fYPosTmp <=> 0 ) );

        $theLogger->logMessage(   "-->Container="
                                . $a2wObjTmp->getName()
                                . ", ImgSize=(W:" . $a2wObjTmp->getWidth() . ", H:" . $a2wObjTmp->getHeight() . ")"
                                . ", BPP=" . $a2wObjTmp->getBitsPerPixel()
                                . ", Compression=" . $a2wObjTmp->getCompression()
                                . ", Pos=(X:" . $fXPosTmp . ", Y:" . $fYPosTmp . ")"
                                . ", PresentationSize=(W:" . $fWidthTmp . ", H:" . $fHeightTmp . ")"
                                . ", Inlined=" . $sImgInlinedTmp
                              );
    }

    return 0;
}
# V116a End

#-----------------------------------------------------------------------
# Write line
#-----------------------------------------------------------------------
sub writeLine{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeLine()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Line object hash (having both kernel object and pom object)
    #
    my $hrefLineObjPar = shift;

    #---- Get actual objects
    my $a2wObjTmp = $hrefLineObjPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefLineObjPar->{ 'POMOBJ' };

    #---- Skip, if skip mode is on
    if ( $this->{ 'SkipMode' } == $TRUE ){
        return 0;
    }

    #---- Add line to body
    #---- Get active page context
    my $pgHandleTmp = $this->{ 'ActivePage' };

    #---- Get active block
    my $blkActiveTmp = $pgHandleTmp->{ 'ActiveBlock' };

    #---- Evaluate object left within block
    my $iBlkLeftTmp = $blkActiveTmp->getStartX();
    my $iRelXPosTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS } - $iBlkLeftTmp;
    if ( $iRelXPosTmp != 0 ){
        if ( $this->{ 'UnitBase' } eq "px" ){
            $iRelXPosTmp *= $this->{ 'ContentScale' };
        }
        $iRelXPosTmp = int( $iRelXPosTmp + .5 * ( $iRelXPosTmp <=> 0 ) );
    }

    # $V106 Begin
    #---- Evaluate object top within block
    my $iBlkTopTmp = $blkActiveTmp->getStartY();
    my $iRelYPosTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS } - $iBlkTopTmp;
    if ( $iRelYPosTmp != 0 ){
        if ( $this->{ 'UnitBase' } eq "px" ){
            $iRelYPosTmp *= $this->{ 'ContentScale' };
        }
        $iRelYPosTmp = int( $iRelYPosTmp + .5 * ( $iRelYPosTmp <=> 0 ) );
    }
    # $V106 End

    #---- Fetch line attributes
    my $fLineWidthTmp  = $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_LINE_WIDTH };
    my $fLineLengthTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_LINE_LENGTH };
    if ( $this->{ 'UnitBase' } eq "px" ){
        $fLineWidthTmp  *= $this->{ 'ContentScale' };
        $fLineLengthTmp *= $this->{ 'ContentScale' };
    }
    my $sOrnTmp = "HLine";
    if ( $a2wObjTmp->isVertical() ){
        $sOrnTmp = "VLine";
        #---- Swap width and length
        my $fSwapTmp = $fLineWidthTmp;
        $fLineWidthTmp = $fLineLengthTmp;
        $fLineLengthTmp = $fSwapTmp;
    }

    $fLineWidthTmp = int( $fLineWidthTmp + .5 * ( $fLineWidthTmp <=> 0 ) );
    $fLineLengthTmp = int( $fLineLengthTmp + .5 * ( $fLineLengthTmp <=> 0 ) );

    #---- Assert line length for negative value
    if ( $fLineLengthTmp < 0.0 ){
        $fXPosTmp += $fLineLengthTmp;
        $fLineLengthTmp *= -1;
    }

    #---- Assert line width for negative value
    if ( $fLineWidthTmp < 0.0 ){
        $fYPosTmp += $fLineWidthTmp;
        $fLineWidthTmp *= -1;
    }

    my @arrClassTmp   = ();
    my $sClassTmp     = '';
    my $sLineStyleTmp = ' style="';
    if ( $iRelXPosTmp != 0 ){
        @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' };
        $sLineStyleTmp .= 'left:' . $iRelXPosTmp . $this->{ 'UnitBase' } . ';';
    }
    # $V106 Begin
    else {
        @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' };
        $sLineStyleTmp .= 'left:0' . $this->{ 'UnitBase' } . ';';
    }
    if ( $iRelYPosTmp != 0 ){
        if ( "@arrClassTmp" !~ /$this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' }/ ){ @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' }; }
        $sLineStyleTmp .= 'top:' . $iRelYPosTmp . $this->{ 'UnitBase' } . ';';
    }
    else {
        if ( "@arrClassTmp" !~ /$this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' }/ ){ @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' }; }
        $sLineStyleTmp .= 'top:0' . $this->{ 'UnitBase' } . ';';
    }
    # $V106 End

    if ( @arrClassTmp > 0 ){
        $sClassTmp = ' class="' . "@arrClassTmp" . '"';
    }

    $sLineStyleTmp .= 'width:'  . $fLineLengthTmp . $this->{ 'UnitBase' } . ';';
    $sLineStyleTmp .= 'height:'  . $fLineWidthTmp . $this->{ 'UnitBase' } . ';';
    $sLineStyleTmp .= 'background-color:#' . sprintf( "%06X", $pomObjTmp->{ $a2w::core::dm::Constants::AT_COLOR } ) . ';';
    $sLineStyleTmp .= 'border:none;"';

    # V124d Begin
    my $sElementTmp =   "\n"
                      . '<hr' . $sClassTmp . $sLineStyleTmp . '/>'
                      ;
    $this->{ 'Content' } .= $sElementTmp;
    $this->{ 'ResponsiveContent' } .= $sElementTmp;
    # V124d End

    if ( $bLog == $TRUE ){
        #---- Fetch line position
        my $fXPosTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS };
        my $fYPosTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS };
        if ( $this->{ 'UnitBase' } eq "px" ){
            $fXPosTmp *= $this->{ 'ContentScale' };
            $fYPosTmp *= $this->{ 'ContentScale' };
        }
        $fXPosTmp = int( $fXPosTmp + .5 * ( $fXPosTmp <=> 0 ) );
        $fYPosTmp = int( $fYPosTmp + .5 * ( $fYPosTmp <=> 0 ) );

        $theLogger->logMessage(   "-->Line="
                                . " Orn=" . $sOrnTmp
                                . ", Width=" . $fLineWidthTmp
                                . ", Length=" . $fLineLengthTmp
                                . ", StartPos=(X:" . $fXPosTmp . ", Y:" . $fYPosTmp . ")"
                              );
    }
    return 0;
}

#-----------------------------------------------------------------------
# Write text
#-----------------------------------------------------------------------
sub writeText{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeText()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Text object hash (having both kernel object and pom object)
    #
    my $hrefTextObjPar = shift;

    #---- Get actual objects
    my $a2wObjTmp = $hrefTextObjPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefTextObjPar->{ 'POMOBJ' };

    #---- Skip, if skip mode is on
    if ( $this->{ 'SkipMode' } == $TRUE ){
        return 0;
    }

    $this->{ 'BlockTextId' }++; # V124c Change

    #---- Fetch text font
    my $a2wFontTmp = $a2wObjTmp->getFont();

    #---- Update font table, if font is added to it already
    my $iFontLocalIdTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_FONTID };

    #---- Update resource table with text used font
    if ( not defined( $this->{ 'FontTable' }{ $iFontLocalIdTmp } ) ){
        $this->updateResourceTable( $iFontLocalIdTmp, $a2wFontTmp, $hrefTextObjPar );
    }
    if ( $this->{ 'PrevFontId' } != $iFontLocalIdTmp ){
        $this->{ 'PrevFontId' } = $iFontLocalIdTmp;

        if ( $bLog == $TRUE ){
            $theLogger->logMessage(   "-->UsedFont="
                                    . $this->{ 'FontTable' }{ $iFontLocalIdTmp }{ 'FAMILY' }
                                    . ", W=" . ( ( $this->{ 'FontTable' }{ $iFontLocalIdTmp }{ 'BOLD' } == $TRUE ) ? "B" : "M" )
                                    . ", S=" . $a2wFontTmp->getHeight()
                                    . ", FLId=" . $iFontLocalIdTmp # V125 Change
                                  );
        }
    }

    #---- Add text to body
    #---- Get active page context
    my $pgHandleTmp = $this->{ 'ActivePage' };

    #---- Get active block
    my $blkActiveTmp = $pgHandleTmp->{ 'ActiveBlock' };

    #---- Evaluate object left within block
    my $iBlkLeftTmp = $blkActiveTmp->getStartX();
    my $iRelXPosTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS } - $iBlkLeftTmp;
    if ( $iRelXPosTmp != 0 ){
        if ( $this->{ 'UnitBase' } eq "px" ){
            $iRelXPosTmp *= $this->{ 'ContentScale' };
        }
        $iRelXPosTmp = int( $iRelXPosTmp + .5 * ( $iRelXPosTmp <=> 0 ) );
    }

    # $V106 Begin
    #---- Evaluate object top within block
    my $iBlkTopTmp = $blkActiveTmp->getStartY();
    my $iRelYPosTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS } - $iBlkTopTmp;

    # $V107 Begin
    #---- Adjust text position to top left based on font height
    #$iRelYPosTmp -= ( $a2wFontTmp->getHeight() / 72 ) * $pgHandleTmp->{ 'ContentRes' }; # V116c Change
    $iRelYPosTmp -= ( ( $a2wFontTmp->getHeight() / 72 ) * $pgHandleTmp->{ 'ContentRes' } ) * 0.75; # V116c Change
    # $V107 End

    if ( $iRelYPosTmp != 0 ){
        if ( $this->{ 'UnitBase' } eq "px" ){
            $iRelYPosTmp *= $this->{ 'ContentScale' };
        }
        $iRelYPosTmp = int( $iRelYPosTmp + .5 * ( $iRelYPosTmp <=> 0 ) );
    }
    # $V106 End

    #---- Escape text for special characters
    my $sTextTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_VALUE };
    # V124d Begin
    my $sEscapedTextTmp = $this->_escapeText( $sTextTmp );
    $sTextTmp = $this->findAndReplaceText( 'html', $hrefTextObjPar, $sEscapedTextTmp ); # Find and replace text, if configured
    # V124d End

    my @arrClassTmp = ();
    my $sClassTmp   = '';
    my $sStyleTmp   = '';
    @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'FontTable' }->{ $iFontLocalIdTmp }{ 'STYLENAME' };
    if ( $iRelXPosTmp != 0 ){
        @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' };
        $sStyleTmp .= 'left:' . $iRelXPosTmp . $this->{ 'UnitBase' } . ';';
    }
    # $V106 Begin
    if ( $iRelYPosTmp != 0 ){
        if ( "@arrClassTmp" !~ /$this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' }/ ){ @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' }; }
        $sStyleTmp .= 'top:' . $iRelYPosTmp . $this->{ 'UnitBase' } . ';';
    }
    # $V106 End
    if ( $this->{ 'CSS' }{ 'Styles' }{ 'ROT' . $pomObjTmp->{ $a2w::core::dm::Constants::AT_ANGLE } } ne "" ){
        @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'CSS' }{ 'Styles' }{ 'ROT' . $pomObjTmp->{ $a2w::core::dm::Constants::AT_ANGLE } };
    }
    if ( @arrClassTmp > 0 ){
        $sClassTmp = 'class="' . "@arrClassTmp" . '"';
    }

    #--- Add text to output
    my $iColorTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_COLOR };
    if ( $iColorTmp != $this->getDefForeColor() ){
        $sStyleTmp .= 'color:#' . sprintf( "%06X", $iColorTmp ) . ';';
    }
    if ( $sStyleTmp ne "" ){
        $sStyleTmp = 'style="' . $sStyleTmp . '"';
    }

    # V124c Begin
    my $hrefOutConfTmp = $blkActiveTmp->getOutputConfig( 'html' );
    my $sIDTmp = '';
    if (    $hrefOutConfTmp
         && defined $hrefOutConfTmp->{ 'generateIDs' }
         && $hrefOutConfTmp->{ 'generateIDs' } == 1
       ){
        $sIDTmp = 'id="' . $blkActiveTmp->getId() . '_' . $this->{ 'BlockTextId' } . '"';
    }
    # V124c End

    # V124d Begin
    my $sElementTmp = "\n" . '<span';
    if ( $sIDTmp ne '' ){ $sElementTmp .= ' ' . $sIDTmp; }
    if ( $sClassTmp ne '' ){ $sElementTmp .= ' ' . $sClassTmp; }
    if ( $sStyleTmp ne '' ){ $sElementTmp .= ' ' . $sStyleTmp; }
    $sElementTmp .= '>' . $sTextTmp . '</span>';
    $this->{ 'Content' } .= $sElementTmp;

    # Check whether find and replace applied to the text
    if ( $sEscapedTextTmp ne $sTextTmp ){
        # Check whether any href points to div id, if yes, update div for responsive html
        if ( $sTextTmp =~ /href="\#/ ){ $sTextTmp =~ s/href="\#/href="\#r-/; }

		$sElementTmp = "\n" . '<span';
		if ( $sIDTmp ne '' ){ $sElementTmp .= ' ' . $sIDTmp; }
		if ( $sClassTmp ne '' ){ $sElementTmp .= ' ' . $sClassTmp; }
		if ( $sStyleTmp ne '' ){ $sElementTmp .= ' ' . $sStyleTmp; }
		$sElementTmp .= '>' . $sTextTmp . '</span>';
    }

	#---- Get content to prepend for responsive html
	my $sCont2PrependTmp = $this->getContentToPrepend( 'responsivehtml', $hrefTextObjPar, $sEscapedTextTmp );
	if ( $sCont2PrependTmp ne "" ){
		$this->{ 'ResponsiveContent' } .= $sCont2PrependTmp;
		if ( $bLog == $TRUE ){ $theLogger->logMessage( "Text:>$sEscapedTextTmp< got content to prepend:>$sCont2PrependTmp<" ); }
	}

    $sElementTmp =~ s/id="/id="r-/;
    $this->{ 'ResponsiveContent' } .= $sElementTmp;

	#---- Get content to append for responsive html
	my $sCont2AppendTmp = $this->getContentToAppend( 'responsivehtml', $hrefTextObjPar, $sEscapedTextTmp );
	if ( $sCont2AppendTmp ne "" ){
		$this->{ 'ResponsiveContent' } .= $sCont2AppendTmp;
		if ( $bLog == $TRUE ){ $theLogger->logMessage( "Text:>$sEscapedTextTmp< got content to append:>$sCont2AppendTmp<" ); }
	}
    # V124d End

    if ( $bLog == $TRUE ){
        #---- Fetch text position
        my $fXPosTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS };
        my $fYPosTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS };
        if ( $this->{ 'UnitBase' } eq "px" ){
            $fXPosTmp *= $this->{ 'ContentScale' };
            $fYPosTmp *= $this->{ 'ContentScale' };
        }
        $fXPosTmp = int( $fXPosTmp + .5 * ( $fXPosTmp <=> 0 ) );
        $fYPosTmp = int( $fYPosTmp + .5 * ( $fYPosTmp <=> 0 ) );

        $theLogger->logMessage(   "@("
                                . $fXPosTmp
                                . ",". $fYPosTmp
                                . ")>" . $sTextTmp . "<"
                              );
    }
    return 0;
}

#-----------------------------------------------------------------------
# Write vector
#-----------------------------------------------------------------------
sub writeVector{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeVector()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Vector object hash (having both kernel object and pom object)
    #
    my $hrefVectorObjPar = shift;

    #---- Get actual objects
    my $a2wObjTmp = $hrefVectorObjPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefVectorObjPar->{ 'POMOBJ' };

    #---- Skip, if skip mode is on
    if ( $this->{ 'SkipMode' } == $TRUE ){
        return 0;
    }

    return 0;
}

# V120a Begin
#-----------------------------------------------------------------------
# Write annotation
#-----------------------------------------------------------------------
sub writeAnnotation{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeAnnotation()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Annotation object hash (having both kernel object and pom object)
    #
    my $hrefAnnotObjPar = shift;

    #---- Get actual objects
    my $a2wObjTmp = $hrefAnnotObjPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefAnnotObjPar->{ 'POMOBJ' };

    #---- Skip, if skip mode is on
    if ( $this->{ 'SkipMode' } == $TRUE ){
        return 0;
    }

    #---- Add annotation to body
    #---- Get active page context
    my $pgHandleTmp = $this->{ 'ActivePage' };

    #---- Get active block
    my $blkActiveTmp = $pgHandleTmp->{ 'ActiveBlock' };

    #---- Evaluate object left within block
    my $iBlkLeftTmp = $blkActiveTmp->getStartX();
    my $iRelXPosTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS } - $iBlkLeftTmp;
    if ( $iRelXPosTmp != 0 ){
        if ( $this->{ 'UnitBase' } eq "px" ){
            $iRelXPosTmp *= $this->{ 'ContentScale' };
        }
        $iRelXPosTmp = int( $iRelXPosTmp + .5 * ( $iRelXPosTmp <=> 0 ) );
    }

    #---- Evaluate object top within block
    my $iBlkTopTmp = $blkActiveTmp->getStartY();
    my $iRelYPosTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS } - $iBlkTopTmp;
    if ( $iRelYPosTmp != 0 ){
        if ( $this->{ 'UnitBase' } eq "px" ){
            $iRelYPosTmp *= $this->{ 'ContentScale' };
        }
        $iRelYPosTmp = int( $iRelYPosTmp + .5 * ( $iRelYPosTmp <=> 0 ) );
    }

    #---- Fetch annotation attributes
    my $fAnnotWidthTmp  = $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_ANNOTATION_WIDTH };
    my $fAnnotHeightTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_ANNOTATION_HEIGHT };
    my $fAnnotBorderWidthTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_ANNOTATION_BORDER_WIDTH };
    my $bAddBorderStyleTmp = ( $fAnnotBorderWidthTmp > 0 ) ? $TRUE : $FALSE;
    if ( $this->{ 'UnitBase' } eq "px" ){
        $fAnnotWidthTmp  *= $this->{ 'ContentScale' };
        $fAnnotHeightTmp *= $this->{ 'ContentScale' };
        $fAnnotBorderWidthTmp *= $this->{ 'ContentScale' };
    }
    $fAnnotWidthTmp = int( $fAnnotWidthTmp + .5 * ( $fAnnotWidthTmp <=> 0 ) );
    $fAnnotHeightTmp = int( $fAnnotHeightTmp + .5 * ( $fAnnotHeightTmp <=> 0 ) );
    $fAnnotBorderWidthTmp = int( $fAnnotBorderWidthTmp + .5 * ( $fAnnotBorderWidthTmp <=> 0 ) );

    my @arrClassTmp   = ();
    my $sClassTmp     = '';
    my $sAnnotStyleTmp = ' style="';
    if ( $iRelXPosTmp != 0 ){
        @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' };
        $sAnnotStyleTmp .= 'left:' . $iRelXPosTmp . $this->{ 'UnitBase' } . ';';
    }
    else {
        @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' };
        $sAnnotStyleTmp .= 'left:0' . $this->{ 'UnitBase' } . ';';
    }
    if ( $iRelYPosTmp != 0 ){
        if ( "@arrClassTmp" !~ /$this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' }/ ){ @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' }; }
        $sAnnotStyleTmp .= 'top:' . $iRelYPosTmp . $this->{ 'UnitBase' } . ';';
    }
    else {
        if ( "@arrClassTmp" !~ /$this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' }/ ){ @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' }; }
        $sAnnotStyleTmp .= 'top:0' . $this->{ 'UnitBase' } . ';';
    }
    if ( @arrClassTmp > 0 ){
        $sClassTmp = ' class="' . "@arrClassTmp" . '"';
    }

    $sAnnotStyleTmp .= 'width:'  . $fAnnotWidthTmp . $this->{ 'UnitBase' } . ';';
    $sAnnotStyleTmp .= 'height:'  . $fAnnotHeightTmp . $this->{ 'UnitBase' } . ';';
    # Force the annotation to be on top of all (z-index: 999), so that the link is active
    # all over the box area (otherwise, active only in non text area in the box)
    $sAnnotStyleTmp .= 'z-index:999;';
    if ( $bAddBorderStyleTmp == $TRUE ){
        $sAnnotStyleTmp .= 'border-width:'  . $fAnnotBorderWidthTmp . $this->{ 'UnitBase' } . ';';
        $sAnnotStyleTmp .= 'border-style:solid;';
        $sAnnotStyleTmp .= 'border-color:#' . sprintf( "%06X", $pomObjTmp->{ $a2w::core::dm::Constants::AT_COLOR } ) . ';';
    }
    else {
        $sAnnotStyleTmp .= 'border:none;';
    }
    $sAnnotStyleTmp .= '"';

    # V124d Begin
    my $sElementTmp =   "\n"
                      . '<a'
                      . $sClassTmp
                      . $sAnnotStyleTmp
                      . ' title="' . $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_ANNOTATION_URL } . '"'
                      . ' href="' . $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_ANNOTATION_URL } . '"'
                      . '>'
                      . '</a>'
                      ;
    $this->{ 'Content' } .= $sElementTmp;
    $this->{ 'ResponsiveContent' } .= $sElementTmp;
    # V124d End

    if ( $bLog == $TRUE ){
        #---- Fetch annotation position
        my $fXPosTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS };
        my $fYPosTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS };
        if ( $this->{ 'UnitBase' } eq "px" ){
            $fXPosTmp *= $this->{ 'ContentScale' };
            $fYPosTmp *= $this->{ 'ContentScale' };
        }
        $fXPosTmp = int( $fXPosTmp + .5 * ( $fXPosTmp <=> 0 ) );
        $fYPosTmp = int( $fYPosTmp + .5 * ( $fYPosTmp <=> 0 ) );

        $theLogger->logMessage(   "-->Annotation="
                                . ", Width=" . $fAnnotWidthTmp
                                . ", Length=" . $fAnnotHeightTmp
                                . ", StartPos=(X:" . $fXPosTmp . ", Y:" . $fYPosTmp . ")"
                              );
    }
    return 0;
}
# V120a End

#---- Block related writing methods ----#
#-----------------------------------------------------------------------
# Begin block
#-----------------------------------------------------------------------
sub beginBlock{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "beginBlock()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Block
    # 2. Skip flag
    #
    my $blkToBeginPar = shift;
    my $bSkipPar = shift;

    $this->{ 'BlockTextId' } = 0; # V124c Change

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::beginBlock( $blkToBeginPar, $bSkipPar );
    if ( $iRetTmp < 0 ){
        return $iRetTmp;
    }
    if ( $bSkipPar == $TRUE ){
        return 0;
    }

    #---- Get active page context
    my $pgHandleTmp = $this->{ 'ActivePage' };

    #---- Get active page
    my $dbPageTmp = $pgHandleTmp->{ 'POM' };

    #---- Get block start X
    my $iBlkLeftTmp  = $blkToBeginPar->getStartX();
    my $iBlkLeft2Tmp = $iBlkLeftTmp;
    if ( $this->{ 'UnitBase' } eq "px" ){
        $iBlkLeftTmp *= $this->{ 'ContentScale' };
    }
    $iBlkLeftTmp = int( $iBlkLeftTmp + .5 * ( $iBlkLeftTmp <=> 0 ) );

    my $sBlkIdTmp   = $blkToBeginPar->getId();
    my $reDefPfxTmp = qr/$a2w::core::dm::Constants::BLK_PAGE_DEF_PREFIX/;
    if ( $sBlkIdTmp =~ $reDefPfxTmp ){
        $sBlkIdTmp =~ s/$reDefPfxTmp/page/g;
    }
    my $sAttrIdTmp = 'id="' . $sBlkIdTmp . '"';

    # $V106 Begin
    my $sStyleTmp =    'style="'
                     . 'position:absolute;'
                     . 'left:' . $iBlkLeftTmp . $this->{ 'UnitBase' } . ';'
                     ;

    #---- If previous block is table, evaluate start Y based on previous table rows
    my $iTopTmp = 0;
    if ( $this->{ 'PageCount' } == 1 ){ # First page
        $iTopTmp = $blkToBeginPar->getStartY();
    }
    else { # Following pages
        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "Block " . $blkToBeginPar->getId() . " start Y:>" . $blkToBeginPar->getStartY() . "<" );
        }
        $iTopTmp = $blkToBeginPar->getStartY() - $pgHandleTmp->{ 'StartY' };
    }

    # V124d Begin
    my $bPrevBlkIsTableTmp = $FALSE;
    if ( $this->{ 'PrevBlock' } != undef && $this->{ 'PrevBlock' }->isTable() == $TRUE ){ $bPrevBlkIsTableTmp = $TRUE; }
    # V124d End

    my $iTableHeightTmp = 0;
    if ( $bPrevBlkIsTableTmp == $TRUE && $this->{ 'TableHeight' } > 0 ){
        # $V109 Begin
        $iTableHeightTmp = ( $this->{ 'TableHeight' } / $this->getOutputRes() ) * $pgHandleTmp->{ 'ContentRes' };
        if ( $bLog == $TRUE ){
            my $iPrevBlkHeightTmp = $this->{ 'PrevBlock' }->getContentHeight();

            $theLogger->logMessage( "Previous block " . $this->{ 'PrevBlock' }->getId() . " is table block" );
            $theLogger->logMessage( "Table height:>" . $iTableHeightTmp . "< Previous block height:>" . $iPrevBlkHeightTmp . "<" );
        }
        $this->{ 'BlockTopDiff' } = ( $this->{ 'BlockTop' } + $iTableHeightTmp ) - $iTopTmp;
        # $V109 End
    }
    # $V109 Begin
    if ( $bLog == $TRUE ){
        $theLogger->logMessage( "Actual Top:>" . $iTopTmp . "< Evaluated Top:>" . ( $iTopTmp + $this->{ 'BlockTopDiff' } ) . "< Block Top Difference:>" . $this->{ 'BlockTopDiff' } . "<" );
        $theLogger->logMessage( "Content height:>" . $blkToBeginPar->getContentHeight() . "<" );
    }
    # $V109 End

    $iTopTmp += $this->{ 'BlockTopDiff' };

    # $V109 Begin
    if ( $this->{ 'FirstBlock' } == undef && $iTopTmp < 0 ){ $iTopTmp = 0; }
    # $V109 End
    $this->{ 'BlockTop' } = $iTopTmp;

    if ( $this->{ 'UnitBase' } eq "px" ){
        $iTopTmp *= $this->{ 'ContentScale' };
    }
    $iTopTmp = int( $iTopTmp + .5 * ( $iTopTmp <=> 0 ) );
    $sStyleTmp .= 'top:' . $iTopTmp . $this->{ 'UnitBase' } . ';';
    # $V106 End

    #---- Evaluate width
    #my $iBlkWidthTmp = $blkToBeginPar->getWidth(); # V124 Change
    my $iBlkWidthTmp = $blkToBeginPar->getContentWidth(); # V124 Change
    if ( $iBlkWidthTmp == 0 ){    # Undefined block
        $iBlkWidthTmp = $blkToBeginPar->getEndX() - $iBlkLeft2Tmp;
    }
    my $iPageWidthTmp = $dbPageTmp->getPageWidth();
    if ( ( $iBlkLeft2Tmp + $iBlkWidthTmp ) > $iPageWidthTmp ){
        $iBlkWidthTmp = $iPageWidthTmp - $iBlkLeft2Tmp;
    }
    if ( $this->{ 'UnitBase' } eq "px" ){
        $iBlkWidthTmp *= $this->{ 'ContentScale' };
    }
    $iBlkWidthTmp = int( $iBlkWidthTmp + .5 * ( $iBlkWidthTmp <=> 0 ) );
    $sStyleTmp .= 'width:' . $iBlkWidthTmp . $this->{ 'UnitBase' } . ';';

    #---- Use configured height
    my $iBlkHeightTmp = $blkToBeginPar->getHeight();
    # V124a Begin
    my $iBlkContentHeightTmp = $blkToBeginPar->getContentHeight();
    if ( $bLog == $TRUE ){
        $theLogger->logMessage( "Block:>" . $sBlkIdTmp . " Height:>" . $iBlkHeightTmp . " Content height:>" . $iBlkContentHeightTmp . "<" );
    }
    if ( $iBlkContentHeightTmp > $iBlkHeightTmp ){ $iBlkHeightTmp = $iBlkContentHeightTmp; }

    if ( $iBlkHeightTmp != 0 ){
        if ( $this->{ 'UnitBase' } eq "px" ){
            $iBlkHeightTmp *= $this->{ 'ContentScale' };
        }
        $iBlkHeightTmp = int( $iBlkHeightTmp + .5 * ( $iBlkHeightTmp <=> 0 ) );
        $sStyleTmp .= 'height:' . $iBlkHeightTmp . $this->{ 'UnitBase' } . ';';
    }
    # V124a End

    $sStyleTmp .= '"';

    #---- Start block
    # $V105 Begin
    # V124a Begin
    my $hrefHTMLConfTmp = $blkToBeginPar->getOutputConfig( 'html' );
    if ( $hrefHTMLConfTmp && defined $hrefHTMLConfTmp->{ 'CSSDiv' } ){
        $sStyleTmp = $hrefHTMLConfTmp->{ 'CSSDiv' };
    }

    my $sRespStyleTmp = $sStyleTmp;
    my $hrefRespHTMLConfTmp = $blkToBeginPar->getOutputConfig( 'responsivehtml' );
    if ( $hrefRespHTMLConfTmp && defined $hrefRespHTMLConfTmp->{ 'CSSDiv' } ){
        $sRespStyleTmp = $hrefRespHTMLConfTmp->{ 'CSSDiv' };
        if ( $sRespStyleTmp !~ /style=".*"/ ){ $sRespStyleTmp .= ' ' . $sStyleTmp; }
        #else { ...TODO... }
    }
    # V124a End

    # V124d Begin
    if ( $blkToBeginPar->isComposite() == $TRUE ){
		# Do not apply any style to composite block, sub blocks will have their own
		$sStyleTmp = '';
    }
    # V124d End

    my $sBlkInfoTmp = $blkToBeginPar->getInfo();
    if ( $sBlkInfoTmp ne "" ){
        # V124d Begin
        my $sElementTmp = "\n" . '<div ' . $sAttrIdTmp . ' ' . $sBlkInfoTmp;
        if ( $sStyleTmp ne '' ){ $sElementTmp .= ' ' . $sStyleTmp; }
        $sElementTmp .= '>';
        $this->{ 'Content' } .= $sElementTmp;

        $sRespStyleTmp =~ s/position:absolute;left:[^;]*;top:[^;]*;//;
        $sElementTmp = "\n" . '<div ' . $sAttrIdTmp . ' ' . $sBlkInfoTmp;
        if ( $sRespStyleTmp ne '' ){ $sElementTmp .= ' ' . $sRespStyleTmp; }
        $sElementTmp .= '>';
        $sElementTmp =~ s/id="/id="r-/;
        $this->{ 'ResponsiveContent' } .= $sElementTmp;
        # V124d End
    }
    else {
        # V124d Begin
        my $sElementTmp = "\n" . '<div ' . $sAttrIdTmp;
        if ( $sStyleTmp ne '' ){ $sElementTmp .= ' ' . $sStyleTmp; }
        $sElementTmp .= '>';
        $this->{ 'Content' } .= $sElementTmp;

        $sRespStyleTmp =~ s/position:absolute;left:[^;]*;top:[^;]*;//;
        if ( $blkToBeginPar->isTable() == $TRUE ){ $sRespStyleTmp =~ s/height:[^;]*;//; }
        $sElementTmp = "\n" . '<div ' . $sAttrIdTmp;
        if ( $sRespStyleTmp ne '' ){ $sElementTmp .= ' ' . $sRespStyleTmp; }
        $sElementTmp .= '>';
        $sElementTmp =~ s/id="/id="r-/;
        $this->{ 'ResponsiveContent' } .= $sElementTmp;
        # V124d End
    }
    # $V105 End

    # $V106 Begin
    $this->{ 'TableHeight' } = 0;
    $this->{ 'PrevBlock' } = $blkToBeginPar;
    if ( $this->{ 'FirstBlock' } == undef ){ $this->{ 'FirstBlock' } = $blkToBeginPar; }
    # $V106 End

    return 0;
}

#-----------------------------------------------------------------------
# End block
#-----------------------------------------------------------------------
sub endBlock{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "endBlock()" );
    }

    #---- Fetch parameter(s)
    #
    # 2. Skip flag
    #
    my $blkToEndPar = shift;
    my $bSkipPar = shift;

    if ( $bSkipPar == $FALSE ){
        # V124d Begin
        my $sElementTmp = "\n" . '</div>';
        $this->{ 'Content' } .= $sElementTmp;
        $this->{ 'ResponsiveContent' } .= $sElementTmp;
        # V124d End
    }

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::endBlock( $blkToEndPar, $bSkipPar );

    # $V106 Begin
    #---- Get active page context
    my $pgHandleTmp = $this->{ 'ActivePage' };

    if (    $blkToEndPar->isTable() == $TRUE
         && $this->{ 'TableHeight' } > 0
       ){
        $iTableHeightTmp = ( $this->{ 'TableHeight' } / $this->getOutputRes() ) * $pgHandleTmp->{ 'ContentRes' };
        $pgHandleTmp->{ 'ContentHeight' } = $this->{ 'BlockTop' } + $iTableHeightTmp;
    }
    # $V109 Begin
    #else {
    #    $pgHandleTmp->{ 'ContentHeight' } = $this->{ 'BlockTop' } + $blkToEndPar->getContentHeight();
    #}
    # $V106 End
    elsif ( $blkToEndPar->isParagraph() == $TRUE ){
        $pgHandleTmp->{ 'ContentHeight' } = $this->{ 'BlockTop' } + $blkToEndPar->getContentHeight();
    }
    elsif ( $blkToEndPar->isSkip() == $TRUE ){
        $pgHandleTmp->{ 'ContentHeight' } = $this->{ 'BlockTop' } + ( $blkToEndPar->getContentHeight() * 0.54 );
    }

    #if ( $bSkipPar == $TRUE ){ $this->{ 'BlockTopDiff' } -= $blkToEndPar->getContentHeight(); } # $V114 Change
    # $V109 End

    return $iRetTmp;
}

#---- Paragraph related writing methods ----#
#-----------------------------------------------------------------------
# Begin paragraph
#-----------------------------------------------------------------------
sub beginParagraph{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "beginParagraph()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Block
    #
    my $blkParagraphPar = shift;

    return 0;
}

#-----------------------------------------------------------------------
# Write pargraph line
#-----------------------------------------------------------------------
sub writeParagraphLine{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeParagraphLine()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Array of line objects
    # 2. Line count
    # 3. Line gap (vertical difference between previous and current line, zero for first line)
    #
    my $arefObjectsPar = shift;
    my $iLineCountPar  = shift;
    my $iLineGapPar    = shift;

    my @arrObjsTmp = @{ $arefObjectsPar };
    if ( @arrObjsTmp <= 0 ){
        return -1;
    }

    # V116g Begin
    #---- Sort objects based on input sequence id
    @arrObjsTmp = sort { $a->{ 'POMOBJ' }{ $a2w::core::dm::Constants::AT_OBJSEQID } <=> $b->{ 'POMOBJ' }{ $a2w::core::dm::Constants::AT_OBJSEQID } } @arrObjsTmp;
    # V116g End

    #---- Fetch objects list
    my $bAllImageTmp = $TRUE;
    foreach my $a2wObjTmp ( @arrObjsTmp ){
        if (    $bAllImageTmp
             && $a2wObjTmp->{ 'POMOBJ' }{ $a2w::core::dm::Constants::AT_OBJTYPE } != $a2w::core::dm::Constants::OT_IMAGE
           ){
            $bAllImageTmp = $FALSE;
        }
    
        #---- Write object
        $this->writeObject( $a2wObjTmp );
    }
    # $V106 Begin
    # Removed adding break <br> (when there is no image)
    # $V106 End

    # $V101 Begin
    # Removed adding zero padding
    # $V101 End

    return 0;
}

#-----------------------------------------------------------------------
# End paragraph
#-----------------------------------------------------------------------
sub endParagraph{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "endParagraph()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Block
    #
    my $blkParagraphPar = shift;

    return 0;
}

#---- Table related writing methods ----#
#-----------------------------------------------------------------------
# Begin table
#-----------------------------------------------------------------------
sub beginTable{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "beginTable()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Block
    # 2. Columns X positions array
    # 3. Starting Y position
    #
    my $blkTablePar        = shift;
    my $arefColumnsXPosPar = shift;
    my $iStartYPar         = shift;

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::beginTable( $blkTablePar, $arefColumnsXPosPar, $iStartYPar );
    if ( $iRetTmp < 0 ){
        return $iRetTmp;
    }

    #---- Evaluate table position
    my $fXPosTmp = $arefColumnsXPosPar->[ 0 ];
    my $fYPosTmp = $iStartYPar;
    my $iBlockWidthTmp = $blkTablePar->getWidth();
    my $fTableWidthTmp = $iBlockWidthTmp;

    if ( $this->{ 'UnitBase' } eq "px" ){
        $fXPosTmp *= $this->{ 'ContentScale' };
        $fYPosTmp *= $this->{ 'ContentScale' };
        $fTableWidthTmp *= $this->{ 'ContentScale' };
    }
    $fXPosTmp = int( $fXPosTmp + .5 * ( $fXPosTmp <=> 0 ) );
    $fYPosTmp = int( $fYPosTmp + .5 * ( $fYPosTmp <=> 0 ) );
    $fTableWidthTmp = int( $fTableWidthTmp + .5 * ( $fTableWidthTmp <=> 0 ) );

    # V124a Begin
    my $hrefHTMLConfTmp = $blkTablePar->getOutputConfig( 'html' );
    my $sWidthPropTmp = 'width:' . $fTableWidthTmp . $this->{ 'UnitBase' } . ';';
    my $sCSSTmp = 'style="' . $sWidthPropTmp . '"';
    if ( $hrefHTMLConfTmp && defined $hrefHTMLConfTmp->{ 'CSSTable' } ){
        my $sTblCSSTmp = $hrefHTMLConfTmp->{ 'CSSTable' };
        if ( $sTblCSSTmp =~ /style="(.*)"/ ){
            #---- Check and add width property (if not given already)
            $sCSSTmp = '';
            if ( $1 !~ /width:.*;/ ){
                my $sNewStyleTmp = $sWidthPropTmp . $1;
                $sTblCSSTmp =~ s/(.*style=")(.*)(".*)/$1$sNewStyleTmp$3/;
            }
        }

        # Final CSS = <Block def given CSS> + <Generated CSS>
        $sCSSTmp = $sTblCSSTmp . ' ' . $sCSSTmp;
    }

    my $sRespCSSTmp = 'style="' . $sWidthPropTmp . '"';
    my $hrefRespHTMLConfTmp = $blkTablePar->getOutputConfig( 'responsivehtml' );
    if ( $hrefRespHTMLConfTmp && defined $hrefRespHTMLConfTmp->{ 'CSSTable' } ){
        my $sTblCSSTmp = $hrefRespHTMLConfTmp->{ 'CSSTable' };
        if ( $sTblCSSTmp =~ /style="(.*)"/ ){
            #---- Check and add width property (if not given already)
            $sRespCSSTmp = '';
            if ( $1 !~ /width:.*;/ ){
                my $sNewStyleTmp = $sWidthPropTmp . $1;
                $sTblCSSTmp =~ s/(.*style=")(.*)(".*)/$1$sNewStyleTmp$3/;
            }
        }

        # Final CSS = <Block def given CSS> + <Generated CSS>
        $sRespCSSTmp = $sTblCSSTmp . ' ' . $sRespCSSTmp;
    }
    # V124a End

    #---- Begin table
    # V124d Begin
    $this->{ 'Content' } .= "\n" . '<table ' . $sCSSTmp . '>';
    $this->{ 'ResponsiveContent' } .= "\n" . '<table ' . $sRespCSSTmp . '>';
    # V124d End

    #---- Evaluate and write column group ----#
    my @arrXPosTmp = @{ $arefColumnsXPosPar };

    #---- Add block width as last element
    @arrXPosTmp[ ( $#arrXPosTmp + 1 ) ] = $iBlockWidthTmp + $arefColumnsXPosPar->[ 0 ];

    #---- Iterate x positions and evaluate column percentages
    my @arrColGroupTmp = ();
    for $x ( 0 .. ( @arrXPosTmp - 2 ) ){
        @arrColGroupTmp[ ( $#arrColGroupTmp + 1 ) ] = ( ( $arrXPosTmp[ $x + 1 ] - $arrXPosTmp[ $x ] ) / $iBlockWidthTmp ) * 100;
    }

    #---- Write column group
    if ( @arrColGroupTmp > 0 ){
        #---- Begin column group
        $this->{ 'Content' } .=   "\n" . '<colgroup>';

        #---- Write columns
        foreach $c ( @arrColGroupTmp ){
            $this->{ 'Content' } .=   "\n" . '<col style="width:' . int( $c + .5 * ( $c <=> 0 ) ) . '%">';
        }

        #---- End column group
        $this->{ 'Content' } .=   "\n" . '</colgroup>';
    }

    return 0;
}

#-----------------------------------------------------------------------
# Begin table header
#-----------------------------------------------------------------------
sub beginTableHeader{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "beginTableHeader()" );
    }

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::beginTableHeader();
    if ( $iRetTmp < 0 ){
        return $iRetTmp;
    }

    # V124d Begin
    my $sElementTmp = "\n" . '<thead>';
    $this->{ 'Content' } .= $sElementTmp;
    $this->{ 'ResponsiveContent' } .= $sElementTmp;
    # V124d End

    return 0;
}

#-----------------------------------------------------------------------
# End table header
#-----------------------------------------------------------------------
sub endTableHeader{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "endTableHeader()" );
    }

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::endTableHeader();
    if ( $iRetTmp < 0 ){
        return $iRetTmp;
    }

    # V124d Begin
    my $sElementTmp = "\n" . '</thead>';
    $this->{ 'Content' } .= $sElementTmp;
    $this->{ 'ResponsiveContent' } .= $sElementTmp;
    # V124d End

    # $V106 Begin
    $this->{ 'TableHdrHeight' } += ( $this->{ 'TableCellPadding' } * 2 );
    $this->{ 'TableHeight' } += $this->{ 'TableHdrHeight' };
    # $V106 End

    return 0;
}

#-----------------------------------------------------------------------
# Begin table body
#-----------------------------------------------------------------------
sub beginTableBody{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "beginTableBody()" );
    }

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::beginTableBody();
    if ( $iRetTmp < 0 ){
        return $iRetTmp;
    }

    # V124d Begin
    my $sElementTmp = "\n" . '<tbody>';
    $this->{ 'Content' } .= $sElementTmp;
    $this->{ 'ResponsiveContent' } .= $sElementTmp;
    # V124d End

    return 0;
}

#-----------------------------------------------------------------------
# End table body
#-----------------------------------------------------------------------
sub endTableBody{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "endTableBody()" );
    }

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::endTableBody();
    if ( $iRetTmp < 0 ){
        return $iRetTmp;
    }

    # V124d Begin
    my $sElementTmp = "\n" . '</tbody>';
    $this->{ 'Content' } .= $sElementTmp;
    $this->{ 'ResponsiveContent' } .= $sElementTmp;
    # V124d End

    return 0;
}

#-----------------------------------------------------------------------
# Begin table footer
#-----------------------------------------------------------------------
sub beginTableFooter{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "beginTableFooter()" );
    }

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::beginTableFooter();
    if ( $iRetTmp < 0 ){
        return $iRetTmp;
    }

    # V124d Begin
    my $sElementTmp = "\n" . '<tfoot>';
    $this->{ 'Content' } .= $sElementTmp;
    $this->{ 'ResponsiveContent' } .= $sElementTmp;
    # V124d End

    return 0;
}

#-----------------------------------------------------------------------
# End table footer
#-----------------------------------------------------------------------
sub endTableFooter{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "endTableFooter()" );
    }

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::endTableFooter();
    if ( $iRetTmp < 0 ){
        return $iRetTmp;
    }

    # V124d Begin
    my $sElementTmp = "\n" . '</tfoot>';
    $this->{ 'Content' } .= $sElementTmp;
    $this->{ 'ResponsiveContent' } .= $sElementTmp;
    # V124d End

    return 0;
}

#-----------------------------------------------------------------------
# Begin row
#-----------------------------------------------------------------------
sub beginRow{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "beginRow()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Row number
    # 2. Page id (where row starts on) # V115 Change
    #
    my $iRowNrPar  = shift;
    my $iPageIdPar = shift; # V115 Change

    # V124d Begin
    my $sElementTmp = "\n" . '<tr>';
    $this->{ 'Content' } .= $sElementTmp;
    $this->{ 'ResponsiveContent' } .= $sElementTmp;
    # V124d End

    # $V106 Begin
    $this->{ 'TableRowHeight' } = 0;
    $this->{ 'TableRowSpan' }   = 0;
    # $V106 End

    return 0;
}

#-----------------------------------------------------------------------
# Write cell
#-----------------------------------------------------------------------
sub writeCell{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeCell()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Array of cell objects
    # 2. Column header name
    # 3. Column id (as in on block definition)
    #
    my $arefObjectsPar = shift;
    my $sColHdrNamePar = shift;
    my $sColIdPar      = shift;
    if ( $bLog == $TRUE ){
        $theLogger->logMessage( "Header Name:>" . $sColHdrNamePar . "< ColumnId:>" . $sColIdPar . "<" );
    }

    #---- Fetch objects list
    my @arrObjsTmp = @{ $arefObjectsPar };

    my $sTableDataElementNameTmp = "td";
    if ( ( $this->{ 'InTable' } & 2 ) != 0 ){
        $sTableDataElementNameTmp = "th";
    }

    #---- Write cell contents ----#
    if ( @arrObjsTmp <= 0 ){
        # V124d Begin
        my $sElementTmp = "\n" . '<' . $sTableDataElementNameTmp . '></' . $sTableDataElementNameTmp . '>';
        $this->{ 'Content' } .= $sElementTmp;
        $this->{ 'ResponsiveContent' } .= $sElementTmp;
        # V124d End
        return 0;
    }

    # $V106 Begin
    my $iFontIdTmp       = -1;
    my $iRowSpanCntTmp   = 1;
    my $a2wObjTmp        = undef;
    my $pomObjTmp        = undef;
    my $a2wFontTmp       = undef;

    #---- Group all cell texts ----#
    my @arrTextsTmp      = ();
    my $sTextTmp         = "";
    my $hrefTableInfoTmp = {
          'Cell'   => $TRUE
        , 'ColHdr' => $sColHdrNamePar
        , 'ColId'  => $sColIdPar
    };
    my $iPrevYPosTmp     = 0;  # $V104 Change

    foreach my $hrefObjTmp ( @arrObjsTmp ){
        if (    ref( $hrefObjTmp ) eq "HASH"
             && defined $hrefObjTmp->{ 'A2WOBJ' }
             && defined $hrefObjTmp->{ 'POMOBJ' }
           ){
            $a2wObjTmp = $hrefObjTmp->{ 'A2WOBJ' };
            $pomObjTmp = $hrefObjTmp->{ 'POMOBJ' };

            if ( $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJTYPE } == $a2w::core::dm::Constants::OT_TEXT ){
                # $V104 Begin
                if ( $iPrevYPosTmp == 0 ){ $iPrevYPosTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS }; }
                # $V104 End

                #---- Collect texts with same style
                @arrTextsTmp[ $#arrTextsTmp + 1 ] = $hrefObjTmp;

                #---- Update font table, if font is added to it already ----#
                if ( @arrTextsTmp == 1 ){
                    #---- Fetch text font
                    my $a2wFontTmp = @arrTextsTmp[ 0 ]->{ 'A2WOBJ' }->getFont();
                    $iFontIdTmp = @arrTextsTmp[ 0 ]->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_FONTID };

                    #---- Update resource table with text used font
                    if ( not defined( $this->{ 'FontTable' }{ $iFontIdTmp } ) ){
                        $this->updateResourceTable( $iFontIdTmp, $a2wFontTmp, @arrTextsTmp[ 0 ] );
                    }
                    if ( $this->{ 'PrevFontId' } != $iFontIdTmp ){
                        $this->{ 'PrevFontId' } = $iFontIdTmp;
                    }
                    if ( $this->{ 'TableRowHeight' } < $this->{ 'FontTable' }->{ $iFontIdTmp }{ 'HEIGHT' } ){
                        $this->{ 'TableRowHeight' } = $this->{ 'FontTable' }->{ $iFontIdTmp }{ 'HEIGHT' };
                    }
                }

                # $V104 Begin
                #if ( $sTextTmp ne "" ){ $sTextTmp .= ' '; }
                if ( $sTextTmp ne "" && $iPrevYPosTmp == $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS } ){ $sTextTmp .= ' '; }
                if ( $iPrevYPosTmp != $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS } ){
                    $iPrevYPosTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS };
                    $iRowSpanCntTmp++; # $V106 Change
                    # $V109 Begin
                    $sTextTmp .= '<br>';
                    # $V109 End
                }
                # $V104 End
                $sTextTmp .= $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_VALUE };
            }
            #elsif ( $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJTYPE } == $a2w::core::dm::Constants::OT_LINE ){
            #    $this->writeLine( $hrefObjTmp );
            #}
        }
        else {
            $sTextTmp .= $hrefObjTmp;
        }
    } # foreach my $hrefObjTmp ( @arrObjsTmp )

    # V124d Begin
    my $sOrgTextTmp = $sTextTmp;
    if ( $sTextTmp ne "" ){
        #---- Begin cell
        my $sElementTmp = "\n" . '<' . $sTableDataElementNameTmp;
        $this->{ 'Content' } .= $sElementTmp;
        $this->{ 'ResponsiveContent' } .= $sElementTmp;

        if ( @arrTextsTmp > 0 ){
            #---- Escape text for special characters
            # $V109 Begin
            #$sTextTmp = $this->_escapeText( $sTextTmp );
            # $V109 End

            # V124d Change: Added output format as first parameter
            $sTextTmp = $this->findAndReplaceText( 'html', @arrTextsTmp[ 0 ], $sTextTmp, $hrefTableInfoTmp ); # Find and replace text, if configured

            $sElementTmp = ' class="' . $this->{ 'FontTable' }->{ $iFontIdTmp }{ 'STYLENAME' } . '"';
            $this->{ 'Content' } .= $sElementTmp;
            $this->{ 'ResponsiveContent' } .= $sElementTmp;
        }
        $sElementTmp = '>';
        $this->{ 'Content' } .= $sElementTmp;
        $this->{ 'ResponsiveContent' } .= $sElementTmp;

        #---- Write so far collected texts
        $this->{ 'Content' } .= $sTextTmp;
        # Check whether find and replace applied to the text
        if ( $sOrgTextTmp ne $sTextTmp ){
            # Check whether any href points to div id, if yes, update div for responsive html
            if ( $sTextTmp =~ /href="\#/ ){ $sTextTmp =~ s/href="\#/href="\#r-/; }
        }
        $this->{ 'ResponsiveContent' } .= $sTextTmp;

        #---- End cell
        $sElementTmp = '</' . $sTableDataElementNameTmp . '>';
        $this->{ 'Content' } .= $sElementTmp;
        $this->{ 'ResponsiveContent' } .= $sElementTmp;
    }
    # V124d End

    #---- Reset to collect next group of texts
    $sTextTmp = "";
    @arrTextsTmp = ();

    if ( $this->{ 'TableRowSpan' } < $iRowSpanCntTmp ){ $this->{ 'TableRowSpan' } = $iRowSpanCntTmp; }
    # $V106 End

    return 0;
}

#-----------------------------------------------------------------------
# End row
#-----------------------------------------------------------------------
sub endRow{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "endRow()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Row number
    #
    my $iRowNrPar = shift;

    # V124d Begin
    my $sElementTmp = "\n" . '</tr>';
    $this->{ 'Content' } .= $sElementTmp;
    $this->{ 'ResponsiveContent' } .= $sElementTmp;
    # V124d End

    # $V106 Begin
    $this->{ 'TableRowHeight' } *= $this->{ 'TableRowSpan' }; # Multiply by row span
    $this->{ 'TableRowHeight' } += ( $this->{ 'TableCellPadding' } * 2 );
    $this->{ 'TableHeight' } += $this->{ 'TableRowHeight' };
    # $V106 End

    return 0;
}

#-----------------------------------------------------------------------
# End table
#-----------------------------------------------------------------------
sub endTable{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "endTable()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Block
    #
    my $blkTablePar = shift;

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::endTable( $blkTablePar );
    if ( $iRetTmp < 0 ){
        return $iRetTmp;
    }

    # V124d Begin
    my $sElementTmp = "\n" . '</table>';
    $this->{ 'Content' } .= $sElementTmp;
    $this->{ 'ResponsiveContent' } .= $sElementTmp;
    # V124d End

    # $V109 Begin
    $this->{ 'TableHeight' } += 10;
    # $V109 End

    # $V106 Begin
    if ( $bLog == $TRUE ){
        $theLogger->logMessage( "endTable: " . $blkTablePar->getId() . " Height:>" . $this->{ 'TableHeight' } . "<" );
    }
    # $V106 End

    return 0;
}

#-----------------------------------------------------------------------
# Initialize HTML
#-----------------------------------------------------------------------
sub _initializeHTML{
    my $this = shift;

    #---- Write html begin
    my $handleFileTmp = $this->{ 'fHandle' };
    printf $handleFileTmp ( '<!DOCTYPE html>' );
    # V116e Begin
    if ( $this->{ 'HTMLVersion' } == $HTML_VERSION_5 ){
        my $sOutLangTmp = $this->getOutputLanguage();
        if ( $sOutLangTmp eq "" ){ $sOutLangTmp = "en"; }
        printf $handleFileTmp ( "\n" . '<html lang="' . $sOutLangTmp . '">' );
    }
    else {
        printf $handleFileTmp ( "\n" . '<html>' );
    }
    # V116e End

    # V116d Begin
    #---- Initialize Additional header
    my $iRetTmp = $this->_initializeAdditionalHeader();
    if ( $iRetTmp < 0 ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Unable to initialze additional header, rc=$iRetTmp" ); }
        return $iRetTmp;
    }
    # V116d End

    #---- Initialize CSS
    $iRetTmp = $this->_initializeCSS();

    return $iRetTmp;
}

#-----------------------------------------------------------------------
# Finalize HTML
#-----------------------------------------------------------------------
sub _finalizeHTML{
    my $this = shift;

    #---- Write html end
    my $handleFileTmp = $this->{ 'fHandle' };
    printf $handleFileTmp ( "\n" . '</html>' );

    # V116d Begin
    my $sHTMLFilenameTmp = $this->{ 'Filename' };

    #---- Finalize additional header
    my $sHdrFilenameTmp = $this->{ 'HDR' }{ 'Filename' };
    my $iHdrRetTmp = $this->_finalizeAdditionalHeader();

    if ( $iHdrRetTmp < 0 ){ return $iHdrRetTmp; }

    #---- Read in additional header file and inline it on html ----#
    my $fHdrHandleTmp = undef;
    if ( $this->{ 'bInlineHDR' } == $TRUE ){
        if ( open( $fHdrHandleTmp, $sHdrFilenameTmp ) ){
            #---- Read data
            local $/ = undef;
            my $sAddHdrTmp = <$fHdrHandleTmp>;
            close( $fHdrHandleTmp );

            if ( $bLog == $TRUE ){ $theLogger->logMessage( "Additional header:>" . $sAddHdrTmp . "<" ); }

            #---- Close HTML
            if ( $this->{ 'PageOutput' } == $FALSE ){ $this->closeOutputStream(); }
            else { $this->closeOutputStream( $this->{ 'ActivePage' } ); }

            #---- Update HTML with additional header ----#
            my $fHTMLHandleTmp = undef;
            if ( open( $fHTMLHandleTmp, $sHTMLFilenameTmp ) ){
                #---- Read HTML data
                local $/ = undef;
                my $sHTMLTmp = <$fHTMLHandleTmp>;
                close( $fHTMLHandleTmp );
    
                #---- Remove additional header include and inline additional content
                my $sSearchTmp = $this->{ 'HDR' }{ 'IncludeStatement' };
                my $reReplaceTmp = qr/$sSearchTmp/;
                $sHTMLTmp =~ s/$reReplaceTmp/$sAddHdrTmp/;
    
                #---- Delete HTML file and recreate with in lined additional header
                unlink( $sHTMLFilenameTmp );
                if ( open( $fHTMLHandleTmp, ">$sHTMLFilenameTmp" ) ){
                    if ( $bLog == $TRUE ){ $theLogger->logMessage( "File (" . $sHTMLFilenameTmp . ") opened successfully to inline additional header" ); }
                    print $fHTMLHandleTmp ( $sHTMLTmp ); # V120b Change
                    close( $fHTMLHandleTmp );
    
                    #---- Delete additional header file
                    unlink( $sHdrFilenameTmp );
    
                    if ( $bLog == $TRUE ){ $theLogger->logMessage( $sHTMLFilenameTmp . " successfully inlined with additional header" ); }
                }
                else {
                    $fHTMLHandleTmp = undef;
                    if ( $bLog == $TRUE ){
                        $theLogger->logMessage( "Error! Unable to open file (" . $sHTMLFilenameTmp . "), reason: " . $! );
                        $theLogger->logMessage( "Warning! Unable to in line additional header on HTML" );
                    }
                    $iRetTmp = -1;
                }
            }
        }
    }
    # V116d End

    #---- Finalize CSS
    my $sCSSFilenameTmp = $this->{ 'CSS' }{ 'Filename' };    # $V102 Change
    my $iRetTmp = $this->_finalizeCSS();

    # $V102 Begin
    if ( $iRetTmp < 0 ){ return $iRetTmp; }
    if ( $this->{ 'bInlineCSS' } == $TRUE ){
        #---- Read in CSS file and inline it on html ----#
        my $fCSSHandleTmp = undef;
        if ( open( $fCSSHandleTmp, $sCSSFilenameTmp ) ){
            #---- Read CSS data
            local $/ = undef;
            my $sCSSTmp = <$fCSSHandleTmp>;
            close( $fCSSHandleTmp );

            #---- Add style tag
            #$sCSSTmp =~ s/%/%%/g; # To take care of 'transform-origin:0% 100%' rotation style # V121 Change
            $sCSSTmp = '<style>'
                       . "\n" . $sCSSTmp
                       . "\n" . '</style>'
                       ;
            if ( $bLog == $TRUE ){
                $theLogger->logMessage( "CSS>" . $sCSSTmp . "<" );
            }

            #---- Close HTML
            if ( $this->{ 'PageOutput' } == $FALSE ){
                $this->closeOutputStream();
            }
            else {
                $this->closeOutputStream( $this->{ 'ActivePage' } );
            }

            #---- Update HTML with CSS ----#
            my $fHTMLHandleTmp = undef;
            if ( open( $fHTMLHandleTmp, $sHTMLFilenameTmp ) ){
                #---- Read HTML data
                local $/ = undef;
                my $sHTMLTmp = <$fHTMLHandleTmp>;
                close( $fHTMLHandleTmp );

                #---- Remove CSS include and inline CSS content
                my $sSearchTmp = $this->{ 'CSS' }{ 'IncludeStatement' };
                my $reReplaceCSSTmp = qr/$sSearchTmp/;
                $sHTMLTmp =~ s/$reReplaceCSSTmp/$sCSSTmp/;

                #---- Delete HTML file and recreate with in lined CSS
                unlink( $sHTMLFilenameTmp );
                if ( open( $fHTMLHandleTmp, ">$sHTMLFilenameTmp" ) ){
                    if ( $bLog == $TRUE ){
                        $theLogger->logMessage( "File (" . $sHTMLFilenameTmp . ") opened successfully to inline CSS" );
                    }
                    print $fHTMLHandleTmp ( $sHTMLTmp ); # V120b Change
                    close( $fHTMLHandleTmp );

                    #---- Delete CSS file
                    unlink( $sCSSFilenameTmp );

                    if ( $bLog == $TRUE ){
                        $theLogger->logMessage( $sHTMLFilenameTmp . " successfully inlined with CSS" );
                    }
                }
                else {
                    $fHTMLHandleTmp = undef;
                    if ( $bLog == $TRUE ){
                        $theLogger->logMessage( "Error! Unable to open file (" . $sHTMLFilenameTmp . "), reason: " . $! );
                        $theLogger->logMessage( "Warning! Unable to in line CSS on HTML" );
                    }
                    $iRetTmp = -1;
                }
            }
        }
    }
    # $V102 End

    return $iRetTmp;
}

# V116d Begin
#-----------------------------------------------------------------------
# Initialize additional header
#-----------------------------------------------------------------------
sub _initializeAdditionalHeader{
    my $this = shift;

    #---- Evaluate additional header file name
    $this->{ 'HDR' }{ 'Filename' } = $this->getFilename();                 # Html file name
    my $sOutputPathTmp    = $this->getOutputPath();                              # Html output path
    my $sHdrOutputPathTmp = $sOutputPathTmp . $this->{ 'ResPaths' }{ 'HDR' }; # Additional header output path
    $this->{ 'HDR' }{ 'Filename' } =~ s/\Q$sOutputPathTmp\E/$sHdrOutputPathTmp/; # Replace html output path with additional header output path

    #---- Check and create additional header sub directory, if not exist already
    unless ( -d $sHdrOutputPathTmp ){
        #---- Create additional header output path
        unless ( mkdir( $sHdrOutputPathTmp ) ){
            my $sMsgTmp = "Unable to create sub directory (" . $this->{ 'ResPaths' }{ 'HDR' } . ") on output path (" . $sOutputPathTmp . "), reason:" . $!;
            if ( $bLog == $TRUE ){ $theLogger->logMessage( "Error! " . $sMsgTmp ); }
            return ( -1, $sMsgTmp );
        }
    }

    #---- Open additional header file
    $this->{ 'HDR' }{ 'fHandle' } = $this->open( $this->{ 'HDR' }{ 'Filename' } );
    if ( $this->{ 'HDR' }{ 'fHandle' } == undef ){
        return ( -2, "Unable to open file (" . $this->{ 'HDR' }{ 'Filename' } . ")" );
    }

    if ( $this->{ 'PageOutput' } == $TRUE ){
        $this->{ 'ActivePage' }->{ 'HDR' }{ 'Filename' } = $this->{ 'HDR' }{ 'Filename' };
    }

    return 0;
}

#-----------------------------------------------------------------------
# Finalize additional header
#-----------------------------------------------------------------------
sub _finalizeAdditionalHeader{
    my $this = shift;

    #---- Close additional header file
    $this->{ 'HDR' }{ 'Filename' } = '';
    my $iRetTmp = 0;
    if ( $this->close( $this->{ 'HDR' }{ 'fHandle' } ) == $FALSE ){
        $iRetTmp = -1;
    }
    $this->{ 'HDR' }{ 'fHandle' } = undef;

    return $iRetTmp;
}
# V116d End

#-----------------------------------------------------------------------
# Initialize CSS
#-----------------------------------------------------------------------
sub _initializeCSS{
    my $this = shift;

    #---- Evaluate CSS file name
    $this->{ 'CSS' }{ 'Filename' } = $this->getFilename();                       # Html file name
    my $sOutputPathTmp    = $this->getOutputPath();                              # Html output path
    my $sCSSOutputPathTmp = $sOutputPathTmp . $this->{ 'ResPaths' }{ 'CSS' };    # CSS output path
    $this->{ 'CSS' }{ 'Filename' } =~ s/\Q$sOutputPathTmp\E/$sCSSOutputPathTmp/; # Replace html output path with CSS output path
    $this->{ 'CSS' }{ 'Filename' } =~ s/\.html/\.css/;                           # Replace html extension with CSS

    #---- Check and create css sub directory, if not exist already
    unless ( -d $sCSSOutputPathTmp ){
        #---- Create css output path
        unless ( mkdir( $sCSSOutputPathTmp ) ){
            my $sMsgTmp = "Unable to create sub directory (" . $this->{ 'ResPaths' }{ 'CSS' } . ") on output path (" . $sOutputPathTmp . "), reason:" . $!;
            if ( $bLog == $TRUE ){
                $theLogger->logMessage( "Error! " . $sMsgTmp );
            }
            return ( -1, $sMsgTmp );
        }
    }

    #---- Open CSS file
    $this->{ 'CSS' }{ 'fHandle' } = $this->open( $this->{ 'CSS' }{ 'Filename' } );
    if ( $this->{ 'CSS' }{ 'fHandle' } == undef ){
        return ( -2, "Unable to open file (" . $this->{ 'CSS' }{ 'Filename' } . ")" );
    }

    # V102 Begin
    if ( $this->{ 'PageOutput' } == $TRUE ){
        $this->{ 'ActivePage' }->{ 'CSS' }{ 'Filename' } = $this->{ 'CSS' }{ 'Filename' };
    }
    # V102 End

	# Reset classes
	$this->{ 'CSS' }{ 'CLASSES' } = undef; # V124d Change

    return 0;
}

#-----------------------------------------------------------------------
# _writeCSSHeader
#-----------------------------------------------------------------------
sub _writeCSSHeader{
    my $this = shift;

    #---- Write CSS header
    my $fCSSHandleTmp = $this->{ 'CSS' }{ 'fHandle' };
    printf $fCSSHandleTmp ( "/* AFP2web generated CSS */" );

    #---- Write rotation styles
    printf $fCSSHandleTmp ( "\n.org90{transform-origin:0%% 100%%;-ms-transform-origin:0%% 100%%;-webkit-transform-origin:0%% 100%%;}" );
    printf $fCSSHandleTmp ( "\n.r90{transform:rotate(90deg);-webkit-transform:rotate(90deg);-moz-transform:rotate(90deg);-ms-transform:rotate(90deg);-o-transform:rotate(90deg);filter: progid:DXImageTransform.Microsoft.BasicImage(rotation=1);}" );
    printf $fCSSHandleTmp ( "\n.r180{transform:rotate(180deg);-webkit-transform:rotate(180deg);-moz-transform:rotate(180deg);-ms-transform:rotate(180deg);-o-transform:rotate(180deg);filter: progid:DXImageTransform.Microsoft.BasicImage(rotation=2);}" );
    printf $fCSSHandleTmp ( "\n.r270{transform:rotate(270deg);-webkit-transform:rotate(270deg);-moz-transform:rotate(270deg);-ms-transform:rotate(270deg);-o-transform:rotate(270deg);filter: progid:DXImageTransform.Microsoft.BasicImage(rotation=3);}" );

    #---- Write position styles
    printf $fCSSHandleTmp ( "\n.pa{position:absolute;}" ); # V116i Change
    printf $fCSSHandleTmp ( "\n.pr{position:relative;}" );

    # V122 Begin
    #---- Write body style
    my $sBodyStyleTmp =   'body{'
                        . 'margin:0;'
                        . 'padding:0;'
                        . '}';
    printf $fCSSHandleTmp ( "\n" . $sBodyStyleTmp );
    # V122 End

    #---- Write div style
    my $sDivStyleTmp =   'div{'
                       . 'border-width:0px;'
                       # V106 Begin
                       . 'display:inline-block;'
                       # V106 End
                       . '}';
    if ( ( $theLogger->getLevel() & $a2w::core::log::Logger::LEVEL_DEBUG ) != 0 ){
        $sDivStyleTmp =   'div{'
                        . 'border-width:medium;'
                        . 'border-style:solid;'
                        . 'border-color:red;'
                        # V106 Begin
                        . 'display:inline-block;'
                        # V106 End
                        . '}';
    }
    printf $fCSSHandleTmp ( "\n" . $sDivStyleTmp ); # V116i Change

    # $V107 Begin
    #---- Add line style to force zero margin
    printf $fCSSHandleTmp ( "\nhr{border:0px;margin:0px;}" );
    # $V107 End

    #---- Write table header style
    my $iTHFontHeightTmp = 10.8 * $this->{ 'FontScale' };
    $iTHFontHeightTmp = int( $iTHFontHeightTmp + .5 * ( $iTHFontHeightTmp <=> 0 ) );

    # $V106 Begin
    #printf $fCSSHandleTmp ( "\n" . 'th{font-size:' . $iTHFontHeightTmp . 'px;font-family:Arial;font-weight:bold;text-align:left;}' );
    $this->{ 'TableHdrHeight' } = $iTHFontHeightTmp;

    printf $fCSSHandleTmp ( "\n" . 'table {border:0px;border-spacing:0;margin:0px;}' );
    printf $fCSSHandleTmp ( "\n" . 'table th {padding:' . $this->{ 'TableCellPadding' } . 'px;line-height:' . $iTHFontHeightTmp . 'px;font-size:' . $iTHFontHeightTmp . 'px;font-family:Arial;font-weight:bold;text-align:left;}' );
    printf $fCSSHandleTmp ( "\n" . 'table td {padding:' . $this->{ 'TableCellPadding' } . 'px;}' );
    # $V106 End
    # $V108 Begin
    #---- Forced table first column vertical alignment as "top"
    #printf $fCSSHandleTmp ( "\n" . 'table td:first-child {vertical-align:top;}' ); # V124d Change
    # $V108 End

    # V124d Begin
    printf $fCSSHandleTmp ( "\n" . 'table td {vertical-align:top;}' );
    my $sResponsiveCSSTmp =   "\n" . 'table.bt tbody th,table.bt thead{display:none}table.bt tbody td,table.bt tfoot td,table.bt tfoot th{border:0;display:block;display:-webkit-box;display:-webkit-flex;display:-ms-flexbox;display:flex;vertical-align:top}table.bt tbody td::before,table.bt tfoot td::before,table.bt tfoot th::before{content:attr(data-th) \': \';display:inline-block;-webkit-flex-shrink:0;-ms-flex-shrink:0;flex-shrink:0;font-weight:700;width:6.5em}table.bt tbody td.bt-hide,table.bt tfoot td.bt-hide,table.bt tfoot th.bt-hide{display:none}table.bt tbody td .bt-content,table.bt tfoot td .bt-content,table.bt tfoot th .bt-content{vertical-align:top}.bt-wrapper.active{max-height:310px;overflow:auto;-webkit-overflow-scrolling:touch}table.bt.bt--no-header tbody td::before,table.bt.bt--no-header tfoot td::before{display:none}@media (max-width:1024px){#responsive_container{width:100%!important;overflow:hidden}#responsive_container div{display:block!important;margin-left:0!important;margin-top:0!important;border:5px solid transparent;box-sizing:border-box;position:relative}div:not(.center){top:0;width:100%!important}.responsive_table{width:100%!important;border-top:10px solid transparent}.responsive_table tr{border-bottom:1px solid rgba(0,0,0,.12);display:block}.responsive_notableheader td{border-bottom:1px solid rgba(0,0,0,.12)!important}.responsive_notableheader td::before{content:" "!important;display:flex!important;width:0!important}}'
                            . "\n"
                            ;
    printf $fCSSHandleTmp ( $sResponsiveCSSTmp );
    # V124d End

    #---- Style classes
    $this->{ 'CSS' }{ 'Styles' } = {
          'ROT90'  => 'r270'
        , 'ROT180' => 'r180'
        , 'ROT270' => 'org90 r90'
        , 'ABSPOS' => 'pa'
        , 'RELPOS' => 'pr'
    };

    return 0;
}

#-----------------------------------------------------------------------
# Finalize CSS
#-----------------------------------------------------------------------
sub _finalizeCSS{
    my $this = shift;

    #---- Close CSS file
    $this->{ 'CSS' }{ 'Filename' } = '';
    my $iRetTmp = 0;
    if ( $this->close( $this->{ 'CSS' }{ 'fHandle' } ) == $FALSE ){
        $iRetTmp = -1;
    }
    $this->{ 'CSS' }{ 'fHandle' } = undef;

    return $iRetTmp;
}

#-----------------------------------------------------------------------
# Escape text
#
# Escapes special characters on text for HTML presentation
#
#-----------------------------------------------------------------------
# $V110 Begin
sub _escapeText{
    my $this = shift;

    #---- Fetch parameter(s)
    #
    # 1. sTextPar (of type string)
    # 2. bEscapeSpacePar (of type bool) OPTIONAL, default is FALSE
    #
    my $sTextPar = shift;
    my $bEscapeSpacePar = $FALSE;
    if ( @_ > 0 ){
        $bEscapeSpacePar = shift;
    }

    #---- Escape table
    my %hshEscapeTableTmp = (
         '<' => '&lt;'
       , '>' => '&gt;'
    );
    if ( $bEscapeSpacePar == $TRUE ){
        %hshEscapeTableTmp->{ ' ' } = '&nbsp;';
    }

    #---- Escape characters string for reqular expression
    my $sAllEscapeCharsTmp = join '', keys %hshEscapeTableTmp;

    #---- Escape characters with equivalent escape values
    my $sEscapedTextTmp = $sTextPar;
    $sEscapedTextTmp =~ s/([\Q$sAllEscapeCharsTmp\E])/$hshEscapeTableTmp{ $1 }/g;

    return $sEscapedTextTmp;
}
# $V110 End

#-----------------------------------------------------------------------
# Fetch equivalent HTML font name
#
# Search for text font equivalent HTML font name and returns it
#
#-----------------------------------------------------------------------
sub _fetchHTMLFont{
    my $this = shift;

    #---- Fetch parameter(s)
    #
    # 1. Font name (of type string)
    #
    my $sFontNamePar = shift;

    # V116d Begin
    my $sHTMLFontNameTmp = $sFontNamePar;
    my $sHTMLFontLocationTmp = undef;
    my $sHTMLFontAddAttrTmp  = undef; # V118 Change

    my $hrefFontMapTable = $this->getFontMapTable();
    if (    $hrefFontMapTable != undef
         && defined( $hrefFontMapTable->{ $sFontNamePar } )
       ){
        $sHTMLFontNameTmp = $hrefFontMapTable->{ $sFontNamePar }{ 'font-family' };
        $sHTMLFontLocationTmp = $hrefFontMapTable->{ $sFontNamePar }{ 'font-location' };
        # V118 Begin
        $sHTMLFontAddAttrTmp = "";
        if ( defined( $hrefFontMapTable->{ $sFontNamePar }{ 'font-attributes' } ) ){
            $sHTMLFontAddAttrTmp = $hrefFontMapTable->{ $sFontNamePar }{ 'font-attributes' }
        }
        # V118 End
    }
    if ( $bLog == $TRUE ){ $theLogger->logMessage( "AFP Font=" . $sFontNamePar . " HTML Font=" . $sHTMLFontNameTmp ); }

    return ($sHTMLFontNameTmp, $sHTMLFontLocationTmp, $sHTMLFontAddAttrTmp); # V118 Change
    # V116d End
}

#-----------------------------------------------------------------------
# Fetch HTML font source for @font-face "src" attribute value
#
# Returns a string having following syntax
# ur(<font filename>)format(<font file format>)
#
#-----------------------------------------------------------------------
sub _fetchHTMLFontSource{
    my $this = shift;

    #---- Fetch parameter(s)
    #
    # 1. HTML font name (of type string)
    #
    my $sHTMLFontNamePar = shift;

    #TODO: Prepare and deliver a list
    return "";
}

#-----------------------------------------------------------------------
# Is Font Bold
#
# Checks whether given font has BOLD weight and returns true if yes
#
#-----------------------------------------------------------------------
sub _isFontBold{
    my $this = shift;

    #---- Fetch parameter(s)
    #
    # 1. Font (of type a2w::Font)
    #
    my $a2wFontPar = shift;
    return $a2wFontPar->isBold();
}

#-----------------------------------------------------------------------
# Is Font Italic
#
# Checks whether given font has Italic slant and returns true if yes
#
#-----------------------------------------------------------------------
sub _isFontItalic{
    my $this = shift;

    #---- Fetch parameter(s)
    #
    # 1. Font (of type a2w::Font)
    #
    my $a2wFontPar = shift;
    return $a2wFontPar->isItalic();
}

# $V106 Begin
#-----------------------------------------------------------------------
# Get page top
#
# Evaluates and returns current page top offset
#
#-----------------------------------------------------------------------
sub _getPageTop{
    my $this = shift;

    #---- Fetch parameter(s)
    #
    # 1. Page id
    #
    my $iPageIdPar = shift;

    if ( $this->{ 'PageOutput' } == $TRUE ){ return 0; } # V116h Change

    #---- Evaluate height for all previous pages and sum them up ----#
    my $iTopOffsetTmp = 0;
    my $pgHandleTmp = undef;

    for ( my $i = 1; $i < $iPageIdPar; $i++ ){
        #---- Get page handle
        $pgHandleTmp = $this->{ 'Pages' }{ $i };
        if ( $pgHandleTmp != undef ){
            # $V109 Begin
            #$iTopOffsetTmp += $pgHandleTmp->{ 'ContentHeight' } * 1.02;
            $iTopOffsetTmp +=   $pgHandleTmp->{ 'ContentHeight' }
                              + ( $a2w::core::dm::Constants::PAGE_MARGIN_TOP * $pgHandleTmp->{ 'ContentRes' } )
                              + ( $a2w::core::dm::Constants::PAGE_MARGIN_BOTTOM * $pgHandleTmp->{ 'ContentRes' } )
                              ;
            # $V109 End
        }
    }
    if ( $bLog == $TRUE ){
        $theLogger->logMessage( "Page " . $iPageIdPar . " Top:>" . $iTopOffsetTmp . "<" );
    }

    return $iTopOffsetTmp;
}

#-----------------------------------------------------------------------
# Get table height
#-----------------------------------------------------------------------
sub getTableHeight{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "getTableHeight()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Block
    #
    my $blkTablePar = shift;

    if ( $this->{ 'InTable' } == 0 ){ return 0; }

    return $this->{ 'TableHeight' };
}
# $V106 End

# $V116a Begin
#-----------------------------------------------------------------------
# Check whether container object type is supported or not
#
# Returns true if supported else false
#-----------------------------------------------------------------------
sub _assertObjectType{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "_assertObjectType()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. object type
    #
    my $objTypePar = shift;

    #---- Check and return true for supported object types
    if (    $objTypePar == $a2w::ContainerConstants::OBJECT_TYPE_TIFF
         || $objTypePar == $a2w::ContainerConstants::OBJECT_TYPE_PCX
         || $objTypePar == $a2w::ContainerConstants::OBJECT_TYPE_GIF
         || $objTypePar == $a2w::ContainerConstants::OBJECT_TYPE_JPEG
         || $objTypePar == $a2w::ContainerConstants::OBJECT_TYPE_PNG
         || $objTypePar == $a2w::ContainerConstants::OBJECT_TYPE_BMP
       ){
        return $TRUE;
    }

    return $FALSE;
}
# $V116a End

# $V116f Begin
#-----------------------------------------------------------------------
# Include auto page scaling header code
#
# Returns >=0 in case of succes, <0 in case of error
#-----------------------------------------------------------------------
sub _includeAutoPageScale{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "_includeAutoPageScale()" );
    }

    #---- Get HTML file handle
    my $handleFileTmp = $this->{ 'fHandle' };

    #---- Build auto page scaling header code
    my $sAutoPageScaleCodeTmp =
    # V124d Begin
    # Include jquery once and set the flag for auto zoom
      '<script src="https://code.jquery.com/jquery-3.6.0.slim.min.js" integrity="sha256-u7e5khyithlIdTpu22PHhENmPcRdFiHRjhAuHcs05RI=" crossorigin="anonymous"></script>'
    . "\n" . '<script>var bAutoZoom = true;</script>'
    . "\n"

    #. "\n" . '<script>'
    # V119 Begin
    #. "\n" . 'var _0x379f=[\'width\',\'undefined\',\'isChrome\',\'isEdgeChromium\',\'hidden\',\'test\',\'each\',\'body\',\'pushNotification\',\'top\x20left\',\'opera\',\'addons\',\'-moz-transform-origin\',\'safari\',\'zoom\',\'isBlink\',\'opr\',\'overflow-x\',\'scale(\',\'StyleMedia\',\'toString\',\'isFirefox\',\'userAgent\',\'ready\',\'-moz-transform\',\'indexOf\',\'HTMLElement\',\'isIE\',\'[object\x20SafariRemoteNotification]\',\'div\',\'css\'];(function(_0x10d2b7,_0x379fb7){var _0x235002=function(_0x4929d5){while(--_0x4929d5){_0x10d2b7[\'push\'](_0x10d2b7[\'shift\']());}};_0x235002(++_0x379fb7);}(_0x379f,0xaa));var _0x2350=function(_0x10d2b7,_0x379fb7){_0x10d2b7=_0x10d2b7-0x0;var _0x235002=_0x379f[_0x10d2b7];return _0x235002;};var calculateDocWidth=!![];var actualDocumentWidth=0x0;function getBrowser(){var _0x2833fb={\'isOpera\':!!window[_0x2350(\'0x1\')]&&!!opr[_0x2350(\'0x1b\')]||!!window[_0x2350(\'0x1a\')]||navigator[_0x2350(\'0x7\')][\'indexOf\'](\'\x20OPR/\')>=0x0,\'isFirefox\':typeof InstallTrigger!==_0x2350(\'0x11\'),\'isSafari\':/constructor/i[\'test\'](window[_0x2350(\'0xb\')])||function(_0x7fe64a){return _0x7fe64a[_0x2350(\'0x5\')]()===_0x2350(\'0xd\');}(!window[_0x2350(\'0x1d\')]||typeof safari!==_0x2350(\'0x11\')&&safari[_0x2350(\'0x18\')]),\'isIE\':![]||!!document[\'documentMode\'],\'isChrome\':navigator[_0x2350(\'0x7\')][_0x2350(\'0xa\')](\'Chrome\')!==-0x1};_0x2833fb[\'isEdge\']=!_0x2833fb[_0x2350(\'0xc\')]&&!!window[_0x2350(\'0x4\')];_0x2833fb[_0x2350(\'0x13\')]=_0x2833fb[_0x2350(\'0x12\')]&&navigator[_0x2350(\'0x7\')][_0x2350(\'0xa\')](\'Edg\')!=-0x1;_0x2833fb[_0x2350(\'0x0\')]=(_0x2833fb[_0x2350(\'0x12\')]||_0x2833fb[\'isOpera\'])&&!!window[\'CSS\'];return _0x2833fb;}function setZoom(){var _0x1e6870=actualDocumentWidth;var _0x48f738=-0x1;var _0x250b19=window[\'innerWidth\']/_0x1e6870*0x64+_0x48f738;var _0x3e30ba=getBrowser();if(_0x3e30ba[_0x2350(\'0x6\')]){$(_0x2350(\'0x17\'))[_0x2350(\'0xf\')](_0x2350(\'0x9\'),\'scale(\'+_0x250b19/0x64+\')\')[_0x2350(\'0xf\')](_0x2350(\'0x1c\'),_0x2350(\'0x19\'));}else if(_0x3e30ba[_0x2350(\'0xc\')]){$(\'body\')[_0x2350(\'0xf\')](\'-ms-transform\',_0x2350(\'0x3\')+_0x250b19/0x64+\')\')[_0x2350(\'0xf\')](\'-ms-transform-origin\',_0x2350(\'0x19\'))[\'css\'](_0x2350(\'0x2\'),_0x2350(\'0x14\'));}else{$(_0x2350(\'0x17\'))[_0x2350(\'0xf\')](_0x2350(\'0x1e\'),_0x250b19+\'%\');}}$(document)[_0x2350(\'0x8\')](function(){if(calculateDocWidth){calculateDocWidth=![];var _0x46a839=0x0;$(_0x2350(\'0xe\'))[_0x2350(\'0x16\')](function(_0x21c35f,_0x4155f8){if(/^page\d+$/[_0x2350(\'0x15\')]($(_0x4155f8)[\'attr\'](\'id\'))){if(_0x46a839<parseInt($(_0x4155f8)[\'css\'](_0x2350(\'0x10\')))){_0x46a839=parseInt($(_0x4155f8)[_0x2350(\'0xf\')](_0x2350(\'0x10\')));}}});actualDocumentWidth=_0x46a839+0x14;}if(actualDocumentWidth){setZoom();$(window)[\'resize\'](function(){setZoom();});}});'
    # V119 End
    #. "\n" . '</script>'
    #. "\n"
    # V124d Begin
    ;

    print $handleFileTmp ( $sAutoPageScaleCodeTmp );

    return 0;
}
# $V116a End

# $V124d Begin
#-----------------------------------------------------------------------
# Include responsive html JS
#
# Returns >=0 in case of succes, <0 in case of error
#-----------------------------------------------------------------------
sub _includeResponsiveJS{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "_includeResponsiveJS()" );
    }

    #---- Get HTML file handle
    my $handleFileTmp = $this->{ 'fHandle' };

    #---- Build responsive JS includes
    my $sResponsiveJSTmp = '<meta name="viewport" content="width=device-width, initial-scale=1, user-scalable=1" />';
    if ( $this->{ 'AutoPageScale' } == $FALSE ){
        $sResponsiveJSTmp .= "\n" . '<script src="https://code.jquery.com/jquery-3.6.0.slim.min.js" integrity="sha256-u7e5khyithlIdTpu22PHhENmPcRdFiHRjhAuHcs05RI=" crossorigin="anonymous"></script>';
    }
    $sResponsiveJSTmp .= "\n" . '<script>var _0x51e5e8=_0x1f2f;function _0x4807(){var _0x5e50df=[\'removeAttr\',\'data-th\',\'isIE\',\'innerWidth\',\'matches\',\'find\',\'isChrome\',\'addons\',\'html\',\'safari\',\'-moz-transform-origin\',\'isFirefox\',\'show\',\'containerBreakpoint\',\'opera\',\'-ms-transform-origin\',\'contentWrap\',\'-moz-transform\',\'length\',\'index\',\'ready\',\'HTMLElement\',\'defaults\',\'tr:first\x20td\',\'parent\',\'tbody\x20tr\x20th\',\'header\',\'css\',\'bind\',\'.bt-content\',\'isEdgeChromium\',\'8gixgIh\',\'matchMedia\',\'1934980OOQRrl\',\'wrapInner\',\'userAgent\',\'forceResponsive\',\'resize\',\'#desktop_container\',\'top\x20left\',\'width\',\'<span\x20class=\x22bt-content\x22\x20/>\',\'noResize\',\'outerWidth\',\'hide\',\'position\',\'breakpoint\',\'2PbxHrg\',\'pushNotification\',\'test\',\'hidden\',\'height\',\'isBlink\',\'push\',\'documentMode\',\'5Dpblez\',\'removeClass\',\'auto\',\'div[class*=\x22center\x22]\',\'relative\',\'data\',\'left\',\'top\',\'bt\x20bt--no-header\',\'addListener\',\'body\',\'tableWrap\',\'<div\x20class=\x22bt-wrapper\x22></div>\',\'indexOf\',\'thead\x20tr\x20th\',\'isOpera\',\'padding-right\',\'StyleMedia\',\'children\',\'CSS\',\'display\',\'inline-block\',\'45217430xxEVsf\',\'Chrome\',\'overflow-x\',\'3599934aXXrOq\',\'7121996ZKZDxT\',\'isEdge\',\'div[class*=\x22normalize\x22]\',\'padding-top\',\'Edg\',\'addClass\',\'tr:first\x20th\',\'bt-content\',\'bt-hide\',\'toString\',\'.bt-wrapper\',\'&nbsp;\',\'#responsive_container\',\'start\',\'wrap\',\'span\',\'basictable\',\'5px\',\'520527KxbWHv\',\'unwrap\',\'closest\',\'div\',\'showEmptyCells\',\'[object\x20SafariRemoteNotification]\',\'zoom\',\'active\',\'each\',\'2215746DsPTUM\',\'tbody\x20tr\',\'scale(\',\'attr\',\'undefined\',\'5585868MTBYvS\',\'-ms-transform\'];_0x4807=function(){return _0x5e50df;};return _0x4807();}(function(_0x5c1a01,_0x5145a3){var _0x4588b5=_0x1f2f,_0x4c178f=_0x5c1a01();while(!![]){try{var _0x2936dc=-parseInt(_0x4588b5(0x20e))/0x1+-parseInt(_0x4588b5(0x24d))/0x2*(parseInt(_0x4588b5(0x217))/0x3)+parseInt(_0x4588b5(0x23f))/0x4*(-parseInt(_0x4588b5(0x1e2))/0x5)+-parseInt(_0x4588b5(0x1fb))/0x6+-parseInt(_0x4588b5(0x1fc))/0x7*(parseInt(_0x4588b5(0x23d))/0x8)+-parseInt(_0x4588b5(0x21c))/0x9+parseInt(_0x4588b5(0x1f8))/0xa;if(_0x2936dc===_0x5145a3)break;else _0x4c178f[\'push\'](_0x4c178f[\'shift\']());}catch(_0x365a3a){_0x4c178f[\'push\'](_0x4c178f[\'shift\']());}}}(_0x4807,0x84094),!function(_0x1978c6){var _0x403606=_0x1f2f;_0x1978c6[\'fn\'][_0x403606(0x20c)]=function(_0x164c06){var _0x22f895=_0x403606;const _0xf1f999=function(_0x115b11,_0x30c39c){var _0x44ac84=_0x1f2f;const _0x2ae17d=[];if(_0x30c39c[_0x44ac84(0x1ed)]&&_0x115b11[_0x44ac84(0x20a)](_0x44ac84(0x1ee)),_0x30c39c[\'header\']){let _0x3c2b7f=\'\';_0x3c2b7f=_0x115b11[\'find\'](_0x44ac84(0x1f0))[_0x44ac84(0x230)]?\'thead\x20th\':_0x115b11[\'find\'](_0x44ac84(0x237))[_0x44ac84(0x230)]?_0x44ac84(0x237):_0x115b11[_0x44ac84(0x223)](\'th\')[\'length\']?_0x44ac84(0x202):_0x44ac84(0x235),_0x1978c6[\'each\'](_0x115b11[\'find\'](_0x3c2b7f),function(){var _0x4695b7=_0x44ac84;const _0x5672b8=_0x1978c6(this),_0x4d7d6c=parseInt(_0x5672b8[_0x4695b7(0x21a)](\'colspan\'),0xa)||0x1,_0x4eedeb=_0x5672b8[_0x4695b7(0x210)](\'tr\')[_0x4695b7(0x231)]();_0x2ae17d[_0x4eedeb]||(_0x2ae17d[_0x4eedeb]=[]);for(let _0x194e55=0x0;_0x194e55<_0x4d7d6c;_0x194e55++)_0x2ae17d[_0x4eedeb][_0x4695b7(0x1e0)](_0x5672b8);});}_0x1978c6[_0x44ac84(0x216)](_0x115b11[\'find\'](_0x44ac84(0x218)),function(){_0x5cff38(_0x1978c6(this),_0x2ae17d,_0x30c39c);}),_0x1978c6[_0x44ac84(0x216)](_0x115b11[\'find\'](\'tfoot\x20tr\'),function(){_0x5cff38(_0x1978c6(this),_0x2ae17d,_0x30c39c);});},_0x5cff38=function(_0x286c6a,_0x78f994,_0x323531){var _0x394a16=_0x1f2f;_0x286c6a[_0x394a16(0x1f4)]()[\'each\'](function(){var _0x477c3d=_0x394a16;const _0x267acb=_0x1978c6(this);if(\'\'!==_0x267acb[_0x477c3d(0x226)]()&&_0x477c3d(0x207)!==_0x267acb[_0x477c3d(0x226)]()||_0x323531[_0x477c3d(0x212)]){const _0x3dbe24=_0x267acb[_0x477c3d(0x231)]();let _0x1aeadb=\'\';for(let _0xb97ac2=0x0;_0xb97ac2<_0x78f994[_0x477c3d(0x230)];_0xb97ac2++){0x0!==_0xb97ac2&&(_0x1aeadb+=\':\x20\'),_0x1aeadb+=_0x78f994[_0xb97ac2][_0x3dbe24][\'text\']();}_0x267acb[\'attr\'](\'data-th\',_0x1aeadb),_0x323531[_0x477c3d(0x22e)]&&!_0x267acb[\'children\']()[\'hasClass\'](_0x477c3d(0x203))&&_0x267acb[_0x477c3d(0x240)](_0x477c3d(0x247));}else _0x267acb[\'addClass\'](_0x477c3d(0x204));});},_0x14e16a=function(_0x1c3b4b,_0x34b689){var _0x98d037=_0x1f2f;_0x34b689[_0x98d037(0x242)]?null!==_0x34b689[_0x98d037(0x24c)]&&_0x1978c6(window)[\'width\']()<=_0x34b689[_0x98d037(0x24c)]||null!==_0x34b689[_0x98d037(0x22b)]&&_0x1c3b4b[_0x98d037(0x236)]()[_0x98d037(0x246)]()<=_0x34b689[_0x98d037(0x22b)]?_0x44685f(_0x1c3b4b,_0x34b689):_0x52611b(_0x1c3b4b,_0x34b689):_0x1c3b4b[_0x98d037(0x1e3)](\'bt\')[_0x98d037(0x249)]()>_0x1c3b4b[_0x98d037(0x236)]()[_0x98d037(0x246)]()?_0x44685f(_0x1c3b4b,_0x34b689):_0x52611b(_0x1c3b4b,_0x34b689);},_0x44685f=function(_0x332034,_0x1a6bd7){var _0x46f27b=_0x1f2f;_0x332034[_0x46f27b(0x201)](\'bt\'),_0x1a6bd7[_0x46f27b(0x238)]||_0x332034[_0x46f27b(0x201)](\'bt--no-header\'),_0x1a6bd7[_0x46f27b(0x1ed)]&&_0x332034[\'parent\'](_0x46f27b(0x206))[_0x46f27b(0x201)](_0x46f27b(0x215));},_0x52611b=function(_0x2a8cd2,_0x1f4005){var _0x1bcabe=_0x1f2f;_0x2a8cd2[\'removeClass\'](_0x1bcabe(0x1ea)),_0x1f4005[_0x1bcabe(0x1ed)]&&_0x2a8cd2[_0x1bcabe(0x236)](_0x1bcabe(0x206))[_0x1bcabe(0x1e3)](_0x1bcabe(0x215));},_0x52ea58=function(_0x288017,_0x5d2b48){var _0x960765=_0x1f2f;_0x288017[\'removeClass\'](_0x960765(0x1ea)),_0x288017[_0x960765(0x223)](\'td\')[_0x960765(0x21e)](_0x960765(0x21f)),_0x5d2b48[_0x960765(0x1ed)]&&_0x288017[_0x960765(0x20f)](),_0x5d2b48[\'contentWrap\']&&function(_0x34b347){var _0x4ac82d=_0x960765;_0x1978c6[_0x4ac82d(0x216)](_0x34b347[\'find\'](\'td\'),function(){var _0x46d8bf=_0x4ac82d;const _0xe817ca=_0x1978c6(this),_0x3ca83d=_0xe817ca[_0x46d8bf(0x1f4)](_0x46d8bf(0x23b))[\'html\']();_0xe817ca[_0x46d8bf(0x226)](_0x3ca83d);});}(_0x288017),_0x288017[\'removeData\'](_0x960765(0x20c));};this[_0x22f895(0x216)](function(){var _0x1617e5=_0x22f895;const _0x200245=_0x1978c6(this);if(0x0===_0x200245[_0x1617e5(0x230)]||_0x200245[_0x1617e5(0x1e7)](_0x1617e5(0x20c))){if(_0x200245[_0x1617e5(0x1e7)](\'basictable\')){const _0x2f7151=_0x200245[_0x1617e5(0x1e7)](\'basictable\');\'destroy\'===_0x164c06?_0x52ea58(_0x200245,_0x2f7151):\'restart\'===_0x164c06?(_0x52ea58(_0x200245,_0x2f7151),_0x200245[_0x1617e5(0x1e7)](_0x1617e5(0x20c),_0x2f7151),_0xf1f999(_0x200245,_0x2f7151),_0x14e16a(_0x200245,_0x2f7151)):_0x1617e5(0x209)===_0x164c06?_0x44685f(_0x200245,_0x2f7151):\'stop\'===_0x164c06?_0x52611b(_0x200245,_0x2f7151):_0x14e16a(_0x200245,_0x2f7151);}return!0x1;}const _0x4132fc=_0x1978c6[\'extend\']({},_0x1978c6[\'fn\'][_0x1617e5(0x20c)][_0x1617e5(0x234)],_0x164c06),_0x167bbf={\'breakpoint\':_0x4132fc[_0x1617e5(0x24c)],\'containerBreakpoint\':_0x4132fc[_0x1617e5(0x22b)],\'contentWrap\':_0x4132fc[_0x1617e5(0x22e)],\'forceResponsive\':_0x4132fc[_0x1617e5(0x242)],\'noResize\':_0x4132fc[_0x1617e5(0x248)],\'tableWrap\':_0x4132fc[_0x1617e5(0x1ed)],\'showEmptyCells\':_0x4132fc[\'showEmptyCells\'],\'header\':_0x4132fc[_0x1617e5(0x238)]};null===_0x167bbf[\'breakpoint\']&&null===_0x167bbf[_0x1617e5(0x22b)]&&(_0x167bbf[\'breakpoint\']=0x238),_0x200245[\'data\'](_0x1617e5(0x20c),_0x167bbf),_0xf1f999(_0x200245,_0x200245[_0x1617e5(0x1e7)](_0x1617e5(0x20c))),_0x167bbf[\'noResize\']||(_0x14e16a(_0x200245,_0x200245[_0x1617e5(0x1e7)](\'basictable\')),_0x1978c6(window)[_0x1617e5(0x23a)](\'resize.basictable\',function(){!function(_0x3528d2){var _0x9261ad=_0x1f2f;_0x3528d2[_0x9261ad(0x1e7)](_0x9261ad(0x20c))&&_0x14e16a(_0x3528d2,_0x3528d2[_0x9261ad(0x1e7)](_0x9261ad(0x20c)));}(_0x200245);}));});},_0x1978c6[\'fn\'][_0x403606(0x20c)][_0x403606(0x234)]={\'breakpoint\':null,\'containerBreakpoint\':null,\'contentWrap\':!0x0,\'forceResponsive\':!0x0,\'noResize\':!0x1,\'tableWrap\':!0x1,\'showEmptyCells\':!0x1,\'header\':!0x0};}(jQuery));var calculateDocWidth=!![],actualDocumentWidth=0x0,RESPONSIVE_MAXWIDTH=0x400,RESPONSIVE_ZOOM=0x5a;function getBrowser(){var _0x4baee2=_0x1f2f,_0x47a8f8={\'isOpera\':!!window[\'opr\']&&!!opr[_0x4baee2(0x225)]||!!window[_0x4baee2(0x22c)]||navigator[_0x4baee2(0x241)][\'indexOf\'](\'\x20OPR/\')>=0x0,\'isFirefox\':typeof InstallTrigger!==\'undefined\',\'isSafari\':/constructor/i[_0x4baee2(0x24f)](window[_0x4baee2(0x233)])||function(_0x37874a){var _0x1d6343=_0x4baee2;return _0x37874a[_0x1d6343(0x205)]()===_0x1d6343(0x213);}(!window[_0x4baee2(0x227)]||typeof safari!==_0x4baee2(0x21b)&&safari[_0x4baee2(0x24e)]),\'isIE\':![]||!!document[_0x4baee2(0x1e1)],\'isChrome\':navigator[\'userAgent\'][_0x4baee2(0x1ef)](_0x4baee2(0x1f9))!==-0x1};return _0x47a8f8[_0x4baee2(0x1fd)]=!_0x47a8f8[\'isIE\']&&!!window[_0x4baee2(0x1f3)],_0x47a8f8[_0x4baee2(0x23c)]=_0x47a8f8[\'isChrome\']&&navigator[\'userAgent\'][\'indexOf\'](_0x4baee2(0x200))!=-0x1,_0x47a8f8[_0x4baee2(0x252)]=(_0x47a8f8[_0x4baee2(0x224)]||_0x47a8f8[_0x4baee2(0x1f1)])&&!!window[_0x4baee2(0x1f5)],_0x47a8f8;}function setZoom(){var _0x31cf13=_0x1f2f,_0x18390e=actualDocumentWidth,_0x56c98e=-0x1,_0x58d65f=window[_0x31cf13(0x221)]/_0x18390e*0x64+_0x56c98e,_0x5e4a43=getBrowser();if(_0x5e4a43[\'isFirefox\'])$(\'body\')[\'css\'](\'-moz-transform\',_0x31cf13(0x219)+_0x58d65f/0x64+\')\')[\'css\'](_0x31cf13(0x228),\'top\x20left\');else _0x5e4a43[_0x31cf13(0x220)]?$(\'body\')[_0x31cf13(0x239)](\'-ms-transform\',_0x31cf13(0x219)+_0x58d65f/0x64+\')\')[\'css\'](\'-ms-transform-origin\',_0x31cf13(0x245))[_0x31cf13(0x239)](_0x31cf13(0x1fa),_0x31cf13(0x250)):$(_0x31cf13(0x1ec))[_0x31cf13(0x239)](_0x31cf13(0x214),_0x58d65f+\'%\');}function resetZoom(){var _0x3a1c0c=_0x1f2f,_0x4878f7=getBrowser();if(_0x4878f7[_0x3a1c0c(0x229)])$(\'body\')[_0x3a1c0c(0x239)](_0x3a1c0c(0x22f),_0x3a1c0c(0x219)+RESPONSIVE_ZOOM/0x64+\')\')[_0x3a1c0c(0x239)](_0x3a1c0c(0x228),\'top\x20left\');else _0x4878f7[_0x3a1c0c(0x220)]||_0x4878f7[_0x3a1c0c(0x1fd)]?$(\'body\')[\'css\'](_0x3a1c0c(0x21d),_0x3a1c0c(0x219)+RESPONSIVE_ZOOM/0x64+\')\')[_0x3a1c0c(0x239)](_0x3a1c0c(0x22d),_0x3a1c0c(0x245))[_0x3a1c0c(0x239)](_0x3a1c0c(0x1fa),\'hidden\'):$(_0x3a1c0c(0x1ec))[_0x3a1c0c(0x239)](_0x3a1c0c(0x214),RESPONSIVE_ZOOM+\'%\');}function handleZoom(){var _0x281429=_0x1f2f;$(window)[_0x281429(0x246)]()>RESPONSIVE_MAXWIDTH?setZoom():resetZoom();}function calculateDocumentWidth(){var _0x341561=_0x1f2f;if(calculateDocWidth){calculateDocWidth=!![];var _0x24fc7f=0x0;$(_0x341561(0x211))[_0x341561(0x216)](function(_0x24fbca,_0x2ce63e){var _0x3c013a=_0x341561;/^page\d+$/[_0x3c013a(0x24f)]($(_0x2ce63e)[\'attr\'](\'id\'))&&(_0x24fc7f<parseInt($(_0x2ce63e)[\'css\'](_0x3c013a(0x246)))&&(_0x24fc7f=parseInt($(_0x2ce63e)[\'css\'](\'width\'))));}),actualDocumentWidth=_0x24fc7f+0x14;}}function setElementsToCenter(){var _0x496b58=_0x1f2f,_0x39c330=$(window)[\'width\']();$(_0x496b58(0x1e5))[_0x496b58(0x216)](function(){var _0x5f5d92=_0x496b58;$(this)[_0x5f5d92(0x239)](_0x5f5d92(0x1e8),(_0x39c330-this[\'offsetWidth\'])/0x2+\'px\');});}function _0x1f2f(_0x224ff6,_0x135de3){var _0x4807f9=_0x4807();return _0x1f2f=function(_0x1f2f0c,_0x556e00){_0x1f2f0c=_0x1f2f0c-0x1e0;var _0x470579=_0x4807f9[_0x1f2f0c];return _0x470579;},_0x1f2f(_0x224ff6,_0x135de3);}function normalizeParagraph(){var _0x345524=_0x1f2f;$(_0x345524(0x1fe))[_0x345524(0x216)](function(_0x48f985,_0xb044b){var _0x237f95=_0x345524;_0x48f985==0x0&&$(_0xb044b)[_0x237f95(0x239)](_0x237f95(0x1ff),\'10px\'),$(_0xb044b)[\'css\'](_0x237f95(0x251),\'auto\'),$(_0xb044b)[\'children\'](\'span\')[_0x237f95(0x216)](function(_0x523310,_0x16caf8){var _0x2b9560=_0x237f95;$(_0x16caf8)[_0x2b9560(0x239)](_0x2b9560(0x1e9),\'auto\'),$(_0x16caf8)[\'css\'](_0x2b9560(0x1e8),_0x2b9560(0x1e4)),$(_0x16caf8)[_0x2b9560(0x239)](_0x2b9560(0x1f6),_0x2b9560(0x1f7)),$(_0x16caf8)[_0x2b9560(0x239)](\'padding-right\',\'5px\'),$(_0x16caf8)[_0x2b9560(0x239)](\'position\',_0x2b9560(0x1e6));}),$(_0xb044b)[\'children\'](\'a\')[_0x237f95(0x216)](function(_0x43f525,_0x4e5dc0){var _0x26da58=_0x237f95;$(_0x4e5dc0)[_0x26da58(0x1f4)](_0x26da58(0x20b))[_0x26da58(0x216)](function(_0x3090e3,_0x24e7d8){var _0x19e016=_0x26da58;$(_0x24e7d8)[_0x19e016(0x239)](_0x19e016(0x1e9),_0x19e016(0x1e4)),$(_0x24e7d8)[_0x19e016(0x239)](_0x19e016(0x1e8),_0x19e016(0x1e4)),$(_0x24e7d8)[_0x19e016(0x239)](_0x19e016(0x1f6),_0x19e016(0x1f7)),$(_0x24e7d8)[_0x19e016(0x239)](_0x19e016(0x1f2),_0x19e016(0x20d)),$(_0x24e7d8)[_0x19e016(0x239)](_0x19e016(0x24b),_0x19e016(0x1e6));});}),$(_0xb044b)[\'children\'](\'hr\')[\'remove\']();});}function onDocumentReady(){calculateDocumentWidth(),setElementsToCenter(),normalizeParagraph(),handleZoom();}$(document)[\'ready\'](function(){var _0xd9f6e0=_0x1f2f;onDocumentReady(),$(window)[_0xd9f6e0(0x243)](function(){onDocumentReady();});}),$(document)[_0x51e5e8(0x232)](function(){var _0x478d5d=_0x51e5e8,_0x4b6e69=$(_0x478d5d(0x244)),_0x1edb57=$(_0x478d5d(0x208)),_0x131f3c=function(_0x25d7cc){var _0x5e7e51=_0x478d5d;_0x1edb57[_0x5e7e51(0x24a)](),_0x4b6e69[_0x5e7e51(0x22a)]();},_0x4e6841=function(_0x58e69d){var _0x103d4e=_0x478d5d;_0x4b6e69[_0x103d4e(0x24a)](),_0x1edb57[_0x103d4e(0x22a)](),$(\'.responsive_table\')[_0x103d4e(0x20c)]({\'breakpoint\':0x400});},_0x416e5b=function(_0xc95e72){var _0x86ee67=_0x478d5d;return _0xc95e72[_0x86ee67(0x222)]?_0x4e6841():_0x131f3c();},_0x4b0192=window[_0x478d5d(0x23e)](\'(max-width:\x201024px)\');_0x416e5b(_0x4b0192),_0x4b0192[_0x478d5d(0x1eb)](_0x416e5b);});</script>'
                         . "\n"
                         ;
    print $handleFileTmp ( $sResponsiveJSTmp );

    return 0;
}

#-----------------------------------------------------------------------
# Update resource table with font info
#-----------------------------------------------------------------------
sub _updateResourceTableForFont{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "_updateResourceTableForFont()" ); }

    #---- Fetch parameters
    #
    # 1. Unique resource id
    # 2. Resource reference
    # 3. Object hash (having both kernel object and pom object)
    #
    my $iUniqueResIdPar = shift;
    my $a2wResourcePar  = shift;
    my $hrefObjectPar   = shift;

    #---- Get actual objects
    my $a2wObjTmp = $hrefObjectPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefObjectPar->{ 'POMOBJ' };

    #---- Get active page context
    my $pgHandleTmp = $this->{ 'Pages' }{ $pomObjTmp->{ $a2w::core::dm::Constants::AT_PAGE_ID } };

    #---- Get active page
    my $dbPageTmp = $pgHandleTmp->{ 'POM' };

    #---- Get page resource table
    my $dbpResTableTmp = $dbPageTmp->getResourceTable();

    #---- Get resource table reference
    my $hrefResTableTmp = $this->{ 'FontTable' };

    #---- Build font class style and store it on font table
    my ($sHTMLFontnameTmp, $sHTMLFontLocationTmp, $sHTMLFontAttrTmp) = $this->_fetchHTMLFont( $a2wResourcePar->getName() ); # V116d/V118 Change
    $hrefResTableTmp->{ $iUniqueResIdPar } = {
          'STYLENAME'  => "ft" . sprintf( "%03d", $iUniqueResIdPar )
        , 'BOLD'       => $dbpResTableTmp->{ $iUniqueResIdPar }{ 'FONT' }{ 'BOLD' }
        , 'ITALIC'     => $dbpResTableTmp->{ $iUniqueResIdPar }{ 'FONT' }{ 'ITALIC' }
        , 'HEIGHT'     => int( $dbpResTableTmp->{ $iUniqueResIdPar }{ 'FONT' }{ 'HEIGHT' } * $this->{ 'FontScale' } )
        , 'FAMILY'     => $sHTMLFontnameTmp
        , 'LOCATION'   => $sHTMLFontLocationTmp # V116d Change
        , 'ATTRIBUTES' => $sHTMLFontAttrTmp     # V118 Change
    };

    if ( $bLog == $TRUE ){
        $theLogger->logMessage(   "Font Details: Name=" . $a2wResourcePar->getName()
                                . " HTMLName=" . $sHTMLFontnameTmp
                                . " UniqueID=" . $iUniqueResIdPar
                                . " Height=" . $hrefResTableTmp->{ $iUniqueResIdPar }{ 'HEIGHT' }
                                . " Bold=" . $hrefResTableTmp->{ $iUniqueResIdPar }{ 'BOLD' }
                                . " Italic=" . $hrefResTableTmp->{ $iUniqueResIdPar }{ 'ITALIC' }
                                . " Location=" . $hrefResTableTmp->{ $iUniqueResIdPar }{ 'LOCATION' } # V116d Change
                                . " Attributes=" . $hrefResTableTmp->{ $iUniqueResIdPar }{ 'ATTRIBUTES' } # V118 Change
                              );
    }

    # $V106 Begin
    my $sFormatTmp = ".ft%03d{line-height:%dpx;font-size:%dpx;font-family:%s;";
    # $V106 End
    # V118 Begin
    if ( $sHTMLFontAttrTmp ne "" ){ $sFormatTmp .= $sHTMLFontAttrTmp . ';'; }
    # V118 End

    if ( $hrefResTableTmp->{ $iUniqueResIdPar }{ 'BOLD' } == $TRUE ){ $sFormatTmp .= "font-weight:bold;"; }
    if ( $hrefResTableTmp->{ $iUniqueResIdPar }{ 'ITALIC' } == $TRUE ){ $sFormatTmp .= "font-style:italic;"; }
    $sFormatTmp .= "}";

    $hrefResTableTmp->{ $iUniqueResIdPar }{ 'CSS' } = sprintf(   $sFormatTmp
                                                               , $iUniqueResIdPar
                                                               , $hrefResTableTmp->{ $iUniqueResIdPar }{ 'HEIGHT' }
                                                               # $V106 Begin
                                                               , $hrefResTableTmp->{ $iUniqueResIdPar }{ 'HEIGHT' }
                                                               # $V106 End
                                                               , $sHTMLFontnameTmp
                                                      );
    return 0;
}

#-----------------------------------------------------------------------
# Update resource table with image info
#-----------------------------------------------------------------------
sub _updateResourceTableForImage{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "_updateResourceTableForImage()" ); }

    #---- Fetch parameters
    #
    # 1. Unique resource id
    # 2. Resource reference
    # 3. Object hash (having both kernel object and pom object)
    #
    my $iUniqueResIdPar = shift;
    my $a2wResourcePar  = shift;
    my $hrefObjectPar   = shift;

    #---- Get actual objects
    my $a2wObjTmp = $hrefObjectPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefObjectPar->{ 'POMOBJ' };

    #---- Get active page context
    my $pgHandleTmp = $this->{ 'Pages' }{ $pomObjTmp->{ $a2w::core::dm::Constants::AT_PAGE_ID } };

    #---- Get active page
    my $dbPageTmp = $pgHandleTmp->{ 'POM' };

    #---- Store the image to disk ----#
    #---- Check and create images sub directory, if not exist already
    my $sImgOutputPathTmp = $this->{ 'OutputPath' } . $this->{ 'ResPaths' }{ 'IMAGE' };
    unless ( -d $sImgOutputPathTmp ){
        #---- Create image output path
        unless ( mkdir( $sImgOutputPathTmp ) ){
            my $sMsgTmp = "Unable to create sub directory (" . $this->{ 'ResPaths' }{ 'IMAGE' } . ") on output path (" . $this->{ 'OutputPath' } . "), reason:" . $!;
            if ( $bLog == $TRUE ){
                $theLogger->logMessage( "Error! " . $sMsgTmp );
            }
            return ( -1, $sMsgTmp );
        }
    }

    #---- Scale down only
    my $hrefFormatParamsTmp = {};
    # V121 Begin
    #
    # Scaling down the container images result in blurry content
    # To avoid blurred content, do not scale down
    #
    #if (    $a2wObjTmp->getWidth() > int( $fWidthTmp )
    #     || $a2wObjTmp->getHeight() > int( $fHeightTmp )
    #   ){
    #    $hrefFormatParamsTmp = {
    #          'Width' => int( $fWidthTmp )
    #        , 'Height' => int( $fHeightTmp )
    #    };
    #}
    # V121 End
    # $V113 End

    #---- Evaluate output file name
    $pgHandleTmp->{ 'ImageCounter' }++;
    # V123b Begin
    #my $sImgFormatTmp   = "png";
    my $sImgFormatTmp   = $this->{ 'ImageFormat' };
    if ( $sImgFormatTmp eq 'jpeg' ){
        $sImgFormatTmp = 'jpg';

        my $hrefOutConfTmp = $this->getOutputConfig();
        if (    $hrefOutConfTmp != undef
             && $hrefOutConfTmp->{ 'jpegQuality' } != undef
             && $hrefOutConfTmp->{ 'jpegQuality' } >= 1 && $hrefOutConfTmp->{ 'jpegQuality' } <= 100
           ){
            $hrefFormatParamsTmp->{ 'Quality' } = $hrefOutConfTmp->{ 'jpegQuality' };
        }
    }
    # V123b End

    my $sImgFilenameTmp =   sprintf(   "%simg%03d."
                                     , $this->{ 'ResPaths' }{ 'IMAGE' }
                                     , ( $this->getDocumentId() . $dbPageTmp->getPageID() . $pgHandleTmp->{ 'ImageCounter' } )
                                   )
                          . $sImgFormatTmp;

    #---- Save image ----#
    $a2wObjTmp->saveAs(   $sImgFilenameTmp
                        , $sImgFormatTmp
                        # $V112 Begin
                        #, {   'ResizeWidth' => ( $a2wObjTmp->getWidth() / 2 )
                        #    , 'ResizeHeight' => ( $a2wObjTmp->getHeight() / 2 )
                        #}
                        # $V112 End
                        # $V113 Begin
                        , $hrefFormatParamsTmp
                        # $V113 End
                      );

    #---- Cache the image info in resource cache ----#
    my $hrefResTableTmp = $this->{ 'ResCache' };                    # get resource cache
    my $sUniqueNameTmp = $a2wObjTmp->_getAttribute( 'uniquename' ); # get unique name

    # V116b Begin
    $hrefResTableTmp->{ $sUniqueNameTmp } = {
          'SIMPLEFILENAME' => $sImgFilenameTmp
        , 'FILENAME' => $this->{ 'OutputPath' } . $sImgFilenameTmp
    };
    $hrefResTableTmp->{ $sUniqueNameTmp }{ 'FILESIZE' } = -s $hrefResTableTmp->{ $sUniqueNameTmp }{ 'FILENAME' };
    $hrefResTableTmp->{ $sUniqueNameTmp }{ 'DATAEXISTS' } = ( $hrefResTableTmp->{ $sUniqueNameTmp }{ 'FILESIZE' } > 0 ) ? $TRUE : $FALSE;
    if ( $bLog == $TRUE && $hrefResTableTmp->{ $sUniqueNameTmp }{ 'DATAEXISTS' } == $TRUE ){
        $theLogger->logMessage( "Saved image " . $a2wObjTmp->getName() . " as $sImgFormatTmp file $sImgFilenameTmp with parameters:" ); # V121 Change
        $theLogger->logHashMessage( $hrefFormatParamsTmp ); # V121 Change
    }

    #---- For AFP IM Raster/IOCA images, render solid rectangle with foreground color when image data is missing
    my $iImgFormatTmp = $a2wObjTmp->getFormat();
    my $bDrawSolidRectTmp = $FALSE;
    if (    (    $iImgFormatTmp == $a2w::ImageConstants::FORMAT_AFP_IMRASTER
              || $iImgFormatTmp == $a2w::ImageConstants::FORMAT_IOCA
            )
         && ( $hrefResTableTmp->{ $sUniqueNameTmp }{ 'DATAEXISTS' } == $FALSE )
       ){
        $bDrawSolidRectTmp = $TRUE;
    }
    #if ( $bLog == $TRUE ){ $theLogger->logMessage( "Image:>" . $a2wObjTmp->getName() . "< Format:>" . $iImgFormatTmp . "< SolidRect:>" . (($bDrawSolidRectTmp == $TRUE) ? "Yes" : "No") . "<" ); }
    # V116b End
    $hrefResTableTmp->{ $sUniqueNameTmp }{ 'SOLIDRECT' } = $bDrawSolidRectTmp;

    #---- Make base64 data from image (if image has to be inlined in HTML) ----#
    if (    $bDrawSolidRectTmp == $FALSE
         && $this->{ 'bInlineImage' } == $TRUE
       ){
        # V102 Begin
        my $fImageHandleTmp = undef;
        if ( open( $fImageHandleTmp, $hrefResTableTmp->{ $sUniqueNameTmp }{ 'FILENAME' } ) ){
            #---- Read image data
            local $/ = undef;
            binmode $fImageHandleTmp;
            my $binImageTmp = <$fImageHandleTmp>;
            close( $fImageHandleTmp );

            #---- Encode image data in base64
            my $b64ImageDataTmp = a2w::core::utils::Base64::encode( $binImageTmp );
            $binImageTmp = undef;

            if ( $this->{ 'HTMLVersion' } == $HTML_VERSION_5 ){ $b64ImageDataTmp =~ s/\n//g; } # V116e Change: Remove newlines and make as one line for HTML5 compatibility
            $hrefResTableTmp->{ $sUniqueNameTmp }{ 'B64DATA' } = $b64ImageDataTmp;

            #---- Delete image file
            unlink( $hrefResTableTmp->{ $sUniqueNameTmp }{ 'FILENAME' } );
        }
        else {
            $fImageHandleTmp = undef;
            $theLogger->logMessage( "Error! Unable to open file (" . $hrefResTableTmp->{ $sUniqueNameTmp }{ 'FILENAME' } . "), reason: " . $! );
            $theLogger->logMessage( "Warning! Image " . $hrefResTableTmp->{ $sUniqueNameTmp }{ 'FILENAME' } . " could not be inlined on HTML" );
            return (-2, "Unable to open file (" . $hrefResTableTmp->{ $sUniqueNameTmp }{ 'FILENAME' } . "), reason: " . $!);
        }
        # V102 End
    }

    return 0;
}

#-----------------------------------------------------------------------
# Update resource table with container info
#-----------------------------------------------------------------------
sub _updateResourceTableForContainer{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "_updateResourceTableForImage()" ); }

    #---- Fetch parameters
    #
    # 1. Unique resource id
    # 2. Resource reference
    # 3. Object hash (having both kernel object and pom object)
    #
    my $iUniqueResIdPar = shift;
    my $a2wResourcePar  = shift;
    my $hrefObjectPar   = shift;

    #---- Get actual objects
    my $a2wObjTmp = $hrefObjectPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefObjectPar->{ 'POMOBJ' };

    #---- Get active page context
    my $pgHandleTmp = $this->{ 'Pages' }{ $pomObjTmp->{ $a2w::core::dm::Constants::AT_PAGE_ID } };

    #---- Get active page
    my $dbPageTmp = $pgHandleTmp->{ 'POM' };

    #---- Get page resource table
    my $dbpResTableTmp = $dbPageTmp->getResourceTable();

    #---- Get resource table reference
    my $hrefResTableTmp = $this->{ 'FontTable' };

    #---- Save image ----#
    #---- Check and create images sub directory, if not exist already
    my $sImgOutputPathTmp = $this->{ 'OutputPath' } . $this->{ 'ResPaths' }{ 'IMAGE' };
    unless ( -d $sImgOutputPathTmp ){
        #---- Create image output path
        unless ( mkdir( $sImgOutputPathTmp ) ){
            my $sMsgTmp = "Unable to create sub directory (" . $this->{ 'ResPaths' }{ 'IMAGE' } . ") on output path (" . $this->{ 'OutputPath' } . "), reason:" . $!;
            if ( $bLog == $TRUE ){
                $theLogger->logMessage( "Error! " . $sMsgTmp );
            }
            return ( -1, $sMsgTmp );
        }
    }

    #---- Scale down only
    my $hrefFormatParamsTmp = {};
    # V121 Begin
    #
    # Scaling down the container images result in blurry content
    # To avoid blurred content, do not scale down
    #
    #if (    $a2wObjTmp->getWidth() > int( $fWidthTmp )
    #     || $a2wObjTmp->getHeight() > int( $fHeightTmp )
    #   ){
    #    $hrefFormatParamsTmp = {
    #          'Width' => int( $fWidthTmp )
    #        , 'Height' => int( $fHeightTmp )
    #    };
    #}
    # V121 End

    #---- Evaluate output file name
    $pgHandleTmp->{ 'ImageCounter' }++;
    my $sImgFormatTmp   = "png";
    my $sImgFilenameTmp =   sprintf(   "%simg%03d."
                                     , $this->{ 'ResPaths' }{ 'IMAGE' }
                                     , ( $this->getDocumentId() . $dbPageTmp->getPageID() . $pgHandleTmp->{ 'ImageCounter' } )
                                   )
                          . $sImgFormatTmp;
    $a2wObjTmp->saveAs(   $sImgFilenameTmp
                        , $sImgFormatTmp
                        # $V112 Begin
                        #, {   'ResizeWidth' => ( $a2wObjTmp->getWidth() / 2 )
                        #    , 'ResizeHeight' => ( $a2wObjTmp->getHeight() / 2 )
                        #}
                        # $V112 End
                        # $V113 Begin
                        , $hrefFormatParamsTmp
                        # $V113 End
                      );

    #---- Cache the image info in resource cache ----#
    my $hrefResTableTmp = $this->{ 'ResCache' };                    # get resource cache
    my $sUniqueNameTmp = $a2wObjTmp->_getAttribute( 'uniquename' ); # get unique name
    $sUniqueNameTmp =~ s/[\.\#]/-/;

    $hrefResTableTmp->{ $sUniqueNameTmp } = {
          'SIMPLEFILENAME' => $sImgFilenameTmp
        , 'FILENAME' => $this->{ 'OutputPath' } . $sImgFilenameTmp
    };
    $hrefResTableTmp->{ $sUniqueNameTmp }{ 'FILESIZE' } = -s $hrefResTableTmp->{ $sUniqueNameTmp }{ 'FILENAME' };
    $hrefResTableTmp->{ $sUniqueNameTmp }{ 'DATAEXISTS' } = ( $hrefResTableTmp->{ $sUniqueNameTmp }{ 'FILESIZE' } > 0 ) ? $TRUE : $FALSE;
    if ( $bLog == $TRUE ){
        $theLogger->logMessage( "Saved container " . $a2wObjTmp->getName() . "  as $sImgFormatTmp file $sImgFilenameTmp with parameters:" ); # V121 Change
        $theLogger->logHashMessage( $hrefFormatParamsTmp ); # V121 Change
    }

    #---- Make base64 data from image (if image has to be inlined in HTML) ----#
    if ( $this->{ 'bInlineImage' } == $TRUE ){
        my $fImageHandleTmp = undef;
        if ( open( $fImageHandleTmp, $hrefResTableTmp->{ $sUniqueNameTmp }{ 'FILENAME' } ) ){
            #---- Read image data
            local $/ = undef;
            binmode $fImageHandleTmp;
            my $binImageTmp = <$fImageHandleTmp>;
            close( $fImageHandleTmp );

            #---- Encode image data in base64
            my $b64ImageDataTmp = a2w::core::utils::Base64::encode( $binImageTmp );
            $binImageTmp = undef;

            if ( $this->{ 'HTMLVersion' } == $HTML_VERSION_5 ){ $b64ImageDataTmp =~ s/\n//g; } # V116e Change: Remove newlines and make as one line for HTML5 compatibility
            $hrefResTableTmp->{ $sUniqueNameTmp }{ 'B64DATA' } = $b64ImageDataTmp;

            #---- Delete image file
            unlink( $hrefResTableTmp->{ $sUniqueNameTmp }{ 'FILENAME' } );
        }
        else {
            $fImageHandleTmp = undef;
            $theLogger->logMessage( "Error! Unable to open file (" . $hrefResTableTmp->{ $sUniqueNameTmp }{ 'FILENAME' } . "), reason: " . $! );
            $theLogger->logMessage( "Warning! Image " . $hrefResTableTmp->{ $sUniqueNameTmp }{ 'FILENAME' } . " could not be inlined on HTML" );
            return (-2, "Unable to open file (" . $hrefResTableTmp->{ $sUniqueNameTmp }{ 'FILENAME' } . "), reason: " . $!);
        }
    }

    return 0;
}
# $V124d End

#-----------------------------------------------------------------------
# Don't remove the following lines !!!
#-----------------------------------------------------------------------
1;
__END__
