#-------------------------------------------------------------------------------
#  a2w::core::visitor::Visitor.pm:
#  PERL module to write the objects
#
#  Author  : AFP2web Team, Maas Holding GmbH
#
#  V100   2014-05-08    Initial Release
#
#  V101   2015-07-20    Extended to inline CSS on HTML output
#
#  V102   2015-10-01    Extended with attribute to configure following page's top margin
#
#  V103   2015-11-04    - Extended to evaluate page start, end Y positions and content height
#                       - Extended with "getTableHeight" function
#
#  V104   2018-01-19    Extended with "Illustration" block functions
#                       a. beginIllustration
#                       b. writeIllustrationLine
#                       c. endIllustration
#                       d. writeContainer
#
#  V105   2018-01-29    Extended with flag to skip output
#
#  V106   2018-08-10    Extended to write tagging info at document level instead of page level
#
#  V107   2018-09-03    a. Extended with RAW block type
#                       b. Extended with spool filename attribute
#                       c. Extended to control opening file in binary or text mode (default is text mode)
#
#  V108   2018-10-03    a. Extended with COMPOSITE block type
#                       b. Extended with table caption support
#
#  V109   2018-11-29    a. AFP-771: Extended with "List" block write functions
#                          - beginList
#                          - beginListItem
#                          - writeListLabel
#                          - writeListBody
#                          - endListItem
#                          - endList
#
#  V110   2018-12-10    AFP-772: Extended to tag line objects under one <P> tag based on "TagLineAsParagraph" flag
#                       Added following functions
#                       - beginParagraphLine
#                       - endParagraphLine
#
#  V111   2020-03-05    a. AFP-929: Extended to use visitor specific font mapping configuration
#                       b. AFP-929: Extended to get/set output language
#
#  V112   2020-07-01    AFP-951: Extended with write annotation object
#
#  V113   2020-10-07    a. AFP-979: Moved 'UnitBase' mutator/accessor APIs to base visitor
#         2021-03-18    b. AFP-1028: Extended with output configuration to control output generation
#
#  V114   2021-06-04    a. AFP-1046: Extended to sort page blocks (as needed by output visitor)
#                       b. AFP-1046: Extended to generate responsive html content
#
#  V115   2021-07-27    OXS-12591: Extended to specify properties for SF generated outputs
#
#  V116   2021-09-21    AFP-1068: Extended with FindAndPrepend and FindAndAppend content processing objects
#
#-------------------------------------------------------------------------------
package a2w::core::visitor::Visitor;

#-----------------------------------------------------------------------
# Include required modules
#-----------------------------------------------------------------------
use a2w::TypeConstants;
use a2w::core::log::Logger;
use a2w::core::dm::Constants;
use a2w::core::dm::Block;

#-----------------------------------------------------------------------
# Constructor
#-----------------------------------------------------------------------
sub new{
    my $proto = shift;
    my $class = ref( $proto ) || $proto;

    #---- Define boolean values
    $TRUE  = $a2w::TypeConstants::TRUE;     # TRUE  boolean value
    $FALSE = $a2w::TypeConstants::FALSE;    # FALSE boolean value

    my $this = {
        # Flag indicating whether file is opened or not
          'bFileOpen'    => $FALSE
        # Output filename
        , 'Filename'     => ""
        # File handle
        , 'fHandle'      => undef
        # Output file path
        , 'OutputPath'   => ""
        # Output simple filename
        , 'SimpleName'   => ""
        # Output resolution
        , 'OutputRes'    => 0
        # Default foreground color
        , 'DefForeColor' => 0
        # Default background color
        , 'DefBackColor' => 0xFFFFFF
        # Document id
        , 'DocumentId'   => 0
        # Flag to indicate to create each page as one output
        , 'PageOutput'   => $FALSE
        # Output page count
        , 'PageCount'    => 0
        # Hash of pages
        , 'Pages'        => undef
        # Active page context
        , 'ActivePage'   => undef
        # Initialize flag
        , 'bInitialized' => $FALSE
        # Flag indicating table is being written currently
        # Value of >0 mean in table
        # where
        # Bit 1 mean in table
        # Bit 2 mean in table header
        # Bit 3 mean in table body
        # Bit 4 mean in table footer
        , 'InTable'      => 0
        # Skip mode (TRUE means skip object presentation, FALSE means do not skip objects from presentation)
        , 'SkipMode'     => $FALSE
        # $V102 Begin
        # Constant for following page's top margin (in millimetres)
        #
        , 'FollowingPageTopMargin' => 5
        # $V102 End
        # $V103 Begin
        # Table height
        #
        , 'TableHeight' => 0
        # Table row count
        #
        , 'TableRowCount' => 0
        # $V103 End
        # $V105 Begin
        # Flag indicating whether output writing should be skipped or not
        # Default is to write output
        , 'SkipOutput' => $FALSE
        # $V105 End
        # $V107 Begin
        , 'SpoolFilename' => undef   # Spool filename
        , 'TextOpenMode' => $TRUE    # Text opening mode used to open output files (TRUE -> text mode, FALSE -> binary mode)
        # $V107 End
        # $V109 Begin
        # Flag indicating currently writting list context
        # Value of >0 mean in list
        # where
        # Bit 1 mean in list
        # Bit 2 mean in list item
        # Bit 3 mean in item label
        # Bit 4 mean in item body
        # Bit 5 mean in list caption
        , 'InList' => 0
        # $V109 End
        , 'FontMapTable' => undef # V111a Change
        , 'OutputLanguage' => undef # V111b Change
        , 'OutputConfig' => undef # V113b Change
        # Units base (possible values: pixel, mm)
        , 'UnitBase' => "" # V113a Change
        , 'Properties' => undef # V115 Change
    };

    bless( $this, $class );

    #---- Get logger
    our $theLogger = a2w::core::log::Logger->getSingleton();
    our $bLog = $theLogger->isRegistered( __PACKAGE__ );

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "new()" );
    #}
    return $this;
}

#-----------------------------------------------------------------------
# Destructor
#-----------------------------------------------------------------------
sub DESTROY{
    my $this = shift;

    #---- Finalize
    if ( 'bInitialized' == $TRUE ){
        $this->finalize();
    }
}

#-----------------------------------------------------------------------
# Mutator
#-----------------------------------------------------------------------
sub setOutputPath{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "setOutputPath()" );
    }

    #---- Set output path
    $this->{ 'OutputPath' } = shift;
}

sub setSimpleName{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "setSimpleName()" );
    }

    #---- Set simple name
    $this->{ 'SimpleName' } = shift;
}

sub setFilename{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "setFilename()" );
    }

    #---- Set filename
    $this->{ 'Filename' } = shift;
}

sub setOutputRes{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "setOutputRes()" );
    }

    $this->{ 'OutputRes' } = shift;
}

sub setDefForeColor{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "setDefForeColor()" );
    }

    $this->{ 'DefForeColor' } = shift;
}

sub setDefBackColor{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "setDefBackColor()" );
    }

    $this->{ 'DefBackColor' } = shift;
}

sub setDocumentId{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "setDocumentId()" );
    }

    $this->{ 'DocumentId' } = shift;
}

sub setPageOutput{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "setPageOutput()" );
    }

    $this->{ 'PageOutput' } = shift;
}

# V107 Begin
sub setSpoolFilename{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setSpoolFilename" );
    #}

    $this->{ 'SpoolFilename' } = shift;
}
# V107 End

# V111a Begin
sub setFontMapTable{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setFontMapTable" );
    #}

    $this->{ 'FontMapTable' } = shift;
}
# V111a End

# V111b Begin
sub setOutputLanguage{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setOutputLanguage" );
    #}

    $this->{ 'OutputLanguage' } = shift;
}
# V111b End

# V113b Begin
sub setOutputConfig{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setOutputConfig" );
    #}

    $this->{ 'OutputConfig' } = shift;
}

sub setFollowingPageTopMargin{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setFollowingPageTopMargin" );
    #}

    $this->{ 'FollowingPageTopMargin' } = shift;
}
# V113b End

# V113a Begin
sub setUnitBase{
    my $this = shift;

    $this->{ 'UnitBase' } = shift;
    if ( lc( $this->{ 'UnitBase' } ) eq "pixel" ){
        $this->{ 'UnitBase' } = "px";
    }
    elsif ( lc( $this->{ 'UnitBase' } ) eq "mm" ){
        $this->{ 'UnitBase' } = "mm";
    }
    else {
        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "Warning! Invalid unit base (" . $this->{ 'UnitBase' } . ") using default px" );
        }
        $this->{ 'UnitBase' } = "px";
    }
}
# V113a End

# V115 Begin
sub setProperties{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "setProperties()" );
    }

    $this->{ 'Properties' } = shift;
}
# V115 End

#-----------------------------------------------------------------------
# Accessor
#-----------------------------------------------------------------------
sub getOutputPath{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "getOutputPath()" );
    }

    #---- Get output path
    return $this->{ 'OutputPath' };
}

sub getSimpleName{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "getSimpleName()" );
    }

    #---- Get simple name
    return $this->{ 'SimpleName' };
}

sub getFilename{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "getFilename()" );
    }

    #---- Get filename
    return $this->{ 'Filename' };
}

sub getOutputRes{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "getOutputRes( " . $this->{ 'OutputRes' } . " )" );
    }

    return $this->{ 'OutputRes' };
}

sub getDefForeColor{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "getDefForeColor()" );
    }

    return $this->{ 'DefForeColor' };
}

sub getDocumentId{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "getDocumentId()" );
    }

    return $this->{ 'DocumentId' };
}

sub getDefBackColor{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "getDefBackColor()" );
    }

    return $this->{ 'DefBackColor' };
}

sub getPageOutput{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "getPageOutput()" );
    }

    return $this->{ 'PageOutput' };
}

# V107 Begin
sub getSpoolFilename{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getSpoolFilename" );
    #}

    return $this->{ 'SpoolFilename' };
}
# V107 End

# V111a Begin
sub getFontMapTable{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getFontMapTable" );
    #}

    return $this->{ 'FontMapTable' };
}
# V111a End

# V111b Begin
sub getOutputLanguage{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getOutputLanguage" );
    #}

    return $this->{ 'OutputLanguage' };
}
# V111b End

# V113b Begin
sub getOutputConfig{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getOutputConfig" );
    #}

    return $this->{ 'OutputConfig' };
}

sub getFollowingPageTopMargin{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getFollowingPageTopMargin" );
    #}

    return $this->{ 'FollowingPageTopMargin' };
}
# V113b End

# V113a Begin
sub getUnitBase{
    my $this = shift;

    return $this->{ 'UnitBase' };
}
# V113a End

# V115 Begin
sub getProperties{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "getProperties()" );
    }

    return $this->{ 'Properties' };
}
# V115 End

#-----------------------------------------------------------------------
# Workers
#-----------------------------------------------------------------------
#-----------------------------------------------------------------------
# Initialize
#-----------------------------------------------------------------------
sub initialize{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "initialize()" );
    }

    #---- Get parameter
    #
    # 1. Output path
    # 2. Simple filename
    #
    my $sOutputPathPar = shift;
    my $sSimpleNamePar = shift;
    $this->setOutputPath( $sOutputPathPar );
    $this->setSimpleName( $sSimpleNamePar );

    #---- Open output file
    if ( $this->{ 'PageOutput' } == $FALSE ){
        # $V101 Begin
        if ( $this->openOutputStream( $sOutputPathPar, $sSimpleNamePar ) < 0 ){ return -1 };
        # $V101 End
    }

    $this->{ 'bInitialized' } = $TRUE;

    return 0;
}

#-----------------------------------------------------------------------
# Finalize
#-----------------------------------------------------------------------
sub finalize{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "finalize()" );
    }

    #---- Fetch parameters
    #
    # 1. Additional content for body end (optional)
    # 2. Additional content for footer (optional)
    #
    my $sBodyAddContentPar   = "";
    my $sAddFooterContentPar = "";
    if ( @_ > 0 ){
        $sBodyAddContentPar = shift;
    }
    if ( @_ > 0 ){
        $sAddFooterContentPar = shift;
    }

    $this->{ 'bInitialized' } = $FALSE;

    #---- Close output file
    if ( $this->{ 'PageOutput' } == $FALSE ){
        # $V101 Begin
        if ( $this->closeOutputStream() < 0 ){ return -1 };
        # $V101 End
    }

    return 0;
}

# V113b Begin
#-----------------------------------------------------------------------
# Update output configuration
#
# Returns
# >= 0 in case of success
#  < 0 in case of failure
#
#-----------------------------------------------------------------------
sub updateConfig{
    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "updateConfig()" ); }

    #---- Get output config
    my $hrefOutConfTmp = $this->getOutputConfig();
    if ( $hrefOutConfTmp == undef ){ return 0; }

    #---- Check and set configured page gap
    if ( $hrefOutConfTmp->{ 'continuousPageGap' } != undef ){
        $this->setFollowingPageTopMargin( $hrefOutConfTmp->{ 'continuousPageGap' } );
    }

    return 0;
}
# V113b End

# $V101 Begin
#-----------------------------------------------------------------------
# Open output stream
#
# Returns
# >= 0 in case of success
#  < 0 in case of failure
#
#-----------------------------------------------------------------------
sub openOutputStream{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "openOutputStream()" );
    }

    #---- Fetch parameters
    #
    # 1. Output path (mandatory)
    # 2. Simple file name (mandatory)
    # 3. Page id (optional)
    # 4. Page handle (optional)
    #
    my $sOutputPathPar = shift;
    my $sSimpleNamePar = shift;
    my $iPageIdPar = 0;
    my $pgHandlePar = undef;
    if ( @_ > 0 ){
        $iPageIdPar = shift;
    }
    if ( @_ > 0 ){
        $pgHandlePar = shift;
    }

    # $V105 Begin
    if ( $this->{ 'SkipOutput' } == $TRUE ){
        $pgHandlePar->{ 'fHandle' }   = undef;
        $pgHandlePar->{ 'bFileOpen' } = $FALSE;
        $pgHandlePar->{ 'Filename' }  = '';

        return 0;
    }
    # $V105 End

    #---- Check whether file opened already or not
    my $bPageModeTmp = $FALSE;
    #---- Document output mode
    if (    $iPageIdPar == 0
         && $pgHandlePar == undef
         && $this->{ 'bFileOpen' } == $TRUE
       ){
        return 0;
    }

    #---- Page output mode
    if (    $iPageIdPar > 0
         && $pgHandlePar != undef
       ){
        $bPageModeTmp = $TRUE;
        if ( $pgHandlePar->{ 'bFileOpen' } == $TRUE ){ return 0 };
    }

    #---- Evaluate output file name
    my $sFilenameTmp =   $sOutputPathPar
                       . $sSimpleNamePar
                       . $this->getFileExtension();
    if ( $bPageModeTmp == $TRUE ){
        $sFilenameTmp =   $sOutputPathPar
                        . $sSimpleNamePar
                        . '_' . sprintf( "%04d", $this->{ 'PageCount' } )
                        . $this->getFileExtension();
    }
    if ( $bLog == $TRUE ){
        $theLogger->logMessage( "Filename:>" . $sFilenameTmp . "<" );
    }

    #---- Open
    my $fHandleTmp = $this->open( $sFilenameTmp );
    if ( $fHandleTmp == undef ){
        return -1;
    }
    if ( $this->{ 'TextOpenMode' } == $FALSE ){ binmode( $fHandleTmp ); } # V107 Change

    if ( $bPageModeTmp == $FALSE ){
        $this->{ 'fHandle' }   = $fHandleTmp;
        $this->{ 'bFileOpen' } = $TRUE;
        $this->{ 'Filename' }  = $sFilenameTmp;
    }
    else {
        $pgHandlePar->{ 'fHandle' }   = $fHandleTmp;
        $pgHandlePar->{ 'bFileOpen' } = $TRUE;
        $pgHandlePar->{ 'Filename' }  = $sFilenameTmp;
    }

    return 0;
}

#-----------------------------------------------------------------------
# Close output stream
#
# Returns
# >= 0 in case of success
#  < 0 in case of failure
#
#-----------------------------------------------------------------------
sub closeOutputStream{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "closeOutputStream()" );
    }

    #---- Fetch parameters
    #
    # 1. Page handle (optional)
    #
    my $pgHandlePar = undef;
    if ( @_ > 0 ){
        $pgHandlePar = shift;
    }

    #---- Check whether file opened already or not
    my $bClosedTmp   = $FALSE;
    if ( $pgHandlePar != undef ){
        if ( $pgHandlePar->{ 'bFileOpen' } == $FALSE ){ return 0 };

        #---- Close
        $bClosedTmp = $this->close( $pgHandlePar->{ 'fHandle' } );
        $pgHandlePar->{ 'bFileOpen' } = $FALSE;
    }
    else {
        if ( $this->{ 'bFileOpen' } == $FALSE ){ return 0 };

        #---- Close
        $bClosedTmp = $this->close( $this->{ 'fHandle' } );
        $this->{ 'bFileOpen' } = $FALSE;
    }

    if ( $bClosedTmp == $FALSE ){
        return -1;
    }

    return 0;
}
# $V101 End

# V114a Begin
#-----------------------------------------------------------------------
# Sort page blocks
#-----------------------------------------------------------------------
sub sortPageBlocks{
    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "sortPageBlocks()" ); }

    #---- Get parameter
    #
    # 1. Page blocks (array reference)
    #
    my $arefPageBlocksPar = shift;

    # do nothing, derived visitors must sort the blocks as per output presentation needs

    return $arefPageBlocksPar;
}
# V114a End

#-----------------------------------------------------------------------
# Initialize page
#
# Returns
# Page handle in case of success
# undef in case of error
#
#-----------------------------------------------------------------------
sub initializePage{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "initializePage()" );
    }

    #---- Fetch parameters
    #
    # 1. Database
    # 2. Additional content for header
    #
    my $dbPagePar = shift;
    my $sAddHeaderContentPar = "";
    if ( @_ > 0 ){
        $sAddHeaderContentPar = shift;
    }

    #---- Increment page count
    $this->{ 'PageCount' } += 1;

    #---- Fetch page info
    my $iPageIdTmp = $dbPagePar->getPageID();

    #---- Create page context ----#
    $this->{ 'Pages' }{ $iPageIdTmp } = {
          'ID'            => $iPageIdTmp
        , 'POM'           => $dbPagePar
        , 'ResourceTable' => undef  # Resource table
        , 'Content'       => undef  # Content of page
        , 'ContentScale'  => 0      # Content scale factor
        , 'Width'         => 0      # Page width
        , 'Height'        => 0      # Page height
        , 'bFileOpen'     => $FALSE # Flag indicating whether file is opened or not
        , 'Filename'      => ""     # Output filename
        , 'fHandle'       => undef  # File handle
        , 'ActiveBlock'   => undef  # Active block
        , 'CurrentY'      => 0      # Current Y position
        , 'ImageCounter'  => 0      # Image counter
        , 'BlockCount'    => 0      # Blocks count
        , 'PrevBlkEndY'   => 0      # Previous blocks bottom most Y position
        # $V103 Begin
        , 'StartY'        => 0      # Page start Y position
        , 'EndY'          => 0      # Page end Y position
        , 'ContentRes'    => 0      # Page content resolution
        , 'ContentHeight' => 0      # Page content height
        # $V103 End
    };
    my $pgHandleTmp = $this->{ 'Pages' }{ $iPageIdTmp };

    #---- Open output file
    if ( $this->{ 'PageOutput' } == $TRUE ){
        # $V101 Begin
        my $sSimpleNameTmp = $this->getSimpleName();
        if ( $sSimpleNameTmp eq "" ){
            $sSimpleNameTmp = $dbPagePar->getPageFilename();
            if ( $sSimpleNameTmp =~ /(.*)\..{3,4}$/ ){
                $sSimpleNameTmp = $1;
            }
        }
        if ( $this->openOutputStream( $this->getOutputPath(), $sSimpleNameTmp, $iPageIdTmp, $pgHandleTmp ) < 0 ){ return undef };
        # $V101 End
    }

    #---- Select this page as active
    $this->activatePage( $iPageIdTmp );

    # $V103 Begin
    $pgHandleTmp->{ 'ContentRes' } = $dbPagePar->getPageRes();
    # $V103 End

    return $pgHandleTmp;
}

#-----------------------------------------------------------------------
# Finalize page
#-----------------------------------------------------------------------
sub finalizePage{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "finalizePage()" );
    }

    #---- Fetch parameters
    #
    # Additional content for footer
    #
    my $sAddFooterContentPar = "";
    if ( @_ > 0 ){
        $sAddFooterContentPar = shift;
    }

    #---- Get page handle
    my $pgHandleTmp = $this->{ 'ActivePage' };

    #---- Close output file
    if ( $this->{ 'PageOutput' } == $TRUE ){
        # $V101 Begin
        if ( $this->closeOutputStream( $pgHandleTmp ) < 0 ){ return -1 };
        # $V101 End
    }

    # $V103 Begin
    if ( $bLog == $TRUE ){
        $theLogger->logMessage( "Page " . $pgHandleTmp->{ 'ID' } . " Height:>" . ( $pgHandleTmp->{ 'EndY' } - $pgHandleTmp->{ 'StartY' } ) . "< Content height:>" . $pgHandleTmp->{ 'ContentHeight' } . "<" );
    }
    # $V103 End

    # $V101 Begin
    #---- Deactivate page
    $this->{ 'ActivePage' } = undef;
    # $V101 End

    return 0;
}

#-----------------------------------------------------------------------
# Activate page (for output presentation)
#-----------------------------------------------------------------------
sub activatePage{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "activatePage()" );
    }

    #---- Get parameter
    #
    # 1. Page Id
    #
    my $sPageIdPar = shift;
    if ( not defined( $this->{ 'Pages' }{ $sPageIdPar } ) ){
        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "  Invalid page id (" . $sPageIdPar . ") (" . $this->{ 'Pages' }{ $sPageIdPar } . ") to activate page" );
        }
        return -1;
    }

    #---- Set page as active page
    $this->{ 'ActivePage' } = $this->{ 'Pages' }{ $sPageIdPar };
    if ( $this->{ 'PageOutput' } == $TRUE ){
        $this->{ 'fHandle' }   = $this->{ 'ActivePage' }{ 'fHandle' };
        $this->{ 'bFileOpen' } = $this->{ 'ActivePage' }{ 'bFileOpen' };
        $this->{ 'Filename' }  = $this->{ 'ActivePage' }{ 'Filename' };
    }
    return 0;
}

#-----------------------------------------------------------------------
# Open
#-----------------------------------------------------------------------
sub open{
    my $this = shift;

    #---- Get parameter
    #
    # 1. Filename
    #
    my $sFilenamePar = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "open()" );
    }

    #---- Open file
    my $fileHandleTmp = undef;
    if ( open( $fileHandleTmp, ">$sFilenamePar" ) ){
        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "File (" . $sFilenamePar . ") opened successfully" );
        }
    }
    else {
        $fileHandleTmp = undef;
        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "Error! Unable to open file (" . $sFilenamePar . "), reason: " . $! );
        }
    }
    return $fileHandleTmp;
}

#-----------------------------------------------------------------------
# close
#-----------------------------------------------------------------------
sub close{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "close()" );
    }

    #---- Get parameter
    #
    # 1. File handle
    #
    my $fileHandlePar = shift;
    if ( $fileHandlePar == undef ){ return $FALSE; }

    #---- Close file
    close( $fileHandlePar );

    return $TRUE;
}

#-----------------------------------------------------------------------
# Get File Extension
#
# Returns output file extension based on visitor type. Base returns empty
# string and concrete visitors MUST return appropriate extension
#
#-----------------------------------------------------------------------
sub getFileExtension{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "getFileExtension()" );
    }

    return "";
}

#-----------------------------------------------------------------------
# Write header
#-----------------------------------------------------------------------
sub writeHeader{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeHeader()" );
    }

    #---- Parameter(s)
    #
    # 1. Title
    # 2. Additional content
    #
    my $sTitlePar      = shift;
    my $sAddContentPar = shift;

    return 0;
}

#-----------------------------------------------------------------------
# Write footer
#-----------------------------------------------------------------------
sub writeFooter{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeFooter()" );
    }

    #---- Parameter(s)
    #
    # 1. Additional content
    #
    my $sAddContentPar = shift;

    return 0;
}

#-----------------------------------------------------------------------
# write block
#-----------------------------------------------------------------------
sub writeBlock{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeBlock()" );
    }

    #---- Parameter(s)
    #
    # 1. Block
    #
    my $blkToWritePar = shift;

    #---- Write block
    return $blkToWritePar->write( $this );
}

#-----------------------------------------------------------------------
# Update resource table
#-----------------------------------------------------------------------
sub updateResourceTable{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "updateResourceTable()" );
    }

    #---- Fetch parameters
    #
    # 1. Unique resource id
    # 2. Resource reference
    # 3. Object hash (having both kernel object and pom object)
    #
    my $iUniqueResIdPar = shift;
    my $a2wResourcePar  = shift;
    my $hrefObjectPar   = shift;

    return 0;
}

#-----------------------------------------------------------------------
# isWritable
#
# Asserts whether given object is writable by visitor
#
# Returns
# TRUE is visitor can write the object
# FALSE is visitor can not write the object
#
#-----------------------------------------------------------------------
sub isWritable{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "isWritable()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Object hash (having both kernel object and pom object)
    #
    my $hrefObjPar = shift;

    #---- Assert object
    my $a2wObjTmp = $hrefObjPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefObjPar->{ 'POMOBJ' };
    my $bRetTmp   = $TRUE;

    my $iObjTypeTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJTYPE };
    unless (    $iObjTypeTmp == $a2w::core::dm::Constants::OT_IMAGE
             || $iObjTypeTmp == $a2w::core::dm::Constants::OT_LINE
             || $iObjTypeTmp == $a2w::core::dm::Constants::OT_TEXT
             || $iObjTypeTmp == $a2w::core::dm::Constants::OT_VECTOR
           ){
        $bRetTmp = $FALSE;
    }

    return $bRetTmp;
}

#-----------------------------------------------------------------------
# Write raw content
#-----------------------------------------------------------------------
sub writeRawContent{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeRawContent()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Raw content
    #
    my $sRawContentPar = shift;

    return 0;
}

#-----------------------------------------------------------------------
# Write grouped text
#-----------------------------------------------------------------------
sub writeGroupedText{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeGroupedText()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. First Object hash (having both kernel object and pom object)
    # 2. Grouped Text
    # 3. Line gap (vertical difference between previous and current line, zero for first line)
    # 5. Table info (hash { Cell => TRUE|FALSE, ColHdr => <Header name>, ColId => <Value> } reference, optional)
    #
    my $hrefFirstObjPar  = shift;
    my $sGroupedTextPar  = shift;
    my $iLineGapPar      = shift;
    my $hrefTableInfoPar = undef;
    if ( @_ > 0 ){
        $hrefTableInfoPar = shift;
    }

    #---- Get actual objects
    my $a2wObjTmp = $hrefFirstObjPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefFirstObjPar->{ 'POMOBJ' };

    return 0;
}

#-----------------------------------------------------------------------
# Write object
#-----------------------------------------------------------------------
sub writeObject{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeObject()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Object hash (having both kernel object and pom object)
    #
    my $hrefObjPar = shift;

    #---- Get actual objects
    my $a2wObjTmp = $hrefObjPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefObjPar->{ 'POMOBJ' };

    #---- Write object ----#
    my $hrefMethodMapTmp = {
          $a2w::core::dm::Constants::OT_IMAGE       => 'writeImage'
        , $a2w::core::dm::Constants::OT_CONTAINER   => 'writeContainer'  # $V104 Change
        , $a2w::core::dm::Constants::OT_LINE        => 'writeLine'
        , $a2w::core::dm::Constants::OT_TEXT        => 'writeText'
        , $a2w::core::dm::Constants::OT_VECTOR      => 'writeVector'
        , $a2w::core::dm::Constants::OT_ANNOTATION  => 'writeAnnotation' # $V112 Change
    };
    my $iObjTypeTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJTYPE };
    my $sMethodNameTmp = $hrefMethodMapTmp->{ $iObjTypeTmp };
    if ( $sMethodNameTmp eq "" ){
        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "Error! Unsupported Object Type (" . $iObjTypeTmp . ") to write" );
        }
        return -1;
    }
    $this->$sMethodNameTmp( $hrefObjPar );

    return 0;
}

#-----------------------------------------------------------------------
# Write image
#-----------------------------------------------------------------------
sub writeImage{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeImage()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Image object hash (having both kernel object and pom object)
    #
    my $hrefImageObjPar = shift;

    #---- Get actual objects
    my $a2wObjTmp = $hrefImageObjPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefImageObjPar->{ 'POMOBJ' };

    return 0;
}

# $V104 Begin
#-----------------------------------------------------------------------
# Write container
#-----------------------------------------------------------------------
sub writeContainer{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeContainer()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Container object hash (having both kernel object and pom object)
    #
    my $hrefContainerObjPar = shift;

    #---- Get actual objects
    my $a2wObjTmp = $hrefContainerObjPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefContainerObjPar->{ 'POMOBJ' };

    return 0;
}
# $V104 End

#-----------------------------------------------------------------------
# Write line
#-----------------------------------------------------------------------
sub writeLine{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeLine()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Line object hash (having both kernel object and pom object)
    #
    my $hrefLineObjPar = shift;

    #---- Get actual objects
    my $a2wObjTmp = $hrefLineObjPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefLineObjPar->{ 'POMOBJ' };

    return 0;
}

#-----------------------------------------------------------------------
# Write text
#-----------------------------------------------------------------------
sub writeText{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeText()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Text object hash (having both kernel object and pom object)
    #
    my $hrefTextObjPar = shift;

    #---- Get actual objects
    my $a2wObjTmp = $hrefTextObjPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefTextObjPar->{ 'POMOBJ' };

    return 0;
}

#-----------------------------------------------------------------------
# Write vector
#-----------------------------------------------------------------------
sub writeVector{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeVector()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Vector object hash (having both kernel object and pom object)
    #
    my $hrefVectorObjPar = shift;

    #---- Get actual objects
    my $a2wObjTmp = $hrefVectorObjPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefVectorObjPar->{ 'POMOBJ' };

    return 0;
}

# V112 Begin
#-----------------------------------------------------------------------
# Write annotation
#-----------------------------------------------------------------------
sub writeAnnotation{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeAnnotation()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Annotation object hash (having both kernel object and pom object)
    #
    my $hrefAnnotObjPar = shift;

    #---- Get actual objects
    my $a2wObjTmp = $hrefAnnotObjPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefAnnotObjPar->{ 'POMOBJ' };

    return 0;
}
# V112 End

#---- Block related writing methods ----#
#-----------------------------------------------------------------------
# Begin block
#-----------------------------------------------------------------------
sub beginBlock{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "beginBlock()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Block
    # 2. Skip flag
    #
    my $blkToBeginPar = shift;
    my $bSkipPar = shift;

    #---- Get active page context
    my $pgHandleTmp = $this->{ 'ActivePage' };

    #---- Set active block
    $pgHandleTmp->{ 'ActiveBlock' } = $blkToBeginPar;

    #---- Increment block count
    if ( $bSkipPar == $FALSE ){ $pgHandleTmp->{ 'BlockCount' }++; }

    # $V103 Begin
    #---- Get start of page
    if ( $pgHandleTmp->{ 'BlockCount' } == 1 ){
        $pgHandleTmp->{ 'StartY' } = $blkToBeginPar->getStartY();
	    if ( $bLog == $TRUE ){
            $theLogger->logMessage( "Page " . $pgHandleTmp->{ 'ID' } . " Start Y:>" . $pgHandleTmp->{ 'StartY' } . "<" );
	    }
    }
    # $V103 End

    return 0;
}

#-----------------------------------------------------------------------
# End block
#-----------------------------------------------------------------------
sub endBlock{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "endBlock()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Block
    # 2. Skip flag
    #
    my $blkToEndPar = shift;
    my $bSkipPar = shift;

    #---- Get active page context
    my $pgHandleTmp = $this->{ 'ActivePage' };

    #---- Reset active block
    $pgHandleTmp->{ 'ActiveBlock' } = undef;

    # $V103 Begin
    #---- Evaluate end of page
    $pgHandleTmp->{ 'EndY' } = $blkToEndPar->getEndY();
    # $V103 End

    return 0;
}

#---- Paragraph related writing methods ----#
#-----------------------------------------------------------------------
# Begin paragraph
#-----------------------------------------------------------------------
sub beginParagraph{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "beginParagraph()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Block
    #
    my $blkParagraphPar = shift;

    return 0;
}

# V110 Begin
#-----------------------------------------------------------------------
# Begin paragraph line
#-----------------------------------------------------------------------
sub beginParagraphLine{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "beginParagraphLine()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Flag indicating whether line should be tagged as paragraph or not
    #
    my $bTagLineAsParagraphPar = shift;

    return 0;
}
# V110 End

#-----------------------------------------------------------------------
# Write paragraph line
#-----------------------------------------------------------------------
sub writeParagraphLine{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeParagraphLine()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Array of line objects
    # 2. Line count
    # 3. Line gap (vertical difference between previous and current line, zero for first line)
    #
    my $arefObjectsPar = shift;
    my $iLineCountPar  = shift;
    my $iLineGapPar    = shift;

    #---- Fetch objects list
    my @arrObjsTmp = @{ $arefObjectsPar };
    foreach my $a2wObjTmp ( @arrObjsTmp ){
        $this->writeObject( $a2wObjTmp );
    }

    return 0;
}

# V110 Begin
#-----------------------------------------------------------------------
# End paragraph line
#-----------------------------------------------------------------------
sub endParagraphLine{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "endParagraphLine()" );
    }

    return 0;
}
# V110 End

#-----------------------------------------------------------------------
# End paragraph
#-----------------------------------------------------------------------
sub endParagraph{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "endParagraph()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Block
    #
    my $blkParagraphPar = shift;

    return 0;
}

#---- Skip related writing methods ----#
#-----------------------------------------------------------------------
# Begin skip
#-----------------------------------------------------------------------
sub beginSkip{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "beginSkip()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Block
    #
    my $blkSkipPar = shift;

    #---- Start skip mode
    $this->{ 'SkipMode' } = $TRUE;

    return 0;
}

#-----------------------------------------------------------------------
# End Skip
#-----------------------------------------------------------------------
sub endSkip{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "endSkip()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Block
    #
    my $blkSkipPar = shift;

    #---- End skip mode
    $this->{ 'SkipMode' } = $FALSE;

    return 0;
}

# V107 Begin
#---- RAW block related writing methods ----#
#-----------------------------------------------------------------------
# Begin raw
#-----------------------------------------------------------------------
sub beginRaw{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "beginRaw()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Block
    #
    my $blkRawPar = shift;

    return 0;
}

#-----------------------------------------------------------------------
# Write raw objects
#-----------------------------------------------------------------------
sub writeRawObjects{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeRawObjects()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Array of objects
    #
    my $arefObjectsPar = shift;

    #---- Fetch objects list
    my @arrObjsTmp = @{ $arefObjectsPar };
    foreach my $elemObjTmp ( @arrObjsTmp ){
        $this->writeObject( $elemObjTmp );
    }

    return 0;
}

#-----------------------------------------------------------------------
# End Raw
#-----------------------------------------------------------------------
sub endRaw{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "endRaw()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Block
    #
    my $blkRawPar = shift;

    return 0;
}
# V107 End

# V108 Begin
#---- COMPOSITE block related writing methods ----#
#-----------------------------------------------------------------------
# Begin composite
#-----------------------------------------------------------------------
sub beginComposite{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "beginComposite()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Block
    #
    my $blkCompsitePar = shift;

    return 0;
}

#-----------------------------------------------------------------------
# End composite
#-----------------------------------------------------------------------
sub endComposite{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "endComposite()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Block
    #
    my $blkCompsitePar = shift;

    return 0;
}
# V108 End

#---- Table related writing methods ----#
#-----------------------------------------------------------------------
# Begin table
#-----------------------------------------------------------------------
sub beginTable{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "beginTable()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Block
    # 2. Columns X positions array
    # 3. Starting Y position
    #
    my $blkTablePar        = shift;
    my $arefColumnsXPosPar = shift;
    my $iStartYPar         = shift;
    if ( @{ $arefColumnsXPosPar } <= 0 ){
        return -1;
    }

    #---- Set in table flag
    $this->{ 'InTable' } = 1;

    # $V103 Begin
    $this->{ 'TableHeight' } = 0;
    $this->{ 'TableRowCount' } = 0;
    # $V103 End

    return 0;
}

#-----------------------------------------------------------------------
# Begin table header
#-----------------------------------------------------------------------
sub beginTableHeader{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "beginTableHeader()" );
    }

    #---- Set in table header flag
    $this->{ 'InTable' } |= ( 1 << 1 );

    return 0;
}

#-----------------------------------------------------------------------
# End table header
#-----------------------------------------------------------------------
sub endTableHeader{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "endTableHeader()" );
    }

    #---- Reset in table header flag
    $this->{ 'InTable' } ^= ( 1 << 1 );

    return 0;
}

# V108 Begin
#-----------------------------------------------------------------------
# Write table caption
#-----------------------------------------------------------------------
sub writeTableCaption{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeTableCaption()" );
    }

    #---- Get parameter
    #
    # 1. Objects list (array reference)
    #
    my $arefCaptionObjectsPar = shift;
    my @arrCaptionObjectsTmp = @{ $arefCaptionObjectsPar };

    #---- Write table caption
    #TODO: Extend to write given list of objects as table caption

    return 0;
}
# V108 End

#-----------------------------------------------------------------------
# Begin table body
#-----------------------------------------------------------------------
sub beginTableBody{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "beginTableBody()" );
    }

    #---- Set in table body flag
    $this->{ 'InTable' } |= ( 1 << 2 );

    return 0;
}

#-----------------------------------------------------------------------
# End table body
#-----------------------------------------------------------------------
sub endTableBody{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "endTableBody()" );
    }

    #---- Reset in table body flag
    $this->{ 'InTable' } ^= ( 1 << 2 );

    return 0;
}

#-----------------------------------------------------------------------
# Begin table footer
#-----------------------------------------------------------------------
sub beginTableFooter{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "beginTableFooter()" );
    }

    #---- Set in table footer flag
    $this->{ 'InTable' } |= ( 1 << 3 );

    return 0;
}

#-----------------------------------------------------------------------
# End table footer
#-----------------------------------------------------------------------
sub endTableFooter{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "endTableFooter()" );
    }

    #---- Reset in table footer flag
    $this->{ 'InTable' } ^= ( 1 << 3 );

    return 0;
}

#-----------------------------------------------------------------------
# Begin row
#-----------------------------------------------------------------------
sub beginRow{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "beginRow()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Row number
    # 2. Page id (where row starts on) # V106 Change
    #
    my $iRowNrPar  = shift;
    my $iPageIdPar = shift; # V106 Change

    # $V103 Begin
    $this->{ 'TableRowCount' } = $iRowNrPar;
    # $V103 End

    return 0;
}

#-----------------------------------------------------------------------
# Write cell
#-----------------------------------------------------------------------
sub writeCell{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeCell()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Array of cell objects
    # 2. Column header name
    # 3. Column id (as in on block definition)
    #
    my $arefObjectsPar = shift;
    my $sColHdrNamePar = shift;
    my $sColIdPar      = shift;

    #---- Fetch objects list
    my @arrObjsTmp = @{ $arefObjectsPar };
    foreach my $a2wObjTmp ( @arrObjsTmp ){
        $this->writeObject( $a2wObjTmp );
    }

    return 0;
}

#-----------------------------------------------------------------------
# End row
#-----------------------------------------------------------------------
sub endRow{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "endRow()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Row number
    #
    my $iRowNrPar    = shift;

    return 0;
}

#-----------------------------------------------------------------------
# End table
#-----------------------------------------------------------------------
sub endTable{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "endTable()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Block
    #
    my $blkTablePar = shift;

    #---- Reset in table flag
    $this->{ 'InTable' } = 0;

    return 0;
}

#---- Find and replace (as configured in block content processor) ----#
#-----------------------------------------------------------------------
# Process given content for find and replace
#
# Returns processed text in case of success, else the original text
#
#-----------------------------------------------------------------------
sub findAndReplaceText{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "findAndReplaceText()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Output format
    # 2. Object hash (having both kernel object and pom object)
    # 3. Text
    # 4. Table info (hash { Cell => TRUE|FALSE, ColHdr => <Header name>, ColId => <Value> } reference, optional)
    #
    my $sOutFmtPar = shift; # V116 Change
    my $hrefObjPar = shift;
    my $sTextPar   = shift;
    my $hrefTableInfoPar = undef;
    if ( @_ > 0 ){
        $hrefTableInfoPar = shift;
    }

    #---- Get actual objects
    my $a2wObjTmp = $hrefObjPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefObjPar->{ 'POMOBJ' };

    if ( $bLog == $TRUE ){
        $theLogger->logMessage( "Text:>" . $sTextPar . "<" );
        if ( $hrefTableInfoPar != undef ){
            $theLogger->logMessage( "Table: Cell:>" . (($hrefTableInfoPar->{ 'Cell' } == $TRUE) ? 'Yes' : 'No') . "< Header:>" . $hrefTableInfoPar->{ 'ColHdr' } . "< Column:>" . $hrefTableInfoPar->{ 'ColId' } . "<" );
        }
    }

    #---- Get active block
    my $pgHandleTmp  = $this->{ 'ActivePage' };
    my $blkActiveTmp = $pgHandleTmp->{ 'ActiveBlock' };
    if ( $blkActiveTmp == undef ){
        return $sTextPar;
    }

    #---- Get content processor of block
    my $contProcTmp = $blkActiveTmp->getContentProcessor();
    if (    $contProcTmp == undef
         || $contProcTmp->hasFNRs() == $FALSE
       ){
        return $sTextPar;
    }

    #---- Get find and replace list
    my $arefFNRListTmp = $contProcTmp->getFindAndReplaceList();
    my @arrFNRListTmp  = @{ $arefFNRListTmp };

    # V116 Begin
    # Filter FNR applicable for given output format alone
    if ( $sOutFmtPar ne '' ){
        my $sOutFmtTmp = ',' + $sOutFmtPar + ',';
        @arrFNRListTmp = grep {
            my $sFmts1Tmp = $_->getApplicableOutputs();
            my $sFmts2Tmp =  ',' . $sFmts1Tmp . ',';

            # Return true when
            # - FNR is applicable for all output formats (i.e, applicable outputs is empty)
            # - FNR is applicable for given output format (i.e, applicable outputs list contains given format)
            (    $sFmts1Tmp eq ''
              || $sFmts2Tmp =~ /$sOutFmtTmp/
            );
	    } @arrFNRListTmp;
	}
    # V116 End

    #---- Filter out FNRs applicable for table column ----#
    # if text is of table cell, Filter FNR applicable for cell column
    my @arrColFNRsTmp = ();
    if ( $hrefTableInfoPar != undef ){
        @arrColFNRsTmp = grep { $_->getColumn() eq $hrefTableInfoPar->{ 'ColId' } } @arrFNRListTmp;
    }

    # Filter FNR applicable for all content of block (i.e FNRs without column)
    my @arrBlkFNRsTmp = grep { $_->getColumn() eq '' } @arrFNRListTmp;
    @arrFNRListTmp = ( @arrColFNRsTmp, @arrBlkFNRsTmp );

    #---- Find and replace ----#
    my $iFNRResultTmp = 0;
    my $sTextTmp = $sTextPar;
    foreach my $fnrTmp ( @arrFNRListTmp ){
        $iFNRResultTmp = $fnrTmp->execute( $a2wObjTmp, $pomObjTmp, \$sTextTmp );
    }

    return $sTextTmp;
}

# V116 Begin
#---- Find and prepend (as configured in block content processor) ----#
#-----------------------------------------------------------------------
# Process given content for find and prepend
#
# In case of success, returns content to be prepended or empty string
#-----------------------------------------------------------------------
sub getContentToPrepend{
    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "getContentToPrepend()" ); }

    #---- Fetch parameter(s)
    #
    # 1. Output format
    # 2. Object hash (having both kernel object and pom object)
    # 3. Text
    # 4. Table info (hash { Cell => TRUE|FALSE, ColHdr => <Header name>, ColId => <Value> } reference, optional)
    #
    my $sOutFmtPar = shift;
    my $hrefObjPar = shift;
    my $sTextPar   = shift;
    my $hrefTableInfoPar = undef;
    if ( @_ > 0 ){ $hrefTableInfoPar = shift; }

    #---- Get actual objects
    my $a2wObjTmp = $hrefObjPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefObjPar->{ 'POMOBJ' };

    if ( $bLog == $TRUE ){
        $theLogger->logMessage( "Text:>" . $sTextPar . "<" );
        if ( $hrefTableInfoPar != undef ){
            $theLogger->logMessage( "Table: Cell:>" . (($hrefTableInfoPar->{ 'Cell' } == $TRUE) ? 'Yes' : 'No') . "< Header:>" . $hrefTableInfoPar->{ 'ColHdr' } . "< Column:>" . $hrefTableInfoPar->{ 'ColId' } . "<" );
        }
    }

    #---- Get active block
    my $pgHandleTmp  = $this->{ 'ActivePage' };
    my $blkActiveTmp = $pgHandleTmp->{ 'ActiveBlock' };
    if ( $blkActiveTmp == undef ){ return ''; }

    #---- Get content processor of block
    my $contProcTmp = $blkActiveTmp->getContentProcessor();
    if ( $contProcTmp == undef || $contProcTmp->hasFNPs() == $FALSE ){ return ''; }

    #---- Get find and prepend list
    my $arefFNPListTmp = $contProcTmp->getFindAndPrependList();
    my @arrFNPListTmp  = @{ $arefFNPListTmp };

    #---- Filter out FNPs applicable for given output format alone ----#
    if ( $sOutFmtPar ne '' ){
        my $sOutFmtTmp = ',' . $sOutFmtPar . ',';
        @arrFNPListTmp = grep {
            my $sFmts1Tmp = $_->getApplicableOutputs();
            my $sFmts2Tmp =  ',' . $sFmts1Tmp . ',';

            # Return true when
            # - FNP is applicable for all output formats (i.e, applicable outputs is empty)
            # - FNP is applicable for given output format (i.e, applicable outputs list contains given format)
            (    $sFmts1Tmp eq ''
              || $sFmts2Tmp =~ /$sOutFmtTmp/
            );
	    } @arrFNPListTmp;
	}
    if ( $bLog == $TRUE && @arrFNPListTmp <= 0 ){
        $theLogger->logMessage( "No find and prepend to execute for given output format:>$sOutFmtPar<" );
    }

    #---- Filter out FNPs applicable for table column ----#
    # if text is of table cell, Filter FNP applicable for cell column
    my @arrColFNPsTmp = ();
    if ( $hrefTableInfoPar != undef ){
        @arrColFNPsTmp = grep { $_->getColumn() eq $hrefTableInfoPar->{ 'ColId' } } @arrFNPListTmp;
    }

    #---- Filter out FNPs applicable for all content of block (i.e FNPs without column) ----#
    my @arrBlkFNPsTmp = grep { $_->getColumn() eq '' } @arrFNPListTmp;
    @arrFNPListTmp = ( @arrColFNPsTmp, @arrBlkFNPsTmp );

    #---- Find and prepend ----#
    my $iFNPResultTmp = 0;
    my $sCont2PrependTmp = '';
    foreach my $fnpTmp ( @arrFNPListTmp ){
        $iFNPResultTmp = $fnpTmp->execute( $a2wObjTmp, $pomObjTmp, \$sTextTmp );
        if ( $iFNPResultTmp == 0 ){ $sCont2PrependTmp .= $fnpTmp->getProcessedResult(); }
    }

    return $sCont2PrependTmp;
}

#---- Find and append (as configured in block content processor) ----#
#-----------------------------------------------------------------------
# Process given content for find and append
#
# In case of success, returns content to be appended or empty string
#-----------------------------------------------------------------------
sub getContentToAppend{
    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "getContentToAppend()" ); }

    #---- Fetch parameter(s)
    #
    # 1. Output format
    # 2. Object hash (having both kernel object and pom object)
    # 3. Text
    # 4. Table info (hash { Cell => TRUE|FALSE, ColHdr => <Header name>, ColId => <Value> } reference, optional)
    #
    my $sOutFmtPar = shift;
    my $hrefObjPar = shift;
    my $sTextPar   = shift;
    my $hrefTableInfoPar = undef;
    if ( @_ > 0 ){ $hrefTableInfoPar = shift; }

    #---- Get actual objects
    my $a2wObjTmp = $hrefObjPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefObjPar->{ 'POMOBJ' };

    if ( $bLog == $TRUE ){
        $theLogger->logMessage( "Text:>" . $sTextPar . "<" );
        if ( $hrefTableInfoPar != undef ){
            $theLogger->logMessage( "Table: Cell:>" . (($hrefTableInfoPar->{ 'Cell' } == $TRUE) ? 'Yes' : 'No') . "< Header:>" . $hrefTableInfoPar->{ 'ColHdr' } . "< Column:>" . $hrefTableInfoPar->{ 'ColId' } . "<" );
        }
    }

    #---- Get active block
    my $pgHandleTmp  = $this->{ 'ActivePage' };
    my $blkActiveTmp = $pgHandleTmp->{ 'ActiveBlock' };
    if ( $blkActiveTmp == undef ){ return ''; }

    #---- Get content processor of block
    my $contProcTmp = $blkActiveTmp->getContentProcessor();
    if ( $contProcTmp == undef || $contProcTmp->hasFNPs() == $FALSE ){ return ''; }

    #---- Get find and append list
    my $arefFNAListTmp = $contProcTmp->getFindAndAppendList();
    my @arrFNAListTmp  = @{ $arefFNAListTmp };

    #---- Filter out FNAs applicable for given output format alone ----#
    if ( $sOutFmtPar ne '' ){
        my $sOutFmtTmp = ',' . $sOutFmtPar . ',';
        @arrFNAListTmp = grep {
            my $sFmts1Tmp = $_->getApplicableOutputs();
            my $sFmts2Tmp =  ',' . $sFmts1Tmp . ',';

            # Return true when
            # - FNA is applicable for all output formats (i.e, applicable outputs is empty)
            # - FNA is applicable for given output format (i.e, applicable outputs list contains given format)
            (    $sFmts1Tmp eq ''
              || $sFmts2Tmp =~ /$sOutFmtTmp/
            );
	    } @arrFNAListTmp;
	}

    #---- Filter out FNAs applicable for table column ----#
    # if text is of table cell, Filter FNP applicable for cell column
    my @arrColFNAsTmp = ();
    if ( $hrefTableInfoPar != undef ){
        @arrColFNAsTmp = grep { $_->getColumn() eq $hrefTableInfoPar->{ 'ColId' } } @arrFNAListTmp;
    }

    #---- Filter out FNAs applicable for all content of block (i.e FNPs without column) ----#
    my @arrBlkFNAsTmp = grep { $_->getColumn() eq '' } @arrFNAListTmp;
    @arrFNAListTmp = ( @arrColFNAsTmp, @arrBlkFNAsTmp );

    #---- Find and append ----#
    my $iFNAResultTmp = 0;
    my $sCont2AppendTmp = '';
    foreach my $fnaTmp ( @arrFNAListTmp ){
        $iFNAResultTmp = $fnaTmp->execute( $a2wObjTmp, $pomObjTmp, \$sTextTmp );
        if ( $iFNAResultTmp == 0 ){ $sCont2AppendTmp .= $fnaTmp->getProcessedResult(); }
    }

    return $sCont2AppendTmp;
}
# V116 End

# $V103 Begin
#-----------------------------------------------------------------------
# Get table height
#-----------------------------------------------------------------------
sub getTableHeight{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "getTableHeight()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Block
    #
    my $blkTablePar = shift;

    if ( $this->{ 'InTable' } == 0 ){ return 0; }

    return 0;
}
# $V103 End

# $V104 Begin
#---- Illustration related writing methods ----#
#-----------------------------------------------------------------------
# Begin illustration
#-----------------------------------------------------------------------
sub beginIllustration{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "beginIllustration()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Block
    #
    my $blkIllustrationPar = shift;

    return 0;
}

#-----------------------------------------------------------------------
# Write illustration line
#-----------------------------------------------------------------------
sub writeIllustrationLine{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeIllustrationLine()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Array of line objects
    # 2. Line count
    # 3. Line gap (vertical difference between previous and current line, zero for first line)
    #
    my $arefObjectsPar = shift;
    my $iLineCountPar  = shift;
    my $iLineGapPar    = shift;

    #---- Fetch objects list
    my @arrObjsTmp = @{ $arefObjectsPar };
    foreach my $a2wObjTmp ( @arrObjsTmp ){
        $this->writeObject( $a2wObjTmp );
    }

    return 0;
}

#-----------------------------------------------------------------------
# End illustration
#-----------------------------------------------------------------------
sub endIllustration{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "endIllustration()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Block
    #
    my $blkIllustrationPar = shift;

    return 0;
}
# $V104 End

# $V109 Begin
#---- List related writing methods ----#
#-----------------------------------------------------------------------
# Begin list
#-----------------------------------------------------------------------
sub beginList{
    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "beginList()" ); }

    #---- Fetch parameter(s)
    #
    # 1. Block
    # 2. Columns X positions array
    # 3. Starting Y position
    #
    my $blkListPar = shift;
    my $arefColumnsXPosPar = shift;
    my $iStartYPar         = shift;
    if ( @{ $arefColumnsXPosPar } <= 0 ){ return -1; }

    #---- Set in list flag
    $this->{ 'InList' } = 1;

    return 0;
}

#-----------------------------------------------------------------------
# Write list caption
#-----------------------------------------------------------------------
sub writeListCaption{
    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "writeListCaption()" ); }

    #---- Fetch parameter(s)
    #
    # 1. Array of label objects
    #
    my $arefObjectsPar = shift;

    #---- Set in item label flag
    $this->{ 'InList' } |= (1 << 4);

    #---- Fetch objects list
    my @arrObjsTmp = @{ $arefObjectsPar };
    foreach my $a2wObjTmp ( @arrObjsTmp ){
        $this->writeObject( $a2wObjTmp );
    }

    #---- Reset in item label flag
    $this->{ 'InList' } ^= (1 << 4);

    return 0;
}

#-----------------------------------------------------------------------
# Begin list item
#-----------------------------------------------------------------------
sub beginListItem{
    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "beginListItem()" ); }

    #---- Fetch parameter(s)
    #
    # 1. List item number
    # 2. Page id (where list item starts on)
    #
    my $iListItemNrPar = shift;
    my $iPageIdPar     = shift;

    #---- Set in list item flag
    $this->{ 'InList' } |= (1 << 1);

    return 0;
}

#-----------------------------------------------------------------------
# Write list label
#-----------------------------------------------------------------------
sub writeListLabel{
    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "writeListLabel()" ); }

    #---- Fetch parameter(s)
    #
    # 1. Array of label objects
    #
    my $arefObjectsPar = shift;

    #---- Set in item label flag
    $this->{ 'InList' } |= (1 << 2);

    #---- Fetch objects list
    my @arrObjsTmp = @{ $arefObjectsPar };
    foreach my $a2wObjTmp ( @arrObjsTmp ){
        $this->writeObject( $a2wObjTmp );
    }

    #---- Reset in item label flag
    $this->{ 'InList' } ^= (1 << 2);

    return 0;
}

#-----------------------------------------------------------------------
# Write list body
#-----------------------------------------------------------------------
sub writeListBody{
    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "writeListBody()" ); }

    #---- Fetch parameter(s)
    #
    # 1. Array of body objects
    #
    my $arefObjectsPar = shift;

    #---- Set in item body flag
    $this->{ 'InList' } |= (1 << 3);

    #---- Fetch objects list
    my @arrObjsTmp = @{ $arefObjectsPar };
    foreach my $a2wObjTmp ( @arrObjsTmp ){
        $this->writeObject( $a2wObjTmp );
    }

    #---- Reset in item body flag
    $this->{ 'InList' } ^= (1 << 3);

    return 0;
}

#-----------------------------------------------------------------------
# End list item
#-----------------------------------------------------------------------
sub endListItem{
    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "endListItem()" ); }

    #---- Fetch parameter(s)
    #
    # 1. List item number
    #
    my $iListItemNrPar = shift;

    #---- Reset in list item flag
    $this->{ 'InList' } ^= (1 << 1);

    return 0;
}

#-----------------------------------------------------------------------
# End list
#-----------------------------------------------------------------------
sub endList{
    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "endList()" ); }

    #---- Fetch parameter(s)
    #
    # 1. Block
    #
    my $blkListPar = shift;

    #---- Reset in list flag
    $this->{ 'InList' } = 0;

    return 0;
}
# $V109 End

# V114b Begin
#-----------------------------------------------------------------------
# isFontResource
#
# Checks whether given resource is font or not
#
# Returns true if resource is a2w::Font else false
#-----------------------------------------------------------------------
sub isFontResource{
    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "endList()" ); }

    #---- Fetch parameter(s)
    #
    # 1. Resource (of type a2w::Font)
    #
    my $a2wFontPar = shift;

    return ( $a2wFontPar != undef && ref( $a2wFontPar ) eq "a2w::Font" ) ? $TRUE : $FALSE;
}
# $V109 End

#-----------------------------------------------------------------------
# Don't remove the following lines !!!
#-----------------------------------------------------------------------
1;
__END__
