#-------------------------------------------------------------------------------
#  ada.pm:
#
#  Module to process page contents for language specific natural reading order and
#  tag the contents with appropriate identifiers to aid PDF/UA transformation
#
#  Call:
#  On Windows : afp2web.exe -q -c -doc_cold -sp:ada.pm samples\insure.afp
#  On Unix    : ./afp2web   -q -c -doc_cold -sp:ada.pm samples/insure.afp
#
#  Author    : Fa. Maas (AFP2web Team)
#  Copyright : (C) 2017-2020 by Maas Holding GmbH
#
#  $V100    2017-10-12    Initial Release
#
#  $V101    2018-03-13    Extended to load block definition from NOP value and priority of loaded block definitions are
#                         Priority 1: Block definition specified using script argument
#                         Priority 2. Spool level block definition (block definition included in NOPs before first BDT)
#                         Priority 3. Document level block definition (block definition included in NOPs within BNG)
#
#  $V102    2018-03-23    a. Fixed issue in loading NOP embedded block definition
#                         b. Modified to read in NOP embedded block definition data AS IS (instead of getting translated)
#
#  $V103    2018-03-24    a. Modified to read in NOP embedded block definition data AS IS or as translated data by checking
#                            'BLOCK_DEF_' start identifier
#
#  $V104    2018-04-27    Fixed minor bug in handling priority of block definitions defined at script argument and document
#                         NOP SFIs
#
#  $V105    2018-11-29    a. AFP-771: Extended to handle Creator, Subject and Keywords document properties passed through DOCUMENTBLOCK TLE
#                         b. AFP-771: Extended to process multiple DOCUMENTBLOCK TLEs (when document properties cannot be specified using
#                            single DOCUMENTBLOCK TLE due to 254 characters TLE value length limitation)
#
#  $V106    2020-02-26    AFP-928: Extended AFP2PDF/UA as generic transformation
#-------------------------------------------------------------------------------

#-----------------------------------------------------------------------
# BEGIN block of module
#
# Extends PERL module search path array (@INC) with new element having
# this script modules path in order to have better module portability
#-----------------------------------------------------------------------
BEGIN {
    #---- Fetch script filename
    my $sScriptFilenameTmp = $0;

    #---- Extract script file path from script filename
    my $sScriptFilePathTmp = "";
    if ( $sScriptFilenameTmp =~ /(.*)\/.*\.pm/ ){
        $sScriptFilePathTmp = $1;
    }

    #printf STDERR ( "Script filename: " . $0 . " Script filepath: " . $sScriptFilePathTmp . "\n" );
    if ( $sScriptFilePathTmp eq "" ){
        $sScriptFilePathTmp = ".";
    }
    else {
        my $sScriptFileParentPathTmp = "";
        if ( $sScriptFilePathTmp =~ /(.*)\/sfsamples/ ){
            $sScriptFileParentPathTmp = $1;
        }

        #---- Add script file parent path to module search path
        if ( $sScriptFileParentPathTmp ne "" ){
            unshift( @INC, $sScriptFileParentPathTmp );
        }
    }

    #---- Add script file path to module search path
    unshift( @INC, $sScriptFilePathTmp );
    unshift( @INC, $sScriptFilePathTmp . "/a2w" );

    # V106 Begin
    #---- Add perl path to module search path
    unshift( @INC, $sScriptFilePathTmp . "/perl/lib" );
    unshift( @INC, $sScriptFilePathTmp . "/perl/site/lib" );
    unshift( @INC, $sScriptFilePathTmp . "/../../../perl/lib" );
    unshift( @INC, $sScriptFilePathTmp . "/../../../perl/site/lib" );
    # V106 End
}

use a2w::Config;
use a2w::ContentBlock;
use a2w::Document;
use a2w::Font;
use a2w::Index;
use a2w::Kernel;
use a2w::Line;
use a2w::MediumMap;
use a2w::NOP;
use a2w::Overlay;
use a2w::Page;
use a2w::PSEG;
use a2w::Text;

use a2w::ConfigConstants;
use a2w::ContentBlockConstants;
use a2w::DocumentConstants;
use a2w::PageConstants;
use a2w::FontConstants;

use a2w::core::dm::MiningEngine;     # Data mining module
use a2w::core::log::Logger;          # Log engine
use a2w::core::visitor::AccessibilityVisitor;

use JSON::Tiny;

#-----------------------------------------------------------------------
# Initialize once per process
#-----------------------------------------------------------------------
sub initialize(){

    #---- Get Parameter of initialize( Par: a2w::Config, a2w::Kernel )
    ( $a2wConfigPar, $a2wKernelPar ) = @_;

    #---- Define boolean values
    $TRUE  = 1;    # TRUE  boolean value
    $FALSE = 0;    # FALSE boolean value

    #---- Set/Reset Logging
    $bLog = $FALSE;
    if (index( lc($a2wConfigPar->getAttribute( $a2w::ConfigConstants::LOGGINGLEVEL )), "sf") >= 0 ){
        $bLog = $TRUE;
    }

    my $sScriptProcTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SCRIPTPROCEDURE );
    my $sScriptArgsTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SCRIPTARGUMENT );
    $sIndexFilePath    = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::INDEXPATH );
    $sDocumentTitle    = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::TITLE );
    $sDocumentCreator  = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::CREATOR );  # V105 Change
    $sDocumentSubject  = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SUBJECT );  # V105 Change
    $sDocumentKeywords = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::KEYWORDS ); # V105 Change
    $sOutputLanguage   = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::OUTPUTLANGUAGE );
    $sOutputFilePath   = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::OUTPUTFILEPATH );
    $sTempFilePath     = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::TEMPPATH );
    $iOutputResolution = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::RESOLUTION );
    $sSpoolFilename    = $a2wKernelPar->getSpoolFilename();

    #---- Set AutoSplit to true
    $a2wConfigPar->setAttribute( $a2w::ConfigConstants::AUTOSPLIT, "on" );

    #---- Add temp path to module search path
    unshift( @INC, $sTempFilePath ); # V101 Change

    #---- Instantiate Logger
    $theLogger = undef;
    $bDebugLog = $FALSE;
    if ( $bLog == $TRUE ){
        #---- Set log on log engine
        $theLogger = a2w::core::log::Logger->getSingleton();
        $theLogger->setStartTime( $sStartTime );

        #---- Register modules that has to be logged
        $theLogger->registerClasses(   "a2w::Main"
                                     . ",a2w::core::dm::Block"
                                     . ",a2w::core::dm::MiningEngine"
                                     . ",a2w::core::visitor::AccessibilityVisitor"
                                   );

        #---- Open log file
        my $sLogFilenameTmp = $sSpoolFilename;
        $sLogFilenameTmp =~ s/^.*[\\\/]//;
        $sLogFilenameTmp .= ".ada.log";
        $theLogger->setFilename( $sLogFilenameTmp );
        if ( $theLogger->open( $sOutputFilePath ) == $FALSE ){
            return ( -1, "[ERROR] Unable to open log file (" . $sOutputFilePath . $sLogFilenameTmp . ")" );
        }
        $bLog = $theLogger->isRegistered( "a2w::Main" );

        if (index( lc($a2wConfigPar->getAttribute( $a2w::ConfigConstants::LOGGINGLEVEL )), "sfdbg") >= 0 ){
            $theLogger->setLevel( $a2w::core::log::Logger::LEVEL_DEBUG );
            $bDebugLog = $TRUE;
        }

        $theLogger->logFunctionName( "a2w::Main", "initialize()" );
        $theLogger->logMessage( "Running $sScriptProcTmp..." );
        $theLogger->logMessage( "initialize(): Processing " . $sSpoolFilename );
        $theLogger->logMessage( "initialize(): Args: $sScriptArgsTmp, OutputFilePath: $sOutputFilePath, OutputLanguage: $sOutputLanguage" );
    }

    #---- Page process flags
    $APPEND = 0;    # append page to Current Document
    $SKIP   = 1;    # skip page
    $NEWDOC = 2;    # new document

    #---- Create mining engine
    $dmMiningEngine = new a2w::core::dm::MiningEngine();

    #---- Fill info on mining engine
    $dmMiningEngine->registerClassesForLog();
    $dmMiningEngine->setSpoolFilename( $sSpoolFilename );
    $dmMiningEngine->setOutputPath( $sOutputFilePath );
    $dmMiningEngine->setOutputRes( $iOutputResolution );

    # V101 Begin
    #---- Load config
    @arrBlocks = ();
    $hrefGlobalBlocks = undef; # Script argument specified block definition
    $hrefSpoolBlocks = undef;  # Block definition specified in NOPs before first document (spool level)
    $hrefBlocks = undef;       # Block definition specified in NOPs within named page group (document level)
    if ( $sScriptArgsTmp ne "" && $sScriptArgsTmp =~ /^.*\:\:.*$/ ){
        my $sErrMsgTmp = "";
        ( $hrefGlobalBlocks, $sErrMsgTmp ) = _loadBlockDefinition( $sScriptArgsTmp );
        if ( lc( ref( $hrefGlobalBlocks ) ) ne "hash" && $hrefGlobalBlocks < 0 ){ return ( $hrefGlobalBlocks, $sErrMsgTmp ); }
        $hrefBlocks = $hrefGlobalBlocks;
    }
    else {
        if ( $bLog == $TRUE ){ if ( $sScriptArgsTmp ne "" ){ $theLogger->logMessage( "Invalid script argument: $sScriptArgsTmp" ); } }
    }

    $iPageId = 0;
    # V101 End

    #---- Document properties
    $hrefDocProperties = {};

    return $iRetTmp;
}

#-----------------------------------------------------------------------
# InitializeDoc for each document
#-----------------------------------------------------------------------
sub initializeDoc(){

    #---- Get Parameter of initializeDoc( Par: a2w::Document )
    ($a2wDocumentPar) = @_;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( "a2w::Main", "initializeDoc()" );
        $theLogger->logMessage( "Name=" . $a2wDocumentPar->getName() . " Id=" . $iDocumentId );
    }

    $iPageId = 0; # V101 Change

    return 0;
}

#-----------------------------------------------------------------------
# InitializePage for each page
#-----------------------------------------------------------------------
sub initializePage(){

    #---- Get Parameter of initializePage( Par: a2w::Page )
    ($a2wPagePar) = @_;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "initializePage()" ); }

    $iPageId++; # V101 Change

    return 0;
}

#-----------------------------------------------------------------------
# Main entry method
# Return values:
#        < 0:    error
#         0:    append page to Current Document
#         1:    skip page
#         2:    first page / new document
#-----------------------------------------------------------------------
sub afp2web(){

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( "a2w::Main", "afp2web()" );
        $theLogger->logMessage( "PageId=" . $a2wPagePar->getParseId() );
    }

    #---- Set default return value
    my $iRetTmp = $APPEND; # append page to Current Document

    # V101 Begin
    if ( $iPageId == 1 ){
        #---- Process NOPs (for block definition) ----#
        # $V104 Begin
        my $iRetTmp = 0;
		my $sMsgTmp = undef;
        if ( $hrefBlocks == undef ){ ( $iRetTmp, $sMsgTmp ) = _processDocumentNOPs( $a2wPagePar ); }
        # $V104 End
        if ( $iRetTmp < 0 ){ return ( $iRetTmp, $sMsgTmp ); }

        #---- Assert for block definition ----#
        # If block definition is not specified in script argument and also miss in NOPs, then throw error
        if ( $hrefBlocks == undef ){ return ( -1, "Missing block definition in script argument and in AFP NOPs" ); }

        #---- Process indexes ----#
        $iRetTmp = _processDocumentIndexes( $a2wDocumentPar );
        if ( $iRetTmp < 0 ){ return ( -1, "Unable to process document " . $iDocumentId . " indexes" ); }

        #---- Process document block attributes ----#
        my $sOutLangTmp  = $sOutputLanguage;
        my $sDocTitleTmp = $sDocumentTitle;
        my $sDocCreatorTmp  = $sDocumentCreator;  # V105 Change
        my $sDocSubjectTmp  = $sDocumentSubject;  # V105 Change
        my $sDocKeywordsTmp = $sDocumentKeywords; # V105 Change
        if ( ref( $hrefDocProperties ) eq "HASH" ){
            #---- Get document language
            if ( defined $hrefDocProperties->{ 'lang' } ){ $sOutLangTmp = $hrefDocProperties->{ 'lang' }; }

            #---- Get document title
            if ( defined $hrefDocProperties->{ 'title' } ){ $sDocTitleTmp = $hrefDocProperties->{ 'title' }; }

            # V105 Begin
            # Handle creator, subject and keywords document properties
            if ( defined $hrefDocProperties->{ 'creator' } ){ $sDocCreatorTmp = $hrefDocProperties->{ 'creator' }; }
            if ( defined $hrefDocProperties->{ 'subject' } ){ $sDocSubjectTmp = $hrefDocProperties->{ 'subject' }; }
            if ( defined $hrefDocProperties->{ 'keywords' } ){ $sDocKeywordsTmp = $hrefDocProperties->{ 'keywords' }; }
            # V105 End
        }
        $a2wConfigPar->setAttribute( $a2w::ConfigConstants::OUTPUTLANGUAGE, $sOutLangTmp );
        $a2wConfigPar->setAttribute( $a2w::ConfigConstants::TITLE, $sDocTitleTmp );
        # V105 Begin
        $a2wConfigPar->setAttribute( $a2w::ConfigConstants::CREATOR, $sDocCreatorTmp );
        $a2wConfigPar->setAttribute( $a2w::ConfigConstants::SUBJECT, $sDocSubjectTmp );
        $a2wConfigPar->setAttribute( $a2w::ConfigConstants::KEYWORDS, $sDocKeywordsTmp );
        # V105 End

        #---- Initialize document
        $dmMiningEngine->initializeDoc(   'a2w::core::visitor::AccessibilityVisitor'
                                        , $a2wDocumentPar
                                        , $sDocTitleTmp
                                        , $FALSE # Page output is false always for ADA accessibility visitor
                                      );
    }
    # V101 End

    #---- Process page
    my $iPPRetTmp = $dmMiningEngine->processPage( $a2wPagePar, $TRUE );
    if ( $iPPRetTmp < 0 ){ return ( -1, "Unable to process page " . $a2wPagePar->getParseId() . " content" ); }

    #---- Write page
    $dmMiningEngine->writePage( $a2wPagePar->getParseId() );

    return $iRetTmp;
}

#-----------------------------------------------------------------------
# FinalizePage for each page
#-----------------------------------------------------------------------
sub finalizePage(){

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "finalizePage()" ); }

    return 0;
}

#-----------------------------------------------------------------------
# FinalizeDoc for each document
#-----------------------------------------------------------------------
sub finalizeDoc(){

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "finalizeDoc()" ); }

    #---- Dump the page contents
    if ( $bDebugLog == $TRUE ){
        $dmMiningEngine->dumpDOM( $sOutputFilePath, $TRUE );
    }

    #---- Finalize document
    my $iRetTmp = $dmMiningEngine->finalizeDoc();
    return $iRetTmp;
}

#-----------------------------------------------------------------------
# Finalize once per process
#-----------------------------------------------------------------------
sub finalize(){

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "finalize()" ); }

    #---- Set end time and close log file
    if ( $bLog == $TRUE ){        
        $theLogger->setEndTime( time() );    # set end time
        $theLogger->close();                # close log file
    }

    #---- Cleanup
    $theLogger      = undef;
    $dmMiningEngine = undef;

    return 0;
}

# V101 Begin
#-----------------------------------------------------------------------
# Load block definition
#
# Loads given block definition module and returns the hash structure
# Returns <0 and message in case of error
#
#-----------------------------------------------------------------------
sub _loadBlockDefinition{

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "_loadBlockDefinition()" ); }

    #---- Get parameter
    #
    # 1. Module name
    # 2. Module filename (optional)
    #
    my $sBlkDefModPar = shift;
    my $sBlkDefFilePar = "";
    if ( @_ > 0 ){ $sBlkDefFilePar = shift; }

    #---- Assert parameter
    if ( $sBlkDefFilePar eq "" && $sBlkDefModPar eq "" ){ return ( -1, "Empty block definition module name" ); } # Return if name is empty
    if ( $sBlkDefFilePar eq "" && $sBlkDefModPar !~ /^.*\:\:.*$/ ){ return ( -1, "Invalid block definition module name ($sBlkDefModPar)" ); } # Return if name is not like module name

    #---- Parse and load block definition module
    if ( $sBlkDefFilePar eq "" ){ # $V102 Change
        eval "require $sBlkDefModPar";
        if ( $@ ){ return ( -2, "Unable to parse block definition module $sBlkDefModPar. reason=" . $@ ); }
    }
    else {
        require $sBlkDefFilePar;
    }
    my $hrefBlocksTmp = \%{ $sBlkDefModPar . '::Blocks' }; # Fetch defined blocks list

    #---- Load all blocks
    $dmMiningEngine->initialize(); # Initialize mining engine
    $dmMiningEngine->loadConfig( \%{ $sBlkDefModPar . '::Blocks' } );

    #---- Fetch all config module defined blocks
    @arrBlocks = sort keys( %{ $hrefBlocksTmp } );
    if ( $bLog == $TRUE ){ $theLogger->logMessage( "$sBlkDefModPar Defined Blocks:>@arrBlocks<" ); }

    #---- Iterate through blocks list and add them as document block
    my $blkCurrentTmp = undef;
    foreach my $sBlkNameTmp ( @arrBlocks ){
        #---- Get block
        $blkCurrentDefTmp = $hrefBlocksTmp->{ $sBlkNameTmp };
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Name:>" . $sBlkNameTmp . "< Id:>" . $blkCurrentDefTmp->{ 'Id' } . "< Info:>" . $blkCurrentDefTmp->{ 'Info' } . "<" ); }

        #---- Add block to document
        $dmMiningEngine->addDocumentBlock( $blkCurrentDefTmp->{ 'Id' }, $blkCurrentDefTmp->{ 'Info' } );
    }

    return $hrefBlocksTmp;
}

#-----------------------------------------------------------------------
# Process document NOPs (for block definition)
#-----------------------------------------------------------------------
sub _processDocumentNOPs{

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "_processDocumentNOPs()" ); }

    #---- Get parameter
    #
    # 1. Page
    #
    my $a2wFirstPagePar = shift;

    #---- Process document NOPs and collect block definition chunks ----#
    #---- Get first NOP
    my $iNOPCountTmp = 0;
    my $sNOPDataTmp = "";
    my $hrefBlockChunksTmp = {};
    my $a2wNOPTmp = $a2wFirstPagePar->getFirstNOP();
    while ( $a2wNOPTmp != 0 ){
        $iNOPCountTmp++;

        #---- Collect NOP with BLOCK_DEF_<4 digits> start up
        $sNOPDataTmp = $a2wNOPTmp->getValue(); # $V102 change
        if ( $sNOPDataTmp !~ /^BLOCK_DEF_(\d{4})/ ){ # $V103 change
            $sNOPDataTmp = $a2wNOPTmp->getEBCDICValue(); # $V102 change
        }
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "NOP " . sprintf( "%04d", $iNOPCountTmp ) . ":>" . $sNOPDataTmp . "<" ); }
        if ( $sNOPDataTmp =~ /^BLOCK_DEF_(\d{4})/ ){
            $hrefBlockChunksTmp->{ $1 } = substr( $sNOPDataTmp, 15 ); # 15 is length of BLOCK_DEF_<4 digits>
        }

        #---- Get next NOP
        $a2wNOPTmp = $a2wFirstPagePar->getNextNOP();
    }

    #---- Merge block definition chunks ----#
    my $sBlockDefBufferTmp = "";
    my @arrChunkIdsTmp = sort keys( %{ $hrefBlockChunksTmp } );
    foreach my $chunkIdTmp ( @arrChunkIdsTmp ){
        $sBlockDefBufferTmp .= $hrefBlockChunksTmp->{ $chunkIdTmp };
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "BLOCK_DEF_" . $chunkIdTmp . ":>" . $hrefBlockChunksTmp->{ $chunkIdTmp } . "<" ); }
    }
    if ( $sBlockDefBufferTmp eq "" ){ return 0; }

    #---- Write block definition in temp path ----#
    my $sBlockDefFilenameTmp = $sTempFilePath;
    my $tCurrentTmp = time();

    #---- Build filename
    #   0    1    2     3     4    5     6     7     8
    #($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime(time);
    my @arrCurrentTimeTmp = localtime( $this->{ 'tStartTime' } );
    @arrCurrentTimeTmp[ 5 ] += 1900;  # Year is not complete 4 digit, so adding 1900
    @arrCurrentTimeTmp[ 4 ]++;        # Day of month starts from 0, so adding 1
    $sBlockDefFilenameTmp .= sprintf(   "%04d%02d%02d_%02d%02d%02d_%06d_%03d.pm"
                                      , @arrCurrentTimeTmp[ 5 ]
                                      , @arrCurrentTimeTmp[ 4 ]
                                      , @arrCurrentTimeTmp[ 3 ]
                                      , @arrCurrentTimeTmp[ 2 ]
                                      , @arrCurrentTimeTmp[ 1 ]
                                      , @arrCurrentTimeTmp[ 0 ]
                                      , $$             # Process id
                                      , rand( 999 )    # 3 digit random value
                                    );

    #---- Write block definition buffer into file
    if ( open( BLOCK_DEF, ">" . $sBlockDefFilenameTmp ) ){
        binmode( BLOCK_DEF );
        syswrite( BLOCK_DEF, $sBlockDefBufferTmp );
        close( BLOCK_DEF );
    }
    else {
        return ( -1, "Unable to dump block definition in file (" . $sBlockDefFilenameTmp . "), reason: " . $! );
    }

    #---- Load the module ----#
    my $sModNameTmp = "";
    if ( $sBlockDefBufferTmp =~ /package\s+([^\:]*\:\:[^;]*)/ ){ $sModNameTmp = $1; }
    if ( $sModNameTmp eq "" ){ return ( -2, "Unable to detect 'package <module name>;' in NOP embedded block definition content" ); }
    my ( $iRetTmp, $sMsgTmp ) = _loadBlockDefinition( $sModNameTmp, $sBlockDefFilenameTmp );
    if ( lc( ref( $iRetTmp ) ) ne "hash" && $iRetTmp < 0 ){
        #---- Delete the temporary module file
        unlink( $sBlockDefFilenameTmp );

        return ( $iRetTmp, $sMsgTmp );
    }
    $hrefBlocks = $iRetTmp;
    if ( $hrefSpoolBlocks == undef ){ $hrefSpoolBlocks = $iRetTmp; }

    #---- Delete the temporary module file once loaded
    unlink( $sBlockDefFilenameTmp );

    return 0;
}
# V101 End

#-----------------------------------------------------------------------
# Process document indexes
#-----------------------------------------------------------------------
sub _processDocumentIndexes{

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "_processDocumentIndexes()" ); }

    #---- Get parameter
    #
    # 1. Document
    #
    my $a2wDocPar = shift;

    #---- Process document indexes ----#
    #---- Get first index
    my $sIdxNameTmp  = "";
    my $hrefAttrsTmp = { 'Info' => "" };
    my $a2wIndexTmp  = $a2wDocPar->getFirstIndex();
    while ( $a2wIndexTmp != 0 ){
        #---- Process index
        $sIdxNameTmp = $a2wIndexTmp->getName();

        #---- Jump to next index, if name is empty
        if ( $sIdxNameTmp eq "" ){ $a2wIndexTmp = $a2wDocPar->getNextIndex(); next; }
        #if ( $bLog == $TRUE ){ $theLogger->logMessage( "  Index: " . $sIdxNameTmp . "=>" . $hrefAttrsTmp->{ 'Info' } . "<" ); }

        #---- Handle DOCUMENTBLOCK index for document properties
        $hrefAttrsTmp->{ 'Info' } = $a2wIndexTmp->getValue();
        if ( lc( $sIdxNameTmp ) eq "documentblock" ){
            # V105 Begin
            # Handle multiple DOCUMENTBLOCK TLEs
            #
            #---- Fill in document properties (Parse json data)
            my $jsonRetTmp = JSON::Tiny::from_json( $hrefAttrsTmp->{ 'Info' } );
            if ( ref( $jsonRetTmp ) ne "HASH" ){
                if ( $bLog == $TRUE ){ $theLogger->logMessage( "Unable to parse index " . $sIdxNameTmp . " JSON value (" . $hrefAttrsTmp->{ 'Info' } . "). Reason: $jsonRetTmp" ); }
                return ( -2, "Unable to parse index " . $sIdxNameTmp . " JSON value (" . $hrefAttrsTmp->{ 'Info' } . "). Reason: $jsonRetTmp" );
            }

            #---- Merge current document block properties with existing
            @$hrefDocProperties{keys %$jsonRetTmp} = values %$jsonRetTmp;
            # V105 End

            #---- Get next index
            $a2wIndexTmp = $a2wDocPar->getNextIndex();

            next;
        }


        #---- Update document block with info
        $dmMiningEngine->updateDocumentBlock( $sIdxNameTmp, $hrefAttrsTmp );

        #---- Get next index
        $a2wIndexTmp = $a2wDocPar->getNextIndex();
    }

    return 0;
}

__END__
