#-------------------------------------------------------------------------------
#  dumpPageVectorObjs.pm:
#
#  Scripting Facility to dump page vector objects
#
#  Call:
#
#  On Windows:    afp2web.exe -q -c -doc_cold -sp:sfsamples\dumpPageVectorObjs.pm samples\insure.afp
#
#  On Unix:    ./afp2web   -q -c -doc_cold -sp:sfsamples/dumpPageVectorObjs.pm samples/insure.afp
#
#  Author  : Fa. Maas
#  Date    : 2013-12-09
#  Version : 1.0.0
#
#  $V100    2013-12-09    Initial Release
#
#-------------------------------------------------------------------------------

#-----------------------------------------------------------------------
# BEGIN block of module
#
# Extends PERL module search path array (@INC) with new element having
# this script modules path in order to have better module portability
#-----------------------------------------------------------------------
BEGIN {
    #---- Fetch script filename
    my $sScriptFilenameTmp = $0;

    #---- Extract script file path from script filename
    my $sScriptFilePathTmp = "";
    if ( $sScriptFilenameTmp =~ /(.*)\/.*\.pm/ ){
        $sScriptFilePathTmp = $1;
    }

    #printf STDERR ( "Script filename: " . $0 . " Script filepath: " . $sScriptFilePathTmp . "\n" );
    if ( $sScriptFilePathTmp eq "" ){
        $sScriptFilePathTmp = ".";
    }
    else {
        my $sScriptFileParentPathTmp = "";
        if ( $sScriptFilePathTmp =~ /(.*)\/sfsamples/ ){
            $sScriptFileParentPathTmp = $1;
        }

        #---- Add script file parent path to module search path
        if ( $sScriptFileParentPathTmp ne "" ){
            unshift( @INC, $sScriptFileParentPathTmp );
        }
    }

    #---- Add script file path to module search path
    unshift( @INC, $sScriptFilePathTmp );
}

use a2w::Config;
use a2w::Document;
use a2w::Kernel;
use a2w::Page;
use a2w::Vector;

use a2w::ConfigConstants;
use a2w::DocumentConstants;
use a2w::PageConstants;

#-----------------------------------------------------------------------
# Initialize once per process
#-----------------------------------------------------------------------
sub initialize(){

    #---- Get Parameter of initialize( Par: a2w::Config, a2w::Kernel )
    ( $a2wConfigPar, $a2wKernelPar ) = @_;

    #---- Define boolean values
    $TRUE  = 1;    # TRUE  boolean value
    $FALSE = 0;    # FALSE boolean value

    #---- Set/Reset Logging
    $bLog = $FALSE;
    if (index( lc($a2wConfigPar->getAttribute( $a2w::ConfigConstants::LOGGINGLEVEL )), "sf") >= 0 ){
        $bLog = $TRUE;
    }

    my $sScriptProcTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SCRIPTPROCEDURE );
    my $sScriptArgsTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SCRIPTARGUMENT );
    $sIndexFilePath    = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::INDEXPATH );
    $sOutputFilePath   = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::OUTPUTFILEPATH );
    $sSpoolFilename    = $a2wKernelPar->getSpoolFilename();

    if ( $bLog == $TRUE ){
        printf STDERR ( "Running $sScriptProcTmp...\n" );
        printf STDERR ( "initialize(): Processing $sSpoolFilename\n" );
        printf STDERR ( "initialize(): Args: $sScriptArgsTmp, IndexFilePath: $sIndexFilePath, OutputFilePath: $sOutputFilePath\n" );
    }

    #---- Page process flags
    $APPEND = 0;    # append page to Current Document
    $SKIP   = 1;    # skip page
    $NEWDOC = 2;    # new document

    #---- Open Dump file
    my ($sSpoolFilenamePathTmp, $sDumpFilenameTmp) = ($sSpoolFilename =~ /^((?:.*[:\\\/])?)(.*)/s);
    $sDumpFilenameTmp = $sOutputFilePath . $sDumpFilenameTmp . ".vectors.txt";
    open( fDumpFile, ">$sDumpFilenameTmp" );
    printf STDERR ( "Running $sScriptProcTmp: Dumping to $sDumpFilenameTmp...\n" );
    return 0;
}

#-----------------------------------------------------------------------
# InitializeDoc for each document
#-----------------------------------------------------------------------
sub initializeDoc(){

    #---- Get Parameter of initializeDoc( Par: a2w::Document )
    ($a2wDocumentPar) = @_;

    if ( $bLog == $TRUE ){
        printf STDERR ( "initializeDoc(): DocId " . $a2wDocumentPar->getId() . "\n" );
    }
    return 0;
}

#-----------------------------------------------------------------------
# InitializePage for each page
#-----------------------------------------------------------------------
sub initializePage(){

    #---- Get Parameter of initializePage( Par: a2w::Page )
    ($a2wPagePar) = @_;

    if ( $bLog == $TRUE ){
        printf STDERR ( "initializePage()\n" );
    }
    return 0;
}

#-----------------------------------------------------------------------
# Main entry method
# Return values:
#        < 0:    error
#         0:    append page to Current Document
#         1:    skip page
#         2:    first page / new document
#-----------------------------------------------------------------------
sub afp2web(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "afp2web(): PageId " . $a2wPagePar->getParseId() . "\n" );
    }

    #---- Set default return value
    my $iRetTmp = $APPEND; # default: append page

    #---- Get Page Id
    my $iPageIdTmp = $a2wPagePar->getParseId();

    #---- Dump
    printf fDumpFile ("================================= ");
    printf fDumpFile ("Page %06d", $iPageIdTmp);
    printf fDumpFile (" =================================\n");

    #---- Define temp Variables
    my $iTmp = 0;
    my $iFontLocalIdTmp = 0;
    my $a2wFontTmp = "";
    my $sFontEntryTmp = "";

    #---- Fetch first vector
    my $a2wVectorTmp = $a2wPagePar->getFirstVector();

    #---- Loop thru all the vectors
    my $sLogMsgTmp = "";
    while ( $a2wVectorTmp != 0 ){
        #---- Build String to be dumped with details
        $sLogMsgTmp = "  vec\@(" . sprintf( "% 6d, % 6d", $a2wVectorTmp->getXPos(), $a2wVectorTmp->getYPos() ) . "):"
                      . " Width=" . $a2wVectorTmp->getWidth()
                      . ", Height=" . $a2wVectorTmp->getHeight()
                      . ", Rotation=" . ( ( $a2wVectorTmp->getAngle() == 360 ) ? 0 : $a2wVectorTmp->getAngle() )
                      . ", Unique Colors Count=" . $a2wVectorTmp->getUniqueColorsCount()
                      ;
        printf fDumpFile ( $sLogMsgTmp . "\n" );

        #---- Get the next vector
        $a2wVectorTmp = $a2wPagePar->getNextVector();
    } # end-while

    return $iRetTmp;
}

#-----------------------------------------------------------------------
# FinalizePage for each page
#-----------------------------------------------------------------------
sub finalizePage(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "finalizePage()\n" );
    }
    return 0;
}

#-----------------------------------------------------------------------
# FinalizeDoc for each document
#-----------------------------------------------------------------------
sub finalizeDoc(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "finalizeDoc()\n" );
    }
    return 0;
}

#-----------------------------------------------------------------------
# Finalize once per process
#-----------------------------------------------------------------------
sub finalize(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "finalize()\n" );
    }
    close(fDumpFile);
    return 0;
}

__END__
