#-------------------------------------------------------------------------------
#  a2w::core::dm::Block.pm:
#  Data Mining Framework Block
#
#  Author  : AFP2web Team, Maas Holding GmbH
#
#  $V100   2014-05-08    Initial Release
#
#  $V101   2015-05-20    - Fixed minor bug in image positioning within block  (AFP-224)
#                        - Extended to dump block contents when log is set to debug
#
#  $V102   2015-09-28    Extended to group texts based on font (identifier) (AFP-297)
#
#  $V103   2015-10-15    Extended to pass in custom block info like HTML style (AFP-298)
#
#  $V104   2015-11-06    - Extended with "isParagraph", "isTable" and "isSkip" functions
#                        - Extended with "ContentHeight" attribute
#
#-------------------------------------------------------------------------------
package a2w::core::dm::Block;

#-----------------------------------------------------------------------
# Include required modules
#-----------------------------------------------------------------------
use a2w::TypeConstants;
use a2w::core::log::Logger;
use a2w::core::dm::Constants;

use a2w::core::dm::Anchor;
use a2w::core::dm::ContentProcessor;
use a2w::core::dm::ParagraphContentDef;
use a2w::core::dm::SkipContentDef;
use a2w::core::dm::TableContentDef;

use a2w::core::dm::ParagraphBlockFormatter;
use a2w::core::dm::SkipBlockFormatter;
use a2w::core::dm::TableBlockFormatter;

use a2w::core::visitor::Visitor;

#-----------------------------------------------------------------------
# Constructor
#-----------------------------------------------------------------------
sub new{
    my $proto = shift;
    my $class = ref( $proto ) || $proto;

    #---- Define boolean values
    $TRUE  = $a2w::TypeConstants::TRUE;     # TRUE  boolean value
    $FALSE = $a2w::TypeConstants::FALSE;    # FALSE boolean value

    my $this = {
        #---- Config attributes ----#
        # Identifier
          'Id' => ''

        # Start anchor
        , 'StartAnchor' => undef

        # End anchor
        , 'EndAnchor' => undef

        # Width of block
        , 'Width' => 0

        # Height of block
        , 'Height' => 0

        # Content definition
        , 'ContentDef' => undef

        # Id of next block in chain
        , 'Next' => undef

        # Quantifier - Block repetition quantity
        # Quantifier is optional
        # If defined, specifies repetition of block. Possible values are
        #
        # Quantifier Description
        #
        # 1          Block occurs only once (default)
        # ?          Block occurs zero or once
        # *          Block occurs zero or many
        # +          Block occurs minimum once and many (!!!???)
        # n          Block occurs exactly 'n' count
        , 'Quantifier' => '1'

        # Content Processor
        , 'ContentProcessor' => undef

        #---- Processing attributes ----#
        # Block formatter
        , 'Formatter' => undef

        # Reference to next block in chain
        , 'NextRef' => undef

        # List of objects belong to this block (array reference of objects)
        , 'ObjsList' => []

        # List of starting anchor objects of this block (array reference of objects)
        , 'StAncList' => []

        # List of ending anchor objects of this block (array reference of objects)
        , 'EdAncList' => []

        # Block starting X position (starting anchor's X position)
        , 'StartX' => 0

        # Block starting Y position (starting anchor's Y position)
        , 'StartY' => 0

        # Block starting Y position as on start page
        , 'BeginY' => 0

        # Block ending X position (starting anchor's X position + block width)
        , 'EndX' => 0

        # Block ending Y position (ending anchor's Y position)
        , 'EndY' => 0

        # Flag indicating whether start anchor found or not
        , 'StartFound' => $FALSE

        # Flag indicating whether end anchor found or not
        , 'EndFound' => $FALSE

        # Page id where block started
        , 'StartedOn' => 0

        # Flag indicating whether block is flushed or not
        , 'Flushed' => $FALSE

        # Block configuration reference (used to create repetitive blocks)
        , 'ConfigReference' => undef

        # Repeat count
        , 'RepeatCount' => 1

        # $V103 Begin
        , 'Info' => ''
        # $V103 End

        # $V104 Begin
        , 'ContentHeight' => 0
        # $V104 End
    };

    bless( $this, $class );

    #---- Evaluate unique id for this instance
    $this->{ 'UniqueId' } = "$this";

    #---- Get logger
    our $theLogger = a2w::core::log::Logger->getSingleton();
    our $bLog = $theLogger->isRegistered( __PACKAGE__ );

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "new()" );
    #}
    return $this;
}

#-----------------------------------------------------------------------
# Destructor
#-----------------------------------------------------------------------
sub DESTROY{
    my $this = shift;
}

#-----------------------------------------------------------------------
# Mutators
#-----------------------------------------------------------------------
sub setId{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setId()" );
    #}

    $this->{ 'Id' } = shift;
}

sub setStartAnchor{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setStartAnchor()" );
    #}

    $this->{ 'StartAnchor' } = shift;
}

sub setEndAnchor{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setEndAnchor()" );
    #}

    $this->{ 'EndAnchor' } = shift;
}

sub setWidth{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setWidth()" );
    #}

    $this->{ 'Width' } = shift;
}

sub setHeight{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setHeight()" );
    #}

    $this->{ 'Height' } = shift;
}

sub setContentDef{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setContentDef()" );
    #}

    $this->{ 'ContentDef' } = shift;
}

sub setNext{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setNext()" );
    #}

    $this->{ 'Next' } = shift;
}

sub setQuantifier{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setQuantifier()" );
    #}

    $this->{ 'Quantifier' } = shift;
}

sub setContentProcessor{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setContentProcessor()" );
    #}

    $this->{ 'ContentProcessor' } = shift;
}

sub setNextRef{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setNextRef()" );
    #}

    $this->{ 'NextRef' } = shift;
}

sub setStartX{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setStartX()" );
    #}

    $this->{ 'StartX' } = shift;
}

sub setStartY{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setStartY()" );
    #}

    $this->{ 'StartY' } = shift;
}

sub setEndX{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setEndX()" );
    #}

    $this->{ 'EndX' } = shift;
}

sub setEndY{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setEndY()" );
    #}

    $this->{ 'EndY' } = shift;
}

sub setStartedOn{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setStartedOn()" );
    #}

    $this->{ 'StartedOn' } = shift;
}

sub setFlushed{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setFlushed()" );
    #}

    $this->{ 'Flushed' } = shift;
}

sub setConfigReference{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setConfigReference()" );
    #}

    $this->{ 'ConfigReference' } = shift;
}

sub setRepeatCount{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setRepeatCount()" );
    #}

    $this->{ 'RepeatCount' } = shift;
}

# $V103 Begin
sub setInfo{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setInfo()" );
    #}

    $this->{ 'Info' } = shift;
}
# $V103 End

# $V104 Begin
sub setContentHeight{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setContentHeight()" );
    #}

    $this->{ 'ContentHeight' } = shift;
}
# $V104 End

#-----------------------------------------------------------------------
# Accessors
#-----------------------------------------------------------------------
sub getId{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getId()" );
    #}

    return $this->{ 'Id' };
}

sub getStartAnchor{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getStartAnchor()" );
    #}

    return $this->{ 'StartAnchor' };
}

sub getEndAnchor{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getEndAnchor()" );
    #}

    return $this->{ 'EndAnchor' };
}

sub getWidth{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getWidth()" );
    #}

    return $this->{ 'Width' };
}

sub getHeight{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getHeight()" );
    #}

    return $this->{ 'Height' };
}

sub getContentDef{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getContentDef()" );
    #}

    return $this->{ 'ContentDef' };
}

sub getNext{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getNext()" );
    #}

    return $this->{ 'Next' };
}

sub getQuantifier{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getQuantifier()" );
    #}

    return $this->{ 'Quantifier' };
}

sub getContentProcessor{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getContentProcessor()" );
    #}

    return $this->{ 'ContentProcessor' };
}

sub getNextRef{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getNextRef()" );
    #}

    return $this->{ 'NextRef' };
}

sub getObjsList{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getObjsList()" );
    #}

    return $this->{ 'ObjsList' };
}

sub getStartAnchorsList{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getStartAnchorsList()" );
    #}

    return $this->{ 'StAncList' };
}

sub getEndAnchorsList{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getEndAnchorsList()" );
    #}

    return $this->{ 'EdAncList' };
}

sub getStartX{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getStartX()" );
    #}

    return $this->{ 'StartX' };
}

sub getStartY{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getStartY()" );
    #}

    return $this->{ 'StartY' };
}

sub getBeginY{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getBeginY()" );
    #}

    return $this->{ 'BeginY' };
}

sub getEndX{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getEndX()" );
    #}

    return $this->{ 'EndX' };
}

sub getEndY{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getEndY()" );
    #}

    return $this->{ 'EndY' };
}

sub getFormatter{
    $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getFormatter()" );
    #}

    return $this->{ 'Formatter' };
}

sub hasStarted{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "hasStarted()" );
    #}

    return $this->{ 'StartFound' };
}

sub hasEnded{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "hasEnded()" );
    #}

    return $this->{ 'EndFound' };
}

sub getStartedOn{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getStartedOn()" );
    #}

    return $this->{ 'StartedOn' };
}

sub isFlushed{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "isFlushed()" );
    #}

    return $this->{ 'Flushed' };
}

sub getConfigReference{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getConfigReference()" );
    #}

    return $this->{ 'ConfigReference' };
}

sub getConfId{
    $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getConfId()" );
    #}

    return $this->{ 'ConfigReference' }{ 'Id' };
}

sub getUniqueId{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getUniqueId()" );
    #}

    return $this->{ 'UniqueId' };
}

sub getRepeatCount{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getRepeatCount()" );
    #}

    return $this->{ 'RepeatCount' };
}

# $V103 Begin
sub getInfo{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getInfo()" );
    #}

    return $this->{ 'Info' };
}
# $V103 End

# $V104 Begin
sub getContentHeight{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getContentHeight()" );
    #}

    #---- Evaluate if block is chained
    my $iContHtTmp = $this->{ 'ContentHeight' };
    if ( $this->isChained() == $TRUE ){
        #---- Get chained block
        $iContHtTmp += $this->{ 'NextRef' }->getContentHeight();
    }
    return $iContHtTmp;
}
# $V104 End

#-----------------------------------------------------------------------
# Workers
#-----------------------------------------------------------------------
#-----------------------------------------------------------------------
# updateBoundary
#-----------------------------------------------------------------------
sub updateBoundary{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "updateBoundary()" );
    #}

    #---- Get parameter
    #
    # 1. Kernel object
    # 2. POM object
    #
    my $a2wObjectPar = shift;
    my $pomObjectPar = shift;

    my $iXPosTmp = $pomObjectPar->{ $a2w::core::dm::Constants::AT_XPOS };
    my $iYPosTmp = $pomObjectPar->{ $a2w::core::dm::Constants::AT_YPOS };
    my $iTextHeightTmp = 0;
    if ( $pomObjectPar->{ $a2w::core::dm::Constants::AT_OBJTYPE } == $a2w::core::dm::Constants::OT_TEXT ){
	    $iTextHeightTmp = $pomObjectPar->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_FONTHT };
	    $iTextHeightTmp *= ( $pomObjectPar->{ $a2w::core::dm::Constants::AT_RESOLUTION } / 72 );
	    $iTextHeightTmp = int( $iTextHeightTmp + .5 * ( $iTextHeightTmp <=> 0 ) );
	    $iYPosTmp -= $iTextHeightTmp;
	}

    #---- Evaluate block top-left
    if (    $iXPosTmp < $this->{ 'StartX' }
         || $this->{ 'StartX' } == 0
       ){
        $this->{ 'StartX' } = $iXPosTmp;
    }

    if (    $iYPosTmp < $this->{ 'StartY' }
         || $this->{ 'StartY' } == 0
       ){
        $this->{ 'StartY' } = $iYPosTmp;
    }

    #---- Evaluate block bottom-right
    if (    $iXPosTmp > $this->{ 'EndX' }
         || $this->{ 'EndX' } == 0
       ){
        $this->{ 'EndX' } = $iXPosTmp;
    }

    $iYPosTmp += $iTextHeightTmp;
    if (    $iYPosTmp > $this->{ 'EndY' }
         || $this->{ 'EndY' } == 0
       ){
        $this->{ 'EndY' } = $iYPosTmp;
    }
}

#-----------------------------------------------------------------------
# _createAnchor
#-----------------------------------------------------------------------
sub _createAnchor{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "_createAnchor()" );
    #}

    #---- Get parameter
    #
    # 1. Anchor config hash reference
    #
    my $hrefAnchorPar = shift;

    #---- Create anchor
    my $objAnchorTmp = new a2w::core::dm::Anchor();

    #---- Fill in anchor details
    $objAnchorTmp->createAndAddConstraints( $hrefAnchorPar->{ 'Constraints' } );
    if ( $hrefAnchorPar->{ 'Skip' } == $TRUE ){
        $objAnchorTmp->setSkip( $hrefAnchorPar->{ 'Skip' } );
    }

    return $objAnchorTmp;
}

#-----------------------------------------------------------------------
# createAndSetStartAnchor
#-----------------------------------------------------------------------
sub createAndSetStartAnchor{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "createAndSetStartAnchor()" );
    #}

    #---- Get parameter
    #
    # 1. Anchor config hash reference
    #
    my $hrefAnchorPar = shift;

    #---- Create start anchor
    my $ancStartTmp = $this->_createAnchor( $hrefAnchorPar );

    $this->setStartAnchor( $ancStartTmp );
}

#-----------------------------------------------------------------------
# createAndSetEndAnchor
#-----------------------------------------------------------------------
sub createAndSetEndAnchor{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "createAndSetEndAnchor()" );
    #}

    #---- Get parameter
    #
    # 1. Anchor config hash reference
    #
    my $hrefAnchorPar = shift;

    #---- Create end anchor
    my $ancEndTmp = $this->_createAnchor( $hrefAnchorPar );

    $this->setEndAnchor( $ancEndTmp );
}

#-----------------------------------------------------------------------
# createAndSetContentDef
#-----------------------------------------------------------------------
sub createAndSetContentDef{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "createAndSetContentDef()" );
    #}

    #---- Get parameter
    #
    # 1. ContentDef config hash reference
    #
    my $hrefContentDefPar = shift;

    #---- Create content definition
    my $objContDefTmp = undef;
    my $sContDefTypeTmp = lc( $hrefContentDefPar->{ 'Type' } );
    if ( $sContDefTypeTmp eq "paragraph" ){
        $objContDefTmp = new a2w::core::dm::ParagraphContentDef();

        #---- Fill in content definition
        if ( lc( $hrefContentDefPar->{ 'Definition' }{ 'Group' } ) eq "true" ){
            $objContDefTmp->setGroup( $TRUE );
        }
        $objContDefTmp->setSeparator( $hrefContentDefPar->{ 'Definition' }{ 'Separator' } );
        # $V102 Begin
        if ( lc( $hrefContentDefPar->{ 'Definition' }{ 'GroupBy' } ) eq "font" ){
            $objContDefTmp->setGroupBy( "font" );
        }
        else {
            if ( $bLog == $TRUE
                 # Since line is default, it is not asserted on above 'if' condition, so 'line' must be skipped while warning about invalid value
                 && lc( $hrefContentDefPar->{ 'Definition' }{ 'GroupBy' } ) ne "line"
               ){
                $theLogger->logMessage( "Warning! Invalid GroupBy (" . $hrefContentDefPar->{ 'Definition' }{ 'GroupBy' } . "), valid values are line and font" );
            }
        }
        # $V102 End
    }
    elsif ( $sContDefTypeTmp eq "skip" ){
        $objContDefTmp = new a2w::core::dm::SkipContentDef();
    }
    elsif ( $sContDefTypeTmp eq "table" ){
        $objContDefTmp = new a2w::core::dm::TableContentDef();

        #---- Fill in content definition
        unless (    $hrefContentDefPar->{ 'Definition' }{ 'Header' } == undef
                 || lc( $hrefContentDefPar->{ 'Definition' }{ 'Header' } ) eq "false"
               ){
            $objContDefTmp->setHeader( $hrefContentDefPar->{ 'Definition' }{ 'Header' } );
        }
        $objContDefTmp->setColumns( $hrefContentDefPar->{ 'Definition' }{ 'Columns' } );
    }
    else {
        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "Warning! Invalid content definition type (" . $hrefContentDefPar->{ 'Type' } . "), valid values are Paragraph, Table and Skip" );
        }
        return;
    }

    #---- Create and add additional contents
    my @arrKeysTmp = sort keys( %{ $hrefContentDefPar->{ 'AddContent' } } );
    foreach my $sIdTmp ( @arrKeysTmp ){
        $objContDefTmp->createAndAddContent(   $sIdTmp
                                             , $hrefContentDefPar->{ 'AddContent' }{ $sIdTmp }
                                           );
    }

    $this->setContentDef( $objContDefTmp );
}

#-----------------------------------------------------------------------
# Create and set content processor
#-----------------------------------------------------------------------
sub createAndSetContentProcessor{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "createAndSetContentProcessor()" );
    #}

    #---- Get parameter
    #
    # 1. ContentProcessor config hash reference
    #
    my $hrefContentProcPar = shift;

    #---- Iterate through content processor and add appropriate objects ----#
    my $contProcessorTmp   = undef;
    my $sNameTmp           = '';
    my @arrContProcKeysTmp = sort keys( %{ $hrefContentProcPar } );
    foreach my $sKeyTmp ( @arrContProcKeysTmp ){
        if ( $hrefContentProcPar->{ $sKeyTmp } == undef ){
            next;
        }

        #---- Identify the content processor type (Find & Replace or Index)
        if ( $sKeyTmp =~ /^FNR\_(.*)/ ){
            $sNameTmp = $1;
            if ( $contProcessorTmp == undef ){
                $contProcessorTmp = new a2w::core::dm::ContentProcessor();
            }
            $contProcessorTmp->addFindAndReplace( $sNameTmp, $hrefContentProcPar->{ $sKeyTmp } );
        }
        elsif ( $sKeyTmp =~ /^IDX\_(.*)/ ){
            $sNameTmp = $1;
            if ( $contProcessorTmp == undef ){
                $contProcessorTmp = new a2w::core::dm::ContentProcessor();
            }
            $contProcessorTmp->addIndex( $sNameTmp, $hrefContentProcPar->{ $sKeyTmp } );
        }
        else {
            if ( $bLog == $TRUE ){
                $theLogger->logMessage( "Warning! Ignored unknown content processor object type (" . $sKeyTmp . ")" );
            }
        }
    }

    $this->setContentProcessor( $contProcessorTmp );
}

#-----------------------------------------------------------------------
# Add start anchor
#
# Adds given object to start anchor list
#
#-----------------------------------------------------------------------
sub addStartAnchor{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "addStartAnchor()" );
    #}

    #---- Get parameter
    #
    # 1. Object
    # 2. POM object wrapper
    #
    my $a2wObjectPar = shift;
    my $pomObjectPar = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logMessage(   "Object added to block (" . $this->getId() . ") start anchor list" );
    }

    #---- Evaluate existing object count
    my $iObjCountTmp = @{ $this->{ 'StAncList' } };

    #---- Add object at count index
    $this->{ 'StAncList' }[ $iObjCountTmp ] = {
          'A2WOBJ' => $a2wObjectPar
        , 'POMOBJ' => $pomObjectPar
    };

    #---- Collect index ----#
    if ( $this->{ 'StartAnchor' } != undef && $this->{ 'StartAnchor' }->isSkipped() == $FALSE ){
        #---- Get content processor
        my $contProcTmp = $this->getContentProcessor();

        if ( $contProcTmp != undef && $contProcTmp->doesAllIndexesCollected() == $FALSE ){
            #---- Collect index
            $contProcTmp->collectIndex( $a2wObjectPar, $pomObjectPar );
        }
    }
}

#-----------------------------------------------------------------------
# Add object
#
# Adds given object to list
#
#-----------------------------------------------------------------------
sub addObject{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "addObject()" );
    #}

    #---- Get parameter
    #
    # 1. Object
    # 2. POM object wrapper
    #
    my $a2wObjectPar = shift;
    my $pomObjectPar = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logMessage(   "Object added to block (" . $this->getId() . ")" );
    }

    #---- Evaluate existing object count
    my $iObjCountTmp = @{ $this->{ 'ObjsList' } };

    #---- Add object at count index
    $this->{ 'ObjsList' }[ $iObjCountTmp ] = {
          'A2WOBJ' => $a2wObjectPar
        , 'POMOBJ' => $pomObjectPar
    };

    #---- Collect index ----#
    #---- Get content processor
    my $contProcTmp = $this->getContentProcessor();

    if ( $contProcTmp != undef && $contProcTmp->doesAllIndexesCollected() == $FALSE ){
        #---- Collect index
        $contProcTmp->collectIndex( $a2wObjectPar, $pomObjectPar );
    }
}

#-----------------------------------------------------------------------
# Add end anchor
#
# Adds given object to end anchor list
#
#-----------------------------------------------------------------------
sub addEndAnchor{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "addEndAnchor()" );
    #}

    #---- Get parameter
    #
    # 1. Object
    # 2. POM object wrapper
    #
    my $a2wObjectPar = shift;
    my $pomObjectPar = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logMessage(   "Object added to block (" . $this->getId() . ") end anchor list" );
    }

    #---- Evaluate existing object count
    my $iObjCountTmp = @{ $this->{ 'EdAncList' } };

    #---- Add object at count index
    $this->{ 'EdAncList' }[ $iObjCountTmp ] = {
          'A2WOBJ' => $a2wObjectPar
        , 'POMOBJ' => $pomObjectPar
    };

    #---- Collect index ----#
    if ( $this->{ 'EndAnchor' } != undef && $this->{ 'EndAnchor' }->isSkipped() == $FALSE ){
        #---- Get content processor
        my $contProcTmp = $this->getContentProcessor();

        if ( $contProcTmp != undef && $contProcTmp->doesAllIndexesCollected() == $FALSE ){
            #---- Collect index
            $contProcTmp->collectIndex( $a2wObjectPar, $pomObjectPar );
        }
    }
}

#-----------------------------------------------------------------------
# doesStartedOn
#
# Returns true if block started on page (having given id)
#
#-----------------------------------------------------------------------
sub doesStartedOn{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "doesStartedOn()" );
    #}

    #---- Get parameter
    #
    # 1. Page id
    #
    my $iPageIdPar = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logMessage(   "Block " . $this->getId() . " started on " . $this->{ 'StartedOn' } . ", checking against " . $iPageIdPar );
    #}

    return ( $this->{ 'StartedOn' } == $iPageIdPar ) ? $TRUE : $FALSE;
}

#-----------------------------------------------------------------------
# getContentDefType
#
# Returns type of content definition
#
#-----------------------------------------------------------------------
sub getContentDefType{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getContentDefType()" );
    #}

    #---- Get content definition
    my $contDefTmp = $this->getContentDef();
    my $sRetTmp = "";
    if ( $contDefTmp != undef ){
        $sRetTmp = $contDefTmp->getType();
    }

    return $sRetTmp;
}

#-----------------------------------------------------------------------
# isRepetitive
#
# Returns true if block is repeatable
#
#-----------------------------------------------------------------------
sub isRepetitive{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "isRepetitive()" );
    #}

    my $bRetTmp = $FALSE;
    if ( $this->{ 'Quantifier' } ne "1" ){
        $bRetTmp = $TRUE;
    }

    return $bRetTmp;
}

#-----------------------------------------------------------------------
# hasMoreRepeatation
#
# Returns true if block configured to repeat more
#
#-----------------------------------------------------------------------
sub hasMoreRepeatation{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "hasMoreRepeatation()" );
    #}

    my $bRetTmp = $FALSE;
    my $sQuantifierTmp = $this->{ 'Quantifier' };

    # 1 - Block occurs only once (default)
    # n - Block occurs exactly 'n' count
    if ( $sQuantifierTmp =~ /^\d+$/ ){
        if ( $this->{ 'RepeatCount' } < $sQuantifierTmp ){
            $bRetTmp = $TRUE;
        }
    }
    # * - Block occurs zero or many
    # + - Block occurs minimum once and many (!!!???)
    elsif (    $sQuantifierTmp eq '*'
            || $sQuantifierTmp eq '+'
          ){
        $bRetTmp = $TRUE;
    }
    # ? - Block occurs zero or once
    elsif ( $sQuantifierTmp eq '?' ){
        # Already current instance exists, so nothing more to repeat
    }

    return $bRetTmp;
}

#-----------------------------------------------------------------------
# isChained
#
# Returns true if block is chained with next block
#
#-----------------------------------------------------------------------
sub isChained{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "isChained()" );
    #}

    return ( $this->{ 'Next' } ne "" ) ? $TRUE : $FALSE;
}

#-----------------------------------------------------------------------
# isFilled
#
# Asserts whether whole block content is collected or not
#
# Returns
# 1 in case of all content of block were collected
# 0 in case of none of block content is collected or still collecting block content
#
#-----------------------------------------------------------------------
sub isFilled{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "isFilled()" );
    #}

    return (    ( $this->{ 'StartFound' } == $TRUE )
             && ( $this->{ 'EndFound'   } == $TRUE )
           );
}

#-----------------------------------------------------------------------
# setAsFilled
#-----------------------------------------------------------------------
sub setAsFilled{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setAsFilled()" );
    #}

    $this->{ 'StartFound' } = $TRUE;
    $this->{ 'EndFound'   } = $TRUE;

    # $V104 Begin
    #---- Evaluate collected content height
    $this->{ 'ContentHeight' } += $this->{ 'EndY' } - $this->{ 'StartY' };
    # $V104 End
}

#-----------------------------------------------------------------------
# hasToBeSkipped
#
# Returns whether whole block content has to be skipped from presentation
#
# Returns
# 1 in case of block has to be skipped from presentation
# 0 in case of block has to be presented
#
#-----------------------------------------------------------------------
sub hasToBeSkipped{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "hasToBeSkipped()" );
    #}

    my $bRetTmp = $FALSE;

    #---- Get content def
    my $contDefTmp = $this->getContentDef();
    if (    $contDefTmp != undef
         && $contDefTmp->getType() eq "skip"
       ){
        $bRetTmp = $TRUE;
    }
    return $bRetTmp;
}

#-----------------------------------------------------------------------
# Reset block for next document processing
#
# Returns
# >=0 in case of success
#  <0 in case of error
#
#-----------------------------------------------------------------------
sub reset{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "reset()" );
    }

    $this->{ 'ObjsList' }   = [];
    $this->{ 'StAncList' }  = [];
    $this->{ 'EdAncList' }  = [];
    $this->{ 'StartX' }     = 0;
    $this->{ 'StartY' }     = 0;
    $this->{ 'BeginY' }     = 0;
    $this->{ 'EndX' }       = 0;
    $this->{ 'EndY' }       = 0;
    $this->{ 'StartFound' } = $FALSE;
    $this->{ 'EndFound' }   = $FALSE;
    $this->{ 'Flushed' }    = $FALSE;
    $this->{ 'StartedOn' }  = 0;
    $this->{ 'RepeatCount' }  = 1;
    $this->setId( $this->getConfId() );

    return 0;
}

#-----------------------------------------------------------------------
# Reset starting anchor (used when block content runs across pages)
#
# Returns
# >=0 in case of success
#  <0 in case of error
#
#-----------------------------------------------------------------------
sub resetStartAnchor{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "resetStartAnchor()" );
    }

    # $V104 Begin
    #---- Evaluate collected content height
    $this->{ 'ContentHeight' } += $this->{ 'EndY' } - $this->{ 'StartY' };
    # $V104 End

    $this->{ 'StartFound' } = $FALSE;
    $this->{ 'StartX' }     = 0;
    $this->{ 'StartY' }     = 0;
    $this->{ 'EndX' }       = 0;

    return 0;
}

#-----------------------------------------------------------------------
# fillIn
#
# Fill in block with configured values
#
# Returns
# >=0 in case of success
# < 0 in case of error
#
#-----------------------------------------------------------------------
sub fillIn{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "fillIn()" );
    #}

    #---- Get parameter
    #
    # 1. Block config reference
    #
    my $blkConfigRefPar = shift;

    #---- Fill in configured values in block
    $this->setConfigReference( $blkConfigRefPar );
    $this->setId( $blkConfigRefPar->{ 'Id' } );
    $this->createAndSetStartAnchor( $blkConfigRefPar->{ 'StartAnchor' } );
    $this->createAndSetEndAnchor( $blkConfigRefPar->{ 'EndAnchor' } );
    $this->setWidth( $blkConfigRefPar->{ 'Width' } );
    if ( defined $blkConfigRefPar->{ 'Height' } ){
        $this->setHeight( $blkConfigRefPar->{ 'Height' } );
    }
    # $V103 Begin
    if ( defined $blkConfigRefPar->{ 'Info' } ){
        $this->setInfo( $blkConfigRefPar->{ 'Info' } );
    }
    # $V103 End
    if ( $blkConfigRefPar->{ 'ContentDef' } == undef ){
        #---- Create default content definition of type paragraph
        $this->createAndSetContentDef( {   'Type' => 'Paragraph'
                                                  , 'Definition' => {
                                                          'Group' => $FALSE
                                                        , 'Separator' => ""
                                                    }
                                                }
                                              );
    }
    else {
        $this->createAndSetContentDef( $blkConfigRefPar->{ 'ContentDef' } );
    }
    $this->setNext( $blkConfigRefPar->{ 'Next' } );
    if ( defined $blkConfigRefPar->{ 'Quantifier' } ){
        $this->setQuantifier( $blkConfigRefPar->{ 'Quantifier' } );
    }
    if ( defined $blkConfigRefPar->{ 'ContentProcessor' } ){
        $this->createAndSetContentProcessor( $blkConfigRefPar->{ 'ContentProcessor' } );
    }

    return 0;
}

#-----------------------------------------------------------------------
# createRepetitiveBlock
#
# Creates repetitive block and fills in with configured values
#
# Returns
# Repetitive Block instance in case of success
# undefined in case of error
#
#-----------------------------------------------------------------------
sub createRepetitiveBlock{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "createRepetitiveBlock()" );
    #}

    #---- Create block
    my $blkRepetitiveTmp = new a2w::core::dm::Block();

    #---- Fill in block
    $blkRepetitiveTmp->fillIn( $this->getConfigReference() );
    $blkRepetitiveTmp->setRepeatCount( $this->getRepeatCount() + 1 );

    if ( $bLog == $TRUE ){
        $theLogger->logMessage( "Created repetitive block (" . $this->getId() . "-" . $blkRepetitiveTmp->getRepeatCount() . ")" );
    }

    return $blkRepetitiveTmp;
}

#-----------------------------------------------------------------------
# update
#
# Updates block
# - with actual id, if id has place holder from index value
#
# Returns
# >=0 in case of success
# < 0 in case of error
#
#-----------------------------------------------------------------------
sub update{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "update()" );
    #}

    #---- Assert whether block has id with place holder
    my $sIdTmp = $this->{ 'Id' };
    my $iSepIdxTmp = index( $sIdTmp, '' );
    if ( $iSepIdxTmp > 0 ){
        $sIdTmp = substr( $sIdTmp, $iSepIdxTmp );
    }
    my @arrIdxTmp = split( //, $sIdTmp );
    @arrIdxTmp = grep { $_ ne '' } @arrIdxTmp;
    foreach $sIdxTmp ( @arrIdxTmp ){
        #---- Get content processor
        my $sIdxValueTmp = "";
        my $contProcTmp = $this->getContentProcessor();
        if ( $contProcTmp != undef ){
		    #---- Get index with place holder name
		    my $idxPlaceHolderTmp = $contProcTmp->getIndex( $sIdxTmp );
		    if ( $idxPlaceHolderTmp != undef ){
		        $sIdxValueTmp = $idxPlaceHolderTmp->getResult();
		    }
        }

        #---- Update id with actual value
        my $sIdTmp = $this->{ 'Id' };
        $sIdTmp =~ s/(\Q$sIdxTmp\E)/$sIdxValueTmp/;
        $this->setId( $sIdTmp );
    }
    return 0;
}

#-----------------------------------------------------------------------
# write
#
# Write current block using formatter
#
#-----------------------------------------------------------------------
sub write{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "write(" . $this->getId() . ")" );
    }

    #---- Get parameter
    #
    # 1. Visitor
    #
    my $outVisitorPar = shift;

    #---- Set flushed flag
    $this->{ 'Flushed' } = $TRUE;

    #---- Write begin block
    my $iRetTmp = 0;
    my $bSkipWrapperTmp = $FALSE;
    my $contDefTmp = $this->getContentDef();
    my $hrefACTmp = $contDefTmp->getAddContent();
    my @arrACKeysTmp = sort keys( %{ $hrefACTmp } );
    if (    $contDefTmp->getType() eq "skip"
         && @arrACKeysTmp <= 0
       ){
        $bSkipWrapperTmp = $TRUE;
    }
    $iRetTmp = $outVisitorPar->beginBlock( $this, $bSkipWrapperTmp );
    if ( $iRetTmp < 0 ){ return $iRetTmp };

    #---- Write additional content to be added before block content
    foreach my $sIdTmp ( @arrACKeysTmp ){
        if ( lc( $hrefACTmp->{ $sIdTmp }->getInclude() ) eq "before" ){
            $outVisitorPar->writeRawContent( $hrefACTmp->{ $sIdTmp }->getContent() );
        }
    }

    #---- Write block content
    $iRetTmp = $this->writeContent( $outVisitorPar );
    if ( $iRetTmp < 0 ){ return $iRetTmp };

    #---- Write additional content to be added after block content
    foreach my $sIdTmp ( @arrACKeysTmp ){
        if ( lc( $hrefACTmp->{ $sIdTmp }->getInclude() ) eq "after" ){
            $outVisitorPar->writeRawContent( $hrefACTmp->{ $sIdTmp }->getContent() );
        }
    }

    #---- Write end block
    $iRetTmp = $outVisitorPar->endBlock( $this, $bSkipWrapperTmp );

    return $iRetTmp;
}

#-----------------------------------------------------------------------
# write block content
#
# Write block content using formatter
#
#-----------------------------------------------------------------------
sub writeContent{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeContent()" );
    }

    #---- Get parameter
    #
    # 1. Visitor
    #
    my $outVisitorPar = shift;

    # $V101 Begin
    if ( ( $theLogger->getLevel() & $a2w::core::log::Logger::LEVEL_DEBUG ) != 0 ){
        #---- Collect block contents ----#
        my @arrObjsListTmp = ();

        # Check and add start anchor objects
        my $ancStTmp = $this->getStartAnchor();
        if ( $ancStTmp != undef && $ancStTmp->isSkipped() == $FALSE ){
            push( @arrObjsListTmp, @{ $this->getStartAnchorsList() } );
        }
		
        # Add Block objects
        push( @arrObjsListTmp, @{ $this->getObjsList() } );
		
        # Check and add end anchor objects
        my $ancEdTmp = $this->getEndAnchor();
        if ( $ancEdTmp != undef && $ancEdTmp->isSkipped() == $FALSE ){
            push( @arrObjsListTmp, @{ $this->getEndAnchorsList() } );
        }

        # Dump block objects
        foreach my $objdmp ( @arrObjsListTmp ){
            if ( $objdmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJTYPE } == $a2w::core::dm::Constants::OT_LINE ){
                $theLogger->logMessage(   "Line@"
                                        . "(" . $objdmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_XPOS }
                                        . "," . $objdmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_YPOS }
                                        . ") W=" . $objdmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_LINE_WIDTH }
                                        . " L=" . $objdmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_LINE_LENGTH }
                                      );

            }
            elsif ( $objdmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJTYPE } == $a2w::core::dm::Constants::OT_TEXT ){
                $theLogger->logMessage(   "Text@"
                                        . "(" . $objdmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_XPOS }
                                        . "," . $objdmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_YPOS }
                                        . ")>" . $objdmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_VALUE }
                                        . "<"
                                      );
            }
            elsif ( $objdmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJTYPE } == $a2w::core::dm::Constants::OT_IMAGE ){
                $theLogger->logMessage(   "Image@"
                                        . "(" . $objdmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_XPOS }
                                        . "," . $objdmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_YPOS }
                                        . ")>" . $objdmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_IMAGE_NAME }
                                        . "<"
                                      );
            }
        }
    }
    # $V101 End

    #---- Check and create formatter
    $this->_createContentFormatter();
    my $fmtBlockTmp = $this->{ 'Formatter' };

    #---- Write block using formatter
    return $fmtBlockTmp->writeFormattedBlock( $this, $outVisitorPar );
}

#-----------------------------------------------------------------------
# Process content
#
# Process block content and do the following
# - Fetch indexes defined as specified content processor
# - Find and replace contents as specified content processor
# - Update block identifier with actual by replacing place holder
#
# Returns
# >=0 in case of success
# < 0 in case of error
#
#-----------------------------------------------------------------------
sub processContent{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "processContent()" );
    }

    #---- Get content processor ----#
    my $contProcTmp = $this->getContentProcessor();
    if (    $contProcTmp == undef
         || (    $contProcTmp->hasIndexes() == $FALSE
              && $contProcTmp->hasFNRs() == $FALSE
            )
       ){
        #---- No content processor, just return
        return 0;
    }

    #---- Get block objects ----#
    my @arrObjsListTmp = ();

    # Check and add start anchor objects
    my $ancStTmp = $blkCurrentPar->getStartAnchor();
    if ( $ancStTmp != undef && $ancStTmp->isSkipped() == $FALSE ){
        push( @arrObjsListTmp, @{ $blkCurrentPar->getStartAnchorsList() } );
    }
	
    # Add Block objects
    push( @arrObjsListTmp, @{ $blkCurrentPar->getObjsList() } );
	
    # Check and add end anchor objects
    my $ancEdTmp = $blkCurrentPar->getEndAnchor();
    if ( $ancEdTmp != undef && $ancEdTmp->isSkipped() == $FALSE ){
        push( @arrObjsListTmp, @{ $blkCurrentPar->getEndAnchorsList() } );
    }

    if ( @arrObjsListTmp <= 0 ){
        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "Warning! No object found on block to process" );
        }
        return -1;
    }

    #---- Iterate through the content and process them ----#
    my @arrIDXTmp = @{ $contProcTmp->getIndexList() };
    my @arrFNRTmp = @{ $contProcTmp->getFindAndReplaceList() };
    foreach my $objTmp ( @arrObjsListTmp ){
        #---- Process and fetch indexes ----#
        #---- Process find and replace ----#
    }

    return 0;
}

#-----------------------------------------------------------------------
# _createContentFormatter
#
# Creates content formatter
#
#-----------------------------------------------------------------------
sub _createContentFormatter{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "_createContentFormatter()" );
    }

    #---- Assert whether formatter is alreay created
    unless ( $this->{ 'Formatter' } eq undef ){
        return 0;
    }

    #---- Create formatter
    my $contDefTmp = $this->getContentDef();
    if ( lc( $contDefTmp->getType() ) eq "paragraph" ){
        $this->{ 'Formatter' } = new a2w::core::dm::ParagraphBlockFormatter();
    }
    elsif ( lc( $contDefTmp->getType() ) eq "skip" ){
        $this->{ 'Formatter' } = new a2w::core::dm::SkipBlockFormatter();
    }
    elsif ( lc( $contDefTmp->getType() ) eq "table" ){
        $this->{ 'Formatter' } = new a2w::core::dm::TableBlockFormatter();
    }
    else {
        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "  Warning! Unknown content type (" . $contDefTmp->getType() . ") to format the block " . $this->getId() );
        }
        return -1;
    }

    return 0;
}

#-----------------------------------------------------------------------
# isStartAnchor
#
# Asserts whether object is start anchor or not
#
# 1, if object is start anchor of a block
# 0, if object is not start anchor of a block
#
#-----------------------------------------------------------------------
sub isStartAnchor{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "isStartAnchor()" );
    #}

    #---- Parameter
    #
    # 1. POM page
    # 2. object
    # 3. POM object wrapper (optional)
    #
    my $pomPagePar   = shift;
    my $a2wObjectPar = shift;
    my $pomObjectPar = undef;
    if ( @_ > 0 ){
        $pomObjectPar = shift;
    }
    if (    $this->{ 'StartFound' } == $TRUE
         || $a2wObjectPar == undef
       ){
        return $FALSE;
    }

    #---- Fetch starting anchor
    my $ancStartTmp = $this->getStartAnchor();

    #---- Assert whether object match start anchor
    my $bStartAnchorTmp = $ancStartTmp->doesConstraintsMatch( $a2wObjectPar, $pomObjectPar );
    if ( $bStartAnchorTmp == $TRUE ){
        $this->{ 'StartX' } = $pomObjectPar->{ $a2w::core::dm::Constants::AT_XPOS };
        $this->{ 'StartY' } = $pomObjectPar->{ $a2w::core::dm::Constants::AT_YPOS };

        if ( $a2wObjectPar->_getType() eq "text" ){
            my $iTextHeightTmp = $pomObjectPar->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_FONTHT };
            $iTextHeightTmp *= ( $pomPagePar->getPageRes() / 72 );
            $iTextHeightTmp = int( $iTextHeightTmp + .5 * ( $iTextHeightTmp <=> 0 ) );
            $this->{ 'StartY' } -= $iTextHeightTmp;
        }

        if ( $this->{ 'BeginY' } <= 0 ){
            $this->{ 'BeginY' } = $this->{ 'StartY' };
        }
        $this->{ 'EndX' }   = $this->{ 'StartX' } + $this->{ 'Width' };
        $this->{ 'StartFound' } = $TRUE;

        if ( $bLog == $TRUE ){
            if ( $this->isRepetitive() == $TRUE ){
                $theLogger->logMessage(   "Object is starting anchor of block " . $this->getId() . "(" . $this->{ 'RepeatCount' } . ")" );
            }
            else {
                $theLogger->logMessage(   "Object is starting anchor of block " . $this->getId() );
            }
        }
    }

    return $bStartAnchorTmp;
}

#-----------------------------------------------------------------------
# isEndAnchor
#
# Asserts whether object is end anchor or not
#
# 1, if object is end anchor of a block
# 0, if object is not end anchor of a block
#
#-----------------------------------------------------------------------
sub isEndAnchor{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "isEndAnchor()" );
    #}

    #---- Parameter
    #
    # 1. POM page
    # 2. object
    # 3. POM object wrapper (optional)
    #
    my $pomPagePar   = shift;
    my $a2wObjectPar = shift;
    my $pomObjectPar = undef;
    if ( @_ > 0 ){
        $pomObjectPar = shift;
    }
    if (    $this->isFilled() == $TRUE
         || $this->{ 'StartFound' } == $FALSE
         || $a2wObjectPar == undef
       ){
        return $FALSE;
    }

    #---- Fetch ending anchor
    my $ancEndTmp = $this->getEndAnchor();

    #---- Assert whether object match end anchor
    my $bEndAnchorTmp = $ancEndTmp->doesConstraintsMatch( $a2wObjectPar, $pomObjectPar );
    if ( $bEndAnchorTmp == $TRUE ){
        $this->{ 'EndFound' } = $TRUE;
        if ( $bLog == $TRUE ){
            if ( $this->isRepetitive() == $TRUE ){
                $theLogger->logMessage(   "Object is ending anchor of block " . $this->getId() . "(" . $this->{ 'RepeatCount' } . ")" );
            }
            else {
                $theLogger->logMessage(   "Object is ending anchor of block " . $this->getId() );
            }
        }

        # $V104 Begin
        #---- Evaluate collected content height
        $this->{ 'ContentHeight' } += $this->{ 'EndY' } - $this->{ 'StartY' };
        # $V104 End
    }

    return $bEndAnchorTmp;
}

# $V101 Begin
#-----------------------------------------------------------------------
# doesObjectFallIn
#
# Asserts whether object fall in given block and returns following value
#
# 1, if object fall in given block defined region
# 0, if object does not fall in given block defined region
#
#-----------------------------------------------------------------------
sub doesObjectFallIn{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "doesObjectFallIn()" );
    #}

    #---- Parameter
    #
    # 1. POM page
    # 2. object
    # 3. POM object wrapper (optional)
    #
    my $pomPagePar   = shift;
    my $a2wObjectPar = shift;
    my $pomObjectPar = undef;
    if ( @_ > 0 ){
        $pomObjectPar = shift;
    }
    if ( $a2wObjectPar == undef ){ return $FALSE; }

    #---- Assert whether block starting found
    if (    $this->{ 'StartFound' } == $FALSE
         || $this->{ 'EndFound' } == $TRUE
       ){
        return $FALSE;
    }
    my $bObjFallInTmp = $FALSE;

    #---- Fetch object position
    my $iXPosTmp = $a2wObjectPar->getXPos();
    my $iYPosTmp = $a2wObjectPar->getYPos();
    if ( $pomObjectPar->{ $a2w::core::dm::Constants::AT_OBJTYPE } == $a2w::core::dm::Constants::OT_IMAGE ){
        $iXPosTmp = $pomObjectPar->{ $a2w::core::dm::Constants::AT_XPOS };
        $iYPosTmp = $pomObjectPar->{ $a2w::core::dm::Constants::AT_YPOS };
    }

    #---- Assert whether object fall beyond start anchor
    if (    $iXPosTmp >= $this->{ 'StartX' }
         && $iYPosTmp >= $this->{ 'StartY' }
         && $iXPosTmp <= $this->{ 'EndX' }
       ){
        $bObjFallInTmp = $TRUE;
        $this->{ 'EndY' } = $iYPosTmp;
        if ( $bLog == $TRUE ){
            $theLogger->logMessage(   "Object fall in block " . $this->getId() );
        }
    }

    return $bObjFallInTmp;
}
# $V101 End

# $V104 Begin
#-----------------------------------------------------------------------
# isParagraph
#
# Asserts whether block type is paragraph or not
#
# 1, if block is paragraph
# 0, if block is NOT paragraph
#
#-----------------------------------------------------------------------
sub isParagraph{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "isParagraph()" );
    #}

    #---- Check type
    if ( $this->{ 'ContentDef' } == undef ){ return $FALSE; }

    my $bRetTmp = $FALSE;
    my $sContDefTypeTmp = lc( $this->{ 'ContentDef' }->getType() );
    if ( $sContDefTypeTmp eq "paragraph" ){
        $bRetTmp = $TRUE;
    }

    return $bRetTmp;
}

#-----------------------------------------------------------------------
# isTable
#
# Asserts whether block type is table or not
#
# 1, if block is table
# 0, if block is NOT table
#
#-----------------------------------------------------------------------
sub isTable{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "isTable()" );
    #}

    #---- Check type
    if ( $this->{ 'ContentDef' } == undef ){ return $FALSE; }

    my $bRetTmp = $FALSE;
    my $sContDefTypeTmp = lc( $this->{ 'ContentDef' }->getType() );
    if ( $sContDefTypeTmp eq "table" ){
        $bRetTmp = $TRUE;
    }

    return $bRetTmp;
}

#-----------------------------------------------------------------------
# isSkip
#
# Asserts whether block type is skip or not
#
# 1, if block is skip
# 0, if block is NOT skip
#
#-----------------------------------------------------------------------
sub isSkip{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "isSkip()" );
    #}

    #---- Check type
    if ( $this->{ 'ContentDef' } == undef ){ return $FALSE; }

    my $bRetTmp = $FALSE;
    my $sContDefTypeTmp = lc( $this->{ 'ContentDef' }->getType() );
    if ( $sContDefTypeTmp eq "skip" ){
        $bRetTmp = $TRUE;
    }

    return $bRetTmp;
}
# $V104 End

#-----------------------------------------------------------------------
# Don't remove the following lines !!!
#-----------------------------------------------------------------------
1;
__END__
