#-------------------------------------------------------------------------------
#  a2w/core/dm/Constraints.pm
#
#  Collection of constraints (base for Anchor, Index and FindAndReplace
#  for common usage of constraints)
#
#  Author   : Panneer, AFP2web Team
#  Date     : 2014-02-14
#  Version  : 1.0.0
#
#  $V100   2014-02-14    Initial Release
#
#-------------------------------------------------------------------------------
package a2w::core::dm::Constraints;

#-----------------------------------------------------------------------
# Include required modules
#-----------------------------------------------------------------------
use a2w::TypeConstants;
use a2w::core::log::Logger;

use a2w::core::dm::Constants;
use a2w::core::dm::Constraint;

#-----------------------------------------------------------------------
# Constructor
#-----------------------------------------------------------------------
sub new{
    my $proto = shift;
    my $class = ref( $proto ) || $proto;

    my $this = {
        # Constraints count
          'ConstraintsCount' => 0
        # Array of Constraint IDs
        , 'ConstraintIDs'    => undef
        # Constraints collection
        , 'Constraints'      => undef
        # Result
        , 'Result'           => ''
    };

    bless( $this, $class );

    #---- Define boolean values
    $TRUE  = $a2w::TypeConstants::TRUE;     # TRUE  boolean value
    $FALSE = $a2w::TypeConstants::FALSE;    # FALSE boolean value

    #---- Get logger
    our $theLogger = a2w::core::log::Logger->getSingleton();
    our $bLog = $theLogger->isRegistered( __PACKAGE__ );

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "new()" );
    #}

    return $this;
}

#-----------------------------------------------------------------------
# Destructor
#-----------------------------------------------------------------------
sub DESTROY{
    my $this = shift;
}

#-----------------------------------------------------------------------
# Mutators
#-----------------------------------------------------------------------
sub setConstraints{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setConstraints()" );
    #}

    $this->{ 'Constraints' } = shift;

    #---- Evaluate and set constraints info
    my @arrConstraintsTmp = sort keys( %{ $this->{ 'Constraints' } } );
    $this->{ 'ConstraintCount' } = @arrConstraintsTmp;
    $this->{ 'ConstraintIDs' }   = \@arrConstraintsTmp;
}

sub setResult{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setResult()" );
    #}

    $this->{ 'Result' } = shift;

    my @arrConstraintsTmp = @{ $this->{ 'ConstraintIDs' } };
    my $consTmp = undef;
    for ( my $i = 0; $i < @arrConstraintsTmp; $i++ ){
        $consTmp = $this->{ 'Constraints' }{ @arrConstraintsTmp[ $i ] };

        #---- Assert for constraint
        $consTmp->setResult( $this->{ 'Result' } );
    }
}

#-----------------------------------------------------------------------
# Accessors
#-----------------------------------------------------------------------
sub getConstraints{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getConstraints()" );
    #}

    return $this->{ 'Constraints' };
}

sub getConstraintCount{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getConstraintCount()" );
    #}

    return $this->{ 'ConstraintCount' };
}

sub getConstraintIDs{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getConstraintIDs()" );
    #}

    return $this->{ 'ConstraintIDs' };
}

sub getResult{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getResult()" );
    #}

    return $this->{ 'Result' };
}

#-----------------------------------------------------------------------
# Workers
#-----------------------------------------------------------------------
sub addConstraint{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "addConstraint()" );
    }

    #---- Parameter
    #
    # 1. Constraint Id
    # 1. Constraint
    #
    my $sConIdPar   = shift;
    my $conToAddPar = shift;

    $this->{ 'ConstraintCount' } += 1;
    $this->{ 'Constraints' }{ $sConIdPar } = $conToAddPar;
    my @arrConstraintsTmp = sort keys( %{ $this->{ 'Constraints' } } );
    $this->{ 'ConstraintIDs' }  = \@arrConstraintsTmp;
}

sub createAndAddConstraints{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "createAndAddConstraints()" );
    }

    #---- Parameter
    #
    # 1. Constraints config hash reference
    #
    my $hrefConstraintsPar = shift;
    my @arrConstraintsTmp = sort keys( %{ $hrefConstraintsPar } );

    #---- Create constraint
    my $conCurrentTmp = undef;
    foreach my $sIdTmp ( @arrConstraintsTmp ){
        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "Constraint: Id=>" . $sIdTmp . "< ObjType=>" . $hrefConstraintsPar->{ $sIdTmp }{ 'OBJ_TYPE' } . "< Cond=>" . $hrefConstraintsPar->{ $sIdTmp }{ 'CONDITION' } . "<" );
        }

        #---- Create constraint and fill in details ---#
        if (    $conCurrentTmp == undef
             || $conCurrentTmp->getObjectType() != $hrefConstraintsPar->{ $sIdTmp }{ 'OBJ_TYPE' }
           ){
            $conCurrentTmp = new a2w::core::dm::Constraint();

            #---- Set object type
            $conCurrentTmp->setObjectType( $hrefConstraintsPar->{ $sIdTmp }{ 'OBJ_TYPE' } );

            #---- Add constraint
            $this->addConstraint( $sIdTmp, $conCurrentTmp );
        }

        #---- Add condition
        $conCurrentTmp->addCondition( $hrefConstraintsPar->{ $sIdTmp }{ 'CONDITION' } );
    }
}

sub doesConstraintsMatch{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "doesConstraintsMatch()" );
    }

    #---- Parameter
    #
    # 1. Object
    # 2. POM object wrapper (optional)
    #
    my $a2wObjectPar = shift;
    my $pomObjectPar = undef;
    if ( @_ > 0 ){
        $pomObjectPar = shift;
    }

    #---- Iterate through constraints and assert them against object ----#
    my $bRetTmp    = $TRUE;
    my @arrConstraintsTmp = @{ $this->{ 'ConstraintIDs' } };
    my $iConsCntTmp = @arrConstraintsTmp;
    my $consTmp = undef;
    for (   my $i = 0
          ; (    $bRetTmp == $TRUE
              && $i < $iConsCntTmp
            )
          ; $i++
        ){
        $consTmp = $this->{ 'Constraints' }{ @arrConstraintsTmp[ $i ] };

        #---- Assert for constraint
        $bRetTmp &&= $consTmp->assert( $a2wObjectPar, $pomObjectPar );
        if (    $bRetTmp == $TRUE
             && $this->{ 'Result' } eq ""
           ){
            $this->{ 'Result' } = $consTmp->getResult();
        }
    }

    return $bRetTmp;
}

#-----------------------------------------------------------------------
# Don't remove the following lines !!!
#-----------------------------------------------------------------------
1;
__END__

