#-------------------------------------------------------------------------------
#  a2w/core/dm/ContentParser.pm
#
#  Perl module to parse page content and to build a database out of page content
#
#  Author   : Panneer, AFP2web Team
#  Date     : 2014-02-14
#  Version  : 1.0.0
#
#  $V100   2014-02-14    Initial Release
#
#  $V101   2015-08-27    Fixed minor bug in evaluating output file name when page
#                        output is turned on
#
#-------------------------------------------------------------------------------
package a2w::core::dm::ContentParser;

#-----------------------------------------------------------------------
# Include required modules
#-----------------------------------------------------------------------
use a2w::TypeConstants;
use a2w::core::log::Logger;

use a2w::core::dm::Constants;
use a2w::core::dm::Constraint;
use a2w::core::dm::Database;

#-----------------------------------------------------------------------
# Constructor
#-----------------------------------------------------------------------
sub new{
    my $proto = shift;
    my $class = ref( $proto ) || $proto;

    my $this = {
        'POM' => undef    # Current page database reference
    };

    bless( $this, $class );

    #---- Define boolean values
    $TRUE  = $a2w::TypeConstants::TRUE;     # TRUE  boolean value
    $FALSE = $a2w::TypeConstants::FALSE;    # FALSE boolean value

    #---- Get logger
    our $theLogger = a2w::core::log::Logger->getSingleton();
    our $bLog = $theLogger->isRegistered( __PACKAGE__ );

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "new()" );
    #}

    return $this;
}

#-----------------------------------------------------------------------
# Destructor
#-----------------------------------------------------------------------
sub DESTROY{
    my $this = shift;
}

#-----------------------------------------------------------------------
# Parse (generic interface)
# Returns a database containing the page content
#-----------------------------------------------------------------------
sub parse{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "parse" );
    }

    require a2w::Page;    # Include page module

    #---- Get parameter of parse ( Par: Page instance[, Flag to include page resource contents] )
    my $a2wPagePar = shift;
    my $bIncludePageResPar = $FALSE;

    #---- Get include page resource contents flag, if exist
    if ( @_ ){
        $bIncludePageResPar = shift;
    }

    #---- Parse page content and building database
    my $iPageIdTmp = $a2wPagePar->getParseId();

    #---- Create database
    my $dbPageTmp = new a2w::core::dm::Database( $iPageIdTmp );

    #---- Fill in page details
    $dbPageTmp->setPageRes( $a2wPagePar->getResolution() );
    $dbPageTmp->setPageWidth( $a2wPagePar->getWidth() );
    $dbPageTmp->setPageHeight( $a2wPagePar->getHeight() );
    $dbPageTmp->setPageBackground( $a2wPagePar->getOutputFilename() );

    #---- Fill Database with page content
    #
    #---- Add Text Objects
    $this->_addTextObjs( $a2wPagePar, $dbPageTmp );

    #---- Add Line Objects
    $this->_addLineObjs( $a2wPagePar, $dbPageTmp );

    #---- Add Vector Objects
    $this->_addVectorObjs( $a2wPagePar, $dbPageTmp );

    #---- Add Image Objects
    $this->_addImageObjs( $a2wPagePar, $dbPageTmp );

    #---- Add Overlays
    if ( $bIncludePageResPar ){
        $this->_addOverlays( $a2wPagePar, $dbPageTmp );        
    }
    return $dbPageTmp;
}

#-----------------------------------------------------------------------
# Initialize page
#
# Returns page database (POM) in case of success, undef in case of error
#-----------------------------------------------------------------------
sub initializePage{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "initializePage" );
    }

    require a2w::Page;    # Include page module

    #---- Get parameter ( Par: Page instance )
    my $a2wPagePar = shift;

    #---- Parse page content and building database
    my $iPageIdTmp = $a2wPagePar->getParseId();

    #---- Create database
    $this->{ 'POM' } = new a2w::core::dm::Database( $iPageIdTmp );
    my $dbPageTmp = $this->{ 'POM' };

    #---- Fill in page details
    $dbPageTmp->setPageRes( $a2wPagePar->getResolution() );
    $dbPageTmp->setPageWidth( $a2wPagePar->getWidth() );
    $dbPageTmp->setPageHeight( $a2wPagePar->getHeight() );

    # $V101 Begin
    #return 0;
    return $dbPageTmp;
    # $V101 End
}

#-----------------------------------------------------------------------
# Add object
#
# Adds given object to active page database (POM)
#
# Returns 0 in case of success, <0 in case of error
#-----------------------------------------------------------------------
sub addObject{
    my $this = shift;

    #---- Get parameter
    #
    # 1. Object
    #
    my $a2wObjPar = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "addObject" );
    }

    my $dbPageTmp = $this->{ 'POM' };
    my $sObjTypeTmp = $a2wObjPar->_getType();
    if ( $sObjTypeTmp eq "text" ){
        #---- Add text to database
        $dbPageTmp->addObject(   $a2w::core::dm::Constants::OT_TEXT    # Object type
                               , $a2wObjPar->getXPos()                 # X Position
                               , $a2wObjPar->getYPos()                 # Y Position
                               , $a2wObjPar->getAngle()                # Angle
                               , $a2wObjPar->getColor()                # Color
                               , $a2wObjPar                            # Object reference
                             );
    }
    elsif ( $sObjTypeTmp eq "line" ){
        #---- Add line to database
        $dbPageTmp->addObject(   $a2w::core::dm::Constants::OT_LINE    # Object type
                               , $a2wObjPar->getXPos()                 # X Position
                               , $a2wObjPar->getYPos()                 # Y Position
                               #!!! Line does not have Angle property !!!
                               , 0                                     # Angle
                               , $a2wObjPar->getColor()                # Color
                               , $a2wObjPar                            # Object reference
                             );
    }
    elsif ( $sObjTypeTmp eq "image" ){
        #---- Add image to database
        $dbPageTmp->addObject(   $a2w::core::dm::Constants::OT_IMAGE   # Object type
                               , $a2wObjPar->getXPos()                 # X Position
                               , $a2wObjPar->getYPos()                 # Y Position
                               , $a2wObjPar->getAngle()                # Angle
                               #!!! Image does not have Color property !!!
                               , 0                                     # Color
                               , $a2wObjPar                            # Object reference
                             );
    }
    elsif ( $sObjTypeTmp eq "vector" ){
        #---- Add vector to database
        $dbPageTmp->addObject(   $a2w::core::dm::Constants::OT_VECTOR  # Object type
                               , $a2wObjPar->getXPos()                 # X Position
                               , $a2wObjPar->getYPos()                 # Y Position
                               , $a2wObjPar->getAngle()                # Angle
                               , $a2wObjPar->getColor()                # Color
                               , $a2wObjPar                            # Object reference
                             );
    }
    return 0;
}

#-----------------------------------------------------------------------
# Finalize page
#
# Returns page database (POM) in case of success, undef in case of error
#-----------------------------------------------------------------------
sub finalizePage{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "finalizePage" );
    }

    return $this->{ 'POM' };
}

#-----------------------------------------------------------------------
#
# Add the Text objects to the database
#
#-----------------------------------------------------------------------
sub _addTextObjs{
    require a2w::Text;    # Include text module

    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "_addTextObjs" );
    }

    #---- Get parameter of _addTextObjs ( Par: Page/Ovelay instance, Database instance )
    my $a2wContentObjectPar = shift;
    my $dbContentPar        = shift;

    #---- Fill with text contents
    #---- Fetch first text
    my $a2wCurTextTmp = $a2wContentObjectPar->getFirstText();

    #---- Loop thru all texts
    while ( $a2wCurTextTmp != 0 ){
        #---- Get text attributes
        my $iXPosTmp  = $a2wCurTextTmp->getXPos();  # Get X position
        my $iYPosTmp  = $a2wCurTextTmp->getYPos();  # Get Y position
        my $iAngleTmp = $a2wCurTextTmp->getAngle(); # Get angle
        my $iColorTmp = $a2wCurTextTmp->getColor(); # Get color

        #---- Add text to database
        $dbContentPar->addObject(   $a2w::core::dm::Constants::OT_TEXT    # Object type
                                  , $iXPosTmp                             # X Position
                                  , $iYPosTmp                             # Y Position
                                  , $iAngleTmp                            # Angle
                                  , $iColorTmp                            # Color
                                  , $a2wCurTextTmp                        # Object reference
                                );

        #---- Get the next text
        $a2wCurTextTmp = $a2wContentObjectPar->getNextText();
    } # end-while
}

#-----------------------------------------------------------------------
#
# Add the Line objects to the database
#
#-----------------------------------------------------------------------
sub _addLineObjs{
    require a2w::Line;    # Include line module

    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "_addLineObjs" );
    }

    #---- Get parameter of _addLineObjs ( Par: Page/Ovelay instance, Database instance )
    my $a2wContentObjectPar = shift;
    my $dbContentPar        = shift;

    #---- Fill with line contents
    #---- Fetch first line
    my $a2wCurLineTmp = $a2wContentObjectPar->getFirstLine();

    #---- Loop thru all lines
    while ( $a2wCurLineTmp != 0 ){
        #---- Get line attributes
        my $iXPosTmp  = $a2wCurLineTmp->getXPos(); # Get X position
        my $iYPosTmp  = $a2wCurLineTmp->getYPos(); # Get Y position
        #!!! Line does not have Angle property !!!
        my $iAngleTmp = 0;
        my $iColorTmp = $a2wCurLineTmp->getColor(); # Get color

        #---- Add line to database
        $dbContentPar->addObject(   $a2w::core::dm::Constants::OT_LINE    # Object type
                                  , $iXPosTmp                             # X Position
                                  , $iYPosTmp                             # Y Position
                                  , $iAngleTmp                            # Angle
                                  , $iColorTmp                            # Color
                                  , $a2wCurLineTmp                        # Object reference
                                );

        #---- Get the next line
        $a2wCurLineTmp = $a2wContentObjectPar->getNextLine();
    } # end-while
}

#-----------------------------------------------------------------------
#
# Add the Vector objects to the database
#
#-----------------------------------------------------------------------
sub _addVectorObjs{
    require a2w::Vector;    # Include vector module

    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "_addVectorObjs" );
    }

    #---- Get parameter of _addVectorObjs ( Par: Page/Ovelay instance, Database instance )
    my $a2wContentObjectPar = shift;
    my $dbContentPar        = shift;

    #---- Fill with vector contents
    #---- Fetch first vector
    my $a2wCurVectorTmp = $a2wContentObjectPar->getFirstVector();

    #---- Loop thru all vectors
    while ( $a2wCurVectorTmp != 0 ){
        #---- Get vector attributes
        my $iXPosTmp  = $a2wCurVectorTmp->getXPos();  # Get X position
        my $iYPosTmp  = $a2wCurVectorTmp->getYPos();  # Get Y position
        my $iAngleTmp = $a2wCurVectorTmp->getAngle(); # Get angle
        my $iColorTmp = $a2wCurVectorTmp->getColor(); # Get color

        #---- Add vector to database
        $dbContentPar->addObject(   $a2w::core::dm::Constants::OT_VECTOR    # Object type
                                  , $iXPosTmp                               # X Position
                                  , $iYPosTmp                               # Y Position
                                  , $iAngleTmp                              # Angle
                                  , $iColorTmp                              # Color
                                  , $a2wCurVectorTmp                        # Object reference
                                );

        #---- Get the next vector
        $a2wCurVectorTmp = $a2wContentObjectPar->getNextVector();
    } # end-while
}

#-----------------------------------------------------------------------
#
# Add the Image objects to the database
#
#-----------------------------------------------------------------------
sub _addImageObjs{
    require a2w::Image;    # Include image module

    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "_addImageObjs" );
    }

    #---- Get parameter of _addImageObjs ( Par: Page/Ovelay instance, Database instance )
    my $a2wContentObjectPar = shift;
    my $dbContentPar        = shift;

    #---- Fill with image contents
    #---- Fetch first image
    my $a2wCurImageTmp = $a2wContentObjectPar->getFirstImage();

    #---- Loop thru all images
    while ( $a2wCurImageTmp != 0 ){
        #---- Get image attributes
        my $iXPosTmp  = $a2wCurImageTmp->getXPos();        #---- Get X position
        my $iYPosTmp  = $a2wCurImageTmp->getYPos();        #---- Get Y position
        my $iAngleTmp = $a2wCurImageTmp->getAngle();       #---- Get angle
        #!!! Image does not have Color property !!!
        my $iColorTmp = 0;

        #---- Add image to database
        $dbContentPar->addObject(   $a2w::core::dm::Constants::OT_IMAGE    # Object type
                                  , $iXPosTmp                              # X Position
                                  , $iYPosTmp                              # Y Position
                                  , $iAngleTmp                             # Angle
                                  , $iColorTmp                             # Color
                                  , $a2wCurImageTmp                        # Object reference
                                );

        #---- Get the next image
        $a2wCurImageTmp = $a2wContentObjectPar->getNextImage();
    } # end-while
}

#-----------------------------------------------------------------------
#
# Add the Overlaysto the database
#
#-----------------------------------------------------------------------
sub _addOverlays{
    require a2w::Overlay;    # Include overlay module

    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "_addOverlays" );
    }

    #---- Get parameter of _addTextObjs ( Par: Page/Ovelay instance, Database instance )
    my $a2wContentObjectPar = shift;
    my $dbContentPar        = shift;
    
    require a2w::Overlay;    # Include overlay module

    #---- Fetch 1st Overlay
    my $a2wOlyTmp = $a2wContentObjectPar->getFirstOverlay();

    #---- Loop thru all the Overlays
    while ( $a2wOlyTmp != 0 ){
        #---- Fill database with overlay contents
        #
        #---- Fill with text contents
        $this->_addTextObjs( $a2wOlyTmp, $dbContentPar );

        #---- Fill with line contents
        $this->_addLineObjs( $a2wOlyTmp, $dbContentPar );

        #---- Add Vector Objects
        $this->_addVectorObjs( $a2wOlyTmp, $dbContentPar );

        #---- Add Image Objects
        $this->_addImageObjs( $a2wOlyTmp, $dbContentPar );

        #---- Get the next Overlay
        $a2wOlyTmp = $a2wContentObjectPar->getNextOverlay();
    }
}

#-----------------------------------------------------------------------
# Don't remove the following lines !!!
#-----------------------------------------------------------------------
1;
__END__

