#-------------------------------------------------------------------------------
#  a2w/core/dm/ContentProcessor.pm
#
#  Perl module to process block contents
#
#  Author   : Panneer, AFP2web Team
#  Date     : 2014-02-14
#  Version  : 1.0.0
#
#  $V100   2014-02-14    Initial Release
#
#-------------------------------------------------------------------------------
package a2w::core::dm::ContentProcessor;

#-----------------------------------------------------------------------
# Include required modules
#-----------------------------------------------------------------------
use a2w::TypeConstants;
use a2w::core::log::Logger;

use a2w::core::dm::Constants;
use a2w::core::dm::Constraint;
use a2w::core::dm::FindAndReplace;
use a2w::core::dm::Index;

#-----------------------------------------------------------------------
# Constructor
#-----------------------------------------------------------------------
sub new{
    my $proto = shift;
    my $class = ref( $proto ) || $proto;

    my $this = {
          'FNRList'      => []    # List of Find and Replace
        , 'IDXList'      => []    # List of indexes
    };

    bless( $this, $class );

    #---- Define boolean values
    $TRUE  = $a2w::TypeConstants::TRUE;     # TRUE  boolean value
    $FALSE = $a2w::TypeConstants::FALSE;    # FALSE boolean value

    #---- Get logger
    our $theLogger = a2w::core::log::Logger->getSingleton();
    our $bLog = $theLogger->isRegistered( __PACKAGE__ );

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "new()" );
    #}

    return $this;
}

#-----------------------------------------------------------------------
# Destructor
#-----------------------------------------------------------------------
sub DESTROY{
    my $this = shift;
}

#-----------------------------------------------------------------------
# Mutators
#-----------------------------------------------------------------------
sub setFindAndReplaceList{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setFindAndReplaceList" );
    #}

    $this->{ 'FNRList' } = shift;
}

sub setIndexList{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setIndexList" );
    #}

    $this->{ 'IDXList' } = shift;
}

#-----------------------------------------------------------------------
# Accessors
#-----------------------------------------------------------------------
sub getFindAndReplaceList{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getFindAndReplaceList" );
    #}

    return $this->{ 'FNRList' };
}

sub getIndexList{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getIndexList" );
    #}

    return $this->{ 'IDXList' };
}

#-----------------------------------------------------------------------
# Workers
#-----------------------------------------------------------------------
sub hasFNRs{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "hasFNRs" );
    #}

    my @arrFNRsTmp = @{ $this->{ 'FNRList' } };
    my $bRetTmp = $FALSE;
    if ( @arrFNRsTmp > 0 ){
        $bRetTmp = $TRUE;
    }

    return $bRetTmp;
}

sub hasIndexes{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "hasIndexes" );
    #}

    my @arrIndexesTmp = @{ $this->{ 'IDXList' } };
    my $bRetTmp = $FALSE;
    if ( @arrIndexesTmp > 0 ){
        $bRetTmp = $TRUE;
    }

    return $bRetTmp;
}

sub getFNRCount{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getFNRCount" );
    #}

    my @arrFNRsTmp = @{ $this->{ 'FNRList' } };
    my $iCountTmp = @arrFNRsTmp;
    return $iCountTmp;
}

sub getIndexesCount{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getIndexesCount" );
    #}

    my @arrIndexesTmp = @{ $this->{ 'IDXList' } };
    my $iCountTmp = @arrIndexesTmp;
    return $iCountTmp;
}

sub addFindAndReplace{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "addFindAndReplace" );
    #}

    #---- Get parameter
    #
    # 1. Name
    # 2. Find and Replace (config hash)
    #
    my $sNamePar     = shift;
    my $configFNRPar = shift;

    #---- Assert parameter
    if (    $sNamePar eq ""
         || $configFNRPar->{ 'Constraints' } == undef
         || $configFNRPar->{ 'Replace' } eq ""
       ){
        if ( $bLog == $TRUE ){
            if ( $sNamePar ne "" ){
                $theLogger->logMessage( "Warning! Ignored config Find and Replace (" . $sNamePar . ") due to missing Constraints or Replace" );
            }
            else{
                $theLogger->logMessage( "Warning! Ignored config Find and Replace due to missing Name" );
            }
        }
        return -1;
    }

    #---- Create find and replace
    my $fnrTmp = new a2w::core::dm::FindAndReplace();

    #---- Fill in details
    $fnrTmp->setName( $sNamePar );
    $fnrTmp->setReplace( $configFNRPar->{ 'Replace' } );
    $fnrTmp->createAndAddConstraints( $configFNRPar->{ 'Constraints' } );
    if ( defined $configFNRPar->{ 'Column' } ){
        $fnrTmp->setColumn( $configFNRPar->{ 'Column' } );
    }

    #---- Add FNR to list
    $this->{ 'FNRList' }[ @{ $this->{ 'FNRList' } } ] = $fnrTmp;

    if ( $bLog == $TRUE ){
        $theLogger->logMessage( "Added FNR (N=>" . $sNamePar . "< R=>" . $configFNRPar->{ 'Replace' } . "<)" );
    }

    return 0;
}

sub addIndex{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "addIndex" );
    #}

    #---- Get parameter
    #
    # 1. Name
    # 2. Index (config hash)
    #
    my $sNamePar     = shift;
    my $configIDXPar = shift;

    #---- Assert parameter
    if (    $sNamePar eq ""
         || not defined $configIDXPar->{ 'Constraints' }
         || not defined $configIDXPar->{ 'Value' }
       ){
        if ( $bLog == $TRUE ){
            if ( $sNamePar ne "" ){
                $theLogger->logMessage( "Warning! Ignored config Index (" . $sNamePar . ") due to missing Constraints or Value" );
            }
            else{
                $theLogger->logMessage( "Warning! Ignored config Index due to missing Name" );
            }
        }
        return -1;
    }

    #---- Create index
    my $idxTmp = new a2w::core::dm::Index();

    #---- Fill in details
    $idxTmp->setName( $sNamePar );
    $idxTmp->setValue( $configIDXPar->{ 'Value' } );
    $idxTmp->createAndAddConstraints( $configIDXPar->{ 'Constraints' } );

    #---- Add index to list
    $this->{ 'IDXList' }[ @{ $this->{ 'IDXList' } } ] = $idxTmp;

    if ( $bLog == $TRUE ){
        $theLogger->logMessage( "Added index (N=>" . $sNamePar . "< V=>" . $configIDXPar->{ 'Value' } . "<)" );
    }

    return 0;
}

sub collectIndex{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "collectIndex" );
    #}

    #---- Get parameter
    #
    # 1. Object
    # 2. POM object wrapper
    #
    my $a2wObjectPar = shift;
    my $pomObjectPar = shift;

    #---- Iterate through indexes and evaluate value ----#
    my $idxTmp  = undef;
    my $iRetTmp = -1;
    my @arrIndexesTmp = @{ $this->{ 'IDXList' } };

    #---- Filter out detected indexes
    @arrIndexesTmp = grep { $_->getResult() eq '' } @arrIndexesTmp;
    if ( @arrIndexesTmp <= 0 ){
        return 0;
    }

    for (   my $i = 0
          ; (    $iRetTmp < 0
              && $i < @arrIndexesTmp
            )
          ; $i++
        ){
        $idxTmp = @arrIndexesTmp[ $i ];
        $iRetTmp = $idxTmp->evaluate( $a2wObjectPar, $pomObjectPar );
    }

    return $iRetTmp;
}

sub doesAllIndexesCollected{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "doesAllIndexesCollected" );
    #}

    #---- Iterate through indexes and evaluate value ----#
    my @arrIndexesTmp = @{ $this->{ 'IDXList' } };

    #---- Filter out detected indexes
    @arrIndexesTmp = grep { $_->getResult() eq '' } @arrIndexesTmp;

    return ( @arrIndexesTmp == 0 ) ? $TRUE : $FALSE;
}

sub getIndex{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getIndex" );
    #}

    #---- Get parameter
    #
    # 1. Name
    #
    my $sNamePar = shift;

    #---- Fetch index matching given index name ----#
    my @arrIndexesTmp = @{ $this->{ 'IDXList' } };

    #---- Filter out needed index
    @arrIndexesTmp = grep { $_->getName() eq $sNamePar } @arrIndexesTmp;

    return ( @arrIndexesTmp > 0 ) ? @arrIndexesTmp[ 0 ] : undef;
}

#-----------------------------------------------------------------------
# Don't remove the following lines !!!
#-----------------------------------------------------------------------
1;
__END__

