#-------------------------------------------------------------------------------
#  a2w/core/dm/Dumper.pm
#
#  Perl module to dump page content from data mining database
#
#  Author   : Panneer, AFP2web Team
#  Date     : 2014-02-14
#  Version  : 1.0.0
#
#  $V100   2014-02-14    Initial Release
#
#-------------------------------------------------------------------------------
package a2w::core::dm::Dumper;

#-----------------------------------------------------------------------
# Include required modules
#-----------------------------------------------------------------------
use a2w::TypeConstants;
use a2w::core::log::Logger;

use a2w::core::dm::Constants;
use a2w::core::dm::Database;

#-----------------------------------------------------------------------
# Constructor
#-----------------------------------------------------------------------
sub new{
    my $proto = shift;
    my $class = ref( $proto ) || $proto;

    #---- Define boolean values
    $TRUE  = $a2w::TypeConstants::TRUE;     # TRUE  boolean value
    $FALSE = $a2w::TypeConstants::FALSE;    # FALSE boolean value

    my $this = {
          'bFileOpen' => $FALSE      # Flag indicating whether file is opened or not
        , 'Filename'  => ""          # File name
        , 'fHandle'   => undef       # File handle
        , 'Database'   => undef      # Database
    };

    bless( $this, $class );

    #---- Get logger
    our $theLogger = a2w::core::log::Logger->getSingleton();
    our $bLog = $theLogger->isRegistered( __PACKAGE__ );

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "new()" );
    #}
    return $this;
}

#-----------------------------------------------------------------------
# Destructor
#-----------------------------------------------------------------------
sub DESTROY{
    my $this = shift;

    #---- Close the file
    $this->_close();
}

#-----------------------------------------------------------------------
# Mutator
#-----------------------------------------------------------------------
#...

#-----------------------------------------------------------------------
# Accessor
#-----------------------------------------------------------------------
sub getFilename{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getFilename( " . $this->{ 'Filename' } . " )" );
    #}

    #---- Get filename
    return $this->{ 'Filename' };
}

#-----------------------------------------------------------------------
# Workers
#-----------------------------------------------------------------------
sub _open{
    my $this = shift;

    #---- Get parameter of open ( Par : Path )
    my $sPathPar = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "open( $sPathPar )" );
    }

    #---- Build filename
    #   0    1    2     3     4    5     6     7     8
    #($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime(time);
    my @arrCurrentTimeTmp = localtime( time() );
    @arrCurrentTimeTmp[ 5 ] += 1900;        #---- Year is not complete 4 digit, so adding 1900
    @arrCurrentTimeTmp[ 4 ]++;              #---- Day of month starts from 0, so adding 1
    my $sFilenameTmp = sprintf(   "dump_p%03d_%02d%02d%02d_%02d%02d%02d.txt"
								, $this->{ 'Database' }->getPageID()
                                , substr( @arrCurrentTimeTmp[ 5 ], 2, 2 ) # only the last 2 digits of year
                                , @arrCurrentTimeTmp[ 4 ]
                                , @arrCurrentTimeTmp[ 3 ]
                                , @arrCurrentTimeTmp[ 2 ]
                                , @arrCurrentTimeTmp[ 1 ]
                                , @arrCurrentTimeTmp[ 0 ]
                              );

    if ( $sPathPar ne "" ){
        $sFilenameTmp = "$sPathPar/$sFilenameTmp";
    }

    #---- Open file
    if ( open( $this->{ 'fHandle' }, ">$sFilenameTmp" ) ){
        $this->{ 'bFileOpen' } = $TRUE;
        $this->{ 'Filename' } = $sFilenameTmp;

        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "File (" . $sFilenameTmp . ") opened successfully to dump Data Mining DOM" );
        }
    }
    else {
        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "Error! Unable to open file (" . $sFilenameTmp . ") to dump Data Mining DOM." );
        }
    }
    return $this->{ 'bFileOpen' };
}

sub _close{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "close()" );
    }

    if ( $this->{ 'bFileOpen' } == $FALSE ){
        return -1;
    }

    #---- Open file
    close( $this->{ 'fHandle' } );
    $this->{ 'bFileOpen' } = $FALSE;

    return 0;
}

#-----------------------------------------------------------------------
# Dump DOM
#-----------------------------------------------------------------------
sub dumpDOM{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "dumpDOM()" );
    }

    #---- Parameter(s)
    #
    # 1. Output File Path
    # 2. Database instance
    # 3. Flag to retain object order
    #
    my $sOutputFilePathPar = shift;
    $this->{ 'Database' } = shift;
    my $bSortedPar = shift;

    if ( $bLog == $TRUE ){
        if ( $bSortedPar == $TRUE ){
            $theLogger->logMessage( "Dumping sorted DOM of page (" . $this->{ 'Database' }->getPageID() . ")" );
        }
        else {
            $theLogger->logMessage( "Dumping unsorted DOM of page (" . $this->{ 'Database' }->getPageID() . ")" );
        }
    }

    if ( $this->{ 'bFileOpen' } == $FALSE ){
		$this->_open($sOutputFilePathPar);
    }

    #---- Dump DOM header
    my $iObjCountTmp = $this->{ 'Database' }->getObjectCount();

    my $handleFileTmp = $this->{ 'fHandle' };
    
    printf $handleFileTmp ( "Dumping %s DOM of page %s\n", (($bSortedPar) ? "sorted" : "unsorted") ,$this->{ 'Database' }->getPageID() );
    printf $handleFileTmp ( "DOM Header:\n" );
    printf $handleFileTmp ( "  Objects Count=%s\n", $iObjCountTmp );
    printf $handleFileTmp ( "  Max XPos=%s\n", $this->{ 'Database' }->getMaxXPos() );
    printf $handleFileTmp ( "  Min XPos=%s\n", $this->{ 'Database' }->getMinXPos() );
    printf $handleFileTmp ( "  Max YPos=%s\n", $this->{ 'Database' }->getMaxYPos() );
    printf $handleFileTmp ( "  Min YPos=%s\n", $this->{ 'Database' }->getMinYPos() );
    printf $handleFileTmp ( "DOM Hash:\n" );

    #---- Dump DOM objects
    if ( $bSortedPar == $TRUE ){
        #---- Dump objects sorted by Y position
        my $hrefYPosTmp = $this->{ 'Database' }->getHashYPos();

        #---- Sort Y positions
        my @arrSortedYPositionsTmp = sort { $a <=> $b } keys( %{ $hrefYPosTmp } );
        my $hrefObjectTmp = undef;

        #---- Iterate through sorted Y positions
        foreach my $fYPosTmp ( @arrSortedYPositionsTmp ){
            #---- Sort X positions (i.e, sort multiple objects based on X positions that occur on same Y position aka line)
            my @arrSortedXPositionsTmp = sort { $hrefYPosTmp->{ $fYPosTmp }{ $a }{ $a2w::core::dm::Constants::AT_XPOS } <=> $hrefYPosTmp->{ $fYPosTmp }{ $b }{ $a2w::core::dm::Constants::AT_XPOS } } keys( %{ $hrefYPosTmp->{ $fYPosTmp } } );

            #---- Iterate through sorted X positions
            foreach $iOrderIdTmp ( @arrSortedXPositionsTmp ){
                #---- Get object reference
                $hrefObjectTmp = $hrefYPosTmp->{ $fYPosTmp }{ $iOrderIdTmp };

                #---- Dump object
                $this->_dumpObject( $hrefObjectTmp );
            }
        }
    }
    else {
        #---- Dump objects retaining the order
        my $hrefObjectTmp = undef;
        my $hrefDOMObjsTmp = $this->{ 'Database' }->getHashObj();
        for ( my $i = 1; $i <= $iObjCountTmp; $i++ ){
            #---- Get object reference
            $hrefObjectTmp = $hrefDOMObjsTmp->{ $i };

            #---- Dump object
            $this->_dumpObject( $hrefObjectTmp );
        }
    }
    return 0;
}

#-----------------------------------------------------------------------
# Dump object
#-----------------------------------------------------------------------
sub _dumpObject{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "_dumpObject()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Object hash
    #
    my $hrefObjectPar = shift;

    #---- Dump object
    my $sObjTypeTmp = "";
    my $sObjInfoStringTmp = "";
    if ( $hrefObjectPar->{ $a2w::core::dm::Constants::AT_OBJTYPE } == $a2w::core::dm::Constants::OT_TEXT ){
        $sObjTypeTmp = "Text";
        $sObjInfoStringTmp =   " Length=" . $hrefObjectPar->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_LENGTH }
                             . " Text=>" . $hrefObjectPar->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_VALUE } . "<";
    }
    elsif ( $hrefObjectPar->{ $a2w::core::dm::Constants::AT_OBJTYPE } == $a2w::core::dm::Constants::OT_LINE ){
        $sObjTypeTmp = "Line";
        $sObjInfoStringTmp =   " Width=" . $hrefObjectPar->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_LINE_WIDTH }
                             . " Length=" . $hrefObjectPar->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_LINE_LENGTH };
    }
    elsif ( $hrefObjectPar->{ $a2w::core::dm::Constants::AT_OBJTYPE } == $a2w::core::dm::Constants::OT_VECTOR ){
        $sObjTypeTmp = "Vector";
        $sObjInfoStringTmp =   " Width=" . $hrefObjectPar->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_VECTOR_WIDTH }
                             . " Height=" . $hrefObjectPar->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_VECTOR_HEIGHT };
    }
    elsif ( $hrefObjectPar->{ $a2w::core::dm::Constants::AT_OBJTYPE } == $a2w::core::dm::Constants::OT_IMAGE ){
        $sObjTypeTmp = "Image";
        $sObjInfoStringTmp =   " Name=" . $hrefObjectPar->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_IMAGE_NAME }
                             . " Width=" . $hrefObjectPar->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_IMAGE_WIDTH }
                             . " Height=" . $hrefObjectPar->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_IMAGE_HEIGHT };
    }
    my $handleFileTmp = $this->{ 'fHandle' };
    printf $handleFileTmp ( "  " . $sObjTypeTmp # Type
                            . "\@(" . $hrefObjectPar->{ $a2w::core::dm::Constants::AT_XPOS } . ", " . $hrefObjectPar->{ $a2w::core::dm::Constants::AT_YPOS } . ")" # Position
                            . " Rot=" . $hrefObjectPar->{ $a2w::core::dm::Constants::AT_ANGLE }   # Rotation
                            . $sObjInfoStringTmp # object specific info
                            . "\n"
                          );
    return 0;
}

#-----------------------------------------------------------------------
# Don't remove the following lines !!!
#-----------------------------------------------------------------------
1;
__END__

