#-------------------------------------------------------------------------------
#  a2w/core/dm/MiningEngine.pm
#
#  Perl module to process database and to generate reports
#
#  Author   : Panneer, AFP2web Team
#
#  $V100   2014-02-14    Initial Release
#
#  $V101   2015-05-20    Fixed minor bug in positioning starting anchor objects
#                        when block span over pages  (AFP-224)
#
#  $V102   2015-07-20    Fixed minor bug in evaluating output file name when page
#                        output is turned on
#
#  $V103   2015-10-15    Extended to pass in custom block info like HTML style (AFP-298)
#
#  $V104   2015-11-17    Extended to sort line objects based on X position along with sequence id  (AFP-320)
#
#-------------------------------------------------------------------------------
package a2w::core::dm::MiningEngine;

#-----------------------------------------------------------------------
# Include required modules
#-----------------------------------------------------------------------
use a2w::TypeConstants;
use a2w::core::log::Logger;

use a2w::core::dm::Constants;
use a2w::core::dm::Constraint;
use a2w::core::dm::Block;

use a2w::core::dm::Database;
use a2w::core::dm::ContentParser;

use a2w::core::dm::ConfigParser;
use a2w::core::dm::Dumper;

#-----------------------------------------------------------------------
# Constructor
#-----------------------------------------------------------------------
sub new{
    my $proto = shift;
    my $class = ref( $proto ) || $proto;

    my $this = {
          'Config'        => undef   # Data Mining Framework Configuration (blocks definition)
        , 'ContentParser' => undef   # Content parser
        , 'SpoolFilename' => undef   # Spool filename
        , 'OutputPath'    => undef   # Output path
        , 'OutputRes'     => undef   # Output resolution
        , 'DOM'           => undef   # Hash of POMs (where key is Page ID and value is POM)
        , 'DocBlockIDs'   => undef   # List of current document processable block ids
        , 'DocBlocks'     => undef   # List of current document processable blocks
        , 'PageBlocks'    => undef   # Page contained (both not configured and configured) blocks
        , 'PreDefAC'      => undef   # Header/Body/Footer additional contents of predefined blocks
        , 'Visitor'       => undef   # Visitor to render content
        , 'Title'         => undef   # Title of document
        , 'a2wDocument'   => undef   # AFP2web document reference
        , 'DocumentCount' => 0       # Document count
        , 'PageCount'     => 0       # Document page count
    };

    bless( $this, $class );

    #---- Define boolean values
    $TRUE  = $a2w::TypeConstants::TRUE;     # TRUE  boolean value
    $FALSE = $a2w::TypeConstants::FALSE;    # FALSE boolean value

    #---- Get logger
    our $theLogger = a2w::core::log::Logger->getSingleton();
    our $bLog = $theLogger->isRegistered( __PACKAGE__ );

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "new()" );
    #}

    #---- Initialize
    $this->initialize();

    return $this;
}

#-----------------------------------------------------------------------
# Destructor
#-----------------------------------------------------------------------
sub DESTROY{
    my $this = shift;

    #---- Finalize
    $this->finalize();
}

#-----------------------------------------------------------------------
# Mutators
#-----------------------------------------------------------------------
sub setSpoolFilename{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setSpoolFilename" );
    #}

    $this->{ 'SpoolFilename' } = shift;
}

sub setOutputPath{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setOutputPath" );
    #}

    $this->{ 'OutputPath' } = shift;
}

sub setOutputRes{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setOutputRes" );
    #}

    $this->{ 'OutputRes' } = shift;
}

#-----------------------------------------------------------------------
# Accessors
#-----------------------------------------------------------------------
sub getSpoolFilename{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getSpoolFilename" );
    #}

    return $this->{ 'SpoolFilename' };
}

sub getOutputPath{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getOutputPath" );
    #}

    return $this->{ 'OutputPath' };
}

sub getOutputRes{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getOutputRes" );
    #}

    return $this->{ 'OutputRes' };
}

#-----------------------------------------------------------------------
# Workers
#-----------------------------------------------------------------------
#-----------------------------------------------------------------------
# Initialize
#-----------------------------------------------------------------------
sub initialize{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "initialize" );
    }

    #---- Initialize attributes
    $this->{ 'Config' }        = undef;
    $this->{ 'ContentParser' } = new a2w::core::dm::ContentParser();
    $this->{ 'SpoolFilename' } = undef;
    $this->{ 'OutputPath' }    = undef;
    $this->{ 'OutputRes' }     = undef;
    $this->{ 'DOM' }           = undef;
    $this->{ 'DocBlockIDs' }   = undef;
    $this->{ 'DocBlocks' }     = undef;
    $this->{ 'PageBlocks' }    = undef;
    $this->{ 'PreDefAC' }      = undef;
    $this->{ 'Visitor' }       = undef;
    $this->{ 'Title' }         = undef;
    $this->{ 'a2wDocument' }   = undef;
    $this->{ 'PageCount' }     = 0;

    return 0;
}

#-----------------------------------------------------------------------
# Finalize
#-----------------------------------------------------------------------
sub finalize{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "finalize" );
    }

    #---- Cleanup
    $this->{ 'Config' }        = undef;
    $this->{ 'ContentParser' } = undef;
    $this->{ 'SpoolFilename' } = undef;
    $this->{ 'OutputPath' }    = undef;
    $this->{ 'OutputRes' }     = undef;
    $this->{ 'DOM' }           = undef;
    $this->{ 'DocBlockIDs' }   = undef;
    $this->{ 'DocBlocks' }     = undef;
    $this->{ 'PageBlocks' }    = undef;
    $this->{ 'PreDefAC' }      = undef;
    $this->{ 'Visitor' }       = undef;
    $this->{ 'Title' }         = undef;
    $this->{ 'a2wDocument' }   = undef;
    $this->{ 'PageCount' }     = 0;

    return 0;
}

#-----------------------------------------------------------------------
# Initialize document
#
# Parameter
# 1. Visitor module name (concrete visitor module name)
# 2. Document instance (of type a2w::Document)
# 3. Title
# 4. Page output, TRUE means each page as one output, FALSE means each document as one output (default is FALSE)
#
# Returns
# >=0 in case of success
# < 0 in case of error
# 
#-----------------------------------------------------------------------
sub initializeDoc{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "initializeDoc" );
    }

    #---- Get parameter
    #
    # 1. Visitor module name
    # 2. Document instance (of type a2w::Document)
    # 3. Title
    # 4. Page output
    #
    my $sVisitorModNamePar = shift;
    my $a2wDocumentPar = shift;
    my $sTitlePar = shift;
    my $bPageOutputPar = $FALSE;
    if ( @_ > 0 ){
        $bPageOutputPar = shift;
    }

    #---- Set process factors
    $this->{ 'Title' }          = $sTitlePar;
    $this->{ 'a2wDocument' }    = $a2wDocumentPar;
    $this->{ 'PageCount' }      = 0;
    $this->{ 'DocumentCount' } += 1;

    #---- Create visitor
    $this->{ 'Visitor' } = $sVisitorModNamePar->new();

    #---- Fill in visitor with processing info
    $this->{ 'Visitor' }->setUnitBase( "pixel" );
    $this->{ 'Visitor' }->setOutputRes( $this->{ 'OutputRes' } );
    $this->{ 'Visitor' }->setDocumentId( $this->{ 'DocumentCount' } );
    if ( $bPageOutputPar == $TRUE ){
        $this->{ 'Visitor' }->setPageOutput( $TRUE );
    }

    #---- Evaluate output simple file name
    my $sOutputFilenameTmp = $a2wDocumentPar->getOutputFilename();
    my $sSimpleNameTmp = "";
    if ( $sOutputFilenameTmp =~ /(.*)\..{3,4}$/ ){
        $sSimpleNameTmp = $1;
    }

    #---- Initialize visitor
    $this->{ 'Visitor' }->initialize( # Output path
                                        $this->{ 'OutputPath' }
                                      # Simple name
                                      , $sSimpleNameTmp
                                    );

    return 0;
}

#-----------------------------------------------------------------------
# Finalize document
#-----------------------------------------------------------------------
sub finalizeDoc{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "finalizeDoc" );
    }

    # Get A2W_DMF_PAGE_BODY predefined block's additional contents included
    # at after position
    #
    my $sAfterBodyAddContTmp = $this->{ 'PreDefAC' }{ 'BdyAfter' };

    # Get A2W_DMF_PAGE_FOOTER predefined block's additional contents
    my $sFtContTmp = $this->{ 'PreDefAC' }{ 'Footer' };

    #---- Finalize visitor
    $this->{ 'Visitor' }->finalize(   $sAfterBodyAddContTmp
                                    , $sFtContTmp
                                  );

    #---- Cleanup ----#
    my @arrBlkListTmp  = @{ $this->{ 'DocBlocks' } };
    foreach my $blkTmp ( @arrBlkListTmp ){
        if ( $blkTmp == undef ){
            next;
        }
        $blkTmp->reset();
    }
    $this->{ 'DOM' }           = undef;
    $this->{ 'DocBlockIDs' }   = undef;
    $this->{ 'DocBlocks' }     = undef;
    $this->{ 'PageBlocks' }    = undef;
    $this->{ 'Visitor' }       = undef;
    $this->{ 'PageCount' }     = 0;

    return 0;
}

#-----------------------------------------------------------------------
# Load config
#-----------------------------------------------------------------------
sub loadConfig{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "loadConfig" );
    }

    #---- Create config parser
    my $parserConfigTmp = new a2w::core::dm::ConfigParser();

    #---- Load config from hash
    $this->{ 'Config' } = $parserConfigTmp->parseHash( @_ );

    #---- Cleanup
    $parserConfigTmp = undef;

    #---- Resolve next chain links
    my $iRetTmp = $this->{ 'Config' }->resolveNextChain();

    #---- Process predefined blocks ----#
    $this->{ 'PreDefAC' } = $this->_getAdditionalContentsOfPredefinedBlocks();

    return $iRetTmp;
}

#-----------------------------------------------------------------------
# Process page
#
# Process page objects and add them to appropriate block they belong to
#
# Parameters:
# 1. Page Instance
# 2. Process overlays for AFP page type (optional)
#
# Returns 0 in case of success, <0 in case of error
#-----------------------------------------------------------------------
sub processPage{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "processPage" );
    }

    #---- Increment page count
    $this->{ 'PageCount' }++;

    #---- Parse page and create POM
    my $dmContentParserTmp = $this->{ 'ContentParser' };
    my $dmPOMTmp = $dmContentParserTmp->parse( @_ );

    # $V102 Begin
    if ( $_[ 0 ] != undef ){
        my $sPageFilenameTmp = $_[ 0 ]->getOutputFilename();
        if ( $sPageFilenameTmp ne "" ){
            $dmPOMTmp->setPageFilename( $sPageFilenameTmp );
        }
    }
    # $V102 End

    #---- Store POM on DOM
    my $iPageIdTmp = $dmPOMTmp->getPageID();
    $this->{ 'DOM' }{ $iPageIdTmp } = $dmPOMTmp;

    #---- Process POM objects and add them to appropriate blocks ----#
    my $iObjCountTmp = $dmPOMTmp->getObjectCount();

    #---- Process POM ----#
    return $this->_processPOM( $dmPOMTmp, $TRUE );
}

#-----------------------------------------------------------------------
# Initialize page
#
# Returns 0 in case of success, <0 in case of error
#-----------------------------------------------------------------------
sub initializePage{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "initializePage" );
    }

    require a2w::Page;    # Include page module

    #---- Get parameter ( Par: Page instance )
    my $a2wPagePar = shift;

    #---- Increment page count
    $this->{ 'PageCount' }++;

    #---- Initiate page parse
    my $dmContentParserTmp = $this->{ 'ContentParser' };
    # $V102 Begin
    #$dmContentParserTmp->initializePage( $a2wPagePar );
    my $dmPOMTmp = $dmContentParserTmp->initializePage( $a2wPagePar );

    if ( $a2wPagePar != undef ){
        my $sPageFilenameTmp = $a2wPagePar->getOutputFilename();
        if ( $sPageFilenameTmp ne "" ){
            $dmPOMTmp->setPageFilename( $sPageFilenameTmp );
        }
    }
    # $V102 End

    return 0;
}

#-----------------------------------------------------------------------
# Add object
#
# Adds given object to active page database (POM)
#
# Returns 0 in case of success, <0 in case of error
#-----------------------------------------------------------------------
sub addObject{
    my $this = shift;

    #---- Get parameter
    #
    # 1. Object
    #
    my $a2wObjPar = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "addObject" );
    }

    #---- Add object to page
    my $dmContentParserTmp = $this->{ 'ContentParser' };
    $dmContentParserTmp->addObject( $a2wObjPar );
    return 0;
}

#-----------------------------------------------------------------------
# Finalize page
#
# Returns 0 in case of success, <0 in case of error
#-----------------------------------------------------------------------
sub finalizePage{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "finalizePage" );
    }

    #---- Finalize page parse
    my $dmContentParserTmp = $this->{ 'ContentParser' };
    my $dmPOMTmp = $dmContentParserTmp->finalizePage();

    #---- Store POM on DOM
    my $iPageIdTmp = $dmPOMTmp->getPageID();
    $this->{ 'DOM' }{ $iPageIdTmp } = $dmPOMTmp;

    #---- Process POM ----#
    return $this->_processPOM( $dmPOMTmp );
}

# $V103 Begin
#-----------------------------------------------------------------------
# Add document possible block
#-----------------------------------------------------------------------
sub addDocumentBlock{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "addDocumentBlock" );
    }

    #---- Get parameter
    #
    # 1. Block id
    # 2. Block info
    #
    my $sBlockIdPar   = shift;
    my $sBlockInfoPar = shift;

    #---- Add block id to list
    $this->{ 'DocBlockIDs' }[ @{ $this->{ 'DocBlockIDs' } } ] = $sBlockIdPar;

    #---- Get block from configured list
    my $confDMFTmp = $this->{ 'Config' };
    my $blkAppliedTmp = $confDMFTmp->getBlock( $sBlockIdPar );

    #---- Assert and add block to document list
    if ( $blkAppliedTmp != undef ){
        $this->{ 'DocBlocks' }[ @{ $this->{ 'DocBlocks' } } ] = $blkAppliedTmp;

        #---- Set block info
        $blkAppliedTmp->setInfo( $sBlockInfoPar );

        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "Block (" . $sBlockIdPar . ") added, total blocks on list is " . @{ $this->{ 'DocBlocks' } } );
        }
    }
    else {
        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "Error! Block (" . $sBlockIdPar . ") not defined in block definition" );
            return ( -1, "Error! Block (" . $sBlockIdPar . ") not defined in block definition" );
        }
    }

    return 0;
}

#-----------------------------------------------------------------------
# Add repetitive document possible block (used for repetitive blocks adding)
#-----------------------------------------------------------------------
sub addRepetitiveDocumentBlock{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "addRepetitiveDocumentBlock" );
    }

    #---- Get parameter
    #
    # 1. Block reference
    #
    my $blkDocPar = shift;

    #---- Add block id to list
    $this->{ 'DocBlockIDs' }[ @{ $this->{ 'DocBlockIDs' } } ] = $blkDocPar->getId();

    #---- Add block to list
    $this->{ 'DocBlocks' }[ @{ $this->{ 'DocBlocks' } } ] = $blkDocPar;
    if ( $bLog == $TRUE ){
        $theLogger->logMessage( "Repetitive block (" . $blkDocPar->getId() . ") added, total blocks on list is " . @{ $this->{ 'DocBlocks' } } );
    }
}
# $V103 End

#-----------------------------------------------------------------------
# Write page
#-----------------------------------------------------------------------
sub writePage{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writePage" );
    }

    #---- Get parameter
    #
    # 1. Page Id
    #
    my $iPageIdPar = shift;

    #---- Get page blocks
    my $arefBlocksTmp = $this->{ 'PageBlocks' }{ $iPageIdPar };
    my @arrBlkListTmp = @{ $arefBlocksTmp };

    #---- Filter out page blocks ----#
    #---- Filter blocks that are started on current page
    #@arrBlkListTmp = grep { $_->doesStartedOn( $iPageIdPar ) == $TRUE } @arrBlkListTmp;
    #if ( $bLog == $TRUE ){
    #    my $sBlkListTmp = "";
    #    foreach my $blkTmp ( @arrBlkListTmp ){
    #        $sBlkListTmp .= " " . $blkTmp->getId();
    #    }
    #    $theLogger->logMessage( "Page $iPageIdPar blocks:$sBlkListTmp" );
    #}

    #---- Filter blocks that are filled
    @arrBlkListTmp = grep { $_->isFilled() == $TRUE } @arrBlkListTmp;
    #if ( $bLog == $TRUE ){
    #    my $sBlkListTmp = "";
    #    foreach my $blkTmp ( @arrBlkListTmp ){
    #        $sBlkListTmp .= " " . $blkTmp->getId();
    #    }
    #    $theLogger->logMessage( "Page $iPageIdPar filled blocks:$sBlkListTmp" );
    #}

    #---- Filter blocks that are chained
    my $hrefChainedIDsTmp = $this->{ 'Config' }->getChainedIDs();
    @arrBlkListTmp = grep { $hrefChainedIDsTmp->{ $_->getId() } != $TRUE } @arrBlkListTmp;
    #if ( $bLog == $TRUE ){
    #    my $sBlkListTmp = "";
    #    foreach my $blkTmp ( @arrBlkListTmp ){
    #        $sBlkListTmp .= " " . $blkTmp->getId();
    #    }
    #    $theLogger->logMessage( "Page $iPageIdPar filled blocks:$sBlkListTmp" );
    #}

    #---- Filter blocks that are not flushed already
    @arrBlkListTmp = grep { $_->isFlushed() == $FALSE } @arrBlkListTmp;
    #if ( $bLog == $TRUE ){
    #    my $sBlkListTmp = "";
    #    foreach my $blkTmp ( @arrBlkListTmp ){
    #        $sBlkListTmp .= " " . $blkTmp->getId();
    #    }
    #    $theLogger->logMessage( "Page $iPageIdPar not flushed blocks:$sBlkListTmp" );
    #}

    #---- Get block additional contents for header/body/footer
    my $hrefAddContentsTmp = $this->_getAdditionalContents( \@arrBlkListTmp );

    #---- Get page POM ----#
    my $dbPagePOMTmp = $this->{ 'DOM' }{ $iPageIdPar };

    # Get A2W_DMF_PAGE_BODY predefined block's additional contents included
    # at before/after positions
    #
    my $sBeforeBodyAddContTmp = $this->{ 'PreDefAC' }{ 'BdyBefore' };
    my $sAfterBodyAddContTmp  = $this->{ 'PreDefAC' }{ 'BdyAfter' };

    #---- Initialize page
    my $sHdrContTmp = $this->{ 'PreDefAC' }{ 'Header' };
    if ( $hrefAddContentsTmp != undef ){
        $sHdrContTmp .= $hrefAddContentsTmp->{ 'Header' };
    }
    $this->{ 'Visitor' }->initializePage(   $dbPagePOMTmp
                                          , $this->{ 'Title' }
                                          , $sHdrContTmp
                                          , $sBeforeBodyAddContTmp
                                         );

    #---- Flush contents
    foreach my $blkToWriteTmp ( @arrBlkListTmp ){
        #---- List might have duplicates due to repetitive blocks, so assert flushed or not before writing
        #if ( $blkToWriteTmp->isFlushed() == $TRUE ){
        #    next;
        #}

        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "Block (" . $blkToWriteTmp->getId() . ") writing started" );
        }

        #---- Write block
        $blkToWriteTmp->write( $this->{ 'Visitor' } );

        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "Block (" . $blkToWriteTmp->getId() . ") writing ended" );
        }
    }

    #---- Finalize page
    my $sFtContTmp = $this->{ 'PreDefAC' }{ 'Footer' };
    if ( $hrefAddContentsTmp != undef ){
        $sFtContTmp .= $hrefAddContentsTmp->{ 'Footer' };
    }
    $this->{ 'Visitor' }->finalizePage(   $sAfterBodyAddContTmp
                                        , $sFtContTmp
                                      );

    #---- Clean up
    $this->{ 'PageBlocks' }{ $iPageIdPar } = undef;
}

#-----------------------------------------------------------------------
# Dump the DOM to a file
#-----------------------------------------------------------------------
sub dumpDOM{
    my $this = shift;

    #---- Get parameters
    #
    # 1. Output path
    # 2. Flag to sort or not the contents
    #
    my $sOutputFilePathPar = shift;
    my $bSortedPar = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "dumpDOM(" . $sOutputFilePathPar . ", " . (($bSortedPar)? "TRUE" : "FALSE") . ")" );
    }

    #---- Iterate and dump all the POMs from DOM
    my $dmDumperTmp = undef;
    my $domObjTmp = $this->{ 'DOM' };
    my @arrPageIdsTmp = sort keys( %{ $domObjTmp } );
    foreach my $sPageIdTmp ( @arrPageIdsTmp ){
        #---- Create dumper
        $dmDumperTmp = new a2w::core::dm::Dumper();

        #---- Dump POM
        $dmDumperTmp->dumpDOM( $sOutputFilePathPar, $this->{ 'DOM' }{ $sPageIdTmp }, $bSortedPar );

        #---- Release the Dumper instance
        $dmDumperTmp = undef;
    }
}

#-----------------------------------------------------------------------
# Register data mining classes for logging
#-----------------------------------------------------------------------
sub registerClassesForLog{
    my $this = shift;

    #---- Register data mining specific classes for logging
    $theLogger->registerAdditionalClass( "a2w::core::dm::AddContent" );
    $theLogger->registerAdditionalClass( "a2w::core::dm::Anchor" );
    $theLogger->registerAdditionalClass( "a2w::core::dm::Block" );
    $theLogger->registerAdditionalClass( "a2w::core::dm::BlockFormatter" );
    $theLogger->registerAdditionalClass( "a2w::core::dm::Config" );
    $theLogger->registerAdditionalClass( "a2w::core::dm::ConfigParser" );
    $theLogger->registerAdditionalClass( "a2w::core::dm::Constraint" );
    $theLogger->registerAdditionalClass( "a2w::core::dm::Constraints" );
    $theLogger->registerAdditionalClass( "a2w::core::dm::ContentDef" );
    $theLogger->registerAdditionalClass( "a2w::core::dm::ContentParser" );
    $theLogger->registerAdditionalClass( "a2w::core::dm::ContentProcessor" );
    $theLogger->registerAdditionalClass( "a2w::core::dm::Database" );
    $theLogger->registerAdditionalClass( "a2w::core::dm::Dumper" );
    $theLogger->registerAdditionalClass( "a2w::core::dm::FindAndReplace" );
    $theLogger->registerAdditionalClass( "a2w::core::dm::Index" );
    $theLogger->registerAdditionalClass( "a2w::core::dm::ParagraphBlockFormatter" );
    $theLogger->registerAdditionalClass( "a2w::core::dm::ParagraphContentDef" );
    $theLogger->registerAdditionalClass( "a2w::core::dm::SkipBlockFormatter" );
    $theLogger->registerAdditionalClass( "a2w::core::dm::SkipContentDef" );
    $theLogger->registerAdditionalClass( "a2w::core::dm::TableBlockFormatter" );
    $theLogger->registerAdditionalClass( "a2w::core::dm::TableContentDef" );
    $theLogger->registerAdditionalClass( "a2w::core::visitor::Visitor" );
    $theLogger->registerAdditionalClass( "a2w::core::visitor::HTMLVisitor" );
}

#-----------------------------------------------------------------------
# Process POM
#
# Process page objects and add them to appropriate block they belong to
#
# Parameters:
# 1. POM
# 2. Flag to indicate whether objects can be removed or skipped (0=>Skip, 1=>Removable)
#
# Returns 0 in case of success, <0 in case of error
#-----------------------------------------------------------------------
sub _processPOM{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "_processPOM" );
    }

    #---- Get parameter
    #
    # 1. POM
    # 2. Flag to indicate whether objects can be removed or skipped (0=>Skip, 1=>Removable)
    #
    my $dmPOMPar   = shift;
    my $bRemovePar = $FALSE;
    if ( @_ > 0 ){
        $bRemovePar = shift;
    }

    #---- Process POM objects and add them to appropriate blocks ----#
    my $iPageIdTmp   = $dmPOMPar->getPageID();
    my $iObjCountTmp = $dmPOMPar->getObjectCount();

    #---- Process objects sorted by Y position
    my $hrefYPosTmp    = $dmPOMPar->getHashYPos();
    my $hrefObjectsTmp = $dmPOMPar->getHashObj();
    my $hrefObjectTmp  = undef;
    my $a2wObjectTmp   = undef;

    my $blkObjectTmp   = undef;
    my $blkPageDefTmp  = undef;
    my $bObjAddedTmp   = $FALSE;
    my $bObjIsAncTmp   = $FALSE;
    my $bSkipObjTmp    = $FALSE;

    my @arrBlkListTmp = ();
    my @arrBlkLst2Tmp = ();

    #---- Sort Y positions
    my @arrSortedYPositionsTmp = sort { $a <=> $b } keys( %{ $hrefYPosTmp } );

    my $hrefPageBlocksTmp   = {};
    my $arefPageBlocksTmp   = [];
    my $iPageDefBlkCountTmp = 0;

    #---- Iterate through sorted Y positions
    foreach my $fYPosTmp ( @arrSortedYPositionsTmp ){
        #---- Sort X positions (i.e, sort multiple objects based on X positions that occur on same Y position aka line)
        # $V104 Begin
        #my @arrSortedXPositionsTmp = sort { $hrefYPosTmp->{ $fYPosTmp }{ $a }{ $a2w::core::dm::Constants::AT_XPOS } <=> $hrefYPosTmp->{ $fYPosTmp }{ $b }{ $a2w::core::dm::Constants::AT_XPOS } } keys( %{ $hrefYPosTmp->{ $fYPosTmp } } );
        my @arrSortedXPositionsTmp = sort {    $hrefYPosTmp->{ $fYPosTmp }{ $a }{ $a2w::core::dm::Constants::AT_OBJSEQID } <=> $hrefYPosTmp->{ $fYPosTmp }{ $b }{ $a2w::core::dm::Constants::AT_OBJSEQID }
                                            or $hrefYPosTmp->{ $fYPosTmp }{ $a }{ $a2w::core::dm::Constants::AT_XPOS } <=> $hrefYPosTmp->{ $fYPosTmp }{ $b }{ $a2w::core::dm::Constants::AT_XPOS }
                                          }
                                     keys( %{ $hrefYPosTmp->{ $fYPosTmp } } );
        # $V104 End

        #---- Iterate through sorted X positions
        foreach my $iOrderIdTmp ( @arrSortedXPositionsTmp ){
            #---- Get document blocks
            @arrBlkListTmp = @{ $this->{ 'DocBlocks' } };

            #---- Filter out unfilled blocks
            @arrBlkListTmp = grep { $_ != undef && $_->isFilled() == $FALSE } @arrBlkListTmp;

            #---- Get object reference
            $hrefObjectTmp = $hrefYPosTmp->{ $fYPosTmp }{ $iOrderIdTmp };

            #---- Set page id on object
            $hrefObjectTmp->{ $a2w::core::dm::Constants::AT_PAGE_ID } = $iPageIdTmp;

            #---- Get kernel object
            $a2wObjectTmp = $hrefObjectsTmp->{ $iOrderIdTmp }{ 'OBJECT' };
            if ( $bLog == $TRUE ){
                $this->_logObject( $a2wObjectTmp, $hrefObjectTmp );
            }

            #---- Detect all the blocks the object falls in and add to them ----#
            $bObjAddedTmp  = $FALSE;
            $bSkipObjTmp   = $FALSE;
            @arrBlkLst2Tmp = @arrBlkListTmp;
            do {
                $bObjIsAncTmp = $FALSE;
                $blkObjectTmp = $this->_fetchObjectBlock( # POM page
                                                            $dmPOMPar
                                                          # POM object hash
                                                          , $hrefObjectTmp
                                                          # Object
                                                          , $a2wObjectTmp
                                                          # Block list
                                                          , \@arrBlkLst2Tmp
                                                          # Anchor or not flag return value
                                                          , \$bObjIsAncTmp
                                                        );

                #---- Add object to the configured block
                if ( $blkObjectTmp != undef ){
                    if ( $blkPageDefTmp != undef ){
                        $blkPageDefTmp->setAsFilled();
                        $blkPageDefTmp = undef;
                    }
                    my $sBlkIdTmp = $blkObjectTmp->getUniqueId();
                    if ( $hrefPageBlocksTmp->{ $sBlkIdTmp } == undef ){
                        if ( $bLog == $TRUE ){
                            $theLogger->logMessage( "Block " . $blkObjectTmp->getId() . " added to page blocks list" );
                        }
                        $hrefPageBlocksTmp->{ $sBlkIdTmp } = $blkObjectTmp;
                        $arefPageBlocksTmp->[ @{ $arefPageBlocksTmp } ] = $blkObjectTmp;
                    }

                    if ( $bObjIsAncTmp == $FALSE ){
                        $blkObjectTmp->addObject( $a2wObjectTmp, $hrefObjectTmp );
                    }
                    if ( $bSkipObjTmp == $FALSE ){
                        $bSkipObjTmp = $blkObjectTmp->hasToBeSkipped();
                    }
                    $bObjAddedTmp  = $TRUE;

                    #---- Filter blocks other than current block to find object belong to them too
                    my $sBlkIdTmp  = $blkObjectTmp->getId();
                    @arrBlkLst2Tmp = grep { $_ != undef && $_->getId() ne $sBlkIdTmp } @arrBlkLst2Tmp;
                }
            } while ( $blkObjectTmp != undef );

            #---- Add object to default block
            if ( $bObjAddedTmp == $FALSE ){
                #---- Add block to page default block
                if ( $blkPageDefTmp == undef ){
                    $iPageDefBlkCountTmp++;
                    my $sBlkIdTmp = $a2w::core::dm::Constants::BLK_PAGE_DEF_PREFIX . $iPageIdTmp . "_" . $iPageDefBlkCountTmp;
                    $blkPageDefTmp = $this->{ 'Config' }->createDefaultBlock( $sBlkIdTmp );
                    $sBlkIdTmp = $blkPageDefTmp->getUniqueId();
                    if ( $hrefPageBlocksTmp->{ $sBlkIdTmp } == undef ){
                        if ( $bLog == $TRUE ){
                            $theLogger->logMessage( "Block " . $blkPageDefTmp->getId() . " added to page blocks list" );
                        }
                        $hrefPageBlocksTmp->{ $sBlkIdTmp } = $blkPageDefTmp;
                        $arefPageBlocksTmp->[ @{ $arefPageBlocksTmp } ] = $blkPageDefTmp;
                    }

                    #---- Fill in info on default block
                    $blkPageDefTmp->setWidth( $dmPOMPar->getPageWidth() );
                    $blkPageDefTmp->setStartedOn( $iPageIdTmp );
                    $blkPageDefTmp->updateBoundary(   $a2wObjectTmp
                                                    , $hrefObjectTmp
                                                  );
                }
                $blkPageDefTmp->addObject( $a2wObjectTmp, $hrefObjectTmp );
                $blkPageDefTmp->updateBoundary(   $a2wObjectTmp
                                                , $hrefObjectTmp
                                              );
            }

            #---- Assert and remove object from kernel presentation
            if ( $bRemovePar == $TRUE ){
                my $hrefObjTmp = {
                      'A2WOBJ' => $a2wObjectTmp
                    , 'POMOBJ' => $hrefObjectTmp
                };
                if (    $this->{ 'Visitor' } != undef
                     && $this->{ 'Visitor' }->isWritable( $hrefObjTmp ) == $TRUE
                   ){
                    #---- Remove object from kernel presentation
                    $a2wObjectTmp->remove();
                    if ( $bLog == $TRUE ){
                        $theLogger->logMessage( "object removed from presentation" );
                    }
                }
                elsif ( $bSkipObjTmp == $TRUE ){
                    #---- Remove object from kernel presentation
                    $a2wObjectTmp->remove();
                    if ( $bLog == $TRUE ){
                        $theLogger->logMessage( "object removed from presentation" );
                    }
                }
            }
        }
    }

    #---- Set last page default block as filled
    if ( $blkPageDefTmp != undef ){
        $blkPageDefTmp->setAsFilled();
        $blkPageDefTmp = undef;
    }

    #---- Store the page blocks
    $this->{ 'PageBlocks' }{ $iPageIdTmp } = $arefPageBlocksTmp;
    if ( $bLog == $TRUE ){
        my @arrBlkListTmp = @{ $arefPageBlocksTmp };
        my $sBlkListTmp = "";
        foreach my $blkTmp ( @arrBlkListTmp ){
            $sBlkListTmp .= " " . $blkTmp->getId();
        }
        $theLogger->logMessage( "Page $iPageIdTmp blocks:$sBlkListTmp" );
    }

    #---- Post process blocks ----#
    # 1. Reset block start anchor, if not completed on current page
    #    in order to continue content collection on following page
    #
    @arrBlkListTmp = @{ $this->{ 'DocBlocks' } };
    for ( my $i = 0; $i < @arrBlkListTmp; $i++ ){
        if ( @arrBlkListTmp[ $i ] != undef ){
            # 1. Reset block start anchor
            if (    @arrBlkListTmp[ $i ]->hasStarted() == $TRUE
                 && @arrBlkListTmp[ $i ]->hasEnded() == $FALSE
               ){
                @arrBlkListTmp[ $i ]->resetStartAnchor();
            }
        }
    }

    return 0;
}

#-----------------------------------------------------------------------
# Fetch object block
#-----------------------------------------------------------------------
sub _fetchObjectBlock{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "_fetchObjectBlock()" );
    #}

    #---- Get parameter
    #
    # 1. POM page
    # 2. Object hash
    # 3. Object
    # 4. Block array reference
    # 5. Reference to return flag saying whether object is start/end anchor
    #
    my $pomPagePar   = shift;
    my $hrefObjPar   = shift;
    my $a2wObjPar    = shift;
    my $blkListPar   = shift;
    my $bObjIsAncPar = shift;

    #---- Iterate through active blocks list and identify on which block the object falls ----#
    my $blkObjFallInTmp = undef;
    my $bObjFallInTmp   = $FALSE;

    my $bObjIsStartAnchorTmp = $FALSE;
    my $bObjIsEndAnchorTmp   = $FALSE;
    my @arrDocBlocksTmp = @{ $blkListPar };
    for ( my $i = 0; $i < @arrDocBlocksTmp; $i++ ){
        $blkObjFallInTmp = @arrDocBlocksTmp[ $i ];
        if ( $blkObjFallInTmp == undef ){
            next;
        }

        #---- Assert whether object is start anchor of a block
        if ( $blkObjFallInTmp->isStartAnchor( $pomPagePar, $a2wObjPar, $hrefObjPar ) == $TRUE ){
            $$bObjIsAncPar = $TRUE;
            $bObjFallInTmp = $TRUE;
            $bObjIsStartAnchorTmp = $TRUE;

            # $V101 Begin
            #---- Mark on which page block started
            if ( $blkObjFallInTmp->getStartedOn() <= 0 ){
                $blkObjFallInTmp->setStartedOn( $hrefObjPar->{ $a2w::core::dm::Constants::AT_PAGE_ID } );

                #---- Add start anchor to block only when detected first, for following pages add it as regular object
                $blkObjFallInTmp->addStartAnchor( $a2wObjPar, $hrefObjPar );
            }
            else {
                my $ancStTmp = $blkObjFallInTmp->getStartAnchor();
                if ( $ancStTmp != undef && $ancStTmp->isSkipped() == $FALSE ){
                    # Let object added as regular instead of starting anchor
                    $$bObjIsAncPar = $FALSE;
                }
                else {
                    #---- Add start anchor to block
                    $blkObjFallInTmp->addStartAnchor( $a2wObjPar, $hrefObjPar );
                }
            }
            # $V101 End
        }

        #---- Assert whether object fall in block
        # $V101 Begin
        #if ( $blkObjFallInTmp->doesObjectFallIn( $a2wObjPar ) == $TRUE ){
        if ( $blkObjFallInTmp->doesObjectFallIn( $pomPagePar, $a2wObjPar, $hrefObjPar ) == $TRUE ){
        # $V101 End
            $bObjFallInTmp = $TRUE;
        }

        #---- Assert whether object is end anchor of a block
        if ( $blkObjFallInTmp->isEndAnchor( $pomPagePar, $a2wObjPar, $hrefObjPar ) == $TRUE ){
            $$bObjIsAncPar = $TRUE;
            $bObjFallInTmp = $TRUE;
            $bObjIsEndAnchorTmp = $TRUE;

            #---- Add end anchor to block
            if ( $bObjIsStartAnchorTmp == $FALSE ){
                #---- Do not add to block, if same object is start/end anchor
                #     to avoid anchor being written twice
                $blkObjFallInTmp->addEndAnchor( $a2wObjPar, $hrefObjPar );
            }

            if ( $blkObjFallInTmp->isFilled() ){
                #---- Assert and create repetitive block ----#
                if (    $blkObjFallInTmp->isRepetitive() == $TRUE
                     && $blkObjFallInTmp->hasMoreRepeatation() == $TRUE
                   ){
                    #---- Create repetitive block
                    my $blkRepetitiveTmp = $blkObjFallInTmp->createRepetitiveBlock();

                    #---- Add to document block list
                    # $V103 Begin
                    #$this->addDocumentBlock( $blkRepetitiveTmp );
                    $this->addRepetitiveDocumentBlock( $blkRepetitiveTmp );
                    # $V103 End
                }

                #---- Update block
                $blkObjFallInTmp->update();
            }
        }

        #---- Break if object found to fall in a block
        if ( $bObjFallInTmp == $TRUE ){
            last;
        }
    }
    if ( $bObjFallInTmp == $FALSE ){ return undef; }

    return $blkObjFallInTmp;
}

#-----------------------------------------------------------------------
# Get additional contents of predefined blocks
#
# Analyze the blocks, detect header/body/footer/before/after additional
# contents if specified and collect them with out duplicate (using
# additional content id).
#
# Returns
# - undef in case of error
# - Hash reference having following structure, in case of success
#   {
#       'Header'    => <Additional content that belong to header>
#       'HdrBefore' => <Additional content that belong right after header start tag>
#       'HdrAfter'  => <Additional content that belong right above header end tag>
#       'Body'      => <Additional content that belong to body end>
#       'BdyBefore' => <Additional content that belong right after body start tag>
#       'BdyAfter'  => <Additional content that belong right above body end tag>
#       'Footer'    => <Additional content that belong to footer>
#       'FtrBefore' => <Additional content that belong right after footer start tag>
#       'FtrAfter'  => <Additional content that belong right above footer end tag>
#   }
#
#-----------------------------------------------------------------------
sub _getAdditionalContentsOfPredefinedBlocks{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "_getAdditionalContentsOfPredefinedBlocks()" );
    }

    #---- Predefined block names list
    my @arrPDHdrBlkNamesTmp = (
          $a2w::core::dm::Constants::BLK_DOC_HEADER
        , $a2w::core::dm::Constants::BLK_PAGE_HEADER
    );
    my @arrPDFtrBlkNamesTmp = (
          $a2w::core::dm::Constants::BLK_PAGE_FOOTER
        , $a2w::core::dm::Constants::BLK_DOC_FOOTER
    );
    my @arrPDBlkNamesTmp = (
          @arrPDHdrBlkNamesTmp
        , $a2w::core::dm::Constants::BLK_PAGE_BODY
        , @arrPDFtrBlkNamesTmp
    );

    #---- Filter configuration defined blocks
    my @arrPDBlkListTmp = ();
    my $hrefPDBlksTmp   = {};
    my $blkRefTmp = undef;
    foreach $id ( @arrPDBlkNamesTmp ){
        $blkRefTmp = $this->{ 'Config' }->getBlock( $id );
        if ( $blkRefTmp != undef ){
            if ( $bLog == $TRUE ){
                $theLogger->logMessage( "Block:>" . $id . "<" );
            }
            @arrPDBlkListTmp[ ( $#arrPDBlkListTmp + 1 ) ] = $blkRefTmp;
            $hrefPDBlksTmp->{ $id } = $blkRefTmp;
        }
    }

    #---- Get additional contents
    my $hrefPDACTmp = $this->_getAdditionalContents( \@arrPDBlkListTmp );

    #---- Additional processing ----#
    # For A2W_DMF_DOC_HEADER/A2W_DMF_PAGE_HEADER predefined block, additional contents included at
    # 'before'/'after' are treated same as 'header' position
    #
    my $hrefBeforeTmp = {};
    my $hrefAfterTmp  = {};
    my $contDefTmp    = undef;
    my $hrefACTmp     = undef;
    my @arrACKeysTmp  = ();
    foreach my $sHKeyTmp ( @arrPDHdrBlkNamesTmp ){
        if ( $hrefPDBlksTmp->{ $sHKeyTmp } == undef ){
            next;
        }

        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "Block:>" . $sHKeyTmp . "<" );
        }

        #---- Get content defition
        $hrefBeforeTmp = {};
        $hrefAfterTmp  = {};
        $contDefTmp = $hrefPDBlksTmp->{ $sHKeyTmp }->getContentDef();
        if ( $contDefTmp != undef ){
            $hrefACTmp = $contDefTmp->getAddContent();
            @arrACKeysTmp = sort keys( %{ $hrefACTmp } );
            if ( $bLog == $TRUE ){
                $theLogger->logMessage( "ContentDef exists: @arrACKeysTmp" );
            }
            foreach my $sIdTmp ( @arrACKeysTmp ){
                if ( lc( $hrefACTmp->{ $sIdTmp }->getInclude() ) eq "before" ){
                    $hrefBeforeTmp->{ $sIdTmp } = $hrefACTmp->{ $sIdTmp }->getContent();
                }
                elsif ( lc( $hrefACTmp->{ $sIdTmp }->getInclude() ) eq "after" ){
                    $hrefAfterTmp->{ $sIdTmp } = $hrefACTmp->{ $sIdTmp }->getContent();
                }
            }

            my $sBeforeTmp = '';
            my $sAfterTmp  = '';
            my @arrHdrContTmp = sort keys( %{ $hrefBeforeTmp } );
            foreach my $b ( @arrHdrContTmp ){
                $sBeforeTmp .= $hrefBeforeTmp->{ $b };
            }
            @arrHdrContTmp = sort keys( %{ $hrefAfterTmp } );
            foreach my $a ( @arrHdrContTmp ){
                $sAfterTmp .= $hrefBeforeTmp->{ $b };
            }
            $hrefPDACTmp->{ 'HdrBefore' } .= $sBeforeTmp;
            $hrefPDACTmp->{ 'HdrAfter' }  .= $sAfterTmp;
            $hrefPDACTmp->{ 'Header' }    .= $sBeforeTmp . $sAfterTmp;
        }
    }

    # For A2W_DMF_PAGE_BODY predefined block, additional contents included at
    # 'after' are treated same as 'body' position
    #
    if ( $hrefPDBlksTmp->{ $a2w::core::dm::Constants::BLK_PAGE_BODY } != undef ){
        $hrefBeforeTmp = {};
        $hrefAfterTmp  = {};
        $contDefTmp    = undef;
        $hrefACTmp     = undef;
        @arrACKeysTmp  = ();

        #---- Get content defition
        $contDefTmp = $hrefPDBlksTmp->{ $a2w::core::dm::Constants::BLK_PAGE_BODY }->getContentDef();
        if ( $contDefTmp != undef ){
            $hrefACTmp = $contDefTmp->getAddContent();
            @arrACKeysTmp = sort keys( %{ $hrefACTmp } );
            foreach my $sIdTmp ( @arrACKeysTmp ){
                if ( lc( $hrefACTmp->{ $sIdTmp }->getInclude() ) eq "before" ){
                    $hrefBeforeTmp->{ $sIdTmp } = $hrefACTmp->{ $sIdTmp }->getContent();
                }
                elsif ( lc( $hrefACTmp->{ $sIdTmp }->getInclude() ) eq "after" ){
                    $hrefAfterTmp->{ $sIdTmp } = $hrefACTmp->{ $sIdTmp }->getContent();
                }
            }

            my $sBeforeTmp = '';
            my $sAfterTmp  = '';
            my @arrBodyContTmp = sort keys( %{ $hrefBeforeTmp } );
            foreach my $b ( @arrBodyContTmp ){
                $sBeforeTmp .= $hrefBeforeTmp->{ $b };
            }
            @arrBodyContTmp = sort keys( %{ $hrefAfterTmp } );
            foreach my $b ( @arrBodyContTmp ){
                $sAfterTmp .= $hrefAfterTmp->{ $b };
            }
            $hrefPDACTmp->{ 'BdyBefore' } .= $sBeforeTmp;
            $hrefPDACTmp->{ 'BdyAfter' }  .= $sAfterTmp;
            $hrefPDACTmp->{ 'Body' }      .= $sBeforeTmp . $sAfterTmp;
        }
    }

    # For A2W_DMF_DOC_FOOTER/A2W_DMF_PAGE_FOOTER predefined block, additional contents included at
    # 'before'/'after' are treated same as 'footer' position
    #
    $contDefTmp    = undef;
    $hrefACTmp     = undef;
    @arrACKeysTmp  = ();
    foreach my $sFKeyTmp ( @arrPDFtrBlkNamesTmp ){
        if ( $hrefPDBlksTmp->{ $sFKeyTmp } == undef ){
            next;
        }

        #---- Get content defition
        $hrefBeforeTmp = {};
        $hrefAfterTmp  = {};
        $contDefTmp = $hrefPDBlksTmp->{ $sFKeyTmp }->getContentDef();
        if ( $contDefTmp != undef ){
            $hrefACTmp = $contDefTmp->getAddContent();
            @arrACKeysTmp = sort keys( %{ $hrefACTmp } );
            foreach my $sIdTmp ( @arrACKeysTmp ){
                if ( lc( $hrefACTmp->{ $sIdTmp }->getInclude() ) eq "before" ){
                    $hrefBeforeTmp->{ $sIdTmp } = $hrefACTmp->{ $sIdTmp }->getContent();
                }
                elsif ( lc( $hrefACTmp->{ $sIdTmp }->getInclude() ) eq "after" ){
                    $hrefAfterTmp->{ $sIdTmp } = $hrefACTmp->{ $sIdTmp }->getContent();
                }
            }

            my $sBeforeTmp = '';
            my $sAfterTmp  = '';
            my @arrFtContTmp = sort keys( %{ $hrefBeforeTmp } );
            foreach my $b ( @arrFtContTmp ){
                $sBeforeTmp .= $hrefBeforeTmp->{ $b };
            }
            @arrFtContTmp = sort keys( %{ $hrefAfterTmp } );
            foreach my $a ( @arrFtContTmp ){
                $sAfterTmp .= $hrefAfterTmp->{ $a };
            }
            $hrefPDACTmp->{ 'FtrBefore' } .= $sBeforeTmp;
            $hrefPDACTmp->{ 'FtrAfter' }  .= $sAfterTmp;
            $hrefPDACTmp->{ 'Footer' }    .= $sBeforeTmp . $sAfterTmp;
        }
    }

    if ( $bLog == $TRUE ){
        $theLogger->logMessage( "Predefined block additional contents:" );
        $theLogger->logMessage( "Header:>" . $hrefPDACTmp->{ 'Header' } . "<" );
        $theLogger->logMessage( "Header Before:>" . $hrefPDACTmp->{ 'HdrBefore' } . "<" );
        $theLogger->logMessage( "Header After:>" . $hrefPDACTmp->{ 'HdrAfter' } . "<" );
        $theLogger->logMessage( "Body:>" . $hrefPDACTmp->{ 'Body' } . "<" );
        $theLogger->logMessage( "Body Before:>" . $hrefPDACTmp->{ 'BdyBefore' } . "<" );
        $theLogger->logMessage( "Body After:>" . $hrefPDACTmp->{ 'BdyAfter' } . "<" );
        $theLogger->logMessage( "Footer:>" . $hrefPDACTmp->{ 'Footer' } . "<" );
        $theLogger->logMessage( "Footer Before:>" . $hrefPDACTmp->{ 'FtrBefore' } . "<" );
        $theLogger->logMessage( "Footer After:>" . $hrefPDACTmp->{ 'FtrAfter' } . "<" );
    }
    return $hrefPDACTmp;
}

#-----------------------------------------------------------------------
# Get additional contents of blocks
#
# Analyze the blocks, detect header/body/footer additional contents if
# specified and collect them with out duplicate (using additional content
# id).
#
# Parameter
# 1. Array of blocks
#
# Returns
# - undef in case of error
# - Hash reference having following structure, in case of success
#   {
#       'Header' => <Additional content that belong to header>
#       'Body'   => <Additional content that belong to body end>
#       'Footer' => <Additional content that belong to footer>
#   }
#
#-----------------------------------------------------------------------
sub _getAdditionalContents{
    my $this = shift;

    #---- Get parameter
    #
    # 1. Array of blocks
    #
    my $arefBlocksPar = shift;

    #---- Get blocks
    my @arrBlkListTmp = @{ $arefBlocksPar };

    #---- Collect additional header/body/footer contents, if block has any ----#
    my $hrefHeaderTmp = {};
    my $hrefBodyTmp   = {};
    my $hrefFooterTmp = {};
    my $contDefTmp    = undef;
    my $hrefACTmp     = undef;
    my @arrACKeysTmp  = ();
    foreach my $blkTmp ( ( @arrBlkListTmp, @arrPDBlkListTmp ) ){
        #---- Get content defition
        $contDefTmp = $blkTmp->getContentDef();
        if ( $contDefTmp == undef ){ next; }
        $hrefACTmp = $contDefTmp->getAddContent();
        @arrACKeysTmp = sort keys( %{ $hrefACTmp } );
        foreach my $sIdTmp ( @arrACKeysTmp ){
            if ( lc( $hrefACTmp->{ $sIdTmp }->getInclude() ) eq "header" ){
                $hrefHeaderTmp->{ $sIdTmp } = $hrefACTmp->{ $sIdTmp }->getContent();
            }
            elsif ( lc( $hrefACTmp->{ $sIdTmp }->getInclude() ) eq "body" ){
                $hrefBodyTmp->{ $sIdTmp } = $hrefACTmp->{ $sIdTmp }->getContent();
            }
            elsif ( lc( $hrefACTmp->{ $sIdTmp }->getInclude() ) eq "footer" ){
                $hrefFooterTmp->{ $sIdTmp } = $hrefACTmp->{ $sIdTmp }->getContent();
            }
        }
    }

    #---- Prepare return value
    my $hrefAddContTmp = {
          'Header'    => ''
        , 'HdrBefore' => ''
        , 'HdrAfter'  => ''
        , 'Body'      => ''
        , 'BdyBefore' => ''
        , 'BdyAfter'  => ''
        , 'Footer'    => ''
        , 'FtrBefore' => ''
        , 'FtrAfter'  => ''
    };
    my @arrHdrContTmp = sort keys( %{ $hrefHeaderTmp } );
    foreach my $h ( @arrHdrContTmp ){
        $hrefAddContTmp->{ 'Header' } .= $hrefHeaderTmp->{ $h };
    }
    my @arrBdContTmp = sort keys( %{ $hrefBodyTmp } );
    foreach my $b ( @arrBdContTmp ){
        $hrefAddContTmp->{ 'Body' } .= $hrefBodyTmp->{ $b };
    }
    my @arrFtContTmp = sort keys( %{ $hrefFooterTmp } );
    foreach my $f ( @arrFtContTmp ){
        $hrefAddContTmp->{ 'Footer' } .= $hrefFooterTmp->{ $f };
    }

    return $hrefAddContTmp;
}

#-----------------------------------------------------------------------
# Log object
#-----------------------------------------------------------------------
sub _logObject{
    my $this = shift;

    #---- Get parameter
    #
    # 1. Object
    # 2. POM object wrapper (optional)
    #
    my $a2wObjectPar = shift;
    my $pomObjectPar = undef;
    if ( @_ > 0 ){
        $pomObjectPar = shift;
    }

    #---- Build log message ----#
    my $sObjTypeTmp = $a2wObjectPar->_getType();
    my $sLogMsgTmp  = "";
    if ( $sObjTypeTmp eq "image" ){
        $sLogMsgTmp = "img>" . $a2wObjectPar->getName() . "<@(" . $a2wObjectPar->getXPos() . "," . $a2wObjectPar->getYPos() . ") W=>" . $a2wObjectPar->getWidth() . "< H=>" . $a2wObjectPar->getHeight() . "<";
    }
    elsif ( $sObjTypeTmp eq "line" ){
        $sLogMsgTmp = "lin><@(" . $a2wObjectPar->getXPos() . "," . $a2wObjectPar->getYPos() . ") W=>" . $a2wObjectPar->getWidth() . "< L=>" . $a2wObjectPar->getLength() . "<";
    }
    elsif ( $sObjTypeTmp eq "text" ){
        $sLogMsgTmp = "txt>" . $a2wObjectPar->getText() . "<@(" . $a2wObjectPar->getXPos() . "," . $a2wObjectPar->getYPos() . ")";
    }
    elsif ( $sObjTypeTmp eq "vector" ){
        $sLogMsgTmp = "vec><@(" . $a2wObjectPar->getXPos() . "," . $a2wObjectPar->getYPos() . ")";
    }
    $theLogger->logMessage( $sLogMsgTmp );
}

#-----------------------------------------------------------------------
# Don't remove the following lines !!!
#-----------------------------------------------------------------------
1;
__END__
