#-------------------------------------------------------------------------------
#  a2w::core::dm::TableBlockFormatter.pm:
#  Data Mining Framework Table Block Formatter
#
#  Author  : AFP2web Team, Maas Holding GmbH
#
#  $V100   2014-05-08    Initial Release
#
#  $V101   2015-08-26    Issue:
#                        Table have gap in between certain rows (in the ‘CHECKS CLEARED AT A GLANCE’ section
#                        in the html, there are 2 blank rows between Check# 5377 and 5350, and also a blank
#                        row between Check#5378 and 5379)
#
#                        Reason:
#                        Table has empty rows, which visually look like a gap in between rows. Empty rows occur
#                        due to lines (start/end anchor or actual content of block).
#
#                        Fix:
#                        Ensured to skip lines when evaluating table row elements
#
#                        JiRa:
#                        AFP-279
#
#                        Fixed by, Date:
#                        Panneer, 26/08/2015
#
#  $V102   2015-10-07    Issue:
#                        Table cell having multi line value occur now in multiple rows
#
#                        Reason:
#                        Table rows are identified based on change in Y position, so each line of
#                        same cell is treated as different row.
#
#                        Fix:
#                        Take first column as primary and mark beginning of a new row whenever the
#                        first column value on line is not empty. i.e, Following lines of a cell must
#                        have empty first column value on line
#
#                        JiRa:
#                        AFP-224
#
#                        Fixed by, Date:
#                        Panneer, 07/10/2015
#
#  $V103   2015-11-06    Fixed minor bug in writing begin table body '<tbody>' element
#
#                        JiRa:
#                        AFP-320
#
#                        Fixed by, Date:
#                        Panneer, 06/11/2015
#
#  $V104   2015-12-11    Fixed minor bug in writing begin table body '<tbody>' element
#                        when table block have less than or equal to two rows
#
#                        JiRa:
#                        AFP-320
#
#                        Fixed by, Date:
#                        Panneer, 11/12/2015
#
#-------------------------------------------------------------------------------
package a2w::core::dm::TableBlockFormatter;

#-----------------------------------------------------------------------
# Include required modules
#-----------------------------------------------------------------------
use a2w::TypeConstants;
use a2w::core::log::Logger;
use a2w::core::dm::Constants;
use a2w::core::dm::BlockFormatter;

#-----------------------------------------------------------------------
# Inherit from base class
#-----------------------------------------------------------------------
our @ISA = qw( a2w::core::dm::BlockFormatter );

#-----------------------------------------------------------------------
# Constructor
#-----------------------------------------------------------------------
sub new{
    my $proto = shift;
    my $class = ref( $proto ) || $proto;

    #---- Define boolean values
    $TRUE  = $a2w::TypeConstants::TRUE;     # TRUE  boolean value
    $FALSE = $a2w::TypeConstants::FALSE;    # FALSE boolean value

    #---- Instantiate from base class to inherit base class attributes
    my $this = a2w::core::dm::BlockFormatter->new( @_ );

    #---- Add this derived class specific attributes
    #$this->{ 'someattribute' } = 0;

    bless( $this, $class );

    #---- Get logger
    our $theLogger = a2w::core::log::Logger->getSingleton();
    our $bLog = $theLogger->isRegistered( __PACKAGE__ );

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "new()" );
    #}
    return $this;
}

#-----------------------------------------------------------------------
# Destructor
#-----------------------------------------------------------------------
sub DESTROY{
    my $this = shift;
}

#-----------------------------------------------------------------------
# Mutators
#-----------------------------------------------------------------------

#-----------------------------------------------------------------------
# Accessors
#-----------------------------------------------------------------------

#-----------------------------------------------------------------------
# Workers
#-----------------------------------------------------------------------
#-----------------------------------------------------------------------
# writeFormattedBlock
#
# Writes block content formatted as table rows/columns
#
# >=0, in case of successful writing
# < 0, in case of error
#
#-----------------------------------------------------------------------
sub writeFormattedBlock{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeFormattedBlock()" );
    }

    #---- Parameter
    #
    # 1. Block
    # 2. Visitor
    #
    my $blkCurrentPar = shift;
    my $outVisitorPar = shift;

    #---- Fetch content definition of block
    my $contDefTmp = $blkCurrentPar->getContentDef();
    if ( lc( $contDefTmp->getType() ) ne "table" ){
        if ( $bLog == $TRUE ){
            $theLogger->logMessage(   "Warning! Table block formatter can not process other type of blocks<" );
        }
        return -1;
    }

    #---- Assert block has content or not ----#
    my @arrObjsListTmp = ();

    # Check and add start anchor objects
    my $ancStTmp = $blkCurrentPar->getStartAnchor();
    if ( $ancStTmp != undef && $ancStTmp->isSkipped() == $FALSE ){
        push( @arrObjsListTmp, @{ $blkCurrentPar->getStartAnchorsList() } );
    }
	
    # Add Block objects
    push( @arrObjsListTmp, @{ $blkCurrentPar->getObjsList() } );
	
    # Check and add end anchor objects
    my $ancEdTmp = $blkCurrentPar->getEndAnchor();
    if ( $ancEdTmp != undef && $ancEdTmp->isSkipped() == $FALSE ){
        push( @arrObjsListTmp, @{ $blkCurrentPar->getEndAnchorsList() } );
    }

    if ( @arrObjsListTmp <= 0 ){
        if ( $bLog == $TRUE ){
            $theLogger->logMessage(   "Error! No object found on table" );
        }
        return -2;
    }

    #---- Fetch columns X positions
    my $hrefColsTmp = $contDefTmp->getColumns();
    my @arrColsTmp = sort { $hrefColsTmp->{ $a } <=> $hrefColsTmp->{ $b } } keys( %{ $hrefColsTmp } );
    my @arrColXPosTmp = map { $hrefColsTmp->{ $_ } } @arrColsTmp;

    #---- Detect first row Y position
    my $iCurrentYTmp = 0;
    my $iFontHeightTmp = 0;
    foreach my $a2wFirstCellTmp ( @arrObjsListTmp ){
        if (    $a2wFirstCellTmp != undef
             && $outVisitorPar->isWritable( $a2wFirstCellTmp ) == $TRUE
             && $a2wFirstCellTmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJTYPE } != $a2w::core::dm::Constants::OT_IMAGE
             && $a2wFirstCellTmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJTYPE } != $a2w::core::dm::Constants::OT_LINE
           ){
            $iCurrentYTmp = $a2wFirstCellTmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_YPOS };
            my $a2wFontTmp = $a2wFirstCellTmp->{ 'A2WOBJ' }->getFont();
            if ( $a2wFontTmp != undef ){
                $iFontHeightTmp = $a2wFontTmp->getHeight();
                $iFontHeightTmp *= $a2wFirstCellTmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_RESOLUTION } / 72;
            }
            last;
        }
    }

    #---- Begin Table
    my $iRetTmp = $outVisitorPar->beginTable( $blkCurrentPar, \@arrColXPosTmp, ( $iCurrentYTmp - $iFontHeightTmp ) );
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    #---- Write header row, if defined in config
    if ( $contDefTmp->isHeaderPredefined() == $TRUE ){
        #---- Start header
        $iRetTmp = $outVisitorPar->beginTableHeader();
        if ( $iRetTmp < 0 ){ return $iRetTmp; }

        #---- Write a row ----#
        $iRetTmp = $outVisitorPar->beginRow( 0 );    # Row count
        if ( $iRetTmp < 0 ){ return $iRetTmp; }

        #---- Write columns
        my @arrHeaderTmp = @{ $contDefTmp->getHeader() };
        my @arrCellTmp = ();
        for ( my $i = 0; $i < @arrHeaderTmp; $i++ ){
            @arrCellTmp[ 0 ] = @arrHeaderTmp[ $i ];
            $iRetTmp = $outVisitorPar->writeCell( \@arrCellTmp, @arrHeaderTmp[ $i ], @arrColsTmp );
            if ( $iRetTmp < 0 ){ return $iRetTmp; }
        }

        $iRetTmp = $outVisitorPar->endRow( 0 );    # Row count
        if ( $iRetTmp < 0 ){ return $iRetTmp; }

        #---- End header
        $iRetTmp = $outVisitorPar->endTableHeader();
        if ( $iRetTmp < 0 ){ return $iRetTmp; }
    }

    #---- Write formatted content of block ----#
    $iRetTmp = $this->_writeFormattedContent( # Block
                                                $blkCurrentPar
                                              # Visitor
                                              , $outVisitorPar
                                              # Row count
                                              , 0
                                            );
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    #---- End body
    $iRetTmp = $outVisitorPar->endTableBody();
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    #---- Begin footer
    #$iRetTmp = $outVisitorPar->beginTableFooter();
    #if ( $iRetTmp < 0 ){ return $iRetTmp; }
    #....
    #---- End footer
    #$iRetTmp = $outVisitorPar->endTableFooter();
    #if ( $iRetTmp < 0 ){ return $iRetTmp; }

    #---- End Table
    $iRetTmp = $outVisitorPar->endTable( $blkCurrentPar );
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    return 0;
}

#-----------------------------------------------------------------------
# _writeFormattedContent
#
# Writes block content formatted as table rows/columns (low level implementation)
#
# >=0, in case of successful writing
# < 0, in case of error
#
#-----------------------------------------------------------------------
sub _writeFormattedContent{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "_writeFormattedContent()" );
    }

    #---- Parameter
    #
    # 1. Block
    # 2. Visitor
    # 3. Row count
    #
    my $blkCurrentPar = shift;
    my $outVisitorPar = shift;
    my $iRowCountPar  = shift;
    if ( $bLog == $TRUE ){
        $theLogger->logMessage( "Block: Id=>" . $blkCurrentPar->getId() . "<" );
    }

    #---- Set flushed
    $blkCurrentPar->setFlushed( $TRUE );

    #---- Fetch content definition of block
    my $contDefTmp = $blkCurrentPar->getContentDef();
    if ( lc( $contDefTmp->getType() ) ne "table" ){
        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "Warning! Table block formatter can not process other type of blocks" );
        }
        return -1;
    }

    #---- Fetch columns definition
    my $hrefColsTmp = $contDefTmp->getColumns();
    my @arrColsTmp = sort { $hrefColsTmp->{ $a } <=> $hrefColsTmp->{ $b } } keys( %{ $hrefColsTmp } );

    # $V102 Begin
    #---- Hash of row
    my %hshRowTmp = map { $_, undef } @arrColsTmp;
    my $hrefCurrentRowTmp = \%hshRowTmp;
    # $V102 End

    #---- Assert block has content or not ----#
    my @arrObjsListTmp = ();

    # Check and add start anchor objects
    my $ancStTmp = $blkCurrentPar->getStartAnchor();
    if ( $ancStTmp != undef && $ancStTmp->isSkipped() == $FALSE ){
        push( @arrObjsListTmp, @{ $blkCurrentPar->getStartAnchorsList() } );
    }
	
    # Add Block objects
    push( @arrObjsListTmp, @{ $blkCurrentPar->getObjsList() } );
	
    # Check and add end anchor objects
    my $ancEdTmp = $blkCurrentPar->getEndAnchor();
    if ( $ancEdTmp != undef && $ancEdTmp->isSkipped() == $FALSE ){
        push( @arrObjsListTmp, @{ $blkCurrentPar->getEndAnchorsList() } );
    }

    if ( @arrObjsListTmp <= 0 ){
        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "Error! No object found on table" );
        }
        return -2;
    }

    #---- Detect first row Y position
    my $iCurrentYTmp = 0;
    foreach my $a2wFirstCellTmp ( @arrObjsListTmp ){
        if (    $a2wFirstCellTmp != undef
             && $outVisitorPar->isWritable( $a2wFirstCellTmp ) == $TRUE
             # $V101 Begin
             && $this->_isSupported( $a2wFirstCellTmp ) == $TRUE
             # $V101 End
           ){
            $iCurrentYTmp = $a2wFirstCellTmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_YPOS };
            last;
        }
    }

    my $iRetTmp = -1;
    my $a2wObjTmp = undef;
    my $pomObjTmp = undef;
    my $iRowCountTmp = $iRowCountPar;
    my $sColTmp = '';
    my $bEmptyRowTmp = $FALSE;
    my @arrHeaderTmp = @{ $contDefTmp->getHeader() };
    # $V102 Begin
    my $bEndTableHeaderTmp = $FALSE;
    my $hrefPreviousRowTmp = undef;
    # $V102 End

    foreach my $elemObjTmp ( @arrObjsListTmp ){
        $a2wObjTmp = $elemObjTmp->{ 'A2WOBJ' };
        $pomObjTmp = $elemObjTmp->{ 'POMOBJ' };

        #---- Assert whether visitor supports this object type ----#
        if (    $outVisitorPar->isWritable( $elemObjTmp ) == $FALSE
             # $V101 Begin
             || $this->_isSupported( $elemObjTmp ) == $FALSE
             # $V101 End
           ){
            next;    # go to next object
        }

        #---- Write row, if collected ----#
        if ( $iCurrentYTmp != $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS } ){
            #---- Update row position
            $iCurrentYTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS };

            # $V102 Begin
            # Delay first line writing till row is fully found
            if ( $hrefPreviousRowTmp == undef ){
                if ( $bLog == $TRUE ){
                    $theLogger->logMessage( "First Line" );
                }
                $hrefPreviousRowTmp = $hrefCurrentRowTmp;

                # Clean current row
                my %hshRowTmp = map { $_, undef } @arrColsTmp;
                $hrefCurrentRowTmp = \%hshRowTmp;
            } # if ( $hrefPreviousRowTmp == undef )

            # Following line of current row
            elsif ( $hrefCurrentRowTmp->{ @arrColsTmp[ 0 ] } == undef ){
                if ( $bLog == $TRUE ){
                    $theLogger->logMessage( "Following Line" );
                }
                #---- Merge current line with current row
                my @arrElementsTmp = ();
                for ( my $c = 0; $c < @arrColsTmp; $c++ ){
                    @arrElementsTmp = @{ $hrefCurrentRowTmp->{ @arrColsTmp[ $c ] } };
                    for ( my $e = 0; $e < @arrElementsTmp; $e++ ){
                        $hrefPreviousRowTmp->{ @arrColsTmp[ $c ] }[ @{ $hrefPreviousRowTmp->{ @arrColsTmp[ $c ] } } ] = @arrElementsTmp[ $e ];
                    }
                }

                # Clean current row
                my %hshRowTmp = map { $_, undef } @arrColsTmp;
                $hrefCurrentRowTmp = \%hshRowTmp;
            } # elsif ( $hrefCurrentRowTmp->{ @arrColsTmp[ 0 ] } == undef )

            # New row found
            elsif ( $hrefCurrentRowTmp->{ @arrColsTmp[ 0 ] } != undef ){
                if ( $bLog == $TRUE ){
                    $theLogger->logMessage( "Row found" );
                }
                #---- Row found, write it ----#
                $iRowCountTmp++;

                if ( $iRowCountTmp == 1 ){
                    if ( $contDefTmp->isFirstRowHeader() == $TRUE ){
                        #---- Start header
                        $iRetTmp = $outVisitorPar->beginTableHeader();
                        $bEndTableHeaderTmp = $TRUE;
                    }
                    else {
                        #---- Start body
                        $iRetTmp = $outVisitorPar->beginTableBody();
                    }
                    if ( $iRetTmp < 0 ){ return $iRetTmp; }
                }
                # $V103 Begin
                elsif ( $iRowCountTmp == 2 ){
                    if ( $contDefTmp->isFirstRowHeader() == $TRUE ){
                        #---- Start body
                        $iRetTmp = $outVisitorPar->beginTableBody();
                    }
                    if ( $iRetTmp < 0 ){ return $iRetTmp; }
                }
                # $V103 End

                #---- Write a row
                $iRetTmp = $this->_writeRow( # Visitor
                                               $outVisitorPar
                                             # Row
                                             , $hrefPreviousRowTmp
                                             # Row number
                                             , $iRowCountTmp
                                             # Column names
                                             , \@arrColsTmp
                                             # Column header
                                             , \@arrHeaderTmp
                                             # Column positions
                                             , $hrefColsTmp
                                           );
                if ( $iRetTmp < 0 ){ return $iRetTmp; }

                if ( $bEndTableHeaderTmp == $TRUE ){
                    #---- End header
                    $bEndTableHeaderTmp = $FALSE;
                    $iRetTmp = $outVisitorPar->endTableHeader();
                    if ( $iRetTmp < 0 ){ return $iRetTmp; }
                }

                # Mark current as previous
                $hrefPreviousRowTmp = $hrefCurrentRowTmp;

                # Clean current row
                my %hshRowTmp = map { $_, undef } @arrColsTmp;
                $hrefCurrentRowTmp = \%hshRowTmp;
            } # elsif ( $hrefCurrentRowTmp->{ @arrColsTmp[ 0 ] } != undef )
            # $V102 End
        } # if ( $iCurrentYTmp != $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS } )

        #---- Detect to which column the current object belongs to
        for ( my $i = @arrColsTmp - 1; $i >= 0; $i-- ){
            if ( $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS } >= $hrefColsTmp->{ @arrColsTmp[ $i ] } ){
                $hrefCurrentRowTmp->{ @arrColsTmp[ $i ] }[ @{ $hrefCurrentRowTmp->{ @arrColsTmp[ $i ] } } ] = $elemObjTmp;
                $iRetTmp = 0;
                if (    $bLog == $TRUE
                     && $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJTYPE } == $a2w::core::dm::Constants::OT_TEXT
                   ){
                    $theLogger->logMessage( @arrColsTmp[ $i ] . ": Element:>" . $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_VALUE } . "<" );
                }
                last;    # break the loop
            }
        } # for ( my $i = @arrColsTmp - 1; $i >= 0; $i-- ){
    } # foreach my $a2wObjTmp ( @arrObjsListTmp )

    # $V102 Begin
    #---- Write last rows ----#
    # $V104 Begin
    if ( $iRowCountTmp <= 0 ){ $iRowCountTmp = 1; }
    # $V104 End

    # Following line of current row
    if ( $hrefCurrentRowTmp->{ @arrColsTmp[ 0 ] } == undef ){
        #---- Merge current line with current row
        my @arrElementsTmp = ();
        for ( my $c = 0; $c < @arrColsTmp; $c++ ){
            @arrElementsTmp = @{ $hrefCurrentRowTmp->{ @arrColsTmp[ $c ] } };
            for ( my $e = 0; $e < @arrElementsTmp; $e++ ){
                $hrefPreviousRowTmp->{ @arrColsTmp[ $c ] }[ @{ $hrefPreviousRowTmp->{ @arrColsTmp[ $c ] } } ] = @arrElementsTmp[ $e ];
            }
        }
    }
    else {
        # $V103 Begin
        if ( $iRowCountTmp == 1 ){
            if ( $contDefTmp->isFirstRowHeader() == $TRUE ){
                #---- Start header
                $iRetTmp = $outVisitorPar->beginTableHeader();
                $bEndTableHeaderTmp = $TRUE;
            }
            else {
                #---- Start body
                $iRetTmp = $outVisitorPar->beginTableBody();
            }
            if ( $iRetTmp < 0 ){ return $iRetTmp; }
        }
        elsif ( $iRowCountTmp == 2 ){
            if ( $contDefTmp->isFirstRowHeader() == $TRUE ){
                #---- Start body
                $iRetTmp = $outVisitorPar->beginTableBody();
            }
            if ( $iRetTmp < 0 ){ return $iRetTmp; }
        }
        # $V103 End

        #---- Write a row
        $iRowCountTmp++;
        $iRetTmp = $this->_writeRow( # Visitor
                                       $outVisitorPar
                                     # Row
                                     , $hrefPreviousRowTmp
                                     # Row number
                                     , $iRowCountTmp
                                     # Column names
                                     , \@arrColsTmp
                                     # Column header
                                     , \@arrHeaderTmp
                                     # Column positions
                                     , $hrefColsTmp
                                   );
        if ( $iRetTmp < 0 ){ return $iRetTmp; }

        # $V103 Begin
        if ( $bEndTableHeaderTmp == $TRUE ){
            #---- End header
            $bEndTableHeaderTmp = $FALSE;
            $iRetTmp = $outVisitorPar->endTableHeader();
            if ( $iRetTmp < 0 ){ return $iRetTmp; }
        }
        # $V103 End

        $hrefPreviousRowTmp = $hrefCurrentRowTmp;
    }

    # $V103 Begin
    if ( $iRowCountTmp == 1 ){
        if ( $contDefTmp->isFirstRowHeader() == $TRUE ){
            #---- Start header
            $iRetTmp = $outVisitorPar->beginTableHeader();
            $bEndTableHeaderTmp = $TRUE;
        }
        else {
            #---- Start body
            $iRetTmp = $outVisitorPar->beginTableBody();
        }
        if ( $iRetTmp < 0 ){ return $iRetTmp; }
    }
    elsif ( $iRowCountTmp == 2 ){
        if ( $contDefTmp->isFirstRowHeader() == $TRUE ){
            #---- Start body
            $iRetTmp = $outVisitorPar->beginTableBody();
        }
        if ( $iRetTmp < 0 ){ return $iRetTmp; }
    }
    # $V103 End

    #---- Write a row
    $iRowCountTmp++;
    $iRetTmp = $this->_writeRow( # Visitor
                                   $outVisitorPar
                                 # Row
                                 , $hrefPreviousRowTmp
                                 # Row number
                                 , $iRowCountTmp
                                 # Column names
                                 , \@arrColsTmp
                                 # Column header
                                 , \@arrHeaderTmp
                                 # Column positions
                                 , $hrefColsTmp
                               );
    if ( $iRetTmp < 0 ){ return $iRetTmp; }
    # $V102 End

    # $V103 Begin
    if ( $bEndTableHeaderTmp == $TRUE ){
        #---- End header
        $bEndTableHeaderTmp = $FALSE;
        $iRetTmp = $outVisitorPar->endTableHeader();
        if ( $iRetTmp < 0 ){ return $iRetTmp; }
    }
    # $V103 End

    #---- Write chained block contents ----#
    my $blkNextTmp = $blkCurrentPar->getNextRef();
    if ( $blkNextTmp != undef ){
        #---- Check and create formatter
        my $fmtNextTmp = $blkNextTmp->getFormatter();
        if ( $fmtNextTmp == undef ){
            $blkNextTmp->_createContentFormatter();
            $fmtNextTmp = $blkNextTmp->getFormatter();
        }

        #---- Write next block
        $iRetTmp = $fmtNextTmp->_writeFormattedContent( # Block
                                                          $blkNextTmp
                                                        # Visitor
                                                        , $outVisitorPar
                                                        # Row count
                                                        , $iRowCountTmp
                                                      );
        $blkNextTmp->setFlushed( $TRUE );
        if ( $iRetTmp < 0 ){ return $iRetTmp; }
    }

    return 0;
}

#-----------------------------------------------------------------------
# _getCellContentWidth
#
# Evaluates cell width based on contents and returns the same
#
# >=0, in case of successful evaluation
# < 0, in case of error
#
#-----------------------------------------------------------------------
sub _getCellContentWidth{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "_getCellContentWidth()" );
    }

    #---- Parameter
    #
    # 1. Cell start X
    # 2. Cell objects (Array reference)
    #
    my $iStartXPar = shift;
    my $arefObjectsPar = shift;

    #---- Fetch objects list
    my @arrObjsTmp = @{ $arefObjectsPar };

    #TODO: Evaluate cell width to determine column span of objects

    return 0;
}

# $V101 Begin
#-----------------------------------------------------------------------
# _isSupported
#
# Asserts whether given object is supported to be table content
#
# Returns
# TRUE means object can be table content
# FALSE means object can not be table content
#
#-----------------------------------------------------------------------
sub _isSupported{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "_isSupported()" );
    #}

    #---- Fetch parameter(s)
    #
    # 1. Object hash (having both kernel object and pom object)
    #
    my $hrefObjPar = shift;

    #---- Assert object
    my $a2wObjTmp = $hrefObjPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefObjPar->{ 'POMOBJ' };
    my $bRetTmp   = $TRUE;

    my $iObjTypeTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJTYPE };
    unless ( $iObjTypeTmp == $a2w::core::dm::Constants::OT_TEXT ){
        $bRetTmp = $FALSE;
    }

    return $bRetTmp;
}
# $V101 End

# $V102 Begin
#-----------------------------------------------------------------------
# _writeRow
#
# write row of table
#
# Returns
# >=0 in case of success
# <0 in case of error
#
#-----------------------------------------------------------------------
sub _writeRow{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "_writeRow()" );
    #}

    #---- Fetch parameter(s)
    #
    # 1. Visitor (a2w::core::visitor::Visitor object)
    # 2. Row cells (Hash Reference)
    # 3. Row number (integer)
    # 4. Column names (Array reference)
    # 4. Column header (Array reference)
    # 4. Column positions (Hash reference)
    #
    my $outVisitorPar   = shift;
    my $hrefRowPar      = shift;
    my $iRowNrPar       = shift;
    my $arefColNamesPar = shift;
    my $arefColHdrPar   = shift;
    my $hrefColPosPar   = shift;

    #---- Write a row ----#
    my $iRetTmp = $outVisitorPar->beginRow( $iRowNrPar );
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    #---- Write cells
    my $sColTmp = '';
    my $iCellWidthTmp = 0;
    my @arrColsTmp = @{ $arefColNamesPar };
    my @arrColHdrTmp = @{ $arefColHdrPar };
    for ( my $i = 0; $i < @arrColsTmp; $i++ ){
        $sColTmp = @arrColsTmp[ $i ];

        #---- Evaluate cell width
        #$iCellWidthTmp = $this->_getCellContentWidth( $hrefColPosPar->{ $sColTmp }, $hrefRowPar->{ $sColTmp } );

        #---- Write cell
        $iRetTmp = $outVisitorPar->writeCell( $hrefRowPar->{ $sColTmp }, @arrColHdrTmp[ $i ], $sColTmp );
        if ( $iRetTmp < 0 ){ return $iRetTmp; }
        $hrefRowPar->{ $sColTmp } = undef; # Reset cell value
    }

    #---- End a row
    $iRetTmp = $outVisitorPar->endRow( $iRowNrPar );
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    return 0;
}
# $V102 End

#-----------------------------------------------------------------------
# Don't remove the following lines !!!
#-----------------------------------------------------------------------
1;
__END__
