#-------------------------------------------------------------------------------
#  a2w::core::visitor::HTMLVisitor.pm:
#  PERL module to write the objects as HTML
#
#  Author  : AFP2web Team, Maas Holding GmbH
#
#  $V100   2013-10-11    Initial Release
#
#  $V101   2015-05-20    Fixed line overlapping image (within block) issue (AFP-224)
#
#  $V102   2015-07-17    Extended to inline CSS and image data (AFP-263)
#
#  $V103   2015-09-28    - Extended to group texts based on font (identifier) (AFP-297)
#                        - Skipped writing empty <div> for page, when page has no content
#                        - Modified following page's top margin as 5mm (from 10mm)
#
#  $V104   2015-10-07    Issue:
#                        Table cell having multi line value occur now in multiple rows
#
#                        Reason:
#                        Table rows are identified based on change in Y position, so each line of
#                        same cell is treated as different row.
#
#                        Fix:
#                        Take first column as primary and mark beginning of a new row whenever the
#                        first column value on line is not empty. i.e, Following lines of a cell must
#                        have empty first column value on line
#
#                        JiRa:
#                        AFP-224
#
#                        Fixed by, Date:
#                        Panneer, 07/10/2015
#
#  $V105   2015-10-15    Extended to pass in custom block info like HTML style (AFP-298)
#
#  $V106   2015-11-04    - Extended to position content blocks absolute
#                        - Removed line breaks
#                        - Extended with "getTableHeight" function
#
#  $V107   2015-11-16    - Fixed minor disposition (due to default margin) issue with lines. To fix, Forced line margin as zero. (AFP-320)
#                        - Fixed minor disposition issue with texts. To fix, Adjusted position based on text applied font height. (AFP-320)
#
#  $V108   2015-12-08    - Forced table first column vertical alignment as "top" (AFP-320)
#
#  $V109   2015-12-09    - Fixed minor disposition (due to missing page margins). (AFP-320)
#                        - Fixed minor disposition (due to skipped block height not considered for following block top). (AFP-320)
#
#  $V110   2015-12-11    - Modified not to escape space (' ') character as HTML entity (&nbsp;) (AFP-320)
#
#  $V111   2016-01-18    Customized to wrap block images with <a> tag when block info (TLE value) has "href=<url>" (AFP-342)
#
#-------------------------------------------------------------------------------
package a2w::core::visitor::HTMLVisitor;

#-----------------------------------------------------------------------
# Include required modules
#-----------------------------------------------------------------------
use a2w::Font;
use a2w::Text;
use a2w::Line;
use a2w::Image;
use a2w::Vector;

use a2w::TypeConstants;
use a2w::core::log::Logger;
use a2w::core::dm::Constants;
use a2w::core::visitor::Visitor;
# $V102 Begin
use a2w::core::utils::Base64;
# $V102 End

#-----------------------------------------------------------------------
# Inherit from base class
#-----------------------------------------------------------------------
our @ISA = qw( a2w::core::visitor::Visitor );

#-----------------------------------------------------------------------
# Constructor
#-----------------------------------------------------------------------
sub new{
    my $proto = shift;
    my $class = ref( $proto ) || $proto;

    #---- Define boolean values
    $TRUE  = $a2w::TypeConstants::TRUE;     # TRUE  boolean value
    $FALSE = $a2w::TypeConstants::FALSE;    # FALSE boolean value

    #---- Instantiate from base class to inherit base class attributes
    my $this = a2w::core::visitor::Visitor->new( @_ );

    #---- Add this derived class specific attributes
    $this->{ 'BackgroundFilename' } = "";        # Background filename
    $this->{ 'FontTable' }          = undef;     # Font table
    $this->{ 'BODY' }               = undef;     # Body of html
    $this->{ 'BPG_TAG' }            = undef;     # Page begin tag # $V103 Change
    $this->{ 'Content' }            = undef;     # Content of page
    $this->{ 'FontScale' }          = 0;         # Font scale factor
    $this->{ 'UnitBase' }           = "";        # Units base (possible values: pixel, mm)
    $this->{ 'PrevFontId' }         = 0;         # Previous font local id
    $this->{ 'CSS' }                = undef;     # Cascaded style sheet for current document (Hash having css file name, handle, style classes etc)
    $this->{ 'ResPaths' }           = undef;     # Hash of resource paths for image, css etc
    # $V102 Begin
    $this->{ 'bInlineImage' }       = $FALSE;    # Flag to inline image or not
    $this->{ 'bInlineCSS' }         = $FALSE;    # Flag to inline CSS or not
    # $V102 End
    # $V106 Begin
    $this->{ 'TableHdrHeight' }     = 0;         # Active Table header height
    $this->{ 'TableRowHeight' }     = 0;         # Active Table row height
    $this->{ 'TableRowSpan' }       = 0;         # Active Table row span count (multi line cell)
    $this->{ 'TableCellPadding' }   = 2;         # Table cell padding
    $this->{ 'BlockTop' }           = 0;         # Current block top
    $this->{ 'BlockTopDiff' }       = 0;         # Block top difference
    $this->{ 'PrevBlock' }          = undef;     # Previous block
    $this->{ 'FirstBlock' }         = undef;     # First block of page
    # $V106 End

    bless( $this, $class );

    #---- Initialize constants ----#
    $this->{ 'ResPaths' } = {
          'CSS'   => 'css/'
        , 'IMAGE' => 'images/'
        , 'VIDEO' => 'videos/'
	};

    # $V102 Begin
    #--- Force inlining image and css
    $this->{ 'bInlineImage' } = $TRUE;
    $this->{ 'bInlineCSS' }   = $TRUE;
    # $V102 End

    #---- Get logger
    our $theLogger = a2w::core::log::Logger->getSingleton();
    our $bLog = $theLogger->isRegistered( __PACKAGE__ );

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "new()" );
    #}
    return $this;
}

#-----------------------------------------------------------------------
# Destructor
#-----------------------------------------------------------------------
sub DESTROY{
    my $this = shift;
}

#-----------------------------------------------------------------------
# Mutator
#-----------------------------------------------------------------------
sub setBackgroundFilename{
    my $this = shift;

    $this->{ 'BackgroundFilename' } = shift;
}

sub setUnitBase{
    my $this = shift;

    $this->{ 'UnitBase' } = shift;
    if ( lc( $this->{ 'UnitBase' } ) eq "pixel" ){
        $this->{ 'UnitBase' } = "px";
    }
    elsif ( lc( $this->{ 'UnitBase' } ) eq "mm" ){
        $this->{ 'UnitBase' } = "mm";
    }
    else {
        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "Warning! Invalid unit base (" . $this->{ 'UnitBase' } . ") using default px" );
        }
        $this->{ 'UnitBase' } = "px";
    }
}

#-----------------------------------------------------------------------
# Accessor
#-----------------------------------------------------------------------
sub getBackgroundFilename{
    my $this = shift;

    return $this->{ 'BackgroundFilename' };
}

sub getUnitBase{
    my $this = shift;

    return $this->{ 'UnitBase' };
}

#-----------------------------------------------------------------------
# Workers
#-----------------------------------------------------------------------
#-----------------------------------------------------------------------
# Initialize
#-----------------------------------------------------------------------
sub initialize{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "initialize()" );
    }

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::initialize( @_ );
    if ( $iRetTmp < 0 ){
        return $iRetTmp;
    }

    #---- Initialize CSS object
    $this->{ 'CSS' } = {
          'Filename' => ''
        , 'fHandle'  => undef
        , 'Styles'   => undef
        , 'FontId'   => undef
    };

    if ( $this->{ 'PageOutput' } == $FALSE ){
        $this->_initializeHTML();
    }

    return 0;
}

#-----------------------------------------------------------------------
# Finalize
#-----------------------------------------------------------------------
sub finalize{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "finalize()" );
    }

    #---- Fetch parameters
    #
    # 1. Additional content for body end (optional)
    # 2. Additional content for footer (optional)
    #
    my $sBodyAddContentPar   = "";
    my $sAddFooterContentPar = "";
    if ( @_ > 0 ){
        $sBodyAddContentPar = shift;
    }
    if ( @_ > 0 ){
        $sAddFooterContentPar = shift;
    }

    if ( $this->{ 'PageOutput' } == $FALSE ){
        #---- End body
        my $handleFileTmp = $this->{ 'fHandle' };
        if ( $sBodyAddContentPar ne "" ){
            printf $handleFileTmp ( "\n" . $sBodyAddContentPar );
        }
        # $V106 Begin
        printf $handleFileTmp ( "\n" . '</div>' );
        # $V106 End
        printf $handleFileTmp ( "\n" . '</body>' );

        #---- Write footer
        $this->writeFooter( # Additional content
                            $sAddFooterContentPar
                          );

        $this->_finalizeHTML();
    }

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::finalize( @_ );
    if ( $iRetTmp < 0 ){
        return $iRetTmp;
    }

    #---- Clean up image/css sub directories
	if ( $this->{ 'bInlineImage' } == $TRUE ){
	    my $sImgOutputPathTmp = $this->{ 'OutputPath' } . $this->{ 'ResPaths' }{ 'IMAGE' };
	    rmdir( $sImgOutputPathTmp );
	}
	if ( $this->{ 'bInlineCSS' } == $TRUE ){
	    my $sCssOutputPathTmp = $this->{ 'OutputPath' } . $this->{ 'ResPaths' }{ 'CSS' };
	    rmdir( $sCssOutputPathTmp );
	}
    
    return 0;
}

#-----------------------------------------------------------------------
# Write header
#-----------------------------------------------------------------------
sub writeHeader{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeHeader()" );
    }

    #---- Parameter(s)
    #
    # 1. Title
    # 2. Additional content
    #
    my $sTitlePar      = shift;
    my $sAddContentPar = shift;

    #---- Write HTML header
    my $handleFileTmp = $this->{ 'fHandle' };
    printf $handleFileTmp (   "\n" . '<head>'
                            . "\n" . '<title>' . $sTitlePar . '</title>'
                            . "\n" . '<meta http-equiv="Content-Type" content="text/html; charset=Windows-1252">'
                          );

    #---- Create CSS ----#
    my $iRetTmp = 0;
    if ( $this->{ 'CSS' }{ 'fHandle' } != undef ){
        $iRetTmp = $this->_writeCSSHeader();
        my $CSSFilenameTmp = $this->{ 'CSS' }{ 'Filename' };
        if ( $CSSFilenameTmp =~ /.*\/(css\/.*)/ ){
            $CSSFilenameTmp = $1;
        }
        printf $handleFileTmp ( "\n" . '<link rel="stylesheet" type="text/css" href="' . $CSSFilenameTmp . '" />' );
        # $V102 Begin
        $this->{ 'CSS' }{ 'IncludeStatement' } = '<link rel="stylesheet" type="text/css" href="' . $CSSFilenameTmp . '" />';
        # $V102 End
    }

    if ( $sAddContentPar ne "" ){
        printf $handleFileTmp ( "\n" . $sAddContentPar );
    }
    printf $handleFileTmp ( "\n" . '</head>' );

    return 0;
}

#-----------------------------------------------------------------------
# Write footer
#-----------------------------------------------------------------------
sub writeFooter{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeFooter()" );
    }

    #---- Parameter(s)
    #
    # 1. Additional content
    #
    my $sAddContentPar = shift;
    if ( $sAddContentPar eq "" ){
        return 0;
    }

    #---- Write HTML footer
    my $handleFileTmp = $this->{ 'fHandle' };
    printf $handleFileTmp ( "\n" . '<footer>' );
    if ( $sAddContentPar ne "" ){
        printf $handleFileTmp ( "\n" . $sAddContentPar );
    }
    printf $handleFileTmp ( "\n" . '</footer>' );

    return 0;
}

#-----------------------------------------------------------------------
# Initialize Page
#-----------------------------------------------------------------------
sub initializePage{
    my $this = shift;

    #---- Fetch parameters
    #
    # 1. Database
    # 2. Title
    # 3. Additional content for header (optional)
    # 4. Additional content for before body (optional)
    #
    my $dbPagePar = shift;
    my $sTitlePar = shift;
    my $sAddHeaderContentPar = "";
    my $sAddBodyContentPar   = "";
    if ( @_ > 0 ){
        $sAddHeaderContentPar = shift;
    }
    if ( @_ > 0 ){
        $sAddBodyContentPar = shift;
    }
    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "initializePage(" . $dbPagePar->getPageID() . ")" );
    }

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::initializePage( $dbPagePar, $sAddHeaderContentPar );
    if ( $iRetTmp < 0 ){
        return $iRetTmp;
    }

    #---- Evaluate page width/height ----#
    #---- Evaluate scale factors
    my $iOutputResTmp         = $this->getOutputRes();
    $this->{ 'ContentScale' } = $iOutputResTmp / $dbPagePar->getPageRes();
    $this->{ 'FontScale' }    = $iOutputResTmp / 72;
    if ( $bLog == $TRUE ){
        $theLogger->logMessage( "Scale: Content=" . $this->{ 'ContentScale' } . " Font=" . $this->{ 'FontScale' } );
    }

    #---- Fetch page info
    my $iPageIdTmp     = $dbPagePar->getPageID();
    my $iPageWidthTmp  = $dbPagePar->getPageWidth();
    my $iPageHeightTmp = $dbPagePar->getPageHeight();
    if ( $this->{ 'UnitBase' } eq "px" ){
        $iPageWidthTmp  *= $this->{ 'ContentScale' };
        $iPageHeightTmp *= $this->{ 'ContentScale' };
    }

    #---- Activate current page
    $this->activatePage( $iPageIdTmp );

    #---- Store the page info on page context
    my $pgHandleTmp = $this->{ 'ActivePage' };
    $this->{ 'PrevFontId' } = 0;

    $pgHandleTmp->{ 'ContentScale' } = $this->{ 'ContentScale' };
    $pgHandleTmp->{ 'Width' }        = int( $iPageWidthTmp  + .5 * ( $iPageWidthTmp  <=> 0 ) );
    $pgHandleTmp->{ 'Height' }       = int( $iPageHeightTmp + .5 * ( $iPageHeightTmp <=> 0 ) );

    my $sPageBackgroundTmp = $dbPagePar->getPageBackground();
    if ( $sPageBackgroundTmp ne "" ){
        $this->setBackgroundFilename( $sPageBackgroundTmp );
    }

    if ( $this->{ 'PageOutput' } == $TRUE ){
        #---- Write html start
        $this->_initializeHTML();

        #---- Write header
        $this->writeHeader( # Title
                              $sTitlePar
                            # Additional content
                            , $sAddHeaderContentPar
                          );
        $this->{ 'BODY' } =   "\n"
                            . '<body>'
                            # $V106 Begin
                            . "\n"
                            . '<div id="container">';
                            # $V106 End
        if ( $sAddBodyContentPar ne "" ){
            $this->{ 'BODY' } .= "\n" . $sAddBodyContentPar;
        }
    }
    else {
        if ( $this->{ 'PageCount' } == 1 ){
            #---- Write header
            $this->writeHeader( # Title
                                  $sTitlePar
                                # Additional content
                                , $sAddHeaderContentPar
                              );

            #---- Start body
            my $handleFileTmp = $this->{ 'fHandle' };
            printf $handleFileTmp (   "\n"
                                    . '<body>'
                                    # $V106 Begin
                                    . "\n"
                                    . '<div id="container">'
                                    # $V106 End
                                  );

            if ( $sAddBodyContentPar ne "" ){
                printf $handleFileTmp ( "\n" . $sAddBodyContentPar );
            }
        }
        $this->{ 'BODY' } = "";
    }

    # $V106 Begin
    #---- Get page top
    my $iTopTmp = $this->_getPageTop( $iPageIdTmp )
                  + ( ( $this->{ 'FollowingPageTopMargin' } / 25.4 ) * $iOutputResTmp );
    if ( $this->{ 'UnitBase' } eq "px" ){
        $iTopTmp *= $this->{ 'ContentScale' };
    }
    $iTopTmp = int( $iTopTmp + .5 * ( $iTopTmp <=> 0 ) );
    $this->{ 'BlockTop' }     = 0;
    $this->{ 'BlockTopDiff' } = 0;
    $this->{ 'PrevBlock' }    = undef;
    $this->{ 'FirstBlock' }   = undef;
    # $V106 End

    #---- Start page div ----#
    $this->{ 'BPG_TAG' } =  "\n"
                            . '<div'
                            . ' id="page' . $this->{ 'PageCount' } . '"'
                            # $V106 Begin
                            . ' class="' . $this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' } . '"'
                            # $V106 End
                            . ' style="background-color:#' . sprintf( "%0X", $this->getDefBackColor() ) . ';'
                            # $V106 Begin
                            . (( $this->{ 'PageCount' } > 1 && $iTopTmp > 0 ) ? 'top:' . $iTopTmp . $this->{ 'UnitBase' } . ';' : "")
                            #. 'position:relative;'
                            # $V106 End
                            . 'width:' . $pgHandleTmp->{ 'Width' } . $this->{ 'UnitBase' } . ';'
                            #. 'height:' . $pgHandleTmp->{ 'Height' } . $this->{ 'UnitBase' } . ';'
                            . '">';
    # $V103 End
    $this->{ 'Content' } = "";
    $this->{ 'FontTable' } = undef;

    return 0;
}

#-----------------------------------------------------------------------
# Finalize page
#-----------------------------------------------------------------------
sub finalizePage{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "finalizePage()" );
    }

    #---- Fetch parameters
    #
    # 1. Additional content for body end (optional)
    # 2. Additional content for footer (optional)
    #
    my $sBodyAddContentPar   = "";
    my $sAddFooterContentPar = "";
    if ( @_ > 0 ){
        $sBodyAddContentPar = shift;
    }
    if ( @_ > 0 ){
        $sAddFooterContentPar = shift;
    }
    if ( $bLog == $TRUE ){
        $theLogger->logMessage( "Additional Content: Body=>" . $sBodyAddContentPar . "< Footer=>" . $sAddFooterContentPar . "<" );
    }

    #---- Build HTML body ----#
    #---- Get active page context
    my $pgHandleTmp = $this->{ 'ActivePage' };

    #---- Get active page
    my $dbPageTmp = $pgHandleTmp->{ 'POM' };

    # $V103 Begin
    if ( $this->{ 'Content' } ne "" ){
        $this->{ 'BODY' } .= $this->{ 'BPG_TAG' };
        $this->{ 'BPG_TAG' } = "";    # Reset value
    }
    # $V103 End

    #---- Add font table ----#
    #---- Build font table list
    my @arrFontTableListTmp = sort keys( %{ $this->{ 'FontTable' } } );
    if ( @arrFontTableListTmp > 0 ){
        if ( $this->{ 'CSS' }{ 'fHandle' } != undef ){
            my $hrefCSSTmp = $this->{ 'CSS' };
            my $fCSSHandleTmp = $this->{ 'CSS' }{ 'fHandle' };

            #---- Write font table entry
            printf $fCSSHandleTmp ( "\n\n/* Style for page (" . $dbPageTmp->getPageID() . ") */" );
            for ( my $i = 0; $i < @arrFontTableListTmp; $i++ ){
			    if ( $hrefCSSTmp->{ 'FontId' }{ $this->{ 'FontTable' }{ $arrFontTableListTmp[ $i ] }{ 'STYLENAME' } } == undef ){
                    $hrefCSSTmp->{ 'FontId' }{ $this->{ 'FontTable' }{ $arrFontTableListTmp[ $i ] }{ 'STYLENAME' } } = $TRUE;
                    printf $fCSSHandleTmp ( "\n" . $this->{ 'FontTable' }{ $arrFontTableListTmp[ $i ] }{ 'CSS' } );
                }
            }
        }
        else {
            #---- Begin font table style
            my $sFontTableTmp =   "\n" . '<style type="text/css" scoped>'
                                . "\n" . "<!--";
    
            #---- Write font table entry
            for ( my $i = 0; $i < @arrFontTableListTmp; $i++ ){
                $sFontTableTmp .= "\n" . $this->{ 'FontTable' }{ $arrFontTableListTmp[ $i ] }{ 'CSS' };
            }
    
            #---- End font table style
            $sFontTableTmp .=   "\n" . '-->'
                              . "\n" . '</style>';

            $this->{ 'BODY' } .= $sFontTableTmp;
        }
    }

    # $V103 Begin
    if ( $this->{ 'Content' } ne "" ){
        #---- Add background image
        if ( $this->{ 'BackgroundFilename' } ne "" ){
            $this->{ 'BODY' } .= "\n" . '<img width="' . $pgHandleTmp->{ 'Width' } . $this->{ 'UnitBase' }
                                      . '" height="' . $pgHandleTmp->{ 'Height' } . $this->{ 'UnitBase' }
                                      . '" src="' . $this->{ 'BackgroundFilename' }
                                      . '" alt="background image">';
        }

        #---- Add page content
        $this->{ 'BODY' } .= $this->{ 'Content' };

        #---- End page div
        $this->{ 'BODY' } .= "\n" . '</div>';
    }
    # $V103 End

    #---- End body
    if ( $this->{ 'PageOutput' } == $TRUE ){
        if ( $sBodyAddContentPar ne "" ){
            $this->{ 'BODY' } .= "\n" . $sBodyAddContentPar;
        }
        # $V106 Begin
        $this->{ 'BODY' } .= "\n" . '</div>';
        # $V106 End
        $this->{ 'BODY' } .= "\n" . '</body>';
    }

    #---- Write body
    my $handleFileTmp = $this->{ 'fHandle' };
    printf $handleFileTmp ( $this->{ 'BODY' } );

    if ( $this->{ 'PageOutput' } == $TRUE ){
        #---- Write footer
        $this->writeFooter( # Additional content
                            $sAddFooterContentPar
                          );

        #---- Write html end
        $this->_finalizeHTML();
    }

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::finalizePage( $sAddFooterContentPar );
    if ( $iRetTmp < 0 ){
        return $iRetTmp;
    }

    return 0;
}

#-----------------------------------------------------------------------
# Get File Extension
#
# Returns output file extension based on visitor type. Base returns empty
# string and concrete visitors MUST return appropriate extension
#
#-----------------------------------------------------------------------
sub getFileExtension{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "getFileExtension()" );
    }

    return ".html";
}

#-----------------------------------------------------------------------
# Update resource table
#-----------------------------------------------------------------------
sub updateResourceTable{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "updateResourceTable()" );
    }

    #---- Fetch parameters
    #
    # 1. Unique resource id
    # 2. Resource reference
    # 3. Object hash (having both kernel object and pom object)
    #
    my $iUniqueResIdPar = shift;
    my $a2wResourcePar  = shift;
    my $hrefObjectPar   = shift;

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::updateResourceTable( $iUniqueResIdPar, $a2wResourcePar, $hrefObjectPar );
    if ( $iRetTmp < 0 ){
        return $iRetTmp;
    }

    #---- Get actual objects
    my $a2wObjTmp = $hrefObjectPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefObjectPar->{ 'POMOBJ' };

    #---- Get active page context
    my $pgHandleTmp = $this->{ 'Pages' }{ $pomObjTmp->{ $a2w::core::dm::Constants::AT_PAGE_ID } };

    #---- Get active page
    my $dbPageTmp = $pgHandleTmp->{ 'POM' };

    #---- Get page resource table
    my $dbpResTableTmp = $dbPageTmp->getResourceTable();

    #---- Get resource table reference
    my $hrefResTableTmp = $this->{ 'FontTable' };

    #---- Build font class style and store it on font table
    my $sHTMLFontnameTmp   = $this->_fetchHTMLFont( $a2wResourcePar->getName() );
    $hrefResTableTmp->{ $iUniqueResIdPar } = {
          'STYLENAME' => "ft" . sprintf( "%03d", $iUniqueResIdPar )
        , 'BOLD'      => $dbpResTableTmp->{ $iUniqueResIdPar }{ 'FONT' }{ 'BOLD' }
        , 'ITALIC'    => $dbpResTableTmp->{ $iUniqueResIdPar }{ 'FONT' }{ 'ITALIC' }
        , 'HEIGHT'    => int( $dbpResTableTmp->{ $iUniqueResIdPar }{ 'FONT' }{ 'HEIGHT' } * $this->{ 'FontScale' } )
        , 'FAMILY'    => $sHTMLFontnameTmp
    };

    if ( $bLog == $TRUE ){
        $theLogger->logMessage(   "Font Details: Name=" . $a2wResourcePar->getName()
                                . " HTMLName=" . $sHTMLFontnameTmp
                                . " UniqueID=" . $iUniqueResIdPar
                                . " Height=" . $hrefResTableTmp->{ $iUniqueResIdPar }{ 'HEIGHT' }
                                . " Bold=" . $hrefResTableTmp->{ $iUniqueResIdPar }{ 'BOLD' }
                                . " Italic=" . $hrefResTableTmp->{ $iUniqueResIdPar }{ 'ITALIC' }
                              );
    }

    # $V106 Begin
    #my $sFormatTmp = ".ft%03d{font-size:%dpx;font-family:%s;";
    my $sFormatTmp = ".ft%03d{line-height:%dpx;font-size:%dpx;font-family:%s;";
    # $V106 End
    if ( $hrefResTableTmp->{ $iUniqueResIdPar }{ 'BOLD' } == $TRUE ){
        $sFormatTmp .= "font-weight:bold;";
    }
    if ( $hrefResTableTmp->{ $iUniqueResIdPar }{ 'ITALIC' } == $TRUE ){
        $sFormatTmp .= "font-style:italic;";
    }
    $sFormatTmp .= "}";

    $hrefResTableTmp->{ $iUniqueResIdPar }{ 'CSS' } = sprintf(   $sFormatTmp
                                                               , $iUniqueResIdPar
                                                               , $hrefResTableTmp->{ $iUniqueResIdPar }{ 'HEIGHT' }
                                                               # $V106 Begin
                                                               , $hrefResTableTmp->{ $iUniqueResIdPar }{ 'HEIGHT' }
                                                               # $V106 End
                                                               , $sHTMLFontnameTmp
                                                      );
    return 0;
}

#-----------------------------------------------------------------------
# isWritable
#
# Asserts whether given object is writeable by visitor
#
# Returns
# TRUE is visitor can write the object
# FALSE is visitor can not write the object
#
#-----------------------------------------------------------------------
sub isWritable{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "isWritable()" );
    #}

    #---- Fetch parameter(s)
    #
    # 1. Object hash (having both kernel object and pom object)
    #
    my $hrefObjPar = shift;

    #---- Assert object
    my $a2wObjTmp = $hrefObjPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefObjPar->{ 'POMOBJ' };
    my $bRetTmp   = $TRUE;

    my $iObjTypeTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJTYPE };
    unless (    $iObjTypeTmp == $a2w::core::dm::Constants::OT_LINE
             || $iObjTypeTmp == $a2w::core::dm::Constants::OT_TEXT
             || $iObjTypeTmp == $a2w::core::dm::Constants::OT_IMAGE
           ){
        $bRetTmp = $FALSE;
    }

    return $bRetTmp;
}

#-----------------------------------------------------------------------
# Write raw content
#-----------------------------------------------------------------------
sub writeRawContent{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeRawContent()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Raw content
    #
    my $sRawContentPar = shift;

    $this->{ 'Content' } .= "\n" . $sRawContentPar;

    return 0;
}

# $V106 Begin
# Modified to write grouped text alone and separated writing table cell content
#
#-----------------------------------------------------------------------
# Write grouped text
#-----------------------------------------------------------------------
sub writeGroupedText{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeGroupedText()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. First Object hash (having both kernel object and pom object)
    # 2. Grouped Text
    # 3. Line gap (vertical difference between previous and current line, zero for first line)
    #
    my $hrefFirstObjPar  = shift;
    my $sGroupedTextPar  = shift;
    my $iLineGapPar      = shift;

    #---- Get actual objects
    my $a2wObjTmp = $hrefFirstObjPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefFirstObjPar->{ 'POMOBJ' };

    #---- Fetch text font
    my $a2wFontTmp = $a2wObjTmp->getFont();

    #---- Update font table, if font is added to it already
    my $iFontLocalIdTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_FONTID };

    #---- Update resource table with text used font
    if ( not defined( $this->{ 'FontTable' }{ $iFontLocalIdTmp } ) ){
        $this->updateResourceTable( $iFontLocalIdTmp, $a2wFontTmp, $hrefFirstObjPar );
    }
    if ( $this->{ 'PrevFontId' } != $iFontLocalIdTmp ){
        $this->{ 'PrevFontId' } = $iFontLocalIdTmp;

        if ( $bLog == $TRUE ){
            $theLogger->logMessage(   "-->UsedFont="
                                    . $this->{ 'FontTable' }{ $iFontLocalIdTmp }{ 'FAMILY' }
                                    . ", W=" . ( ( $this->{ 'FontTable' }{ $iFontLocalIdTmp }{ 'BOLD' } == $TRUE ) ? "B" : "M" )
                                    . ", S=" . $a2wFontTmp->getHeight()
                                  );
        }
    }

    #---- Add text to body
    #---- Get active page context
    my $pgHandleTmp = $this->{ 'ActivePage' };

    #---- Get active block
    my $blkActiveTmp = $pgHandleTmp->{ 'ActiveBlock' };

    #---- Evaluate object left within block
    my $iBlkLeftTmp = $blkActiveTmp->getStartX();
    my $iRelXPosTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS } - $iBlkLeftTmp;
    if ( $iRelXPosTmp != 0 ){
        if ( $this->{ 'UnitBase' } eq "px" ){
            $iRelXPosTmp *= $this->{ 'ContentScale' };
        }
        $iRelXPosTmp = int( $iRelXPosTmp + .5 * ( $iRelXPosTmp <=> 0 ) );
    }

    
    #---- Evaluate object top within block
    my $iBlkTopTmp = $blkActiveTmp->getStartY();
    my $iRelYPosTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS } - $iBlkTopTmp;
    $iRelYPosTmp -= ( $a2wFontTmp->getHeight() / 72 ) * $pgHandleTmp->{ 'ContentRes' };
    if ( $iRelYPosTmp != 0 ){
        if ( $this->{ 'UnitBase' } eq "px" ){
            $iRelYPosTmp *= $this->{ 'ContentScale' };
        }
        $iRelYPosTmp = int( $iRelYPosTmp + .5 * ( $iRelYPosTmp <=> 0 ) );
    }

    # $V103 Begin
    #---- Apply relative position style to text when text is grouped based on font in paragraph blocks
    #
    my $bForceRelPosStyleTmp = $FALSE;
    my $contDefTmp = $blkActiveTmp->getContentDef();
    if (    lc( $contDefTmp->getType() ) eq "paragraph"
         && $contDefTmp->getGroup() == $TRUE
         && $contDefTmp->getGroupBy() eq "font"
       ){
        $bForceRelPosStyleTmp = $TRUE;
    }
    # $V103 End
    #---- Escape text for special characters
    my $sTextTmp = $this->_escapeText( $sTextTmp );
    $sTextTmp = $this->findAndReplaceText( $hrefFirstObjPar, $sGroupedTextPar, $hrefTableInfoPar );    # Find and replace text, if configured

    #--- Add the text
    my @arrClassTmp = ();
    my $sClassTmp   = '';
    my $sStyleTmp   = '';
    @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'FontTable' }->{ $iFontLocalIdTmp }{ 'STYLENAME' };
    # $V103 Begin
    if ( $bForceRelPosStyleTmp == $TRUE ){
        @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'CSS' }{ 'Styles' }{ 'RELPOS' };
        $sStyleTmp .= 'left:' . $iRelXPosTmp . $this->{ 'UnitBase' } . ';';
    }
    elsif ( $iRelXPosTmp != 0 ){
        @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' };
        $sStyleTmp .= 'left:' . $iRelXPosTmp . $this->{ 'UnitBase' } . ';';
    }
    # $V103 End
    # $V106 Begin
    if ( $bForceRelPosStyleTmp == $TRUE ){
        if ( "@arrClassTmp" !~ /$this->{ 'CSS' }{ 'Styles' }{ 'RELPOS' }/ ){ @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'CSS' }{ 'Styles' }{ 'RELPOS' }; }
        $sStyleTmp .= 'top:' . $iRelYPosTmp . $this->{ 'UnitBase' } . ';';
    }
    elsif ( $iRelYPosTmp != 0 ){
        if ( "@arrClassTmp" !~ /$this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' }/ ){ @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' }; }
        $sStyleTmp .= 'top:' . $iRelYPosTmp . $this->{ 'UnitBase' } . ';';
    }
    # $V106 End
    if ( $this->{ 'CSS' }{ 'Styles' }{ 'ROT' . $pomObjTmp->{ $a2w::core::dm::Constants::AT_ANGLE } } ne "" ){
        @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'CSS' }{ 'Styles' }{ 'ROT' . $pomObjTmp->{ $a2w::core::dm::Constants::AT_ANGLE } };
    }
    if ( @arrClassTmp > 0 ){
        $sClassTmp = ' class="' . "@arrClassTmp" . '"';
    }

    my $iColorTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_COLOR };
    if ( $iColorTmp != $this->getDefForeColor() ){
        $sStyleTmp .= 'color:#' . sprintf( "%06X", $iColorTmp ) . ';';
    }
    if ( $sStyleTmp ne "" ){
        $sStyleTmp = ' style="' . $sStyleTmp . '"';
    }

    $this->{ 'Content' } .=   "\n"
                            . '<span' . $sClassTmp . $sStyleTmp . '>'
                            . $sTextTmp
                            . '</span>'
                            ;

    if ( $bLog == $TRUE ){
        #---- Fetch text position
        my $fXPosTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS };
        my $fYPosTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS };
        if ( $this->{ 'UnitBase' } eq "px" ){
            $fXPosTmp *= $this->{ 'ContentScale' };
            $fYPosTmp *= $this->{ 'ContentScale' };
        }
        $fXPosTmp = int( $fXPosTmp + .5 * ( $fXPosTmp <=> 0 ) );
        $fYPosTmp = int( $fYPosTmp + .5 * ( $fYPosTmp <=> 0 ) );

        $theLogger->logMessage(   "@("
                                . $fXPosTmp
                                . ",". $fYPosTmp
                                . ")>" . $sTextTmp . "<"
                              );
    }
    return 0;
}
# $V106 End

#-----------------------------------------------------------------------
# Write image
#-----------------------------------------------------------------------
sub writeImage{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeImage()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Image object hash (having both kernel object and pom object)
    #
    my $hrefImageObjPar = shift;

    #---- Get actual objects
    my $a2wObjTmp = $hrefImageObjPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefImageObjPar->{ 'POMOBJ' };

    #---- Get active page context
    my $pgHandleTmp = $this->{ 'ActivePage' };

    #---- Get active page
    my $dbPageTmp = $pgHandleTmp->{ 'POM' };

    #---- Get active block
    my $blkActiveTmp = $pgHandleTmp->{ 'ActiveBlock' };

    #---- Evaluate object left within block
    my $iBlkLeftTmp = $blkActiveTmp->getStartX();
    my $iRelXPosTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS } - $iBlkLeftTmp;
    if ( $iRelXPosTmp != 0 ){
        if ( $this->{ 'UnitBase' } eq "px" ){
            $iRelXPosTmp *= $this->{ 'ContentScale' };
        }
        $iRelXPosTmp = int( $iRelXPosTmp + .5 * ( $iRelXPosTmp <=> 0 ) );
    }

    # $V106 Begin
    #---- Evaluate object top within block
    my $iBlkTopTmp = $blkActiveTmp->getStartY();
    my $iRelYPosTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS } - $iBlkTopTmp;
    if ( $iRelYPosTmp != 0 ){
        if ( $this->{ 'UnitBase' } eq "px" ){
            $iRelYPosTmp *= $this->{ 'ContentScale' };
        }
        #$iRelYPosTmp = int( $iRelYPosTmp + .5 * ( $iRelYPosTmp <=> 0 ) );
    }
    # $V106 End

    #---- Fetch image attributes
    my $fWidthTmp     = $a2wObjTmp->getPresentationWidth();
    my $fHeightTmp    = $a2wObjTmp->getPresentationHeight();
    my $iResoluionTmp = $a2wObjTmp->getResolution();

    #---- Extend block end Y (add presentation size to with image position)
    my $fYPosTmp    = $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS };
    my $fImgEndYTmp = $fYPosTmp
                      + ( $fHeightTmp * ( $dbPageTmp->getPageRes() / $iResoluionTmp ) );
    if ( $pgHandleTmp->{ 'PrevBlkEndY' } < $fImgEndYTmp ){
        $pgHandleTmp->{ 'PrevBlkEndY' } = $fImgEndYTmp;
    }

    #---- Skip, if skip mode is on
    if ( $this->{ 'SkipMode' } == $TRUE ){
        return 0;
    }

    #---- Save image ----#
    #---- Check and create images sub directory, if not exist already
    my $sImgOutputPathTmp = $this->{ 'OutputPath' } . $this->{ 'ResPaths' }{ 'IMAGE' };
    unless ( -d $sImgOutputPathTmp ){
        #---- Create image output path
        unless ( mkdir( $sImgOutputPathTmp ) ){
            my $sMsgTmp = "Unable to create sub directory (" . $this->{ 'ResPaths' }{ 'IMAGE' } . ") on output path (" . $this->{ 'OutputPath' } . "), reason:" . $!;
            if ( $bLog == $TRUE ){
                $theLogger->logMessage( "Error! " . $sMsgTmp );
            }
            return ( -1, $sMsgTmp );
        }
    }

    #---- Evaluate output file name
    $pgHandleTmp->{ 'ImageCounter' }++;
    my $sImgFormatTmp   = "png";
    my $sImgFilenameTmp =   sprintf(   "%simg%03d."
                                     , $this->{ 'ResPaths' }{ 'IMAGE' }
                                     , ( $this->getDocumentId() . $dbPageTmp->getPageID() . $pgHandleTmp->{ 'ImageCounter' } )
                                   )
                          . $sImgFormatTmp;
    $a2wObjTmp->saveAs(   $sImgFilenameTmp
                        , $sImgFormatTmp
                        , { 'Resolution' => $this->getOutputRes() }
                      );
    if ( $bLog == $TRUE ){
        $theLogger->logMessage( "Saved image as " . $sImgFilenameTmp );
    }

    #---- Add image to body
    if ( $this->{ 'UnitBase' } eq "px" ){
        $fWidthTmp  *= $this->getOutputRes() / $iResoluionTmp;
        $fHeightTmp *= $this->getOutputRes() / $iResoluionTmp;
    }
    $fWidthTmp  = int( $fWidthTmp + .5 * ( $fWidthTmp <=> 0 ) );
    $fHeightTmp = int( $fHeightTmp + .5 * ( $fHeightTmp <=> 0 ) );

    my $sStyleTmp = ' width='  . $fWidthTmp;
    $sStyleTmp   .= ' height='  . $fHeightTmp;

    my @arrClassTmp = ();
    my $sClassTmp   = '';
    my $sPosStyleTmp = '';
    if ( $iRelXPosTmp != 0 ){
        @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' };
        $sPosStyleTmp .= 'left:' . $iRelXPosTmp . $this->{ 'UnitBase' } . ';';
    }
    # $V106 Begin
    else {
        @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' };
        $sPosStyleTmp .= 'left:0' . $this->{ 'UnitBase' } . ';';
    }
    if ( $iRelYPosTmp != 0 ){
        if ( "@arrClassTmp" !~ /$this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' }/ ){ @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' }; }
        $sPosStyleTmp .= 'top:' . $iRelYPosTmp . $this->{ 'UnitBase' } . ';';
    }
    else {
        if ( "@arrClassTmp" !~ /$this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' }/ ){ @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' }; }
        $sPosStyleTmp .= 'top:0' . $this->{ 'UnitBase' } . ';';
    }
    # $V106 End
    if ( @arrClassTmp > 0 ){
        $sClassTmp = ' class="' . "@arrClassTmp" . '"';
    }
    if ( $sPosStyleTmp ne "" ){
        $sPosStyleTmp = ' style="' . $sPosStyleTmp . '"';
    }

    # $V111 Begin
    #---- Assert and wrap image with <a> tag (when block info has "href=<url>")
    #
    # Fetch block info
    my $sBlkInfoTmp = $blkActiveTmp->getInfo();
    my $bWrappedWithAnchorTmp = $FALSE;
    if ( $sBlkInfoTmp =~ /\s+href\=\"(.*)\"\s*/ ){
        $bWrappedWithAnchorTmp = $TRUE;
        my $sURLTmp = $1;
        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "Block info has href URL (" . $sURLTmp . "), wrapping image with <a> tag" );
        }
        $this->{ 'Content' } .=   "\n"
                                . '<a href="' . $sURLTmp . '">'
                                ;
    }
    # $V111 End

    # $V102 Begin
    #---- Process image file content and make data uri to inline image in html ----#
    #---- Open image file
    my $sImgInlinedTmp = "No";
    my $fImageHandleTmp = undef;
    my $sFQImgFilenameTmp = $this->{ 'OutputPath' } . $sImgFilenameTmp;
    if (    $this->{ 'bInlineImage' } == $TRUE
         && open( $fImageHandleTmp, $sFQImgFilenameTmp )
       ){
        #---- Read image data
        local $/ = undef;
        binmode $fImageHandleTmp;
        my $binImageTmp = <$fImageHandleTmp>;
        close( $fImageHandleTmp );

        #---- Encode image data in base64
        my $b64ImageDataTmp = a2w::core::utils::Base64::encode( $binImageTmp );
        $binImageTmp = undef;

        $this->{ 'Content' } .=   "\n"
                                . '<img' . $sClassTmp . $sPosStyleTmp . $sStyleTmp . ' src="data:image/' . $sImgFormatTmp . ';base64,' . $b64ImageDataTmp . '" alt="' . $a2wObjTmp->getName() . '"/>'
                                ;
        $sImgInlinedTmp = "Yes";

        #---- Delete image file
        unlink( $sFQImgFilenameTmp );
    }
    else {
        $fImageHandleTmp = undef;
        if (    $bLog == $TRUE
             && $this->{ 'bInlineImage' } == $TRUE
           ){
            $theLogger->logMessage( "Error! Unable to open file (" . $sFQImgFilenameTmp . "), reason: " . $! );
            $theLogger->logMessage( "Warning! Image $sFQImgFilenameTmp could not be inlined on HTML" );
        }

        $this->{ 'Content' } .=   "\n"
                                # $V101 Begin
                                . '<img' . $sClassTmp . $sPosStyleTmp . $sStyleTmp . ' src="' . $sImgFilenameTmp . '" alt="' . $a2wObjTmp->getName() . '"/>'
                                # $V101 End
                                ;
    }
    # $V102 End

    # $V111 Begin
    if ( $bWrappedWithAnchorTmp == $TRUE ){
        $this->{ 'Content' } .=   "\n"
                                . '</a>'
                                ;
    }
    # $V111 End

    if ( $bLog == $TRUE ){
        #---- Fetch image position
        my $fXPosTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS };
        if ( $this->{ 'UnitBase' } eq "px" ){
            $fXPosTmp *= $this->{ 'ContentScale' };
            $fYPosTmp *= $this->{ 'ContentScale' };
        }
        $fXPosTmp = int( $fXPosTmp + .5 * ( $fXPosTmp <=> 0 ) );
        $fYPosTmp = int( $fYPosTmp + .5 * ( $fYPosTmp <=> 0 ) );

        $theLogger->logMessage(   "-->Image="
                                . $a2wObjTmp->getName()
                                . ", ImgSize=(W:" . $a2wObjTmp->getWidth() . ", H:" . $a2wObjTmp->getHeight() . ")"
                                . ", BPP=" . $a2wObjTmp->getBitsPerPixel()
                                . ", Compression=" . $a2wObjTmp->getCompression()
                                . ", Pos=(X:" . $fXPosTmp . ", Y:" . $fYPosTmp . ")"
                                . ", PresentationSize=(W:" . $fWidthTmp . ", H:" . $fHeightTmp . ")"
                                # $V102 Begin
                                . ", Inlined=" . $sImgInlinedTmp
                                # $V102 End
                              );
    }

    return 0;
}

#-----------------------------------------------------------------------
# Write line
#-----------------------------------------------------------------------
sub writeLine{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeLine()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Line object hash (having both kernel object and pom object)
    #
    my $hrefLineObjPar = shift;

    #---- Get actual objects
    my $a2wObjTmp = $hrefLineObjPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefLineObjPar->{ 'POMOBJ' };

    #---- Skip, if skip mode is on
    if ( $this->{ 'SkipMode' } == $TRUE ){
        return 0;
    }

    #---- Add line to body
    #---- Get active page context
    my $pgHandleTmp = $this->{ 'ActivePage' };

    #---- Get active block
    my $blkActiveTmp = $pgHandleTmp->{ 'ActiveBlock' };

    #---- Evaluate object left within block
    my $iBlkLeftTmp = $blkActiveTmp->getStartX();
    my $iRelXPosTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS } - $iBlkLeftTmp;
    if ( $iRelXPosTmp != 0 ){
        if ( $this->{ 'UnitBase' } eq "px" ){
            $iRelXPosTmp *= $this->{ 'ContentScale' };
        }
        $iRelXPosTmp = int( $iRelXPosTmp + .5 * ( $iRelXPosTmp <=> 0 ) );
    }

    # $V106 Begin
    #---- Evaluate object top within block
    my $iBlkTopTmp = $blkActiveTmp->getStartY();
    my $iRelYPosTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS } - $iBlkTopTmp;
    if ( $iRelYPosTmp != 0 ){
        if ( $this->{ 'UnitBase' } eq "px" ){
            $iRelYPosTmp *= $this->{ 'ContentScale' };
        }
        $iRelYPosTmp = int( $iRelYPosTmp + .5 * ( $iRelYPosTmp <=> 0 ) );
    }
    # $V106 End

    #---- Fetch line attributes
    my $fLineWidthTmp  = $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_LINE_WIDTH };
    my $fLineLengthTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_LINE_LENGTH };
    if ( $this->{ 'UnitBase' } eq "px" ){
        $fLineWidthTmp  *= $this->{ 'ContentScale' };
        $fLineLengthTmp *= $this->{ 'ContentScale' };
    }
    my $sOrnTmp = "HLine";
    if ( $a2wObjTmp->isVertical() ){
        $sOrnTmp = "VLine";
        #---- Swap width and length
        my $fSwapTmp = $fLineWidthTmp;
        $fLineWidthTmp = $fLineLengthTmp;
        $fLineLengthTmp = $fSwapTmp;
    }

    $fLineWidthTmp = int( $fLineWidthTmp + .5 * ( $fLineWidthTmp <=> 0 ) );
    $fLineLengthTmp = int( $fLineLengthTmp + .5 * ( $fLineLengthTmp <=> 0 ) );

    #---- Assert line length for negative value
    if ( $fLineLengthTmp < 0.0 ){
        $fXPosTmp += $fLineLengthTmp;
        $fLineLengthTmp *= -1;
    }

    #---- Assert line width for negative value
    if ( $fLineWidthTmp < 0.0 ){
        $fYPosTmp += $fLineWidthTmp;
        $fLineWidthTmp *= -1;
    }

    my @arrClassTmp   = ();
    my $sClassTmp     = '';
    my $sLineStyleTmp = ' style="';
    if ( $iRelXPosTmp != 0 ){
        @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' };
        $sLineStyleTmp .= 'left:' . $iRelXPosTmp . $this->{ 'UnitBase' } . ';';
    }
    # $V106 Begin
    else {
        @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' };
        $sLineStyleTmp .= 'left:0' . $this->{ 'UnitBase' } . ';';
    }
    if ( $iRelYPosTmp != 0 ){
        if ( "@arrClassTmp" !~ /$this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' }/ ){ @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' }; }
        $sLineStyleTmp .= 'top:' . $iRelYPosTmp . $this->{ 'UnitBase' } . ';';
    }
    else {
        if ( "@arrClassTmp" !~ /$this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' }/ ){ @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' }; }
        $sLineStyleTmp .= 'top:0' . $this->{ 'UnitBase' } . ';';
    }
    # $V106 End

    if ( @arrClassTmp > 0 ){
        $sClassTmp = ' class="' . "@arrClassTmp" . '"';
    }

    $sLineStyleTmp .= 'width:'  . $fLineLengthTmp . $this->{ 'UnitBase' } . ';';
    $sLineStyleTmp .= 'height:'  . $fLineWidthTmp . $this->{ 'UnitBase' } . ';';
    $sLineStyleTmp .= 'background-color:#' . sprintf( "%06X", $pomObjTmp->{ $a2w::core::dm::Constants::AT_COLOR } ) . ';';
    $sLineStyleTmp .= 'border:none;"';

    $this->{ 'Content' } .=   "\n"
                            . '<hr' . $sClassTmp . $sLineStyleTmp . '/>'
                            ;

    if ( $bLog == $TRUE ){
        #---- Fetch line position
        my $fXPosTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS };
        my $fYPosTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS };
        if ( $this->{ 'UnitBase' } eq "px" ){
            $fXPosTmp *= $this->{ 'ContentScale' };
            $fYPosTmp *= $this->{ 'ContentScale' };
        }
        $fXPosTmp = int( $fXPosTmp + .5 * ( $fXPosTmp <=> 0 ) );
        $fYPosTmp = int( $fYPosTmp + .5 * ( $fYPosTmp <=> 0 ) );

        $theLogger->logMessage(   "-->Line="
                                . " Orn=" . $sOrnTmp
                                . ", Width=" . $fLineWidthTmp
                                . ", Length=" . $fLineLengthTmp
                                . ", StartPos=(X:" . $fXPosTmp . ", Y:" . $fYPosTmp . ")"
                              );
    }
    return 0;
}

#-----------------------------------------------------------------------
# Write text
#-----------------------------------------------------------------------
sub writeText{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeText()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Text object hash (having both kernel object and pom object)
    #
    my $hrefTextObjPar = shift;

    #---- Get actual objects
    my $a2wObjTmp = $hrefTextObjPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefTextObjPar->{ 'POMOBJ' };

    #---- Skip, if skip mode is on
    if ( $this->{ 'SkipMode' } == $TRUE ){
        return 0;
    }

    #---- Fetch text font
    my $a2wFontTmp = $a2wObjTmp->getFont();

    #---- Update font table, if font is added to it already
    my $iFontLocalIdTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_FONTID };

    #---- Update resource table with text used font
    if ( not defined( $this->{ 'FontTable' }{ $iFontLocalIdTmp } ) ){
        $this->updateResourceTable( $iFontLocalIdTmp, $a2wFontTmp, $hrefTextObjPar );
    }
    if ( $this->{ 'PrevFontId' } != $iFontLocalIdTmp ){
        $this->{ 'PrevFontId' } = $iFontLocalIdTmp;

        if ( $bLog == $TRUE ){
            $theLogger->logMessage(   "-->UsedFont="
                                    . $this->{ 'FontTable' }{ $iFontLocalIdTmp }{ 'FAMILY' }
                                    . ", W=" . ( ( $this->{ 'FontTable' }{ $iFontLocalIdTmp }{ 'BOLD' } == $TRUE ) ? "B" : "M" )
                                    . ", S=" . $a2wFontTmp->getHeight()
                                  );
        }
    }

    #---- Add text to body
    #---- Get active page context
    my $pgHandleTmp = $this->{ 'ActivePage' };

    #---- Get active block
    my $blkActiveTmp = $pgHandleTmp->{ 'ActiveBlock' };

    #---- Evaluate object left within block
    my $iBlkLeftTmp = $blkActiveTmp->getStartX();
    my $iRelXPosTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS } - $iBlkLeftTmp;
    if ( $iRelXPosTmp != 0 ){
        if ( $this->{ 'UnitBase' } eq "px" ){
            $iRelXPosTmp *= $this->{ 'ContentScale' };
        }
        $iRelXPosTmp = int( $iRelXPosTmp + .5 * ( $iRelXPosTmp <=> 0 ) );
    }

    # $V106 Begin
    #---- Evaluate object top within block
    my $iBlkTopTmp = $blkActiveTmp->getStartY();
    my $iRelYPosTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS } - $iBlkTopTmp;

    # $V107 Begin
    #---- Adjust text position to top left based on font height
    $iRelYPosTmp -= ( $a2wFontTmp->getHeight() / 72 ) * $pgHandleTmp->{ 'ContentRes' };
    # $V107 End

    if ( $iRelYPosTmp != 0 ){
        if ( $this->{ 'UnitBase' } eq "px" ){
            $iRelYPosTmp *= $this->{ 'ContentScale' };
        }
        $iRelYPosTmp = int( $iRelYPosTmp + .5 * ( $iRelYPosTmp <=> 0 ) );
    }
    # $V106 End

    #---- Escape text for special characters
    my $sTextTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_VALUE };
    $sTextTmp = $this->_escapeText( $sTextTmp );
    $sTextTmp = $this->findAndReplaceText( $hrefFirstObjPar, $sTextTmp );    # Find and replace text, if configured

    my @arrClassTmp = ();
    my $sClassTmp   = '';
    my $sStyleTmp   = '';
    @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'FontTable' }->{ $iFontLocalIdTmp }{ 'STYLENAME' };
    if ( $iRelXPosTmp != 0 ){
        @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' };
        $sStyleTmp .= 'left:' . $iRelXPosTmp . $this->{ 'UnitBase' } . ';';
    }
    # $V106 Begin
    if ( $iRelYPosTmp != 0 ){
        if ( "@arrClassTmp" !~ /$this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' }/ ){ @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'CSS' }{ 'Styles' }{ 'ABSPOS' }; }
        $sStyleTmp .= 'top:' . $iRelYPosTmp . $this->{ 'UnitBase' } . ';';
    }
    # $V106 End
    if ( $this->{ 'CSS' }{ 'Styles' }{ 'ROT' . $pomObjTmp->{ $a2w::core::dm::Constants::AT_ANGLE } } ne "" ){
        @arrClassTmp[ $#arrClassTmp + 1 ] = $this->{ 'CSS' }{ 'Styles' }{ 'ROT' . $pomObjTmp->{ $a2w::core::dm::Constants::AT_ANGLE } };
    }
    if ( @arrClassTmp > 0 ){
        $sClassTmp = ' class="' . "@arrClassTmp" . '"';
    }

    #--- Add text to output
    my $iColorTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_COLOR };
    if ( $iColorTmp != $this->getDefForeColor() ){
        $sStyleTmp .= 'color:#' . sprintf( "%06X", $iColorTmp ) . ';';
    }
    if ( $sStyleTmp ne "" ){
        $sStyleTmp = ' style="' . $sStyleTmp . '"';
    }

    $this->{ 'Content' } .=   "\n"
                            . '<span' . $sClassTmp . $sStyleTmp . '>'
                            . $sTextTmp
                            . '</span>'
                            ;

    if ( $bLog == $TRUE ){
        #---- Fetch text position
        my $fXPosTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS };
        my $fYPosTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS };
        if ( $this->{ 'UnitBase' } eq "px" ){
            $fXPosTmp *= $this->{ 'ContentScale' };
            $fYPosTmp *= $this->{ 'ContentScale' };
        }
        $fXPosTmp = int( $fXPosTmp + .5 * ( $fXPosTmp <=> 0 ) );
        $fYPosTmp = int( $fYPosTmp + .5 * ( $fYPosTmp <=> 0 ) );

        $theLogger->logMessage(   "@("
                                . $fXPosTmp
                                . ",". $fYPosTmp
                                . ")>" . $sTextTmp . "<"
                              );
    }
    return 0;
}

#-----------------------------------------------------------------------
# Write vector
#-----------------------------------------------------------------------
sub writeVector{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeVector()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Vector object hash (having both kernel object and pom object)
    #
    my $hrefVectorObjPar = shift;

    #---- Get actual objects
    my $a2wObjTmp = $hrefVectorObjPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefVectorObjPar->{ 'POMOBJ' };

    #---- Skip, if skip mode is on
    if ( $this->{ 'SkipMode' } == $TRUE ){
        return 0;
    }

    return 0;
}

#---- Block related writing methods ----#
#-----------------------------------------------------------------------
# Begin block
#-----------------------------------------------------------------------
sub beginBlock{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "beginBlock()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Block
    # 2. Skip flag
    #
    my $blkToBeginPar = shift;
    my $bSkipPar = shift;

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::beginBlock( $blkToBeginPar, $bSkipPar );
    if ( $iRetTmp < 0 ){
        return $iRetTmp;
    }
    if ( $bSkipPar == $TRUE ){
        return 0;
    }

    #---- Get active page context
    my $pgHandleTmp = $this->{ 'ActivePage' };

    #---- Get active page
    my $dbPageTmp = $pgHandleTmp->{ 'POM' };

    #---- Get block start X
    my $iBlkLeftTmp  = $blkToBeginPar->getStartX();
    my $iBlkLeft2Tmp = $iBlkLeftTmp;
    if ( $this->{ 'UnitBase' } eq "px" ){
        $iBlkLeftTmp *= $this->{ 'ContentScale' };
    }
    $iBlkLeftTmp = int( $iBlkLeftTmp + .5 * ( $iBlkLeftTmp <=> 0 ) );

    my $sBlkIdTmp   = $blkToBeginPar->getId();
    my $reDefPfxTmp = qr/$a2w::core::dm::Constants::BLK_PAGE_DEF_PREFIX/;
    if ( $sBlkIdTmp =~ $reDefPfxTmp ){
        $sBlkIdTmp =~ s/$reDefPfxTmp/page/g;
    }
    my $sAttrIdTmp = ' id="' . $sBlkIdTmp . '"';

    # $V106 Begin
    my $sStyleTmp =   ' style="'
                     . 'position:absolute;'
                     . 'left:' . $iBlkLeftTmp . $this->{ 'UnitBase' } . ';'
                     ;

    #---- If previous block is table, evaluate start Y based on previous table rows
    my $iTopTmp = 0;
    if ( $this->{ 'PageCount' } == 1 ){ # First page
        $iTopTmp = $blkToBeginPar->getStartY();
    }
    else { # Following pages
        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "Block " . $blkToBeginPar->getId() . " start Y:>" . $blkToBeginPar->getStartY() . "<" );
        }
        $iTopTmp = $blkToBeginPar->getStartY() - $pgHandleTmp->{ 'StartY' };
    }

    my $iTableHeightTmp = 0;
    if ( $this->{ 'PrevBlock' } != undef ){
        if (    $this->{ 'PrevBlock' }->isTable() == $TRUE
             && $this->{ 'TableHeight' } > 0
           ){
            # $V109 Begin
            $iTableHeightTmp = ( $this->{ 'TableHeight' } / $this->getOutputRes() ) * $pgHandleTmp->{ 'ContentRes' };
            if ( $bLog == $TRUE ){
                my $iPrevBlkHeightTmp = $this->{ 'PrevBlock' }->getContentHeight();

                $theLogger->logMessage( "Previous block " . $this->{ 'PrevBlock' }->getId() . " is table block" );
                $theLogger->logMessage( "Table height:>" . $iTableHeightTmp . "< Previous block height:>" . $iPrevBlkHeightTmp . "<" );
            }
            $this->{ 'BlockTopDiff' } = ( $this->{ 'BlockTop' } + $iTableHeightTmp ) - $iTopTmp;
            # $V109 End
        }
    }
	if ( $bLog == $TRUE ){
		# $V109 Begin
		$theLogger->logMessage( "Actual Top:>" . $iTopTmp . "< Evaluated Top:>" . ( $iTopTmp + $this->{ 'BlockTopDiff' } ) . "< Block Top Difference:>" . $this->{ 'BlockTopDiff' } . "<" );
		# $V109 End
	}
    $iTopTmp += $this->{ 'BlockTopDiff' };
    # $V109 Begin
    if ( $this->{ 'FirstBlock' } == undef && $iTopTmp < 0 ){ $iTopTmp = 0; }
    # $V109 End
    $this->{ 'BlockTop' } = $iTopTmp;

    if ( $this->{ 'UnitBase' } eq "px" ){
        $iTopTmp *= $this->{ 'ContentScale' };
    }
    $iTopTmp = int( $iTopTmp + .5 * ( $iTopTmp <=> 0 ) );
    $sStyleTmp .= 'top:' . $iTopTmp . $this->{ 'UnitBase' } . ';';
    # $V106 End

    #---- Evaluate width
    my $iBlkWidthTmp = $blkToBeginPar->getWidth();
    if ( $iBlkWidthTmp == 0 ){    # Undefined block
        $iBlkWidthTmp = $blkToBeginPar->getEndX() - $iBlkLeft2Tmp;
    }
    my $iPageWidthTmp = $dbPageTmp->getPageWidth();
    if ( ( $iBlkLeft2Tmp + $iBlkWidthTmp ) > $iPageWidthTmp ){
        $iBlkWidthTmp = $iPageWidthTmp - $iBlkLeft2Tmp;
    }
    if ( $this->{ 'UnitBase' } eq "px" ){
        $iBlkWidthTmp *= $this->{ 'ContentScale' };
    }
    $iBlkWidthTmp = int( $iBlkWidthTmp + .5 * ( $iBlkWidthTmp <=> 0 ) );
    $sStyleTmp .= 'width:' . $iBlkWidthTmp . $this->{ 'UnitBase' } . ';';

    #---- Use configured height
    my $iBlkHeightTmp = $blkToBeginPar->getHeight();
    if ( $iBlkHeightTmp != 0 ){
        if ( $this->{ 'UnitBase' } eq "px" ){
            $iBlkHeightTmp *= $this->{ 'ContentScale' };
        }
        $iBlkHeightTmp = int( $iBlkHeightTmp + .5 * ( $iBlkHeightTmp <=> 0 ) );
        $sStyleTmp .= 'height:' . $iBlkHeightTmp . $this->{ 'UnitBase' } . ';';
    }

    $sStyleTmp .= '"';

    #---- Start block
    # $V105 Begin
    #$this->{ 'Content' } .= "\n" . '<div' . $sAttrIdTmp . $sStyleTmp . '>';

    my $sBlkInfoTmp = $blkToBeginPar->getInfo();
    if ( $sBlkInfoTmp ne "" ){
        $this->{ 'Content' } .= "\n" . '<div' . $sAttrIdTmp . ' ' . $sBlkInfoTmp . $sStyleTmp . '>';
    }
    else {
        $this->{ 'Content' } .= "\n" . '<div' . $sAttrIdTmp . $sStyleTmp . '>';
    }
    # $V105 End

    # $V106 Begin
    $this->{ 'TableHeight' } = 0;
    $this->{ 'PrevBlock' } = $blkToBeginPar;
    if ( $this->{ 'FirstBlock' } == undef ){ $this->{ 'FirstBlock' } = $blkToBeginPar; }
    # $V106 End

    return 0;
}

#-----------------------------------------------------------------------
# End block
#-----------------------------------------------------------------------
sub endBlock{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "endBlock()" );
    }

    #---- Fetch parameter(s)
    #
    # 2. Skip flag
    #
    my $blkToEndPar = shift;
    my $bSkipPar = shift;

    if ( $bSkipPar == $FALSE ){
        $this->{ 'Content' } .= "\n" . '</div>';
    }

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::endBlock( $blkToEndPar, $bSkipPar );

    # $V106 Begin
    #---- Get active page context
    my $pgHandleTmp = $this->{ 'ActivePage' };

    if (    $blkToEndPar->isTable() == $TRUE
         && $this->{ 'TableHeight' } > 0
       ){
        $iTableHeightTmp = ( $this->{ 'TableHeight' } / $this->getOutputRes() ) * $pgHandleTmp->{ 'ContentRes' };
        $pgHandleTmp->{ 'ContentHeight' } = $this->{ 'BlockTop' } + $iTableHeightTmp;
    }
    # $V109 Begin
    #else {
    #    $pgHandleTmp->{ 'ContentHeight' } = $this->{ 'BlockTop' } + $blkToEndPar->getContentHeight();
    #}
    # $V106 End
    elsif ( $blkToEndPar->isParagraph() == $TRUE ){
        $pgHandleTmp->{ 'ContentHeight' } = $this->{ 'BlockTop' } + $blkToEndPar->getContentHeight();
    }
    elsif ( $blkToEndPar->isSkip() == $TRUE ){
        $pgHandleTmp->{ 'ContentHeight' } = $this->{ 'BlockTop' } + ( $blkToEndPar->getContentHeight() * 0.54 );
    }

    if ( $bSkipPar == $TRUE ){
        $this->{ 'BlockTopDiff' } -= $blkToEndPar->getContentHeight();
    }
    # $V109 End

    return $iRetTmp;
}

#---- Paragraph related writing methods ----#
#-----------------------------------------------------------------------
# Begin paragraph
#-----------------------------------------------------------------------
sub beginParagraph{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "beginParagraph()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Block
    #
    my $blkParagraphPar = shift;

    return 0;
}

#-----------------------------------------------------------------------
# Write pargraph line
#-----------------------------------------------------------------------
sub writeParagraphLine{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeParagraphLine()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Array of line objects
    # 2. Line count
    # 3. Line gap (vertical difference between previous and current line, zero for first line)
    #
    my $arefObjectsPar = shift;
    my $iLineCountPar  = shift;
    my $iLineGapPar    = shift;

    my @arrObjsTmp = @{ $arefObjectsPar };
    if ( @arrObjsTmp <= 0 ){
        return -1;
    }

    #---- Fetch objects list
    my $bAllImageTmp = $TRUE;
    foreach my $a2wObjTmp ( @arrObjsTmp ){
        if (    $bAllImageTmp
             && $a2wObjTmp->{ 'POMOBJ' }{ $a2w::core::dm::Constants::AT_OBJTYPE } != $a2w::core::dm::Constants::OT_IMAGE
           ){
            $bAllImageTmp = $FALSE;
        }
    
        #---- Write object
        $this->writeObject( $a2wObjTmp );
    }
    # $V106 Begin
    #if ( $bAllImageTmp == $FALSE ){
    #    $this->{ 'Content' } .= "\n" . '<br>';
    #}
    # $V106 End

    # $V101 Begin
    #else {
    #   $this->{ 'Content' } .= "\n" . '<p style="padding:0px;"/>';
    #}
    # $V101 End

    return 0;
}

#-----------------------------------------------------------------------
# End paragraph
#-----------------------------------------------------------------------
sub endParagraph{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "endParagraph()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Block
    #
    my $blkParagraphPar = shift;

    return 0;
}

#---- Table related writing methods ----#
#-----------------------------------------------------------------------
# Begin table
#-----------------------------------------------------------------------
sub beginTable{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "beginTable()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Block
    # 2. Columns X positions array
    # 3. Starting Y position
    #
    my $blkTablePar        = shift;
    my $arefColumnsXPosPar = shift;
    my $iStartYPar         = shift;

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::beginTable( $blkTablePar, $arefColumnsXPosPar, $iStartYPar );
    if ( $iRetTmp < 0 ){
        return $iRetTmp;
    }

    #---- Evaluate table position
    my $fXPosTmp = $arefColumnsXPosPar->[ 0 ];
    my $fYPosTmp = $iStartYPar;
    my $iBlockWidthTmp = $blkTablePar->getWidth();
    my $fTableWidthTmp = $iBlockWidthTmp;

    if ( $this->{ 'UnitBase' } eq "px" ){
        $fXPosTmp *= $this->{ 'ContentScale' };
        $fYPosTmp *= $this->{ 'ContentScale' };
        $fTableWidthTmp *= $this->{ 'ContentScale' };
    }
    $fXPosTmp = int( $fXPosTmp + .5 * ( $fXPosTmp <=> 0 ) );
    $fYPosTmp = int( $fYPosTmp + .5 * ( $fYPosTmp <=> 0 ) );
    $fTableWidthTmp = int( $fTableWidthTmp + .5 * ( $fTableWidthTmp <=> 0 ) );

    #---- Begin table
    $this->{ 'Content' } .=   "\n" . '<table style="'
                            #. 'position:relative;'
                            #. 'left:' . $fXPosTmp . $this->{ 'UnitBase' } . ';'
                            . 'width:' . $fTableWidthTmp . $this->{ 'UnitBase' } . ';'
                            . '">';

    #---- Evaluate and write column group ----#
    my @arrXPosTmp = @{ $arefColumnsXPosPar };

    #---- Add block width as last element
    @arrXPosTmp[ ( $#arrXPosTmp + 1 ) ] = $iBlockWidthTmp + $arefColumnsXPosPar->[ 0 ];

    #---- Iterate x positions and evaluate column percentages
    my @arrColGroupTmp = ();
    for $x ( 0 .. ( @arrXPosTmp - 2 ) ){
        @arrColGroupTmp[ ( $#arrColGroupTmp + 1 ) ] = ( ( $arrXPosTmp[ $x + 1 ] - $arrXPosTmp[ $x ] ) / $iBlockWidthTmp ) * 100;
    }

    #---- Write column group
    if ( @arrColGroupTmp > 0 ){
        #---- Begin column group
        $this->{ 'Content' } .=   "\n" . '<colgroup>';

        #---- Write columns
        foreach $c ( @arrColGroupTmp ){
            $this->{ 'Content' } .=   "\n" . '<col style="width:' . int( $c + .5 * ( $c <=> 0 ) ) . '%">';
        }

        #---- End column group
        $this->{ 'Content' } .=   "\n" . '</colgroup>';
    }

    return 0;
}

#-----------------------------------------------------------------------
# Begin table header
#-----------------------------------------------------------------------
sub beginTableHeader{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "beginTableHeader()" );
    }

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::beginTableHeader();
    if ( $iRetTmp < 0 ){
        return $iRetTmp;
    }

    $this->{ 'Content' } .= "\n" . '<thead>';

    return 0;
}

#-----------------------------------------------------------------------
# End table header
#-----------------------------------------------------------------------
sub endTableHeader{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "endTableHeader()" );
    }

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::endTableHeader();
    if ( $iRetTmp < 0 ){
        return $iRetTmp;
    }

    $this->{ 'Content' } .= "\n" . '</thead>';

    # $V106 Begin
    $this->{ 'TableHdrHeight' } += ( $this->{ 'TableCellPadding' } * 2 );
    $this->{ 'TableHeight' } += $this->{ 'TableHdrHeight' };
    # $V106 End

    return 0;
}

#-----------------------------------------------------------------------
# Begin table body
#-----------------------------------------------------------------------
sub beginTableBody{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "beginTableBody()" );
    }

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::beginTableBody();
    if ( $iRetTmp < 0 ){
        return $iRetTmp;
    }

    $this->{ 'Content' } .= "\n" . '<tbody>';

    return 0;
}

#-----------------------------------------------------------------------
# End table body
#-----------------------------------------------------------------------
sub endTableBody{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "endTableBody()" );
    }

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::endTableBody();
    if ( $iRetTmp < 0 ){
        return $iRetTmp;
    }

    $this->{ 'Content' } .= "\n" . '</tbody>';

    return 0;
}

#-----------------------------------------------------------------------
# Begin table footer
#-----------------------------------------------------------------------
sub beginTableFooter{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "beginTableFooter()" );
    }

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::beginTableFooter();
    if ( $iRetTmp < 0 ){
        return $iRetTmp;
    }

    $this->{ 'Content' } .= "\n" . '<tfoot>';

    return 0;
}

#-----------------------------------------------------------------------
# End table footer
#-----------------------------------------------------------------------
sub endTableFooter{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "endTableFooter()" );
    }

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::endTableFooter();
    if ( $iRetTmp < 0 ){
        return $iRetTmp;
    }

    $this->{ 'Content' } .= "\n" . '</tfoot>';

    return 0;
}

#-----------------------------------------------------------------------
# Begin row
#-----------------------------------------------------------------------
sub beginRow{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "beginRow()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Row number
    #
    my $iRowNrPar = shift;

    $this->{ 'Content' } .= "\n" . '<tr>';

    # $V106 Begin
    $this->{ 'TableRowHeight' } = 0;
    $this->{ 'TableRowSpan' }   = 0;
    # $V106 End

    return 0;
}

#-----------------------------------------------------------------------
# Write cell
#-----------------------------------------------------------------------
sub writeCell{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeCell()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Array of cell objects
    # 2. Column header name
    # 3. Column id (as in on block definition)
    #
    my $arefObjectsPar = shift;
    my $sColHdrNamePar = shift;
    my $sColIdPar      = shift;
    if ( $bLog == $TRUE ){
        $theLogger->logMessage( "Header Name:>" . $sColHdrNamePar . "< ColumnId:>" . $sColIdPar . "<" );
    }

    #---- Fetch objects list
    my @arrObjsTmp = @{ $arefObjectsPar };

    my $sTableDataElementNameTmp = "td";
    if ( ( $this->{ 'InTable' } & 2 ) != 0 ){
        $sTableDataElementNameTmp = "th";
    }

    #---- Write cell contents ----#
    if ( @arrObjsTmp <= 0 ){
        $this->{ 'Content' } .= "\n" . '<' . $sTableDataElementNameTmp . '></' . $sTableDataElementNameTmp . '>';
        return 0;
    }

    # $V106 Begin
    my $iFontIdTmp       = -1;
    my $iRowSpanCntTmp   = 1;
    my $a2wObjTmp        = undef;
    my $pomObjTmp        = undef;
    my $a2wFontTmp       = undef;

    #---- Group all cell texts ----#
    my @arrTextsTmp      = ();
    my $sTextTmp         = "";
    my $hrefTableInfoTmp = {
          'Cell'   => $TRUE
        , 'ColHdr' => $sColHdrNamePar
        , 'ColId'  => $sColIdPar
    };
    my $iPrevYPosTmp     = 0;  # $V104 Change

    foreach my $hrefObjTmp ( @arrObjsTmp ){
        if (    ref( $hrefObjTmp ) eq "HASH"
             && defined $hrefObjTmp->{ 'A2WOBJ' }
             && defined $hrefObjTmp->{ 'POMOBJ' }
           ){
            $a2wObjTmp = $hrefObjTmp->{ 'A2WOBJ' };
            $pomObjTmp = $hrefObjTmp->{ 'POMOBJ' };

            if ( $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJTYPE } == $a2w::core::dm::Constants::OT_TEXT ){
                # $V104 Begin
                if ( $iPrevYPosTmp == 0 ){ $iPrevYPosTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS }; }
                # $V104 End

                #---- Collect texts with same style
                @arrTextsTmp[ $#arrTextsTmp + 1 ] = $hrefObjTmp;

                #---- Update font table, if font is added to it already ----#
                if ( @arrTextsTmp == 1 ){
                    #---- Fetch text font
                    my $a2wFontTmp = @arrTextsTmp[ 0 ]->{ 'A2WOBJ' }->getFont();
                    $iFontIdTmp = @arrTextsTmp[ 0 ]->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_FONTID };

                    #---- Update resource table with text used font
                    if ( not defined( $this->{ 'FontTable' }{ $iFontIdTmp } ) ){
                        $this->updateResourceTable( $iFontIdTmp, $a2wFontTmp, @arrTextsTmp[ 0 ] );
                    }
                    if ( $this->{ 'PrevFontId' } != $iFontIdTmp ){
                        $this->{ 'PrevFontId' } = $iFontIdTmp;
                    }
                    if ( $this->{ 'TableRowHeight' } < $this->{ 'FontTable' }->{ $iFontIdTmp }{ 'HEIGHT' } ){
                        $this->{ 'TableRowHeight' } = $this->{ 'FontTable' }->{ $iFontIdTmp }{ 'HEIGHT' };
                    }
                }

                # $V104 Begin
                #if ( $sTextTmp ne "" ){ $sTextTmp .= ' '; }
                if ( $sTextTmp ne "" && $iPrevYPosTmp == $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS } ){ $sTextTmp .= ' '; }
                if ( $iPrevYPosTmp != $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS } ){
                    $iPrevYPosTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS };
                    $iRowSpanCntTmp++; # $V106 Change
                    # $V109 Begin
                    $sTextTmp .= '<br>';
                    # $V109 End
                }
                # $V104 End
                $sTextTmp .= $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_VALUE };
            }
            #elsif ( $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJTYPE } == $a2w::core::dm::Constants::OT_LINE ){
            #    $this->writeLine( $hrefObjTmp );
            #}
        }
        else {
            $sTextTmp .= $hrefObjTmp;
        }
    } # foreach my $hrefObjTmp ( @arrObjsTmp )

    if ( $sTextTmp ne "" ){
        #---- Begin cell
        $this->{ 'Content' } .= "\n" . '<' . $sTableDataElementNameTmp;

        if ( @arrTextsTmp > 0 ){
            #---- Escape text for special characters
            # $V109 Begin
            #$sTextTmp = $this->_escapeText( $sTextTmp );
            # $V109 End
            $sTextTmp = $this->findAndReplaceText( @arrTextsTmp[ 0 ], $sTextTmp, $hrefTableInfoTmp );    # Find and replace text, if configured

            $this->{ 'Content' } .= ' class="' . $this->{ 'FontTable' }->{ $iFontIdTmp }{ 'STYLENAME' } . '"';
        }
        $this->{ 'Content' } .= '>';

        #---- Write so far collected texts
        $this->{ 'Content' } .= $sTextTmp;

        #---- End cell
        $this->{ 'Content' } .= '</' . $sTableDataElementNameTmp . '>';
    }

    #---- Reset to collect next group of texts
    $sTextTmp = "";
    @arrTextsTmp = ();

    if ( $this->{ 'TableRowSpan' } < $iRowSpanCntTmp ){ $this->{ 'TableRowSpan' } = $iRowSpanCntTmp; }
    # $V106 End

    return 0;
}

#-----------------------------------------------------------------------
# End row
#-----------------------------------------------------------------------
sub endRow{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "endRow()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Row number
    #
    my $iRowNrPar = shift;

    $this->{ 'Content' } .= "\n" . '</tr>';

    # $V106 Begin
    $this->{ 'TableRowHeight' } *= $this->{ 'TableRowSpan' }; # Multiply by row span
    $this->{ 'TableRowHeight' } += ( $this->{ 'TableCellPadding' } * 2 );
    $this->{ 'TableHeight' } += $this->{ 'TableRowHeight' };
    # $V106 End

    return 0;
}

#-----------------------------------------------------------------------
# End table
#-----------------------------------------------------------------------
sub endTable{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "endTable()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Block
    #
    my $blkTablePar = shift;

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::endTable( $blkTablePar );
    if ( $iRetTmp < 0 ){
        return $iRetTmp;
    }

    $this->{ 'Content' } .= "\n" . '</table>';

    # $V109 Begin
    $this->{ 'TableHeight' } += 10;
    # $V109 End

    # $V106 Begin
    if ( $bLog == $TRUE ){
        $theLogger->logMessage( "endTable: " . $blkTablePar->getId() . " Height:>" . $this->{ 'TableHeight' } . "<" );
    }
    # $V106 End

    return 0;
}

#-----------------------------------------------------------------------
# Initialize HTML
#-----------------------------------------------------------------------
sub _initializeHTML{
    my $this = shift;

    #---- Write html begin
    my $handleFileTmp = $this->{ 'fHandle' };
    printf $handleFileTmp ( '<!DOCTYPE html>' );
    printf $handleFileTmp ( "\n" . '<html>' );

    #---- Initialize CSS
    my $iRetTmp = $this->_initializeCSS();

    return $iRetTmp;
}

#-----------------------------------------------------------------------
# Finalize HTML
#-----------------------------------------------------------------------
sub _finalizeHTML{
    my $this = shift;

    #---- Write html end
    my $handleFileTmp = $this->{ 'fHandle' };
    printf $handleFileTmp ( "\n" . '</html>' );

    #---- Finalize CSS
    my $sCSSFilenameTmp = $this->{ 'CSS' }{ 'Filename' };    # $V102 Change
    my $iRetTmp = $this->_finalizeCSS();

    # $V102 Begin
    if ( $iRetTmp < 0 ){ return $iRetTmp; }
    my $sHTMLFilenameTmp = $this->{ 'Filename' };
    if ( $this->{ 'bInlineCSS' } == $TRUE ){
        #---- Read in CSS file and inline it on html ----#
        my $fCSSHandleTmp = undef;
        if ( open( $fCSSHandleTmp, $sCSSFilenameTmp ) ){
            #---- Read CSS data
            local $/ = undef;
            my $sCSSTmp = <$fCSSHandleTmp>;
            close( $fCSSHandleTmp );

            #---- Add style tag
            $sCSSTmp =~ s/%/%%/g; # To take care of 'transform-origin:0% 100%' rotation style
            $sCSSTmp = '<style>' . "\n" . $sCSSTmp . "\n" . '</style>';
            if ( $bLog == $TRUE ){
                $theLogger->logMessage( "CSS>" . $sCSSTmp . "<" );
            }

            #---- Close HTML
            if ( $this->{ 'PageOutput' } == $FALSE ){
                $this->closeOutputStream();
            }
            else {
                $this->closeOutputStream( $this->{ 'ActivePage' } );
            }

            #---- Update HTML with CSS ----#
            my $fHTMLHandleTmp = undef;
            if ( open( $fHTMLHandleTmp, $sHTMLFilenameTmp ) ){
                #---- Read HTML data
                local $/ = undef;
                my $sHTMLTmp = <$fHTMLHandleTmp>;
                close( $fHTMLHandleTmp );

                #---- Remove CSS include and inline CSS content
                my $sSearchTmp = $this->{ 'CSS' }{ 'IncludeStatement' };
                my $reReplaceCSSTmp = qr/$sSearchTmp/;
                $sHTMLTmp =~ s/$reReplaceCSSTmp/$sCSSTmp/;

                #---- Delete HTML file and recreate with in lined CSS
                unlink( $sHTMLFilenameTmp );
                if ( open( $fHTMLHandleTmp, ">$sHTMLFilenameTmp" ) ){
                    if ( $bLog == $TRUE ){
                        $theLogger->logMessage( "File (" . $sHTMLFilenameTmp . ") opened successfully to inline CSS" );
                    }
                    printf $fHTMLHandleTmp ( $sHTMLTmp );
                    close( $fHTMLHandleTmp );

                    #---- Delete CSS file
                    unlink( $sCSSFilenameTmp );

                    if ( $bLog == $TRUE ){
                        $theLogger->logMessage( $sHTMLFilenameTmp . " successfully inlined with CSS" );
                    }
                }
                else {
                    $fHTMLHandleTmp = undef;
                    if ( $bLog == $TRUE ){
                        $theLogger->logMessage( "Error! Unable to open file (" . $sHTMLFilenameTmp . "), reason: " . $! );
                        $theLogger->logMessage( "Warning! Unable to in line CSS on HTML" );
                    }
                    $iRetTmp = -1;
                }
            }
        }
    }
    # $V102 End

    return $iRetTmp;
}

#-----------------------------------------------------------------------
# Initialize CSS
#-----------------------------------------------------------------------
sub _initializeCSS{
    my $this = shift;

    #---- Evaluate CSS file name
    $this->{ 'CSS' }{ 'Filename' } = $this->getFilename();                       # Html file name
    my $sOutputPathTmp    = $this->getOutputPath();                              # Html output path
    my $sCSSOutputPathTmp = $sOutputPathTmp . $this->{ 'ResPaths' }{ 'CSS' };    # CSS output path
    $this->{ 'CSS' }{ 'Filename' } =~ s/\Q$sOutputPathTmp\E/$sCSSOutputPathTmp/; # Replace html output path with CSS output path
    $this->{ 'CSS' }{ 'Filename' } =~ s/\.html/\.css/;                           # Replace html extension with CSS

    #---- Check and create css sub directory, if not exist already
    unless ( -d $sCSSOutputPathTmp ){
        #---- Create css output path
        unless ( mkdir( $sCSSOutputPathTmp ) ){
            my $sMsgTmp = "Unable to create sub directory (" . $this->{ 'ResPaths' }{ 'CSS' } . ") on output path (" . $sOutputPathTmp . "), reason:" . $!;
            if ( $bLog == $TRUE ){
                $theLogger->logMessage( "Error! " . $sMsgTmp );
            }
            return ( -1, $sMsgTmp );
        }
    }

    #---- Open CSS file
    $this->{ 'CSS' }{ 'fHandle' } = $this->open( $this->{ 'CSS' }{ 'Filename' } );
    if ( $this->{ 'CSS' }{ 'fHandle' } == undef ){
        return ( -2, "Unable to open file (" . $this->{ 'CSS' }{ 'Filename' } . ")" );
    }

    # V102 Begin
    if ( $this->{ 'PageOutput' } == $TRUE ){
        $this->{ 'ActivePage' }->{ 'CSS' }{ 'Filename' } = $this->{ 'CSS' }{ 'Filename' };
    }
    # V102 End
    return 0;
}

#-----------------------------------------------------------------------
# _writeCSSHeader
#-----------------------------------------------------------------------
sub _writeCSSHeader{
    my $this = shift;

    #---- Write CSS header
    my $fCSSHandleTmp = $this->{ 'CSS' }{ 'fHandle' };
    printf $fCSSHandleTmp ( "/* AFP2web generated CSS */" );

    #---- Write rotation styles
    printf $fCSSHandleTmp ( "\n.org90{transform-origin:0%% 100%%;-ms-transform-origin:0%% 100%%;-webkit-transform-origin:0%% 100%%;}" );
    printf $fCSSHandleTmp ( "\n.r90{transform:rotate(90deg);-webkit-transform:rotate(90deg);-moz-transform:rotate(90deg);-ms-transform:rotate(90deg);-o-transform:rotate(90deg);filter: progid:DXImageTransform.Microsoft.BasicImage(rotation=1);}" );
    printf $fCSSHandleTmp ( "\n.r180{transform:rotate(180deg);-webkit-transform:rotate(180deg);-moz-transform:rotate(180deg);-ms-transform:rotate(180deg);-o-transform:rotate(180deg);filter: progid:DXImageTransform.Microsoft.BasicImage(rotation=2);}" );
    printf $fCSSHandleTmp ( "\n.r270{transform:rotate(270deg);-webkit-transform:rotate(270deg);-moz-transform:rotate(270deg);-ms-transform:rotate(270deg);-o-transform:rotate(270deg);filter: progid:DXImageTransform.Microsoft.BasicImage(rotation=3);}" );

    #---- Write position styles
    printf $fCSSHandleTmp ( "\n\n.pa{position:absolute;}" );
    printf $fCSSHandleTmp ( "\n.pr{position:relative;}" );

    #---- Write div style
    my $sDivStyleTmp =   'div{'
                       . 'border-width:0px;'
                       # V106 Begin
                       . 'display:inline-block;'
                       # V106 End
                       . '}';
    if ( ( $theLogger->getLevel() & $a2w::core::log::Logger::LEVEL_DEBUG ) != 0 ){
        $sDivStyleTmp =   'div{'
                        . 'border-width:medium;'
                        . 'border-style:solid;'
                        . 'border-color:red;'
                        # V106 Begin
                        . 'display:inline-block;'
                        # V106 End
                        . '}';
    }
    printf $fCSSHandleTmp ( "\n\n" . $sDivStyleTmp );

    # $V107 Begin
    #---- Add line style to force zero margin
    printf $fCSSHandleTmp ( "\nhr{border:0px;margin:0px;}" );
    # $V107 End

    #---- Write table header style
    my $iTHFontHeightTmp = 10.8 * $this->{ 'FontScale' };
    $iTHFontHeightTmp = int( $iTHFontHeightTmp + .5 * ( $iTHFontHeightTmp <=> 0 ) );

    # $V106 Begin
    #printf $fCSSHandleTmp ( "\n" . 'th{font-size:' . $iTHFontHeightTmp . 'px;font-family:Arial;font-weight:bold;text-align:left;}' );
    $this->{ 'TableHdrHeight' } = $iTHFontHeightTmp;

    printf $fCSSHandleTmp ( "\n" . 'table {border:0px;border-spacing:0;margin:0px;}' );
    printf $fCSSHandleTmp ( "\n" . 'table th {padding:' . $this->{ 'TableCellPadding' } . 'px;line-height:' . $iTHFontHeightTmp . 'px;font-size:' . $iTHFontHeightTmp . 'px;font-family:Arial;font-weight:bold;text-align:left;}' );
    printf $fCSSHandleTmp ( "\n" . 'table td {padding:' . $this->{ 'TableCellPadding' } . 'px;}' );
    # $V106 End
    # $V108 Begin
    #---- Forced table first column vertical alignment as "top"
    printf $fCSSHandleTmp ( "\n" . 'table td:first-child {vertical-align:top;}' );
    # $V108 End

    #---- Style classes
    $this->{ 'CSS' }{ 'Styles' } = {
          'ROT90'  => 'r270'
        , 'ROT180' => 'r180'
        , 'ROT270' => 'org90 r90'
        , 'ABSPOS' => 'pa'
        , 'RELPOS' => 'pr'
    };

    return 0;
}

#-----------------------------------------------------------------------
# Finalize CSS
#-----------------------------------------------------------------------
sub _finalizeCSS{
    my $this = shift;

    #---- Close CSS file
    $this->{ 'CSS' }{ 'Filename' } = '';
    my $iRetTmp = 0;
    if ( $this->close( $this->{ 'CSS' }{ 'fHandle' } ) == $FALSE ){
        $iRetTmp = -1;
    }
    $this->{ 'CSS' }{ 'fHandle' } = undef;

    return $iRetTmp;
}

#-----------------------------------------------------------------------
# Escape text
#
# Escapes special characters on text for HTML presentation
#
#-----------------------------------------------------------------------
# $V110 Begin
sub _escapeText{
    my $this = shift;

    #---- Fetch parameter(s)
    #
    # 1. sTextPar (of type string)
    # 2. bEscapeSpacePar (of type bool) OPTIONAL, default is FALSE
    #
    my $sTextPar = shift;
    my $bEscapeSpacePar = $FALSE;
    if ( @_ > 0 ){
        $bEscapeSpacePar = shift;
    }

    #---- Escape table
    my %hshEscapeTableTmp = (
         '<' => '&lt;'
       , '>' => '&gt;'
    );
    if ( $bEscapeSpacePar == $TRUE ){
        %hshEscapeTableTmp->{ ' ' } = '&nbsp;';
    }

    #---- Escape characters string for reqular expression
    my $sAllEscapeCharsTmp = join '', keys %hshEscapeTableTmp;

    #---- Escape characters with equivalent escape values
    my $sEscapedTextTmp = $sTextPar;
    $sEscapedTextTmp =~ s/([\Q$sAllEscapeCharsTmp\E])/$hshEscapeTableTmp{ $1 }/g;

    return $sEscapedTextTmp;
}
# $V110 End

#-----------------------------------------------------------------------
# Fetch equivalent HTML font name
#
# Search for text font equivalent HTML font name and returns it
#
#-----------------------------------------------------------------------
sub _fetchHTMLFont{
    my $this = shift;

    #---- Fetch parameter(s)
    #
    # 1. Text font name (of type string)
    #
    my $sTextFontNamePar = shift;

    #TODO: No mapping is done now, just return the name AS IS
    return $sTextFontNamePar;
}

#-----------------------------------------------------------------------
# Fetch HTML font source for @font-face "src" attribute value
#
# Returns a string having following syntax
# ur(<font filename>)format(<font file format>)
#
#-----------------------------------------------------------------------
sub _fetchHTMLFontSource{
    my $this = shift;

    #---- Fetch parameter(s)
    #
    # 1. HTML font name (of type string)
    #
    my $sHTMLFontNamePar = shift;

    #TODO: Prepare and deliver a list
    return "";
}

#-----------------------------------------------------------------------
# Is Font Bold
#
# Checks whether given font has BOLD weight and returns true if yes
#
#-----------------------------------------------------------------------
sub _isFontBold{
    my $this = shift;

    #---- Fetch parameter(s)
    #
    # 1. Font (of type a2w::Font)
    #
    my $a2wFontPar = shift;
    return $a2wFontPar->isBold();
}

#-----------------------------------------------------------------------
# Is Font Italic
#
# Checks whether given font has Italic slant and returns true if yes
#
#-----------------------------------------------------------------------
sub _isFontItalic{
    my $this = shift;

    #---- Fetch parameter(s)
    #
    # 1. Font (of type a2w::Font)
    #
    my $a2wFontPar = shift;
    return $a2wFontPar->isItalic();
}

# $V106 Begin
#-----------------------------------------------------------------------
# Get page top
#
# Evaluates and returns current page top offset
#
#-----------------------------------------------------------------------
sub _getPageTop{
    my $this = shift;

    #---- Fetch parameter(s)
    #
    # 1. Page id
    #
    my $iPageIdPar = shift;

    #---- Evaluate height for all previous pages and sum them up ----#
    my $iTopOffsetTmp = 0;
    my $pgHandleTmp = undef;

    for ( my $i = 1; $i < $iPageIdPar; $i++ ){
        #---- Get page handle
        $pgHandleTmp = $this->{ 'Pages' }{ $i };
        if ( $pgHandleTmp != undef ){
            # $V109 Begin
            #$iTopOffsetTmp += $pgHandleTmp->{ 'ContentHeight' } * 1.02;
            $iTopOffsetTmp +=   $pgHandleTmp->{ 'ContentHeight' }
                              + ( $a2w::core::dm::Constants::PAGE_MARGIN_TOP * $pgHandleTmp->{ 'ContentRes' } )
                              + ( $a2w::core::dm::Constants::PAGE_MARGIN_BOTTOM * $pgHandleTmp->{ 'ContentRes' } )
                              ;
            # $V109 End
        }
    }
    if ( $bLog == $TRUE ){
        $theLogger->logMessage( "Page " . $iPageIdPar . " Top:>" . $iTopOffsetTmp . "<" );
    }

    return $iTopOffsetTmp;
}

#-----------------------------------------------------------------------
# Get table height
#-----------------------------------------------------------------------
sub getTableHeight{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "getTableHeight()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Block
    #
    my $blkTablePar = shift;

    if ( $this->{ 'InTable' } == 0 ){ return 0; }

    return $this->{ 'TableHeight' };
}
# $V106 End

#-----------------------------------------------------------------------
# Don't remove the following lines !!!
#-----------------------------------------------------------------------
1;
__END__
