#-------------------------------------------------------------------------------
#  docCounter.pm:
#  Scripting Facility module to count documents from input AFP spool. Counting will work
#  only if input documents are indexed.
#
#  Call:
#
#  On Windows:    afp2web.exe -q -c -doc_cold -sp:docCounter.pm samples\insure.afp
#
#  On Unix:    ./afp2web   -q -c -doc_cold -sp:docCounter.pm samples/insure.afp
#
#  Author  : Fa. Maas
#
#  $V100   2015-07-27    Initial Release
#
#  $V101   2015-10-12    Extended to write document count on job file for AFP2web Server
#
#-------------------------------------------------------------------------------

#-----------------------------------------------------------------------
# BEGIN block of module
#
# Extends PERL module search path array (@INC) with new element having
# this script modules path in order to have better module portability
#-----------------------------------------------------------------------
BEGIN {
    #---- Fetch script filename
    my $sScriptFilenameTmp = $0;

    #---- Extract script file path from script filename
    my $sScriptFilePathTmp = "";
    if ( $sScriptFilenameTmp =~ /(.*)\/.*\.pm/ ){
        $sScriptFilePathTmp = $1;
    }

    #printf STDERR ( "Script filename: " . $0 . " Script filepath: " . $sScriptFilePathTmp . "\n" );
    if ( $sScriptFilePathTmp eq "" ){
        $sScriptFilePathTmp = ".";
    }
    else {
        my $sScriptFileParentPathTmp = "";
        if ( $sScriptFilePathTmp =~ /(.*)\/sfsamples/ ){
            $sScriptFileParentPathTmp = $1;
        }

        #---- Add script file parent path to module search path
        if ( $sScriptFileParentPathTmp ne "" ){
            unshift( @INC, $sScriptFileParentPathTmp );
        }
    }

    #---- Add script file path to module search path
    unshift( @INC, $sScriptFilePathTmp );
}

use a2w::Config;
use a2w::Document;
use a2w::Font;
use a2w::Index;
use a2w::Kernel;
use a2w::Line;
use a2w::MediumMap;
use a2w::NOP;
use a2w::Overlay;
use a2w::Page;
use a2w::PSEG;
use a2w::Text;

use a2w::ConfigConstants;
use a2w::DocumentConstants;
use a2w::PageConstants;
use a2w::FontConstants;

#-----------------------------------------------------------------------
# Initialize once per process
#-----------------------------------------------------------------------
sub initialize(){

    #---- Get Parameter of initialize( Par: a2w::Config, a2w::Kernel )
    ( $a2wConfigPar, $a2wKernelPar ) = @_;

    #---- Define boolean values
    $TRUE  = 1;    # TRUE  boolean value
    $FALSE = 0;    # FALSE boolean value

    #---- Set/Reset Logging
    $bLog = $FALSE;
    if (index( lc($a2wConfigPar->getAttribute( $a2w::ConfigConstants::LOGGINGLEVEL )), "sf") >= 0 ){
        $bLog = $TRUE;
    }

    my $sScriptProcTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SCRIPTPROCEDURE );
    my $sScriptArgsTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SCRIPTARGUMENT );
    $sSpoolFilename    = $a2wKernelPar->getSpoolFilename();
    $sINIFilePath      = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::INIPATH ); # $V101 Change

    if ( $bLog == $TRUE ){
        printf STDERR ( "Running $sScriptProcTmp...\n" );
        printf STDERR ( "initialize(): Processing $sSpoolFilename\n" );
        printf STDERR ( "initialize(): Args: $sScriptArgsTmp\n" );
    }

    #---- Page process flags
    $APPEND = 0;    # append page to Current Document
    $SKIP   = 1;    # skip page
    $NEWDOC = 2;    # new document

    #---- Set AutoSplit to true
    #INFO:
    # Your AFP Spool must have documents grouped in Name Page Groups and Tag Logical Elements (TLEs).
    # Setting AutoSplit to "on" will force AFP2web to split based on the same grouping.
    $a2wConfigPar->setAttribute( $a2w::ConfigConstants::AUTOSPLIT, "on" );

    #---- Force output format to AFP
    #INFO:
    # AFP2web is optimized to read pages quickly when output format is AFP. Since
    # the process must just count the documents and pages has to be skipped, forcing
    # AFP output format will ensure the pages are read quick to skip and counting
    # documents also be done quick.
    $a2wConfigPar->setAttribute( $a2w::ConfigConstants::OUTPUTFORMAT, "afp" );

    #---- Documents counter
    $iDocsCount = 0;

    # $V101 Begin
    #---- Job file name
    $sJobFilename =   $sINIFilePath
                    # Service directory
                    . "services/a2wtransformation/"
                    # Job id
                    . $sScriptArgsTmp
                    # Job status
                    . "_working"
                    ;
    if ( $bLog == $TRUE ){
        printf STDERR ( "Job filename: " . $sJobFilename . "\n" );
    }
    # $V101 End

    return 0;
}

#-----------------------------------------------------------------------
# InitializeDoc for each document
#-----------------------------------------------------------------------
sub initializeDoc(){

    #---- Get Parameter of initializeDoc( Par: a2w::Document )
    ($a2wDocumentPar) = @_;

    if ( $bLog == $TRUE ){
        printf STDERR ( "initializeDoc(): DocId " . $a2wDocumentPar->getId() . "\n" );
    }

    #---- Increment document counter
    $iDocsCount++;

    return 0;
}

#-----------------------------------------------------------------------
# InitializePage for each page
#-----------------------------------------------------------------------
sub initializePage(){

    #---- Get Parameter of initializePage( Par: a2w::Page )
    ($a2wPagePar) = @_;

    if ( $bLog == $TRUE ){
        printf STDERR ( "initializePage()\n" );
    }
    return 0;
}

#-----------------------------------------------------------------------
# Main entry method
# Return values:
#        < 0:    error
#         0:    append page to Current Document
#         1:    skip page
#         2:    first page / new document
#-----------------------------------------------------------------------
sub afp2web(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "afp2web(): PageId " . $a2wPagePar->getParseId() . "\n" );
    }

    #---- Set default return value
    my $iRetTmp = $SKIP; # skip all pages

    return $iRetTmp;
}

#-----------------------------------------------------------------------
# FinalizePage for each page
#-----------------------------------------------------------------------
sub finalizePage(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "finalizePage()\n" );
    }
    return 0;
}

#-----------------------------------------------------------------------
# FinalizeDoc for each document
#-----------------------------------------------------------------------
sub finalizeDoc(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "finalizeDoc()\n" );
    }

    return 0;
}

#-----------------------------------------------------------------------
# Finalize once per process
#-----------------------------------------------------------------------
sub finalize(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "finalize()\n" );
    }

    #---- Print total documents count
    if ( $bLog == $TRUE ){
        printf STDERR ( "Documents count: " . $iDocsCount . "\n" );
    }

    #---- Update job file with documents count ----#
    if ( -e $sJobFilename ){
        #---- Open file
        my $fhJobFileTmp = undef;
        unless ( open( $fhJobFileTmp, ">>" . $sJobFilename ) ){
            my $sErrMsgTmp = "Error! Unable to open file (" . $sJobFilename . "), reason: " . $!;
            if ( $bLog == $TRUE ){
                printf STDERR ( "$sErrMsgTmp\n" );
            }
            return ( -1, $sErrMsgTmp );
        }

        #---- Update job file with documents count
        printf $fhJobFileTmp ( "DocumentCount:" . $iDocsCount . "\n" );

        #---- Close job file
        close( $fhJobFileTmp );
        if ( $bLog == $TRUE ){
            printf STDERR ( "File (" . $sJobFilename . ") appended successfully with document count (DocumentCount:" . $iDocsCount . ")\n" );
        }
    }

    return 0;
}

__END__
