#-------------------------------------------------------------------------------
#  html.pm:
#  Script writer to generate HTML output
#
#  Call:
#
#  On Windows:
#  afp2web.exe -q -c -html samples\insure.afp
#
#  On Unix:
#  ./afp2web   -q -c -html samples/insure.afp
#
#  Author  : Fa. Maas
#  Date    : 2014-08-04
#
#  $V100    2014-08-04    Initial Release
#
#  $V101    2015-03-10    Modification:
#                         Minor bug in evaluating log file name (*nix specific). Input
#                         spool file name (without path is used as part of log file name),
#                         extracting the same failed due to regex problem.
#
#                         Fixed by correcting the regex that extract spool file name and
#                         also reported the log file name on error message.
#
#  $V102   2015-10-15    Extended to pass in custom block info like HTML style (AFP-298)
#
#-------------------------------------------------------------------------------

#-----------------------------------------------------------------------
# BEGIN block of module
#
# Extends PERL module search path array (@INC) with new element having
# this script modules path in order to have better module portability
#-----------------------------------------------------------------------
BEGIN {
    #---- Fetch script filename
    my $sScriptFilenameTmp = $0;

    #---- Extract script file path from script filename
    my $sScriptFilePathTmp = "";
    if ( $sScriptFilenameTmp =~ /(.*)\/.*\.pm/ ){
        $sScriptFilePathTmp = $1;
    }

    #printf STDERR ( "Script filename: " . $0 . " Script filepath: " . $sScriptFilePathTmp . "\n" );
    if ( $sScriptFilePathTmp eq "" ){
        $sScriptFilePathTmp = ".";
    }
    else {
        my $sScriptFileParentPathTmp = "";
        if ( $sScriptFilePathTmp =~ /(.*)\/sfsamples/ ){
            $sScriptFileParentPathTmp = $1;
        }

        #---- Add script file parent path to module search path
        if ( $sScriptFileParentPathTmp ne "" ){
            unshift( @INC, $sScriptFileParentPathTmp );
        }
    }

    #---- Add script file path to module search path
    unshift( @INC, $sScriptFilePathTmp );
}

use a2w::Config;
use a2w::Kernel;
use a2w::Document;
use a2w::Page;
use a2w::Text;
use a2w::Line;
use a2w::Image;
use a2w::Vector;
use a2w::Index;
use a2w::Font;

use a2w::TypeConstants;
use a2w::ConfigConstants;
use a2w::DocumentConstants;
use a2w::PageConstants;
use a2w::FontConstants;

use a2w::core::dm::MiningEngine;     # Data mining module
use a2w::core::log::Logger;          # Log engine
use a2w::core::visitor::HTMLVisitor;

#-----------------------------------------------------------------------
# Initialize once per process
#-----------------------------------------------------------------------
sub initialize(){

    #---- Get Parameter of initialize( Par: a2w::Config, a2w::Kernel )
    ( $a2wConfigPar, $a2wKernelPar ) = @_;

    #---- Mark start time
    $sStartTime = time();                  # start time

    #---- Define boolean values
    $TRUE  = $a2w::TypeConstants::TRUE;    # TRUE  boolean value
    $FALSE = $a2w::TypeConstants::FALSE;   # FALSE boolean value

    #---- Set/Reset Logging
    $bLog = $FALSE;
    if (index( lc($a2wConfigPar->getAttribute( $a2w::ConfigConstants::LOGGINGLEVEL )), "sw") >= 0 ){
        $bLog = $TRUE;
    }

    my $sScriptProcTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SCRIPTWRITER );
    my $sScriptArgsTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SCRIPTARGUMENT );
    $sIndexFilePath    = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::INDEXPATH );
    $sOutputFilePath   = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::OUTPUTFILEPATH );
    $iOutputResolution = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::RESOLUTION );
    $sPageOutput       = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::PAGEOUTPUT );
    $sSpoolFilename    = $a2wKernelPar->getSpoolFilename();

    #---- Instantiate Logger
    $theLogger = undef;
    $bDebugLog = $FALSE;
    if ( $bLog == $TRUE ){
        #---- Set log on log engine
        $theLogger = a2w::core::log::Logger->getSingleton();
        $theLogger->setStartTime( $sStartTime );

        #---- Register modules that has to be logged
        $theLogger->registerClasses(   "a2w::Main"
                                     . ",a2w::core::dm::MiningEngine"
                                   );

        #---- Open log file
        # V101 Begin
        #my $sLogFilenameTmp = "";
        #if ( $sSpoolFilename =~ /[\\\/](.*)$/ ){
        #    $sLogFilenameTmp = $1 . ".dmf.log";
        #}
        my $sLogFilenameTmp = $sSpoolFilename;
        $sLogFilenameTmp =~ s/^.*[\\\/]//;
        $sLogFilenameTmp .= ".dmf.log";
        # V101 End
        $theLogger->setFilename( $sLogFilenameTmp );
        if ( $theLogger->open( $sOutputFilePath ) == $FALSE ){
            # V101 Begin
            #return ( -1, "[ERROR] Unable to open log file" );
            return ( -1, "[ERROR] Unable to open log file (" . $sOutputFilePath . $sLogFilenameTmp . ")" );
            # V101 End
        }
        $bLog = $theLogger->isRegistered( "a2w::Main" );

        if (index( lc($a2wConfigPar->getAttribute( $a2w::ConfigConstants::LOGGINGLEVEL )), "swdbg") >= 0 ){
            $theLogger->setLevel( $a2w::core::log::Logger::LEVEL_DEBUG );
            $bDebugLog = $TRUE;
        }

        $theLogger->logFunctionName( "a2w::Main", "initialize()" );
        $theLogger->logMessage( "Running $sScriptProcTmp..." );
        $theLogger->logMessage( "initialize(): Processing " . $sSpoolFilename );
        $theLogger->logMessage( "initialize(): Args: $sScriptArgsTmp, OutputFilePath: $sOutputFilePath, PageOutput: $sPageOutput" );
    }

    #---- Create mining engine
    $dmMiningEngine = new a2w::core::dm::MiningEngine();

    #---- Fill info on mining engine
    $dmMiningEngine->registerClassesForLog();
    $dmMiningEngine->setSpoolFilename( $sSpoolFilename );
    $dmMiningEngine->setOutputPath( $sOutputFilePath );
    $dmMiningEngine->setOutputRes( $iOutputResolution );

    #---- Load config
    my $sBlkDefModTmp = $sScriptArgsTmp;
    eval "require $sBlkDefModTmp";
    my $iRetTmp = $dmMiningEngine->loadConfig( \%{ $sBlkDefModTmp . '::Blocks' } );

    #---- Initialize process variables
    $iDocumentId = 0;
    $iPageId     = 0;

    return $iRetTmp;
}

#-----------------------------------------------------------------------
# InitializeDoc for each document
#-----------------------------------------------------------------------
sub initializeDoc(){

    #---- Get Parameter of initializeDoc( Par: a2w::Document )
    ($a2wDocumentPar) = @_;

    #---- Increment document id
    $iDocumentId++;

    #---- Reset page id
    $iPageId = 0;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( "a2w::Main", "initializeDoc()" );
        $theLogger->logMessage( "Name=" . $a2wDocumentPar->getName() . " Id=" . $iDocumentId );
    }

    #---- Process indexes ----#
    my $iRetTmp = _processDocumentIndexes( $a2wDocumentPar );
    if ( $iRetTmp < 0 ){
        return ( -1, "Unable to process document " . $iDocumentId . " indexes" );
    }

    #---- Initialize document
    my $bPageOutputTmp = ( lc( $sPageOutput ) eq "on" ) ? $TRUE : $FALSE;
    $iRetTmp = $dmMiningEngine->initializeDoc(   'a2w::core::visitor::HTMLVisitor'
                                               , $a2wDocumentPar
                                               , qw( January February March April May June July August September October November December )[ (localtime)[ 4 ] ] . " Statement | eStatements"
                                               , $bPageOutputTmp
                                             );
    return $iRetTmp;
}

#-----------------------------------------------------------------------
# InitializePage for each page
#-----------------------------------------------------------------------
sub initializePage(){

    #---- Get Parameter of initializePage( Par: a2w::Page )
    ($a2wPagePar) = @_;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( "a2w::Main", "initializePage()" );
    }

    #---- Process first page indexes ----#
    $iPageId++;
    my $iRetTmp = 0;
    if ( $iPageId == 1 ){
        $iRetTmp = _processFirstPageIndexes( $a2wPagePar );
        if ( $iRetTmp < 0 ){
            return ( -1, "Unable to process first page indexes of document(" . $iDocumentId . ")" );
        }
    }

    #---- Initialize page
    $iRetTmp = $dmMiningEngine->initializePage( $a2wPagePar );
    return $iRetTmp;
}

#-----------------------------------------------------------------------
# Write text
#-----------------------------------------------------------------------
sub writeText(){

    #---- Get Parameter of writeText( Par: a2w::Text )
    ($a2wTextPar) = @_;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( "a2w::Main", "writeText()" );
    }

    #---- Add text to page
    my $iRetTmp = $dmMiningEngine->addObject( $a2wTextPar );
    return $iRetTmp;
}

#-----------------------------------------------------------------------
# Write line
#-----------------------------------------------------------------------
sub writeLine(){

    #---- Get Parameter of writeLine( Par: a2w::Line )
    ($a2wLinePar) = @_;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( "a2w::Main", "writeLine()" );
    }

    #---- Skip line, if attributes are not correct like zero width, length so on ----#
    if (    $a2wLinePar->getWidth() == 0
         || $a2wLinePar->getLength() == 0
       ){
        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "Line skipped as width or length is zero" );
        }
        return 0;
    }

    #---- Add line to page
    my $iRetTmp = $dmMiningEngine->addObject( $a2wLinePar );
    return $iRetTmp;
}

#-----------------------------------------------------------------------
# Write image
#-----------------------------------------------------------------------
sub writeImage(){

    #---- Get Parameter of writeImage( Par: a2w::Image )
    ($a2wImagePar) = @_;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( "a2w::Main", "writeImage()" );
        if ( $a2wImagePar != undef ){
            $theLogger->logMessage( "img>" . $a2wImagePar->getName() . "<@(" . $a2wImagePar->getXPos() . "," . $a2wImagePar->getYPos() . ") W=" . $a2wImagePar->getWidth() . " H=" . $a2wImagePar->getHeight() );
        }
    }

    #---- Add image to page
    my $iRetTmp = $dmMiningEngine->addObject( $a2wImagePar );
    return $iRetTmp;
}

#-----------------------------------------------------------------------
# Write vector
#-----------------------------------------------------------------------
sub writeVector(){

    #---- Get Parameter of writeVector( Par: a2w::Vector )
    ($a2wVectorPar) = @_;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( "a2w::Main", "writeVector()" );
    }

    #---- Add vector to page
    my $iRetTmp = $dmMiningEngine->addObject( $a2wVectorPar );
    return $iRetTmp;
}

#-----------------------------------------------------------------------
# FinalizePage for each page
#-----------------------------------------------------------------------
sub finalizePage(){

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( "a2w::Main", "finalizePage()" );
    }

    #---- Finalize page
    my $iRetTmp = $dmMiningEngine->finalizePage();
    if ( $iRetTmp < 0 ){
        return $iRetTmp;
    }

    #---- Write page
    $dmMiningEngine->writePage( $a2wPagePar->getParseId() );
    return 0;
}

#-----------------------------------------------------------------------
# FinalizeDoc for each document
#-----------------------------------------------------------------------
sub finalizeDoc(){

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( "a2w::Main", "finalizeDoc()" );
    }

    #---- Finalize document
    if ( $bDebugLog == $TRUE ){
        $dmMiningEngine->dumpDOM( $sOutputFilePath, $TRUE );
    }
    my $iRetTmp = $dmMiningEngine->finalizeDoc();
    return $iRetTmp;
}

#-----------------------------------------------------------------------
# Finalize once per process
#-----------------------------------------------------------------------
sub finalize(){

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( "a2w::Main", "finalize()" );
    }

    #---- Set end time and close log file
    if ( $bLog == $TRUE ){        
        $theLogger->setEndTime( time() );    # set end time
        $theLogger->close();                # close log file
    }

    #---- Cleanup
    $theLogger      = undef;
    $dmMiningEngine = undef;

    return 0;
}

#-----------------------------------------------------------------------
# Process document indexes
#-----------------------------------------------------------------------
sub _processDocumentIndexes(){

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( "a2w::Main", "_processDocumentIndexes()" );
    }

    #---- Get parameter
    #
    # 1. Document
    #
    my $a2wDocPar = shift;

    #---- Process document indexes ----#
    #---- Get first index
    my $sIdxNameTmp  = "";
    my $sIdxValueTmp = "";
    my $a2wIndexTmp  = $a2wDocPar->getFirstIndex();
    while ( $a2wIndexTmp != 0 ){
        #---- Process index
        $sIdxNameTmp = $a2wIndexTmp->getName();
        # $V102 Begin
        $sIdxValueTmp = $a2wIndexTmp->getValue();

        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "  Index: " . $sIdxNameTmp . "=>" . $sIdxValueTmp . "<" );
        }

        #---- Process index name and add blocks
        if ( $sIdxNameTmp ne "" ){
            #$dmMiningEngine->addDocumentBlockId( $sIdxNameTmp );
            if ( $dmMiningEngine->addDocumentBlock( $sIdxNameTmp, $sIdxValueTmp ) < 0 ){ return ( -1, "Error! Adding document block (" . $sIdxNameTmp . ") failed." ); }
        }
        # $V102 End

        #---- Get next index
        $a2wIndexTmp = $a2wDocPar->getNextIndex();
    }

    return 0;
}

#-----------------------------------------------------------------------
# Process first page indexes
#-----------------------------------------------------------------------
sub _processFirstPageIndexes(){

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( "a2w::Main", "_processFirstPageIndexes()" );
    }

    #---- Get parameter
    #
    # 1. Page
    #
    my $a2wPagePar = shift;

    #---- Process page indexes ----#
    #---- Get first index
    my $sIdxNameTmp  = "";
    my $sIdxValueTmp = "";
    my $a2wIndexTmp  = $a2wPagePar->getFirstIndex();
    while ( $a2wIndexTmp != 0 ){
        #---- Process index
        $sIdxNameTmp = $a2wIndexTmp->getName();
        # $V102 Begin
        $sIdxValueTmp = $a2wIndexTmp->getValue();

        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "  Index: " . $sIdxNameTmp . "=>" . $sIdxValueTmp . "<" );
        }

        #---- Process index name and add blocks
        if ( $sIdxNameTmp ne "" ){
            #$dmMiningEngine->addDocumentBlockId( $sIdxNameTmp );
            if ( $dmMiningEngine->addDocumentBlock( $sIdxNameTmp, $sIdxValueTmp ) < 0 ){ return ( -1, "Error! Adding document block (" . $sIdxNameTmp . ") failed." ); }
        }
        # $V102 End

        #---- Get next index
        $a2wIndexTmp = $a2wPagePar->getNextIndex();
    }

    return 0;
}

__END__
