#-------------------------------------------------------------------------------
#  eyecatcher.pm:
#  Scripting Facility Module to demonstrate how to use page content to identify
#  documents and collect indexes
#
#  Call:
#
#  On Windows:    afp2web.exe -q -c -doc_cold -sp:sfsamples\eyecatcher.pm samples\insure.afp
#
#  On Unix:    ./afp2web   -q -c -doc_cold -sp:sfsamples/eyecatcher.pm samples/insure.afp
#
#  Author  : Fa. OXSEED
#  Date    : 2010-01-04
#  Version : 1.0.2
#
#  $V100   2005-08-01    Initial Release
#
#  $V101   2008-07-10    Extended to add script file path as module search path
#                        on "BEGIN" block
#
#  $V102   2010-01-04    a. Extended to utilize
#                           "a2w::ConfigConstants"
#                           "a2w::DocumentConstants"
#                           "a2w::PageConstants"
#                           "a2w::FontConstants"
#                           packages
#                        b. Optimized constant declarations
#                        c. Modified PERL "@INC" extending to consider the
#                           Scripting Facility samples grouping under "sfsamples"
#                           directory
#-------------------------------------------------------------------------------

# $V101 Begin
#-----------------------------------------------------------------------
# BEGIN block of module
#
# Extends PERL module search path array (@INC) with new element having
# this script modules path in order to have better module portability
#-----------------------------------------------------------------------
BEGIN {
    #---- Fetch script filename
    my $sScriptFilenameTmp = $0;

    #---- Extract script file path from script filename
    my $sScriptFilePathTmp = "";
    if ( $sScriptFilenameTmp =~ /(.*)\/.*\.pm/ ){
        $sScriptFilePathTmp = $1;
    }

    #printf STDERR ( "Script filename: " . $0 . " Script filepath: " . $sScriptFilePathTmp . "\n" );
    # $V102 Begin
    if ( $sScriptFilePathTmp eq "" ){
        $sScriptFilePathTmp = ".";
    }
    else {
        my $sScriptFileParentPathTmp = "";
        if ( $sScriptFilePathTmp =~ /(.*)\/sfsamples/ ){
            $sScriptFileParentPathTmp = $1;
        }

        #---- Add script file parent path to module search path
        if ( $sScriptFileParentPathTmp ne "" ){
            unshift( @INC, $sScriptFileParentPathTmp );
        }
    }
    # $V102 End

    #---- Add script file path to module search path
    unshift( @INC, $sScriptFilePathTmp );
}
# $V101 End

use a2w::Config;
use a2w::Document;
use a2w::Font;
use a2w::Index;
use a2w::Kernel;
use a2w::Line;
use a2w::MediumMap;
use a2w::NOP;
use a2w::Overlay;
use a2w::Page;
use a2w::PSEG;
use a2w::Text;
# $V102 Begin
use a2w::ConfigConstants;
use a2w::DocumentConstants;
use a2w::PageConstants;
use a2w::FontConstants;
# $V102 End
#-----------------------------------------------------------------------
# Initialize once per process
#-----------------------------------------------------------------------
sub initialize(){

    #---- Get Parameter of initialize( Par: a2w::Config, a2w::Kernel )
    ( $a2wConfigPar, $a2wKernelPar ) = @_;

    #---- Define boolean values
    $TRUE  = 1;    # TRUE  boolean value
    $FALSE = 0;    # FALSE boolean value

    #---- Set/Reset Logging
    $bLog = $FALSE;
    if (index( lc($a2wConfigPar->getAttribute( $a2w::ConfigConstants::LOGGINGLEVEL )), "sf") >= 0 ){    # $V102 Change
        $bLog = $TRUE;
    }

    my $sScriptProcTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SCRIPTPROCEDURE );    # $V102 Change
    my $sScriptArgsTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SCRIPTARGUMENT );
    $sIndexFilePath    = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::INDEXPATH );
    $sOutputFilePath   = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::OUTPUTFILEPATH );
    $sDocType          = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::OUTPUTFORMAT );
    $sSpoolFilename    = $a2wKernelPar->getSpoolFilename();

    if ( $bLog == $TRUE ){
        printf STDERR ( "Running $sScriptProcTmp...\n" );
        printf STDERR ( "initialize(): Processing $sSpoolFilename\n" );
        printf STDERR ( "initialize(): Args: $sScriptArgsTmp, IndexFilePath: $sIndexFilePath, OutputFilePath: $sOutputFilePath\n" );
    }

    # $V102 Begin
    #---- Page process flags
    $APPEND = 0;    # append page to Current Document
    $SKIP   = 1;    # skip page
    $NEWDOC = 2;    # new document
    # $V102 End

    #---- Reset Page Id
    $iPageId = 0;

    #---- Reset/Create Current Index List
    @arrIndexList = ();

    return 0;
}

#-----------------------------------------------------------------------
# InitializeDoc for each document
#-----------------------------------------------------------------------
sub initializeDoc(){

    #---- Get Parameter of initializeDoc( Par: a2w::Document )
    ($a2wDocumentPar) = @_;

    if ( $bLog == $TRUE ){
        printf STDERR ( "initializeDoc(): DocId " . $a2wDocumentPar->getId() . "\n" );
    }
    return 0;
}

#-----------------------------------------------------------------------
# InitializePage for each page
#-----------------------------------------------------------------------
sub initializePage(){

    #---- Get Parameter of initializePage( Par: a2w::Page )
    ($a2wPagePar) = @_;

    if ( $bLog == $TRUE ){
        printf STDERR ( "initializePage()\n" );
    }
    return 0;
}

#-----------------------------------------------------------------------
# Main entry method
# Return values:
#        < 0:    error
#         0:    append page to Current Document
#         1:    skip page
#         2:    first page / new document
#-----------------------------------------------------------------------
sub afp2web(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "afp2web(): PageId " . $a2wPagePar->getParseId() . "\n" );
    }    

    #---- Set default return value
    my $iRetTmp = $APPEND; # default: append page

    #---- We are looking for "Disability Income Policy" in following
    #---- area: 782<= x <= 802 and 2298 <= y <= 2318... (@(792,2308)>Disability Income Policy<)
    #---- ...to recognize the first page of a document 

    #---- Fetch first Text Object
    my $a2wTextTmp = $a2wPagePar->getFirstText();

    #---- Loop thru all the Text Objects
    while ( $a2wTextTmp != 0 ){
        
        #---- If true ==> new doc
        if ( $a2wTextTmp->getXPos() >= 782
             && $a2wTextTmp->getXPos() <= 802
             && $a2wTextTmp->getYPos() >= 2298
             && $a2wTextTmp->getYPos() <= 2318
             && $a2wTextTmp->getText() eq "Disability Income Policy"
            ){

            #---- Reset Page Id
            $iPageId = 0;

            $iRetTmp = $NEWDOC;
            last; # leave while loop
        }

        #---- Fetch next Text Object
        $a2wTextTmp = $a2wPagePar->getNextText();
    }

    #---- Increment Page Id
    $iPageId++;

    return $iRetTmp;
}

#-----------------------------------------------------------------------
# FinalizePage for each page
#-----------------------------------------------------------------------
sub finalizePage(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "finalizePage()\n" );
    }
    return 0;
}

#-----------------------------------------------------------------------
# FinalizeDoc for each document
#-----------------------------------------------------------------------
sub finalizeDoc(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "finalizeDoc()\n" );
    }

    #---- Fetch Indexes from the first page of the document
    addFirstPageEyecatcherIndexes();

    #---- Build Index Filename   
    my $svSimpleFilenameTmp = $a2wDocumentPar->getSimpleFilename();    # get document simple filename
    if ( $svSimpleFilenameTmp ne "" ){
        
        my $IndexFilenameTmp = $sIndexFilePath . $svSimpleFilenameTmp . ".idx"; # build Index Filename

        if ( $bLog == $TRUE ){
            printf STDERR ( "Writing Index File ==> $IndexFilenameTmp\n" );
        }

        #---- Write Indexes
        my $docIndexCountTmp = @arrIndexList;

        if ( $bLog == $TRUE ){
            printf STDERR ( "Printing Index List\n" );
            for(my $iIndexPtrTmp = 0; $iIndexPtrTmp < $docIndexCountTmp; $iIndexPtrTmp++){
                printf STDERR ( "Index: $iIndexPtrTmp ==> @arrIndexList[$iIndexPtrTmp]\n" );
            }
        }

        #---- Open Index file
        my $bFileOpenSuccessTmp = open( fIndexFile, ">$IndexFilenameTmp" );

        if ( $bFileOpenSuccessTmp ){

            #---- Write Document Id, Type, Name, PageCount and Size
            print fIndexFile (      "DocId="        . $a2wDocumentPar->getId() .
                                ", DocType="    . $sDocType .
                                ", DocName="    . $sOutputFilePath . $a2wDocumentPar->getOutputFilename() .
                                ", PageCount="    . $a2wDocumentPar->getPageCount() . 
                                ", Size="        . $a2wDocumentPar->getSize() . "\n");

            #---- Write Indexes
            for(my $iIndexPtrTmp = 0; $iIndexPtrTmp < $docIndexCountTmp; $iIndexPtrTmp++){
                print fIndexFile ("@arrIndexList[$iIndexPtrTmp]\n");
            }
            print fIndexFile ("\n");
            close( fIndexFile );
        }
        else{
            return(-999, "Unable to open index file $IndexFilenameTmp (rc=$bFileOpenSuccessTmp), leaving...");
        }

        #---- Reset Current Index List
        @arrIndexList = ();
    }
    else{
        return(-998, "Invalid Simple Filename for Document " . $a2wDocumentPar->getOutputFilename() . ", leaving...");
    }
    return 0;
}

#-----------------------------------------------------------------------
# Finalize once per process
#-----------------------------------------------------------------------
sub finalize(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "finalize()\n" );
    }
    return 0;
}

#-----------------------------------------------------------------------
# Add Eyecatcher indexes from the first Page to the index list
#-----------------------------------------------------------------------
sub addFirstPageEyecatcherIndexes(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "addFirstPageEyecatcherIndexes()\n" );
    }

    #---- Fetch the first Page of Document
    my $a2wFirstPageTmp = $a2wDocumentPar->getFirstPage();

    #---- If a Page exists...
    if ( $a2wFirstPageTmp != 0 ){

        #---- Define the list of Eyecatcher positions we should look for
        my @arrEyeCatcherIndexListTmp = ( ( 2448,5353 ),    # Insured        @(2448,5353)>Geoffrey R Stephens<
                                          ( 2448,6097 ),    # Policy Number    @(2448,6097)>324-1443255-11<
                                          ( 7114,6097 ) );    # Date of Issue    @(7114,6097)>03-25-53<

        #---- Define temp variables
        my $arrECILLenTmp    = @arrEyeCatcherIndexListTmp / 2;
        my $sTextTmp        = "";
        my $iTextXPosTmp    = 0;
        my $iTextYPosTmp    = 0;
        my $svArrPosTmp        = 0;
        my $svIdxTextXTmp    = 0;
        my $svIdxTextYTmp    = 0;
        my $svIndexValueTmp    = "";

        #---- Get the Index list count
        my $iIndexPtrTmp = @arrIndexList;

        #---- Fetch first Text Object
        my $a2wTextTmp = $a2wFirstPageTmp->getFirstText();

        #---- Loop thru all the Text Objects
        while ( $a2wTextTmp != 0 ){

            $sTextTmp     = $a2wTextTmp->getText();
            $iTextXPosTmp = $a2wTextTmp->getXPos();
            $iTextYPosTmp = $a2wTextTmp->getYPos();

            #---- Search all the defined Eyecatchers
            for (my $i = 0; $i < $arrECILLenTmp; $i++ ){
                $svArrPosTmp = $i * 2;
                $svIdxTextXTmp = $arrEyeCatcherIndexListTmp[ $svArrPosTmp ];
                $svIdxTextYTmp = $arrEyeCatcherIndexListTmp[ $svArrPosTmp + 1 ];

                #---- If true ==> we found one of the defined Eyecatchers
                if ( $iTextXPosTmp >= ( $svIdxTextXTmp - 10 )    && 
                     $iTextXPosTmp <= ( $svIdxTextXTmp + 10 )    && 
                     $iTextYPosTmp >= ( $svIdxTextYTmp - 10 )    &&
                     $iTextYPosTmp <= ( $svIdxTextYTmp + 10 ) ){

                    if ( $i == 0 ){
                        $svIndexValueTmp = "Insured=" . $sTextTmp;
                    } elsif ( $i == 1 ){
                        $svIndexValueTmp = "Policy Number=" . $sTextTmp;
                    } elsif ( $i == 2 ){
                        $svIndexValueTmp = "Date of Issue=" . $sTextTmp;
                    }

                    #---- Add Eyecatcher to Index List
                    @arrIndexList[$iIndexPtrTmp++] = $svIndexValueTmp;

                    if ( $bLog == $TRUE ){
                        printf STDERR ( "$svIndexValueTmp\n" );
                    }
                } 
            }            
            #---- Fetch next Text Object
            $a2wTextTmp = $a2wFirstPageTmp->getNextText();
        }
    }
}

__END__
