#-------------------------------------------------------------------------------
#  a2w/core/dm/ContentParser.pm
#
#  Perl module to parse page content and to build a database out of page content
#
#  Author   : Panneer, AFP2web Team
#  Date     : 2014-02-14
#  Version  : 1.0.0
#
#  $V100   2014-02-14    Initial Release
#
#  $V101   2015-08-27    Fixed minor bug in evaluating output file name when page
#                        output is turned on
#
#  $V102   2017-10-30    a. Fixed minor bug in evaluating absolute position of page
#                           inline images when page and image resolution differ.
#                        b. Extended to fill in page reference on Database to handle
#                           empty cells for ADA output
#                        c. Extended to process page segment resources also
#
#  $V103   2018-01-10    a. Extended with container objects parsing (AFP-456, AFP-623)
#                        b. Extended to handle included position of objects within resources
#
#  $V104   2018-05-18    a. Removed evaluating absolute position of page inline images
#                           for Script Visitor process (since AFP2web core already pass
#                           absolute position of image)
#
#  $V105   2018-08-09    - AFP-735: Extended to handle tagging for reusable objects (with different tag info
#                          at each presentation of same object)
#                        - AFP-734: Extended to write tagging info at document level instead of page level
#
#-------------------------------------------------------------------------------
package a2w::core::dm::ContentParser;

#-----------------------------------------------------------------------
# Include required modules
#-----------------------------------------------------------------------
use a2w::TypeConstants;
use a2w::core::log::Logger;

use a2w::core::dm::Constants;
use a2w::core::dm::Constraint;
use a2w::core::dm::Database;

#-----------------------------------------------------------------------
# Constructor
#-----------------------------------------------------------------------
sub new{
    my $proto = shift;
    my $class = ref( $proto ) || $proto;

    my $this = {
        'POM' => undef    # Current page database reference
    };

    bless( $this, $class );

    #---- Define boolean values
    $TRUE  = $a2w::TypeConstants::TRUE;     # TRUE  boolean value
    $FALSE = $a2w::TypeConstants::FALSE;    # FALSE boolean value

    #---- Get logger
    our $theLogger = a2w::core::log::Logger->getSingleton();
    our $bLog = $theLogger->isRegistered( __PACKAGE__ );

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "new()" );
    #}

    return $this;
}

#-----------------------------------------------------------------------
# Destructor
#-----------------------------------------------------------------------
sub DESTROY{
    my $this = shift;
}

#-----------------------------------------------------------------------
# Parse (generic interface)
# Returns a database containing the page content
#-----------------------------------------------------------------------
sub parse{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "parse" );
    }

    require a2w::Page;    # Include page module

    #---- Get parameter of parse ( Par: Page instance[, Flag to include page resource contents] )
    my $a2wPagePar = shift;
    my $bIncludePageResPar = $FALSE;

    #---- Get include page resource contents flag, if exist
    if ( @_ ){
        $bIncludePageResPar = shift;
    }

    #---- Parse page content and building database
    my $iPageIdTmp = $a2wPagePar->getParseId();

    #---- Create database
    my $dbPageTmp = new a2w::core::dm::Database( $iPageIdTmp );

    #---- Fill in page details
    $dbPageTmp->setPageRes( $a2wPagePar->getResolution() );
    $dbPageTmp->setPageWidth( $a2wPagePar->getWidth() );
    $dbPageTmp->setPageHeight( $a2wPagePar->getHeight() );
    $dbPageTmp->setPageBackground( $a2wPagePar->getOutputFilename() );
    $dbPageTmp->setPageReference( $a2wPagePar ); # $V102 Change

    #---- Fill Database with page content
    #
    #---- Add Text Objects
    $this->_addTextObjs( $a2wPagePar, $dbPageTmp );

    #---- Add Line Objects
    $this->_addLineObjs( $a2wPagePar, $dbPageTmp );

    #---- Add Vector Objects
    $this->_addVectorObjs( $a2wPagePar, $dbPageTmp );

    #---- Add Image Objects
    $this->_addImageObjs( $a2wPagePar, $dbPageTmp );

    #---- Add Container Objects
    $this->_addContainerObjs( $a2wPagePar, $dbPageTmp ); # $V103 Change

    #---- Add Overlays
    if ( $bIncludePageResPar ){
        $this->_addOverlays( $a2wPagePar, $dbPageTmp );
    }

    # $V102 Begin
    #---- Add Page Segments
    if ( $bIncludePageResPar ){
        $this->_addPageSegments( $a2wPagePar, $dbPageTmp );
    }
    # $V102 End

    return $dbPageTmp;
}

#-----------------------------------------------------------------------
# Initialize page
#
# Returns page database (POM) in case of success, undef in case of error
#-----------------------------------------------------------------------
sub initializePage{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "initializePage" );
    }

    require a2w::Page;    # Include page module

    #---- Get parameter ( Par: Page instance )
    my $a2wPagePar = shift;

    #---- Parse page content and building database
    my $iPageIdTmp = $a2wPagePar->getParseId();

    #---- Create database
    $this->{ 'POM' } = new a2w::core::dm::Database( $iPageIdTmp );
    my $dbPageTmp = $this->{ 'POM' };

    #---- Fill in page details
    $dbPageTmp->setPageRes( $a2wPagePar->getResolution() );
    $dbPageTmp->setPageWidth( $a2wPagePar->getWidth() );
    $dbPageTmp->setPageHeight( $a2wPagePar->getHeight() );
    $dbPageTmp->setPageReference( $a2wPagePar ); # $V102 Change

    # $V101 Begin
    #return 0;
    return $dbPageTmp;
    # $V101 End
}

#-----------------------------------------------------------------------
# Add object
#
# Adds given object to active page database (POM)
#
# Returns 0 in case of success, <0 in case of error
#-----------------------------------------------------------------------
sub addObject{
    my $this = shift;

    #---- Get parameter
    #
    # 1. Object
    #
    my $a2wObjPar = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "addObject" );
    }

    my $dbPageTmp = $this->{ 'POM' };
    my $sObjTypeTmp = $a2wObjPar->_getType();
    if ( $sObjTypeTmp eq "text" ){
        #---- Add text to database
        $dbPageTmp->addObject(   $a2w::core::dm::Constants::OT_TEXT    # Object type
                               , $a2wObjPar->getXPos()                 # X Position
                               , $a2wObjPar->getYPos()                 # Y Position
                               , $a2wObjPar->getAngle()                # Angle
                               , $a2wObjPar->getColor()                # Color
                               , $a2wObjPar                            # Object reference
                             );
    }
    elsif ( $sObjTypeTmp eq "line" ){
        #---- Add line to database
        $dbPageTmp->addObject(   $a2w::core::dm::Constants::OT_LINE    # Object type
                               , $a2wObjPar->getXPos()                 # X Position
                               , $a2wObjPar->getYPos()                 # Y Position
                               #!!! Line does not have Angle property !!!
                               , 0                                     # Angle
                               , $a2wObjPar->getColor()                # Color
                               , $a2wObjPar                            # Object reference
                             );
    }
    elsif ( $sObjTypeTmp eq "image" ){
        # $V102 Begin
        #---- Assert image/page resolution for inline image and evaluate proper position of image
        my $iXPosTmp = $a2wObjPar->getXPos();
        my $iYPosTmp = $a2wObjPar->getYPos();
        # $V104 Begin
        #if ( $a2wObjPar->isInline() == $TRUE ){
        #    my $iImResTmp = $a2wObjPar->getResolution();
        #    my $iPgResTmp = $dbPageTmp->getPageRes();
        #
        #    if ( $iImResTmp != $iPgResTmp ){
        #        my $fScaleTmp = $iPgResTmp / $iImResTmp;
        #        $iXPosTmp *= $fScaleTmp;
        #        $iYPosTmp *= $fScaleTmp;
        #    }
        #}
        # $V104 End
        # $V102 End

        #---- Add image to database
        $dbPageTmp->addObject(   $a2w::core::dm::Constants::OT_IMAGE   # Object type
                               , $iXPosTmp                             # X Position # $V102 Change
                               , $iYPosTmp                             # Y Position # $V102 Change
                               , $a2wObjPar->getAngle()                # Angle
                               #!!! Image does not have Color property !!!
                               , 0                                     # Color
                               , $a2wObjPar                            # Object reference
                             );
    }
    # $V103 Begin
    elsif ( $sObjTypeTmp eq "container" ){
        #---- Add container to database
        $dbPageTmp->addObject(   $a2w::core::dm::Constants::OT_CONTAINER  # Object type
                               , $a2wObjPar->getXPos()                    # X Position
                               , $a2wObjPar->getYPos()                    # Y Position
                               , $a2wObjPar->getAngle()                   # Angle
                               #!!! Container does not have Color property !!!
                               , 0                                        # Color
                               , $a2wObjPar                               # Object reference
                             );
    }
    # $V103 End
    elsif ( $sObjTypeTmp eq "vector" ){
        #---- Add vector to database
        $dbPageTmp->addObject(   $a2w::core::dm::Constants::OT_VECTOR  # Object type
                               , $a2wObjPar->getXPos()                 # X Position
                               , $a2wObjPar->getYPos()                 # Y Position
                               , $a2wObjPar->getAngle()                # Angle
                               , $a2wObjPar->getColor()                # Color
                               , $a2wObjPar                            # Object reference
                             );
    }
    return 0;
}

#-----------------------------------------------------------------------
# Finalize page
#
# Returns page database (POM) in case of success, undef in case of error
#-----------------------------------------------------------------------
sub finalizePage{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "finalizePage" );
    }

    return $this->{ 'POM' };
}

#-----------------------------------------------------------------------
#
# Add the Text objects to the database
#
#-----------------------------------------------------------------------
sub _addTextObjs{
    require a2w::Text;    # Include text module

    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "_addTextObjs" );
    }

    #---- Get parameter of _addTextObjs ( Par: Page/Overlay instance, Database instance, Position Adjustment )
    my $a2wContentObjectPar     = shift;
    my $dbContentPar            = shift;
    # $V103 Begin
    # Position adjustment (OPTIONAL, type hash reference)
    # $hrefPosAdjustmentPar = {
    #     X_INC => X position increment
    #     Y_INC => Y position increment
    # }
    my $hrefPosAdjustmentPar    = { 'X_INC' => 0, 'Y_INC' => 0 };
    if ( @_ > 0 ){ $hrefPosAdjustmentPar = shift; }
    # $V103 End

    #---- Fill with text contents
    #---- Fetch first text
    my $a2wCurTextTmp = $a2wContentObjectPar->getFirstText();

    #---- Loop thru all texts
    while ( $a2wCurTextTmp != 0 ){
        #---- Get text attributes
        my $iXPosTmp  = $hrefPosAdjustmentPar->{ 'X_INC' } + $a2wCurTextTmp->getXPos(); # Get X position # $V103 Change
        my $iYPosTmp  = $hrefPosAdjustmentPar->{ 'Y_INC' } + $a2wCurTextTmp->getYPos(); # Get Y position # $V103 Change
        my $iAngleTmp = $a2wCurTextTmp->getAngle();                                     # Get angle
        my $iColorTmp = $a2wCurTextTmp->getColor();                                     # Get color

        #---- Add text to database
        $dbContentPar->addObject(   $a2w::core::dm::Constants::OT_TEXT    # Object type
                                  , $iXPosTmp                             # X Position
                                  , $iYPosTmp                             # Y Position
                                  , $iAngleTmp                            # Angle
                                  , $iColorTmp                            # Color
                                  , $a2wCurTextTmp                        # Object reference
                                );

        #---- Get the next text
        $a2wCurTextTmp = $a2wContentObjectPar->getNextText();
    } # end-while
}

#-----------------------------------------------------------------------
#
# Add the Line objects to the database
#
#-----------------------------------------------------------------------
sub _addLineObjs{
    require a2w::Line;    # Include line module

    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "_addLineObjs" );
    }

    #---- Get parameter of _addLineObjs ( Par: Page/Overlay instance, Database instance, Position Adjustment )
    my $a2wContentObjectPar     = shift;
    my $dbContentPar            = shift;
    # $V103 Begin
    # Position adjustment (OPTIONAL, type hash reference)
    # $hrefPosAdjustmentPar = {
    #     X_INC => X position increment
    #     Y_INC => Y position increment
    # }
    my $hrefPosAdjustmentPar    = { 'X_INC' => 0, 'Y_INC' => 0 };
    if ( @_ > 0 ){ $hrefPosAdjustmentPar = shift; }
    # $V103 End

    #---- Fill with line contents
    #---- Fetch first line
    my $a2wCurLineTmp = $a2wContentObjectPar->getFirstLine();

    #---- Loop thru all lines
    while ( $a2wCurLineTmp != 0 ){
        #---- Get line attributes
        my $iXPosTmp  = $hrefPosAdjustmentPar->{ 'X_INC' } + $a2wCurLineTmp->getXPos(); # Get X position # $V103 Change
        my $iYPosTmp  = $hrefPosAdjustmentPar->{ 'Y_INC' } + $a2wCurLineTmp->getYPos(); # Get Y position # $V103 Change
        #!!! Line does not have Angle property !!!
        my $iAngleTmp = 0;
        my $iColorTmp = $a2wCurLineTmp->getColor();                                     # Get color

        #---- Add line to database
        $dbContentPar->addObject(   $a2w::core::dm::Constants::OT_LINE    # Object type
                                  , $iXPosTmp                             # X Position
                                  , $iYPosTmp                             # Y Position
                                  , $iAngleTmp                            # Angle
                                  , $iColorTmp                            # Color
                                  , $a2wCurLineTmp                        # Object reference
                                );

        #---- Get the next line
        $a2wCurLineTmp = $a2wContentObjectPar->getNextLine();
    } # end-while
}

#-----------------------------------------------------------------------
#
# Add the Vector objects to the database
#
#-----------------------------------------------------------------------
sub _addVectorObjs{
    require a2w::Vector;    # Include vector module

    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "_addVectorObjs" );
    }

    #---- Get parameter of _addVectorObjs ( Par: Page/Overlay instance, Database instance, Position Adjustment )
    my $a2wContentObjectPar     = shift;
    my $dbContentPar            = shift;
    # $V103 Begin
    # Position adjustment (OPTIONAL, type hash reference)
    # $hrefPosAdjustmentPar = {
    #     X_INC => X position increment
    #     Y_INC => Y position increment
    # }
    my $hrefPosAdjustmentPar    = { 'X_INC' => 0, 'Y_INC' => 0 };
    if ( @_ > 0 ){ $hrefPosAdjustmentPar = shift; }
    # $V103 End

    #---- Fill with vector contents
    #---- Fetch first vector
    my $a2wCurVectorTmp = $a2wContentObjectPar->getFirstVector();

    #---- Loop thru all vectors
    while ( $a2wCurVectorTmp != 0 ){
        #---- Get vector attributes
        my $iXPosTmp  = $hrefPosAdjustmentPar->{ 'X_INC' } + $a2wCurVectorTmp->getXPos(); # Get X position # $V103 Change
        my $iYPosTmp  = $hrefPosAdjustmentPar->{ 'Y_INC' } + $a2wCurVectorTmp->getYPos(); # Get Y position # $V103 Change
        my $iAngleTmp = $a2wCurVectorTmp->getAngle();                                     # Get angle
        my $iColorTmp = $a2wCurVectorTmp->getColor();                                     # Get color

        #---- Add vector to database
        $dbContentPar->addObject(   $a2w::core::dm::Constants::OT_VECTOR    # Object type
                                  , $iXPosTmp                               # X Position
                                  , $iYPosTmp                               # Y Position
                                  , $iAngleTmp                              # Angle
                                  , $iColorTmp                              # Color
                                  , $a2wCurVectorTmp                        # Object reference
                                );

        #---- Get the next vector
        $a2wCurVectorTmp = $a2wContentObjectPar->getNextVector();
    } # end-while
}

#-----------------------------------------------------------------------
#
# Add the Image objects to the database
#
#-----------------------------------------------------------------------
sub _addImageObjs{
    require a2w::Image;    # Include image module

    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "_addImageObjs" );
    }

    #---- Get parameter of _addImageObjs ( Par: Page/Overlay instance, Database instance, Position Adjustment )
    my $a2wContentObjectPar     = shift;
    my $dbContentPar            = shift;
    # $V103 Begin
    # Position adjustment (OPTIONAL, type hash reference)
    # $hrefPosAdjustmentPar = {
    #     X_INC => X position increment
    #     Y_INC => Y position increment
    # }
    my $hrefPosAdjustmentPar    = { 'X_INC' => 0, 'Y_INC' => 0 };
    if ( @_ > 0 ){ $hrefPosAdjustmentPar = shift; }
    # $V103 End

    #---- Fill with image contents
    #---- Fetch first image
    my $a2wCurImageTmp = $a2wContentObjectPar->getFirstImage();

    #---- Loop thru all images
    while ( $a2wCurImageTmp != 0 ){
        #---- Get image attributes
        my $iXPosTmp  = $a2wCurImageTmp->getXPos();  # Get X position # $V103 Change
        my $iYPosTmp  = $a2wCurImageTmp->getYPos();  # Get Y position # $V103 Change
        my $iAngleTmp = $a2wCurImageTmp->getAngle(); # Get angle
        #!!! Image does not have Color property !!!
        my $iColorTmp = 0;

        # $V102 Begin
        #---- Assert image/page resolution for inline image and evaluate proper position of image
        if ( $a2wCurImageTmp->isInline() == $TRUE ){
            my $iImResTmp = $a2wCurImageTmp->getResolution();
            my $iPgResTmp = $dbContentPar->getPageRes();

            if ( $iImResTmp != $iPgResTmp ){
                my $fScaleTmp = $iPgResTmp / $iImResTmp;
                $iXPosTmp *= $fScaleTmp;
                $iYPosTmp *= $fScaleTmp;
            }
        }
        # $V102 End

        $iXPosTmp = $hrefPosAdjustmentPar->{ 'X_INC' } + $iXPosTmp;
        $iYPosTmp = $hrefPosAdjustmentPar->{ 'Y_INC' } + $iYPosTmp;

        #---- Add image to database
        $dbContentPar->addObject(   $a2w::core::dm::Constants::OT_IMAGE    # Object type
                                  , $iXPosTmp                              # X Position
                                  , $iYPosTmp                              # Y Position
                                  , $iAngleTmp                             # Angle
                                  , $iColorTmp                             # Color
                                  , $a2wCurImageTmp                        # Object reference
                                );

        #---- Get the next image
        $a2wCurImageTmp = $a2wContentObjectPar->getNextImage();
    } # end-while
}

# $V103 Begin
#-----------------------------------------------------------------------
#
# Add the Container objects to the database
#
#-----------------------------------------------------------------------
sub _addContainerObjs{
    require a2w::Container;    # Include container module

    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "_addContainerObjs" );
    }

    #---- Get parameter of _addImageObjs ( Par: Page/Overlay instance, Database instance, Position Adjustment )
    my $a2wContentObjectPar     = shift;
    my $dbContentPar            = shift;

    # Position adjustment (OPTIONAL, type hash reference)
    # $hrefPosAdjustmentPar = {
    #     X_INC => X position increment
    #     Y_INC => Y position increment
    # }
    my $hrefPosAdjustmentPar    = { 'X_INC' => 0, 'Y_INC' => 0 };
    if ( @_ > 0 ){ $a2wContentObjectTypePar = shift; }

    #---- Fill with container contents
    #---- Fetch first container
    my $a2wCurContainerTmp = $a2wContentObjectPar->getFirstContainer();

    #---- Loop thru all containers
    while ( $a2wCurContainerTmp != 0 ){
        #---- Get container attributes
        my $iXPosTmp  =   $hrefPosAdjustmentPar->{ 'X_INC' }  # Get X position # $V103 Change
                        + $a2wCurContainerTmp->getIncludedXPosition()
                        + $a2wCurContainerTmp->getXPos(); 
        my $iYPosTmp  = $hrefPosAdjustmentPar->{ 'Y_INC' }    # Get Y position # $V103 Change
                        + $a2wCurContainerTmp->getIncludedYPosition()
                        + $a2wCurContainerTmp->getYPos();
        my $iAngleTmp = $a2wCurContainerTmp->getAngle();      # Get angle
        #!!! Container does not have Color property !!!
        my $iColorTmp = 0;

        #---- Add container to database
        $dbContentPar->addObject(   $a2w::core::dm::Constants::OT_CONTAINER # Object type
                                  , $iXPosTmp                               # X Position
                                  , $iYPosTmp                               # Y Position
                                  , $iAngleTmp                              # Angle
                                  , $iColorTmp                              # Color
                                  , $a2wCurContainerTmp                     # Object reference
                                );

        #---- Get the next image
        $a2wCurContainerTmp = $a2wContentObjectPar->getNextContainer();
    } # end-while
}
# $V103 End

#-----------------------------------------------------------------------
#
# Add the Overlays to the database
#
#-----------------------------------------------------------------------
sub _addOverlays{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "_addOverlays" );
    }

    #---- Get parameter of _addOverlays ( Par: Overlay instance, Database instance )
    my $a2wContentObjectPar = shift;
    my $dbContentPar        = shift;
    
    require a2w::Overlay;    # Include overlay module

    #---- Fetch 1st Overlay
    my $a2wOlyTmp = $a2wContentObjectPar->getFirstOverlay();

    #---- Loop thru all the Overlays
    # $V103 Begin
    my $inclResTmp    = 0;                           # Overlay resolution
    my $pageResTmp    = $dbContentPar->getPageRes(); # Page resolution
    my $hrefPosAdjTmp = { 'X_INC' => 0, 'Y_INC' => 0 };

    my $iResIdTmp     = 0; # Resource id in page # V105 Change

    while ( $a2wOlyTmp != 0 ){
        #---- Fill database with overlay contents
        #
        $hrefPosAdjTmp->{ 'X_INC' } = $a2wOlyTmp->getIncludedXPosition();
        $hrefPosAdjTmp->{ 'Y_INC' } = $a2wOlyTmp->getIncludedYPosition();
        $inclResTmp  = $a2wOlyTmp->getResolution();
        if ( $inclResTmp != $pageResTmp ){
            $hrefPosAdjTmp->{ 'X_INC' } = ( $hrefPosAdjTmp->{ 'X_INC' } / $inclResTmp ) * $pageResTmp;
            $hrefPosAdjTmp->{ 'Y_INC' } = ( $hrefPosAdjTmp->{ 'Y_INC' } / $inclResTmp ) * $pageResTmp;
        }
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Overlay (" . $a2wOlyTmp->getName() . ") included at position (" . $hrefPosAdjTmp->{ 'X_INC' } . "," . $hrefPosAdjTmp->{ 'Y_INC' } . ")" ); }

        # V105 Begin
        #---- Update resource id in database context
        $iResIdTmp = $a2wOlyTmp->getSequenceId();
        if ( $iResIdTmp > 0 ){ $dbContentPar->setResId( $iResIdTmp ); }
        # V105 End

        #---- Fill with text contents
        $this->_addTextObjs( $a2wOlyTmp, $dbContentPar, $hrefPosAdjTmp );

        #---- Fill with line contents
        $this->_addLineObjs( $a2wOlyTmp, $dbContentPar, $hrefPosAdjTmp );

        #---- Add Vector Objects
        $this->_addVectorObjs( $a2wOlyTmp, $dbContentPar, $hrefPosAdjTmp );

        #---- Add Image Objects
        $this->_addImageObjs( $a2wOlyTmp, $dbContentPar, $hrefPosAdjTmp );

        #---- Add Container Objects
        $this->_addContainerObjs( $a2wOlyTmp, $dbContentPar, $hrefPosAdjTmp );

        #---- Reset resource id in database context
        $dbContentPar->setResId( 0 ); # V105 Change

        #---- Get the next Overlay
        $a2wOlyTmp = $a2wContentObjectPar->getNextOverlay();
    } # while ( $a2wOlyTmp != 0 )
    # $V103 End
}

# $V102 Begin
#-----------------------------------------------------------------------
#
# Add the Page Segments to the database
#
#-----------------------------------------------------------------------
sub _addPageSegments{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "_addPageSegments" );
    }

    #---- Get parameter of _addPageSegments ( Par: Page segment instance, Database instance )
    my $a2wContentObjectPar = shift;
    my $dbContentPar        = shift;
    
    require a2w::PSEG;    # Include Page Segment module

    #---- Fetch 1st Page Segment
    my $a2wPSEGTmp = $a2wContentObjectPar->getFirstPageSegment();

    #---- Loop thru all the Page Segments
    # $V103 Begin
    my $hrefPosAdjTmp = { 'X_INC' => 0, 'Y_INC' => 0 };

    my $iResIdTmp     = 0; # Resource id in page # V105 Change

    while ( $a2wPSEGTmp != 0 ){
        #---- Fill database with page segment contents
        #
        $hrefPosAdjTmp->{ 'X_INC' } = $a2wPSEGTmp->getIncludedXPosition();
        $hrefPosAdjTmp->{ 'Y_INC' } = $a2wPSEGTmp->getIncludedYPosition();
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Page Segment (" . $a2wPSEGTmp->getName() . ") included at position (" . $hrefPosAdjTmp->{ 'X_INC' } . "," . $hrefPosAdjTmp->{ 'Y_INC' } . ")" ); }

        # V105 Begin
        #---- Update resource id in database context
        $iResIdTmp = $a2wPSEGTmp->getSequenceId();
        if ( $iResIdTmp > 0 ){ $dbContentPar->setResId( $iResIdTmp ); }
        # V105 End

        #---- Add Vector Objects
        $this->_addVectorObjs( $a2wPSEGTmp, $dbContentPar, $hrefPosAdjTmp );

        #---- Add Image Objects
        $this->_addImageObjs( $a2wPSEGTmp, $dbContentPar, $hrefPosAdjTmp );

        #---- Reset resource id in database context
        $dbContentPar->setResId( 0 ); # V105 Change

        #---- Get the next Page Segment
        $a2wPSEGTmp = $a2wContentObjectPar->getNextPageSegment();
    } # while ( $a2wPSEGTmp != 0 )
    # $V103 End
}
# $V102 End

#-----------------------------------------------------------------------
# Don't remove the following lines !!!
#-----------------------------------------------------------------------
1;
__END__

