#-------------------------------------------------------------------------------
#  a2w/core/dm/Database.pm
#
#  Perl module to hold page content as database for data mining
#
#  Author   : Panneer, AFP2web Team
#
#  $V100   2014-05-08    Initial Release
#
#  $V101   2015-07-20    Fixed minor bug in evaluating output file name when page
#                        output is turned on
#
#  $V102   2015-11-17    a. Extended with page reference attribute for database
#                        b. Extended with "object sequence id" attribute for objects
#
#  $V103   2018-01-10    Extended with container objects processing (AFP-456, AFP-623)
#
#  $V104   2018-08-09    - AFP-735: Extended to handle tagging for reusable objects (with different tag info
#                          at each presentation of same object)
#                        - AFP-734: Extended to write tagging info at document level instead of page level
#
#-------------------------------------------------------------------------------
package a2w::core::dm::Database;

#-----------------------------------------------------------------------
# Include required modules
#-----------------------------------------------------------------------
use a2w::TypeConstants;
use a2w::core::log::Logger;
use a2w::core::dm::Constants;

#-----------------------------------------------------------------------
# Constructor
#-----------------------------------------------------------------------
sub new{
    my $proto = shift;
    my $class = ref( $proto ) || $proto;

    my $this = {
          'ObjectCount'    => 0        # Total Object Count
        , 'MaxXPos'        => 0        # Maximum X position value
        , 'MinXPos'        => 0        # Minimum X position value
        , 'MaxYPos'        => 0        # Maximum Y position value
        , 'MinYPos'        => 0        # Minimum Y position value
        , 'PageID'         => 0        # Page identifier from which objects were filled
        , 'PageWidth'      => 0        # Page width
        , 'PageHeight'     => 0        # Page height
        , 'PageRes'        => 0        # Page resolution
        , 'PageBackground' => ""       # Page background filename
        , 'hshObj'         => {}       # Hash of Object references
        , 'hshObjInfo'     => {}       # Hash of object specific info
        , 'hshXPos'        => {}       # X position hash
        , 'hshYPos'        => {}       # Y position hash
        , 'hshAngle'       => {}       # Angle hash
        , 'ResTable'       => {}       # Resource table
        , 'PageFilename'   => ''       # Page output file name    # $V101 Change
        , 'PageReference'  => undef    # Reference to core page   # $V102 Change
        , 'ResId'          => 0        # Active resource id       # $V104 Change
    };

    bless( $this, $class );

    #---- Define boolean values
    $TRUE  = $a2w::TypeConstants::TRUE;     # TRUE  boolean value
    $FALSE = $a2w::TypeConstants::FALSE;    # FALSE boolean value

    #---- Get logger
    our $theLogger = a2w::core::log::Logger->getSingleton();
    our $bLog = $theLogger->isRegistered( __PACKAGE__ );

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "new()" );
    #}

    #---- Fetch parameter(s)
    #
    # 1. Page id
    #
    if ( @_ ){
        $this->{ 'PageID' } = shift;
    }
    return $this;
}

#-----------------------------------------------------------------------
# Destructor
#-----------------------------------------------------------------------
sub DESTROY{
    my $this = shift;
}

#-----------------------------------------------------------------------
# Mutators
#-----------------------------------------------------------------------
sub setPageID{
    my $this = shift;
    $this->{ 'PageID' } = shift;
}

sub setPageWidth{
    my $this = shift;
    $this->{ 'PageWidth' } = shift;
}

sub setPageHeight{
    my $this = shift;
    $this->{ 'PageHeight' } = shift;
}

sub setPageRes{
    my $this = shift;
    $this->{ 'PageRes' } = shift;
}

sub setPageBackground{
    my $this = shift;
    $this->{ 'PageBackground' } = shift;
}

# $V101 Begin
sub setPageFilename{
    my $this = shift;
    $this->{ 'PageFilename' } = shift;
}
# $V101 End

# $V102 Begin
sub setPageReference{
    my $this = shift;
    $this->{ 'PageReference' } = shift;
}
# $V102 End

# $V104 Begin
sub setResId{
    my $this = shift;
    $this->{ 'ResId' } = shift;
}
# $V104 End

#-----------------------------------------------------------------------
# Accessors
#-----------------------------------------------------------------------
sub getObjectCount{
    my $this = shift;
    return $this->{ 'ObjectCount' };
}

sub getMaxXPos{
    my $this = shift;
    return $this->{ 'MaxXPos' };
}

sub getMinXPos{
    my $this = shift;
    return $this->{ 'MinXPos' };
}

sub getMaxYPos{
    my $this = shift;
    return $this->{ 'MaxYPos' };
}

sub getMinYPos{
    my $this = shift;
    return $this->{ 'MinYPos' };
}

sub getPageID{
    my $this = shift;
    return $this->{ 'PageID' };
}

sub getPageWidth{
    my $this = shift;
    return $this->{ 'PageWidth' };
}

sub getPageHeight{
    my $this = shift;
    return $this->{ 'PageHeight' };
}

sub getPageRes{
    my $this = shift;
    return $this->{ 'PageRes' };
}

sub getPageBackground{
    my $this = shift;
    return $this->{ 'PageBackground' };
}

sub getHashObj{
    my $this = shift;
    return \%{ $this->{ 'hshObj' } };
}

sub getHashObjInfo{
    my $this = shift;
    return \%{ $this->{ 'hshObjInfo' } };
}

sub getHashXPos{
    my $this = shift;
    return \%{ $this->{ 'hshXPos' } };
}

sub getHashYPos{
    my $this = shift;
    return \%{ $this->{ 'hshYPos' } };
}

sub getHashAngle{
    my $this = shift;
    return \%{ $this->{ 'hshAngle' } };
}

sub getResourceTable{
    my $this = shift;
    return \%{ $this->{ 'ResTable' } };
}

# $V101 Begin
sub getPageFilename{
    my $this = shift;
    return $this->{ 'PageFilename' };
}
# $V101 End

# $V102 Begin
sub getPageReference{
    my $this = shift;
    return $this->{ 'PageReference' };
}
# $V102 End

# $V104 Begin
sub getResId{
    my $this = shift;
    return $this->{ 'ResId' };
}
# $V104 End

#-----------------------------------------------------------------------
# Workers
#-----------------------------------------------------------------------
#-----------------------------------------------------------------------
# Add object
#-----------------------------------------------------------------------
sub addObject{
    my $this = shift;

    #---- Get parameters
    #
    # 1. Object Type (Text, Line so on)
    # 2. X Position
    # 3. Y Position
    # 4. Angle
    # 5. Color
    # 6. Object Reference
    my (   $ObjectTypePar
         , $iXPosPar
         , $iYPosPar
         , $iAnglePar
         , $iColorPar
         , $a2wObjectPar
    ) = @_;

    if ( $bLog == $TRUE ){
        my $sObjTypeTmp = "";
        my $sObjInfoStringTmp = "";

        if ( $ObjectTypePar == $a2w::core::dm::Constants::OT_TEXT ){
            $sObjTypeTmp = "Text";
            $sObjInfoStringTmp = " Length=" . $a2wObjectPar->getTextLen()
                               . " Text=>"  . $a2wObjectPar->getText() . "<";
        }
        elsif ( $ObjectTypePar == $a2w::core::dm::Constants::OT_LINE ){
            $sObjTypeTmp = "Line";
            $sObjInfoStringTmp = " Width="  . $a2wObjectPar->getWidth()
                               . " Length=" . $a2wObjectPar->getLength();
        }
        elsif ( $ObjectTypePar == $a2w::core::dm::Constants::OT_VECTOR ){
            $sObjTypeTmp = "Vector";
            $sObjInfoStringTmp = " Width="  . $a2wObjectPar->getWidth()
                               . " Height=" . $a2wObjectPar->getHeight();
        }
        elsif ( $ObjectTypePar == $a2w::core::dm::Constants::OT_IMAGE ){
            $sObjTypeTmp = "Image";
            $sObjInfoStringTmp =   " Name="  . $a2wObjectPar->getName() # $V103 Change
                                 . " Width="  . $a2wObjectPar->getWidth()
                                 . " Height=" . $a2wObjectPar->getHeight();
        }
        # $V103 Begin
        elsif ( $ObjectTypePar == $a2w::core::dm::Constants::OT_CONTAINER ){
            $sObjTypeTmp = "Container";
            $sObjInfoStringTmp =   " Name="  . $a2wObjectPar->getName()
                                 . " ObjectType="  . $a2wObjectPar->getObjectTypeName()
                                 . " Width="  . $a2wObjectPar->getWidth()
                                 . " Height=" . $a2wObjectPar->getHeight();
        }
        # $V103 End
        $theLogger->logFunctionName( __PACKAGE__, "addObject: " . $sObjTypeTmp . " @(" . $iXPosPar . ", " . $iYPosPar . ") Rot=" . $iAnglePar . $sObjInfoStringTmp );
    }

    #---- Create object of database
    $this->{ 'ObjectCount' }++;         #---- Increment object count

    #---- Set maximum & minimum X/Y positions
    #---- Initialize, if it is first time
    if ( $this->{ 'ObjectCount' } == 1 ){
        $this->{ 'MaxXPos' } = $iXPosPar;
        $this->{ 'MinXPos' } = $iXPosPar;
        $this->{ 'MaxYPos' } = $iYPosPar;
        $this->{ 'MinYPos' } = $iYPosPar;
    }
    else {
        if ( $this->{ 'MaxXPos' } < $iXPosPar ){
            $this->{ 'MaxXPos' } = $iXPosPar;
        }

        if ( $this->{ 'MinXPos' } > $iXPosPar ){
            $this->{ 'MinXPos' } = $iXPosPar;
        }

        if ( $this->{ 'MaxYPos' } < $iYPosPar ){
            $this->{ 'MaxYPos' } = $iYPosPar;
        }

        if ( $this->{ 'MinYPos' } > $iYPosPar ){
            $this->{ 'MinYPos' } = $iYPosPar;
        }
    }

    my $iObjSequenceIDTmp = $this->{ 'ObjectCount' };

    if ( $bLog == $TRUE ){
        $theLogger->logMessage( "Object Sequence ID:>" . $iObjSequenceIDTmp . "<" );
    }

    #---- Set object reference on objects list
    $this->{ 'hshObj' }{ $iObjSequenceIDTmp }{ 'OBJECT' } = $a2wObjectPar;

    #---- Get object reference
    my $hrefObjectTmp = \%{ $this->{ 'hshObj' }{ $iObjSequenceIDTmp } };

    #---- Set attribute references on object database
    $hrefObjectTmp->{ $a2w::core::dm::Constants::AT_OBJTYPE } = $ObjectTypePar;

    # $V102 Begin
    $hrefObjectTmp->{ $a2w::core::dm::Constants::AT_OBJSEQID } = $iObjSequenceIDTmp; # Object sequence id
    # $V102 End

    # V104 Begin
    $hrefObjectTmp->{ $a2w::core::dm::Constants::AT_ID } = $this->getResId();
    $hrefObjectTmp->{ $a2w::core::dm::Constants::AT_ID } <<= 16;
    $hrefObjectTmp->{ $a2w::core::dm::Constants::AT_ID } |= $a2wObjectPar->getSequenceId();
    # V104 End

    if ( $ObjectTypePar == $a2w::core::dm::Constants::OT_TEXT ){
        $hrefObjectTmp->{ $a2w::core::dm::Constants::AT_OBJINFO } = {
              $a2w::core::dm::Constants::OI_TEXT_VALUE  => $a2wObjectPar->getText()     # value
            , $a2w::core::dm::Constants::OI_TEXT_LENGTH => $a2wObjectPar->getTextLen()  # length
        };

        #---- Update resource table with text applied font resource info ----#
        #---- Fetch text font
        my $a2wFontTmp = $a2wObjectPar->getFont();

        #---- Update font table, if font is added to it already
        my $iFontLocalIdTmp = $this->{ 'PageID' } . $a2wObjectPar->getMappedFontLocalId();
        $hrefObjectTmp->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_FONTID } = $iFontLocalIdTmp;
        $hrefObjectTmp->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_FONTHT } = $a2wFontTmp->getHeight();

        #---- Update resource table with text used font
        if ( not defined( $this->{ 'ResTable' }{ $iFontLocalIdTmp } ) ){
            $this->updateResourceTable( # Unique resource id
                                           $iFontLocalIdTmp
                                        # Resource
                                        , $a2wFontTmp
                                        # Resource type
                                        , $a2w::core::dm::Constants::RT_FONT
                                        # Object reference
                                        , $a2wObjectPar
                                      );
        }
    }
    elsif ( $ObjectTypePar == $a2w::core::dm::Constants::OT_LINE ){
        $hrefObjectTmp->{ $a2w::core::dm::Constants::AT_OBJINFO } = {
              $a2w::core::dm::Constants::OI_LINE_WIDTH  => $a2wObjectPar->getWidth()    # width
            , $a2w::core::dm::Constants::OI_LINE_LENGTH => $a2wObjectPar->getLength()   # length
        };
    }
    elsif ( $ObjectTypePar == $a2w::core::dm::Constants::OT_VECTOR ){
        $hrefObjectTmp->{ $a2w::core::dm::Constants::AT_OBJINFO } = {
              $a2w::core::dm::Constants::OI_VECTOR_WIDTH  => $a2wObjectPar->getWidth()  # width
            , $a2w::core::dm::Constants::OI_VECTOR_HEIGHT => $a2wObjectPar->getHeight() # height
        };
    }
    elsif ( $ObjectTypePar == $a2w::core::dm::Constants::OT_IMAGE ){
        $hrefObjectTmp->{ $a2w::core::dm::Constants::AT_OBJINFO } = {
              $a2w::core::dm::Constants::OI_IMAGE_WIDTH  => $a2wObjectPar->getWidth()          # width
            , $a2w::core::dm::Constants::OI_IMAGE_HEIGHT => $a2wObjectPar->getHeight()         # height
            , $a2w::core::dm::Constants::OI_IMAGE_NAME => $a2wObjectPar->getName()             # name
            , $a2w::core::dm::Constants::OI_IMAGE_RESOLUTION => $a2wObjectPar->getResolution() # resolution
        };
    }
    # $V103 Begin
    elsif ( $ObjectTypePar == $a2w::core::dm::Constants::OT_CONTAINER ){
        $hrefObjectTmp->{ $a2w::core::dm::Constants::AT_OBJINFO } = {
              $a2w::core::dm::Constants::OI_CONTAINER_WIDTH  => $a2wObjectPar->getWidth()              # width
            , $a2w::core::dm::Constants::OI_CONTAINER_HEIGHT => $a2wObjectPar->getHeight()             # height
            , $a2w::core::dm::Constants::OI_CONTAINER_NAME => $a2wObjectPar->getName()                 # name
            , $a2w::core::dm::Constants::OI_CONTAINER_RESOLUTION => $a2wObjectPar->getResolution()     # resolution
            , $a2w::core::dm::Constants::OI_CONTAINER_OBJECTTYPE => $a2wObjectPar->getObjectTypeName() # object type
        };
    }
    # $V103 End
    $hrefObjectTmp->{ $a2w::core::dm::Constants::AT_XPOS       } = $iXPosPar;
    $hrefObjectTmp->{ $a2w::core::dm::Constants::AT_YPOS       } = $iYPosPar;
    $hrefObjectTmp->{ $a2w::core::dm::Constants::AT_ANGLE      } = ( $iAnglePar == 360 ) ? 0 : $iAnglePar;
    $hrefObjectTmp->{ $a2w::core::dm::Constants::AT_COLOR      } = $iColorPar;
    $hrefObjectTmp->{ $a2w::core::dm::Constants::AT_RESOLUTION } = $this->{ 'PageRes' };

    #---- Add attributes
    $this->{ 'hshObjInfo' }{ $ObjectTypePar }{ $iObjSequenceIDTmp } = $hrefObjectTmp->{ $a2w::core::dm::Constants::AT_OBJINFO };
    $this->{ 'hshXPos' }{ $iXPosPar }{ $iObjSequenceIDTmp }         = $hrefObjectTmp;
    $this->{ 'hshYPos' }{ $iYPosPar }{ $iObjSequenceIDTmp }         = $hrefObjectTmp;
    $this->{ 'hshAngle' }{ $iAnglePar }{ $iObjSequenceIDTmp }       = $hrefObjectTmp;
}

#-----------------------------------------------------------------------
# Get objects
# Returns a hash which has following format:
# ____________________________________
# Key    Value
# ____________________________________
# 0      Header
# 1..n   Values
# ____________________________________
#-----------------------------------------------------------------------
sub getObjects{
    my $this = shift;

    #---- Get parameter of getObjects ( Par: Attribute type, StartValue[, EndValue] )
    my $attrTypePar = shift;
    my $attrValuePar = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "getObjects($attrTypePar,$attrValuePar)" );
    }

    #---- Assert parameter
    if ( $attrValuePar == -1 ){
        if ( $attrTypePar == $a2w::core::dm::Constants::AT_XPOS ){
            $attrValuePar = $this->getMinXPos();

            #---- Get attribute specific hash table
            $hshrefAttrTmp = $this->getHashXPos();
        }
        elsif ( $attrTypePar == $a2w::core::dm::Constants::AT_YPOS ){
            $attrValuePar = $this->getMinYPos();

            #---- Get attribute specific hash table
            $hshrefAttrTmp = $this->getHashYPos();
        }
    }

    my %hshRetTmp = ();         #---- Return value

    #---- Initialize header section of return value
    %hshRetTmp->{ 0 }{ 'COUNT' } = 0;
    %hshRetTmp->{ 0 }{ 'KEYS' } = ();
    %hshRetTmp->{ 0 }{ 'ACTIVEKEYINDEX' } = 0;
    %hshRetTmp->{ 0 }{ 'POSITIVEDIFF' } = 0;
    %hshRetTmp->{ 0 }{ 'NEGATIVEDIFF' } = 0;

    if ( @_ ){      #---- We've end value of search range
        my $iStartValuePar = $attrValuePar;
        my $iEndValuePar = shift;

        #---- Assert parameter
        if ( $iEndValuePar == -1 ){
            if ( $attrTypePar == $a2w::core::dm::Constants::AT_XPOS ){
                $iEndValuePar = $this->getMaxXPos();
            }
            elsif ( $attrTypePar == $a2w::core::dm::Constants::AT_YPOS ){
                $iEndValuePar = $this->getMaxYPos();
            }
        }

        #---- Create list of all values
        my @arrValueListTmp = sort {$a <=> $b} keys( %{ $hshrefAttrTmp } ) ;

        #---- Refine the list of all values to list of required values
        my $i = 0;
        my $iIndexTmp = 0;
        my @arrRangeListTmp = ();       #---- List to hold values fall in required range
        for ( $i = 0; $i < @arrValueListTmp; $i++ ){
            if ( $arrValueListTmp[ $i ] >= $iStartValuePar &&
                 $arrValueListTmp[ $i ] <= $iEndValuePar ){
                @arrRangeListTmp[ $iIndexTmp++ ] = $arrValueListTmp[ $i ];
            }

            if ( $arrValueListTmp[ $i ] == $iEndValuePar ){
                last;   #---- break the loop
            }
        }

        #---- Build list of objects
        $iIndexTmp = 1;
        for ( $i = 0; $i < @arrRangeListTmp; $i++ ){
            #---- Get object for current value
            %hshValueTmp = %{ $hshrefAttrTmp->{ $arrRangeListTmp[ $i ] } };

            foreach my $sKeyTmp ( sort {$a <=> $b} keys( %hshValueTmp ) ){
                my $hrefObjTmp = %hshValueTmp->{ $sKeyTmp };

                %hshRetTmp->{ $iIndexTmp }{ $a2w::core::dm::Constants::AT_OBJTYPE } = $hrefObjTmp->{ $a2w::core::dm::Constants::AT_OBJTYPE };
                %hshRetTmp->{ $iIndexTmp }{ $a2w::core::dm::Constants::AT_OBJINFO } = $hrefObjTmp->{ $a2w::core::dm::Constants::AT_OBJINFO };
                %hshRetTmp->{ $iIndexTmp }{ $a2w::core::dm::Constants::AT_XPOS    } = $hrefObjTmp->{ $a2w::core::dm::Constants::AT_XPOS    };
                %hshRetTmp->{ $iIndexTmp }{ $a2w::core::dm::Constants::AT_YPOS    } = $hrefObjTmp->{ $a2w::core::dm::Constants::AT_YPOS    };
                %hshRetTmp->{ $iIndexTmp }{ $a2w::core::dm::Constants::AT_ANGLE   } = $hrefObjTmp->{ $a2w::core::dm::Constants::AT_ANGLE   };

                $iIndexTmp++;
            }
        }

        #---- Fill header section of return value
        %hshRetTmp->{ 0 }{ 'COUNT' } = $iIndexTmp;
        %hshRetTmp->{ 0 }{ 'KEYS' } = \@arrRangeListTmp;
    }
    else {          #---- We don't have end value of search range
        #---- Get object for given value
        %hshValueTmp = %{ $hshrefAttrTmp->{ $attrValuePar } };

        my $iIndexTmp = 1;
        foreach my $sKeyTmp ( sort {$a <=> $b} keys( %hshValueTmp ) ){
            my $hrefObjTmp = %hshValueTmp->{ $sKeyTmp };

            %hshRetTmp->{ $iIndexTmp }{ $a2w::core::dm::Constants::AT_OBJTYPE } = $hrefObjTmp->{ $a2w::core::dm::Constants::AT_OBJTYPE };
            %hshRetTmp->{ $iIndexTmp }{ $a2w::core::dm::Constants::AT_OBJINFO } = $hrefObjTmp->{ $a2w::core::dm::Constants::AT_OBJINFO };
            %hshRetTmp->{ $iIndexTmp }{ $a2w::core::dm::Constants::AT_XPOS    } = $hrefObjTmp->{ $a2w::core::dm::Constants::AT_XPOS    };
            %hshRetTmp->{ $iIndexTmp }{ $a2w::core::dm::Constants::AT_YPOS    } = $hrefObjTmp->{ $a2w::core::dm::Constants::AT_YPOS    };
            %hshRetTmp->{ $iIndexTmp }{ $a2w::core::dm::Constants::AT_ANGLE   } = $hrefObjTmp->{ $a2w::core::dm::Constants::AT_ANGLE   };

            $iIndexTmp++;
        }

        #---- Fill header section of return value
        %hshRetTmp->{ 0 }{ 'COUNT' } = $iIndexTmp;
        my @arrTemp = ( $attrValuePar );
        %hshRetTmp->{ 0 }{ 'KEYS' } = \@arrTemp;
    }
    return %hshRetTmp;
}

#-----------------------------------------------------------------------
# Get first result set
# Returns a hash which has following format:
# ____________________________________
# Key    Value
# ____________________________________
# 0      Header
# 1..n   Values
# ____________________________________
#-----------------------------------------------------------------------
sub getFirstResultSet{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "getFirstResultSet" );
    }
    return $this->getResultSet( @_ );
}

#-----------------------------------------------------------------------
# Get next result set
# Returns a hash which has following format:
# ____________________________________
# Key    Value
# ____________________________________
# 0      Header
# 1..n   Values
# ____________________________________
#-----------------------------------------------------------------------
sub getNextResultSet{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "getNextResultSet" );
    }
    return $this->getResultSet( @_ );
}

#-----------------------------------------------------------------------
# Get result set
#
# Parameters:
# 1. Hash reference of objects
# 2. Attribute type
# 3. Tolerable positive position difference (optional)
# 4. Tolerable negative position difference (optional)
#
# Returns a hash which has following format:
# ____________________________________
# Key    Value
# ____________________________________
# 0      Header
# 1..n   Values
# ____________________________________
#-----------------------------------------------------------------------
sub getResultSet{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "getResultSet" );
    }

    #---- Get parameter of getResultSet
    # Par:
    # Hash reference,
    # Attribute type,
    # Tolerable positive position difference
    # Tolerable negative position difference
    my $hshrefListPar = shift;
    my $attrTypePar = shift;
    my $iPositiveDiffPar = 0;
    my $iNegativeDiffPar = 0;

    if ( @_ ){
        $iPositiveDiffPar = shift;
        $hshrefListPar->{ 0 }{ 'POSITIVEDIFF' } = $iPositiveDiffPar;
    }

    if ( @_ ){
        $iNegativeDiffPar = shift;
        $hshrefListPar->{ 0 }{ 'NEGATIVEDIFF' } = $iNegativeDiffPar;
    }

    my %hshRetTmp = ();         #---- Return value

    #---- Get header section of return value
    my $iCountTmp = $hshrefListPar->{ 0 }{ 'COUNT' };
    my @arrKeysTmp = @{ $hshrefListPar->{ 0 }{ 'KEYS' } };
    my $iActiveKeyIndexTmp = $hshrefListPar->{ 0 }{ 'ACTIVEKEYINDEX' };
    my $sActiveKeyTmp = @arrKeysTmp[ $iActiveKeyIndexTmp ];

    if ( $iActiveKeyIndexTmp > @arrKeysTmp ){
        return %hshRetTmp;
    }

    #---- Initialize header section of return value
    %hshRetTmp->{ 0 }{ 'COUNT' } = 0;
    %hshRetTmp->{ 0 }{ 'KEYS' } = ();
    %hshRetTmp->{ 0 }{ 'ACTIVEKEYINDEX' } = 0;
    %hshRetTmp->{ 0 }{ 'POSITIVEDIFF' } = 0;
    %hshRetTmp->{ 0 }{ 'NEGATIVEDIFF' } = 0;

    my $iIndexTmp = 1;
    my @arrValuesTmp = ();

    my $iMaxValueFoundTmp = -1;
    my $iMinValueTmp = $sActiveKeyTmp - $hshrefListPar->{ 0 }{ 'NEGATIVEDIFF' };
    my $iMaxValueTmp = $sActiveKeyTmp + $hshrefListPar->{ 0 }{ 'POSITIVEDIFF' };

    for ( my $i = 1; $i < $iCountTmp; $i++ ){
        my $hrefObjTmp = \%{ $hshrefListPar->{ $i } };
        my $iValueTmp = $hrefObjTmp->{ $attrTypePar };
        
        if ( $iValueTmp >= $iMinValueTmp &&
             $iValueTmp <= $iMaxValueTmp ){
            if ( $iValueTmp > $iMaxValueFoundTmp || $iMaxValueFoundTmp == -1 ){
                $iMaxValueFoundTmp = $iValueTmp;
            }

            $arrValuesTmp[ ( $iIndexTmp - 1 ) ] = $iValueTmp;

            %hshRetTmp->{ $iIndexTmp }{ $a2w::core::dm::Constants::AT_OBJTYPE } = $hrefObjTmp->{ $a2w::core::dm::Constants::AT_OBJTYPE };
            %hshRetTmp->{ $iIndexTmp }{ $a2w::core::dm::Constants::AT_XPOS    } = $hrefObjTmp->{ $a2w::core::dm::Constants::AT_XPOS };
            %hshRetTmp->{ $iIndexTmp }{ $a2w::core::dm::Constants::AT_YPOS    } = $hrefObjTmp->{ $a2w::core::dm::Constants::AT_YPOS };
            %hshRetTmp->{ $iIndexTmp }{ $a2w::core::dm::Constants::AT_ANGLE   } = $hrefObjTmp->{ $a2w::core::dm::Constants::AT_ANGLE };

            $iIndexTmp++;
        }
    }

    #---- Change active key index to next value
    for ( my $j = $iActiveKeyIndexTmp; $j < @arrKeysTmp; $j++ ){
        if ( @arrKeysTmp[ $j ] == $iMaxValueFoundTmp ){
            $hshrefListPar->{ 0 }{ 'ACTIVEKEYINDEX' } = $j + 1;

            last;       #---- break the loop
        }
    }

    #---- Fill header section of return value
    %hshRetTmp->{ 0 }{ 'COUNT' } = $iIndexTmp;
    %hshRetTmp->{ 0 }{ 'KEYS' } = \@arrValuesTmp;

    return %hshRetTmp;
}

#-----------------------------------------------------------------------
# Update resource table
#-----------------------------------------------------------------------
sub updateResourceTable{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "updateResourceTable()" );
    }

    #---- Fetch parameters
    #
    # 1. Unique resource id
    # 2. Resource reference
    # 3. Resource type
    # 4. Object reference
    #
    my $iUniqueResIdPar = shift;
    my $a2wResourcePar  = shift;
    my $iResTypePar     = shift;
    my $a2wObjectPar    = shift;

    #---- Get resource table reference
    my $hrefResTableTmp = $this->{ 'ResTable' };

    #---- Update resource table with given resource info
    $hrefResTableTmp->{ $iUniqueResIdPar } = {
          'NAME'      => $a2wResourcePar->getName()
        , 'TYPE'      => $iResTypePar
    };

    if ( $iResTypePar == $a2w::core::dm::Constants::RT_FONT ){
        $hrefResTableTmp->{ $iUniqueResIdPar }{ 'FONT' } = {
              'BOLD'      => $a2wResourcePar->isBold()
            , 'ITALIC'    => $a2wResourcePar->isItalic()
            , 'HEIGHT'    => $a2wResourcePar->getHeight()
            , 'FAMILY'    => $a2wResourcePar->getTypefaceName()
        };
    }
    elsif ( $iResTypePar == $a2w::core::dm::Constants::RT_PSEG ){
        $hrefResTableTmp->{ $iUniqueResIdPar }{ 'PSEG' } = {
              'XPOS'      => $a2wResourcePar->getIncludedXPosition()
            , 'YPOS'      => $a2wResourcePar->getIncludedYPosition()
        };
    }
    elsif ( $iResTypePar == $a2w::core::dm::Constants::RT_OVERLAY ){
        $hrefResTableTmp->{ $iUniqueResIdPar }{ 'OVERLAY' } = {
              'XPOS'      => $a2wResourcePar->getIncludedXPosition()
            , 'YPOS'      => $a2wResourcePar->getIncludedYPosition()
            , 'WIDTH'     => $a2wResourcePar->getWidth()
            , 'HEIGHT'    => $a2wResourcePar->getHeight()
        };
    }

    if ( $bLog == $TRUE ){
        if ( $iResTypePar == $a2w::core::dm::Constants::RT_FONT ){
            $theLogger->logMessage(   "Resource Details:"
                                    . " Name=" . $hrefResTableTmp->{ $iUniqueResIdPar }{ 'NAME' }
                                    . " Type=Font"
                                    . " Bold=" . ( ( $hrefResTableTmp->{ $iUniqueResIdPar }{ 'FONT' }{ 'BOLD' } == $TRUE ) ? "Y" : "N" )
                                    . " Italic=" . ( ( $hrefResTableTmp->{ $iUniqueResIdPar }{ 'FONT' }{ 'ITALIC' } == $TRUE ) ? "Y" : "N" )
                                    . " Height=" . $hrefResTableTmp->{ $iUniqueResIdPar }{ 'FONT' }{ 'HEIGHT' }
                                    . " Family=" . $hrefResTableTmp->{ $iUniqueResIdPar }{ 'FONT' }{ 'FAMILY' }
                                  );
        }
        elsif ( $iResTypePar == $a2w::core::dm::Constants::RT_PSEG ){
            $theLogger->logMessage(   "Resource Details:"
                                    . " Name=" . $hrefResTableTmp->{ $iUniqueResIdPar }{ 'NAME' }
                                    . " Type=Page Segment"
                                    . " XPos=" . $hrefResTableTmp->{ $iUniqueResIdPar }{ 'PSEG' }{ 'XPOS' }
                                    . " YPos=" . $hrefResTableTmp->{ $iUniqueResIdPar }{ 'PSEG' }{ 'YPOS' }
                                  );
        }
        elsif ( $iResTypePar == $a2w::core::dm::Constants::RT_OVERLAY ){
            $theLogger->logMessage(   "Resource Details:"
                                    . " Name=" . $hrefResTableTmp->{ $iUniqueResIdPar }{ 'NAME' }
                                    . " Type=Overlay"
                                    . " XPos=" . $hrefResTableTmp->{ $iUniqueResIdPar }{ 'OVERLAY' }{ 'XPOS' }
                                    . " YPos=" . $hrefResTableTmp->{ $iUniqueResIdPar }{ 'OVERLAY' }{ 'YPOS' }
                                    . " Width=" . $hrefResTableTmp->{ $iUniqueResIdPar }{ 'OVERLAY' }{ 'WIDTH' }
                                    . " Height=" . $hrefResTableTmp->{ $iUniqueResIdPar }{ 'OVERLAY' }{ 'HEIGHT' }
                                  );
        }
    }

    return 0;
}

#-----------------------------------------------------------------------
# Don't remove the following lines !!!
#-----------------------------------------------------------------------
1;
__END__

