/**-----------------------------------------------------------------------------
 * expressAppUtils.js: Node.js module that provides function to handle Express HTTP routes
 * 
 * Author    :  AFP2web Team
 * Copyright :  (C) 2014 by Maas Holding GmbH
 * Email     :  support@oxseed.de
 * Version   :  V1.0.0
 * 
 * History
 *  V100        02.03.2015     Initial release
 *
 *----------------------------------------------------------------------------*/
'use strict';

var MODULE_NAME		= 'expressAppUtils'
  , MODULE_VERSION 	= '1.0.0'
  , async	    	= require('async')
  , log4js	    	= require('log4js')  
  , url         	= require('url')

  , npsServer   	= require('../server')
  , expressApp		= npsServer.expressApp					// is the Express App
  , npsDir	    	= npsServer.npsDir
  , npsConf 		= npsServer.npsConf					// holds the configuration of the OTS Server   
  , npsLogFile 		= npsServer.npsLogFile  
  ;

// Get Logger
var logger = log4js.getLogger(MODULE_NAME);
logger.setLevel(npsConf.log.level);

/**
 * enableRoutes: 		Add HTTP Routes defined in the plugin configuration file
 * @param  {module} 	handle of the module
 * @param  {json} 		routes
 *         				routes = [
 *         					{
 *         						"enable": "on|off"
 *         						"path": 	"/path/to/the/service",
 *         						...,
 *         					}
 *         					{...},
 *         				]
 * @param  {function} 	callback(err)
 */
exports.enableRoutes = function(module, routes, callback){

	logger.debug(__filename, 'enableRoutes: Adding HTTP following routes: ' + JSON.stringify(routes));

	// Assert routes
	if(routes === null){	// routes are optional
		return callback();	
	}

	// Loop through all routes
	async.each(routes,
		function(route, nextTask){
			route.enable = route.enable || 'on';
			// Add HTTP route if only it's enabled and method (get,post,...) is defined
			if(route.enable === 'on' && route.method && 
			  (route.method.toLowerCase() !== 'dir' && route.method.toLowerCase() !== 'queue')){
				exports.enableRoute(module, route, nextTask);
			} 
			else{
				nextTask();
			}
		},
		function(err){
			return callback(err);
		}
	);
}


/**
 * enableRoute: 		Enable the given http route
 * @param  {module} 	handle of the module
 * @param  {json} 		route
 * 			        	route = {
 * 			        		"path": 	"/path/to/the/services",
 * 			        	 	"method": 	"get|post",
 * 			           		"service": 	"version",
 *                 		}
 * @param  {function}	callback(err)
 * @param  {boolean} 	bInit 	!!!
 */
exports.enableRoute = function(module, route, callback, bInit){

	logger.debug(__filename, 'enableRoute: Service: ' + JSON.stringify(route));
	
	route.method = route.method || 'get';
	if(bInit === undefined){//!!! NOT FULLY IMPLEMENTED
		bInit = true;
	}

	if(!route.path || !route.method || !route.service)
		return callback(new Error('HTTP route does not have value for either "path" or "method" or "service"'));

	try{
		// First add the route to the Express App list of routes
		//!!! Allow route.path the be an array of path!!!
		expressApp[route.method](route.path, module[route.service]);	

		// Last ensure to move the '/*' route to the end of the list of routes
//		if(!bInit){//!!! NOT FULLY IMPLEMENTED
			exports.moveDefaultAppRoute();
//		}
	}
	catch(err){
		return callback(err);
	}
	callback();
}

/**
 * disableRoutes: 		Remove HTTP Routes
 * @param  {json} 		routes
 *         				routes = [
 *         					{
 *         						"enable": "on|off"
 *         						"path": 	"/path/to/the/service",
 *         						...,
 *         					}
 *         					{...},
 *         				]
 * @param  {function} 	callback(err)
 */
exports.disableRoutes = function(routes, callback){

	logger.debug(__filename, 'disableRoutes: Removing HTTP following routes: ' + JSON.stringify(routes));

	// Loop through all routes
	async.each(routes,
		function(route, nextTask){
			// Add HTTP route if only it's enabled and method (get,post,...) is defined
			if(route.enable === 'on' && route.method){
				exports.disableRoute(route, nextTask);
			} 
			else{
				nextTask();
			}
		},
		function(err){
			return callback(err);
		}
	);
}

/**
 * disableRoute: 	Disable the given HTTP route
 * @param  {json} 	route
 * 			        route = { 
 * 			        	"path": 	"/path/to/the/services",
 * 			        }
 * @param  {function} 	callback(err)
 */
exports.disableRoute = function(route, callback){

	logger.debug(__filename, 'disableRoute: Service: ' + JSON.stringify(route));

	try{
		// Remove the route
		_removeAppRoute(route.path, route.method);
	}
	catch(err){
		return callback(err);
	}
	callback();
}

/**
 * moveDefaultAppRoute:  	Ensure that the '/*' route is at the end of the list of routes
 */
exports.moveDefaultAppRoute = function(){
	_removeAppRoute('/*');
	expressApp['get']('/*', exports.routeNotAvailable);
	expressApp['post']('/*', exports.routeNotAvailable);			
}

/**
 * routeNotAvailable: 	Process a unmapped route request
 * @param  {request}  	req Request Object
 * @param  {response} 	res Response Object
 */
exports.routeNotAvailable = function(req, res){
	var retMsg = ''
	  , url_parts = url.parse(req.url, true);

	retMsg = req.method + ' ' + url_parts['pathname'] + ' route is not available or not enabled';
	logger.info('--->routeNotAvailable: ' + retMsg);
	return res.status(404).end(retMsg);
}

/***** Private Functions ***********/
/**
 * _removeAppRoute: 	Remove the given route from the Express App
 * @param  {string} 	route The route to be removed
 */
function _removeAppRoute(route, method){

	var stack = expressApp._router.stack;	// get the Express App router stack

	// Remove passed route
	for(var i = stack.length - 1; i >= 0; i--){
		if(stack[i].route && stack[i].route.path){
			if(method){
				// if method is given, search for path having that method
				if(stack[i].route.path === route &&
					stack[i].route.methods && stack[i].route.methods[method]){
					logger.info('deleting ' + route + ' route using ' + method + ' method');
	  				stack.splice(i, 1);					
	  				break;
				}
			}
			else if(stack[i].route.path === route){
				logger.debug('deleting ' + route);
	  			stack.splice(i, 1);
	  			//break; // do not break here since some path may be in 'get' and 'post'
			}
		}
	}
}	

// V2.0.32 BEGIN
/**
 * @param  {plugin}     plugin      plugin
 * @param  {function} 	callback    callback(err, router) 
 */
exports.setupRouter = function(plugin, callback){
	exports.enableRoutes(plugin, plugin.pluginConf.routes, callback)
}
// V2.0.32 END