/**-----------------------------------------------------------------------------
 * oxsnps.js: 	OXSEED NodeJS Server msoffice2any Plugin
 *
 * Author    :  AFP2web Team
 * Copyright :  (C) 2015 by Maas Holding GmbH
 * Email     :  support@oxseed.de
 * Version   :  2.2.0
 * 
 * History
 *  V2.2.3  01.06.2023  AFP-1155: Fixed a bug that caused "inputBuffer parameter in JSON Input string passed to office converter process method, is not an array." error
 *  V100   	04.03.2014  Initial release
 *  V101   	13.03.2015  Added _logServiceRespTime()
 *  
 *----------------------------------------------------------------------------*/
'use strict';

/**
 * Variable declaration
 * @private
 */
var packageJson 	= require(__dirname + '/package.json')
  , PLUGIN_NAME 	= packageJson.name
  , PLUGIN_VERSION  = packageJson.version // Get Server Version from package.json
  , SERVICE_NAME 	= 'MSOfficeConverter'
  , fs 				= require('fs')
  , async 			= require('async')
  , log4js 			= require('log4js')

  , npsServer		= require('oxsnps-core/server')
  , npsConfDir 		= npsServer.npsConfDir					// holds the configuration Dir of the Server 
  , npsTempDir		= npsServer.npsTempDir
  , npsDir			= npsServer.npsDir

  //!!! This Plugin should NOT need both OTSScenarioConf.json & OTSScenarioList.json. Correct it!
  , confs 			= require(npsConfDir + '/OTSScenarioConf.js')	// holds the Transformation configurations
  , scenarios 		= require(npsConfDir + '/OTSScenarioList.js')
  , javaServices 	= require('./services/javaServices')

  , pluginConf		= require(__dirname + '/conf/' + PLUGIN_NAME + '.js')
  , dateFormat 		= require('oxsnps-core/helpers/date_format')
  , utils 			= require('oxsnps-core/helpers/utils')
  , jsonResWriter   = require('oxsnps-core/helpers/jsonResWriter')
  , expressAppUtils = require('oxsnps-core/helpers/expressAppUtils')
  ;

// Set plugin name, version and longDesc in pluginConf
pluginConf.module 	= PLUGIN_NAME; 
pluginConf.version 	= PLUGIN_VERSION; 
pluginConf.longDesc = packageJson.description;

// Export the Plugin's Configuration Express App
exports.pluginConf = pluginConf;

// Get the Logger
var logger = log4js.getLogger(PLUGIN_NAME);
logger.setLevel(pluginConf.log.level || 'INFO'); // set log level
javaServices.setLogLevel(pluginConf.log.level || 'INFO');

/**************** PUBLIC FUNCTIONS ****************/
/**
 * Initialize Plugin
 * @param  {function} 	callback(err, routes)
 */
exports.initialize = function(callback){

	var self = this
	  , pluginProps = null
	  , classPath = null
	  , libPath = npsDir + '/lib/'
	  , recursive = true	/* Recursively call sub directories to find jar files */
	  ;
	
	if(self._initialized){
		return callback();
	}

	pluginProps = [
		{
			'name': 	SERVICE_NAME,
			'classname': 'de.maas.office2any.MSOfficeTransformerWrapper',
			'startArgs': {'confFile': __dirname + '/' + pluginConf.props.confile}
		}
	];

	async.series([
		// Task 1: Get classpath 
		function(nextTask){
			utils.getFileList(libPath, '.jar', true/* case sensitive ext. comparision*/, recursive, function(err,filesArray){								
				if(err){
					nextTask(new Error('-->Could not get file list from ' + libPath + ', Reason: ' + err.message));
				}else{

					classPath = filesArray;

					// Add plugin conf directory to classpath. Needed for log4j.xml
					if(classPath){
						classPath.push(__dirname + '/conf/');
					}
					nextTask();
				}
			});
		},
		
		// Task 2: Start Services
		function(nextTask){
			javaServices.start(classPath, null, pluginProps, nextTask);
		},

		// Task 3: Map the routes of the plugin
		function(nextTask){
			expressAppUtils.
			enableRoutes(self, pluginConf.routes, nextTask);
		},
	],	function(err){
			if(err){
				return callback(err);
			}
			self._initialized = true;
			logger.info('\t\tPlugin ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION + ' initialized');
			callback(null, pluginConf.routes);
		}
	);
}

/**
 * Finalize Plugin
 * @param  {function} 	callback(err) 
 */
exports.finalize = function(callback){

	var self = this;

	if(!self._initialized){
		callback();
	}

	async.series([
		// Task 1: Unmap the HTTP routes of the plugin
		function(nextTask){
			// Unmap the HTTP routes of the plugin
			expressAppUtils.disableRoutes(pluginConf.routes, nextTask);
		},
		// Task 2: STOP MSOffice Converter Wrapper Service
		function(nextTask){
			javaServices.stopService(SERVICE_NAME, nextTask); 
		},
	],	function(err){
			if(err){
				return callback(err);
			}
			self._initialized = false;
			logger.info('\t\tPlugin ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION + ' finalized');
			callback();
		}
	);
}

/**
 * Return Plugin Version
 * @param  {function} 	callback(err,version)
 */
exports.getVersion = function(callback){
	return callback(null, PLUGIN_VERSION);
}

/**
 * Process a HTTP Version Request
 * @param  {Request}  	req Request Object
 * @param  {Response} 	res Response Object
 */
exports.version = function(req, res){

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);

	logger.info('--->version: Req. id=' + res._Context.reqId);

	javaServices.callMethod(SERVICE_NAME, 'getVersion', null, function(err, result){
		if(err){
			logger.error('--->version: Req. id=' + res._Context.reqId + ', Error! Reason: ' + err.message);
			return res.status(404).end('--->version: Req. id=' + res._Context.reqId + ', Error! Reason: ' + err.message);
		}
		else{		
			var data = '<h3>' +  PLUGIN_NAME + ' v' + result + '</h3>';
			logger.info('--->version: Req. id=' + res._Context.reqId + ', sent back, over');
			res.end(data); 
		}
	});
}

/**
 * Transform Office Document to PDF or TIFF
 * @param  {Request}  	req Request Object
 *                      req.body:{
 *                      	"input":{ 
 *                       		"buffer":  <Input document buffer encoded as base64>,
 *                           	"ownerpw": <Optional. Input document owner password.>,
 *                            	"userpw":  <Optional. Input document user password.>,
 *                         		"type":    <Input document type>
 *                          },
 *                          "output":{
 *                          	"type":    <Output document type>,
 *                           	"ownerpw": <Optional. Output document owner password.>,
 *                            	"userpw":  <Optional. Output document user password.>,
 *                             	"flags":   <Optional. Output document flags.>
 *                          }
 *                      }
 * @param  {Response} 	res Response Object
 *                         	{ 
 *                          	"ResponseMessage":{ 
 *                           		"responseData":              <Output document buffer encoded as base64>,
 *                            		"responseStatus":            <Response status. Values:Done|Error>,
 *                              	"responseStatusDescription": <Response status description>
 *                               }
 *                           }
 */
exports.transform = function(req, res){

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);

	// Add the Transformation Type (sync|async)
	res._Context.transType = 'sync';

	logger.info('--->transform: Req. id=' + res._Context.reqId);

	var data = ''
	  , self = this
	  ;
	
	// V101 Begin  
	res._Context.req = req.body;
	if(res._Context.req.input !== undefined
	   && res._Context.req.input.buffer !== undefined
	  ){
	  	// V102 Begin
		//res._Context.inputDoc = new Buffer(res._Context.req.input.buffer, 'binary');
		res._Context.inputDoc = new Buffer(res._Context.req.input.buffer, 'base64');
		// V102 End
	}

	//logger.debug('--->transform: Req. id=' + res._Context.reqId + ', Params: ' + JSON.stringify(req.body));

	// Validate an AFP2web Server Request
	var err = _validateReq(res);
	if(err !== null){
		logger.error('--->transform: Req.Id:' + res._Context.reqId + ', Error:' + err.message);
		jsonResWriter.sendJSONError(res, 400 /*Bad Req*/, null, err.message);
		_logServiceRespTime(res, 'transform');
		return;
	}

	// Process Request
	exports.transformService(res, function(err, result){
		if(err){
			// Send status code as OK(200) since we pass error status and its description in JSON Response
			jsonResWriter.sendJSONError(res, 200,  null, err.message);
			_logServiceRespTime(res, 'transform');
		}
		else{
			jsonResWriter.sendJSONResponse(res, 200, new Buffer(result, 'binary').toString('base64'), 'Done');
			logger.info('--->transform: Req. id=' + res._Context.reqId + ', Response sent back, over');			
			_logServiceRespTime(res, 'transform');
		}
	});
} 

/**
 * Transform PDF to Secured PDF	
 * @param  {[type]}   res      [description]
 * @param  {Function} callback [description]
 * @return {[type]}            [description]
 */
exports.transformService = function(res, callback){
	var self = this;

	// V101 Begin
	var req = res._Context.req;
	// V101 End

	// Get Configuration
	var o2aConf = scenarios[res._Context.scenario].action.config;
	logger.debug('--->transform: Req. Id=' + res._Context.reqId + ', Input: ' + req.input.type + ', Output: ' + req.output.type + ', Scenario: ' + o2aConf);

	if (confs[o2aConf] === undefined){
		logger.error('--->transform: Req. Id=' + res._Context.reqId + ', Configuration (' + o2aConf + ') does NOT exit!');
		callback(new Error('Office2any: Req. Id=' + res._Context.reqId + ', Configuration (' + o2aConf + ') does NOT exit!'));
		return;
	}
	
	var o2aProps = utils.clone(confs[o2aConf].props); //!!! Using a = b does NOT copy !!!

	var req = res._Context.req;

	// Set the _msofficeService Props
	var props = {
	//	'inputBuffer'    : ...
		'sourceOwnerPWD' : req.input.ownerpw,	
	    'sourceUserPWD'  : req.input.userpw,	
	    'inputType'      : o2aProps.InputType, 
	    'outputType'     : o2aProps.OutputType,
	    'pdfProducer'    : 'OTS v' + npsServer.npsVersion, 
	    'pdfApplication' : 'OTS MSOffice Service'
	};

	logger.info('-->transformService: Req. id=' + res._Context.reqId + ', Parameters to ' + SERVICE_NAME + '.process() '+ JSON.stringify(props));

	// V101 Begin
	//props.inputBuffer = new Buffer(res._Context.inputDoc, 'base64');
	//props.inputBuffer = res._Context.inputDoc;	// V2.2.3
	// V101 End
    props.inputBuffer = Array.prototype.slice.call(res._Context.inputDoc, 0)        // V2.2.3. Pass input as an array instead of Buffer class

	javaServices.callMethod(SERVICE_NAME, 'process', props, function(err, result){
		if (err){
			logger.error('--->transform: Req. Id=' + res._Context.reqId + ', Error when running MSOffice Service. ' + err.message);
			// in case of error, return and do not proceed further
			return callback(err);
		}
		// Store Data in debug mode ONLY
		if (logger.isDebugEnabled()){
			var outputFilename = dateFormat.asString('yyyy-MM-dd_hh-mm-ss', new Date()) + '_' + req.input.type + '.' + req.output.type;
			fs.writeFileSync(npsTempDir + '/' + outputFilename, new Buffer(result));
		}
		callback(null, result);
	});

}

/************ OPTIONAL PUBLIC FUNCTIONS ***********/
/**
 * Set Log Level
 * @param {String} logLevel Log Level. values:DEBUG|INFO|ERROR 
 */
exports.setLogLevel = function(logLevel){
	pluginConf.logLevel = logLevel || 'INFO';
	logger.setLevel(pluginConf.logLevel);
}

/**
 * Get Log Level
 * @return {String} Log Level. values:DEBUG|INFO|ERROR
 */
exports.getLogLevel = function(){
	return pluginConf.logLevel;
}

/**
 * enableRoute: 		Enable the given route
 * @param  {request} 	req
 *                      req.body = {
 *                      	"module":"sample", 
 *                      	"route": { 
 *                      		"path": "/path/to/the/service",
 *                      		"method": 	"get|post",
 *                      		"service": 	"version"
 *                      	}
 *                      }
 * @param  {function} 	callback(err)
 */
exports.enableRoute = function(req, callback){

	var self = this;

	logger.debug(__filename, '--->enableRoute');

	// Enable the route if route has been passed and route has been added previously
	if(req.body.route !== undefined && 
	   req.body.route.path !== undefined &&
	   req.body.route.method !== undefined &&
	   req.body.route.service !== undefined){
		expressAppUtils.enableRoute(self, req.body.route, function(err){
			if(err){
				return callback(err);
			}
			callback();
		});
	}
	else{
		// Display a Warning
		logger.warn('--->enableRoute: Wrong parameters! Not able to enable the route, Parms: ' + JSON.stringify(req.body));
		callback();
	}
} 

/**
 * Disable the given route
 * @param  {Request} 	request Request json has following structure.
 * 		            	req.body = {
 *                      	"module":"sample", 
 *                      	"route": { 
 *                      		"path": "/path/to/the/service"
 * 		            	  	}
 * 		                }
 * @param  {function} 	callback  callback(err)
 */
exports.disableRoute = function(req, callback){

	logger.debug(__filename, '--->disableRoute');

	// Disable the route
	if(req.body.route !== undefined && req.body.route.path !== undefined){
		expressAppUtils.disableRoute(req.body.route, callback);
	}
	else{
		// Display a Warning
		logger.warn('--->disableRoute: Wrong parameters! Not able to disable the route, Parms: ' + JSON.stringify(req.body));
		callback();
	}
} 

/**************** PRIVATE FUNCTIONS ***************/
// V101 Begin
// Validate an Office2Any Request
function  _validateReq(res){
	var req = res._Context.req;

	if (req.input === undefined || 
		req.input.type === undefined || 
		req.input.buffer === undefined || 
		req.input.buffer.length <= 0 ||
		req.output.type === undefined ){
		return new Error( 'Invalid JSON Request. Reason: input.type, output.type ' + 
					 'or input.document is missing !\n' + 'JSON Request:\n' + JSON.stringify(req));
	}

	// Build Scenario Name
	res._Context.scenario = req.input.type + '2' + req.output.type;

	// Check if Scenario exists
	if (scenarios[res._Context.scenario] === undefined){
		return new Error( 'Invalid JSON Request. Reason: ' + 
			 'Scenario (' + res._Context.scenario + ') does NOT exit! \n' + 'JSON Request:\n' + JSON.stringify(req));
	}
	
	var o2aConf = scenarios[res._Context.scenario].action.config;
	logger.debug('--->transform: Req. Id=' + res._Context.reqId + ', Input: ' + req.input.type + ', Output: ' + req.output.type + ', Scenario: ' + o2aConf);

	if (confs[o2aConf] === undefined){
		return new Error( 'Invalid JSON Request. Reason: ' + 
			 'Configuration (' + o2aConf + ') does NOT exit!\n' + 'JSON Request:\n' + JSON.stringify(req));
	}	

	// Add the Scenario Configuration to the request
	res._Context.scenarioConf = confs[o2aConf];

	return null;
}

// V101 End

/**
 * Log a Warning if Service Response time exceeded maxresptime 
 * @param  {Response}	res  	Response Object
 * @param  {String}		service service
 * @private
 */
function _logServiceRespTime(res, service){

    var respTime 	= 0
	  , os 			= require('os')
      ;

    // Check resp. time only if maxresptime is defined
   	if(pluginConf.maxresptime){
		respTime = utils.dateDiff('x', res._Context.date, new Date()); // 'x' means returns diff. in ms

   		if(respTime > pluginConf.maxresptime * 1000){
			logger.warn('--->' + service + ': Req. id=' + res._Context.reqId + ', '
						+ ' Resp. Time: ' + respTime/1000 + 's, Max. allowed: '
						+ pluginConf.maxresptime + 's. Average Load: ' + os.loadavg());
   		}
   	}
}
