//------------------------------------------------------------------------------
// javaServices Module : Module to manage Java services that in 
//						 turn uses js2Java module APIs 
//                    
// Author    : AFP2web Team
// Copyright : (C) 2014 by Maas Holding GmbH
// Email     : support@oxseed.de
// Version   : V1.0.0
// History
//   V100    01.08.2014     Initial release
//   
//
// Todos:
//------------------------------------------------------------------------------
var fs 				= require('fs')
  , async 			= require('async')
  , log4js 			= require('log4js')
  , js2Java 		= require('./js2Java/js2Java.js')
  ;

var logger = log4js.getLogger('javaServices');

/**************** PUBLIC INTERFACE ***************/
exports.setLogLevel = function(logLevel){
	logger.setLevel(logLevel || 'INFO');
}

/**
 * start: Initializes the JVM and starts passed services
 * @Params:
 * classpath  :  String Array. An array of class paths where JVM can find the required jar and class files.
 * jvmOptions :  String Array. Any options that need to be passed to JVM (Example: MaxMemory to be used by JVM)
 * services   :  JSON Array:  An array on JSON objects describing Services. Refer to the example below.
 * callback   :  A callback function that would be called after initializing the JVM and staring the services
 *               callback function will have one argument "err" of node.js type "Error".
 *               In case of success, "err" will be null.
 *               In case of  error, "err" will have information about error occurred during initialization.
 * Example: 
 *  var classpath = [ __dirname + '/lib/', , __dirname + '/lib/gson-2.2.4.jar' ]; 
 *  var jvmOptions = []; 
 *	var services = [{
 *			'name'        : 'MSOfficeConverter'
 *			, 'classname' : 'de.maas.office2any.MSOfficeTransformerWrapper' 
 *			, 'startArgs' : { 'confFile': __dirname + '/conf/o2a.json'  }
 *		}];
 *  var javaServices = new javaServices();
 *  javaServices.start(classpath. jvmOptions, services, callback(err){...});
 * 
 */
exports.start = function(classpath, jvmOptions, services, callback){
	var self = this;
	
	// Create js2Java instance
	//js2Java = new js2Java();

	// List tasks that are to be called one after another to start Java Services
	async.series([
			// Task 1: start the JVM
			function(nextTask){
				// Initialize the Java Virtual Machine with passed classpath and jvmOptions 
				// This method should be called before using any js2Java APIs
				if( !js2Java.initialized ){
					logger.debug(__filename, '-->Initializing the JVM for Java Services, classpath=' + 
						classpath + ', JVMOptions=' + JSON.stringify(jvmOptions) +' ...');	// V3.0.1
					js2Java.initialize(classpath, jvmOptions, function(err){
						if (err){
							logger.error('-->Could not initialize the Java Virtual Machine for Java Services. Reason: ' + err);
							// do not return here. Just call the nextTask so that error has been delegated to the caller
							//return;
						}
						nextTask( err );
					});	
				}
				else {
					logger.debug('-->JVM already initialized for Java Services.');
					nextTask( null );
				}
			},
			
			// Task 2: Start Services
			function(nextTask){
				logger.debug(__filename, '-->Starting Java Services ...');
				_startServices(self, services,  function(err){
					if (err){
						logger.error('-->Could not start Java Services. Reason: ' + err.message);
						// do not return here. Just call the nextTask so that error has been delegated to the caller
						//return;
					}
					nextTask(err);
				});	
			}
		],	function(err){
				callback(err);
			}
		);
}

/**
 * stop: Stop Java Services and terminate JVM
 * @Params:
 * callback   :  A callback function that would be called after stopping services and terminating the JVM
 *               callback function will have one argument "err" of node.js type "Error".
 *               In case of success, "err" will be null.
 *               In case of  error, "err" will have information about error occurred during initialization.
 * Example: 
 *  var javaServices = new javaServices();
 *  javaServices.stop(  callback(err){});
 * 
 */
exports.stop = function(callback){
	var self = this;

	async.series([
		// Task 1: stop services
		function(nextTask){
			if (self.services !== null){
				logger.debug(__filename, '-->Stopping all services ...');
				_stopServices(self.services, function(err){
					if (err){
						logger.error('-->Error: Could not stop Java Services. Reason: ' + err);
					}
					else {
						logger.debug(__filename, '-->Java Services successfully stopped.');
					}
					self.services = null; 
					nextTask( err );
				});
			}
		},
		// Task 2: stop the JVM
		function(nextTask){ 
			// Terminates the Java Virtual Machine.
			logger.debug(__filename, '-->Stopping the Java Virtual Machine...');
			js2Java.finalize(function(err){
				if (err){
					logger.error('-->Error: Could not stop Java Virtual Machine. Reason: ' + err.message);
					nextTask( err );
				}
				else{					
					logger.debug(__filename, '-->Java Virtual Machine successfully stopped.');
					nextTask( null );
				}
				
			});
		}
	],
		function(err){ // This function gets called after the tasks defined aboved have called their task's callback
			callback(err);
			return;
		}
	);		
}

/**
 * stopService: Stop Java Services and terminate JVM
 * @Params:
 * callback   :  A callback function that would be called after stopping services and terminating the JVM
 *               callback function will have one argument "err" of node.js type "Error".
 *               In case of success, "err" will be null.
 *               In case of  error, "err" will have information about error occurred during initialization.
 * Example: 
 *  var javaServices = new javaServices();
 *  javaServices.stop(  callback(err){});
 * 
 */
exports.stopService = function(serviceName, callback){
	var self = this;									  

	// Assert service name
	if (serviceName === null){
		callback(new Error("Invalid Service Name (null)"));
		return;
	}
	else if (serviceName.length <=0){
		callback(new Error("Invalid Service Name (empty)"));
		return;
	}

	var service = self.services[serviceName];
	if (service === null || service === undefined){
		callback(new Error("Invalid Service Name \'" + serviceName + "\'"));
		return;
	}

	async.series([
		// Task 1: stop services
		function(nextTask){
			logger.debug(__filename, '-->Stopping service \'' +  serviceName + '\'...');
			service.callMethod("stop",  null, function(err, result){
				if (err){
					logger.error('-->Error: Could not stop Java Services. Reason: ' + err);
				}
				self.services[serviceName] = null;
				nextTask(err);
				return;
			});
		},

		// Task 2: stop the JVM, if there are no services running
		function(nextTask){ 
			var bStopJVM = true;
			// check if any services running 
			for(var service in self.services){
			    logger.debug(__filename, service + ': ' + self.services[service]);
			    if( self.services[service] !== null){
			    	// some services are running, do not terminate the Java Virtual Machine.
					bStopJVM = false;
					break;
			    }
			}
			if( bStopJVM ){
				// no services running, terminates the Java Virtual Machine.
				logger.debug(__filename, '-->Stopping the Java Virtual Machine...');
				js2Java.finalize(function(err){
					if (err){
						logger.error('-->Error: Could not stop Java Virtual Machine. Reason: ' + err.message);
						nextTask( err );
					}
					else{					
						logger.debug(__filename, '-->Java Virtual Machine successfully stopped.');
						nextTask( null );
					}
					
				});
			}
			else{
				nextTask( null );
			}
		}
	],
		function(err){ // This function gets called after the tasks defined aboved have called their task's callback
			callback(err);
			return;
		}
	);		
}

/**
 * callMethod: Execute a service method
 * @Params:
 * serviceName : String.  Name of the Java service
 * methodname  : String. Name of Java method to call
 * callback    :  A callback function that would be called after the execution of Java method. 
 *             callback function will have two arguments.
 *                "err", of node.js type "Error"
 *                "result", of type JSON
 *             In case of success,
 *                "err" will be null.
 *                "result" a JSON Object containing result of java method execution.
 *             In case of error,
 *                "err" will have information about error occurred during method execution
 *                "result" will be null.
 * Example: 
 *  // Read input file
 *  var inputBuffer = fs.readFileSync(__dirname + '/samples/sample.doc');
 * 
 *  // Arguments of process method
 *  var args = {
 *            'inputBuffer':     inputBuffer,
 *            'sourceUserPWD':   'mht',
 *            'sourceOwnerPWD':  'mhtmht',
 *            'inputType':       'doc',
 *            'outputType':      'pdf'
 *            }  
 *  javaServices.callMethod('MSOfficeConverter', 'process', args, callback(err,result){...});
 * 
 */
exports.callMethod = function(serviceName, method, args, callback){
	var self = this;									  

	// Assert service name
	if (serviceName === null){
		callback(new Error("Invalid Service Name (null)"));
		return;
	}
	else if (serviceName.length <=0){
		callback(new Error("Invalid Service Name (empty)"));
		return;
	}

	var service = self.services[serviceName];
	if (service === null || service === undefined){
		callback(new Error("Invalid Service Name \'" + serviceName + "\'"));
		return;
	}

	// Call passed method for a given service
	service.callMethod(method, args, callback);
} 

/***** Private Functions ***********/

/**
 * Start all passes services parallely 
 */		
function _startServices(self, services, callback){

	// loop through all services
	async.each(services,
		// function to call for each service
		function(service, nextTask){
			//  Start Service
			_startService(self, service, function(err){
				nextTask(err);
			});
		},
		function(err){
			callback(err);
			return;
		}
	);
}

/**
 * Start Service 
 */		
function _startService(self, service, callback){
	
	/* Get Service information */
	var serviceName = service.name;
	var className = service.classname;
	var startArgs = service.startArgs;
	
	// Assert class name
	if (className === null){
		callback(new Error("Invalid Class Name (null)"));
		return;
	}
	else if (className.length <=0){
		callback(new Error("Invalid Class Name (empty)"));
		return;
	}

	// Assert service name
	if (serviceName === null){
		callback(new Error("Invalid Service Name (null)"));
		return;
	}
	else if (serviceName.length <=0){
		callback(new Error("Invalid Service Name (empty)"));
		return;
	}

	if( self.services === undefined ){
		self.services = {};
	}
	
	async.series([ 
			// Task 1: create an instance of  Service class
			function(nextTask){ 
				// Create an instance of javaClass module that encapsulates the actual java class for the passed classname
				logger.debug(__filename, '-->Creating an instance of \'' + className + '\'...');
				js2Java.createClass(className, function(err, result){
					if (err){
						logger.error('-->Error: Could not create an instance of \''+ className  + '\'. Reason: ' + err);
						self.services[serviceName] = null; 
					} else{
						// result is an instance of javaClass module that encapsulates the java class
						self.services[serviceName] = result; 
					}
					nextTask( err );
				});
			},
			
			// Task 2: Start  Service
			function(nextTask){
				// Call start method to initialize  Service.
				logger.debug(__filename, '-->Starting '+ serviceName +  ' Service...');
				self.services[serviceName].callMethod('start', startArgs, function(err, result){
					if (err){
						logger.error('-->Error: Could not start \''+ serviceName + '\' Service. Reason: ' + err);
						self.services[serviceName] = null; 
						//return;
					}
					nextTask(err);
				});
			}
		], function(err){ // This function gets called after the tasks defined aboved have called their task's callback
			callback(err);
			return;
		}
	);		
}

/**
 * Stop all services parallely 
 */		
function _stopServices(services, callback){
	var name = "";

	if( services === undefined || services === null ){
		callback( null );
		return;
	}

	async.each(services,
		//  Stop Service
		function(service, nextTask){
			logger.debug(__filename, '-->Stopping service \'' +  name + '\'...'); //???
			service.callMethod("stop",  null, function(err, result){
				nextTask(err);
			});
		},
		function(err){
			callback(err);
			return;
		}
	);
}
