/**-----------------------------------------------------------------------------
 * oxsnps.js: 	OXSEED NodeJS Server pdf2pdf Plugin
 *
 * Author    :  AFP2web Team
 * Copyright :  (C) 2015 by Maas Holding GmbH
 * Email     :  support@oxseed.de
 * Version   :  2.2.0
 * 
 * History
 *  V100   04.03.2015  Initial release
 *  V102   13.03.2015  Added _logServiceRespTime()
 *  V107   16.09.2022  AFP-1033: OTS V2 Update Delivery
 *  V224   14.10.2022  OTS-3364: use QPDF
 *  
 *----------------------------------------------------------------------------*/
'use strict';

/**
 * Variable declaration
 * @private
 */
var packageJson 	= require(__dirname + '/package.json')
  , PLUGIN_NAME 	= packageJson.name
  , PLUGIN_VERSION  = packageJson.version // Get Server Version from package.json
  , fs 				= require('fs')
  , path 			= require('path')
  , async 			= require('async')
  , log4js 			= require('log4js')
  
  , npsServer		= require('oxsnps-core/server')
  , npsDir			= npsServer.npsDir
  , npsTempDir		= npsServer.npsTempDir

  , pluginConf		= require(__dirname + '/conf/' + PLUGIN_NAME + '.js')
  , pluginManager 	= require('oxsnps-core/pluginManager') 
  , dateFormat 		= require('oxsnps-core/helpers/date_format')
  , utils 			= require('oxsnps-core/helpers/utils')
  , expressAppUtils = require('oxsnps-core/helpers/expressAppUtils')
  , jsonResWriter   = require('oxsnps-core/helpers/jsonResWriter')

// Set plugin name, version and longDesc in pluginConf
pluginConf.module 	= PLUGIN_NAME; 
pluginConf.version 	= PLUGIN_VERSION; 
pluginConf.longDesc = packageJson.description;

// Export the Plugin's Configuration Express App
exports.pluginConf = pluginConf;

// Get the Logger
var logger = log4js.getLogger(PLUGIN_NAME);
logger.setLevel(pluginConf.log.level || 'INFO'); // set log level

/**************** PUBLIC FUNCTIONS ****************/
/**
 * Initialize Plugin
 * @param  {function} 	callback(err, routes)
 */
exports.initialize = function(callback){

	var self 		= this

	if(self._initialized) return callback()

	expressAppUtils.enableRoutes(self, pluginConf.routes, function(err){
		if(err) return callback(err)
		self._initialized = true
		logger.info('\t\tPlugin ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION + ' initialized')
		callback(null, pluginConf.routes)
	})
}

/**
 * Activate Plugin. This function is called from pluginManager once all the enabled plugins have been initialize
 * @param  {function} 	callback  callback(err)
 */
exports.postInitialize = function(callback){
	callback();
}

/**
 * Finalize Plugin
 * @param  {function} 	callback(err) 
 */
exports.finalize = function(callback){

	var self = this;

	if(!self._initialized) return callback();


	async.series([
		// Task 1: Unmap the HTTP routes of the plugin
		function(nextTask){
			// Unmap the HTTP routes of the plugin
			expressAppUtils.disableRoutes(pluginConf.routes, nextTask)
		}
	],	function(err){
			if(err) return callback(err)
			self._initialized = false
			logger.info('\t\tPlugin ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION + ' finalized')
			callback()
		}
	);
}

/**
 * Return Plugin Version
 * @param  {function} 	callback(err,version)
 */
exports.getVersion = function(callback){
	return callback(null, PLUGIN_VERSION);
}

/**
 * Process a HTTP Version Request
 * @param  {Request}  	req Request Object
 * @param  {Response} 	res Response Object
 */
exports.version = function(req, res){
	res._Context = {} // add Context to the response instance
	res._Context.reqId = utils.buildReqId(new Date()) // add a Request Id to the Context
	logger.info('--->version: Req. id=' + res._Context.reqId);
	let data = '<h3>' +  PLUGIN_NAME + ' v' + PLUGIN_VERSION + '</h3>'
	res.end(data)
}

/**
 * Process a HTTP Get Configuration Request
 * @param {Request} 	req Request Object
 * @param {Response}    res Response Object
*/
exports.conf = function(req, res){

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);
	logger.info('--->conf: Req. id=' + res._Context.reqId);

	res.json(pluginConf);

	logger.info('--->conf: Req. id=' + res._Context.reqId + ', over');
	return;
}

/**
 * Transform PDF to Secured PDF	
 * @param  {Request}  	req Request Object
 *                      req.body:{
 *                      	"input":{ 
 *                       		"buffer": <Input document buffer encoded as base64>,
 *                           	"ownerpw": <Optional. Input document owner password.>,
 *                            	"userpw":  <Optional. Input document user password.>,
 *                         		"type":   "pdf"
 *                          },
 *                          "output":{
 *                          	"type":    "pdf",
 *                           	"ownerpw": <Optional. Output document owner password.>,
 *                            	"userpw":  <Optional. Output document user password.>,
 *                             	"flags":   <Optional. Output document flags.>
 *                          }
 *                      }
 * @param  {Response} 	res Response Object
 *                         	{ 
 *                          	"ResponseMessage":{ 
 *                           		"responseData":              <Output document buffer encoded as base64>,
 *                            		"responseStatus":            <Response status. Values:Done|Error>,
 *                              	"responseStatusDescription": <Response status description>
 *                               }
 *                           }
 */
exports.transform = function(req, res){

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);

	// Add the Transformation Type (sync|async)
	res._Context.transType = 'sync';

	res._Context.req = req.body;
	if(res._Context.req.input !== undefined
	   && res._Context.req.input.buffer !== undefined
	  ){
	  	// V102 Begin
		//res._Context.inputDoc = new Buffer(res._Context.req.input.buffer, 'binary');
		res._Context.inputDoc = new Buffer(res._Context.req.input.buffer, 'base64');
		// V102 End
	}

	logger.info('--->transform: Req. id=' + res._Context.reqId);

	// Validate Request
	var err = _validateReq(res);
	if(err !== null){
		logger.error('--->transform: Req.Id:' + res._Context.reqId + ', Error:' + err.message);
		jsonResWriter.sendJSONError(res, 400 /*Bad Req*/, nul, err.message);
	   	_logServiceRespTime(res, 'transform');
	   	return;
	}

	// Process Request
	exports.transformService(res, function(err, result){
		if(err){
			// Send status code as OK(200) since we pass error status and its description in JSON Response
			jsonResWriter.sendJSONError(res, 200, null, err.message);
		   	_logServiceRespTime(res, 'transform');
		}
		else{
			jsonResWriter.sendJSONResponse(res, 200, new Buffer(result, 'binary').toString('base64'), 'Done');
			logger.info('--->transform: Req. id=' + res._Context.reqId + ', Response sent back, over');			
		   	_logServiceRespTime(res, 'transform');
		}
	});
} 

/**
 * Merge PDFs to one PDF,optionally encrypting merged PDF
 * @param  {Request}  	req Request Object
 *                      req.body:{
 *                      	"inputList":[
 *                      		{
 *                       			"source": <Input document filename.>,
 *                           		"ownerpw": <Optional. Input document owner password.>,
 *                                  "userpw":  <Optional. Input document user password.>,
 *                           	},
 *                           	{
 *                           		...
 *                           	}
 *                          ],
 *                          "output":{
 *                          	"type":    "mergedpdf",
 *                           	"ownerpw": <Optional. Output document owner password.>,
 *                            	"userpw":  <Optional. Output document user password.>,
 *                             	"flags":   <Optional. Output document flags.>
 *                          }
 *                      }
 * @param  {Response} 	res Response Object
 *                         	{ 
 *                          	"ResponseMessage":{ 
 *                           		"responseData":              <Merged output document buffer encoded as base64>,
 *                            		"responseStatus":            <Response status. Values:Done|Error>,
 *                              	"responseStatusDescription": <Response status description>
 *                               }
 *                           }
 */
exports.merge = function(req, res){

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);

	// Add the Transformation Type (sync|async)
	res._Context.transType = 'sync';

	res._Context.req = req.body;

	logger.info('--->merge. Req. id=' + res._Context.reqId);

	exports.mergeService(res, function(err, result){
		if(err){
			// Send status code as OK(200) if err.status not passed in err object,
			// since we pass error status and its description in JSON Response
			err.status = err.status || 200;
			jsonResWriter.sendJSONError(res, err.status, null, err.message);
		   	_logServiceRespTime(res, 'merge');
		}
		else{
			jsonResWriter.sendJSONResponse(res, 200, new Buffer(result, 'binary').toString('base64'), 'Done');
			logger.info('--->merge: Req. id=' + res._Context.reqId + ', Response sent back, over');			
		   	_logServiceRespTime(res, 'merge');
		}
	});
} 

/**
 * Merge PDFs to one PDF,optionally encrypting merged PDF
 * @param  {Response}  	res Response Object
 * @param  {Function} callback callback(err)
 * 
 * res._Context.req.inputList: [
 * 	{
 * 		source: "/home/daniel/AFP2web/V4.3.0.96/samples/medform.pdf",
 * 		ownerpw: "mht",
 * 		userpw: "mht"
 * 	},
 * 	{
 * 		source: "/home/daniel/AFP2web/V4.3.0.96/samples/insure.pdf",
 * 		ownerpw: "mht",
 * 		userpw: "mht"
 * 	}
 * ],
 * output: {
 * 	type: "mergedpdf",
 * 	ownerpw: "",
 * 	userpw: "",
 * 	flags: "1|2|3|4|5|6|7|8"
 * }
 * 
 * QPDF Merge Command:
 * 	qpdf --empty --pages in/file1.pdf in/file2.pdf --password=<password> in/file3.pdf --password=<password> -- --encrypt <OutputUserPassword> <OutputOwnerPassword>  256 [flags] -- out/merge.pdf
 * 		where:
 * 			<password> is owner password, if not given user password
 * 			if --encrypt is use than both <OutputUserPassword> and <OutputOwnerPassword> must be specified
 * 			256 is encryption key length
 * 			flags are optional
 */
 exports.mergeService = function(res, callback){
	
	let req 	 		= res._Context.req
	  , resTmp 	 		= undefined
	  , qpdfargs 		= [] 		// args to be passed to qpdf
	  , outputFilename 	= undefined // generated output filename
	  , flags 			= undefined
	  , data			= undefined	// output buffer
	  , tempDir 		= res._Context.tempDir || npsTempDir

	// Assert Input filename list
	if (req.inputList === undefined){
		logger.error( '--->mergeService: Invalid Request. Req.Id:' + res._Context.reqId + ', Reason: Input List is missing or empty!')
		var err = new Error('Invalid Request. Req.Id:' + res._Context.reqId + ', Reason: Input List is missing or empty!')
		err.status = 400; /* Bad Request*/
		return callback(err)
	}

	// Set input type
	res._Context.inputType 	= 'pdf'					

	/* Loop thru inputList
		req.inputList: [
			{
				source: "/home/daniel/AFP2web/V4.3.0.96/samples/medform.pdf",
				ownerpw: "mht",
				userpw: "mht"
			},
			{
				source: "/home/daniel/AFP2web/V4.3.0.96/samples/insure.pdf",
				ownerpw: "mht",
				userpw: "mht"
			},
		],
		output: {
			type: "mergedpdf",
			ownerpw: "",
			userpw: "",
			flags: ""
		}
	*/
	/**
	 * Loop thru inputList to build QPDF args as shown below
	 * 	 qpdf --empty --pages
	 * 		  in/file1.pdf in/file2.pdf --password=<password> in/file3.pdf --password=<password> 
	 * 		  -- 
	 * 		  --encrypt <OutputUserPassword> <OutputOwnerPassword> key-length(i.e. 256) [flags] --
	 * 		  -- out/merge.pdf
	 */
    qpdfargs.push("--empty") 
    qpdfargs.push("--pages") 
    req.inputList.forEach(function(item){
        qpdfargs.push('"' + item.source + '"')
		// Add owner pwd; if not passed, add user pwd if defined
		if(item.ownerpw) qpdfargs.push("--password="+ item.ownerpw)
		else if(item.userpw) qpdfargs.push("--password="+ item.userpw)
        //!!!qpdfargs.push("1-z") // Page range. <startPage>-<endPage> //!!! I think it's not needed
    })

	// Add pwds of output filename
	/**
	 * 	-- 
	 * 	--encrypt <OutputUserPassword> <OutputOwnerPassword> 256 
	 * 	-- 
	 * 	out/merge.pdf
	 */
    qpdfargs.push("--") 	// add separator
	if(req.output.ownerpw || req.output.userpw){
		qpdfargs.push("--encrypt")
		if(req.output.userpw)  qpdfargs.push(req.output.userpw)
		if(req.output.ownerpw) qpdfargs.push(req.output.ownerpw)
		// We must set 2 pwds before the key length (i.e. 256)
		if(req.output.ownerpw && !req.output.userpw)  qpdfargs.push(req.output.ownerpw)
		if(req.output.userpw  && !req.output.ownerpw) qpdfargs.push(req.output.userpw)
		qpdfargs.push("256") // 256 is encryption key length
		// Add Flags
		flags = req.output.flags ? req.output.flags.split("|") : []
		for(let i=0; i<flags.length; i++){
			switch(flags[i]){
				case '0':{ // 0	Disable All flags: --print=none --modify=none --extract=n --accessibility=n --assemble=n
					qpdfargs.push("--print=none")
					qpdfargs.push("--modify=none")
					qpdfargs.push("--extract=n")
					qpdfargs.push("--accessibility=n")
					qpdfargs.push("--assemble=n")
					break;	
				}
				case '1':{ // 1	Allow Low resolution printing: --print=low
					qpdfargs.push("--print=low")
					break
				}
				case '2': { // 2 Allow Changing the document: --modify=all
					qpdfargs.push("--modify=all")
					break
				}
				case '3':{ // 3 Allow Content copying or extraction: --extract=y
					qpdfargs.push("--extract=y")
					break
				}
				case '4': { // 4 Allow Authoring comments and form fields (annotations): --modify=annotate
					qpdfargs.push("--modify=annotate")
					break
				}
				case '5': { // 5 Allow Form field fill-in or signing: --modify=form
					qpdfargs.push("--modify=form")
					break
				}
				case '6': { // 6 Allow Content accessibility: --accessibility=y
					qpdfargs.push("--accessibility=y")
					break
				}
				case '7': { // 7 Allow Document assembly: --assemble=y
					qpdfargs.push("--assemble=y")
					break
				}
				case '8': { // 8 Allow High Quality Printing: --print=full
					qpdfargs.push("--print=full")
					break
				}
				default: {
					break
				}
			}
		}
		qpdfargs.push("--") 	// add separator for output filename
	}

	// Add output filename
	outputFilename = tempDir + '/' + res._Context.reqId + '_' + res._Context.inputType + '.' + req.output.type
    qpdfargs.push(outputFilename)

	// Call QPDF Service (i.e. oxsnps-qpdf.processService())
    resTmp = { 
        '_Context': {
            'reqId': 		  res._Context.reqId,
            'qpdf': 		  {"args": qpdfargs},
			'addDefaultArgs': false // default is true. true => Add oxsnps-qpdf plugin defined default args, in addition to the caller qpdf args; false => Do not add default args
        }
    }   
    if(logger.isDebugEnabled()) logger.debug('--->mergeService: Req.Id=' + res._Context.reqId + ', Calling oxnps-qpdf.processService() to merge PDFs using qpdf, Parms:' + JSON.stringify(resTmp._Context.qpdf))
	async.series([
		// Task 1: Create tempDir
		function(nextTask){
			utils.mkDir(tempDir,parseInt('0777',8),nextTask)
		},
		// Task 2: Call QPDF to merge the files
		function(nextTask){
			pluginManager.callPluginService({'name':'oxsnps-qpdf','service':'processService'}, resTmp, function(err, result){
				/*  result = {"code":0,"stdout":<stdout>,"stderr":<stderr>} */
				if(err) return nextTask(new Error('Unable to merge pdfs using qpdf. ' + err.message + ', Result: ' + JSON.stringify(result))) // err.message has qpdf parms
				if(result. code !== 0) return nextTask(new Error('Unable to merge pdfs using qpdf. Result: ' + JSON.stringify(result) + ', Parms: ' + JSON.stringify(resTmp._Context.qpdf)))
				if(logger.isDebugEnabled()) logger.debug('--->mergeService: Req. id=' + res._Context.reqId + ', qpdf result: ' + JSON.stringify(result))
				nextTask()
			})
		},
		// Task 3: Read output file
		function(nextTask){
			fs.readFile(outputFilename,function(err,result){
                if(err) return nextTask(new Error('Unable to read merged pdf: ' + outputFilename + ', Reason: '+ err.message))
                data = result
				nextTask()
			})
        }
        /*
		// Task 4: Delete the temp dir and its content
		function(nextTask){
			utils.deleteDirectory(tempDir,function(err){
				// Ignore the error if any
				nextTask()
			})
        }
        */
	],	function(err){
            // Delete temp dir. for success/error cases
            utils.deleteDirectory(tempDir,function(err){
                // Ignore the error if any
                if(logger.isDebugEnabled()) logger.debug('--->mergeService: Req. id=' + res._Context.reqId + ', Unable to delete temp dir: ' + tempDir + ', Reason: ' + err.message)
                return
            })        
			return callback(err,data)
		}
	)
}

/**
 * Transform PDF to Secured PDF	
 * @param  {[type]}   res      [description]
 * @param  {Function} callback [description]
 */
exports.transformService = function(res, callback){
	/*
		res._Context: {
			"reqId": "221014134835181-91022",
			"tempDir": "/tmp/nps/221014134835181-91022",
			"inputDoc": [37,80,68,70,37,69],
			"req": {
				"input": {
					"ownerpw": "mht",
					"userpw": "mht",
					"document": "JVBERi0xLjDExNgolJUVPRg==",
					"type": "pdf"
				},
				"output": {
					"type": "pdf",
					"ownerpw": "mht",
					"userpw": "mht",
					"flags": "1|2|3|4|5|6|7|8"
				}
			}
		}
	*/
	let req 			= res._Context.req
	  , resTmp 	 		= undefined
	  , qpdfargs 		= [] 		// args to be passed to qpdf
	  , inputFilename 	= undefined // generated input filename
	  , outputFilename 	= undefined // generated output filename
	  , flags 			= undefined
	  , data			= undefined	// output buffer
	  , tempDir 		= res._Context.tempDir || npsTempDir
	
	// Assert res._Context.inputDoc
	if (!res._Context.inputDoc)
		return callback(new Error('-->transformService: Req. id=' + res._Context.reqId + ', Missing input doc!.'))

	/**
	 * Build QPDF args as shown below
	 * 	 qpdf in/file.pdf
	 * 		  --password=mht 
	 * 		  --encrypt mht mht 256 
	 * 		  --print=full --modify=all --extract=y --accessibility=y --assemble=y 
	 * 		  -- out/pdfsecure.pdf
	 */
	inputFilename = tempDir + '/' + res._Context.reqId + '_' + req.input.type + '.' + req.output.type
	qpdfargs.push('"' + inputFilename + '"')

	// Add owner pwd; if not passed, add user pwd if defined
	if(req.input.ownerpw) qpdfargs.push("--password="+ req.input.ownerpw)
	else if(req.input.userpw) qpdfargs.push("--password="+ req.input.userpw)
 
	 // Add pwds of output filename
	 /**
	  * 	-- 
	  * 	--encrypt <OutputUserPassword> <OutputOwnerPassword> 256 
	  * 	-- 
	  * 	out/merge.pdf
	  */
	qpdfargs.push("--") 	// add separator
	if(req.output.ownerpw || req.output.userpw){
		qpdfargs.push("--encrypt")
		if(req.output.userpw)  qpdfargs.push(req.output.userpw)
		if(req.output.ownerpw) qpdfargs.push(req.output.ownerpw)
		// We must set 2 pwds before the key length (i.e. 256)
		if(req.output.ownerpw && !req.output.userpw)  qpdfargs.push(req.output.ownerpw)
		if(req.output.userpw  && !req.output.ownerpw) qpdfargs.push(req.output.userpw)
		qpdfargs.push("256") // 256 is encryption key length
		// Add Flags
		flags = req.output.flags ? req.output.flags.split("|") : []
		for(let i=0; i<flags.length; i++){
			switch(flags[i]){
				case '0':{ // 0	Disable All flags: --print=none --modify=none --extract=n --accessibility=n --assemble=n
					qpdfargs.push("--print=none")
					qpdfargs.push("--modify=none")
					qpdfargs.push("--extract=n")
					qpdfargs.push("--accessibility=n")
					qpdfargs.push("--assemble=n")
					break;	
				}
				case '1':{ // 1	Allow Low resolution printing: --print=low
					qpdfargs.push("--print=low")
					break
				}
				case '2': { // 2 Allow Changing the document: --modify=all
					qpdfargs.push("--modify=all")
					break
				}
				case '3':{ // 3 Allow Content copying or extraction: --extract=y
					qpdfargs.push("--extract=y")
					break
				}
				case '4': { // 4 Allow Authoring comments and form fields (annotations): --modify=annotate
					qpdfargs.push("--modify=annotate")
					break
				}
				case '5': { // 5 Allow Form field fill-in or signing: --modify=form
					qpdfargs.push("--modify=form")
					break
				}
				case '6': { // 6 Allow Content accessibility: --accessibility=y
					qpdfargs.push("--accessibility=y")
					break
				}
				case '7': { // 7 Allow Document assembly: --assemble=y
					qpdfargs.push("--assemble=y")
					break
				}
				case '8': { // 8 Allow High Quality Printing: --print=full
					qpdfargs.push("--print=full")
					break
				}
				default: {
					break
				}
			}
		}
		qpdfargs.push("--") 	// add separator for output filename
	}
 
	// Add output filename
	outputFilename = tempDir + '/' + res._Context.reqId + '_' + res._Context.inputType + '_secured.' + req.output.type
	qpdfargs.push(outputFilename)
 
	// Call QPDF Service (i.e. oxsnps-qpdf.processService())
	resTmp = { 
		'_Context': {
			'reqId': 		  res._Context.reqId,
			'qpdf': 		  {"args": qpdfargs},
			'addDefaultArgs': false // default is true. true => Add oxsnps-qpdf plugin defined default args, in addition to the caller qpdf args; false => Do not add default args
		}
	}   
 
	async.series([
		// Task 1: Create tempDir
		function(nextTask){
			utils.mkDir(tempDir,parseInt('0777',8),nextTask)
		},
		// Task 2: Write buffer to a temp file
		function(nextTask){
			fs.writeFile(inputFilename,res._Context.inputDoc,nextTask)
		},
		// Task 3: Call QPDF to secure the file
		function(nextTask){
			if(logger.isDebugEnabled()) logger.debug('--->transformService: Req.Id=' + res._Context.reqId + ', Calling oxnps-qpdf.processService() to secure the PDF using qpdf, Parms:' + JSON.stringify(resTmp._Context.qpdf))
			pluginManager.callPluginService({'name':'oxsnps-qpdf','service':'processService'}, resTmp, function(err, result){
				/*  result = {"code":0,"stdout":<stdout>,"stderr":<stderr>} */
				if(err) return nextTask(new Error('Unable to secure the pdf file using qpdf. ' + err.message + ', Result: ' + JSON.stringify(result))) // err.message has qpdf parms
				if(result.code !== 0) return nextTask(new Error('Unable to secure the pdf file using qpdf. Result: ' + JSON.stringify(result) + ', Parms: ' + JSON.stringify(resTmp._Context.qpdf)))
				if(logger.isDebugEnabled()) logger.debug('--->transformService: Req. id=' + res._Context.reqId + ', qpdf result: ' + JSON.stringify(result))
				nextTask()
			})
		},
		// Task 4: Read output file
		function(nextTask){
			fs.readFile(outputFilename,function(err,result){
                if(err) return nextTask(new Error('Unable to read output pdf: ' + outputFilename + ', Reason: '+ err.message))
				data = result
				nextTask()
			})
        }/*,
		// Task 5: Delete the temp dir and its content
		function(nextTask){
			utils.deleteDirectory(tempDir,function(err){
				// Ignore the error if any
				nextTask()
			})
        }
        */
	],	function(err){
            // Delete temp dir. for success/error cases
            utils.deleteDirectory(tempDir,function(err){
                // Ignore the error if any
                if(logger.isDebugEnabled()) logger.debug('--->transformService: Req. id=' + res._Context.reqId + ', Unable to delete temp dir: ' + tempDir + ', Reason: ' + err.message)
                return
            })        

			return callback(err,data)
		}
	)
}	

/************ OPTIONAL PUBLIC FUNCTIONS ***********/
/**
 * Set Log Level
 * @param {String} logLevel Log Level. values:DEBUG|INFO|WARN|ERROR 
 */
exports.setLogLevel = function(logLevel){
	pluginConf.log.level = logLevel || 'INFO';
	logger.setLevel(pluginConf.log.level);
}

/**
 * Get Log Level
 * @return {String} Log Level. values:DEBUG|INFO|WARN|ERROR
 */
exports.getLogLevel = function(){
	return pluginConf.log.level;
}

/**
 * enableRoute: 		Enable the given route
 * @param  {request} 	req
 *                      req.body = {
 *                      	"module":"sample", 
 *                      	"route": { 
 *                      		"path": "/path/to/the/service",
 *                      		"method": 	"get|post",
 *                      		"service": 	"version"
 *                      	}
 *                      }
 * @param  {function} 	callback(err)
 */
exports.enableRoute = function(req, callback){

	var self = this;

	logger.debug(__filename, '--->enableRoute');

	// Enable the route if route has been passed and route has been added previously
	if(req.body.route !== undefined && 
	   req.body.route.path !== undefined &&
	   req.body.route.method !== undefined &&
	   req.body.route.service !== undefined){
		expressAppUtils.enableRoute(self, req.body.route, function(err){
			if(err){
				return callback(err);
			}
			callback();
		});
	}
	else{
		// Display a Warning
		logger.warn('--->enableRoute: Wrong parameters! Not able to enable the route, Parms: ' + JSON.stringify(req.body));
		callback();
	}
} 

/**
 * Disable the given route
 * @param  {Request} 	request Request json has following structure.
 * 		            	req.body = {
 *                      	"module":"sample", 
 *                      	"route": { 
 *                      		"path": "/path/to/the/service"
 * 		            	  	}
 * 		                }
 * @param  {function} 	callback  callback(err)
 */
exports.disableRoute = function(req, callback){

	logger.debug(__filename, '--->disableRoute');

	// Disable the route
	if(req.body.route !== undefined && req.body.route.path !== undefined){
		expressAppUtils.disableRoute(req.body.route, callback);
	}
	else{
		// Display a Warning
		logger.warn('--->disableRoute: Wrong parameters! Not able to disable the route, Parms: ' + JSON.stringify(req.body));
		callback();
	}
} 

/**************** PRIVATE FUNCTIONS ***************/
// Validate Request
function  _validateReq(res){
	var req = res._Context.req;

	if (req.input === undefined || 
		req.input.type === undefined || 
		req.input.buffer === undefined || 
		req.input.buffer.length <= 0 ||
		req.output.type === undefined ){
		return new Error('Invalid JSON Request. Reason: input.type, output.type ' + 
					 'or input.document is missing !\n' + 'JSON Request:\n' + JSON.stringify(req));
	}
	return null;
}

/**
 * Log a Warning if Service Response time exceeded maxresptime 
 * @param  {Response}	res  	Response Object
 * @param  {String}		service service
 * @private
 */
function _logServiceRespTime(res, service){

    var respTime 	= 0
	  , os 			= require('os')
      ;

    // Check resp. time only if maxresptime is defined
   	if(pluginConf.maxresptime){
		respTime = utils.dateDiff('x', res._Context.date, new Date()); // 'x' means returns diff. in ms

   		if(respTime > pluginConf.maxresptime * 1000){
			logger.warn('--->' + service + ': Req. id=' + res._Context.reqId + ', '
						+ ' Resp. Time: ' + respTime/1000 + 's, Max. allowed: '
						+ pluginConf.maxresptime + 's. Average Load: ' + os.loadavg());
   		}
   	}
}
