/**-----------------------------------------------------------------------------
 * index.js: 	OXSNPS Plugin to manage SOAP requests
 *
 * Author    :  AFP2web Team
 * Copyright :  (C) 2015-2017 by Maas Holding GmbH
 * Email     :  support@oxseed.de
 * Version   :  V2.0.6
 * 
 * History
 *  V206   03.03.2017  Bug:
 *                     AFP to PDF transformation produce unknown password protected PDFs
 *
 *                     Reason:
 *                     PDF output password specifying XML elements <ownerpw> and <userpw> have an attribute called "xsi:nil" as given below even for empty
 *                     or null value
 *                     ---------------------------------------------
 *                     <output><type>pdf</type><ownerpw xsi:nil="true"/><userpw xsi:nil="true"/><flags xsi:nil="true"/></output>
 *                     ---------------------------------------------
 *
 *                     XML to JSON transforms above elements as given below
 *                     ---------------------------------------------
 *                     "output":{"type":"pdf","ownerpw":{"$":{"xsi:nil":"false"}},"userpw":{"$":{"xsi:nil":"false"}},"flags":{"$":{"xsi:nil":"true"}}}
 *                     ---------------------------------------------
 *                     i.e, the value of 'ownerpw', 'userpw' and 'flags' elements are not of string type rather object.
 *
 *                     oxsnps-afp2any plugin process the values of these elements as 'string' type. When the value is not string type, then the value for
 *                     PDFSecurity option (evaluated in oxsnps-afp2any:_getPDFSecurityOptions function) goes wrong and thus produce protected PDF with unknown
 *                     password.
 *
 *                     Fix:
 *                     After XML to JSON transformation, validate JSON for 'ownerpw', 'userpw' and 'flags' elements of input, output and make there values
 *                     as string
 *
 *                     Issue, Customer:
 *                     AFP-509, Sanlam
 *
 *                     Fixed by:
 *                     Panneer
 *  ...
 *  V101   13.03.2015  Added _logServiceRespTime()
 *  V100   09.03.2015  Initial release
 *  
 *----------------------------------------------------------------------------*/
'use strict';

/**
 * Variable declaration
 * @private
 */
var packageJson 	= require(__dirname + '/package.json')
  , PLUGIN_NAME 	= packageJson.name
  , PLUGIN_VERSION  = packageJson.version // Get Server Version from package.json
  , fs 				= require('fs')
  , path 			= require('path')
  , async 			= require('async')  
  , log4js 			= require('log4js')

  , npsServer		= require('oxsnps-core/server')
  , npsConfDir 		= npsServer.npsConfDir							// holds the configuration Dir of the nps Server 
  , npsTempDir		= npsServer.npsTempDir  

  //!!! This Plugin should NOT need both OTSScenarioList.json & OTSScenarioConf.json. Correct it!
  , scenarioConf	= require(npsConfDir + '/OTSScenarioConf.js')	// holds the Scenarios configuration
  , scenarios 		= require(npsConfDir + '/OTSScenarioList.js')	// holds the list of Transformation Scenarios
  , a2wsAny 		= scenarios.anytypes.a2ws 						// holds the A2WS Any Types
  , msofficeAny 	= scenarios.anytypes.msoffice 					// holds the MSOffice Any Types
  , soapServices 	= require('./services/soapServices.js')

  , pluginConf		= require(__dirname + '/conf/' + PLUGIN_NAME + '.js')
  , pluginManager 	= require('oxsnps-core/pluginManager') 
  , utils 			= require('oxsnps-core/helpers/utils')
  , expressAppUtils = require('oxsnps-core/helpers/expressAppUtils')
  ;

// Set plugin name, version and longDesc in pluginConf
pluginConf.module 	= PLUGIN_NAME; 
pluginConf.version 	= PLUGIN_VERSION; 
pluginConf.longDesc = packageJson.description;

// Export the Plugin's Configuration Express App
exports.pluginConf = pluginConf;

// Get the Logger
var logger = log4js.getLogger(PLUGIN_NAME);
logger.setLevel(pluginConf.log.level || 'INFO'); // set log level
soapServices.setLogLevel(pluginConf.log.level || 'INFO');

/**************** PUBLIC FUNCTIONS ****************/
/**
 * initialize: 			Initialize Plugin
 * @param  {function} 	callback(err, routes)
 */
exports.initialize = function(callback){

	logger.debug(__filename, '--->initializing ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION );
	var self = this;
	
	if(self._initialized){
		return callback();
	}

	// Map the HTTP routes of the plugin
	expressAppUtils.enableRoutes(self, pluginConf.routes, function(err){
		if(err){
			return callback(err);
		}
		self._initialized = true;
		logger.info('\t\t\tPlugin ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION + ' initialized');
		callback(null, pluginConf.routes);
	});
}

/**
 * Activate Plugin. This function is called from pluginManager once all the enabled plugins have been initialize
 * @param  {function} 	callback  callback(err)
 */
exports.postInitialize = function(callback){
	callback();
}

/**
 * Finalize Plugin
 * @param  {function} 	callback callback(err) 
 */
exports.finalize = function(callback){

	logger.debug(__filename, '--->finalizing ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION );
	var self = this;

	if(!self._initialized){
		callback();
	}

	// Unmap the HTTP routes of the plugin
	expressAppUtils.disableRoutes(pluginConf.routes, function(err){
		if(err){
			return callback(err);
		}
		self._initialized = false;
		logger.info('\t\tPlugin ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION + ' finalized');
		callback();
	});
}

/**
 * getVersion: 			Return Plugin Version
 * @param  {function} 	callback(err,version)
 */
exports.getVersion = function(callback){
	return callback(null, PLUGIN_VERSION);
}

/**
 * version: 			Process a HTTP Version Request
 * @param  {Request}  	req Request Object
 * @param  {Response} 	res Response Object
 */
exports.version = function(req, res){

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);
	logger.info('--->version: Req. id=' + res._Context.reqId);

	var data = '<h3>' +  PLUGIN_NAME + ' v' + PLUGIN_VERSION + '</h3>';
	res.end(data);
	
	logger.info('--->version: Req. id=' + res._Context.reqId + ', sent back ' +
				PLUGIN_NAME + ' v' + PLUGIN_VERSION + ', over');
	return;
}

/**
 * Process a HTTP Get Configuration Request
 * @param {Request} 	req Request Object
 * @param {Response}    res Response Object
*/
exports.conf = function(req, res){

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);
	logger.info('--->conf: Req. id=' + res._Context.reqId);

	res.json(pluginConf);

	logger.info('--->conf: Req. id=' + res._Context.reqId + ', over');
	return;
}

/**
 * syncTransform:  		Process Transformation Request
 * @param  {Request}  	req Request Object
 * @param  {Response} 	res Response Object
 */
exports.syncTransform = function(req, res){

	var data = ''
	  , self = this
	  ;
	
	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);

	// Add the temp Dir to the Context
	res._Context.tempDir = npsTempDir + '/' + res._Context.reqId;

	// Add the Transformation Type (sync|async)
	res._Context.transType = 'sync';

	logger.info('--->syncTransform: Req. id=' + res._Context.reqId);

	req.on('data', function(chunk){
		data += chunk;
	});

	req.on('end', function(){
		soapServices.parseRequest(data, res, function(err){
			if(err){
				soapServices.sendError(res, null, err.message);
			   	_logServiceRespTime(res, 'syncTransform');
				return;
			}
			switch(res._Context.clientReqType){
				case 'transNET':
				case 'transJava': {
			        _transformReq(data, res);
					break;
				}
				case 'mergeNET':
				case 'mergeJava': {
					_pdfMerge(data, res);
					break;
				}
				default: {
					//  Unknown Request.
					logger.error('--->syncTransform: Req. id=' + res._Context.reqId + ', Unknown Request: ' + res._Context.clientReqType);
					soapServices.sendError(res, null, '--->syncTransform: Req. id=' + res._Context.reqId + ', Unknown Request: ' + res._Context.clientReqType);
				   	_logServiceRespTime(res, 'syncTransform');
					return;
					break;
				}
			}
		});
	});
} 

/**
 * asyncTransform:  	Process ASYNC Transformation Request
 * @param  {Request}  	req Request Object
 * @param  {Response} 	res Response Object
 */
exports.asyncTransform = function(req, res){

	var data = ''
	  , self = this
	  ;
	
	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);

	// Add the Transformation Type (sync|async)
	res._Context.transType = 'async';

	logger.info('--->asyncTransform: Req. id=' + res._Context.reqId);

	req.on('data', function(chunk){
		data += chunk;
	});

	req.on('end', function(){
		soapServices.parseRequest(data, res, function(err){
			if(err){
				soapServices.sendError(res, null, err.message);
			   	_logServiceRespTime(res, 'asyncTransform');
			   	return;
			}
			switch(res._Context.clientReqType){
				case 'transNET':
				case 'transJava': {
			        _transformReq(data, res);
					break;
				}
				default: {
					//  Unknown Request.
					logger.error('--->asyncTransform: Req. id=' + res._Context.reqId + ', Unknown Request: ' + res._Context.clientReqType);
					soapServices.sendError(res, null, '--->asyncTransform: Req. id=' + res._Context.reqId + ', Unknown Request: ' + res._Context.clientReqType);
				   	_logServiceRespTime(res, 'asyncTransform');
				   	return;
					break;
				}
			}
		});
	});
} 

/**
 * status:  			Process Status Request
 * @param  {Request}  	req Request Object
 * @param  {Response} 	res Response Object
 */
exports.status = function(req, res){

	var data = ''
	  , self = this
	  ;
	
	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);

	// Add the Transformation Type (sync|async)
	res._Context.transType = 'status';

	logger.info('--->status: Req. id=' + res._Context.reqId);

	req.on('data', function(chunk){
		data += chunk;
	});

	req.on('end', function(){
		soapServices.parseRequest(data, res, function(err){
			if(err){
				soapServices.sendError(res, null, err.message);
			   	_logServiceRespTime(res, 'status');
			   	return;			
			}
			switch(res._Context.clientReqType){
				case 'transNET':
				case 'transJava': {
			        _statusReq(data, res);
					break;
				}
				default: {
					//  Unknown Request.
					logger.error('--->status: Req. id=' + res._Context.reqId + ', Unknown Request: ' + res._Context.clientReqType);
					soapServices.sendError(res, null, '--->status: Req. id=' + res._Context.reqId + ', Unknown Request: ' + res._Context.clientReqType);
				   	_logServiceRespTime(res, 'status');
				   	return;			
					break;
				}
			}
		});
	});
} 

/************ OPTIONAL PUBLIC FUNCTIONS ***********/
/**
 * Set Log Level
 * @param {String} logLevel Log Level. values:DEBUG|INFO|WARN|ERROR 
 */
exports.setLogLevel = function(logLevel){
	pluginConf.log.level = logLevel || 'INFO';
	soapServices.setLogLevel(pluginConf.log.level);
	logger.setLevel(pluginConf.log.level);
}

/**
 * Get Log Level
 * @return {String} Log Level. values:DEBUG|INFO|WARN|ERROR
 */
exports.getLogLevel = function(){
	return pluginConf.log.level;
}

/**
 * enableRoute: 		Enable the given route
 * @param  {request} 	req
 *                      req.body = {
 *                      	"module":"sample", 
 *                      	"route": { 
 *                      		"path": "/path/to/the/service",
 *                      		"method": 	"get|post",
 *                      		"service": 	"version"
 *                      	}
 *                      }
 * @param  {function} 	callback(err)
 */
exports.enableRoute = function(req, callback){

	var self = this;

	logger.debug(__filename, '--->enableRoute');

	// Enable the route if route has been passed and route has been added previously
	if(req.body.route !== undefined && 
	   req.body.route.path !== undefined &&
	   req.body.route.method !== undefined &&
	   req.body.route.service !== undefined){
		expressAppUtils.enableRoute(self, req.body.route, function(err){
			if(err){
				return callback(err);
			}
			callback();
		});
	}
	else{
		// Display a Warning
		logger.warn('--->enableRoute: Wrong parameters! Not able to enable the route, Parms: ' + JSON.stringify(req.body));
		callback();
	}
} 

/**
 * disableRoute: 		Disable the given route
 * @param  {request} 	req
 * 		            	req.body = {
 *                      	"module":"sample", 
 *                      	"route": { 
 *                      		"path": "/path/to/the/service"
 * 		            	  	}
 * 		                }
 * @param  {function} 	callback(err)
 */
exports.disableRoute = function(req, callback){

	logger.debug(__filename, '--->disableRoute');

	// Disable the route
	if(req.body.route !== undefined && req.body.route.path !== undefined){
		expressAppUtils.disableRoute(req.body.route, callback);
	}
	else{
		// Display a Warning
		logger.warn('--->disableRoute: Wrong parameters! Not able to disable the route, Parms: ' + JSON.stringify(req.body));
		callback();
	}
} 

/**************** PRIVATE FUNCTIONS ***************/
// Process a Status Request
function  _statusReq(data, res){

	if (res._Context.req.input.document === undefined){
		logger.error('-->Error: Invalid SOAP Request. Reason: XML Tag Input Type, Output Type ' + 
					 'or Document is missing !\n' + 'SOAP Request:\n' + data);
		soapServices.sendError(res, null, 'Invalid SOAP Request. Reason: XML Tag Input Type, Output Type ' + 
		 					  'or Document is missing !\n' + 'SOAP Request:\n' + data);
	   	_logServiceRespTime(res, '_statusReq');
	   	return;			
	}

	res._Context.inputType 	= res._Context.req.input.type;		// set input type
	res._Context.inputDoc 	= res._Context.req.input.document;	// set input doc
	res._Context.outputType = res._Context.req.output.type;		// set output type
	// Set job Id (inputDoc is base64 encoded, so decode it!)
	res._Context.jobId 		= new Buffer(res._Context.req.input.document, 'base64');

	// Build Scenario Name
	res._Context.scenario = res._Context.inputType + '2' + res._Context.outputType;

	// BuildCheck if Scenario exists
	if (scenarios[res._Context.scenario] === undefined){
		logger.error('Scenario ' + res._Context.scenario + ' does NOT exit!');
        soapServices.sendError(res, null, 'Scenario ' + res._Context.scenario + ' does NOT exit!');
	   	_logServiceRespTime(res, '_statusReq');
	   	return;			
	}
	/*
		Must be meant for A2WS
		A2WS Service:
		 	asyncstatus2pdf		: AFP async Status
	*/
    logger.debug(__filename, 'Calling AFP2web Server Service...');
    return _a2wsService(data, res);
}

// Process a Transform Request
function  _transformReq(data, res){

	if (res._Context.req.input === undefined || 
		res._Context.req.input.type === undefined || 
		res._Context.req.input.document === undefined || 
		res._Context.req.output.type === undefined ){
		logger.error('-->Error: Invalid SOAP Request. Reason: XML Tag Input Type, Output Type ' + 
					 'or Document is missing !\n' + 'SOAP Request:\n' + data);
		soapServices.sendError(res, null, 'Invalid SOAP Request. Reason: XML Tag Input Type, Output Type ' + 
		 					  'or Document is missing !\n' + 'SOAP Request:\n' + data);
	   	_logServiceRespTime(res, '_transformReq');
	   	return;			
	}

    // V206 Begin
    // Validate JSON for 'ownerpw', 'userpw' and 'flags' elements of input, output and make there values as string
    //
    // Input elements
    if ( res._Context.req.input.ownerpw ){
        if ( typeof res._Context.req.input.ownerpw === 'string' ){
            // do nothing, use the value as is
        }
        else if ( typeof res._Context.req.input.ownerpw === 'object' ){
            // XML to JSON will pass element value in '_' and element attributes in '$'
            if ( res._Context.req.input.ownerpw._ ){
                res._Context.req.input.ownerpw = res._Context.req.input.ownerpw._;
            }
            else {
                res._Context.req.input.ownerpw = undefined;
            }
        }
        else {
            var sMsgTmp = 'input.ownerpw: Invalid data type to process (type:>' + typeof res._Context.req.input.ownerpw + '< value:>' + JSON.stringify( res._Context.req.input.ownerpw ) + '<)\n' + 'SOAP Request:\n' + data;
            logger.error( '-->Error: ' + sMsgTmp );
            soapServices.sendError( res, null, sMsgTmp );
            _logServiceRespTime( res, '_transformReq' );
            return;
        }
    }
    if ( res._Context.req.input.userpw ){
        if ( typeof res._Context.req.input.userpw === 'string' ){
            // do nothing, use the value as is
        }
        else if ( typeof res._Context.req.input.userpw === 'object' ){
            // XML to JSON will pass element value in '_' and element attributes in '$'
            if ( res._Context.req.input.userpw._ ){
                res._Context.req.input.userpw = res._Context.req.input.userpw._;
            }
            else {
                res._Context.req.input.userpw = undefined;
            }
        }
        else {
            var sMsgTmp = 'input.userpw: Invalid data type to process (type:>' + typeof res._Context.req.input.userpw + '< value:>' + JSON.stringify( res._Context.req.input.userpw ) + '<)\n' + 'SOAP Request:\n' + data;
            logger.error( '-->Error: ' + sMsgTmp );
            soapServices.sendError( res, null, sMsgTmp );
            _logServiceRespTime( res, '_transformReq' );
            return;
        }
    }
    if ( res._Context.req.input.flags ){
        if ( typeof res._Context.req.input.flags === 'string' ){
            // do nothing, use the value as is
        }
        else if ( typeof res._Context.req.input.flags === 'object' ){
            // XML to JSON will pass element value in '_' and element attributes in '$'
            if ( res._Context.req.input.flags._ ){
                res._Context.req.input.flags = res._Context.req.input.flags._;
            }
            else {
                res._Context.req.input.flags = undefined;
            }
        }
        else {
            var sMsgTmp = 'input.flags: Invalid data type to process (type:>' + typeof res._Context.req.input.flags + '< value:>' + JSON.stringify( res._Context.req.input.flags ) + '<)\n' + 'SOAP Request:\n' + data;
            logger.error( '-->Error: ' + sMsgTmp );
            soapServices.sendError( res, null, sMsgTmp );
            _logServiceRespTime( res, '_transformReq' );
            return;
        }
    }

    // Output elements
    if ( res._Context.req.output.ownerpw ){
        if ( typeof res._Context.req.output.ownerpw === 'string' ){
            // do nothing, use the value as is
        }
        else if ( typeof res._Context.req.output.ownerpw === 'object' ){
            // XML to JSON will pass element value in '_' and element attributes in '$'
            if ( res._Context.req.output.ownerpw._ ){
                res._Context.req.output.ownerpw = res._Context.req.output.ownerpw._;
            }
            else {
                res._Context.req.output.ownerpw = undefined;
            }
        }
        else {
            var sMsgTmp = 'output.ownerpw: Invalid data type to process (type:>' + typeof res._Context.req.output.ownerpw + '< value:>' + JSON.stringify( res._Context.req.output.ownerpw ) + '<)\n' + 'SOAP Request:\n' + data;
            logger.error( '-->Error: ' + sMsgTmp );
            soapServices.sendError( res, null, sMsgTmp );
            _logServiceRespTime( res, '_transformReq' );
            return;
        }
    }
    if ( res._Context.req.output.userpw ){
        if ( typeof res._Context.req.output.userpw === 'string' ){
            // do nothing, use the value as is
        }
        else if ( typeof res._Context.req.output.userpw === 'object' ){
            // XML to JSON will pass element value in '_' and element attributes in '$'
            if ( res._Context.req.output.userpw._ ){
                res._Context.req.output.userpw = res._Context.req.output.userpw._;
            }
            else {
                res._Context.req.output.userpw = undefined;
            }
        }
        else {
            var sMsgTmp = 'output.userpw: Invalid data type to process (type:>' + typeof res._Context.req.output.userpw + '< value:>' + JSON.stringify( res._Context.req.output.userpw ) + '<)\n' + 'SOAP Request:\n' + data;
            logger.error( '-->Error: ' + sMsgTmp );
            soapServices.sendError( res, null, sMsgTmp );
            _logServiceRespTime( res, '_transformReq' );
            return;
        }
    }
    if ( res._Context.req.output.flags ){
        if ( typeof res._Context.req.output.flags === 'string' ){
            // do nothing, use the value as is
        }
        else if ( typeof res._Context.req.output.flags === 'object' ){
            // XML to JSON will pass element value in '_' and element attributes in '$'
            if ( res._Context.req.output.flags._ ){
                res._Context.req.output.flags = res._Context.req.output.flags._;
            }
            else {
                res._Context.req.output.flags = undefined;
            }
        }
        else {
            var sMsgTmp = 'output.flags: Invalid data type to process (type:>' + typeof res._Context.req.output.flags + '< value:>' + JSON.stringify( res._Context.req.output.flags ) + '<)\n' + 'SOAP Request:\n' + data;
            logger.error( '-->Error: ' + sMsgTmp );
            soapServices.sendError( res, null, sMsgTmp );
            _logServiceRespTime( res, '_transformReq' );
            return;
        }
    }
    // V206 End

	res._Context.inputType 	= res._Context.req.input.type;		// set input type
	//!!! req.input.document is base64 encoded, so decode it!
	res._Context.inputDoc 	= new Buffer(res._Context.req.input.document, 'base64');
//!!!	res._Context.inputDoc 	= res._Context.req.input.document;	// set input doc
	res._Context.outputType = res._Context.req.output.type;		// set output type

	// Check whether inputType belongs to any
	if (a2wsAny.indexOf(res._Context.inputType.toLowerCase()) !== -1){
		res._Context.inputType = 'any';
	}
	// Build Scenario Name
	res._Context.scenario = res._Context.inputType + '2' + res._Context.outputType;

	// BuildCheck if Scenario exists
	if (scenarios[res._Context.scenario] === undefined){
		logger.error('Scenario ' + res._Context.scenario + ' does NOT exit!');
        soapServices.sendError(res, null, 'Scenario ' + res._Context.scenario + ' does NOT exit!');
	   	_logServiceRespTime(res, '_transformReq');
	   	return;			
	}
	// Set Scenario Object
	res._Context.scenarioObj = scenarios[res._Context.scenario];

	// Get transformation options from OTSSceneraionConf
	if(res._Context.scenarioObj && res._Context.scenarioObj.action 
			&& res._Context.scenarioObj.action.config
			&& scenarioConf[res._Context.scenarioObj.action.config]
		){
		res._Context.transformOptions = scenarioConf[res._Context.scenarioObj.action.config].props || {};
	} 


	/*
		PDF Service:
		 	pdf2pdf 		: SECURE_PDF
	*/
	if (res._Context.scenario === 'pdf2pdf'){
        return _pdfSecure(data, res);
	}
	/*
		html2any Service:
		 	mht2pdf			: HTML  --> PDF
		 	mht2tif			: HTML  --> TIF
	*/
	else if (['mht'].indexOf(res._Context.inputType.toLowerCase()) !== -1){
        return _html2any(data, res);
	}
	/*
		Office2any Service:
		 	html2pdf		: HTMLT --> PDF
		 	rtf2tif			: RTF   --> TIF
		 	rtf2tif_bw		: RTF   --> TIF
		 	rtf2pdf			: RTF   --> PDF
		 	word2tif		: WORD  --> TIF
		 	word2pdf		: WORD  --> PDF
		 	excel2tif		: EXCEL --> TIF
		 	excel2tif_bw	: EXCEL --> TIF
		 	excel2pdff		: EXCEL --> PDF
	*/
	else if (msofficeAny.indexOf(res._Context.inputType.toLowerCase()) !== -1){
        logger.debug(__filename, 'Calling msoffice Service...');
        return _msoffice(data, res);
	}
	/*
		Must be meant for A2WS
		A2WS Service:
		 	any2pdf			: AFP,..., --> PDF
		 	asyncafppdf		: AFP,..., --> PDF async
	*/
	else {
        logger.debug(__filename, 'Calling a2ws Service...');
        return _a2wsService(data, res);
	}
}

// Process an Office2Any Request
function  _msoffice(data, res){
	var self = this;

	var err = soapServices.validateTransformRequest(data, res);
	if (err !== null){
		soapServices.sendError(res, null, err.message);
	   	_logServiceRespTime(res, '_msoffice');
	   	return;			
	}
	return _processReq(res, 'oxsnps-msoffice2any', 'transformService');
}

// Process an HTML2any Request
function  _html2any(data, res){
	var self = this;

	var err = soapServices.validateTransformRequest(data, res);
	if (err !== null){
		soapServices.sendError(res, null, err.message);
	   	_logServiceRespTime(res, '_html2any');
	   	return;			
	}
	return _processReq(res, 'oxsnps-html2any', 'transformService');
}

// Process an AFP2web Server Request
function  _a2wsService(data, res){

	//!!! Assert
	var a2wsConf = scenarios[res._Context.scenario].action.config;
	logger.debug(__filename, 'Input: ' + res._Context.inputType + ', Output: ' + res._Context.outputType + ', Scenario: ' + a2wsConf);

	if (scenarioConf[a2wsConf] === undefined){
		logger.error(__filename, 'Scenario ' + res._Context.scenario + ': Could not find configuration ' + a2wsConf + ' in OTSScenarioConf.json.');
        return new Error('Scenario ' + res._Context.scenario + ': Could not find configuration ' + a2wsConf + ' in OTSScenarioConf.json.');
	}
	
	// Add the Scenario Configuration to the request
	res._Context.scenarioConf = scenarioConf[a2wsConf];

	// Async Trans., Sync Trans. or Status Req. or ?
	switch(res._Context.transType){
		case 'status':{
	    	return _processReq(res, 'oxsnps-afp2any', 'syncStatusService');
			break;		
		}
		case 'async': {
	    	return _processReq(res, 'oxsnps-afp2any', 'asyncTransformService');
			break;
		}
		default: {
		    return _processReq(res, 'oxsnps-afp2any', 'syncTransformService');
			break;
		}
	}
}

// Process a PDF Merge Request
function  _pdfMerge(data, res){
    return _processReq(res, 'oxsnps-pdf2pdf', 'mergeService');
}

// Process a PDF Secure Request
function  _pdfSecure(data, res){
	
	var self = this;

	// Assert Input buffer is not empty
	if ( res._Context.inputDoc.length <= 0){
		logger.error(__filename, 'Invalid SOAP Request. Reason: Input buffer is empty !\n' + 
					 'SOAP Request:\n' + data);
		soapServices.sendError(res, null, 'Invalid SOAP Request. Reason: Input buffer is empty !\n' + 
					 	 'SOAP Request:\n' + data);
	   	_logServiceRespTime(res, '_pdfSecure');
	   	return;
	}

    return _processReq(res, 'oxsnps-pdf2pdf', 'transformService');
}

// Process a Request
function  _processReq(res, pluginName, serviceName){
	
	logger.debug(__filename, '_processReq: Plugin: ' + pluginName + ', Service: ' + serviceName);

	// Get the handle of the requested plugin if enabled
	pluginManager.getModule(pluginName, function(err, module){
		if(err){
			logger.error('---> Req. id=' + res._Context.reqId + ', calling ' + pluginName + '.' + serviceName + ' failed, Reason: ' + err.message);
			soapServices.sendError(res, null, 'Req. id=' + res._Context.reqId + ', calling ' + pluginName + '.' + serviceName + ' failed, Reason: ' + err.message);
		   	_logServiceRespTime(res, '_processReq');
		   	return;			
		}
		// Try to call the requested Service from the given Plugin
		//!!! We should first check whether the service/route is enabled
		try{
			logger.info('--->_processReq, Calling ' + pluginName + '.' + serviceName + '...');
			module[serviceName](res, function(err, result){
				if(err){
					logger.error('---> Req. id=' + res._Context.reqId + ', calling ' + pluginName + '.' + serviceName + ' failed, Reason: ' + err.message);
					soapServices.sendError(res, null, 'Req. id=' + res._Context.reqId + ', calling ' + pluginName + '.' + serviceName + ' failed, Reason: ' + err.message);
				   	_logServiceRespTime(res, '_processReq');
				   	return;			
				} 
				logger.debug(__filename, '_processReq: Plugin: ' + pluginName + ', Service: ' + serviceName + ', Result Length: ' + result.length);
				soapServices.sendResponse(res, result);
			   	_logServiceRespTime(res, '_processReq');
			   	return;			
			});
		}catch(err){
			logger.error('---> Req. id=' + res._Context.reqId + ', calling ' + pluginName + '.' + serviceName + ' failed, Reason: ' + err.message);
			soapServices.sendError(res, null, 'Req. id=' + res._Context.reqId + ', calling ' + pluginName + '.' + serviceName + ' failed, Reason: ' + err.message);
		   	_logServiceRespTime(res, '_processReq');
		}
	});
}

/**
 * _logServiceRespTime:  	Log a Warning if Service Response time exceeded maxresptime 
 * @param  {res} 			res     	
 * @param  {String} 		service
 */
function _logServiceRespTime(res, service){

    var respTime 	= 0
	  , os 			= require('os')
      ;

    // Check resp. time only if maxresptime is defined
   	if(pluginConf.maxresptime){
		respTime = utils.dateDiff('x', res._Context.date, new Date()); // 'x' means returns diff. in ms

   		if(respTime > pluginConf.maxresptime * 1000){
			logger.warn('--->' + service + ': Req. id=' + res._Context.reqId + ', '
						+ ' Resp. Time: ' + respTime/1000 + 's, Max. allowed: '
						+ pluginConf.maxresptime + 's. Average Load: ' + os.loadavg());
   		}
   	}
}
