//------------------------------------------------------------------------------
// soapServices Module : Module to parse/validate SOAP Requests and to send SOAP Responses
//						 
//                    
// Author    : AFP2web Team
// Copyright : (C) 2014 by Maas Holding GmbH
// Email     : support@oxseed.de
// Version   : V1.0.0
// History
//   V100    09.08.2014     Initial release
//   
//
// Todos:
//------------------------------------------------------------------------------
var path 			= require('path')
  , log4js 			= require('log4js')
  , xml2js 			= require('xml2js') 				// to convert xml to json and vice versa  
  , cheerio 		= require('cheerio')				// to parse HTML

  , npsServer		= require('oxsnps-core/server')
  , npsConfDir 		= npsServer.npsConfDir				// holds the configuration Dir of the nps Server 

  , scenarioConf	= require(npsConfDir + '/OTSScenarioConf.js')
  , utils 			= require('oxsnps-core/helpers/utils')
  ;

var logger = log4js.getLogger('soapServices');

/**************** PUBLIC INTERFACE ***************/
exports.setLogLevel = function(logLevel){
	logger.setLevel(logLevel || 'INFO');
}

/**
 * Parses SOAP Request
 */
exports.parseRequest = function(data, res, callback){
	var inputType = ''			// theformat of the input doc
	  ,	outputType = ''			// the requested output format
	  ,	scenario = ''			// the scenario to be executed
	  ,	soapReq = ''			// the SOAP Request Content
	  ,	clientReqType = ''		// the Client Request Type: transform or merge req.?, Java or .NET Client?
	;

	// Add Context to the response instance
	if(res._Context === undefined || res._Context === null ){
		res._Context = {};
	}

	// Add the SOAP Envelope Namespace to the Context
	res._Context.nsEnv = '';

	// Add the Client type (.NET or Java) to the Context
	res._Context.clientReqType = '';

	// Add the transform or RequestTransformMessage Namespace to the Context
	res._Context.nsTransform = '';

	// Add the transformation Scenario to the Context
	res._Context.scenario = '';

	// Add the input doc type to the Context
	res._Context.inputType = '';

	// Add the output doc type to the Context
	res._Context.outputType = '';

	// Add the input doc buffer to the Context
	res._Context.inputDoc = '';

	// Add the JSON Transformtation Request Content to the Context
	res._Context.req = {};


	//logger.debug(__filename, 'parseRequest: SOAP Request:\n' + data);

	// Convert XML to JSON
	/*
		Java SOAP Request:
			<?xml version='1.0' encoding='UTF-8'?>
			<soapenv:Envelope xmlns:soapenv="http://schemas.xmlsoap.org/soap/envelope/">
				<soapenv:Header />
				<soapenv:Body>
					<ns1:transform xmlns:ns1="http://oxseed.com/services/">
						<ns1:RequestTransformMessage>
							<input xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xsi:type="ns1:InputObjectTransform">
								<document>WgAY06ioAAAC1MjjQEBAQEA</document>
								<type>afp</type>
							</input>
							<output>
								<type>pdf</type>
							</output>
						</ns1:RequestTransformMessage>
					</ns1:transform>
				</soapenv:Body>
			</soapenv:Envelope>

		.NET SOAP Request: does NOT have the "transform" tag

		in Java/Axis2 a request has the following structure:
			Envelope/Body/transform/RequestTransformMessage

		but in .NET it has:
			Envelope/Body/RequestTransformMessage

		The difference is that the .NET Request does NOT contain the "transform" tag (same holds true for merge request)
		 
		In the Response Axis client expects the "transformResponse" tag after the "body", but NOT .NET.
	*/
	var parser = new xml2js.Parser({explicitArray:false});
	parser.parseString(data, function (err, result){
			if (err){
				//logger.error(__filename, '-->Error: Could not parse the SOAP Request. Reason: ' + err.message);
				callback(new Error( 'Could not parse the SOAP Request. Reason: ' + err.message));
				return;
			}

			// Extract the Namespace of the SOAP Envelope and getStapel, if any
			for (prop in result) {
				logger.debug(__filename, 'parseRequest: (1)Prop Name: ' + prop);
				var envPos = prop.indexOf('Envelope');
				if (envPos >=0 ){ // Envelope found
					res._Context.nsEnv = prop.substring(0,envPos); // extract SOAP Envelope Namespace
					logger.debug(__filename, 'parseRequest: SOAP Envelope Namespace: ' + res._Context.nsEnv);
					var envObj = result[prop];
					for (prop in envObj) {	// Look for Body in the Envelope Object
						logger.debug(__filename, 'parseRequest: (2)Prop Name: ' + prop);
						if (prop === res._Context.nsEnv + 'Body'){ // Body found
							var bodyObj = envObj[prop];
							for (prop in bodyObj) {	// Look for 'transform' or 'RequestTransformMessage' in the Body Object
								logger.debug(__filename, 'parseRequest: (3)Prop Name: ' + prop);						
								// Look for 'transform' or 'RequestTransformMessage'
								if (prop.indexOf('transform') >=0 ){	// transform found
									res._Context.nsTransform = prop.substring(0,prop.indexOf('transform'));	// extract transform Namespace
									logger.debug(__filename, 'parseRequest: transform Namespace: ' + res._Context.nsTransform);
								}
								// Look for 'RequestTransformMessage'
								else if (prop.indexOf('RequestTransformMessage') >=0 ){	// RequestTransformMessage found
									res._Context.nsTransform = prop.substring(0,prop.indexOf('RequestTransformMessage'));	// extract RequestTransformMessage Namespace
									logger.debug(__filename, 'parseRequest: RequestTransformMessage Namespace: ' + res._Context.nsTransform);
								}
								// Look for 'merge'
								else if (prop.indexOf('merge') >=0 ){	// merge found
									res._Context.nsMerge = prop.substring(0,prop.indexOf('merge'));	// extract merge Namespace
									logger.debug(__filename, 'parseRequest: merge Namespace: ' + res._Context.nsMerge);
								}
								// Look for 'RequestMergeMessage'
								else if (prop.indexOf('RequestMergeMessage') >=0 ){	// RequestMergeMessage found
									res._Context.nsTransform = prop.substring(0,prop.indexOf('RequestMergeMessage'));	// extract RequestMergeMessage Namespace
									logger.debug(__filename, 'parseRequest: RequestMergeMessage Namespace: ' + res._Context.nsMerge);
								}
							}
						}
					}
				}
			}
			  //, soapbody 	= res._Context.nsEnv 		+ 'Body'
			// Define SOAP Envelope & Body based on the SOAP Namespace used
			var soapenv  	= res._Context.nsEnv 		+ 'Envelope'
			  , soapbody 	= res._Context.nsEnv 		+ 'Body'
			  , transform 	= res._Context.nsTransform	+ 'transform'
			  , reqTransMsg = res._Context.nsTransform	+ 'RequestTransformMessage'
			  , merge 		= res._Context.nsMerge		+ 'merge'
			  , reqMergeMsg = res._Context.nsMerge		+ 'RequestMergeMessage'
			;

			logger.debug(__filename, 'parseRequest: SOAP Envelope                XML Tag: ' + soapenv);
			logger.debug(__filename, 'parseRequest: SOAP Body                    XML Tag: ' + soapbody);
			logger.debug(__filename, 'parseRequest: SOAP RequestTransformMessage XML Tag: ' + reqTransMsg);

			// Check that the SOAP Request is well-formed. If not, leave
			if (result[soapenv] === undefined || result[soapenv][soapbody] === undefined){
				//logger.error(__filename, '-->Error: Invalid SOAP Request. Reason: XML Tag Envelope or Body is missing !\n' +									  'SOAP Request:\n' + data); 
				callback(new Error( 'Invalid SOAP Request. Reason: XML Tag Envelope or Body is missing !\n' + 'SOAP Request:\n' + data));
				return;
			}

			// Check whether the Request is a 'tranform' req. or a 'merge' req.
			if (result[soapenv][soapbody][transform] !== undefined){			// Java Transform Request contains 'ns1:transform'
				res._Context.clientReqType = 'transJava';
		        res._Context.req = result[soapenv][soapbody][transform][reqTransMsg];
			}
			else if (result[soapenv][soapbody][merge] !== undefined){			// Java Merge Request contains 'ns1:merge'
				res._Context.clientReqType = 'mergeJava';
		        res._Context.req = result[soapenv][soapbody][merge][reqMergeMsg];
			}
			else if (result[soapenv][soapbody][reqTransMsg] !== undefined){		// .NET Transform Request does NOT contain 'ns1:transform'
				res._Context.clientReqType = 'transNET';
		        res._Context.req = result[soapenv][soapbody][reqTransMsg];
			}
			else if (result[soapenv][soapbody][reqMergeMsg] !== undefined){		// .NET Merge Request does NOT contain 'ns1:merge'
				res._Context.clientReqType = 'mergeNET';
		        res._Context.req = result[soapenv][soapbody][reqMergeMsg];
			}
			else{
				//!!! Unknown Request.
				callback(new Error( 'Invalid SOAP Request.' + data));
			}
			callback(null);
			return;
   	});
}

/**
 * Validate SOAP Transform Request
 */
exports.validateTransformRequest = function(data, res){

	// Check if Scenario has action.config
	if (!res._Context.scenarioObj || 
		!res._Context.scenarioObj.action || 
		!res._Context.scenarioObj.action.config){
		logger.error(__filename, 'Scenario ' + res._Context.scenario + ' does NOT have a valid config in OTSScenarioList.json. Missing action or action.config');
        return new Error('Scenario ' + res._Context.scenario + ' does NOT have a valid config in OTSScenarioList.json. Missing action or action.config');
	}

	// Build Scenario Conf Name
	res._Context.scenarioConf = res._Context.scenarioObj.action.config;

	// Check if Scenario Conf. exists in OTSScenarioList.json
	if (scenarioConf[res._Context.scenarioConf] === undefined){
		logger.error(__filename, 'Scenario ' + res._Context.scenario + ': Could not find configuration ' + res._Context.scenarioConf + ' in OTSScenarioConf.json.');
        return new Error('Scenario ' + res._Context.scenario + ': Could not find configuration ' + res._Context.scenarioConf + ' in OTSScenarioConf.json.');
	}

	// Assert Input buffer
	if ( res._Context.inputDoc.length <= 0){
		logger.error(__filename, 'Invalid SOAP Request. Reason: Input buffer is empty !\n' + 
					 'SOAP Request:\n' + data);
		return new Error('Invalid SOAP Request. Reason: Input buffer is empty !\n' + 
					 	 'SOAP Request:\n' + data);		
	}
	return null;
}

/**
 * Validate SOAP Merge Request
 */
exports.validateMergeRequest = function (data, res ) {
	var self = this;
	var soapReq = res._Context.req;

	// Assert Java Services are started
	if( self.jServices === undefined || self.jServices === null ){
		//logger.error(__filename, '-->Error: Java Services are not initialized !\n' + 			'SOAP Request:\n' + data);
		return new Error('Java Services are not initialized !\n' + 
				 	     'SOAP Request:\n' + data);
	}

	// Assert Input filename list
	if ( soapReq.inputList === undefined){
		//logger.error(__filename, '-->Error: Invalid SOAP Request. Reason: XML Tag Input List is missing !\n' + 					 'SOAP Request:\n' + data);
		return new Error('Invalid SOAP Request. Reason: XML Tag Input List is missing !\n' + 
						 'SOAP Request:\n' + data);
	}

	// Assert Input filename list is not empty
	if (soapReq.inputList.length <= 0 ){
		//logger.error(__filename, '-->Error: Invalid SOAP Request. Reason: XML Tag Input List is empty !\n' + 					 'SOAP Request:\n' + data);
		return new Error('Invalid SOAP Request. Reason: XML Tag Input List is empty !\n' + 
						 'SOAP Request:\n' + data);
	}
	return null;
}

/**
 * Returns a SOAP Response
 */
exports.sendResponse = function(res, resData, resStatus, resStatusDescription){
	/*
		Build the SOAP Response

		Java SOAP Response:
			<?xml version='1.0' encoding='UTF-8'?>
			<soapenv:Envelope xmlns:soapenv="http://schemas.xmlsoap.org/soap/envelope/">
				<soapenv:Body>
					<transformResponse>
						<ns1:ResponseMessage xmlns:ns1="http://oxseed.com/services/">
							<responseData>JVBERi0xLjI</responseData>
							<responseStatus>Done</responseStatus>
							<responseStatusDescription/>
						</ns1:ResponseMessage>
					</transformResponse>
				</soapenv:Body>
			</soapenv:Envelope>

		in Java/Axis2 a Response has the following structure:
			Envelope/Body/transformResponse/ResponseMessage

		but in .NET it has:
			Envelope/Body/ResponseMessage

		The difference is that the .NET Request does NOT contain the "transformResponse" tag
	*/
	var soapenv 	= res._Context.nsEnv 		+ 'Envelope'
	  , soapbody 	= res._Context.nsEnv 		+ 'Body'
	  , transform 	= 'transformResponse'	// NO namespace expected here
	  //, respMsg 	= res._Context.nsTransform	+ 'ResponseMessage'
	  , respMsg 	= ''
	  , soapObj 	= {}
	  , soapTrans 	= {}
	;

	soapObj[soapenv] = {
		"$": {	// Attributes
			"xmlns:soapenv": "http://schemas.xmlsoap.org/soap/envelope/",
			"xmlns:xsd": 	 "http://www.w3.org/2001/XMLSchema",
			"xmlns:xsi": 	 "http://www.w3.org/2001/XMLSchema-instance"
		}
	};
	soapObj[soapenv][soapbody] = {};

	// 'transform' or 'Merge' request ?
	if(res._Context.clientReqType === 'mergeJava' || res._Context.clientReqType === 'mergeNET'){//  Merge Request
		respMsg = res._Context.nsMerge + 'ResponseMessage';
	}
	else{// Transform  Request
		respMsg = res._Context.nsTransform + 'ResponseMessage';
	}

	soapTrans[respMsg] = {
		"$": {	// Attributes
			"xmlns:ns1": "http://oxseed.com/services/"
		}
	};

	// Async Trans., Sync Trans. or Status Req. or ?
	switch(res._Context.transType){
		case 'status':{
			/** 
			 * Get Status Response from A2W Server 
			 * <html>
			 * <head>
			 * <title>AFP2web Server</title>
			 * </head>
			 * <body>
			 * <pre><h1>AFP2web Server</h1>
			 * PID: 20176<br />
			 * RequestNumber: 58<br />
			 * ApplicationName: AFP2web Server<br />
			 * JobID: 140814170447583<br />
			 * ResponseTime: 1ms<br />
			 * Status: Done | Working | Error, ErrorMsg:<ErrorMsg><br />
			 * ReturnCode: 0<br />
			 * </pre>
			 * </body></html>
			 */
			 // Get Text of Status attribute

			// resData is null in case of error response
			if(resStatus === 'Error'){
				resStatusDescription = _getAttrTextFromA2WSResponse(resStatusDescription, 'pre', 'Status:', 'text');
				if(resStatusDescription === null || resStatusDescription.length <=0){
					return new Error('Status is missing in AFP2web Server Response');
				}
			}
			else if(resData === null){
				return new Error('Status is missing in AFP2web Server Response');
			}
			else{
				resStatus = _getAttrTextFromA2WSResponse(resData, 'pre', 'Status:', 'text');			
				if(resStatus === null || resStatus.length <=0){
					return new Error('Status is missing in AFP2web Server Response');
				}

				 // Is it an error?
		 		var errMsgIdx = resStatus.indexOf('ErrorMsg:');   
				if(errMsgIdx > 0){
					errMsgIdx += 'ErrorMsg:'.length;
					resStatusDescription = resStatus.substring(errMsgIdx ).trim();
					resStatus = 'Error';
					logger.error('AFP2web Server Error, ' + resStatusDescription);

				}
			}
			resData = res._Context.jobId;	// set responseData as JobId
			break;	
		}
		// For Async Trans. set responseData as JobId and Working as responseStatus
		case 'async': {
			resStatus = resStatus || 'Working';
			resData = res._Context.jobId;
			break;
		}
		default: {
			break;
		}
	}

	// resData is null in case of error response
	if(resData !== null){
		soapTrans[respMsg].responseData = new Buffer(resData, 'binary').toString('base64');	//!!! 'binary' is deprecated but I don't have a better way right now !?
	}

	soapTrans[respMsg].responseStatus = resStatus || 'Done'; // MUST BE 'Done', 'Working' or 'Error'
	soapTrans[respMsg].responseStatusDescription = resStatusDescription || '';


	// '.NET' or 'Java' ?
	if(res._Context.clientReqType === 'transJava' || res._Context.clientReqType === 'mergeJava'){
		soapObj[soapenv][soapbody][transform] = soapTrans;	// add the 'transformResponse' XML tag + the Response Message
	}
	else{// .NET
		soapObj[soapenv][soapbody] = soapTrans;	// add the Response Message
	}

	var xml2js 	= require('xml2js');
	// default renderOpts:{'pretty':true,'indent':' ','newline':'\n'}
	// default xmldec: {'version':'1.0','encoding':'utf-8','standalone':true}
	var builderOpts = {
		renderOpts:{
			'pretty': false	// pretty print or not
		},
		xmldec: {
			'version':'1.0',
			'encoding':'utf-8'
		}
	}
	if (logger.isDebugEnabled()){
		builderOpts.renderOpts.pretty = true;
	}
	var builder = new xml2js.Builder(builderOpts);	
	var soapResponse = builder.buildObject(soapObj);
//!!!	logger.debug(__filename, '_sendSOAPResponse: SOAP Response:\n' + soapResponse);
	res.setHeader('Content-Type', 'text/xml; charset=utf-8');	// or  application/xml or application/soap+xml ??
	res.setHeader('Content-Length', utils.strLength(soapResponse));	// _strLength takes in account 2 bytes chars
	res.end(soapResponse); 

	// Async Trans., Sync Trans. or Status Req. or ?
	var functionName = res._Context.transType + 'Transform';
	if(res._Context.transType === 'status'){
		functionName = res._Context.transType;
	}
	logger.info('--->' + functionName + ': Req. id=' + res._Context.reqId + ', SOAP Response sent back, over');
}

/**
 * Return Error as SOAP Response
 */
exports.sendError = function(res, resData, errorMsg){
	var self = this;
	return self.sendResponse(res, resData, 'Error', errorMsg);
}

/***** Private Functions *********/
// Get Text of an Attribute placed under given tag from AFP2web Server Response Body
function _getAttrTextFromA2WSResponse(html, tagName, attrName, attrType){

	// Parse HTML
	var parsedHTML = cheerio.load(html);

	// Get contents under tag. For example <pre>
	var tag = parsedHTML(tagName, 'body');

	// Filter contents for the specific attribute
	var targetLine = "" + tag.contents().
					filter(function(i, el){
						if(el.type === attrType && el.data.indexOf(attrName)>=0){ 
	      					return true;
	  					}
					});
	var attrText = null;

	if(targetLine !== null && targetLine.length>0){
		attrText = targetLine.substring(targetLine.indexOf(':') + 1).trim();
	}
	return attrText;
}

