//------------------------------------------------------------------------------
// a2wsServices Module : Module to manage AFP2web Server services that in 
//						 turn uses the AFP2web Server HTTP APIs
//                    
// Author    : AFP2web Team
// Copyright : (C) 2014 by Maas Holding GmbH
// Email     : support@oxseed.de
// Version   : V1.0.0
// History
//  V100    01.08.2014     Initial release
// 	V101	22.09.2014     Added handling of Async Reqs
// 	V102	03.11.2014     Fixed a bug that lead to passing InputBuffer twice for sync. transformations
//   
//
// Todos:
//------------------------------------------------------------------------------
var fs 			= require('fs')
  , log4js 		= require('log4js')

  , npsServer		= require('oxsnps-core/server')  
  , npsConf 		= npsServer.npsConf  

  , npsLogFile 		= npsServer.npsLogFile
  
  , http 		= require('http')
  , hostname 	= '127.0.0.1'
  , port 		= '80'
  , utils		= require('oxsnps-core/helpers/utils')
  , httptimeout = undefined
  , urlVersionContext 			= ''
  , urlTransformContext 		= ''	
  , urlStatusContext 			= ''
  , urlAsyncTransformContext 	= ''
  ;

// Get Logger
var logger = log4js.getLogger('a2wsServices');
logger.setLevel(npsServer.npsConf.log.level);

/**************** PUBLIC INTERFACE ***************/
exports.setLogLevel = function(logLevel){
	logger.setLevel(logLevel || 'INFO');
}

/**
 * Initialize AFP2web Server's Hostname and Port.
 */
exports.initialize = function(pluginConf){

	var serverConf = undefined
	  , urlContext = undefined
	  ;

	if(npsConf.plugins && npsConf.plugins['oxsnps-afp2any'])  serverConf = npsConf.plugins['oxsnps-afp2any'];
	else serverConf = {};

	// Set Hostname and Port
    hostname 	= serverConf.hostname 	|| pluginConf.name 		|| '127.0.0.1';	// AFP2web Server Host (Default is 127.0.0.1)
	port 		= serverConf.port 		|| pluginConf.port 		|| '80';					// AFP2web Server Port (Default is 80)
	urlContext 	= serverConf.context 	|| pluginConf.context 	|| '';

	if(urlContext && (typeof urlContext == 'string') && urlContext.length >0){
		urlVersionContext 			= '/' + urlContext;	
		urlTransformContext 		= '/' + urlContext;
		urlStatusContext 		 	= '/' + urlContext;	
		urlAsyncTransformContext 	= '/' + urlContext;
	}
	else{
		if(urlContext && urlContext.version && urlContext.version.length >0) urlVersionContext = '/' + urlContext.version;
		else urlVersionContext = '';

		if(urlContext && urlContext.transform && urlContext.transform.length >0 ) urlTransformContext = '/' + urlContext.transform;
		else urlTransformContext = '';

		if(urlContext && urlContext.status && urlContext.status.length >0) urlStatusContext = '/' + urlContext.status;
		else urlStatusContext = '';

		if(urlContext && urlContext.asyncTransform && urlContext.asyncTransform.length >0) urlAsyncTransformContext = '/' + urlContext.asyncTransform;
		else urlAsyncTransformContext = '';
	}

	// Default is 5 minutes, just as apache timeout default value
	httptimeout = pluginConf.httptimeout || 5*60*1000;	// httptimeout in milliseconds.

	//LOG	console.log('host:', hostname, ', port:', port);
}

// Get the AFP2web Server & AFP2web Version
exports.getVersion = function(a2wsProps, callback){

	//LOG console.log('getVersion properties:\n' + JSON.stringify(a2wsProps));

	// Add path to the props
	//a2wsProps.path = '/services/transform';
	a2wsProps.path = urlVersionContext + '/services/version';

	// AFP2web Version or AFP2web Version all ?
	if (a2wsProps.a2w_version === 'all'){
		a2wsProps.a2w_version = 'off';
		a2wsProps.a2w_version_all = 'on'
	}
	
	// AFP2web Server Version or AFP2web Server Version all ?
	if (a2wsProps.transform_version === 'all'){
		a2wsProps.transform_version = 'off';
		a2wsProps.transform_version_all = 'on'
	}

	// Set Response Format to html
	a2wsProps.responseformat = 'html';

	// POST the Multipart Request
	_postMultipartRequest(a2wsProps, callback);		
}

// Call AFP2web Server's syncTransform Service (Buff In, Buf Out)
exports.syncTransformBuf = function(a2wsProps, callback){

	//LOG console.log('syncTransformBuf properties:\n' + JSON.stringify(a2wsProps));

	// Add  path to the props
	a2wsProps.path = urlTransformContext + '/services/transform';

	// POST the Multipart Request
	_postMultipartRequest(a2wsProps, callback);		
}

// Call AFP2web Server's syncMerge Service (nxFiles In as Files URLs, 1 Buf Out)
exports.syncMerge = function(a2wsProps, callback){

	logger.debug(__filename, 'syncMerge: Properties:\n' + JSON.stringify(a2wsProps));

	// Add path and FileCreationMode to the props
	a2wsProps.path = urlTransformContext + '/services/transform';
	a2wsProps.FileCreationMode = 'DOC_MERGE';	// Watch out !!!

	// POST the Multipart Request
	_postMultipartRequest(a2wsProps, callback);		
}
		
/**
	Call AFP2web Server's syncTransformServerFile Service (1..n File(s) In, 1..n File(s) Out)
	The input file(s) MUST reside on the server. The output file(s) will be stored on the server. 
	Possible scenarios are:
		1.  1xFile  In - 1xFile  Out
		2.  1xFile  In - nxFiles Out
		3.  nxFiles In - 1xFile  Out
		4.  nxFiles In - nxFiles Out		
*/
exports.syncTransformServerFile = function(a2wsProps, callback){

	logger.debug(__filename, 'syncTransformServerFile: Properties:\n' + JSON.stringify(a2wsProps));

	// Add path to the props
	a2wsProps.path = urlTransformContext + '/services/transform';
	// OutputFilePath MUST be specified. It will be used to differenciate whether Output is Buffer or File (specified means File(s) out)
	a2wsProps.OutputFilePath = a2wsProps.OutputFilePath || 'pdf'; 

	// POST the Multipart Request
	_postMultipartRequest(a2wsProps, callback);		
}
	
/**
	Call AFP2web Server's asyncTransformServerFile Service 
	Possible scenarios are:
		1.  1xFile  In - 1xFile  Out
		2.  1xFile  In - nxFiles Out
	 	3.  nxFiles In - 1xFile  Out
	 	4.  nxFiles In - nxFiles Out
*/ 	
exports.asyncTransformServerFile = function(a2wsProps, callback){

	logger.debug(__filename, 'asyncTransformServerFile: Properties:\n' + JSON.stringify(a2wsProps));

	// Add path to the props
	a2wsProps.path = urlAsyncTransformContext + '/services/asyncTransform';
	// OutputFilePath MUST be specified. It will be used to differenciate whether Output is Buffer or File (specified means File(s) out)
	a2wsProps.OutputFilePath = a2wsProps.OutputFilePath || 'pdf'; 

	// POST the Multipart Request
	_postMultipartRequest(a2wsProps, callback);		
}

// Call AFP2web Server's getStatus (to be used after a call to asyncTransformServerFile)
exports.getStatus = function(a2wsProps, callback){

	logger.debug(__filename, 'getStatus: Properties:\n' + JSON.stringify(a2wsProps));

	// Add path to the props
	//a2wsProps.path = '/services/jobstatus';
	a2wsProps.path = urlStatusContext + '/services/getStatus';

	// POST the Multipart Request
	_postMultipartRequest(a2wsProps, callback);		
}

// Call AFP2web Server's syncTransformURL Service (1xUrl In, 1xBuff Out)
exports.syncTransformURL = function(a2wsProps, callback){

	logger.debug(__filename, 'syncTransformURL: Properties:\n' + JSON.stringify(a2wsProps));

	// Add  path to the props
	a2wsProps.path = urlTransformContext + '/services/transform';

	// POST the Multipart Request
	_postMultipartRequest(a2wsProps, callback);		
}

/***** Private Functions ***********/
// Post the Request
function _postMultipartRequest(a2wsProps, callback){

	// Build the POST Req Parameters
	var crlf = "\r\n",
		//boundary = '---------------------------9r-SzFGlBRvRAc8RJbwAJ4i-Idwoujo2F-EPQ6v', // Boundary: "--" + up to 70 ASCII chars + "\r\n"
		min = 100000000000000,
		max = 1000000000000000,
		//boundary = '---------------------------' + Math.floor(Math.random() * (max - min + 1) + min),
		boundary = 'oswhTEndxHsTZddqkYwAg6-sbh6HP0e5R',
		delimiter = crlf + "--" + boundary,
		preamble = "", // ignored. a good place for non-standard mime info
		epilogue = "", // ignored. a good place to place a checksum, etc
		jsonResponse	= undefined,
		err 			= undefined,
		contentType 	= undefined,	
		bufferHeader = [
		  'Content-Disposition: form-data; name="InputBuffer"; filename="' + a2wsProps.Filename + '"' + crlf,
		  //'Content-Type: application/octet-stream' + crlf,
		  'Content-Type: application/afp' + crlf,
		  'Content-Transfer-Encoding: binary' + crlf,
		],
		
		optionHeader1 = [
		  'Content-Disposition: form-data; name='
		],
		optionHeader2 = [
		  'Content-Type: text/plain; charset=US-ASCII' + crlf,
		  'Content-Transfer-Encoding: 8bit' + crlf,
		],						
		closeDelimiter = delimiter + "--" + crlf, // crlf is VERY IMPORTANT here
		
	// Here is how the multipartBody MUST be built 
	// multipartBody  =	preamble
	//				  + 1..n[delimiter + crlf + optionHeader1 + optionName + crlf + optionHeader2 + crlf + optionValue]
	//				  + delimiter + crlf + bufferHeader + crlf + InputBuffer
	//				  + closeDelimiter + epilogue
	// Concatenate preamble
	multipartBody = Buffer.concat([
		new Buffer(preamble)
	]);
			
	// Set up Logging options
	if (a2wsProps.transform_logging !== undefined && a2wsProps.transform_logging === "all"){		// Turn on both AFP2web Server & AFP2web Logging
		a2wsProps.transform_logging = "on";
		a2wsProps.logging = "on";
	}
	// Set Response Format to if not given
	if(!a2wsProps.responseformat) a2wsProps.responseformat = 'json';

	// Concatenate a2wProps
	if (a2wsProps) {
		for (var key in a2wsProps) {
			if (key === "Filename" || key === "path"){}	// Is it correct to NOT pass "Filename" ???
			// V102 Begin
			else if (key === "InputBuffer"){}	// InputBuffer is attached separately below.
			// V102 End
			else {
				multipartBody = Buffer.concat([
					multipartBody,
					new Buffer(delimiter + crlf + optionHeader1.join('') + '"' + key + '"' + crlf),
					new Buffer(optionHeader2.join('') + crlf + a2wsProps[key])
				]);	
			}
		}
	}
	
	// Concatenate InputBuffer...if provided
	if (a2wsProps.InputBuffer){
		multipartBody = Buffer.concat([
			multipartBody,
			new Buffer(delimiter + crlf + bufferHeader.join('') + crlf),
			a2wsProps.InputBuffer
		]);
	}

	// Concatenate closeDelimiter + epilogue
	multipartBody = Buffer.concat([
		multipartBody,
		new Buffer(closeDelimiter + epilogue)
	]);	
					
	// Set the POST Req Options
	var post_options = {
		hostname:	hostname,
		port: 		port,
		path: 		a2wsProps.path,
		method: 	'POST',
		headers: 	{
			'Content-Type': 'multipart/form-data; boundary=' + boundary,
			'Content-Length': multipartBody.length
		}
	};

	// Add Host prop in Header if passed
	if(a2wsProps.Host) post_options.headers.Host = a2wsProps.Host // V104 // V2.0.21
	
	//logger.debug('--->_postMultipartRequest: Request:' + JSON.stringify(post_options));
	//logger.debug('--->_postMultipartRequest: Body:' + JSON.stringify(multipartBody.toString()));

	// Set up the request
	var post_req = http.request(post_options);

	// Timeout handling
	logger.debug('--->_postMultipartRequest: Request timeout set to ' + httptimeout + ' ms.');
	post_req.setTimeout(httptimeout, function(){
    	logger.error('--->_postMultipartRequest: Request timeout(' + httptimeout + ' ms) occurred.');
    	post_req.abort();
    	// post_req.abort will trigger 'error' event which invokes callback(). 
    	// So do not invoke callback() here
    	// Refer to http://stackoverflow.com/questions/6214902/how-to-set-a-timeout-on-a-http-request-in-node
    	//callback(new Error('Request timeout(' + httptimeout + ' ms). occurred.'));
	});

	// Post the data
	post_req.write(multipartBody);
	post_req.end();

	// V101 Begin
	// Commented since A2WS returns immediately after scheduling the job
	/*
	if( a2wsProps.path.indexOf( 'async') >= 0 ){
		// If it is a async request, do not wait for response and return immediately 
		return callback();
	}
	// V101 End
	else{
	*/		
	// Response handling
	post_req.on('response', function(res){

		var response = '';				// response of a request as a concatenation of chunks
		res.setEncoding('binary');
		res.on('data', function (chunk) {
			// Concatenate the chunks
			response += chunk;
		});

		// Call the callback function passing the response
		res.on('end', function(){
			contentType = res.headers['content-type'];
			contentType = contentType ? contentType.toLowerCase() : 'text/html';
			//logger.debug('--->_postMultipartRequest: contentType=' + contentType);
			//logger.debug('--->_postMultipartRequest: Response=' + response);
			// Success case
			if(res.statusCode === 200){ // HTTP status code 200 means everything went ok
				/**
				 * Response contains a data buffer for Buffer Outputs
				 * Response contains JSON for File outputs, Async API, GetStatus API
				 * Response contains html for GetVersion API
				 */
				/*
				Success Case: (HTTP StatusCode: 200)
					{
					  "PID": 17847,
					  "RequestNumber": 7,
					  "ApplicationName": "AFP2web Server",
					  "Status": "Success",
					  "ReturnCode": 0,
					  "ResponseTime": 63,
					  "Statistics": {
					    "InputPageCount": 1,
					    "InputFilenames": [
					      "/var/lib/apache2/fcgid/a2wserver/samples/ots-1024.pdf"
					    ],
					    "DocumentCount": 1,
					    "PageCount": 1
					  }
					}
				 */
				if(contentType === 'application/json' || contentType === 'json'){
					jsonResponse = utils.parseJSON(response);
					if(!jsonResponse){
						err = new Error();
						err.httpStatusCode = 422; // set HTTP Status code in error object
						err.message = 'Unable to parse AFP2web Server JSON response. Response=' + response;
						return callback(err); 
					}
					else return callback(null, jsonResponse); 
				} 
				return callback(null, response); 
			}
			// A2WS Failure case
			else if(res.statusCode === 422){
					/*
						Failure case: (HTTP StatusCode: 422)
						{
						  "PID": 17847,
						  "RequestNumber": 3,
						  "ApplicationName": "AFP2web Server",
						  "Status": "Error: Pid=17847 ReqId=00003 Exception while processing request 3. Msg: Input (/home/mathi/temp/ots-1024.pdf) is not found",
						  "ReturnCode": -1,
						  "ResponseTime": 3
						}
					*/
				
				err = new Error();
				err.httpStatusCode = res.statusCode; // Save HTTP Status code in error object

				if(contentType === 'application/json' || contentType === 'json'){
					jsonResponse = utils.parseJSON(response);
					if(jsonResponse) 
						err.message = 'AFP2web Server Error, HTTP Status Code=' + res.statusCode + ', Reason=' + jsonResponse.Status;
					else 
						err.message = 'AFP2web Server Error, HTTP Status Code=' + res.statusCode + ', Unable to parse JSON response. Response=' + response;					}
				else err.message = 'AFP2web Server Error, HTTP Status Code=' + res.statusCode + ', Response=' + response;

				return callback(err);
			}
			// Must be an Apache Error/application termination Errors like segmentation error
			else{
				err = new Error();
				err.httpStatusCode = res.statusCode; // Save HTTP Status code in error object
				err.message = 'AFP2web Server Error, HTTP Status Code=' + res.statusCode + ', Reason: ' + response
				return callback(err);
			}
			//_getA2WSErrorMessage(res, response, callback); // We have either an Apache Error or an AFP2web Server Error
		});	
	});
	
	// Error handling 
	post_req.on('error', function(err){
		if(err.code === 'ECONNRESET')
			return callback(new Error('Request timeout(' + httptimeout + ' ms) occurred. Reason: ' + err.message));
		callback(new Error('AFP2web Server Error, Reason: ' + err.message));
	});
}

/**
 * Extract A2WS Error Message if any or return Apache error msg
 * @param  {JSON}   	res
 * @param  {String} 	errHTML HTML String
 * @param  {Function} 	callback
 */
function  _getA2WSErrorMessage_notused(res, errHTML, callback){

	var status 		= undefined
	  , errMsgIdx 	= undefined
	  , errMsg 		= undefined
	  ;

	/**
	 * Sample error message from A2WS:
	 * 
	 * Get Job Status ERROR Response from AFP2web Server.		
	 * <html>
	 * <head>
	 * <title>AFP2web Server</title>
	 * </head>
	 * <body>
	 * <pre><h1>AFP2web Server</h1>
	 * PID: 20176<br />
	 * RequestNumber: 81<br />
	 * ApplicationName: AFP2web Server<br />
	 * JobID: 140814170447585<br />
	 * ResponseTime: 0ms<br />
	 * Status: failed, Reason: Exception occurred while processing the request. Msg:Job file font found in Job directory. JobID:140814170447585, Job Directory:/var/lib/apache2/fcgid/a2wserver_v1.3.0.2_ms4_20140717/services/a2wtransformation/. Job Id may be invalid or Job file deleted by other programs (rc=-1)<br />
	 * ReturnCode: -1<br />
	 * </pre>
	 * </body>
	 * </html>
	 */

	// Looking for the "* Status: failed, Reason: Exception ..." line
	status = utils.getAttrTextFromHTML(errHTML, 'pre', 'Status:', 'text');
	if(status === null || status.length <=0){ // must be a Apache Error
		return callback(new Error('AFP2web Server Error, HTTP Status Code=' + res.statusCode + ', Reason: ' + errHTML));
	}

	// Extracting & returning the text occuring after "Reason:..."" in the status line
	errMsgIdx = status.indexOf('Reason:');
	errMsg = status;
	if( errMsgIdx > 0 ){
		errMsgIdx += 'Reason:'.length;
		errMsg = status.substring(errMsgIdx).trim();
	}		
	return callback(new Error('AFP2web Server Error, HTTP Status Code=' + res.statusCode + ', Reason: ' + errMsg));
}
