/**-----------------------------------------------------------------------------
 * pluginManager.js: 	Node.js module to manage plugins
 * 
 * Author    :  AFP2web Team
 * Copyright :  (C) 2014 by Maas Holding GmbH
 * Email     :  support@oxseed.de
 * Version   :  V1.0.0
 * 
 * History
 *  V100   24.02.2015  Initial release
 *
 *----------------------------------------------------------------------------*/
'use strict';

var MODULE_NAME		= 'pluginManager'
  , MODULE_VERSION	= '2.0.0'
  , fs          	= require('fs')
  , path        	= require('path')
  , log4js			= require('log4js')
  ,	async 			= require('async')

  , npsServer		= require('./server')  
  , expressApp		= npsServer.expressApp				// is the Express App
  , npsDir			= npsServer.npsDir					
  , npsConf 		= npsServer.npsConf					// holds the configuration of the OXSNPS Server 
  , warn 			= require('oxsnps-core/helpers/warn')  
  , npsLogFile 		= npsServer.npsLogFile
  , utils 			= require('./helpers/utils')
  , expressAppUtils = require('./helpers/expressAppUtils')
  , pluginJSfilename= 'oxsnps.js'						// Mandatory js filename of an OXSNPS Plugin
  , plugins 		= [] 								// holds the list of the plugins
/*
	Structure of the plugins array
    "plugins":[
		{
			"module": 		"Plugin Name (ex.: afp2any)",
			"plugindir": 	"FQ name of the the plugin root directory (ex.: /path/to/pluginDir/)",
			"enable": 		"on|off",
			"desc": 		"Short description of the plugin",
			"longDesc": 	"Long description of the plugin",
			"logLevel":		"DEBUG|INFO|ERROR",
			"routes": [
				{
					"path": 	"/path/to/the/service",
					"enable": 	"on/off"
					// HTTP routes:
					"method": 	"get/post",
					"service": 	"name of the plugin service to be mapped to path",
					// dirwacther routes:
					"events": 	"...to watch for",
					"options": "see node,js chokidar options",
					"action": 	{
		                "interface": "[http | local]: http or local call of a service of a module",
		                "props":    "the props required by the interface",
		                "props":    {
							"module":  "module/plugin name",
							"module":  "./local/ic1",
							"service": "ic1Service"
		                },
		                "parms":    "the props required by the service",
				        "parms":	{
				        	"prop1": "prop1",
				        	"prop2": "prop2"
				        }
					}
				}
			]
		},
		{...}
	]
*/
  ;

// Get Logger
var logger = log4js.getLogger(MODULE_NAME);
logger.setLevel(npsConf.log.level);

// Export Module Version
exports.version = MODULE_VERSION;

/**
 * version: 		Process a Version Request
 * @param  {req}	req Request Object
 * @param  {res}	res Response Object
 */
exports.version = function(req, res){

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);
	logger.info('--->version: Req. id=' + res._Context.reqId);

	var data = '<h3>' +  MODULE_NAME + ' v' + MODULE_VERSION + '</h3>';
	res.end(data);
	
	logger.info('--->version: Req. id=' + res._Context.reqId + ', Version sent back, over');
}

/**
 * enablePlugin: 		Enable the given plugin.
 * @param  {req} 		req as GET or POST 
 * 		   GET URL: 	<server>[:port]/appservices/pm/enable?module=oxsnps-afp2any
 * 		   POST URL: 	<server>[:port]/appservices/pm/enable
 * 		   POST parms:	req.body = {
 * 		            		"module":"oxsnps-afp2any", 
 * 		              	}
 * @param  {res} 		res
 */
exports.enablePlugin = function(req, res){

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);

	// Add the plugin name to the Context
	if(req.method === 'GET') res._Context.module = req.query.module;
	else res._Context.module = req.body.module;

	logger.info('--->enable: Req. id=' + res._Context.reqId);
	logger.debug('--->enable: Req. id=' + res._Context.reqId + ', Parms: ' + (req.method === 'GET'?JSON.stringify(req.query):JSON.stringify(req.body)));

	// Get Plugin
	_getPlugin(res._Context.module, function(err, plugin){
		if(err){
			logger.error('--->enablePlugin: Req. id=' + res._Context.reqId + ', Plugin enable request failed: ' + err.stack + '. Parms:' + JSON.stringify(req.body));
			return res.status(404).end('Plugin enable request failed: ' + err.stack + '. Parms:' + JSON.stringify(req.body));
		}

		// If the plugin is already enabled (plugin is loaded only when it is enabled), leave
		if(plugin.enable === undefined || plugin.enable === 'on'){
			return res.end('Plugin ' + res._Context.module + ' already enabled');
		}

		//  Enable the plugin (means enable the routes of the plugin)
		_enablePlugin(plugin, function(err){

			// Ensure to move the '/*' route to the end of the list of routes
			expressAppUtils.moveDefaultAppRoute();

			if(err){
				logger.error('--->enable: Req. id=' + res._Context.reqId + ', Plugin enable request failed. Reason:' + err.stack);
				return res.status(404).end('Plugin enable request failed. Reason:' + err.stack);
			}
			if(logger.isDebugEnabled()){
				_logAppRoutes();
			}
			logger.info('--->enable: Req. id=' + res._Context.reqId + ', over');			
			return res.end('done');
		});
	});
}

/**
 * Load the given plugin.
 * @param  {req} 		req as GET or POST 
 * 		   GET URL: 	<server>[:port]/appservices/pm/load?module=oxsnps-afp2any
 * 		   POST URL: 	<server>[:port]/appservices/pm/load
 * 		   POST parms:	req.body = {
 * 		            		"module":"oxsnps-afp2any", 
 * 		              	}
 * @param  {res} 	res
 */
exports.loadPlugin = function(req, res){
    
	var dirs = []
	  , plugindir = npsDir + '/node_modules/' // default plugin dir
	  , groups = []
      , bPluginFound = false;

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);

	// Add the plugin name to the Context
	if(req.method === 'GET') res._Context.module = req.query.module;
	else res._Context.module = req.body.module;

	logger.info('--->load: Req. id=' + res._Context.reqId);
	logger.debug('--->load: Req. id=' + res._Context.reqId + ', Parms: ' + (req.method === 'GET'?JSON.stringify(req.query):JSON.stringify(req.body)));

	// Is the Plugin already loaded?
	_getPlugin(res._Context.module, function(err, plugin){
		if(!err){
			logger.info('--->loadPlugin: Req. id=' + res._Context.reqId + ', Plugin already enabled.');
			return res.end('Plugin already enabled.');
		}

		// If the plugin has not been loaded, load it
		// Get the list of the subdirs contained in the npsConf.pluginDirs array
		if(npsConf.pluginDirs){
		    if(npsConf.pluginDirs instanceof Array){
		        for (var i = 0; i< npsConf.pluginDirs.length; i++){
		            utils.listDirsSync(path.resolve(npsDir + '/' + npsConf.pluginDirs[i]), dirs, true);
		        };   
		    }
		    else{   
		        // Not an array, so it must be one dir only
		        plugindir = path.resolve(npsDir + '/' + npsConf.pluginDirs);
		        utils.listDirsSync(plugindir, dirs, true);
		    }
		}
		else{
		    // Use the default plugin dir
		    utils.listDirsSync(plugindir, dirs, true);
		}

		// Loop through all the dirs
		async.each(dirs,
			function(dir, nextTask){

				// Is it the dir of the plugin?
			    groups = dir.name.match(/(.*\/)(.*)/);
				if(groups[2] === res._Context.module){
					if(fs.existsSync(dir.name + '/' + pluginJSfilename)){ // dir.name is the FQ name of the plugin
						bPluginFound = true;
						// Load and initialize the plugin
						_loadPlugin(dir.name, nextTask);
					}
					else{
						nextTask();
					}
				}
				else{
					nextTask();
				}
			},
			function(err){

				// Ensure to move the '/*' route to the end of the list of routes
				expressAppUtils.moveDefaultAppRoute();

				if(err){
					logger.error('--->loadPlugin: Req. id=' + res._Context.reqId + ', Loading Plugin ' + res._Context.module + ' failed, Reason:' + err.stack);
					return res.status(404).end('Loading Plugin ' + res._Context.module + ' failed, Reason:' + err.stack);
				}
				if(bPluginFound){
					return res.end('Plugin ' + res._Context.module + ' loaded.');
				}
				return res.status(404).end('Plugin ' + res._Context.module + ' not found!');
			}
		);
	});
}

/**
 * Disable the given plugin
 * @param  {req} 		req as GET or POST 
 * 		   GET URL: 	<server>[:port]/appservices/pm/disable?module=oxsnps-afp2any
 * 		   POST URL: 	<server>[:port]/appservices/pm/disable
 * 		   POST parms:	req.body = {
 * 		            		"module":"oxsnps-afp2any", 
 * 		              	}
 * @param  {res} 		res
 */
exports.disablePlugin = function(req, res){

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);

	// Add the plugin name to the Context
	if(req.method === 'GET') res._Context.module = req.query.module;
	else res._Context.module = req.body.module;

	logger.info('--->disable: Req. id=' + res._Context.reqId);
	logger.debug('--->disable: Req. id=' + res._Context.reqId + ', Parms: ' + (req.method === 'GET'?JSON.stringify(req.query):JSON.stringify(req.body)));

	// Get Plugin
	_getPlugin(res._Context.module, function(err, plugin){
		if(err){
			logger.error('--->disablePlugin: Req. id=' + res._Context.reqId + ', Plugin disable request failed: ' + err.stack + '. Parms:' + JSON.stringify(req.body));
			return res.status(404).end('Plugin disable request failed: ' + err.stack + '. Parms:' + JSON.stringify(req.body));
		}

		// If Plugin is already disabled, leave
		if(plugin.enable && plugin.enable !== 'on'){
			return res.end('Plugin ' + res._Context.module + ' already disabled');
		}

		// Disable the plugin
		_disablePlugin(plugin, function(err){
			if(err){
				logger.error('--->disable: Req. id=' + res._Context.reqId + ', Plugin disable request failed. ' + err.stack);
				return res.status(404).end('Plugin disable request failed. ' + err.stack);
			}
			if(logger.isDebugEnabled()){
				_logAppRoutes();
			}
			logger.info('--->disable: Req. id=' + res._Context.reqId + ', over');
			return res.end('done');
		});
	});
}

/**
 * Enable the given route
 * @param  {req} 		req as GET or POST 
 * 		   GET URL: 	<server>[:port]/appservices/pm/enableroute?module=oxsnps-afp2any&route={...}
 * 		   POST URL: 	<server>[:port]/appservices/pm/enableroute
 * 		   POST parms:	req.body = {
 * 		            		"module":"oxsnps-afp2any", 
 * 		            	 	"route": { 
 * 		            		 	"path": "/path/to/the/route",
 * 		            		 	"method": <Optional. Values are 'get'|'post'|'dw'|'queue'>. 
 * 		            		 			  No default. If given method+path will be used to find the route 
 * 		            		 			  else path alone is used to find the route.
 * 		            		 			  This is useful since same path (especially HTTP Paths) can be used 
 * 		            		 			  with different methods(get|post)
 * 		            		  	...
 * 		            	 	}
 * 		              	}
 * @param  {res}		res
 */
exports.enableRoute = function(req, res){

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);

	req.body = req.body || {};
	
	// Add Get Parameters to req.body
	if(req.method === 'GET'){
		req.body.module = req.query.module;
		req.body.route = utils.parseJSON(req.query.route);
	}

	res._Context.enableRouteReq = {
		'body':{
			'module': req.body.module,
			'route' : req.body.route
		}
	};

	exports.enableRouteService(res, function(err){
		if(err && err instanceof warn) return res.end(err.message);
		else if(err) return res.status(404).end('Unable to enable a Route. Req. id=' + 
											res._Context.reqId + ', ' + err.message);
		res.end('Plugin ' + req.body.module + ', ' + (req.body.route.method || 'get') + ' route ' + 
					req.body.route.path + ' has been enabled');
	});
}

/**
 * Enable the given route
 * @param  {res} 		Response Object
 * 		   POST parms:	req._Context = {
 * 		   					"enableRouteReq"."body" : {
 * 		            			"module": "oxsnps-afp2any", 
 * 		            	 		"route": { 
 * 	                 	           		 	"path": "/path/to/the/route",
 * 		                                 	"method": <Optional. Values are 'get'|'post'|'dw'|'queue'>. 
 * 		            		 	                      No default. If given method+path will be used to find the route 
 * 		            		 	                      else path alone is used to find the route.
 * 		            		 			              This is useful since same path (especially HTTP Paths) can be used 
 * 		            		 			              with different methods(get|post)
 * 		            		  	             ...
 * 		            	 	             }
 * 		              	        }
 * 		              	}
 * @param  {Function} 	callback(err)
 */
exports.enableRouteService = function(res, callback){

	var fqPlugin 	= ''
	  , req 		= undefined
	  , msg 		= undefined
	  , module 		= undefined
	  ;

	// Add Context to the response instance
	res._Context = res._Context || {};

	// Add the Current Date to the Context
	res._Context.date = res._Context.date || new Date();

	// Add a Request Id to the Context
	res._Context.reqId = res._Context.reqId || utils.buildReqId(res._Context.date);

	req = res._Context.enableRouteReq || {};

	logger.info('--->enableRouteService: Req. id=' + res._Context.reqId + ', Parms:' + JSON.stringify(req.body));

	// 1. Assert Parms
	if(req.body === undefined || req.body.module === undefined || req.body.route === undefined){
		logger.error('--->enableRouteService: Req. id=' + res._Context.reqId + 
					 ', Wrong request: Parms:' + JSON.stringify(req));
		return callback(new Error('Wrong request: Parms:' + JSON.stringify(req)));
	}

	// 1a. Check if req.body.module is in the plugins list and enabled. If not, leave 
	_getPlugin(req.body.module, function(err, plugin){
		if(err){
			logger.error('--->enableRouteService: Req. id=' + res._Context.reqId + ', Request failed: ' + err.message + ' Parms:' + JSON.stringify(req.body));
			return callback(new Error(err.message + ', Parms:' + JSON.stringify(req.body)));
		}

		// If plugin is not enabled
		if(plugin.enable && plugin.enable !== 'on'){
			msg = req.body.module + ' is not enabled. Parms:' + JSON.stringify(req.body);
			logger.error('--->enableRouteService: Req. id=' + res._Context.reqId + ', Error: ' + msg);
			return callback(new Error(msg));
		}

		// Add the plugin to the Context
		res._Context.plugin = plugin;

		// 1b. Fetch whether req.body.route.path in the plugins array
		for (var i = res._Context.plugin.routes.length - 1; i >= 0; i--) {
			// If optional method parameter is given, use it to find the route with that method
			// This is useful since same path (especially HTTP Paths) can be used with different methods(get|post)
			if(req.body.route.method && req.body.route.method.length > 0){
				if(res._Context.plugin.routes[i] !== undefined && 
					res._Context.plugin.routes[i].method !== undefined && 
					res._Context.plugin.routes[i].method === req.body.route.method &&
					res._Context.plugin.routes[i].path !== undefined && 
					res._Context.plugin.routes[i].path === req.body.route.path){
						res._Context.route = res._Context.plugin.routes[i]; // add the route to the Context
						req.body.route = res._Context.plugin.routes[i]; 	//!!! add the route to the req
						break;
				}
			}
			else if(res._Context.plugin.routes[i] !== undefined && 
			   res._Context.plugin.routes[i].path !== undefined && 
			   res._Context.plugin.routes[i].path === req.body.route.path){
					res._Context.route = res._Context.plugin.routes[i]; // add the route to the Context
					req.body.route = res._Context.plugin.routes[i]; 	//!!! add the route to the req
					break;
			}
		};

		// If route does not exist, leave
		if(!res._Context.route){
			logger.warn('--->enableRouteService: Req. id=' + res._Context.reqId + 
						', Plugin ' + req.body.module + ', route ' + req.body.route.path + ' does not exist');
			return callback(new warn('Plugin ' + req.body.module + ', route ' + req.body.route.path + ' does not exist'));
		}

		// If route is already enabled, leave
		if(res._Context.route.enable === 'on'){
			logger.warn('--->enableRouteService: Req. id=' + res._Context.reqId + 
						', Plugin ' + req.body.module + ', ' + (req.body.route.method || 'get') + ' route ' + req.body.route.path + ' already enabled');
			return callback(new warn('Plugin ' + req.body.module + ', ' + (req.body.route.method || 'get') + ' route ' + req.body.route.path + ' already enabled'));
		}

		// Get the module handle
		try{
			fqPlugin = plugin.plugindir + plugin.module + '/' + pluginJSfilename;
			module = require(fqPlugin);
			if(!module){
				logger.error('--->enableRouteService: Req. id=' + res._Context.reqId + 
							 ', Error loading Plugin ' + fqPlugin);
				return callback(new Error('Error loading Plugin ' + fqPlugin + ', Parms:' + JSON.stringify(req.body)));				
			}
		}catch(err){
			logger.error('--->enableRouteService: Req. id=' + res._Context.reqId + 
						 ', Error loading Plugin ' + fqPlugin + '. Reason: ' + err.message);
			return callback(new Error('Error loading Plugin ' + fqPlugin + '. Reason: ' + err.message + ', Parms:' + JSON.stringify(req.body)));
		}

		// Call enableRoute of the plugin if it exists
		if (typeof(module.enableRoute) === 'function'){
			logger.debug('--->enableRouteService: Req. id=' + res._Context.reqId + 
						 ', Calling ' + req.body.module + '.enableRoute...');
			// route.enable must be on to enable the route in oxsnps-dirwatcher and oxsnps-ibmmq
			res._Context.route.enable = 'on';
			module.enableRoute(req, function(err, route){
				if(err){
					// route.enable to off since we are unable to enable the route
					res._Context.route.enable = 'off';
					msg = 'Module=' + req.body.module +
					 	  ', Route=' + req.body.route.path +
					 	  ', Method=' + req.body.route.method +
					 	  ', Reason: ' + err.message;
					logger.error('--->enableRouteService: Req. id=' + res._Context.reqId + 
					 			 ', Unable to enable a Route.' + err.message);
					return callback(new Error(msg));
				}
				logger.info('--->enableRouteService: Req. id=' + res._Context.reqId + 
							' Plugin ' + req.body.module + ', ' + (req.body.route.method || 'get') + ' route ' + req.body.route.path + ' has been enabled');
				return callback();
			});
		}
		else{
			logger.warn('--->enableRouteService: Req. id=' + res._Context.reqId + 
						', Plugin ' + req.body.module + '.enableRoute not available.');
			return callback(new warn('Plugin ' + req.body.module + '.enableRoute() not available.'));	
		}
	});
}

/**
 * Disable the given route
 * @param  {req} 		req as GET or POST 
 * 		   GET URL: 	<server>[:port]/appservices/pm/disableroute?module=oxsnps-afp2any&route={...}
 * 		   POST URL: 	<server>[:port]/appservices/pm/disableroute
 * 		   POST parms:	req.body = {
 * 		            		"module":"oxsnps-afp2any", 
 * 		            	 	"route": { 
 * 		            		 	"path": "/path/to/the/route",
 * 		            		 	"method": <Optional. Values are 'get'|'post'|'dw'|'queue'>. 
 * 		            		 			  No default. If given method+path will be used to find the route 
 * 		            		 			  else path alone is used to find the route.
 * 		            		 			  This is useful since same path (especially HTTP Paths) can be used 
 * 		            		 			  with different methods(get|post)
 * 		            		  	...
 * 		            	 	}
 * 		              	}
 * @param  {res}		res
 */
exports.disableRoute = function(req, res){

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);

	req.body = req.body || {};

	// Add Get Parameters to req.body
	if(req.method === 'GET'){
		req.body.module = req.query.module;
		req.body.route = utils.parseJSON(req.query.route);
	}

	res._Context.disableRouteReq = {
		'body':{
			'module': req.body.module,
			'route' : req.body.route
		}
	};

	exports.disableRouteService(res, function(err){
		if(err && err instanceof warn) return res.end(err.message);
		else if(err)
			return res.status(404).end('Unable to disable a Route. Req. id=' + res._Context.reqId + ', ' + err.message);
		res.end('Plugin ' + req.body.module + ', ' + (req.body.route.method || 'get') + ' route ' + 
				req.body.route.path + ' has been disabled');
	});
}

/**
 * Disable the given route
 * @param  {res} 		Response Object
 * 		   POST parms:	req._Context = {
 * 		   					"disableRouteReq"."body" : {
 * 		            			"module": "oxsnps-afp2any", 
 * 		            	 		"route": { 
 * 	                 	           		 	"path": "/path/to/the/route",
 * 		                                 	"method": <Optional. Values are 'get'|'post'|'dw'|'queue'>. 
 * 		            		 	                      No default. If given method+path will be used to find the route 
 * 		            		 	                      else path alone is used to find the route.
 * 		            		 			              This is useful since same path (especially HTTP Paths) can be used 
 * 		            		 			              with different methods(get|post)
 * 		            		  	             ...
 * 		            	 	             }
 * 		              	        }
 * 		              	}
 * @param  {Function} 	callback(err)
 */
exports.disableRouteService = function(res, callback){

	var fqPlugin 	= ''
	  , req 		= undefined
	  , msg 		= undefined
	  , module 		= undefined
	  ;

	// Add Context to the response instance
	res._Context = res._Context || {};

	// Add the Current Date to the Context
	res._Context.date = res._Context.date || new Date();

	// Add a Request Id to the Context
	res._Context.reqId = res._Context.reqId || utils.buildReqId(res._Context.date);

	req = res._Context.disableRouteReq || {};

	logger.info('--->disableRouteService: Req. id=' + res._Context.reqId + ', Parms:' + JSON.stringify(req.body));

	// 1. Assert Parms
	if(req.body === undefined || req.body.module === undefined || req.body.route === undefined){
		logger.error('--->disableRouteService: Req. id=' + res._Context.reqId + 
					 ', Wrong request: Parms:' + JSON.stringify(req));
		return callback(new Error('Wrong request: Parms:' + JSON.stringify(req)));
	}

	// 1a. Check if req.body.module is in the plugins list and enabled. If not, leave 
	_getPlugin(req.body.module, function(err, plugin){
		if(err){
			logger.error('--->disableRouteService: Req. id=' + res._Context.reqId + ', Request failed: ' + err.message + ' Parms:' + JSON.stringify(req.body));
			return callback(new Error(err.message + ', Parms:' + JSON.stringify(req.body)));
		}

		// If plugin is not enabled
		if(plugin.enable && plugin.enable !== 'on'){
			msg = req.body.module + ' is not enabled. Parms:' + JSON.stringify(req.body);
			logger.error('--->disableRouteService: Req. id=' + res._Context.reqId + ', Error: ' + msg);
			return callback(new Error(msg));
		}

		// Add the plugin to the Context
		res._Context.plugin = plugin;

		// 1b. Fetch whether req.body.route.path in the plugins array
		for (var i = res._Context.plugin.routes.length - 1; i >= 0; i--){
			// If optional method parameter is given use it to find the route with that method
			// This is useful since same path (especially HTTP Paths) can be used with different methods(get|post)
			if(req.body.route.method && req.body.route.method.length > 0){
				if(res._Context.plugin.routes[i] !== undefined && 
					res._Context.plugin.routes[i].method !== undefined && 
					res._Context.plugin.routes[i].method === req.body.route.method &&
					res._Context.plugin.routes[i].path !== undefined && 
					res._Context.plugin.routes[i].path === req.body.route.path){
						res._Context.route = res._Context.plugin.routes[i]; // add the route to the Context
						req.body.route = res._Context.plugin.routes[i]; 	//!!! add the route to the req
						break;
				}
			}			
			else if(res._Context.plugin.routes[i] !== undefined && 
			   res._Context.plugin.routes[i].path !== undefined && 
			   res._Context.plugin.routes[i].path === req.body.route.path){
				res._Context.route = res._Context.plugin.routes[i]; // add the route to the Context
				req.body.route = res._Context.plugin.routes[i]; 	//!!! add the route to the req
				break;
			}
		};
	
		// If route does not exist, leave
		if(!res._Context.route){
			logger.warn('--->disableRouteService: Req. id=' + res._Context.reqId + 
						', Plugin ' + req.body.module + ', route ' + req.body.route.path + ' does not exist');
			return callback(new warn('Plugin ' + req.body.module + ', route ' + req.body.route.path + ' does not exist'));
		}

		// If route is already disabled, leave
		if(res._Context.route.enable !== 'on'){
			logger.warn('--->disableRouteService: Req. id=' + res._Context.reqId + 
						', Plugin ' + req.body.module + ', ' + (req.body.route.method || 'get') + ' route ' + req.body.route.path + ' already disabled');
			return callback(new warn('Plugin ' + req.body.module + ', ' + (req.body.route.method || 'get') + ' route ' + req.body.route.path + ' already disabled'));
		}

		// Get the module handle
		try{
			fqPlugin = plugin.plugindir + plugin.module + '/' + pluginJSfilename;
			module = require(fqPlugin);
			if(!module){
				logger.error('--->disableRouteService: Req. id=' + res._Context.reqId + 
							 ', Error loading Plugin ' + fqPlugin);
				return callback(new Error('Error loading Plugin ' + fqPlugin + ', Parms:' + JSON.stringify(req.body)));
			}
		}catch(err){
			logger.error('--->disableRouteService: Req. id=' + res._Context.reqId + 
						 ', Error loading Plugin ' + fqPlugin + '. Reason: ' + err.message);
			return callback(new Error('Error loading Plugin ' + fqPlugin + '. Reason: ' + err.message + ', Parms:' + JSON.stringify(req.body)));
		}

		// Call disableRoute of the plugin if it exists
		if(typeof(module.disableRoute) === 'function'){
			logger.debug('--->disableRouteService: Req. id=' + res._Context.reqId + 
						 ', Calling ' + req.body.module + '.disableRoute...');
			module.disableRoute(req, function(err, route){
				if(err){
					msg = 'Module=' + req.body.module +
					 	  ', Route=' + req.body.route.path +
					 	  ', Method=' + req.body.route.method +
					 	  ', Reason: ' + err.message;
					logger.error('--->disableRoute: Req. id=' + res._Context.reqId + 
					 			 ', Unable to disable a Route.' + err.message);
					return callback(new Error(msg));
				}
				res._Context.route.enable = 'off';
				logger.info('--->disableRouteService: Req. id=' + res._Context.reqId + 
							' Plugin ' + req.body.module + ', ' + (req.body.route.method || 'get') + ' route ' + req.body.route.path + ', has been disabled');
				return callback();				
			});
		}
		else{
			logger.warn('--->disableRouteService: Req. id=' + res._Context.reqId + 
						', Plugin ' + req.body.module + '.disableRoute not available.');
			return callback(new warn('Plugin ' + req.body.module + '.disableRoute not available.'));
		}
	});
}

/**
 * Return the list of the plugin routes
 * @param  {req} 	GET req
 * @param  {res}	res
 */
exports.getRoutes = function(req, res){

	logger.info('--->getRoutes: Getting the list of the Plugins and their Routes...');

	// Assert plugins array
	if(plugins){
		res.json(plugins);
	}
	else{
		res.json([]); // return an empty array
	}
	logger.info('--->getRoutes: List of the Plugins and their Routes returned, over.');
}

/**
 * Return the log level of the given plugin
 * @param  {req} 	GET req   http://localhost:1026/appservices/pm/getloglevel?module=oxsnps-afp2any
 * @param  {res}	res
 */
exports.getLogLevel = function(req, res){

	logger.info('--->getLogLevel: Getting the log level of a plugin or of the Plugin Manager...');

	//!!! Must be a better way to get parms from a HTTP GET req.
	if(req.query.module === undefined){
		return res.status(404).end('Module name is missing. API call: http://<server>[:<port>/appservices/pm/getloglevel?module=<moduleName>');
	}

	var moduleName 	= req.query.module;

	// Assert parms
	if(!moduleName){
		return res.status(404).end('Missing module, Parms: ' + JSON.stringify(req.query));
	}

	// If the module is Plugin Manager
	if(moduleName.toLowerCase() === 'pm' || moduleName.toLowerCase() === 'plugin manager'){
		return res.end(logger.level.levelStr);
	}
	else{	
		// Get the handle of the given module if enabled
		exports.getModule(moduleName, function(err, module){
	    	if(err) return res.status(404).end(''+err);

			// Call getLogLevel service of the plugin if it exists
			if (typeof(module.getLogLevel) === 'function'){

				// Try to call getloglevel
				try{
			    	return res.end(module.getLogLevel());
					logger.info('--->getLogLevel: Log level of ' + moduleName + ' is ' + loglevel);
				}catch(err){
					return res.status(404).end('Calling ' + moduleName + ' .getloglevel() failed.' + err.stack);
				}
			}
			else{
				logger.warn('--->getLogLevel: ' + moduleName + '.getloglevel not available.');
				return res.status(404).end('Plugin ' + moduleName + '.getloglevel not available.');
			}
		});
	}
}

/**
 * Set the log level of the given plugin
 * @param  {req} 	GET req   http://localhost:1026/appservices/pm/setloglevel?module=oxsnps-afp2any&loglevel=DEBUG|INFO|ERROR
 * @param  {res}	res
 */
exports.setLogLevel = function(req, res){

	logger.info('--->setLogLevel: Setting the log level of a plugin...');

	//!!! Must be a better way to get parms from a HTTP GET req.
	if(req.query.module === undefined){
		return res.status(404).end('Module name is missing. API call: http://<server>[:<port>/appservices/pm/setloglevel?module=<moduleName>loglevel=DEBUG|INFO|ERROR');
	}

	var moduleName 	= req.query.module
	  , loglevel 	= req.query.loglevel || 'INFO'
	  ;

	// Assert parms
	if(!moduleName || !loglevel ){
		return res.status(404).end('Missing module or logLevel, Parms: ' + JSON.stringify(req.query));
	}

	// If the module is Plugin Manager
	if(moduleName.toLowerCase() === 'pm' || moduleName.toLowerCase() === 'plugin manager'){
		logger.setLevel(loglevel);
		return res.end(loglevel);
	}
	else{	
		// Get the handle of the given module if enabled
		exports.getModule(moduleName, function(err, module){
	    	if(err) return res.status(404).end(''+err);

			// Call setLogLevel service of the plugin if it exists
			if (typeof(module.setLogLevel) === 'function'){

				// Try to call setLoglevel
				try{
					module.setLogLevel(loglevel);
					// Save plugin Configuration file
					if (module.saveConfService && typeof(module.saveConfService) === 'function'){
						module.saveConfService(function(err){
							if(err)
								return res.end('--->setLogLevel: Unable to set log level to ' + 
										 loglevel + ', ' + err.message);
							res.end('--->setLogLevel: Log level of ' + moduleName + ' set to ' + loglevel);
						});
					}
					else return res.end('--->setLogLevel: Log level of ' + moduleName + ' set to ' + loglevel);
				}catch(err){
					return res.status(404).end('Calling ' + moduleName + ' .setLoglevel() failed.' + err.stack);
				}
			}
			else{
				logger.warn('--->setLogLevel: ' + moduleName + '.setLoglevel not available.');
				return res.status(404).end('Plugin ' + moduleName + '.setLoglevel not available.');
			}
		});
	}
}

/**
 * Enable the plugins found in the dirs listed in npsConf.pluginDirs
 * @param  {function}   callback(err)
 *
 */
exports.enablePlugins = function(callback){
 	
	logger.info('-->Enabling Plugins...');


	// Search the plugin dir for plugins
	var dirs = []
	  , plugindir = npsDir + '/node_modules/' // default plugin dir
	  ;

	// Get the list of the subdirs contained in the npsConf.pluginDirs array
	if(npsConf.pluginDirs){
	    if(npsConf.pluginDirs instanceof Array){
	        for (var i = 0; i< npsConf.pluginDirs.length; i++){
	            utils.listDirsSync(path.resolve(npsDir + '/' + npsConf.pluginDirs[i]), dirs, true);
	        };   
	    }
	    else{   
	        // Not an array, so it must be one dir only
	        plugindir = path.resolve(npsDir + '/' + npsConf.pluginDirs);
	        utils.listDirsSync(plugindir, dirs, true);
	    }
	}
	else{
	    // Use the default plugin dir
	    utils.listDirsSync(plugindir, dirs, true);
	}

	// Loop through all the dirs
	async.each(dirs,
		function(dir, nextTask){
			if(fs.existsSync(dir.name + '/' + pluginJSfilename)){ // dir.name is the FQ name of the plugin
				// Load and initialize the plugin
				_loadPlugin(dir.name, nextTask);
			}
			else{
				nextTask();
			}
		},
		function(err){
			// Add the '/*' route to the end of the list of routes to map undefined routes
			expressApp['get']('/*', expressAppUtils.routeNotAvailable);
			expressApp['post']('/*', expressAppUtils.routeNotAvailable);			

			if(logger.isDebugEnabled()){
				_logPlugins();
				_logAppRoutes();
			}
			if(err){
				logger.error('--->enablePlugins: Enable Plugins failed: ' + err.stack);
			}
			callback(err);
		}
	);
}

exports.activatePlugins = function(callback){
 	
	logger.info('-->Activating Plugins...');

	// Loop through all plugins
	async.each(plugins,
		function(plugin, nextTask){

			if(plugin.enable !== 'on') return nextTask(); // do nothing if plugin is disabled

			// Activate the plugin
			_activatePlugin(plugin, nextTask);
		},
		function(err){
			if(err) logger.error('--->activatePlugins: Activating Plugins failed. ' + err.stack);
			callback(err);
		}
	);
}

/**
 * Disable all plugins
 * @param  {function} 	callback(err)
 */
exports.disablePlugins = function(callback){
	
	var module = null;

	logger.info('-->Disabling Plugins...');

	// Loop through all plugins
	async.each(plugins,
		function(plugin, nextTask){

			if(plugin.enable !== 'on'){
				nextTask();
			}
			else{ // Disable the plugin
				_disablePlugin(plugin, nextTask);
			}
		},
		function(err){
			if(err){
				logger.error('--->disablePlugins: Disable Plugins failed. ' + err.stack);
			}
			callback(err);
		}
	);
}

/**
 * isPluginEnabled: 	Checks if plugin is enabled or not.
 * @param  {string}  	pluginName Name of the plugin
 * @return {boolean}    Returns true if plugin is enabled, else false.
 */
exports.isPluginEnabled = function(pluginName){

	if(pluginName === undefined || pluginName === null){
		return false;
	}	

	// Check if the plugin is in the list and enabled
	for (var i = plugins.length - 1; i >= 0; i--) {
		if(plugin[i].module === pluginName && plugin[i].enable === 'on'){
			return true;
		}
	};
	return false;
}

/**
 * Returns the list of plugins
 * @return {json} 	plugins List of plugins
 */
exports.getPluginsList = function(){
	return plugins;
}

/**
 * Returns the handle of the requested plugin if enabled
 * @param  {string} 	pluginName Name of the plugin
 * @param  {function}   callback(err, module)
 */
exports.getModule = function(pluginName, callback){

	var module = null
	  , fqPlugin = ''
	  ;
	
	// Get the plugin info
	_getPlugin(pluginName, function(err, plugin){
		if(err){
			return callback(err);
		}
		// Check if the plugin is enabled
		if(plugin.enable === 'on'){
			fqPlugin = plugin.plugindir + plugin.module + '/' + pluginJSfilename;
			// Get the module handle
			try{
				module = require(fqPlugin);
				if(!module){
					return callback(new Error('Error loading Plugin ' + fqPlugin));
				}
				return callback(null, module);
			}catch(err){
				return callback(new Error('Error loading Plugin ' + fqPlugin + '. Reason: ' + err.stack));
			}
		}
		else{
			return callback(new Error('Plugin ' + pluginName + ' not enabled.'));
		}
	});
}

/******************  Private Functions ********************/
/**
 * _loadPlugin: 		Load and initialize a plugin
 * @param  {string}   	plugin 		the FQ plugin directory
 * @param  {function}   callback(err)
 */
function _loadPlugin(plugin, callback){
	
	var module = null
	  , fqPlugin = plugin + '/' + pluginJSfilename
	  , aPlugin = {}	// the new plugin to be added to the list of plugins
	  ;

  	// Assert fqPlugin
	if(!fqPlugin){
		return callback(new Error('Plugin name is null'));
	}

	logger.debug('_loadPlugin: Enabling ' + fqPlugin + ' ...');

	// Load the plugin
	try{
		// Get the module handle
		module = require(fqPlugin);
		if(!module){
			return callback(new Error('Error loading Plugin ' + fqPlugin));
		}
	}catch(err){
		return callback(new Error('Error loading Plugin ' + fqPlugin + '. Reason: ' + err.stack));
	}
	// Set the plugin props
	/*
		Structure of the plugins array
	    "plugins":[
			{
				"module": 		"Plugin Name (ex.: afp2any)",
				"plugindir": 	"FQ name of the the plugin root directory (ex.: /path/to/pluginDir/)",
				"enable": 		"on|off",
				"desc": 		"Short description of the plugin",
				"longDesc": 	"Long description of the plugin",
				"logLevel":		"DEBUG|INFO|ERROR",
				"routes": []
			},
			{...}
		]
	*/
    var groups = plugin.match(/(.*\/)(.*)/);
	aPlugin.module = groups[2]; 	// Get the last dir as default plugin name
	aPlugin.plugindir = groups[1]; 	// Get the path up to last dir (i.e the plugin root dir)

	if(module.pluginConf === undefined){
		aPlugin.enable = 'off';
		aPlugin.desc = fqPlugin;
		aPlugin.longDesc = fqPlugin;
		aPlugin.logLevel = 'INFO';
		aPlugin.version  = 	'Unknown';
	}
	else{
		//!!!aPlugin.module 	 = 	module.pluginConf.module 	|| groups[2]; //!!! Get the last dir (i.e plugin name)
		aPlugin.enable 	 = 	module.pluginConf.enable 	|| 'off';
		aPlugin.desc 	 =	module.pluginConf.desc 		|| fqPlugin;
		aPlugin.longDesc = 	module.pluginConf.longDesc 	|| fqPlugin;
		aPlugin.logLevel = 	module.pluginConf.logLevel 	||'INFO';
		aPlugin.version  = 	module.pluginConf.version 	|| 'Unknown';
	}

	// Initialize the plugin if the plugin is enabled
	if(aPlugin.enable === 'on'){
		_intializePlugin(module, fqPlugin, function(err, routes){
			if (err){
				aPlugin.enable = 'off'; // reset enable prop
				return callback(new Error('Could not intialize Plugin ' + fqPlugin + '. ' + err.stack));
			}
			aPlugin.routes = routes;	// add the routes to the plugin
			plugins.push(aPlugin);		// add the plugin to the list of plugins
			return callback();
		});
	}
	else{
		aPlugin.routes = [];		// add the routes to the plugin
		plugins.push(aPlugin);		// add the plugin to the list of plugins
		return callback();
	}
}

/**
 * _enablePlugin: 		Enable a plugin which is already in the list of plugins (has been once before enabled)
 * @param  {string}     plugindir The plugin directory
 * @param  {function}   callback(err)
 */
function _enablePlugin(plugin, callback){
	
	var module = null
	  , fqPlugin = null
	  ;

	// Assert plugin parameter
	if(!plugin && !plugin.plugindir && !plugin.module){
		return callback(new Error('Plugin configuration parameter is null'));
	}
	fqPlugin = plugin.plugindir + plugin.module + '/' + pluginJSfilename;

	logger.debug('_enablePlugin: Enabling ' + fqPlugin + ' ...');

	// Get the module handle
	try{
		module = require(fqPlugin);
		if(!module){
			return callback(new Error('Error loading Plugin ' + fqPlugin));
		}
	}catch(err){
		return callback(new Error('Error loading Plugin ' + fqPlugin + '. Reason: ' + err.stack));
	}

	// Initialize the plugin
	_intializePlugin(module, fqPlugin, function(err, routes){
		if (err) return callback(new Error('Could not intialize Plugin ' + fqPlugin + '. ' + err.stack));

		// Activate the plugin
		_activatePlugin(plugin, function(err){
			if (err) return callback(new Error('Could not activate Plugin ' + fqPlugin + '. ' + err.stack));

			// Add the list of the plugin routes to the list of the plugins array
			plugin.enable = 'on';
			module.pluginConf.enable = 'on';
			plugin.routes = routes;
			// Save the plugin configuration file
			if (module.saveConfService && typeof(module.saveConfService) === 'function')
				module.saveConfService(callback);
			else callback();
		});
	});
}

/**
 * _disablePlugin: 		Disable a plugin
 * @param  {JSON}       plugin    A JSON object that contains plugin information.
 * @param  {function}   callback(err)
 */
function _disablePlugin(plugin, callback){

	var name = null
	  , fqPlugin = null
	  ;

	// Assert plugin parameter
	if(!plugin || !plugin.plugindir || !plugin.module){
		return callback(new Error('Plugin configuration parameter is null'));
	}
	fqPlugin = plugin.plugindir + plugin.module + '/' + pluginJSfilename;

	logger.debug('_disablePlugin: Disabling ' + fqPlugin + '  ...');

	// Get the module handle
	try{
		module = require(fqPlugin);
		if(!module){
			return callback(new Error('Error loading Plugin ' + fqPlugin));
		}
	}catch(err){
		return callback(new Error('Error loading Plugin ' + fqPlugin + '. Reason: ' + err.stack));
	}

	// Finalize the plugin
	_finalizePlugin(module, fqPlugin, function(err){
		if (err){
			return callback(new Error('Could not finalize Plugin ' + fqPlugin + '. ' + err.stack));
		}
		// Add the list of the plugin routes to the list of the plugins array
		plugin.enable = 'off';
		module.pluginConf.enable = 'off';
		plugin.routes = []; // no more route available
		// Save the plugin configuration file		
		if (module.saveConfService && typeof(module.saveConfService) === 'function')
			module.saveConfService(callback);
		else callback();
	});
}

/**
 * _activatePlugin: 	Activate a plugin
 * @param  {JSON}       plugin    A JSON object that contains plugin information.
 * @param  {function}   callback(err)
 */
function _activatePlugin(plugin, callback){

	var name = null
	  , fqPlugin = null
	  ;

	// Assert plugin parameter
	if(!plugin || !plugin.plugindir || !plugin.module){
		return callback(new Error('Plugin configuration parameter is null'));
	}
	fqPlugin = plugin.plugindir + plugin.module + '/' + pluginJSfilename;

	logger.debug('_activatePlugin: Activating ' + fqPlugin + '  ...');

	// Get the module handle
	try{
		module = require(fqPlugin);
		if(!module){
			return callback(new Error('Error loading Plugin ' + fqPlugin));
		}
	}catch(err){
		return callback(new Error('Error loading Plugin ' + fqPlugin + '. Reason: ' + err.stack));
	}

	// Call activate service of the plugin if it exists
	if (typeof(module.postInitialize) === 'function'){

		// Try to call activate
		try{
			module.postInitialize(function(err){
				if(err) return callback(new Error('Could not activate Plugin ' + fqPlugin + '. ' + err.stack));		
				callback();
			});
		}catch(err){
			callback('Calling ' + plugin.module + ' .postInitialize() failed.' + err.stack);
		}
	}
	else{
		logger.warn('--->_activatePlugin: ' + plugin.module + '.postInitialize() not available.');
		callback();
	}
}
/**
 * _intializePlugin: 	Initialize a plugin
 * @param  {module}     module    	Plugin Module
 * @param  {string} 	plugindir 	Plugin Directory
 * @param  {function}   callback(err, routes)
 */
function _intializePlugin(module, plugindir, callback){

	logger.debug('_intializePlugin: Intializing Plugin ' + plugindir + ' ...');

	// Initialize the plugin
	try{
		module.initialize(function(err, routes){
			if(err){
				return callback(new Error('Initializing Plugin ' + plugindir + ' failed. Reason: ' + err.stack));
			}
			routes.forEach(function(route){
				// For Queue routes add 'path' property and set queue value for 'path'
				if(route.queue)	route.path = route.queue;
			});
			return callback(null, routes);
		});
	}catch(err){
		return callback(new Error('-->Initializing Plugin ' + plugindir + ', failed. Reason: ' + err.stack));
	}
}

/**
 * _finalizePlugin: 	Initialize a plugin
 * @param  {module}     module    	Plugin Module
 * @param  {string} 	plugindir 	Plugin Directory
 * @param  {function}   callback(err, routes)
 */
function _finalizePlugin(module, plugindir, callback){

	logger.debug('_finalizePlugin: Finalizing Plugin ' + plugindir + ' ...');

	// Finalize the plugin
	try{
		module.finalize(function(err){
			if(err){
				return callback(new Error('Finalizing Plugin ' + plugindir + ' failed. Reason: ' + err.stack));
			}
			return callback();
		});
	}catch(err){
		return callback(new Error('Finalizing Plugin ' + plugindir + ', failed. Reason: ' + err.stack));
	}
}

/**
 * _getPlugin: 			Looks in the plugins array for the passed plugin and returns its json desc. if found, return it
 * 						If not found returns error
 * @param  {string}   	pluginName 	FQ Plugin Name
 * @param  {function}   callback(err, plugin)
 */
function _getPlugin(pluginName, callback){

	var retPlugin = null;
	
	if(pluginName === undefined || pluginName === null){
		return callback(new Error('plugin name undefined or null!'));
	}	

	// Check if the plugin is in the plugins array
	async.each(plugins,
		function(plugin, nextTask){
			if(plugin.module === pluginName){
				retPlugin = plugin;	// return the plugin
			}
			nextTask();
		},
		function(err){
			if(err){
				return callback(new Error('Error when getting Plugin ' + pluginName + ', Reason: ' + err.stack));
			}
			if(retPlugin === null || retPlugin === undefined){
				return callback(new Error('Plugin ' + pluginName + ' not found.'));
			}
			return callback(null, retPlugin);
		}
	);
}

/**
 * _logPlugins: 		Logs all loaded plugins
 */
function _logPlugins(){

 	var pluginInfo = [];

	logger.debug('_logPlugins: Listing the Plugins...');

	// Assert plugins parameter
	if(plugins === undefined || plugins === null){
		logger.debug('_logPlugins: No Plugins found in ' + npsDir + '/plugins/');
		return;
	}

	// Check if the plugin is in the list and enabled
	async.each(plugins,
		function(plugin, nextTask){
			pluginInfo.push('Plugin: ' + plugin.module + (plugin.enable === 'on'? ', enabled' : ', disabled'));
			nextTask();
		},
		function(err){
			if(err){
				logger.debug(err.stack);
			}
			else{
				for (var i = pluginInfo.length - 1; i >= 0; i--) {
					logger.debug(pluginInfo[i]);
				};				
			}
		}
	);
}

/**
 * _logAppRoutes: 		Display the routes of the Express App
 */
function _logAppRoutes(){

	var stack = expressApp._router.stack;	// get the Express App router stack

	logger.debug('_logAppRoutes: Listing the Routes...');

    if(stack instanceof Array){
        stack.forEach(function(a){
            var route = a.route;
            if(route){
                route.stack.forEach(function(r){
                    var method = r.method.toUpperCase();
                    logger.debug(method + '\t' + route.path);
                })
            }
        });
    }
}

// V2.0.32 BEGIN
/**
 * Call the Service of the speficied Plugin
 * @param  {JSON}  	plugin Plugin and Service names
 *					plugin:{
 *						'name': 	<plugin name>,
 *						'service': 	<service name>
 *					}
 * @param  {JSON}  	parms Service parameters
 * @param {Function} callback(err, result)	returns the result of the called service
 * 
 */
 exports.callPluginService = function(plugin, parms, callback){
	var reqId = ''; 

	if(parms && parms._Context && parms._Context.reqId) reqId = ' Req.Id=' + parms._Context.reqId + ',';

	if(!plugin || !plugin.name || !plugin.service){
		return callback(new Error('-->callPluginService:' + reqId + ' Missing plugin.name or plugin.service'));
	}

	if(logger.isDebugEnabled()) logger.debug('-->callPluginService:' + reqId + ' Plugin: ' + plugin.name + ', Service: ' + plugin.service);

	// Get the handle of the requested plugin if enabled
	exports.getModule(plugin.name, function(err, module){
		if(err) return callback(err);

		// Try to call the requested Service from the given Plugin
		try{
			module[plugin.service](parms, callback);
		}catch(err){
			return callback(new Error('Calling ' + plugin.name + '.' + plugin.service + '() failed: ' + err.message));
		}
	});
}
// V2.0.32 END
