/**-----------------------------------------------------------------------------
 * utils.js:  Node.js module that provides utility functions
 * 
 * Author    :  AFP2web Team
 * Copyright :  (C) 2014 by Maas Holding GmbH
 * Email     :  support@oxseed.de
 * Version   :  V1.0.0
 * 
 * History
 *  V100   18.09.2014  Initial release
 *  V101   30.09.2014  getFileList refactored to handle sub directories recursively
 *  V102   16.10.2014  Added getAttrTextFromHTML() method
 *  V103   06.11.2014  Added checkAndCreateDir() and parseJSON() methods
 *  V104   07.11.2014  Added mergeJSON() method
 *  V105   13.11.2014  Added symlink() method
 *  V106   05.01.2015  Added deleteDir() methods using async methods
 *  V107   09.01.2015  Added getDocId() and getDocIdFromRegExp() functions
 *  V108   22.01.2015  Extended to have configuration to turn on/off caching
 *  V109   28.01.2015  Added getOutputJSONFilename() method
 *  V110   19.10.2015  Replace ('blueimp-md5').md5 module with 'crypto-md5' module, 
 *                     since ('blueimp-md5').md5 gives following error when used for big files(>20 KB) 
 *                    ................
 *                    /opt/test/node_modules/blueimp-md5/js/md5.js:235
 *                      return unescape(encodeURIComponent(input));
                        ^
 *                    URIError: URI malformed
 *                    at encodeURIComponent (native)
 *                    at str2rstr_utf8 (/opt/test/node_modules/blueimp-md5/js/md5.js:235:25)
 *                    ................
 *                    Note: MD5 Id will not change even if we change the module used to generate MD5 Id
 *----------------------------------------------------------------------------*/
'use strict';

var MODULE_NAME   = 'utils'
  , fs          = require('fs')
  , fse         = require('fs-extra')
  , oldMask     = process.umask(0)
  , path        = require('path')
  , date_format = require('./date_format')
  , log4js      = require('log4js')
  , cheerio     = require('cheerio')            // to parse HTML  
  , mkdirp      = require('mkdirp')  
  , async       = require('async')
  //, md5         = require('blueimp-md5').md5  
  , md5         = require('crypto-md5')
  , url         = require('url')
  , os          = require('os')
  , xml2js      = require('xml2js')         // to convert xml to json and vice versa      
  , hex2utf8    = require(path.resolve(__dirname + '../../../..//conf/hex2utf8.js'))
  , cp          = require('child_process')
  , rimraf      = require('rimraf')
  , underscore  = require('underscore')
  , mv          = require('mv')
  , CRLF_Tabs = '\n\t\t\t\t\t\t\t'
  ;

// Get Logger
var logger = log4js.getLogger(MODULE_NAME);
logger.setLevel("INFO");  // Todo: Has to add a route in core services(index.js) to change utils.loglevel

/*
  Delete dirs if their date is older than (today's date + interval)
    dirs:     array of dirs
    interval: if the dir date is older than (today's date + interval (in seconds)) then delete the dir.
*/
exports.deleteDirs = function(dirs, interval){

  //logger.debug(__filename, 'deleteDirs: ' + dirs.length + ' temp dir(s) queued to be deleted...');

  if (dirs.length <= 0){
    return; // over since array is empty
  }

  // Delete the dir and its content if the dir date is older than (today's date + interval)
  _deleteDir(dirs, new Date(), interval);
}

/**
 * Delete recursively invalid Soft links SYNC 
 * @param  {String} dir
 * @return {void}
 */
exports.deleteInvalidSoftLinksSync = function(dir){

  if(fs.existsSync(dir)){
    fs.readdirSync(dir).forEach(function(file, index){
      var curDir = dir + "/" + file;

      if (!fs.existsSync(curDir)){
        // If the Soft Link is invalid, delete it!
        try{
          // If the Soft Link maps to a non existing dir, it will be deleted else an exception will be thrown
          fs.unlinkSync(curDir); 
        }
        catch(err){
          // Ignore the exception that will be thrown if the soft link is invalid
        }
      }
      else if(fs.lstatSync(curDir).isDirectory()){ 
        deleteInvalidSoftLinksSync(curDir); // recurse
      }
    });
  }
}

/**
 * Delete recursively invalid Soft links
 * Based on the very nicely structured https://github.com/AvianFlu/ncp/blob/master/lib/ncp.js
 * @param  {string}     dir
 * @param  {function}   callback(err)
 */
exports.deleteInvalidSoftLinks = function(dir, callback){

  var errs      = null
    , started   = 0
    , finished  = 0
    , running   = 0 // indicates how many times process has been called recursively
    ;

  // Assert Callback function
  callback = (typeof callback === 'function') ? callback : function(){};
  
  process(dir);

  function process(source) {
    
    started++;
    running++;
    fs.lstat(source, function (err, stats) {
      if (err) {
        return onError(err);
      }
      if (stats.isDirectory()) {
        return onDir(source);
      }
      else if (stats.isFile()) {
        return onFile(source);
      }
      else if (stats.isSymbolicLink()) {
        return onLink(source);
      }
    });
  }

  function onFile(file){
    // Ignore files
    //console.log('Processing file ' + file);
    return cb();
  }
  
  function onDir(dir){
    //console.log('Processing dir  ' + dir);
    fs.readdir(dir, function (err, items){
      if(err){
        return onError(err);
      }
      items.forEach(function(item){
        return process(dir + '/' + item);
      });
      return cb();
    });
  }

  function onLink(link){
    //console.log('Processing link ' + link);
    fs.exists(link, function(exists){
      if(!exists){  // If the Soft Link is invalid, delete it!
        //console.log('Deleting  link ' + link);
        fs.unlink(link, function(err){
          if(err){
            return onError(err);
          }
          return cb();          
        });
      }
      return cb();
    });
  }

  function onError(err) {
    if (!errs) {
      errs = [];
    }
    if (typeof errs.write === 'undefined') {
      errs.push(err);
    }
    else { 
      errs.write(err.stack + '\n\n');
    }
    return cb();
  }

  function cb() {
    running--;
    finished++;
    if ((started === finished) && (running === 0)) {
      return errs ? callback(errs) : callback(null);
    }
  }
}
/**
 * listDirsSync       List the dirs contained in the passed root dir    
 * @param  {String}   dir      the root dir
 * @param  {Array}    dirs     list of dirs found
 * @param  {Boolean}  bResolve If true add the FQ path to the list
 */
exports.listDirsSync = function(dir, dirs, bResolve){

  var curDir = '';

  if (bResolve === undefined) bResolve = true;

  if (fs.existsSync(dir)){
    fs.readdirSync(dir).forEach(function(file, index){
      curDir = path.resolve(dir) + '/' + file;
      if (fs.lstatSync(curDir).isDirectory()){ // recurse
        if(bResolve){
          dirs.push({ 'name': curDir})  // Add the dir to the list         
        }
        else{
          dirs.push({ 'name': file})  // Add the dir to the list        
        }
      }
    });
  }
}

/*
  Delete the dirs and subdirs and and their content
    dir: the root dir
    bDeleteDir: is true if dir must be deleted
*/
exports.deleteDirSync = function(dir, bDeleteDir){ /* bDeleteDir is true if dir must be deleted */

  if (fs.existsSync(dir)){
    fs.readdirSync(dir).forEach(function(file, index){
      var curDir = dir + "/" + file;
      if (fs.lstatSync(curDir).isDirectory()){ // recurse
        exports.deleteDirSync(curDir, true);
      } else { // delete file
        fs.unlinkSync(curDir);
      }
    });
    if (bDeleteDir === true){
      fs.rmdirSync(dir);  // delete all other dirs than the root dir
    }
  }
}

/* 
  Evaluate the date diff between 2 dates  
    datepart: 
      'y' --> years,    'm' --> months, 'w' --> weeks, 
      'd' --> days,     'h' --> hours,  'n' --> minutes, 
      's' --> seconds,  's' --> seconds 'x' --> milliseconds 
*/    
exports.dateDiff = function(datepart, fromdate, todate){

  datepart = datepart.toLowerCase();  
  var diff = todate - fromdate; 
  var divideBy = { 
        w:604800000, 
        d:86400000, 
        h:3600000, 
        n:60000, 
        s:1000,
        x:1
      };  

  return Math.floor(diff/divideBy[datepart]);
}

// Pad the passed number to the passed width and returns it as string
exports.padWithZeros = function(vNumber, width){

  var numAsString = vNumber + "";
  while (numAsString.length < width) {
    numAsString = "0" + numAsString;
  }
  return numAsString;
}

// Create a Request Id as yyMMddhhmmssSSS-x where x is random integer between min (inclusive) and max (inclusive)
exports.buildReqId = function(date){

  var min = 10000 
    , max = 100000
    ;
  // See http://stackoverflow.com/questions/1527803/generating-random-numbers-in-javascript-in-a-specific-range/1527820#1527820
  // Math.floor(Math.random() * (max - min + 1)) + min --> returns a random integer between min (inclusive) and max (inclusive)

  return date_format.asString("yyMMddhhmmssSSS", date) + '-' + Math.floor(Math.random() * (max - min + 1) + min);
}

/**
 * Retuns files in a directory in an ARRAY 
 * @Params:
 * dir  :  String. Name of the directory
 * ext  : File extension like ".jar"
 * caseSensitive:  true: Case sensivie ext comparision, false: No case sensivie ext comparision
 * recursive: Boolean: true: Recursively call subdirectories to find files. false:Find files only in given directory 
 * callback    :  A callback function 
 *             callback function will have two arguments.
 *                "err", of node.js type "Error"
 *                "result", of type JSON
 *             In case of success,
 *                "err" will be null.
 *                "result" a JSON array containing files in that directory
 *             In case of error,
 *                "err" will have information about error occurred during method execution
 *                "result" will be null.
 */
exports.getFileList = function(dir, ext, caseSensitive, recursive, callback){
  
  var self    = this
    , absPath = path.resolve(dir) + '/'
    , extArg  = ext
    , fileExt = undefined
    ;

  if(!caseSensitive) extArg    = ext.toLowerCase();

  fs.readdir(absPath, function (err, files){
    if(err){
        callback(err, null);
        return;
    }

    var retFiles=[];
    var errSave= "";

    if(files === undefined){
      callback(new Error("Invalid directory name. Dir:" + absPath), null);
      return;
    }
    else{
      var entryCount = files.length;
      if(!entryCount){
        return callback(null, retFiles);
      }

      files.forEach(function(file){
        var stats = fs.statSync(absPath + file);
        fileExt = path.extname(file);
        if(!caseSensitive) fileExt = fileExt.toLowerCase();
        if (stats.isFile() && fileExt === extArg){
          retFiles.push( absPath + file );
          if (!--entryCount){
            // Invoke callback only after all entries are processed
            callback(null, retFiles);
            return;
          }
        }
        else if(stats.isDirectory() && recursive){
          self.getFileList(absPath + file, ext, caseSensitive, recursive, function(err, res){
              if(err){
                errSave += err.message;
              }
              else if(res !== undefined && res !== null){
                retFiles = retFiles.concat(res);
              }

              if (!--entryCount){
                // Invoke callback only after all dir entries are processed
                if( errSave !== "" ){
                  callback(new Error(errSave), null);  
                }
                else{
                  callback(null, retFiles);
                }
              }
          });
        }
        else{
          if (!--entryCount){
            // Invoke callback only after all entries are processed
            callback(null, retFiles);
            return;
          }
        }
      });
    }
  });
}
//V101 End

// Evaluate the length of an utf-8 string
exports.strLength = function(str){
  // returns the byte length of an utf8 string
  var s = str.length;
  for (var i=str.length-1; i>=0; i--) {
    var code = str.charCodeAt(i);
    if (code > 0x7f && code <= 0x7ff) s++;
    else if (code > 0x7ff && code <= 0xffff) s+=2;
  }
  return s;
}
 
exports.clone = function(a){
   return JSON.parse(JSON.stringify(a));
}

// Create a Unique JobId
exports.createJobId = function(){
  return date_format.asString('yyMMddhhmmssSSS', new Date());
}

// V102 Begin
// Get Text of an Attribute placed under given tag from AFP2web Server Response Body
exports.getAttrTextFromHTML = function(html, tagName, attrName, attrType){

  // Parse HTML
  var parsedHTML = cheerio.load(html);

  // Get contents under tag. For example <pre>
  var tag = parsedHTML(tagName, 'body');

  // Filter contents for the specific attribute
  var targetLine = "" + tag.contents().
          filter(function(i, el){
            if(el.type === attrType && el.data.indexOf(attrName)>=0){ 
                  return true;
              }
          });
  var attrText = null;

  if(targetLine !== null && targetLine.length>0){
    attrText = targetLine.substring(targetLine.indexOf(':') + 1).trim();
  }
  return attrText;
}
// V102 End

// V103 Begin
/**
 * [checkAndCreateDir description]
 * @param  {String}   dir      [description]
 * @param  {String}   mode     [description]
 * @param  {Function} callback [description]
 * @return {}
 */
exports.checkAndCreateDir = function(dir, mode, callback){
    return exports.mkDir(dir, mode, callback);
}

/**
 * mkDir              Create a dir if it does not exist
 * @param  {String}   dir      
 * @param  {String}   mode     
 * @param  {Function} callback(err)
 */
exports.mkDir = function(dir, mode, callback){

  mkdirp.mkdirp(dir, mode,/*works since we set process.umask(0)*/ callback);

}

/**
 * mkDirSync          Create a dir SYNC if it does not exist
 * @param  {String}   dir      
 * @param  {String}   mode     
 * @result            throw an exception in case of error
 */
exports.mkDirSync = function(dir, mode){

  mkdirp.sync(dir, mode/*works since we set process.umask(0)*/);
}

/**
 * Parse a string as JSON
 * @param  {String} str [description]
 * @return {JSON}     [description]
 */
exports.parseJSON = function(str){
    try {
        return JSON.parse(str);
    } catch (ex) {
        return null;
    }
} 
// V103 End

// V104 Begin
/**
 * Merge obj2 Json with Obj1 Json. If any entry of obj2 already exists in obj1, obj1 entry will be replaced with obj2 entry.
 * @param  {JSON} obj1 JSON to which obj2 entries has to be merged
 * @param  {JSON} obj2 JSON
 * @return {JSON}      Returns merged JSON object.
 */
exports.mergeJSON = function(obj1, obj2){
  if(obj2 === undefined || obj2 === null){
    return obj1;    
  }
  
  if(obj1 === undefined || obj1 === null){
    obj1 = {};
  }

  for(var attrname in obj2){
    obj1[attrname] = obj2[attrname];
  }
  return obj1;
}
// V104 End

/**
 * Create async. a Symbolic / Soft Link
 * @param  {String}   srcpath  Source Path
 * @param  {String}   dstpath  Destinattion path
 * @param  {String}   type     'dir', 'file', or 'junction' (default is 'file')
 * @param  {Function} callback A callback function
 *                               callback function will have one argument "err" of node.js type "Error".
 *                               In case of success, "err" will be null.
 *                               In case of error, "err" will have information about error occurred on link creation
 * @return {}
 * 
 */
exports.symlink = function(srcpath, dstpath, type, callback){
    return _symlink(srcpath, dstpath, type, callback);
}

// V106 Begin
/**
 * Delete directories and files that are older than given interval
 * @param  {String}   dir        Root Directory
 * @param  {Boolean}   bDeleteDir Flag to say whether to delete Root directory or not
 * @param  {String}   datePart   String that specifies the interval
 * @param  {Number}   interval   Intervale
 * @param  {Boolean}   recursive  Flag to delete directories recursively or not
 * @param  {Function} callback   callback(err, dirDeleted(true|false)) 
 */
exports.deleteDir = function(dir, bDeleteDir, datePart, interval, recursive, callback){

  var fileDate    = ''
    , now         = new Date()
    , entryCount  = undefined
    , absPath     = path.resolve(dir) + '/'
    ;

  fs.readdir(absPath, function(err, files){
    if(err)  return callback(err);

    entryCount = files ? files.length : 0;
    //logger.debug('-->deleteDir: Deleting ' + absPath + ', entryCount:' + entryCount);
    async.each(files, function (file, next){
          file = absPath + file;
          fs.stat(file, function(err, stat){
              if(err){
                if(err.code === 'ENOENT') logger.debug('--->deleteDir: Warning! Unable to get statistics of file. ' + file +' file not found.');
                else logger.warn('--->deleteDir: Unable to get statistics of file(' + file +') ' + err.message);
                next();
                // return next(err);
              } 
              else if(stat.isDirectory() && recursive){
                exports.deleteDir(file, true, datePart, interval, recursive, function(err, dirDeleted){
                  if(err){ 
                    if(err.code === 'ENOENT') logger.debug('--->deleteDir: Unable to delete directory. ' + file + ' dir not found.');
                    else logger.warn('--->deleteDir: Unable to delete directory (' + file +'). ' + err.message);
                    // return next(err);
                  }
                  // Directory deleted ? if yes decrement entry count
                  else if(dirDeleted) {
                    //logger.debug('-->deleteDir: Deleted ' + file + ', entryCount:' + entryCount);
                    entryCount--;
                  }
                  next();
                });
              }
              else if(stat.isFile()){
                // Get Last Modification time
                fileDate =  new Date(stat.mtime); 
                if(exports.dateDiff(datePart, fileDate, now) < interval){
                  //skip dirs and files that are not older enough for deletion
                  //logger.debug('-->deleteDir: Skipping ' + file + ' from deletion');
                  return next();
                }
                //logger.debug('-->deleteDir: Deleting ' + file);
                fs.unlink(file, function(err){
                  if(err){
                    if(err.code === 'ENOENT') logger.debug('--->deleteDir: Unable to delete file. ' + file + ' file not found.');
                    else logger.warn('--->deleteDir: Unable to delete file (' + file +'). ' + err.message);
                  }
                  else{
                    entryCount--;
                    //logger.debug('-->deleteDir: Deleted ' + file + ', entryCount:' + entryCount);
                  }
                  next();
                });
              }
              else{
                entryCount--;
                next();
              }
          });
        },
        function(err){
          if(err) return callback(err);

          if(entryCount === 0 && bDeleteDir === true){
            //logger.debug('-->deleteDir: Deleting ' + absPath+ '...');
            fs.rmdir(absPath, function(err){
                return callback(err, err?false:true);
            });
          }
          else return callback(null, false);
        }
    );
  });
}
// V106 End

// V107 Begin
/**
 * Get document id of the request
 * @param  {JSON}   req Request  Object
 * @return {String}     Documen id in case of success else error object
 */
exports.getDocId = function(req){

  if(req.input === undefined){
    return (new Error("request does not contain input document parameters"));
  }

  // V108 Begin
  if(req.input.buffer){
    var srcBuf = new Buffer(req.input.buffer, 'base64')
      , tmpBuf = srcBuf.slice(0, 1024);
    return md5(tmpBuf, 'hex');  
  }
  else if(req.input.filename){
    return md5(req.input.filename, 'hex');
  }
  else if(req.input.url){
    var urlTmp = req.input.url.trim()
      , idx    = urlTmp.indexOf('?')
      ;

    if(idx>=0){
      urlTmp = urlTmp.substring(0,idx);
    }
    return md5(urlTmp, 'hex');
    /*
    var urlParts = url.parse(req.input.url);
    if(urlParts.path && (urlParts.path.length > 0)){
      return md5(urlParts.path);
    }
    else{
     return (new Error("input.url does not contain any path to get the document"));
    }
    */
  }
  else{
    return (new Error("request does not contain input document parameters"));
  }
  /*
  // if docId + url is given, extract docId from url
  if(req.input.docId && req.input.url){
    return this.getValueUsingRegExp(req.input.docId, req.input.url);
  }
  else if(req.input.docId){
    // Return docId, if passed in the request
    return req.input.docId;
  }
  else{
    return null;
  }
  */
 // V108 End
 }

// V110 Begin
/**
 * Read the pass file content and get MD5 Id
 * @param  {String}   filename  File Name
 * @param  {Function}   callback(err, md5Id)
 */
 exports.getMD5Id = function(filename, callback){
  var md5Id = null;

  if(filename === undefined || filename === null || filename.length <=0){
    return callback(new Error("Could not generate MD5Id, passed filename is null or empty"));
  }
  fs.readFile(filename, function(err, data){
    if(err){
      return callback(new Error('Could not generate MD5Id, unable to read "' + filename + '". ' + err.message));
    }
    md5Id = md5(data, 'hex');
    return callback(null, md5Id);
  });
}
// V110 End

/**
 * Get MD5 Id for the passed string
 * @param  {String}   data  data
 * @param  {Function}   callback(err, md5Id)
 */
//!!! Extend it for any object like buffer/JSON
 exports.getObjMD5Id = function(data, callback){

  var md5Id = null;

  if(data && data.length > 0 ){
    md5Id = md5(data, 'hex');
    return callback(null, md5Id);
  }else{
    return callback(new Error("Could not generate MD5Id, passed data is null or empty"));
  }
}

// 
/**
 * Get value from input using regular expression
 * @param  {String} regExpr Regular Expression String
 * @return {String}         If regular expression is valid returns a value extracted from inputString 
 *                          else returns regular expression itself.
 */
exports.getValueUsingRegExp = function(regExpr, inputString){
  if(regExpr === null || inputString === null){
    return null;
  }

  var regexp = new RegExp(regExpr)
    , result = inputString.match(regexp)    // Refer http://msdn.microsoft.com/en-us/library/ie/7df7sf4x%28v=vs.94%29.aspx
    , retValue = ""
    ;

  //console.log('-->getValueUsingRegExp:RegExp: "' + regExpr + '", result:' + JSON.stringify(result));
  if(result && result.length > 1){
    // Values to be extracted start from index 1.
    for(var index=1; index<result.length; index++)
    {
        if(retValue.length > 0){
          retValue += '-';
        }
        retValue += result[index];
    }
  }
  else{
      retValue = regExpr;
  }
  //console.log('-->_getDocIdFromRegExp: evaluated value is "' + retValue + '"');
  return retValue;
}
// V107 End

/**
 * Get Cache filename suffix
 * @param  {JSON} req                  Request Object
 * @param  {JSON} cacheFilenamePattern JSON Array having info about parameter names used to build cache filename suffix
 * @return {String}                    Returns cache filename suffix
 */
exports.getCacheFilenameSuffix = function(req, cacheFilenamePattern){ 
  var name = ""
    ;

  if(req.props === undefined || req.props === null){
    return "";
  }  

  if(cacheFilenamePattern === undefined || cacheFilenamePattern.length <=0){
    return "";
  }

  var value          = null
    , filenameSuffix = ""
    ;

  for(var i = 0; i < cacheFilenamePattern.length; i++){
    // Get Parameter Name
    name = cacheFilenamePattern[i].name;

    // Get Parameter value
    if(req.props[name]){
      value = req.props[name];
    }
    else
    {
      value = cacheFilenamePattern[i].defaultValue;
    }

    if(cacheFilenamePattern[i].type && cacheFilenamePattern[i].type.toLowerCase() === 'bool'){
      if(cacheFilenamePattern[i].defaultValue && cacheFilenamePattern[i].defaultValue.toLowerCase() !== value.toLowerCase()){
        filenameSuffix +=  "-" + cacheFilenamePattern[i].shortForm;
      }
    }
    else{
      if(cacheFilenamePattern[i].valuePattern && value !== null && value.length > 0){
        // Get Parameter value applying regular expression
        value = exports.getValueUsingRegExp(cacheFilenamePattern[i].valuePattern, value);
      }
      if(cacheFilenamePattern[i].replacePattern && value !== null && value.length > 0){
        // Replace given pattern in parameter value with value given in conf.
        //var regexp = XRegExp(cacheFilenamePattern[i].replacePattern.pattern);
        //value = XRegExp.replace(value, regexp, cacheFilenamePattern[i].replacePattern.value, 'all');
        var regexp = new RegExp(cacheFilenamePattern[i].replacePattern.pattern);
        value = value.replace(regexp, cacheFilenamePattern[i].replacePattern.value);
      }     

      filenameSuffix +=  "-" + cacheFilenamePattern[i].shortForm + value;
    }
  }
  return filenameSuffix;
}


/**
 * Get Output JSON Filename 
 * @param  {Response} res              Response Object
 * @param  {Response} req              Request Object
 * @param  {JSON} cacheFilenamePattern JSON Array having info about parameter names used to build cache filename suffix
 * @return {String}                    Returns Output JSON Filename 
 */
exports.getOutputJSONFilename = function(res, req, cacheFilenamePattern){

    var filenameSuffix = exports.getCacheFilenameSuffix(req, cacheFilenamePattern)
      , jsonFilename   = ""
      ;

  // Set the Output  filename (<docId dir>/<docId>-<pageNr>.<res._Context.outputType>)
  jsonFilename = (res._Context.cacheSubdir || res._Context.docId) + '/' + res._Context.docId + '-' + res._Context.pageNr  + filenameSuffix + '.json';
  return jsonFilename;
}

// V109 Begin
/**
 * Get Output JSON Filename 
 * @param  {Response} res Response Object
 * @return {String}       Returns Output JSON Filename 
 */ 
exports.getOutputJSONFilename_del = function(res){
  var req           = res._Context.req
    , jsonFilename  = ""
    , otherProps    = ""
    ;

  if(req.props){
    // Segserver/OCR request parameters
    if(req.props.rotation && req.props.rotation !== 0){
      otherProps += '-r' + req.props.rotation;
    }
    if(req.props.deskew && req.props.deskew.toLowerCase() === 'yes'){
      otherProps += '-d';
    }
    if(req.props.max_width || req.props.max_height){
      otherProps += '-os';  
      if(req.props.max_width){
        otherProps += req.props.max_width;  
      }
      if(req.props.max_width && req.props.max_height){
        otherProps += 'x' ;
      }     
      if(req.props.max_height){
        otherProps += req.props.max_height; 
      }     
    }
    if(req.props.antialias && req.props.antialias.toLowerCase() === 'on'){
      otherProps += '-aa';
    }
    if(req.props.jpgQuality){
      otherProps += '-jq' + req.props.jpgQuality;
    }
    if(req.props.pngCompressionFactor){
      otherProps += '-pcf' + req.props.pngCompressionFactor;
    }   

    // Image service properties
    if(req.props.Rotation && req.props.Rotation !== 0){
      otherProps += '-r' + req.props.Rotation;
    }
    if(req.props.OutputSize){
      /*
      var idx = req.props.OutputSize.indexOf(",");
      if(idx >= 0){
        var osw = req.props.OutputSize.substring(0, idx);
        var osh = req.props.OutputSize.substring(idx+1);
        otherProps += '-os' + osw + 'x' + osh;
      } 
      */
      // Replace ',' char' with 'x' char
      otherProps += '-os' + req.props.OutputSize.replace(/,/g, 'x');
    }
    if(req.props.JPEGQuality){
      otherProps += '-jq' + req.props.JPEGQuality;
    }
    if(req.props.FormatType){
      otherProps += '-ft' + req.props.FormatType;
    }
    if(req.props.Resolution){
      otherProps += '-pr' + req.props.Resolution;
    }
    if(req.props.OutputScale){
      otherProps += '-oscale' + req.props.OutputScale;
    }
    if(req.props.Color && req.props.Color.toLowerCase() === 'on'){
      otherProps += '-c' + req.props.Color;
    }
  }

  // Set the Output  filename (<docId dir>/<docId>-<pageNr>.<res._Context.outputType>)
  jsonFilename = res._Context.docId + '/' + res._Context.docId + '-' + res._Context.pageNr  + otherProps + '-' + res._Context.outputType + '.json';
  return jsonFilename;
}

/**
 * removeTrailingSlash:   Remove the trailing slash if any
 * @param  {string}       path
 * @return {string}       path without the trailing slash
 */
exports.removeTrailingSlash = function(path){
    return path.replace(/\/$/, "");
} 

/**
 * Check if passed value is positive integer
 * @param  {[number|string]}  inputValue value to be verified
 * @return {Boolean}            Returns trun if passed value is integer and > 0
 */
exports.isPositiveInteger = function(value){
  var ret = /^[1-9]\d*$/.test(value);
  return ret;
}

exports.isValidPageRange_doesnotwork = function(value){
  var ret = /^[1-9]\d*-[1-9]\d*/.test(value);
  return ret;
}

/**
 * [isValidPageRange description]
 * @param  {[type]}  value [description]
 * @return {Boolean}       [description]
 */
exports.isValidPageRange = function(value){
  if(value === null || value.length <= 0){
    return false;
  }

  var idx = value.indexOf('-');   

  if(idx<0){
    return false;
  }
  var startPage = value.substring(0, idx);
  if(startPage && !exports.isPositiveInteger(startPage)){
    return false;
  }

  var endPage = value.substring(idx+1); 
  if(endPage && !exports.isPositiveInteger(endPage)){
    return false;
  }
  return true;
  /*
    10-=> true
    0-0=> false
    0=> false
    -5=> true
    -55=> true
    672-=> true
    2-3=> true
    25-300=> true
    25+300=> false
    xyz=> false
    123=> false
  */  
}

/**
 * getHardwareInfo
 * @return {String}     Hardware Info
 */
exports.getHardwareInfo = function(){

  var cpus = os.cpus()
    , CRLF_Tabs = '\n\t\t\t\t\t\t\t'
    , hardwareInfo = 'Server Info - Hardware:'
                   + CRLF_Tabs + 'Hostname: ' + os.hostname() + ', ' + os.platform() + ' ' + os.arch() + ' ' + os.release()
                   + CRLF_Tabs + 'Average Load: ' + os.loadavg()
                   + CRLF_Tabs + 'Total Mem.: ' + (os.totalmem()/1024/1024/1024).toFixed(2) + 'GB, Free Mem.: ' + (os.freemem()/1024/1024/1024).toFixed(2) + 'GB.'
    ;
  
  for (var i = 0; i < cpus.length; i++) {
    hardwareInfo += CRLF_Tabs + 'CPU ' + (i+1) +', Model: ' + cpus[i].model;
  };
  
  hardwareInfo += CRLF_Tabs + 'Network Interfaces: ' + _networkInterfaces();

  /* Other Hardware Info
    logger.info(os.endianness());
    logger.info(os.type());
    logger.info(os.tmpdir());
    logger.info(os.uptime());
    logger.info(os.cpus());
    logger.info(os.networkInterfaces());
  */
  return hardwareInfo;
}

/**
 * getEnvironmentInfo   
 * @return {String}     Environment Info
 */
exports.getEnvironmentInfo = function(){

  /* Let's try to beautify the output as:
    {
    "TERM":"xterm",
    "SHELL":"/bin/bash",
    "XDG_SESSION_COOKIE":"d1938ee4627d4f0a3849455d00000007-1461134155.149860-472728795",
    "USER":"root",
    "LD_LIBRARY_PATH":"./node_modules/oxsnps-ibmmq/services/",
    "LS_COLORS":"rs=0:di=01;34:ln=01;36:... <cut> ...:*.spx=00;36:*.xspf=00;36:",
    "SVN_USERNAME":"daniel",
    "SUDO_USER":"adminuser",
    "SUDO_UID":"1000",
    "USERNAME":"root",
    "PATH":"/usr/local/sbin:/usr/local/bin:/usr/sbin:/usr/bin:/sbin:/bin:/usr/games",
    "MAIL":"/var/mail/root",
    "PWD":"/opt/OXS.IMPORT",
    "UV_THREADPOOL_SIZE":"24",
    "LANG":"en_US.UTF-8",
    "SVN_PASSWORD":"EkLj9/3I1BF6f8x4Y0Ko",
    "NODE_PATH":"/usr/lib/nodejs:/usr/lib/node_modules:/usr/share/javascript",
    "HOME":"/root",
    "SUDO_COMMAND":"/bin/su",
    "SHLVL":"2",
    "LOGNAME":"root",
    "LESSOPEN":"| /usr/bin/lesspipe %s",
    "DISPLAY":":0",
    "SUDO_GID":"1000",
    "LESSCLOSE":"/usr/bin/lesspipe %s %s",
    "COLORTERM":"gnome-terminal",
    "XAUTHORITY":"/home/adminuser/.Xauthority",
    "_":"/usr/bin/node"
    }  
  */    
  // Walk thru the JSON Object, sort it and build the Environment Info message
  return exports.walk(process.env);
}

/**
 * Returns info about the Server Environment
 * @param  {Object}     obj           A Javascript Object
 * @return {String}     Environment Info
 * 
 * see http://am.aurlien.net/post/1221493460/sorting-javascript-objects
 */
exports.walk = function(obj){

  var environmentInfo = 'Server Info - Environment:'
    , temp_array = []
    ;

  // Make an array out of the obj
  for (var key in obj) {
    if (obj.hasOwnProperty(key)) {
      temp_array.push(key + '=' + obj[key]);
    }
  }
  temp_array.sort(); // sort the array

  // Build the result
  for (var i=0; i<temp_array.length; i++) {
    environmentInfo += CRLF_Tabs + temp_array[i];
  }
  return environmentInfo;
}

/**
 * Returns info about the Server's mount points
 * @return {String}     Mount points Info
*/
exports.getMountInfoSync = function(){

  try{
    var stdout = 'Server Info - Mount:\n' + cp.execSync('mount'); // execute mount command to get the mount points
    stdout = stdout.replace(/(\r?\n|\r)$/, '');     // remove last new line
    stdout = stdout.replace(/\r?\n|\r/g, CRLF_Tabs);// replace new lines with CRLF_Tabs
    return stdout;
  }
  catch(err){
    return 'Mount information: ERROR when getting the mount points, Reason: ' + err.message;
  }
}

/**
 * Returns info about the Server's /etc/hosts
 * @return {String}     /etc/hosts Info
 */
exports.getHostsSync = function(){

  try{
    var stdout = 'Server Info - Hosts (/etc/hosts):\n' + cp.execSync('cat /etc/hosts'); // execute cat command to get/etc/hosts content
    stdout = stdout.replace(/(\r?\n|\r)$/, '');     // remove last new line
    stdout = stdout.replace(/\r?\n|\r/g, CRLF_Tabs);// replace new lines with CRLF_Tabs
    return stdout;
  }
  catch(err){
    return 'Hosts (/etc/hosts) information: ERROR when getting the content of /etc/hosts, Reason: ' + err.message;
  }
}

/**
 * pad:               Pad the String (str) in the Length (length) using the Char (char)
 * @param  {string}   str
 * @param  {int}      length
 * @param  {char}     char
 * @return {string}   the modified str
 */
exports.pad = function(str, length, char){
  while (str.length < length){
    str = char + str;
  }
  return str;
}

/**
 * Convert XML file to JSON string
 * @param  {String}   xmlData         XML Data
 * @param  {Object}   parseOptions    Optional. XML Parser Options. Refer to https://www.npmjs.com/package/xml2js
 * @param  {Function} callback        callback(err, jsonObject)
 */
exports.xml2json = function(xmlData, parseOptions, callback){
  
  var parser = undefined;

  if(!callback){
    callback = parseOptions;
    parseOptions = undefined;
  }

  if(!parseOptions){
    parseOptions = {
      explicitArray:false, 
      tagNameProcessors:[xml2js.processors.stripPrefix]
    }    
  };

  if(!parseOptions.tagNameProcessors) parseOptions.tagNameProcessors = [xml2js.processors.stripPrefix];

  parser = new xml2js.Parser(parseOptions);
  parser.parseString(xmlData, callback);
}

/**
 * Convert XML file to JSON string
 * @param  {String}   xmlFile         Fully qualified XML filename
 * @param  {Object}   readOptions     File Read options like {encoding:'utf8'}
 * @param  {Object}   parseOptions    Optional. XML Parser Options. Refer to https://www.npmjs.com/package/xml2js
 * @param  {Function} callback        callback(err, jsonObject)
 */
exports.xmlFile2json = function(xmlFile, readOptions, parseOptions, callback){

    var fileEnc 		= undefined
      , result 			= undefined
      , passedEnc 		= undefined
      , fileContent 	= undefined
      , regEx           = undefined
      , regExResult     = undefined
      , xmlDefaultEnc   = undefined
      ;

/*      
http://www.w3schools.com/xml/xml_syntax.asp
 -- UTF-8 is the default character encoding for XML documents.
 -- XML Attribute Values Must be Quoted
*/
  // Reg. exp to find encoding attribute value from xml content
  regEx           = /\s*encoding\s*=\s*"\s*([^\s]+)\s*"/i
  xmlDefaultEnc   = 'utf-8'; // UTF-8 is the default character encoding for XML documents.

  if(!readOptions) readOptions = {};

  if(!callback){
    callback = parseOptions;
    parseOptions = undefined;
  }

  if(!parseOptions) parseOptions = {};

  // Get Passed encoding
  passedEnc = readOptions.encoding || null; // Refer to fs.readFile in fs.js from node.js
  if(passedEnc) passedEnc = passedEnc.toLowerCase();

  async.series([
    // Read the file
    function(nextTask){
      fs.readFile(xmlFile, readOptions, function(err, data){
        if(err){
          return nextTask(new Error('Unable to read "' + xmlFile + '", ReadOptions=' + JSON.stringify(readOptions) +', ' + err.message));
        }
        fileContent = data;
        nextTask();
      });
    },

    // Get encoding from file content
    function(nextTask){
      regExResult = fileContent.match(regEx);
      if(regExResult && regExResult.length > 1){
        fileEnc = regExResult[1]; 
      }
      else{
        fileEnc = xmlDefaultEnc;
      }
      fileEnc = fileEnc.toLowerCase();
      if(fileEnc.match(/ISO-8859-\d+/i)) fileEnc = 'binary';
      nextTask();
    },

    // If encoding specified in readOptions is not same as encoding specified within XML file 
    // read the XML file again using the encoding specified within XML file 
    function(nextTask){
      //console.log('filenc:>'+ fileEnc +'<');
      //console.log('passedEnc:>'  + passedEnc +'<');

      if(passedEnc && passedEnc !== fileEnc){
        readOptions.encoding = fileEnc;
        //console.log('reading file again using ' + JSON.stringify(readOptions));
        fs.readFile(xmlFile, readOptions, function(err, data){
          if(err){
            return nextTask(new Error('Unable to read "' + xmlFile + '", ReadOptions=' + JSON.stringify(readOptions) +', ' + err.message));
          }
          fileContent = data;
          nextTask();          
        });
      }
      else nextTask();
    },    

    // convert xml 2 json
    function(nextTask){
      exports.xml2json(fileContent, parseOptions, function(err, result){
        if(err){
          return callback(new Error('Unable to parse XML data read from "' + xmlFile + '". ' + err.message));
        }
        callback(null,result);
      });
    }

  ],  function(err){
      return callback(err, result)
    }
  );    
}

/**
 * Convert JSON as XML
 * @param  {JOSN}       json            JSON Object
 * @param  {JSON}       xmlBuildOptions XML Build Options. Refer to https://www.npmjs.com/package/xml2js
 * @param  {Function}   callback        callback(err, xmlData)
 */
exports.json2xml = function(json, xmlBuildOptions, callback){
  
  var xmlData = null
    , builder = null
    ;

  if(!callback){
    callback = xmlBuildOptions;
    xmlBuildOptions = undefined;
  }

  if(!xmlBuildOptions){
    xmlBuildOptions = {
        renderOpts:{
          'pretty': false // pretty print or not
        },
        xmldec: {
          'version':'1.0',
          'encoding': exports.getEncoding('binary')
        }
    }    
  };

  // Convert JSON to XML
  builder = new xml2js.Builder(xmlBuildOptions);
  
  try{
    xmlData = builder.buildObject(json); /* xmlData is not a buffer it is a string in UTF8) */
  }
  catch(err){
    return callback(new Error('JSON to XML conversion failed. ' + err.message));
  }
  
  return callback(null, xmlData);
}

/**
 * Write passed JSON as XML to given filename
 * @param  {JOSN}       json            JSON Object
 * @param  {String}     xmlFilename     XML Filename to write to
 * @param  {JSON}       xmlBuildOptions XML Build Options. Refer to https://www.npmjs.com/package/xml2js
 * @param  {JSON}       writeOptions    File Write options.
 * @param  {Function}   callback        callback(err, xmlData)
 */
exports.json2xmlFile = function(json, xmlFilename, writeOptions, xmlBuildOptions, callback){
  
  if(!writeOptions) writeOptions = {};

  if(!callback){
    callback        = xmlBuildOptions;
    xmlBuildOptions = undefined;
  }

  exports.json2xml(json, xmlBuildOptions, function(err, xmlData){
    if(err) return callback(err);
    fs.writeFile(xmlFilename, xmlData, writeOptions, function(err1){
      if(err1) return callback(new Error('Writing JSON to  "' + xmlFilename + '" failed. ' + err1.message));
      return callback(null, xmlData);
    }); 
  });
}

/**
 * Get equivalent standard encoding string that corresponds to passed node.js encoding
 * @param  {String} nodeJSencoding Encoding as specified by node.js
 * @return {String}                returns equivalent standard encoding string that corresponds to passed node.js encoding
 */
exports.getEncoding = function(nodeJSencoding){
  // Refer to https://nodejs.org/api/buffer.html to get list of node.js supported encodings
  // Refer to https://en.wikipedia.org/wiki/Character_encoding
  switch(nodeJSencoding.toLowerCase()){
    case 'hex':
      return 'hex';

    case 'base64':
      return 'base64';

    case 'utf8':
    case 'utf-8':
      return 'UTF-8';

    case 'ascii':
      return 'ASCII';

    case 'binary':
      return 'iso-8859-1';

    case 'ucs2':
    case 'ucs-2':
      return 'UCS-2';

    case 'utf16le':
    case 'utf-16le':
      return 'UTF-16';

    default:
      return new Error('Unknown encoding: ' + nodeJSencoding);
  }
}

/**
 * Get Last Token separated by given delimiter
 * @param  {String} string    Input String
 * @param  {String} delimiter Delimiter character or string
 * @return {String}           Last token of string separated by given delimiter
 */
exports.getLastToken = function (string, delimiter){
  var idx       = 0
    , retValue  = string
    ;

  if(string === undefined || string === null || string.length === 0) return retValue;

  idx   = string.lastIndexOf(delimiter);

  if(idx>0) retValue = string.substring(idx+1);
  return retValue;
}

/**
 * Iterate a JSON and call <handleKey>  function for each key
 * @param  {[JSON]} obj           JSON Object
 * @param  {String} parentObjName Parent Object name. For Root or top level object it is string
 * @param  {Function} handleKey   Function that gets key name and its value.
 * @return {[type]}               None
 */
exports.iterateJSON=function(obj, parentObjName, handleKey){
  var value = ''
    , name  = ''
    ;

  for(name in obj){
    value = obj[name];
    if(parentObjName !== null && parentObjName.length > 0){
      name = parentObjName + '.'  + name;
    }
    if(value instanceof Object){
      exports.iterateJSON(value, name, handleKey);
    }
    else{
      handleKey(name, value);
    }
  }
}

/**
 * Convert booelan string to boolean 
 * @param  {String} string Input String
 * @return {boolean}        
 */
exports.stringToBoolean=function(string){
    switch(string.toLowerCase().trim()){
        case "true": case "yes": case "1": return true;
        case "false": case "no": case "0": case null: return false;
        default: return Boolean(string);
    }
}

/**
 * Strip a trial comma in the passed string
 * @param  {[String]} str Input Sting
 * @return {[String]}     String with stripped trailing comma, if any              
 */
exports.stripTrailingComma=function(str){
  var idx = 0;

  if(str && str.length > 0){
    idx = str.lastIndexOf(',');
    if(idx>=0 && idx === (str.length -1)) str = str.substring(0, idx);
  }
  return str;
}

/**
 * Check whether a string matches a pattern specified as regular exp.
 * @param  {String}     str
 * @param  {Reg. Exp.}  pattern
 * @return {Boolean}    Returns true if str matches pattern
 */
exports.strEndsWith=function(str, pattern){
//!!!IMPORTANT NOTE: just looks for now if str ends with pattern (see $)
  if (str.match(pattern + '$')) return true;
    return false;
}

/**
 * Copy a file or directory. 
 * @param  {String}   src      Source file or dir.
 * @param  {String}   dest     Destination file or dir.
 * @param  {JSON}     options  Refer to https://www.npmjs.com/package/fs-extra#outputjsonfile-data-callback
 * @param  {Function} callback callback(err)
 */
exports.copyFile=function(src, dest, options, callback){
  fse.copy(src, dest, options, callback);
}

/**
 * Encode string data passed as part of Get or Post http requests
 * @param  {String} str   Data to be encoded 
 * @return {String}       Encoded data
 */
exports.encodeURIData=function(str){
  /*
    Refer to  
    https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/encodeURIComponent
    https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/encodeURI

    encodeURI() and encodeURIComponent() escapes all characters except the following: alphabetic, decimal digits, - _ . ! ~ * ' ( )
    encodeURI does not encode "&", "+", and "=" chars, which are treated as special characters in GET and POST requests
    however encodeURIComponent() does encode these characters. 
   */
  return encodeURIComponent(str);
}

/**
 * Search of %HH where HH is a hex digit and replace with corresponding utf-8 char
 * @param  {String} str input string
 * @return {String}     String with replaced values for %HH 
 */
//!!! To be moved to customer specific common module
exports.replaceHexWithUTF8Char=function(str) {

  var regexp1    = /%u{0,1}([0-9A-F]{1,8})/ig
    , regexp    = /%u([0-9A-F]{1,4})|%([0-9A-F]{1,2})/ig
    , result    = str.match(regexp)  
    , ret       = undefined
    , utf8char  = undefined
    ;

  if(result && result.length > 0){      
    //console.log(str);
    str = str.replace(regexp, function(match, p1, p2){
      if(p1) p1 = p1.toUpperCase();
      if(p2) p1 = p2.toUpperCase();
      utf8char = hex2utf8[p1];
      //console.log('utils:'+'p1:' + p1 +', ' + p2 + ', utf8char:' + utf8char );
      if(utf8char){
        // We use Buffer since in hex2utf8.js we can define utf8 char(lefthand side) as array of bytes too. 
        // It is useful when we can not enter utf8 char in js file as character.
        ret = new Buffer(utf8char).toString();
      }
      else{
        // hex2utf8 does not contain mapping for p1.
        ret = match;  
      }
      return ret;
    }); 
    //console.log(str);
  }
  return str;
};

/**
 * Pad string with given 'pad' char for specified length
 * @param  {String} str   Input String
 * @param  {Int}    len   Padding Length
 * @param  {String} pad   Padding Character
 * @return {String}       Padded String
 */
exports.padLeft = function(str, len, pad){
    pad = typeof pad === 'undefined' ? '0' : pad + '';
    str = str + '';
    while(str.length < len) str = pad + str;
    return str;
}

/**
 * Ensure passed file is readable on return
 * @param  {String}   filename   Filename
 * @param  {int}   retryCount Retry count. Number of time to check for file readabiliy
 * @param  {int}   interval   Time in milliseconds, to pause between the file readable checks
 * @param  {int}   filesize   Optional. Used internally by recursive calls
 * @param  {Function} callback   callback(err)
 *                                 If file is ready for reading, err=null
 *                                 else err object contains error information
 */
exports.ensureFileIsReadable = function(filename, retryCount, interval, filesize, callback){
  var newSize = 0;

  if(typeof filesize === 'function'){
    callback = filesize;
    filesize = 0;
  }
  
  //console.log('filename:' + filename +', retryCount:'+ retryCount +', filesize:' + filesize);

  // fs.open() and fs.readFile() returns control, even if file is being written, so not used
  // Get File statistics
  fs.stat(filename, function(err, stats){
    if(retryCount == 0 && err) 
        return callback(new Error(filename + ' is not accessible. ' + err.message));

    if(err){
       //console.log('filename:' + filename + ' not readable yet. ' + err.message + ', retryCount:'  + retryCount + ', filesize:' + filesize + ', scheduled to check again...');
    }
    else{  
      newSize = stats.size;

      // if file size is same as prev. iteration file size, file is fully written 
      if(filesize !== 0 && filesize === newSize)
           return callback();

      // File is not accessible even after waiting for specified time
      if(retryCount == 0)
        return callback(new Error(filename + ' is not accessible even after waiting for specified time'));

       //console.log('filename:' + filename + ' not readable yet. ' + ', retryCount:'  + retryCount + ', filesize:' + newSize + ', scheduled to check again...');
    }
    // File not found still or still file is being written, so call _ensureFileIsReady after a delay
    setTimeout(function(){
        exports.ensureFileIsReadable(filename, --retryCount, interval,  newSize, callback);
      },
      interval
    );      
   });
}

/**
 * Replace not allowed chars in simple filename with '-' or passed replacement char
 * @param  {String}   simpleFilename    Simple filename without path
 * @param  {String}   replaceChar       Replace Char
 * @return {String}                     
 */
exports.replaceInvalidCharsInFilename = function(simpleFilename, replaceChar){
  
  if(!replaceChar) replaceChar = '-';

  // Refer to http://www.mtu.edu/umc/services/web/cms/characters-avoid/
  // to see Characters to Avoid in Directories and Filenames
  simpleFilename = simpleFilename.replace(/[#<$+%>!`&*‘|{?"=}/:\@ ]/g, replaceChar); 

  return simpleFilename;
}

/**
 * DeleteDirectory recursively
 * @param  {String|glob pattern}  dir      Directory
 * @param  {JSON}                 opts     Options. Refer to https://github.com/isaacs/rimraf
 * @param  {Function}             callback callback(err)
 */
exports.deleteDirectory = function (dir, opts, callback){
  rimraf(dir, opts, callback);
}

/**
 * Retuns files in a directory in an ARRAY matching the pattern
 * @Params:
 * dir  :  String. Name of the directory
 * ext  : File extension like ".jar"
 * caseSensitive:  true: Case sensivie ext comparision, false: No case sensivie ext comparision
 * recursive: Boolean: true: Recursively call subdirectories to find files. false:Find files only in given directory 
 * callback    :  A callback function 
 *             callback function will have two arguments.
 *                "err", of node.js type "Error"
 *                "result", of type JSON
 *             In case of success,
 *                "err" will be null.
 *                "result" a JSON array containing files in that directory
 *             In case of error,
 *                "err" will have information about error occurred during method execution
 *                "result" will be null.
 */
exports.getFiles = function(dir, pattern, recursive, callback){
  var absPath     = path.resolve(dir) + '/'
    , retFiles    = []
    , fqFile      = undefined
    ;

  // If pattern is string, convert pattern to regular expression  
  if(typeof(pattern) === 'string') pattern = new RegExp(pattern);

  fs.readdir(absPath, function (err, files){
    if(err) return callback(err, retFiles);

    if(files === undefined) 
        return callback(new Error("Invalid directory name. Dir:" + absPath), retFiles);

    if(!files.length) return callback(null, retFiles);

    async.forEachOfLimit(files, 1,
      function(file, index, next){
          fqFile = absPath + file;
          fs.stat(fqFile, function(err, stat){
              if(err)
                next(new Error('--->getFiles: Unable to get statistics of file(' + fqFile +'). ' + 
                              err.message));
              else if(stat.isDirectory() && recursive){
                exports.getFiles(fqFile, pattern, recursive, function(err, result){
                  if(err) return next(err);
                  if(result) retFiles = retFiles.concat(result);
                  next();
                });
              }
              else if(stat.isFile() && file.match(pattern)){
                retFiles.push(fqFile);
                next();
              }
              else next();
            });
        },
        function(err){
          return callback(err, retFiles);
        }
    );        
  });
}

/**
 * Performs an optimized deep comparison between the two objects, to determine if they should be considered equal. 
 * Refer to http://underscorejs.org/#rest
 * @param  {Object}   json1 JSON 
 * @param  {Object}   json2 JSON
 * @return {boolean}        Returns true if both JSON have same props and values
 *                          else returns false
 */
exports.jsonEqual = function(json1, json2){
  return underscore.isEqual(json1, json2)
}

/**
 * Move 'soruce' dir or file to 'dest' location
 * @param  {String}   source  source dir/file
 * @param  {String}   dest    destination dir/file
 * @param  {JSON}     options   see https://github.com/andrewrk/node-mv
 * @param  {Function} callback(err)      
 */
exports.move = function(source, dest, options, callback){
  mv(source, dest, options, callback)
}

/**
 * Search for chars defined in utf82hex and replace with %HH representation
 * @param  {String} str input string
 * @return {String}     String with %HH valuesfor special chars defined in utf82hex
 */
/*
exports.replaceUTF8withHex = function(str){

  var ret = '';
  for(var i = 0; i < str.length; i++){
    //console.log('"' + str[i] + '"=>'  + utf82hex[str[i]]);
    if(utf82hex[str[i]]){
      ret = ret + '%' + utf82hex[str[i]].toUpperCase();
    }
    else ret = ret + [str[i]];
  }
  return ret;
};
*/

/**************** PRIVATE FUNCTIONS ***************/
/**
 * _networkInterfaces:  Return the Network Interfaces
 * @return {String}     Network Interfaces as string
 */
function _networkInterfaces(){

  //console.log('----------------->_networkInterfaces...');

  var ifaces = os.networkInterfaces()
    , alias = 0
    , networkInterfaces = ''
    ;


  Object.keys(ifaces).forEach(function (ifname){

    alias = 0;

    ifaces[ifname].forEach(function (iface) {
      if ('IPv4' !== iface.family || iface.internal !== false) {
        // skip over internal (i.e. 127.0.0.1) and non-ipv4 addresses
        return;
      }

      if (alias >= 1) {
        // this single interface has multiple ipv4 addresses
        //console.log(ifname + ':' + alias, iface.address);
        networkInterfaces += ifname + ': ' + alias + ', ' + iface.address + '';
      } else {
        // this interface has only one ipv4 adress
        //console.log(ifname, iface.address);
        networkInterfaces += ifname + ': ' + iface.address + '';
      }
    });
  });
  return networkInterfaces;
}

/*
  Delete the dir and its content if the dir date is older than (today's date + interval)
    dirs: array of dirs
    now:  today's date
    interval: if the dir date is older than (today's date + interval (in seconds)) then delete the dir.
*/
function _deleteDir(dirs, now, interval){
  
  // Any dir?
  if (dirs.length <= 0){
    return; // over since array is empty
  }

  // Get the first dir to check
  var dir = dirs[0]; 

  // If no date passed, try to extract it from the name of the dir
  // works only for /opt/OTS/nodeDocServer/temp/140912110440272-78578
  if (dir.date === undefined){

    // Extract the relative dir out of the fully qualified dir (the last 21 digits)
    // /opt/OTS/nodeDocServer/temp/140912110440272-78578
    var relDir = dir.name.substring(dir.name.length - 21, dir.name.length);

    // Extract the date from the relative dir
    dir.date = new Date(20 + exports.padWithZeros(relDir.substring(0,  2), 2),    // year (4 digits year)
                             exports.padWithZeros(relDir.substring(2,  4)-1 ,2),  // month (month starts with 0)
                             exports.padWithZeros(relDir.substring(4,  6), 2),    // day
                             exports.padWithZeros(relDir.substring(6,  8), 2),    // hours
                             exports.padWithZeros(relDir.substring(8,  10), 2),   // minutes
                             exports.padWithZeros(relDir.substring(10, 12), 2),   // seconds
                             exports.padWithZeros(relDir.substring(12, 15), 3)    // milliseconds
                        );
  }

  // If dir is older that interval seconds, then delete it
  if (exports.dateDiff('s', dir.date, now) >= interval){
    //logger.debug(__filename, '_deleteDir: deleting ' + dir.name + ', date: ' + dir.date + '...');
    try{
      exports.deleteDirSync(dir.name, true);  // false means delete the content of the temp/ dir but NOT the the temp/ dir
      dirs.shift();         // remove dir from array
    }
    catch(err){
      //logger.debug(__filename, '_deleteDir: Error when deleting ' + dir.name + ', Message: ' + err.message);
    }
      return _deleteDir(dirs, now, interval); // delete the next dir in the array
  }
}

/**
 * Low level function to create async. a Symbolic / Soft Link
 * @param  {String}   srcpath  Source Path
 * @param  {String}   dstpath  Destinattion path
 * @param  {String}   type     'dir', 'file', or 'junction' (default is 'file')
 * @param  {Function} callback A callback function
 *                               callback function will have one argument "err" of node.js type "Error".
 *                               In case of success, "err" will be null.
 *                               In case of error, "err" will have information about error occurred on link creation
 * @return {}
 * 
 */
function _symlink(srcpath, dstpath, type, callback){  

  type = type || 'file';      // assert type

  // Create a Symbolic / Soft Link from tsPath (<cachedir>/objects) to the temp dir (res._Context.tempDir)
  fs.symlink(srcpath, dstpath, type, callback);
} 

/** This code would not work since there is no guranteed order in which keys in JSON are returned
 *  Refer to http://book.mixu.net/node/ch5.html
 */
/*
exports.getOutputJSONFilename_del = function(res){
  
  var req        = res._Context.req
  , jsonFilename = ""
  , otherProps   = ""
  , keyTmp       = ""
  , value        = ""
  ;
  
  var prefix={
    "pngcompressionfactor": "pcf",  
    "rotation":    "r",
    "dskew":       "d",
    "max_width":   "mw",
    "max_height":  "mh",
    "antialias":   "aa",
    "jpgquality":  "jq",
    "resolution":  "pr",
    "formattype":  "ft",
    "jpegquality": "jq",
    "outputsize":  "os",
    "outputscale": "oscale",
    "color":       "c"
  };

  for(var keyTmp in req.props){
    value = req.props[keyTmp] + "";
    if(value.toLowerCase() === 'off' || value.toLowerCase() === 'no'){
      //do nothing  
    }
    else if(value.toLowerCase() === 'on' || value.toLowerCase() === 'yes'){
      otherProps += '-' + prefix[keyTmp];
    }
    else{
        otherProps += '-' + prefix[keyTmp] + value;
    }
  }

  // Set the Output  filename (<docId dir>/<docId>-<pageNr>.<res._Context.outputType>)
  jsonFilename = res._Context.docId + '/' + res._Context.docId + '-' + res._Context.pageNr  + otherProps + '-' + res._Context.outputType + '.json';
  return jsonFilename;  
}
*/

// V109 End

// V2.0.32
/**
 * Set the Log Level for the given logger object
 *
 * @param  {logger}   logger    Log4j logger
 * @param  {String}   level     level to set
 */
exports.log4jsSetLogLevel = function(logger, level) {
  level = level || "INFO"
  if (logger.setLevel)  // Log4j 1.1.1 version supports setLevel function.
      return logger.setLevel(level)
  logger.level = level  // Log4j >= 3.0.6 version supports level property.
}
