//------------------------------------------------------------------------------
// javaClass.js: Node.js module that contains APIs to call Java methods 
//				 from a java service either synchronously or asynchronously
// 
// Author    :  AFP2web Team
// Copyright :  (C) 2014 by Maas Holding GmbH
// Email     :  support@oxseed.de
// Version   :  V1.0.0
//
// History
//  V100        29.07.2014     Initial release
//   
//
// Todos:
//
//------------------------------------------------------------------------------
var fs = require('fs')
  , java = require("java")
  ;

/**
 * Constructor
 */
javaClass = function(className, callback){
	
	// We use try-catch block here since java.import does not have a callback function.
	try{
		// Import the class
		var importedClass = java.import(className);
		
		// Create a class instance
		var result = new importedClass();
		
		// Check Result
		if( result instanceof Error ){
			callback(result);
		}
		else{
			// result is a java class instance
			this._class = result;	
			callback(null, this);
		}
	} catch(err){
		callback(err);
	}
	
	return;
};

/**
 * callMethod: To call a Java method asynchronously
 * @Params:
 * name: 		String. Name of Java method to call
 * args: 		null | JSON Object.
 *             JSON Object containing arguments to be passed to Java method.
 *             If Java method does not have any parameters, null must be passed.
 * callback:   A callback function that would be called after the execution of Java method. 
 *             callback function will have two arguments.
 *                "err", of node.js type "Error"
 *                "result", of type JSON
 *             In case of success,
 *                "err" will be null.
 *                "result" a JSON Object containing result of java method execution.
 *             In case of error,
 *                "err" will have information about error occurred during method execution
 *                "result" will be null.
 */
javaClass.prototype.callMethod =  function(name, args, callback){
	var self = this;

	// Assert name
	if (name === null){
		callback( new Error("Invalid Java Method Name (null)"));
		return;
	}
	else if (name.length <=0){
		callback( new Error("Invalid Java Method Name (empty)"));
		return;
	}
	
	// We have a callback function associated with "self._class[name]"
	// This callback will be triggered only when method name is valid and after method execution is over.
	// If method name is not valid, this callback will not be triggered, that is why we use try-catch block here
	// to catch other exceptions than thrown by the method execution
	try{
		if (args === null){	// call the Java method without parameter
			
			self._class[name](function(err, result){
				if (err){
					callback(err);
				}
				else{
					var resultJson = JSON.parse(result);	// Convert result JSON String to JSON Object
					callback(null, resultJson);
				}
			});
		}
		else{						// call the Java method with parameter(s)
			self._class[name](JSON.stringify(args), function(err, result){
				if (err){
					callback(err);
				}
				else{
					var resultJson = JSON.parse(result);	// Convert result JSON String to JSON Object
					callback(null, resultJson);
				}
			});
		}
	}catch(err){
		callback(err);
	}
}

/**
 * callMethod: To call a Java method ssynchronously
 * @Params:
 * name: 		String. Name of Java method to call
 * args: 		null | JSON Object.
 *             JSON Object containing arguments to be passed to Java method.
 *             If Java method does not have any parameters, null must be passed.
 * @Returns:
 *   In case of success,
 *      JSON Object containing result of Java method
 *   In case of error,
 *      Error object containing information about errors occurred during method execution.
 */
javaClass.prototype.callMethodSync =  function(name, args){
	var self = this;

	// Assert name
	if (name === null){
		return new Error("Invalid Java Method Name (null)");
	}
	else if (name.length <=0){
		return new Error("Invalid Java Method Name (empty)");
	}
	
	var result = null;
	var syncName = name+"Sync";
	
	// We use try-catch block here since there are no callback function.
	try{
		if (args === null){
			result = self._class[syncName]();						// call the Java method without parameter
		}
		else{
			result = self._class[syncName](JSON.stringify(args));	// call the Java method with parameter(s)
		}
		if (!(result instanceof Error)){
			result = JSON.parse(result); 						// convert result JSON String to JSON Object
		}
	}catch(err){
		result = err;
	}
	return result;
}

module.exports = javaClass; 
