/**-----------------------------------------------------------------------------
 * httpUtils.js:  Node.js module that provides http utility functions
 * 
 * Author    :  AFP2web Team
 * Copyright :  (C) 2014 by Maas Holding GmbH
 * Email     :  support@oxseed.de
 * Version   :  V1.0.0
 * 
 * History
 *  V100   07.11.2014  Initial release
 *  V101   06.02.2015  Added postURL & processURL
 *  V102   25.02.2015  Added https support
 *
 *----------------------------------------------------------------------------*/
'use strict';

var fs      = require('fs')
  , http    = require('http')
  , https   = require('https')
  , url     = require('url')
  , request = require('request')
  , isHttps = /^https:/i  
  ;

/**
 * getURL:                  Get data from the passed URL
 * @param  {object|string}  options  http.request options, can be an object or a string
 * @param  {function}       callback(err, data)
 */
exports.getURL = function(options, callback){

  var req         = null
    , httptimeout = undefined
    ;

  try{
    // Set up the request
    if (isHttps.test(options)) {
      req = https.request(options);   
    }
    else{
      req = http.request(options);   
    }
    req.end();
  }
  catch(err){
    return callback(err);
  }

  if(options !== null && typeof options === 'object' && options.httptimeout) httptimeout = options.httptimeout;
  // Timeout handling
  if(httptimeout){
    req.setTimeout(httptimeout, function(){
        req.abort();
        // req.abort will trigger 'error' event which invokes callback(). 
        // So do not invoke callback() here
        // Refer to http://stackoverflow.com/questions/6214902/how-to-set-a-timeout-on-a-http-request-in-node
        //callback(new Error('Request timeout(' + httptimeout + ' ms). occurred.'));
    });
  }

  // Response handling
  req.on('response', function(resp){
    var data = '';  // response of a request as a concatenation of chunks
    resp.setEncoding('binary');

    resp.on('data', function(chunk){
      // Concatenate the chunks
      data += chunk;
    });

    // Once over, send the data through callback
    resp.on('end', function(){
      if (resp.statusCode === 200){ // 200 means everything went ok
        return callback(null, data);
      }
      else{
        return callback(new Error('getURL: Failed to get complete data. Data: ' + data));
      }     
    }); 
  });
  
  // Error handling 
  req.on('error', function(err){
    if(err.code === 'ECONNRESET')
      return callback(new Error('Request timeout(' + httptimeout + ' ms) occurred. Reason: ' + err.message));
    callback(err);
  });    
} 

/**
 * [postURL description]
 * @param  {object | string}  options  http.request options, can be an object or a string
 * @param  {object}           parms    parameters to be posted. Must be JSON
 * @param  {function}         callback(errr, data)
 */
exports.postURL = function(options, parms, callback){

  var body = JSON.stringify(parms)
    , req = null
    , httptimeout = undefined
    ;

  //!!! Check is options is a string (for ex.: localhost:84/services/version.html)

  options.method  = 'post';
  options.headers = {
    'Content-Type': 'application/json',
    'Content-Length': Buffer.byteLength(body)
  }

  try{
    // Set up the request
    if (isHttps.test(options)) {
      req = https.request(options);   
    }
    else{
      req = http.request(options);   
    }
   
    // Post the data
    req.write(body);
    req.end();
  }
  catch(err){
    return callback(err);
  }

  if(options !== null && typeof options === 'object' && options.httptimeout) httptimeout = options.httptimeout;
  // Timeout handling
  if(httptimeout){
    req.setTimeout(httptimeout, function(){
        req.abort();
        // req.abort will trigger 'error' event which invokes callback(). 
        // So do not invoke callback() here
        // Refer to http://stackoverflow.com/questions/6214902/how-to-set-a-timeout-on-a-http-request-in-node
        //callback(new Error('Request timeout(' + httptimeout + ' ms). occurred.'));
    });
  }

  // Response handling
  req.on('response', function(resp){
    var data = '';  // response of a request as a concatenation of chunks
    resp.setEncoding('binary');

    resp.on('data', function(chunk){
      // Concatenate the chunks
      data += chunk;
    });

    // Once over, send the data through callback
    resp.on('end', function(){
      if (resp.statusCode === 200){ // 200 means everything went ok
        return callback(null, data);
      }
      else{
        return callback(new Error('postURL: Failed to get complete data. Data: ' + data), null);
      }     
    }); 
  });
  
  // Error handling 
  req.on('error', function(err){
    if(err.code === 'ECONNRESET')
      return callback(new Error('Request timeout(' + httptimeout + ' ms) occurred. Reason: ' + err.message));
    callback(err);
  });  
} 

/**
 * Service to setup a http(s) Request
 * @param  {object | string}  options  http.request options, can be:
 *                                       protocol: Protocol to use. Defaults to 'http:'.
 *                                       hostname: To support url.parse() hostname is preferred over host
 *                                       port: Port of remote server. Defaults to 80.
 *                                       method: A string specifying the HTTP request method (GET,POST,PUT,HEAD); defaults to 'GET'.
 *                                       path: Request path. Defaults to '/'. 
 *                                       headers: An object containing request headers.
 *                                       ...and more, see https://nodejs.org/docs/latest-v0.12.x/api/http.html#http_http_request_options_callback
 * @param  {JSON}             parms    parameters to be posted. Must be JSON
 * @param  {function}         callback(err, result)
 *
 *
 * TODO: Why parms must be JSON ?
 * TODO: How to pass the timeout value ?
 * TODO: Fix resp.setEncoding('binary'); // !!! This should be passed from the Request. Default could be binary
 */
exports.processURL = function(options, parms, callback){

  var body        = undefined
    , req         = null
    , httptimeout = undefined
    ;

  // Any parm specified?
  if (typeof(parms) === "function"){
    callback = parms;
    parms = undefined;
  }

  // If options is a string, parse it (for ex.: http://localhost:84/services/version.html)
  if (typeof options === 'string') options = url.parse(options);

  // Is options specified?
  if(options === undefined){  // no go!
    return callback(new Error('processURL: options is undefined'));
  }

  // Set get as default method
  options.method === options.method || 'get';

  // Lowercase the method name
  options.method = options.method.toLowerCase();

  // Add Content-Type as json if parms is specified
  if(parms){
    body = JSON.stringify(parms);
    options.headers = options.headers || {'Content-Type':'application/json','Content-Length': Buffer.byteLength(body)};
  }

  try{
    // Set up the request
    if(isHttps.test(options.protocol)) req = https.request(options);   
    else req = http.request(options);   

    // Write the body if specified
    if(body) req.write(body);

    // Flush the request
    req.end();
  }
  catch(err){
    return callback(err);
  }

  if(options.httptimeout) httptimeout = options.httptimeout;
  // Timeout handling
  if(httptimeout){
    req.setTimeout(httptimeout, function(){
        req.abort();
        /*
           req.abort will trigger 'error' event which invokes callback(). 
           So do not invoke callback() here
           Refer to http://stackoverflow.com/questions/6214902/how-to-set-a-timeout-on-a-http-request-in-node
         */
    });
  }

  // Response handling
  req.on('response', function(resp){
    var data = '';  // response of a request as a concatenation of chunks
    resp.setEncoding('binary'); // !!! This should be passed from the Request. Default could be binary

    resp.on('data', function(chunk){
      data += chunk; // concatenate the chunks
    });

    // Once over, send the data through callback
    resp.on('end', function(){
      // 200 means everything went ok
      if (resp.statusCode === 200) return callback(null, data);
      return callback(new Error('processURL: Failed to get complete data. Data: ' + data));
    }); 
  });
  
  // Error handling 
  req.on('error', function(err){
    if(err.code === 'ECONNRESET')
      return callback(new Error('Request timeout(' + httptimeout + ' ms) occurred. Reason: ' + err.message));
  });  
} 

/**
 * uploadFileAsBuffer upload of a buffer
 * @param  {json}     options
 *                    options ={
 *                      hostname:     "<hostname>",
 *                      [port:        "<port>",]
 *                      method:       "<method>, POST or PUT"
 *                      [path:        "<URI>"],
 *                      [httptimeout:        "<Request timeout in milliseconds>"],
 *                    }
 * @param  {Buffer}   body            "<buffer to be uploaded>",
 * @param  {Function} callback
 */
exports.uploadFileAsBuffer = function(options, body, callback){

  var req         = undefined
    , httptimeout = undefined
    ;

  // If options is a string, parse it (for ex.: http://localhost:1029/services/oxsnps-.../version)
  if (typeof options === 'string'){
    options = url.parse(options, true, true);
  }
  else{
    // Encode Path (i.e. escape special chars like space)
    options.path = encodeURI(options.path);
  }

/*!!! needed ?
  options.headers = {
    'Content-Length': Buffer.byteLength(body)
  }
*/
  try{
    // Set up the request
    if(isHttps.test(options)) req = https.request(options);   
    else req = http.request(options);   
   
    // Post the data
    req.write(body);
    req.end();
  }
  catch(err){
    return callback(err);
  }

  if(options !== null && typeof options === 'object' && options.httptimeout) httptimeout = options.httptimeout;
  // Timeout handling
  if(httptimeout){
    req.setTimeout(httptimeout, function(){
        req.abort();
        // req.abort will trigger 'error' event which invokes callback(). 
        // So do not invoke callback() here
        // Refer to http://stackoverflow.com/questions/6214902/how-to-set-a-timeout-on-a-http-request-in-node
        //callback(new Error('Request timeout(' + httptimeout + ' ms). occurred.'));
    });
  }

  // Response handling
  req.on('response', function(resp){
    var data = '';  // response of a request as a concatenation of chunks
    //!!!resp.setEncoding('binary');

    resp.on('data', function(chunk){
      // Concatenate the chunks
      data += chunk;
    });

    // Once over, send the data through callback
    resp.on('end', function(){
      if(resp.statusCode === 200) return callback(null, data); // 200 means everything went ok
      return callback(new Error('postURL: Failed to upload buffer. Data: ' + data), resp);
    }); 
  });
  
  // Error handling 
  req.on('error', function(err){
    if(err.code === 'ECONNRESET')
      return callback(new Error('Request timeout(' + httptimeout + ' ms) occurred. Reason: ' + err.message));
    callback(err);
  });
} 

exports._uploadFile = function(options, parms, callback){

  var formData = {}
    , url = 'http://' + options.hostname + (options.port ? ':' + options.port : '') + (options.path ? options.path : '/');
  console.log('uploadFile: url: ' + url + ',  formData:' + JSON.stringify(formData));

  // Assert buffer
  if(!parms.buffer){
    return callback(new Error('parms.buffer undefined.'));
  }

  // Assert parms
  parms.filename    = parms.filename || 'unknown_file';
  parms.contentType = parms.contentType || 'application/pdf';

  if(options.method === 'POST'){
    // Post the request
    var req = request.post({'url':url, 'formData': formData}, function(err, httpResponse, data) {
      if (err) {
        return callback(err);
      }
      //console.log('uploadFile: Upload successful! Server responded:', data);
      return callback(null, data);
    });

    // Add buffer to the form data (see http://stackoverflow.com/questions/25344879/uploading-file-using-post-request-in-node-js)
    var form = req.form();
    form.append('file', new Buffer(new Uint8Array(parms.buffer)), {filename: parms.filename, contentType: parms.contentType});  
  }
  else if(options.method === 'PUT'){
    // Put the request
    var req = request.put({'url':url, 'formData': formData}, function(err, httpResponse, data) {
      if (err) {
        return callback(err);
      }
      //console.log('uploadFile: Upload successful! Server responded:', data);
      return callback(null, data);
    });

    // Add buffer to the form data (see http://stackoverflow.com/questions/25344879/uploading-file-using-post-request-in-node-js)
    var form = req.form();
    form.append('file', new Buffer(new Uint8Array(parms.buffer)), {filename: parms.filename, contentType: parms.contentType});  
  }
  else{
    return callback(new Error('HTTP method undefined: ' + options.method));
  }
}

/**
 * !!! NOT COMPLETED !!! has to be checked before being used
 * 
 * postMultipartRequest 
 * @param  {json}       props    
 *                      props = {
 *                        filename:     "<filename>",
 *                        buffer:       "<file buffer>",
 *                        contentType:  "application/pdf",
 *                        encoding:     "binary",
 *                        hostname:     "<hostname>",
 *                        port:         "<port>",
 *                        method:       "<method>"
 *                        path:         "<URI>"
 *                      }
 * @param  {Function}   callback
 */
exports.postMultipartRequest = function(props, callback){

  // Build the POST Req Parameters
  var crlf = "\r\n",
    //boundary = '---------------------------9r-SzFGlBRvRAc8RJbwAJ4i-Idwoujo2F-EPQ6v', // Boundary: "--" + up to 70 ASCII chars + "\r\n"
    min = 100000000000000,
    max = 1000000000000000,
    //boundary = '---------------------------' + Math.floor(Math.random() * (max - min + 1) + min),
    boundary = 'oswhTEndxHsTZddqkYwAg6-sbh6HP0e5R',
    delimiter = crlf + "--" + boundary,
    preamble = "", // ignored. a good place for non-standard mime info
    epilogue = "", // ignored. a good place to place a checksum, etc
    
    bufferHeader = [
      'Content-Disposition: form-data; name="InputBuffer"; filename="' + props.filename + '"' + crlf,
      //'Content-Type: application/octet-stream' + crlf,
      'Content-Type: ' + props.contentType + crlf,
      'Content-Transfer-Encoding: ' + props.encoding + crlf,
    ],
    
    optionHeader1 = [
      'Content-Disposition: form-data; name='
    ],
    optionHeader2 = [
      'Content-Type: text/plain; charset=US-ASCII' + crlf,
      'Content-Transfer-Encoding: 8bit' + crlf,
    ],            
    closeDelimiter = delimiter + "--" + crlf, // crlf is VERY IMPORTANT here
    
  // Here is how the multipartBody MUST be built 
  // multipartBody  = preamble
  //          + 1..n[delimiter + crlf + optionHeader1 + optionName + crlf + optionHeader2 + crlf + optionValue]
  //          + delimiter + crlf + bufferHeader + crlf + InputBuffer
  //          + closeDelimiter + epilogue
  // Concatenate preamble
  multipartBody = Buffer.concat([
    new Buffer(preamble)
  ]);
      
  // Concatenate InputBuffer...if provided
  if (props.buffer){
    multipartBody = Buffer.concat([
      multipartBody,
      new Buffer(delimiter + crlf + bufferHeader.join('') + crlf),
      props.buffer
    ]);
  }

  // Concatenate closeDelimiter + epilogue
  multipartBody = Buffer.concat([
    multipartBody,
    new Buffer(closeDelimiter + epilogue)
  ]); 
          
  // Set the POST Req Options
  var post_options = {
    hostname: props.hostname,
    port:     props.port,
    path:     props.path,
    method:   props.method,
    headers:  {
      'Content-Type': 'multipart/form-data; boundary=' + boundary,
      'Content-Length': multipartBody.length
    }
  };


  // Set up the request
  var post_req = http.request(post_options);

  // Post the data
  post_req.write(multipartBody);
  post_req.end();

  // V101 Begin
  if( props.path.indexOf( 'async') >= 0 ){
    // If it is a async request, do not wait for response and return immediately 
    return callback(null);
  }
  // V101 End
  else{
    // Response handling
    post_req.on('response', function(res) {

      var response = '';        // response of a request as a concatenation of chunks
      res.setEncoding('binary');
      res.on('data', function (chunk) {
        // Concatenate the chunks
        response += chunk;
      });

      // Call the callback function passing the response
      res.on('end', function () {
        if (res.statusCode === 200){ // HTTP status code (200, 404, 500, ...), 200 means everything went ok
          return callback(null, response);
        }
        else{
          return callback(response);
        }
      }); 
    });
    
    // Error handling 
    post_req.on('error', function(error) {
      return callback(error);
    }); 
  }         
}
