/**-----------------------------------------------------------------------------
 * index.js: 	Node.js module to handle request processing
 * 
 * Author    :  AFP2web Team
 * Copyright :  (C) 2014 by Maas Holding GmbH
 * Email     :  support@oxseed.de
 * Version   :  V1.0.0
 * 
 * History
 *  V100   09.08.2014  Initial release
 *  V101   26.08.2014  Changes done
 *                     1. Ensure 'init' function modified to return only after 
 *                        initialization is over through 'callback' function
 *                     2. Extended to include version information about MSOffice 
 *                        Converter Wrapper service and PDF Converter Wrapper
 *                        service in response to version request 
 *                     3. Added a 'quit' function to handle 'quit' request. 
 *                        'quit' function will take care of stopping java services. 
 *                     4. Implemented MSofficeConverter service
 *                     5. Implemented PDF MergeAndSecure service
 *                     6. Implemented A2W Async. Transformation service 
 *                     7. Implemented Job Status request service
 *                     8. Extended to send proper SOAP response in case of errors
 *                     9. In case of A2W Server error, parsed error html response 
 *                        to get error information and built error SOAP response 
 *                        for sync. and async. transformations.
 *  V102   26.01.2015	Added the pluginManager services                        
 *
 *----------------------------------------------------------------------------*/
'use strict';

var MODULE_NAME 	= 'core'
  , MODULE_VERSION 	= '2.0.0'
  , fs 				= require('fs')
  , path 			= require('path')
  , log4js 			= require('log4js')
  ,	async 			= require('async')
  , util 			= require('util')
  , cheerio 		= require('cheerio')	// to parse HTML and get to get value of tags
  , url     		= require('url')
  , npsServer		= require('./server')  
  , npsDir			= npsServer.npsDir
  , npsConf 		= npsServer.npsConf		// holds the configuration of the Server 
  , npsTempDir 		= npsConf.tempDir
  , npsLogFile 		= npsServer.npsLogFile
  , pluginManager 	= require('./pluginManager')
  , cronjob 		= require('./cronjob')
  , utils 			= require('./helpers/utils')
  , httpUtils		= require('oxsnps-core/helpers/httpUtils')
  , expressAppUtils = require('./helpers/expressAppUtils')
  , deploy 			= require('./deploy')
  , backup 			= require('./backup')
//!!!  , mochatest	= require('./mochatest')
  , pluginJSfilename= 'oxsnps.js'
  // Add any core service here as '/appservices/<coreModule>/<service>'
  , coreModules = [
		{	// PluginManager services
			handle: 	pluginManager,
			name: 		'Plugin Manager',
			baseRoute: 	'/appservices/pm',
			routes: [
				{method: 'get', 	path: '/appservices/pm/version', 			service: 'version'}, 
				{method: 'get', 	path: '/appservices/pm/routes', 			service: 'getRoutes'},
				{method: 'get', 	path: '/appservices/pm/enable',				service: 'enablePlugin'},
				{method: 'post', 	path: '/appservices/pm/enable', 			service: 'enablePlugin'},
				{method: 'get', 	path: '/appservices/pm/load',				service: 'loadPlugin'},
				{method: 'post', 	path: '/appservices/pm/load', 				service: 'loadPlugin'},		
				{method: 'get', 	path: '/appservices/pm/disable',			service: 'disablePlugin'},
				{method: 'post', 	path: '/appservices/pm/disable', 			service: 'disablePlugin'},				
				{method: 'get', 	path: '/appservices/pm/enableRoute',		service: 'enableRoute'},
				{method: 'post', 	path: '/appservices/pm/enableRoute',		service: 'enableRoute'},
				{method: 'get', 	path: '/appservices/pm/disableRoute',		service: 'disableRoute'},
				{method: 'post', 	path: '/appservices/pm/disableRoute',		service: 'disableRoute'},
				{method: 'get', 	path: '/appservices/pm/getloglevel',		service: 'getLogLevel'},
				{method: 'get', 	path: '/appservices/pm/setloglevel',		service: 'setLogLevel'}
			]
		},
		{	// Cronjob services
			handle: 	cronjob,
			name: 		'Cron Job',
			baseRoute: 	'/appservices/jobs',
			routes: [
				{method: 'get', 	path: '/appservices/jobs/list', 			service: 'list'},
				{method: 'post', 	path: '/appservices/jobs/add', 				service: 'add'},	//!!! NOT IMPLEMENTED
				{method: 'post', 	path: '/appservices/jobs/update', 			service: 'update'},	//!!! NOT IMPLEMENTED
				{method: 'post', 	path: '/appservices/jobs/delete', 			service: 'delete'},
				{method: 'post', 	path: '/appservices/jobs/start', 			service: 'start'},
				{method: 'post', 	path: '/appservices/jobs/stop', 			service: 'stop'},
				{method: 'post', 	path: '/appservices/jobs/stopall', 			service: 'stopAll'},
				{method: 'post', 	path: '/appservices/jobs/exists', 			service: 'exists'},
				{method: 'get', 	path: '/appservices/jobs/getloglevel', 		service: 'getLogLevel'},
				{method: 'get', 	path: '/appservices/jobs/setloglevel',		service: 'setLogLevel'}
			]
		},
		{	// Deploy services
			handle: 	deploy,
			name: 		'Deploy',
			baseRoute: 	'/appservices/deploy',
			routes: [
				{method: 'get', 	path: '/appservices/deploy/form', 			service: 'displayform'},
				{method: 'get', 	path: '/appservices/deploy/list', 			service: 'list'},
				{method: 'post', 	path: '/appservices/deploy/upload', 		service: 'process'},
				{method: 'get', 	path: '/appservices/deploy/getloglevel', 	service: 'getLogLevel'},
				{method: 'get', 	path: '/appservices/deploy/setloglevel',	service: 'setLogLevel'}
			]
		},
		{	// Backup services
			handle: 	backup,
			name: 		'Backup',
			baseRoute: 	'/appservices/backup',
			routes: [
				{method: 'get', 	path: '/appservices/backup/getloglevel', 	service: 'getLogLevel'},
				{method: 'get', 	path: '/appservices/backup/setloglevel',	service: 'setLogLevel'},
				{method: 'get', 	path: '/appservices/backup?*', 				service: 'backup'}
			]
		},
		{	// Cache services
			handle: 	backup,
			name: 		'Cache',
			baseRoute: 	'/appservices/cache',
			routes: [
				{method: 'get', 	path: '/appservices/cache/getloglevel', 	service: 'getLogLevel'},
				{method: 'get', 	path: '/appservices/cache/setloglevel',		service: 'setLogLevel'}
			]
		}
	/*!!!
		},
		{	// Mocha Test services
			handle: 	mochatest,
			name: 		'Test',
			baseRoute: 	'/appservices/test',
			routes: [
				{method: 'get', 	path: '/appservices/test/getloglevel', 		service: 'getLogLevel'},
				{method: 'get', 	path: '/appservices/test/setloglevel',		service: 'setLogLevel'},
				{method: 'get', 	path: '/appservices/test/plugin', 			service: 'pluginTest'}
			]
		}
	!!!*/
	]
  ;

// Get Logger
var logger = log4js.getLogger(MODULE_NAME);
logger.setLevel(npsConf.log.level);

// Export Module Version
exports.version = MODULE_VERSION;

/**
 * init: Initializes Server services 
 * @Params:
 * callback    :  A callback function that would be called after initialization is over
 *                callback function will one argument "err", of node.js type "Error"
 *                In case of success,
 *                    "err" will be null.
 *               In case of error,
 *                  "err" will have information about intialization error
 */
exports.init = function(callback){
	
	logger.debug(__filename, 'init: Initializing routes...');

	var self 	= this
	  , app 	= npsServer.expressApp
	  , options = {}
	  ;

	// Set up default parms used by cleanTempDir
	if(npsConf.tempDirProps){
		// Set up the cron job options
		npsConf.tempDirProps.cronTime 	= npsConf.tempDirProps.cronTime || '00 00 * * * *'; // Run cron at 0th min of every hour

		// Set up options
		npsConf.tempDirProps.interval 	= npsConf.tempDirProps.interval || 30;  // delete files older than 30 mins
		npsConf.tempDirProps.unit 		= npsConf.tempDirProps.unit ||'n';   // n => minutes
	}
	else{
		// Set up default options
		npsConf.tempDirProps = {'cronTime': '00 00 * * * *', 'interval': 30, 'unit': 'n'};
	}

	// Clean the temp dir
	exports.cleanTempDir(function(err){
		// Error already logged by cleanTempDir. Ignore the error since we are deleting temp. directories
		//if(err) return callback(err);	

		// Initialize the cronjob module
		cronjob.initialize(function(err){
			if(err) return callback(err);

			// Set up a Cron Job to clean the temp dir
			options.name         = npsTempDir;		
			options.cronFunction = self.cleanTempDir;
			self.function 		 = 'oxsnps-core/index.js->cleanTempDir'; // (Optional) a String to identify the cronFunction
			options.cronOptions  = {'start': true, 'context': self};
			options.cronTime     = npsConf.tempDirProps.cronTime;

			// Add the Cron job
			logger.debug('--->_createAndStartCronJob: Creating cron job to clean the temp directory (' + npsTempDir + ')...');
			cronjob.addCronJobService(options, function(err){
				if(err) return callback(err);
			});

			// Enable the core services
			async.each(coreModules,
				function(coreModule, nextTask){
					expressAppUtils.enableRoutes(coreModule.handle, coreModule.routes, function(err){
						if(err){
							nextTask(err);
						}
						nextTask();
					});
				},
				function(err){
					if(err){
						logger.error(__filename, 'init: Initializing core services failed, Reason: ' + err.message);
						return callback(err);
					}
				}
			);

			// Enable plugins
			pluginManager.enablePlugins(function(err){
				if(err) return callback(err);
				pluginManager.activatePlugins(callback);
			});
		});
	});
};

// Process the default Request (i.e. localhost/)
exports.main = function(req, res){
	logger.info('main:\nreq: ' + req.url + '\nheaders:' + JSON.stringify(req.headers)  + '\nbody:'+ JSON.stringify(req.body));
	return _getVersion(res);
}

// Process a ping request (i.e. localhost/ping)
exports.ping = function(req, res){
	res.end();
}

/**
 * getVersion: Returns the Server version the version of its components
 * @Params: 
 *    req: Request Object
 *    res: Response Object 
 */
exports.getVersion = function(req, res){
	return _getVersion(res);
}

/**
 * isAlive: Check whether the Servers behing the plugins are still alive
 * @Params: 
 *    req: Request Object
 *    res: Response Object 
 */
exports.isAlive = function(req, res){
	return _isAlive(res);
}

// Returns Server History
exports.getHistory = function(req, res) {

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);

	logger.info('--->History Req. id=' + res._Context.reqId + ', Server History requested');
	fs.createReadStream(npsDir + '/' + 'history.txt').pipe(res);
	logger.info('--->History Req. id=' + res._Context.reqId + ', Server History returned');
}

/**
 * getRoutes 			Returns the list of core services
 * @param  {Request} 	req Express request
 * @param  {Response} 	res Express response
 */
exports.getRoutes = function(req, res) {

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);

	logger.info('--->getRoutes Req. id=' + res._Context.reqId + ', List of Core Services requested');
	res.json(coreModules);
	logger.info('--->getRoutes Req. id=' + res._Context.reqId + ', List of Core Services returned');
}

/**
 * Set the log level of this module
 * @param  {req} 	GET req   http://localhost:1026/appservices/deploy/setloglevel?loglevel=DEBUG|INFO|ERROR
 * @param  {res}	res
 */
exports.setLogLevel = function(req, res){
	var loglevel = req.query.loglevel || 'INFO';
	logger.setLevel(loglevel);
	return res.end('--->setLogLevel: Log level of ' + MODULE_NAME + ' set to ' + loglevel);
}

/**
 * Return the log level of this module
 * @param  {req} 	GET req   http://localhost:1026/appservices/deploy/getloglevel
 * @param  {res}	res
 */
exports.getLogLevel = function(req, res){
	logger.info('--->getLogLevel: Getting the log level of ' + MODULE_NAME + '...');
	return res.end(logger.level.levelStr);
}

/**
 * Called from the cron Job to clean the temp directory
 */
exports.cleanTempDir = function(callback){

	var self = this;

	logger.info('--->cleanTempDir: Cleaning the temp directory (' + npsTempDir + ')...');

	// Delete temp dirs and files that are older than 30mns
	utils.deleteDir(npsTempDir, false /* do not delete the root dir */, npsConf.tempDirProps.unit /* mns */, npsConf.tempDirProps.interval, true /* recursively delete dir entries */, function(err){
		if(err){
			logger.error('--->cleanTempDir: Cleaning the temp directory (' + npsTempDir + ') failed: ' + err.message);
		} 
		else{
			logger.info('--->cleanTempDir: Temp directory (' + npsTempDir + ') cleaned.');
		}
		if(callback) callback(err);
	});
}

/**
 * quit: Terminate Server
 * @Params: 
 *    req: Request Object
 *    res: Response Object 
 */
exports.quit = function(req, res) {

	logger.debug(__filename, 'quit: Leaving Server...');
	logger.info('--->Quit Req.: Leaving Server...');

	// Finalize the cronjob module
	cronjob.finalize(function(err){	
		if(err){
			logger.error(__filename, 'quit, Error: ' + err);
		}

		pluginManager.disablePlugins(function(err){
			// Build Response
			var quitMsg = null;
			if (err){
				logger.error(__filename, 'quit, Error: ' + err);
				quitMsg = 'Error on closing the server. ' + err
			}else{
				quitMsg = 'OXSNPS Server quit successfully.';
				logger.info('--->Quit Req.: ' + quitMsg);
			} 

			// Send  Response
			res.end('<html><head><title>OXSNPS Server</title></head>' +
					'<body><br><h2>'+ quitMsg + '</h2></body></html>'
					);

			// Terminate the process
			process.exit(0);
			return;
		});
	});
}

/**
 * Return the log level of this module
 * @param  {req} 	GET req   http://<server>:<port>/http?parm={"method":"post","url": "http://locahost:1030/postkorb/online/postkorb/services/pk.frombpm/existGeVo","body": "{\"geVoTypKey\": 2,\"ordnungsBegriffe\":[{\"definitionId\":901,\"value\":\"67\",\"sequenceNo\":1,\"uiPriority\":1}]}"}
 * @param  {res}	res
 */
exports.processHttpReq = function(req, res){

	var parm 	= undefined
	  , options = undefined
	  , helpmsg	= 'This API expects following parameters:<br>' +
	  			  '<br>parm={"method":"&lt;method&gt;","url":"&lt;url&gt;","body": "{&lt;body&gt;}<br>' +
	  			  '<br>where:' +
	  			  '<br>\t\t"&lt;method&gt;":\t\t get|post|put|head (a http verb; optional, defaults to get)' + 
	  			  '<br>\t\t"&lt;url&gt;":\t\t the url to call' + 
	  			  '<br>\t\t"&lt;body&gt;":\t\t the body of the request; optional' + 
	  			  '<br><br>Examples:<br>' + 
	  			  '\t\tGET http://&lt;hostname&gt;:&lt;port&gt;/http?parm={"method":"post","url":"http://locahost:1030/...",' + 
	  			  '"body":{&lt;some JSON object&gt;}}<br><br>'
	  ;
	logger.info('--->processHttpReq: Processing req.query: ' + JSON.stringify(req.query));

	if(Object.keys(req.query).length === 0){
		logger.warn('--->processHttpReq: Wrong Request, the parm parameter is missing, leaving...');
		return res.end('<html><body>--->processHttpReq: Wrong Request, the "parm" parameter is missing!<br><br>' + helpmsg + '</body></html>');
	}

	logger.info('--->processHttpReq: Processing req.query.parm: ' + JSON.stringify(req.query.parm));
	parm = utils.parseJSON(req.query.parm);
	if(parm === null){
		logger.warn('--->processHttpReq: Wrong Request, the "parm" parameter is missing, leaving...');
		return res.end('<html><body>--->processHttpReq: Wrong Request, the "parm" parameter is missing!<br><br>' + helpmsg + '</body></html>');
	}

	logger.info('--->processHttpReq: parm as JSON Object: ' + JSON.stringify(parm));
	if(parm.url === undefined){
		logger.warn('--->processHttpReq: Wrong Request, the "url" parameter is missing, leaving...');
		return res.end('<html><body>--->processHttpReq: Wrong Request, the "url" parameter is missing!<br><br>' + helpmsg + '</body></html>');
	}

	// Ensure parm.method is define
	parm.method = parm.method || 'get' 

	logger.info('--->processHttpReq: method:' + parm.method.toLowerCase() + ', url:' + parm.url + ', body:' + (parm.body? JSON.stringify(parm.body) : 'not passed'));
	/*
		if(parm.method.toLowerCase() === 'post' && !parm.body){
			logger.warn('--->processHttpReq: Wrong Request, if the post method is used, body must be passed, leaving...');
			return res.end('<html><body>--->processHttpReq: Wrong Request, if the post method is used, body must be passed!<br><br>' + helpmsg + '</body></html>');
		}
	 */	

	// Parse the URL
	options = url.parse(parm.url);
	if(options === undefined){
		logger.warn('--->processHttpReq: Wrong Request, the given "url" parameter (' + parm.url + ') is not a valid url, leaving...');
		return res.end('<html><body>--->processHttpReq: Wrong Request, the given "url" parameter (' + parm.url + ') is not a valid url!<br><br>' + helpmsg + '</body></html>');
	}
	logger.info('--->processHttpReq: url as JSON Object: ' + JSON.stringify(options));

	// Add the method to options
	options.method = parm.method

	// Execute the HTTP(s) request
	httpUtils.processURL(options, parm.body, function(err, data){
		if(err){
			logger.error('--->processHttpReq: Executing HTTP request with options:' + JSON.stringify(options) +
						 ' and parms:' + JSON.stringify(parm.body) + ' failed, Error: ' + err.message);
			return res.end('<html><body>Executing HTTP request with options:' + 
						   JSON.stringify(options) + ' failed, Error: ' + err.message + '</body></html>');

		}
		logger.info('--->processHttpReq: req.query: ' + JSON.stringify(req.query) + ' successfully processed. ' +
					'httpUtils.processURL returned ' + data.length + ' bytes');						 
		return res.end(data); 
	});
}

/******************  Private Functions ********************/
function _isAlive(res){

	var data 	= ''
	  , plugins = pluginManager.getPluginsList()
	  , module 	= null
	  , bError 	= false
	  ;

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);

	logger.info('--->isAlive. Req. id=' + res._Context.reqId);

	// Assert plugins parameter
	if(plugins === undefined || plugins === null|| (typeof plugins !== "object") || (!util.isArray(plugins)) || (plugins.length <= 0)){
		logger.info('--->isAlive. Req. id=' + res._Context.reqId + ', over.');
		return res.status(404).end('No Plugin available.');
	}

	// Loop through all the plugins
	async.each(plugins,
		function(plugin, nextTask){
			if(plugin.enable === undefined || plugin.enable !== 'on'){
        		data += plugin.module + ' not enabled\n';
				nextTask();
			}
			else{
				//  Get plugin version
				logger.debug(__filename, '_isAlive: Getting ' + plugin.module + ' version...');

				try{
					// Get the module handle
					module = require(path.resolve(plugin.plugindir) + '/' + plugin.module + '/' + pluginJSfilename);

					if (typeof(module.isAlive) === "function"){
						// Call isAlive
						module.isAlive(function(err, result){
				        	if (err){
								logger.error(__filename, '_isAlive, ' + plugin.module + ' Version Error: ' + err.message);
				        		data += plugin.module + ' Error: ' + err.message + '\n';
								bError = true;
				        	}
				        	else{
				        		data += plugin.module + ' v' + result + '\n';
				        	}
							nextTask();
						});
					}
					else{				
						// Get the version of the module
						module.getVersion(function(err, result){
				        	if (err){
								logger.error(__filename, '_isAlive, ' + plugin.module + ' Version Error: ' + err.message);
				        		data += plugin.module + ' Error: ' + err.message + '\n';
								bError = true;
				        	}
				        	else{
				        		data += plugin.module + ' v' + result + '\n';
				        	}
							nextTask();
						});
					}
				}catch(err){
					logger.error(__filename, '_isAlive, ' + plugin.module + ' Version Error: ' + err.message);
	        		data += plugin.module + ' Error: ' + err.message + '\n';
					bError = true;
					nextTask();
				}
			}
		},
		function(err){
			logger.debug(__filename, '_isAlive: Plugin Info:\n' + data);
	        if (bError){
				logger.error('--->isAlive. Req. id=' + res._Context.reqId + ', at least one Component is not alive.');
				return res.status(500).end('--->isAlive. Req. id=' + res._Context.reqId + ' Error: At least one Component is not alive:\n' + data);
			}
			logger.info('--->isAlive. Req. id=' + res._Context.reqId + ', over.');
			res.end('All the Components are alive:\n' + data);
		}
	);
}

function _getVersion(res){

	var data 	= ''
	  , plugins = pluginManager.getPluginsList()
	  , module 	= null
	  ;

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);

	// Add the HTML file to server to the Context
	res._Context.htmlFile = '/public/index.html';

	logger.info('--->Ver. Req. id=' + res._Context.reqId + ', Servers Version required');

	// Assert plugins parameter
	if(plugins === undefined || plugins === null|| (typeof plugins !== "object") || (!util.isArray(plugins)) || (plugins.length <= 0)){
		logger.info('--->Ver. Req. id=' + res._Context.reqId + ', over.');
		// Send back HTTP Response 
		return _serveHTML(res, 'Plugins are not available.');
	}

	// Loop through all the plugins
	async.each(plugins,
		function(plugin, nextTask){
			if(plugin.enable === undefined || plugin.enable !== 'on'){
				nextTask();
			}
			else{
				//  Get plugin version
				logger.debug(__filename, '_getVersion: Getting ' + plugin.module + ' version...');

				try{
					// Get the module handle
					module = require(path.resolve(plugin.plugindir) + '/' + plugin.module + '/' + pluginJSfilename);

					// Get the version of the module
					module.getVersion(function(err, result){
			        	if (err){
							logger.error(__filename, '_getVersion,' + plugin.module + ' Version Error: ' + err);
							data += '<h3>' +  plugin.module + ' Version Error: ' + err + '</h3>';
			        	}
			        	else{
							// result => v1.0.x.x
							logger.debug(__filename, '_getVersion: ' + plugin.module + ' Version: ' + result);
							data += '<h3>' +  plugin.module + ' v' + result + '</h3>';
						}
						nextTask(err);
					});
				}catch(err){
					nextTask(err);
				}
			}
		},
		function(err){
	        if (err){
				logger.error(__filename, '_getVersion: Error: ' + err);
				logger.info('--->Ver. Req. id=' + res._Context.reqId + ', over.');
				return _serveHTML(res, 'Error when retrieving a Server Version: ' + err.message);
			} 
			logger.info('--->Ver. Req. id=' + res._Context.reqId + ', Servers Version returned');

			// Send back HTTP Response 
			_serveHTML(res, data);
		}
	);
}

function _serveHTML(res, body){

	// Read the HTML file, set some nps... vars, and send back the modified HTML
	fs.readFile(npsDir + res._Context.htmlFile, 'utf8', function(err, html){
		if(err){
			return res.status(404).end(npsDir + res._Context.htmlFile + ' not found!' );
		}
		html = html.replace(/&npsVersion;/g, npsServer.npsVersion);	// replace version
		html = html.replace(/&npsName;/g, npsServer.npsDesc);		// replace description
		html = html.replace(/&npsBody;/g, body);					// replace body
		res.end(html);
	});
}
