//------------------------------------------------------------------------------
// js2Java.js: 	Node.js module to manage Java Virtual Machines (JVM) 
//				and helps in getting Java class handles.
// 
// Author    :  AFP2web Team
// Copyright :  (C) 2014 by Maas Holding GmbH
// Email     :  support@oxseed.de
// Version   :  V1.0.0
//
// History
//  V100   29.07.2014  Initial release
//  V101   01.08.2014  Removed packages parameter from js2Java.initialize() 
//
// Todos:
//
//------------------------------------------------------------------------------
var fs = require('fs')
  , java = require("java")
  , util = require('util')
  , javaClass = require('./javaClass.js')
  , log4js = require('log4js')  
  ;
/*
var logger = log4js.getLogger('js2Java');
var loggerConfigLevel = config.otsConf.otsServer.logLevel || 'INFO';
logger.setLevel(loggerConfigLevel);
*/
exports.initialized = false;

/**
 * initialize: Initialize the Java Virtual Machine (JVM)
 * @Params:
 * classpath:  String Array. An array of class paths where JVM can find the required jar and class files.
 * jvmOptions: String Array. Any options that need to be passed to JVM (Example: MaxMemory to be used by JVM)
 *                           Optional.  Can be null or an empty array.
 
 * callback:   A callback function that would be called after initializing the JVM.
 *             callback function will have one argument "err" of node.js type "Error".
 *             In case of success, "err" will be null.
 *             In case of  error, "err" will have information about error occurred during initialization.
 */
exports.initialize =  function(classpath, jvmOptions, callback){
	var self = this
	  , result = null;
	
	// Check if already initialized
	if(self.initialized !== undefined && self.initialized === true ){
		callback(result);
		return;
	};
	
	// Set Classpath
	result = _setClasspath(classpath);

	// Set JVM Options
	if( !(result instanceof Error) ){
		result = _setOptions(jvmOptions);
	}
	
	if( !(result instanceof Error) ){
		self.initialized = true;
	}
	
	// call passed callback function with result
	callback(result);
}

/**
 * createClass: Create a Java Class
 * @Params:
 * className: String, name of the Java class with complete package name for which the instance has to be created.
 *                    Example: de.maas.pdfengine.PDFConverterWrapper  
 * callback:   A callback function that would be called after creating the class instance.
 *             callback function will have two arguments
 *                "err" of node.js type "Error"
 *                "handle" of type javaClass (defined in javaClass.js).
 *             In case of success,
 *                "err" will be null.
 *                "handle" is the java class Handle that must be used when calling the callMethod & callMethodSync APIs.
 *             In case of error,
 *                "err" will have information about error occurred during class instance creation.
 *                "handle" will be null.
 */
exports.createClass = function(className, callback){
	var self = this;

	// Check if already initialized
	if ( self.initialized === undefined || !self.initialized){
		callback(new Error("Can not create java class instance since js2Java module is not initialized."));
		return;
	};

	// Assert className
	if (className === null){
		callback(new Error("Classname can not be null"));
	}
	else if (className.length <=0){
		callback(new Error("Classname can not be empty string"));
	}
	else{
		// Create a javaClass for the passed classname
		new javaClass(className, function(err, result){
			if (err){
				callback(err);
			}else{
				callback(null, result);
			}
		});
	}
}

/**
 * finalize: Finalize Java Virtual Machine (JVM)
 * @Params:
 * callback:   A callback function that would be called after termination of JVM.
 *             callback function will have one argument "err" of node.js type "Error."
 *             In case of success, "err" will be null.
 *             In case of  error,  "err" will have information about errors occurred during JMV termination.
 */
exports.finalize =  function(callback){
	var self = this;

	if (self.initialized === undefined || !self.initialized){
		callback(new Error("Can not finalize js2Java module since it is not initialized."));
		return;
	};

	// We use try-catch block here since require.resolve does not have a callback function.
	try{
		// unload java module
		var name = require.resolve('java');
		delete require.cache[name];	
		self.initialized = false;
		callback();
	}catch(err){
		self.initialized = false;
		callback(err);
	}
}

/**
 * _setClasspath: Set the Java classpath
 * @Params:
 * classpath: An array of class paths where JVM can find the required jar and class files.
 * @Returns:
 *   In case of success,
 *      Nothing is returned.
 *   In case of error,
 *      Error object containing the error that occurred when setting the classpath
 */
function _setClasspath(classpath){
	var self = this;

	// Assert Classpath
	if (classpath === null){
		return new Error("Java classpath is null. Could not initialize js2Java module");
	}
	
	// Assert type of classpath
	if (typeof classpath !== "object" || !util.isArray(classpath)){ 
		return new Error("Java classpath is not an array object. Could not initialize js2Java module");
	}
	
	// Set Classpath
	for (var i = 0; i < classpath.length; i++){
		var strTmp = classpath[i];
		if (classpath[i] !== null && classpath[i].length > 0){
			java.classpath.push(classpath[i]);
		}
	}
}

/**
 * _importPackages: Set JVM Options
 * @Params:
 * jvmOptions: Any options that need to be passed to JVM (Example: MaxMemory to be used by JVM)
 *             Optional.  Can be null or an empty array    
 * @Returns:
 *   In case of success,
 *      Nothing is returned.
 *   In case of error,
 *      Error object containing the error that occurred when setting JVM options
 */
function _setOptions(jvmOptions){
	var self = this;

	// Assert Required Packages
	if (jvmOptions === null){
		return;	// JVM options are optional. so do not throw error, just return 
	}
	
	// Assert type of Required Packages
	if (typeof jvmOptions !== "object" || !util.isArray(jvmOptions)){ 
		return new Error("Java Virtual Machine(JVM) options parameter is not an array object. Could not initialize js2Java module");
	}
	
	// Set JVM Options
	for (var i = 0; i < jvmOptions.length; i++){
		if( jvmOptions[i] !== null && jvmOptions[i].length > 0){
			java.options.push( jvmOptions[i] );
		}
	}
}
