package ActiveState::Unix::Crontab;

# http://www.opengroup.org/onlinepubs/007904975/utilities/crontab.html
# man crontab

use strict;
use ActiveState::Run qw(run);

use base 'Exporter';
our @EXPORT_OK = qw(cron_edit cron_add cron_format cron_install
		    cron_list cron_cmds cron_parse
		    cron_allow_user cron_deny_user cron_remove_user
		   );

my $crontab = "crontab";
my $cron_allow;
my $cron_deny;
my $blank_lines_ok = 1;

if ($^O eq "linux") {
    $cron_allow = "/etc/cron.allow";
    $cron_deny  = "/etc/cron.deny";
}
elsif ($^O eq "solaris") {
    $cron_allow = "/etc/cron.d/cron.allow";
    $cron_deny = "/etc/cron.d/cron.deny";
    $blank_lines_ok = 0;
}
elsif ($^O eq "freebsd") {
    $cron_allow = "/var/cron/allow";
    $cron_deny = "/var/cron/deny";
}
else {
    # posix says this is the way
    $cron_allow = "/usr/lib/cron/cron.allow";
    $cron_deny = "/usr/lib/cron/cron.allow";
}

my $RE_min = qr/[0-5]?[0-9]/;
my $RE_min_range = qr/$RE_min(?:-$RE_min)*/;
my $RE_min_field = qr/(?:(?:\*|$RE_min_range(?:,$RE_min_range)*)(?:\/$RE_min)?)/;

sub cron_edit {
    run($crontab, "-e");
}

sub cron_add {
    my @list = cron_list();
    if (@_ > 1) {
	my %opt = @_;
	my $comment = delete $opt{comment};
	if ($comment) {
	    $comment =~ s/^/\# /g;
	    $comment .= "\n" unless $comment =~ /\n\z/;
	    push(@list, "\n") if $blank_lines_ok && @list && $list[-1] =~ /\S/;
	    push(@list, $comment);
	}
	push(@list, cron_format(%opt));
    }
    else {
	push(@list, shift);
    }

    cron_install(@list);
}

sub cron_format {
    my %opt = @_;
    my $line;
    if (my $cmd = delete $opt{cmd}) {
	my $min = delete $opt{min};
	my $hour = delete $opt{hour};
	my $mday = delete $opt{mday};
	my $mon = delete $opt{mon};
	my $wday = delete $opt{wday};
	my $enabled = exists $opt{enabled} ? delete $opt{enabled} : 1;

	for ($min, $hour, $mday, $mon, $wday) {
	    $_ = "*" unless defined($_) && length($_);
	    die "Bad timespec" if /\s/;
	}
	$cmd =~ s/%/\\%/g;

	$line = "$min $hour $mday $mon $wday $cmd";
	$line = "#$line" unless $enabled;
    
	if (defined(my $data = delete $opt{data})) {
	    $data =~ s/%/\\%/g;
	    $data =~ s/\n/%/g;
	    $line .= "%$data";
	}
	$line .= "\n";
    }
    elsif (my $env = delete $opt{env}) {
	my $val = delete $opt{value};
	$val = "" unless defined($val);
	$line = "$env=$val\n";
    }
    elsif (defined(my $com = delete $opt{comment})) {
	$com = "# $com" unless $com =~ /^\s*\#/;
	$com .= "\n" unless $com =~ /\n\z/;
	$line = $com;
    }
    elsif ($blank_lines_ok) {
	$line = "\n";
    }
    else {
	$line = "#\n";
    }

    warn "Unrecognized cron_fmt option: " . join(", ", keys %opt)
	if $^W && %opt;
    
    return $line;
}

sub cron_parse {
    my $line = shift;
    $line =~ s/\n\z//;
    if ($line =~ /^(\w+)=(.*)/) {
	return (env => $1, value => $2);
    }
    elsif ($line =~ /^\s*$/) {
	return ();
    }
    elsif ($line =~ /^(\s*\#)?\s*((?:$RE_min_field\s+){3}(?:\S+\s+){2}\S.*)/) {
	my $enabled = !$1;
	my($min, $hour, $mday, $mon, $wday, $cmd) = split(' ', $2, 6);
	my $data;
	if ($cmd =~ s/(?<!\\)%(.*)//) {
	    $data = $1;
	    $data =~ s/%/\n/g;
	}
	$cmd =~ s/\\(.)/$1/g;
	return ($min  eq "*" ? () : (min  => $min),
		$hour eq "*" ? () : (hour => $hour),
		$mday eq "*" ? () : (mday => $mday),
		$mon  eq "*" ? () : (mon  => $mon),
		$wday eq "*" ? () : (wday => $wday),
		cmd => $cmd,
		defined($data) ? (data => $data) : (),
		enabled => $enabled,
	       );
    }
    else {
	return (comment => $line);
    }
}

sub cron_install {
    # Linux/*BSD crontab insists on the "-" to read from stdin
    my $stdin = $^O =~ /^(?:linux|freebsd|darwin)$/ ? "-" : "";

    open(my $c, "|$crontab $stdin") || die "Can't run $crontab\n";
    print $c @_;
    return close($c);
}

sub cron_list {
    my @list = split(/^/, `crontab -l 2>/dev/null`);
    if (@list >= 3 &&
	$list[0] =~ /\# DO NOT EDIT THIS FILE/ &&
	$list[2] =~ /^\# \(Cron version .* vixie /)
    {
	# vixie-cron on Linux systems prepend 3 lines we don't want to see
	splice(@list, 0, 3);
    }
    return wantarray ? @list : join("", @list);
}

sub cron_cmds {
    grep defined, map +{cron_parse($_)}->{cmd}, cron_list();
}

sub cron_allow_user {
    my $user = shift;
    if ($cron_allow && open(CRON_ALLOW, "<", $cron_allow)) {
	local $_;
	my $found;
	while (<CRON_ALLOW>) {
	    $found++, last if /^\Q$user\E$/;
	}
	unless ($found) {
	    print "Adding $user to $cron_allow...\n";
	    open(CRON_ALLOW, ">>", $cron_allow) ||
		die "Can't append to $cron_allow: $!";
	    print CRON_ALLOW "$user\n";
	}
	close(CRON_ALLOW);
    }
}

sub cron_deny_user {
    die "NYI";
}

sub cron_remove_user {
    my $user = shift;
    die "Most be root to remove the cron user '$user'" if $> != 0;
    run(qq(-su - $user -c "crontab -r" </dev/null 1>/dev/null 2>&1));
    _remove_line("$user\n", $cron_allow);
    _remove_line("$user\n", $cron_deny);
}

sub _remove_line {
    my($line, $file) = @_;
    open(my $f, "<", $file) || return;
    my @lines = <$f>;
    if (grep $_ eq $line, @lines) {
	@lines = grep $_ ne $line, @lines;
	undef($f);
	open(my $f, ">", $file) || die "Can't write $file: $!";
	print $f grep $_ ne $line, @lines;
	close($f) || die "Can't close $file: $!";
    }
}

1;


__END__

=head1 NAME

ActiveState::Unix::Crontab - Portable manipulation of the crontab system

=head1 SYNOPSIS

 use ActiveState::Unix::Crontab qw(cron_add);

 cron_add(cmd => "ls -l /etc");

=head1 DESCRIPTION

This module provides function that allow portable manipulation of the
crontab system found on Unix.

The following functions are provided and manipulate the crontab entry
for the current user

=over

=item cron_edit()

This function invokes the editor on the current crontab file.
The function will croak on failure.

=item cron_add( $line )

=item cron_add( %opts )

This function add an entry to the current crontab file.  The function
either takes a single cron line or a set of key/value paired
parameters; see cron_parse() description below.  A C<comment> can be
passed together with a C<cmd>.

=item $line = cron_format( %opts )

This formats a single crontab entry line.  This is the inverse of the
cron_parse() function.  See the description of cron_parse() for what
options to pass.

=item %opts = cron_parse( $line )

This taks a single crontab line and parse it out as a set of key/value
pairs.  This is the inverse of the cron_format() function.  The
cron_list() function is used to read out lines from crontab.

There are 4 different results that can be returned:

=over

=item 1.

An empty line simply returns an empty %opts.

=item 2.

An line containing the setting of an environment variable returns:

  env:     name of the environment variable
  value:   the value it takes

=item 3.

An line containing a scheduled command contains:

  cmd:     what command to execute
  data:    stdin data provided to the command
  enabled: is the command enabled or not
  min:     which minute to run
  hour:    which hour to run
  mday:    which day of month to run
  mon:     which month to run
  wday:    which week day to run

Only C<cmd> will always be there.  The C<cmd> and C<data> is plain
strings.  The C<enabled> is a boolean.  The time and date fields take
a list of numbers or ranges:

  5
  1,2
  1-3
  1,2,5-9,12

Note that no space is allowed in these values.  The range for C<wday>
is 0 .. 6, with 0 as Sunday.

Portable applications should limit the settings to this format.  Some
implementations of cron also allow step values and/or plain English
names for the weekdays or months.

=item 4

Anything else is returned as a comment:

  comment: comment text (including leading '#')

=back

=item cron_install( @lines )

This installes the given crontab lines as the current crontab file.

=item @lines = cron_list()

This extract the crontab lines from the current crontab file.  The
lines are returned as is and can be parsed with the cron_parse()
function.

=item @cmds = cron_cmds()

This extracts the currently installed crontab commands.

=back

The following commands are available for 'root' to use:

=over

=item cron_allow_user( $user )

This make sure that the given user is allowed to use the crontab
system.

=item cron_deny_user( $user )

The tell the system the the given user is to allowed to use the
crontab system.

=item cron_remove_user( $user )

This remove all records of the given user in the crontab system.

=back

=head1 EXAMPLE

This disables all cron commands:

 #!/usr/bin/perl -w
 use ActiveState::Unix::Crontab qw(cron_list cron_install
                                   cron_parse cron_format);

 my $mod;
 my @list = cron_list();

 for (@list) {
     my %line = cron_parse($_);
     if ($line{cmd} && $line{enabled}) {
	 $line{enabled} = 0;
	 $_ = cron_format(%line);
	 $mod++;
     }
 }

 cron_install(@list);

Enabling them again is left as an exercise for the interested
developer.

cron_install(@list) if $mod;

=head1 SEE ALSO

L<crontab>

=cut
