#-------------------------------------------------------------------------------
#  afp2xml.pm:
#  Scripting Facility to create XML output
#
#  Call:
#
#  On Windows:
#            FOR /F %%i IN ( 'perl -e "{$,=';'; print @INC;}"' ) DO @SET PERLLIB=%%i
#            afp2web.exe -q -c -doc_cold -sp:sfsamples\afp2xml.pm samples\insure.afp
#
#  On Unix:
#            export PERLLIB=`perl -e "{$,=' '; print @INC;}"`
#            ./afp2web   -q -c -doc_cold -sp:sfsamples/afp2xml.pm samples/insure.afp
# 
#  Author  : Fa. OXSEED
#  Date    : 2010-08-12
#  Version : 1.0.3
#
#  $V100   2005-08-01    Initial Release
#
#  $V101   2008-07-10    Extended to add script file path as module search path
#                        on "BEGIN" block
#
#  $V102   2010-01-04    a. Extended to utilize
#                           "a2w::ConfigConstants"
#                           "a2w::DocumentConstants"
#                           "a2w::PageConstants"
#                           "a2w::FontConstants"
#                           packages
#                        b. Optimized constant declarations
#                        c. Modified PERL "@INC" extending to consider the
#                           Scripting Facility samples grouping under "sfsamples"
#                           directory
#
#  $V103   2010-08-12    Extended to support MMD/LPD inputs
#
#-------------------------------------------------------------------------------

# $V101 Begin
#-----------------------------------------------------------------------
# BEGIN block of module
#
# Extends PERL module search path array (@INC) with new element having
# this script modules path in order to have better module portability
#-----------------------------------------------------------------------
BEGIN {
    #---- Fetch script filename
    my $sScriptFilenameTmp = $0;

    #---- Extract script file path from script filename
    my $sScriptFilePathTmp = "";
    if ( $sScriptFilenameTmp =~ /(.*)\/.*\.pm/ ){
        $sScriptFilePathTmp = $1;
    }

    #printf STDERR ( "Script filename: " . $0 . " Script filepath: " . $sScriptFilePathTmp . "\n" );
    # $V102 Begin
    if ( $sScriptFilePathTmp eq "" ){
        $sScriptFilePathTmp = ".";
    }
    else {
        my $sScriptFileParentPathTmp = "";
        if ( $sScriptFilePathTmp =~ /(.*)\/sfsamples/ ){
            $sScriptFileParentPathTmp = $1;
        }

        #---- Add script file parent path to module search path
        if ( $sScriptFileParentPathTmp ne "" ){
            unshift( @INC, $sScriptFileParentPathTmp );
        }
    }
    # $V102 End

    #---- Add script file path to module search path
    unshift( @INC, $sScriptFilePathTmp );
}
# $V101 End

use a2w::Config;
use a2w::Document;
use a2w::Font;
use a2w::Index;
use a2w::Kernel;
use a2w::Line;
use a2w::MediumMap;
use a2w::NOP;
use a2w::Overlay;
use a2w::Page;
use a2w::PSEG;
use a2w::Text;
# $V102 Begin
use a2w::ConfigConstants;
use a2w::DocumentConstants;
use a2w::PageConstants;
use a2w::FontConstants;
# $V102 End

#---- Required Perl Modules for XML output
use IO::File;
use XML::Writer;
#-----------------------------------------------------------------------
# Initialize once per process
#-----------------------------------------------------------------------
sub initialize(){

    #---- Get Parameter of initialize( Par: a2w::Config, a2w::Kernel )
    ( $a2wConfigPar, $a2wKernelPar ) = @_;

    #---- Define boolean values
    $TRUE  = 1;    # TRUE  boolean value
    $FALSE = 0;    # FALSE boolean value

    #---- Set/Reset Logging
    $bLog = $FALSE;
    if (index( lc($a2wConfigPar->getAttribute( $a2w::ConfigConstants::LOGGINGLEVEL )), "sf") >= 0 ){    # $V102 Change
        $bLog = $TRUE;
    }

    my $sScriptProcTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SCRIPTPROCEDURE );    # $V102 Change
    my $sScriptArgsTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SCRIPTARGUMENT );
    $sIndexFilePath    = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::INDEXPATH );
    $sOutputFilePath   = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::OUTPUTFILEPATH );
    $sSpoolType        = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::INPUTFORMAT );        # $V103 Change
    $sSpoolFilename    = $a2wKernelPar->getSpoolFilename();

    if ( $bLog == $TRUE ){
        printf STDERR ( "Running $sScriptProcTmp...\n" );
        printf STDERR ( "initialize(): Processing $sSpoolFilename\n" );
        printf STDERR ( "initialize(): Args: $sScriptArgsTmp, IndexFilePath: $sIndexFilePath, OutputFilePath: $sOutputFilePath\n" );
    }

    # $V102 Begin
    #---- Page process flags
    $APPEND = 0;    # append page to Current Document
    $SKIP   = 1;    # skip page
    $NEWDOC = 2;    # new document
    # $V102 End

    #---- Reset Page Id
    $iPageId = 0;

    # $V103 Begin
    #---- Page formatting constants
    $iPageWidth  = 842.0;
    $iPageHeight = 595.0;

    $iLeftMargin  = 15.0;
    $iTopMargin   = 0.0;

    $a2wDefaultFont = undef;
    $sScriptUnitBase = "pixel";

    $bTRCExist = $FALSE;    # Flag indicating whether records have TRC or not

    $iLineSpacing = 9;

    #---- Initialize current page index fields line
    $iIndexFieldsLineNr = 64;
    $sIndexFieldsLine   = "";

    if ( lc( $sSpoolType ) eq "afp" ){
        #---- Set AutoSplit on
        $a2wConfigPar->setAttribute( $a2w::ConfigConstants::AUTOSPLIT, "on" );
    }
    elsif ( lc( $sSpoolType ) eq "lpd" ){
        #---- Create default font
        $a2wDefaultFont = new a2w::Font( $a2w::FontConstants::TYPE_TYPE1, "Courier" );
        $a2wDefaultFont->setHeight( $iLineSpacing );

        #---- Set script unit base as Pixels    
        $a2wConfigPar->setAttribute( $a2w::ConfigConstants::SCRIPTUNITBASE, $sScriptUnitBase );
    }    
    # $V103 End

    #---- afp2xml.pm Version
    $sVersion = "v1.0.3";

    return 0;
}

#-----------------------------------------------------------------------
# InitializeDoc for each document
#-----------------------------------------------------------------------
sub initializeDoc(){

    #---- Get Parameter of initializeDoc( Par: a2w::Document )
    ($a2wDocumentPar) = @_;

    #---- Fetch DocId
    my $DocIdTmp = $a2wDocumentPar->getId();

    if ( $bLog == $TRUE ){
        printf STDERR ( "initializeDoc(): DocId $DocIdTmp\n" );
    }

    #---- Generating FileHandle for XML Writer (needed since a2wDocumentPar->getSimpleFilename() not available from initializeDoc())
    $xmlDocIdFilename = $sOutputFilePath . $DocIdTmp . ".xml";
    $xmlDocOutput = new IO::File( ">" . $xmlDocIdFilename );

    #---- Generating XML Writer Object
    $xmlDocWriter = new XML::Writer(   OUTPUT        => $xmlDocOutput
                                     , NAMESPACES    => 1
                                   );

    #---- Processing Instruction
    $xmlDocWriter->xmlDecl( "ISO-8859-1" );
    $xmlDocWriter->comment( "Generated by $svScriptProc " . $sVersion . ", " . localtime() );
    $xmlDocWriter->doctype( 'Document', "", 'afp2xml.dtd' );

    #---- Starting with root element
    $xmlDocWriter->startTag(   'Document'
                             , 'Id'    => $DocIdTmp                  # Doc Id element
                             , 'Name'  => $a2wDocumentPar->getName()
                             , 'Spool' => $sSpoolFilename
                             , 'PID'   => $a2wDocumentPar->getPID()
                           );

    #---- Reset Page Id
    $iPageId = 0;

    return 0;
}

#-----------------------------------------------------------------------
# InitializePage for each page
#-----------------------------------------------------------------------
sub initializePage(){

    #---- Get Parameter of initializePage( Par: a2w::Page )
    ($a2wPagePar) = @_;

    if ( $bLog == $TRUE ){
        printf STDERR ( "initializePage()\n" );
    }
    return 0;
}

#-----------------------------------------------------------------------
# Main entry method
# Return values:
#        < 0:    error
#         0:    append page to Current Document
#         1:    skip page
#         2:    first page / new document
#-----------------------------------------------------------------------
sub afp2web(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "afp2web(): PageId " . $a2wPagePar->getParseId() . "\n" );
    }

    #---- Set default return value
    my $iRetTmp = $APPEND; # default: append page

    # $V103 Begin
    #---- Format page, for LPD input ----#
    if ( lc( $sSpoolType ) eq "lpd" ){
        my ( $iRcTmp, $sMsgTmp ) = _formatPage( $a2wPagePar );
        if ( $iRcTmp < 0 ){
            return ( $iRcTmp, $sMsgTmp );
        }
    }
    # $V103 End

    #---- Increment PageId
    $iPageId++;

    #---- Add Document Indexes to Index List
    if ( $iPageId == 1 ){
        addIndexes( $a2wDocumentPar );
        $xmlDocWriter->startTag( 'PageList' );    # <PageList>
    }

    #---- Starting with Element Page
    $xmlDocWriter->startTag(   'Page'
                             , 'Id'         => $iPageId                        # Page Id element
                             , 'Name'       => $a2wPagePar->getName()
                             , 'Width'      => $a2wPagePar->getWidth()
                             , 'Height'     => $a2wPagePar->getHeight()
                             , 'Resolution' => $a2wPagePar->getResolution()
                           );

    #---- Add Page Applied Medium Map Info
    addPageMediumMapInfo();

    #---- Add Page Applied Medium Overlay List
    addPageMediumOverlayList();

    #---- Add Page Included Overlay List
    addIncludedOverlayList( $a2wPagePar );

    #---- Add Page Included PageSegment List
    addIncludedPageSegmentList( $a2wPagePar );

    #---- Add Page Indexes to IndexList of Page
    addIndexes( $a2wPagePar );

    #---- Add NOPS to NOPList of Page
    addPageNOPs();

    #---- Add Text Objects to TextObjectList of Page
    addTextObjects( $a2wPagePar );

    #---- Add Line Objects to LineObjectList of Page
    addLineObjects( $a2wPagePar );

    return $iRetTmp;
}

#-----------------------------------------------------------------------
# FinalizePage for each page
#-----------------------------------------------------------------------
sub finalizePage(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "finalizePage()\n" );
    }
    $xmlDocWriter->endTag();                    # closing Element Page ==> </Page>

    return 0;
}

#-----------------------------------------------------------------------
# FinalizeDoc for each document
#-----------------------------------------------------------------------
sub finalizeDoc(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "finalizeDoc()\n" );
    }

    #---- Closing Element PageList
    $xmlDocWriter->endTag();    # </PageList>

    #---- Closing Element Document
    $xmlDocWriter->endTag();    # </Document>

    #---- Writing Buffer into the File
    $xmlDocWriter->end();

    #---- Closing FileHandle
    $xmlDocOutput->close();

    #---Rename DocId.xml to DocumentOutputFilename.xml
    rename( $xmlDocIdFilename , $sOutputFilePath . $a2wDocumentPar->getSimpleFilename() . ".xml" );

    return 0;
}

#-----------------------------------------------------------------------
# Finalize once per process
#-----------------------------------------------------------------------
sub finalize(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "finalize()\n" );
    }
    return 0;
}

#-----------------------------------------------------------------------
# Add Indexes to Index List
#-----------------------------------------------------------------------
sub addIndexes(){

    #---- Get Parameter (Page or Overlay)
    my $svMediaPar = $_[ 0 ];

    if ( $bLog == $TRUE ){
        printf STDERR ( "addIndexes()\n" );
    }

    #---- Fetch first Index
    my $a2wIndexTmp = $svMediaPar->getFirstIndex();

    if ( $a2wIndexTmp != 0 ){

        $xmlDocWriter->startTag( 'IndexList' );    # opening element IndexList ==> <IndexList>

        #---- Loop thru the Doc Indexes
        while ( $a2wIndexTmp != 0 ){

            #---- Write Index Name & Value
            $xmlDocWriter->dataElement(   'Index'
                                        , $a2wIndexTmp->getValue()
                                        , 'Name'    => $a2wIndexTmp->getName()
                                      );

            #---- Fetch next Index
            $a2wIndexTmp = $svMediaPar->getNextIndex();
        }
        $xmlDocWriter->endTag();                # closing Element IndexList ==> </IndexList>
    }
}

#-----------------------------------------------------------------------
# Add Page Applied Medium Map Info
#-----------------------------------------------------------------------
sub addPageMediumMapInfo(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "addPageMediumMapInfo()\n" );
    }

    #---- Fetch applied medium map
    my $a2wMediumMapTmp = $a2wPagePar->getAppliedMediumMap();

    if ( $a2wMediumMapTmp != 0 ){
        #---- Starting with Medium Map
        $xmlDocWriter->startTag(   'MediumMap'                       # opening Element MediumMap ==> <MediumMap>
                                 , 'Name'           => $a2wMediumMapTmp->getName()
                                 , 'Formdef'        => $a2wMediumMapTmp->getFormdefName()
                                 , 'Width'          => $a2wMediumMapTmp->getWidth()
                                 , 'Height'         => $a2wMediumMapTmp->getHeight()
                                 , 'Resolution'     => $a2wMediumMapTmp->getResolution()
                                 , 'DuplexControl'  => $a2wMediumMapTmp->getDuplexControl()
                                 , 'N-UpControl'    => $a2wMediumMapTmp->getNupControl()
                               );
                            
        #---- Closing medium map
        $xmlDocWriter->endTag();                # </MediumMap>
    }
}

#-----------------------------------------------------------------------
# Add Page Applied Medium Overlay List
#-----------------------------------------------------------------------
sub addPageMediumOverlayList(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "addPageMediumOverlayList()\n" );
    }

    #---- Fetch first Page Medium Overlay
    my $a2wOverlayTmp = $a2wPagePar->getFirstMediumOverlay();

    if ( $a2wOverlayTmp != 0 ){

        $xmlDocWriter->startTag( 'MediumOverlayList' );    # opening Element MediumOverlayList ==> <MediumOverlayList>

        #---- Loop thru the Page Medium Overlays
        while ( $a2wOverlayTmp != 0 ){

            #---- Starting with Overlay
            $xmlDocWriter->startTag(   'Overlay'
                                     , 'Name'       => $a2wOverlayTmp->getName             # Overlay name
                                     , 'Width'      => $a2wOverlayTmp->getWidth()
                                     , 'Height'     => $a2wOverlayTmp->getHeight()
                                     , 'Resolution' => $a2wOverlayTmp->getResolution()
                                     , 'XPos'       => $a2wOverlayTmp->getIncludedXPosition()
                                     , 'YPos'       => $a2wOverlayTmp->getIncludedYPosition()
                                   );

            #---- Closing overlay
            $xmlDocWriter->endTag();                # </Overlay>

            #---- Fetch next Page Medium Overlay
            $a2wOverlayTmp = $a2wPagePar->getNextMediumOverlay();
        }
        $xmlDocWriter->endTag();                # closing Element MediumOverlayList ==> </MediumOverlayList>
    }
}

#-----------------------------------------------------------------------
# Add Included Overlay List
#-----------------------------------------------------------------------
sub addIncludedOverlayList(){

    #---- Get Parameter (Page or Overlay)
    my $svMediaPar = $_[ 0 ];

    if ( $bLog == $TRUE ){
        printf STDERR ( "addIncludedOverlayList()\n" );
    }

    #---- Fetch first Included Overlay
    my $a2wOverlayTmp = $svMediaPar->getFirstOverlay();

    if ( $a2wOverlayTmp != 0 ){

        $xmlDocWriter->startTag( 'IncludedOverlayList' );    # opening Element IncludedOverlayList ==> <IncludedOverlayList>

        #---- Loop thru the Included Overlays
        while ( $a2wOverlayTmp != 0 ){

            #---- Starting with Overlay
            $xmlDocWriter->startTag(   'Overlay'
                                     , 'Name'       => $a2wOverlayTmp->getName             # Overlay name
                                     , 'Width'      => $a2wOverlayTmp->getWidth()
                                     , 'Height'     => $a2wOverlayTmp->getHeight()
                                     , 'Resolution' => $a2wOverlayTmp->getResolution()
                                     , 'XPos'       => $a2wOverlayTmp->getIncludedXPosition()
                                     , 'YPos'       => $a2wOverlayTmp->getIncludedYPosition()
                                   );

            #---- Closing overlay
            $xmlDocWriter->endTag();                # </Overlay>

            #---- Fetch next Included Overlay
            $a2wOverlayTmp = $svMediaPar->getNextOverlay();
        }
        $xmlDocWriter->endTag();                # closing Element IncludedOverlayList ==> </IncludedOverlayList>
    }
}

#-----------------------------------------------------------------------
# Add Included Page Segment List
#-----------------------------------------------------------------------
sub addIncludedPageSegmentList(){

    #---- Get Parameter (Page or Overlay)
    my $svMediaPar = $_[ 0 ];

    if ( $bLog == $TRUE ){
        printf STDERR ( "addIncludedPageSegmentList()\n" );
    }

    #---- Fetch first Included Page Segment
    my $a2wPageSegmentTmp = $svMediaPar->getFirstPageSegment();

    if ( $a2wPageSegmentTmp != 0 ){

        $xmlDocWriter->startTag( 'IncludedPageSegmentList' );    # opening Element IncludedPageSegmentList ==> <IncludedPageSegmentList>

        #---- Loop thru the Included Page Segments
        while ( $a2wPageSegmentTmp != 0 ){

            #---- Starting with Page Segment
            $xmlDocWriter->startTag( 'PageSegment'
                                     , 'Name' => $a2wPageSegmentTmp->getName()               # Page Segment name
                                     , 'XPos' => $a2wPageSegmentTmp->getIncludedXPosition()  # XPosition
                                     , 'YPos' => $a2wPageSegmentTmp->getIncludedYPosition()  # YPosition
                                   );

            #---- Closing PageSegment
            $xmlDocWriter->endTag();                # </PageSegment>

            #---- Fetch next Included Page Segment
            $a2wPageSegmentTmp = $svMediaPar->getNextPageSegment();
        }
        $xmlDocWriter->endTag();                # closing Element IncludedPageSegmentList ==> </IncludedPageSegmentList>
    }
}

#-----------------------------------------------------------------------
# Add NOPs to NOPList
#-----------------------------------------------------------------------
sub addPageNOPs(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "addPageNOPs()\n" );
    }

    #---- Fetch first NOP
    my $a2wNOPTmp = $a2wPagePar->getFirstNOP();

    if ( $a2wNOPTmp != 0 ){

        $xmlDocWriter->startTag( 'NOPList' );    # starting Element NOPList ==> <NOPList>

        #---- Loop thru the NOPs
        while ( $a2wNOPTmp != 0 ){

            #---- Starting Elmement TextObject
            $xmlDocWriter->dataElement( 'NOP', $a2wNOPTmp->getValue() );

            #---- Fetch next NOP
            $a2wNOPTmp = $a2wPagePar->getNextNOP();
        }
        $xmlDocWriter->endTag();    # closing Element NOPList ==> </NOPList>
    }
}

#-----------------------------------------------------------------------
# Add Text Objects to TextObjectList of Page
#-----------------------------------------------------------------------
sub addTextObjects(){

    #---- Get Parameter (Page or Overlay)
    my $svMediaPar = $_[ 0 ];

    if ( $bLog == $TRUE ){
        printf STDERR ( "addTextObjects()\n" );
    }

    #---- Fetch first Text Object
    my $a2wTextTmp = $svMediaPar->getFirstText();

    my $a2wFontTmp = "";

    if ( $a2wTextTmp != 0 ){

        $xmlDocWriter->startTag( 'TextObjectList' );    # starting Element TextObjectList ==> <TextObjectList>

        #---- Loop thru the TextObjects
        while ( $a2wTextTmp != 0 ){

            $a2wFontTmp = $a2wTextTmp->getFont();

            #---- Starting Elmement TextObject
            $xmlDocWriter->startTag(   'TextObject'
                                     , 'XPos'         => $a2wTextTmp->getXPos()
                                     , 'YPos'         => $a2wTextTmp->getYPos()
                                     , 'CodedFont'    => $a2wFontTmp->getCodedFontName(),
                                     , 'CharacterSet' => $a2wFontTmp->getCharacterSetName()
                                     , 'CodePage'     => $a2wFontTmp->getCodePageName()
                                     , 'Typeface'     => $a2wFontTmp->getTypefaceName()
                                     , 'Height'       => $a2wFontTmp->getHeight()
                                     , 'Width'        => $a2wFontTmp->getWidth()
                                     , 'Weight'       => $a2wFontTmp->isBold() ? "Bold" : ($a2wFontTmp->isLight() ? "Light" : "Normal")
                                     , 'Slant'        => $a2wFontTmp->isItalic() ? "Italic" : "Medium"
                                     , 'Underline'    => $a2wFontTmp->isUnderline() ? "yes" : "no"
                                     , 'Strikeover'   => $a2wFontTmp->isStrikeover() ? "yes" : "no"
                                   );

            $xmlDocWriter->characters($a2wTextTmp->getText());

            $xmlDocWriter->endTag();    # closing Element TextObject

            #---- Fetch next Text Object
            $a2wTextTmp = $svMediaPar->getNextText();
        }
        $xmlDocWriter->endTag();    # closing Element TextObjectList ==> </TextObjectList>
    }
}

#-----------------------------------------------------------------------
# Add Line Objects to LineObjectList of Page
#-----------------------------------------------------------------------
sub addLineObjects(){

    #---- Get Parameter (Page or Overlay)
    my $svMediaPar = $_[ 0 ];

    if ( $bLog == $TRUE ){
        printf STDERR ( "addLineObjects()\n" );
    }

    #---- Fetch first Line Object
    my $a2wLineTmp = $svMediaPar->getFirstLine();

    if ( $a2wLineTmp != 0 ){

        $xmlDocWriter->startTag( 'LineObjectList' );    # starting Element LineObjectList ==> <LineObjectList>

        my $LineColorTmp = "";
        my $NegativeLineTmp = "";
        my $LineOrientationTmp = "";

        #---- Loop thru the LineObjects
        while ( $a2wLineTmp != 0 ){

            #---- Starting Elmement LineObject
            $xmlDocWriter->startTag(   'LineObject'
                                     , 'XPos'        => $a2wLineTmp->getXPos()
                                     , 'YPos'        => $a2wLineTmp->getYPos()
                                     , 'Length'      => $a2wLineTmp->getLength()
                                     , 'Width'       => $a2wLineTmp->getWidth()
                                     , 'Color'       => $a2wLineTmp->getColor()
                                     , 'Orientation' => $a2wLineTmp->isHorizontal() ? "Horizontal" : "Vertical"
                                     , 'Negative'    => $a2wLineTmp->isNegative() ? "yes" : "no"
                                   );

            #---- Closing Element LineObject
            $xmlDocWriter->endTag();

            #---- Fetch next Line Object
            $a2wLineTmp = $svMediaPar->getNextLine();
        }
        $xmlDocWriter->endTag();    # closing Element LineObjectList ==> </LineObjectList>
    }
}

# $V103 Begin
#-----------------------------------------------------------------------
# Format page
#
# Parameter(s)
# 1. Page (of type a2w::Page)
#
# Returns
# >0 in case of success
# <0 (+ the reason message) in case of error
#
# Remarks
# Processes each text (aka line text) of input and formats it based on
# ASA carriage controls
#
# *** IMPORTANT NOTE ON PROCESSING LINES ***:
# AFP2web will process each line of input and will make a text object
# out of each line of text. So processing a text object mean processing
# a line from input.
#
# Also the text (aka line) will have Carriage Control character at the
# beginning which can be processed in below function for formatting the
# lines
#
#-----------------------------------------------------------------------
sub _formatPage(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "_formatPage()\n" );
    }

    #---- Get parameter(s)
    #
    # 1. Page (of type a2w::Page)
    #
    my $a2wCurrentPageTmp = shift;

    #---- Fetch page resolution
    my $iPageResTmp = $a2wCurrentPageTmp->getResolution();

    #---- Set new page width/height
    $a2wCurrentPageTmp->setHeight( $iPageHeight );
    $a2wCurrentPageTmp->setWidth( $iPageWidth );

    #---- Evaluate line height
    my $fFactorTmp = $iPageResTmp / 72;

    if ( lc( $sScriptUnitBase ) eq "mm" ){
        $fFactorTmp = 25.4 / 72;
    }

    #---- Initialize LineId
    my $iLineIdTmp = 0;

    #---- Origin is at the bottom left corner
    my $iLeftMarginTmp = $iLeftMargin * $fFactorTmp;
    my $iTopMarginTmp  = $iTopMargin * $fFactorTmp;

    #---- Modify boundary evaluation based on Top-Left co-ordinate system
    my $iXPosTmp = $iLeftMarginTmp;    # X Position starts from Left Margin
    my $iYPosTmp = $iTopMarginTmp;     # Y Position is "Top Margin"

    #---- Process and format page content ----#
    my $a2wTextTmp = $a2wCurrentPageTmp->getFirstText();

    #---- Format each and every text (aka line text)
    my $sTextTmp  = "";
    my $cCCCharTmp = "";
    my $cTRCharTmp = "";
    my $sTextWithoutCCTmp = "";

    #---- Index Field Insured Name
    my $sInsuredNameTmp = "";
    
    while ( $a2wTextTmp != 0 ){
        $iLineIdTmp++;                 # Increment line number
        $iXPosTmp = $iLeftMarginTmp;   # X Position starts from Left Margin

        #---- Assert Y position
        #---- Modify boundary evaluation based on Top-Left co-ordinate system
        if ( $iYPosTmp > $iPageHeight ){
            if ( $bLog == $TRUE ){
                printf STDERR "\tERROR! Spool File Type (" . $sSpoolFileType . "), Line=" . $iLineIdTmp . ": Not enough space on page.\n";
            }

            return ( -98, "Spool File Type (" . $sSpoolFileType . "), Line=" . $iLineIdTmp . ": Not enough space on page." );
        }

        #---- Process current line and add texts appropriately
        $sTextWithoutCCTmp = "";
        $sTextTmp = $a2wTextTmp->getText();

        #---- Collect current page's index fields line
        if ( $iLineIdTmp == $iIndexFieldsLineNr ){
            $sIndexFieldsLine = $sTextTmp;
        }

        if ( $bTRCExist == $TRUE ){
            if ( $sTextTmp =~ /(.)(.)(.*)/ ){
                $cCCCharTmp        = $1;   # Contains CC
                $cTRCharTmp        = $2;   # Contains TRC
                $sTextWithoutCCTmp = $3;   # Contains line data
            }
        }
        else {
            if ( $sTextTmp =~ /(.)(.*)/ ){
                $cCCCharTmp        = $1;   # Contains CC
                $sTextWithoutCCTmp = $2;   # Contains line data
            }
        }

        #----- ASA Carriage Control action is applied before printing current line
        #      so process apply actions for given CC character
        #
        if ( $cCCCharTmp eq "+" ){                   # Do not space
            # Overwrite existing line
            $iYPosTmp -= $iLineSpacing;
        }
        elsif ( $cCCCharTmp eq " " ){                # Space one line
            # One line space is already evaluated,
            # so nothing to do here
        }
        elsif ( $cCCCharTmp eq "0" ){                # Space two lines
            # One line space is already evaluated,
            # so leave only one line space now
            $iYPosTmp += $iLineSpacing;
        }
        elsif ( $cCCCharTmp eq "-" ){                # Space three lines
            # One line space is already evaluated,
            # so leave only two line space now
            $iYPosTmp += $iLineSpacing;
            $iYPosTmp += $iLineSpacing;
        }
        elsif ( $cCCCharTmp eq "1" ){                # Skip to channel 1/New page
            # A2W kernel itself identifies new page,
            # so nothing to do here
        }
        elsif ( $cCCCharTmp eq "2" ){                # Skip to channel 2
        }
        elsif ( $cCCCharTmp eq "3" ){                # Skip to channel 3
        }
        elsif ( $cCCCharTmp eq "4" ){                # Skip to channel 4
        }
        elsif ( $cCCCharTmp eq "5" ){                # Skip to channel 5
        }
        elsif ( $cCCCharTmp eq "6" ){                # Skip to channel 6
        }
        elsif ( $cCCCharTmp eq "7" ){                # Skip to channel 7
        }
        elsif ( $cCCCharTmp eq "8" ){                # Skip to channel 8
        }
        elsif ( $cCCCharTmp eq "9" ){                # Skip to channel 9
        }
        elsif ( $cCCCharTmp eq "A" ){                # Skip to channel 10
        }
        elsif ( $cCCCharTmp eq "B" ){                # Skip to channel 11
        }
        elsif ( $cCCCharTmp eq "C" ){                # Skip to channel 12
        }

        #---- Skip trailing spaces
        $sTextWithoutCCTmp =~ s/\s+$//g;

        #---- Skip empty lines
        if ( length( $sTextWithoutCCTmp ) <= 0 ){
            #---- Evaluate next YPos (YPos = YPos - FontHeight)
            $iYPosTmp += $iLineSpacing;

            #---- Don't present this text on output
            $a2wTextTmp->remove();
        }
        else {
            if ( $bLog == $TRUE ){
                printf STDERR ( "(" . $iXPosTmp . ", " . $iYPosTmp . ")=>" . $sTextWithoutCCTmp . "<=\n" );
            }

            #---- Fill in text details
            $a2wTextTmp->setText( $sTextWithoutCCTmp ); # Text value
            $a2wTextTmp->setXPos( $iXPosTmp );          # Text X position
            $a2wTextTmp->setYPos( $iYPosTmp );          # Text Y position
            $a2wTextTmp->setFont( $a2wDefaultFont );    # Text font

            #---- Evaluate next YPos
            $iYPosTmp += $iLineSpacing;
        }

        #---- Get next text
        $a2wTextTmp = $a2wCurrentPageTmp->getNextText();
    }

    if ( $bLog == $TRUE ){
        printf STDERR ( "\tIndexFields=" . $sIndexFieldsLine . "\n" );
    }

    return 0;
}
# $V103 End

__END__
