/****************************************************************
 * Module Name: %W%                                             *
 *                                                              *
 * Description: See below                                       *
 *                                                              *
 *   <copyright                                                 *
 *   notice="lm-source-program"                                 *
 *   pids="5724-H72,"                                           *
 *   years="2008,2012"                                          *
 *   crc="3305572466" >                                         *
 *   Licensed Materials - Property of IBM                       *
 *                                                              *
 *   5724-H72,                                                  *
 *                                                              *
 *   (C) Copyright IBM Corp. 2008, 2012 All Rights Reserved.    *
 *                                                              *
 *   US Government Users Restricted Rights - Use, duplication or  *
 *   disclosure restricted by GSA ADP Schedule Contract with    *
 *   IBM Corp.                                                  *
 *   </copyright>                                               *
 *                                                              *
 * Version: @(#) MQMBID sn=p800-003-150615.2 su=_O7DtFhOPEeWRp7-5NNeHyQ pn=java/http/code/samples/P2PConsole/HTTPDELETE.java                                 *
 *                                                              *
 ****************************************************************/
/**
 * This sample shows how to delete a message. It has slightly enhanced function
 * of the amqsget command in that it will print out the timestamp, expiry and
 * persistence of the messages. The program continues until the queue is empty
 * or a request fails. This program can potentially take in three parameters:
 * <queueName>
 * <host:port> <context-root (the MQ Bridge for HTTP's context-root)>
 * defaults are: SYSTEM.DEFAULT.LOCAL.QUEUE localhost:8080 mq 
 * 
 * If there are any exceptions thrown from this program or errors returned from the server then they are 
 * printed to standard output as-is.
 *   
 * Example output of this program:
 * 
 * Java HTTPDELETE
 * 
 * HTTP DELETE Sample start
 * Target server       is 'localhost:8080'
 * Target queue        is 'SYSTEM.DEFAULT.LOCAL.QUEUE'
 * Target context-root is 'mq'
 * _________________________________________________________________________________________
 * my test message that was on the queue
 * timestamp   = Sun, 25 Mar 2007 08:49:37 GMT
 * expiry      = UNLIMITED
 * persistence = NON_PERSISTENT
 * _________________________________________________________________________________________
 * my other test message that was on the queue
 * timestamp   = Sun, 25 Mar 2007 08:49:36 GMT
 * expiry      = UNLIMITED
 * persistence = NON_PERSISTENT
 * 
 * No more messages
 * HTTP DELETE Sample end
 */

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;

public class HTTPDELETE
{
   private static final String DEFAULT_HOSTANDPORT = "localhost:8080";

   private static final String DEFAULT_QUEUE = "SYSTEM.DEFAULT.LOCAL.QUEUE";

   private static final String DEFAULT_CONTEXT_ROOT = "mq";

   public static String newline = System.getProperty("line.separator");

   private static final String MESSAGE_BOUNDARY = "_________________________________________________________________________________________";

   // the maximum length of the message that we want to print to the screen
   private static final int MAX_OUTPUT_MESSAGE_SIZE = 256;

   private URL url;

   /* some statics for HTTP return codes */
   private static int NO_MESSAGE_RC = 504;

   private static int OK_RC = 200;

   private static boolean exception = false;

   /**
    * @param args either none, 1 (the queue name) or 2 (queuename and host:port)
    * @throws MalformedURLException if the given host and port are incorrect
    *            format
    */
   public HTTPDELETE(String[] args) throws MalformedURLException
   {
      System.out.println("HTTP DELETE Sample start");

      // Is there a different queue name?
      String queueName = null;
      if (args.length > 0)
      {
         queueName = args[0];
      }
      else
      {
         // use the default
         queueName = DEFAULT_QUEUE;
      }

      // Is there a host:port?
      String hostAndPort = null;
      if (args.length > 1)
      {
         // We can't really do any checking here so just get the next arg.
         // The HTTP Connection will throw an exception if there are too many
         // issues
         hostAndPort = args[1];
      }
      else
      {
         // set to the default
         hostAndPort = DEFAULT_HOSTANDPORT;
      }

      // Is MQ bridge for HTTP deployed to a non-default context-root?
      String contextRoot = null;
      if (args.length > 2)
      {
         contextRoot = args[2];
      }
      else
      {
         // use the default
         contextRoot = DEFAULT_CONTEXT_ROOT;
      }

      System.out.println("Target server       is '" + hostAndPort + "'");
      System.out.println("Target queue        is '" + queueName + "'");
      System.out.println("Target context-root is '" + contextRoot + "'");

      // add them all up to get the url
      String urlString = "http://";
      urlString += hostAndPort + "/" + contextRoot + "/msg/queue/" + queueName;
      url = new URL(urlString);
   }

   /**
    * This is the main part of the program. It reads in messages from the queue
    * and writes them out to the console.
    * 
    * @throws IOException if there is a problem writing out to the console
    */
   public void printMessagesToConsole() throws IOException
   {
      String deletedMessage = null;
      do
      {
         deletedMessage = deleteMessage();
         if (deletedMessage != null)
         {
            System.out.println(deletedMessage);
         }
      }
      while (deletedMessage != null);

      if (!exception)
         System.out.println("No more messages");

      System.out.println("HTTP DELETE Sample end");
   }

   /**
    * @throws IOException if there is a problem connecting to the server.
    * @return the message. Already formatted for System.out or null if there was
    *         a problem. If there was a problem then an error message would have
    *         been put out to system.err already
    */
   private String deleteMessage() throws IOException
   {
      HttpURLConnection connection = (HttpURLConnection) url.openConnection();

      /* Build the headers */
      // the verb first.
      connection.setRequestMethod("DELETE");

      // write out what headers we want back
      // the header names are case-sensitive
      connection.setRequestProperty("x-msg-require-headers",
            "timestamp, expiry, persistence");

      // Now actually send the request message. There is no content as this is a
      // DELETE
      connection.connect();
      String formattedMessage = null;
      if (connection.getResponseCode() == NO_MESSAGE_RC)
      {
         return null;
      }
      // check the response for errors
      if (connection.getResponseCode() != OK_RC)
      {
         exception = true;
         printErrorMessage(connection.getErrorStream());
         connection.disconnect();
      }
      else
      {
         // Get the headers first
         String timestamp = connection.getHeaderField("x-msg-timestamp");
         String expiry = connection.getHeaderField("x-msg-expiry");
         String persistence = connection.getHeaderField("x-msg-persistence");
         // now get the message data
         BufferedReader reader = new BufferedReader(new InputStreamReader(
               connection.getInputStream()));
         String line = null;
         StringBuffer messageBuffer = new StringBuffer();
         while ((line = reader.readLine()) != null)
         {
            messageBuffer.append(line);
         }
         formattedMessage = formatMessage(timestamp, expiry, persistence,
               messageBuffer.toString());
      }
      connection.disconnect();
      return formattedMessage;
   }

   /**
    * Formats the newly found message
    * 
    * @param timestamp
    * @param expiry
    * @param persistence
    * @param messageBody
    * @return the message and properties formatted ready to print to System out.
    */
   private String formatMessage(String timestamp, String expiry,
         String persistence, String messageBody)
   {
      String formattedMessage = MESSAGE_BOUNDARY + newline;
      // Which is greater the max output message size or the message length?
      int messageSizeToPrint = messageBody.length() > MAX_OUTPUT_MESSAGE_SIZE ? MAX_OUTPUT_MESSAGE_SIZE
            : messageBody.length();
      formattedMessage += messageBody.substring(0, messageSizeToPrint)
            + newline;
      formattedMessage += "timestamp   = " + timestamp + newline;
      formattedMessage += "expiry      = " + expiry + newline;
      formattedMessage += "persistence = " + persistence + newline;

      return formattedMessage;
   }

   /**
    * Reads the response from the input stream and prints it to standard out.
    * 
    * @param inputStream the input stream from the HTTPUrlConnection
    * @throws IOException if there is a problem reading from the stream
    */
   private void printErrorMessage(InputStream inputStream) throws IOException
   {
      System.out
            .println("There has been an error trying to DELETE a message from '"
                  + url + "'. HTTP error response follows...");
      BufferedReader reader = new BufferedReader(new InputStreamReader(
            inputStream));
      String line = null;
      while ((line = reader.readLine()) != null)
      {
         System.out.println(line);
      }
   }

   /**
    * Main method which creates the HTTPDELETE object and then calls its
    * printMessagesToConsole() method.
    * 
    * @param args the command line arguments either zero, [queueName] or
    *           [queueName host:port]
    */
   public static void main(String[] args)
   {
      HTTPDELETE deleteExample = null;
      try
      {
         // args are OK start the program
         deleteExample = new HTTPDELETE(args);
      }
      catch (MalformedURLException e)
      {
         // thrown if the host name or port is not in a good format.
         printUsage();
      }
      try
      {
         deleteExample.printMessagesToConsole();
      }
      catch (IOException e1)
      {
         System.out.println("Failed to DELETE a message from "
               + deleteExample.url.toString());

         e1.printStackTrace();
      }
   }

   /**
    * Prints the usage to System.out
    */
   private static void printUsage()
   {
      System.out.println("Usage: java HTTPDELETE <Queue Name> (default="
            + DEFAULT_QUEUE + ") <host:port> (default=" + DEFAULT_HOSTANDPORT
            + ") e.g. MY.QUEUE myhost:8080 <context-root> (default="
            + DEFAULT_CONTEXT_ROOT + ")");
   }
}
