/****************************************************************
 * Module Name: %W%                                             *
 *                                                              *
 * Description: See below                                       *
 *                                                              *
 *   <copyright                                                 *
 *   notice="lm-source-program"                                 *
 *   pids="5724-H72,"                                           *
 *   years="2008,2012"                                          *
 *   crc="3964041921" >                                         *
 *   Licensed Materials - Property of IBM                       *
 *                                                              *
 *   5724-H72,                                                  *
 *                                                              *
 *   (C) Copyright IBM Corp. 2008, 2012 All Rights Reserved.    *
 *                                                              *
 *   US Government Users Restricted Rights - Use, duplication or  *
 *   disclosure restricted by GSA ADP Schedule Contract with    *
 *   IBM Corp.                                                  *
 *   </copyright>                                               *
 *                                                              *
 * Version: @(#) MQMBID sn=p800-003-150615.2 su=_O7DtFhOPEeWRp7-5NNeHyQ pn=java/http/code/samples/P2PConsole/HTTPPOST.java                                 *
 *                                                              *
 ****************************************************************/

/**
 * This sample shows how to post a message. It has the same behaviour as the
 * amqsput command in that it will read in lines from the command line and put
 * them to the queue. It will put non-persistent String messages on to the queue
 * with UNLIMITED expiry and LOW (0) priority. The program is terminated by
 * either EOF being put into the entry line (^Z on windows) or a blank line.
 * usage: java HTTPPOST <Queue (default=SYSTEM.DEFAULT.LOCAL.QUEUE)> <host:port
 * (default localhost:8080> <context-root (the MQ Bridge for HTTP's
 * context-root)>
 */

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;

public class HTTPPOST
{
   private static final String DEFAULT_HOSTANDPORT = "localhost:8080";

   private static final String DEFAULT_QUEUE = "SYSTEM.DEFAULT.LOCAL.QUEUE";

   private static final String DEFAULT_CONTEXT_ROOT = "mq";

   private static final String CRLF = "\r\n";

   public static int MALFORMED_URL_EXCEPTION_RC = -1;

   public static int END_IOEXCEPTION_RC = -2;

   private URL url;

   /**
    * @param args either none, 1 (the queue name) or 2 (queuename and host:port)
    * @throws MalformedURLException if the given host and port are incorrect
    *            format
    */
   public HTTPPOST(String[] args) throws MalformedURLException
   {
      System.out.println("HTTP POST Sample start");

      // Is there a queue name?
      String queueName = null;
      if (args.length > 0)
      {
         queueName = args[0];
      }
      else
      {
         // use the default
         queueName = DEFAULT_QUEUE;
      }

      // Is there a host:port?
      String hostAndPort = null;
      if (args.length > 1)
      {
         // We can't really do any checking here so just get the next arg.
         // The HTTP Connection will throw an exception if there are too many
         // issues
         hostAndPort = args[1];
      }
      else
      {
         // set to the default
         hostAndPort = DEFAULT_HOSTANDPORT;
      }

      // Is MQ bridge for HTTP deployed to a non-default context-root?
      String contextRoot = null;
      if (args.length > 2)
      {
         contextRoot = args[2];
      }
      else
      {
         // use the default
         contextRoot = DEFAULT_CONTEXT_ROOT;
      }

      System.out.println("Target server       is '" + hostAndPort + "'");
      System.out.println("Target queue        is '" + queueName + "'");
      System.out.println("Target context-root is '" + contextRoot + "'");

      // add them all up to get the url
      String urlString = "http://";
      urlString += hostAndPort + "/" + contextRoot + "/msg/queue/" + queueName;
      url = new URL(urlString);
   }

   /**
    * This is the main part of the program. It reads in lines from the command
    * line and sends them to the MQ HTTP support server.
    * 
    * @throws IOException if there is a problem reading in from the command line
    */
   public void sendMessagesFromConsole() throws IOException
   {
      String line = null;
      boolean messageSentOK = true;
      while (messageSentOK && (line = readLine()) != null && line.length() != 0)
      {
         messageSentOK = sendMessage(line);
      }
      System.out.println("HTTP POST Sample end");
   }

   /**
    * @param message the line of text to send
    * @throws IOException if there is a problem connecting to the server.
    * @return whether the message was handled by the server OK. If this is false
    *         then this method would have printed out a nice message to the
    *         screen with the reason for failure that came back from the server
    */
   private boolean sendMessage(String message) throws IOException
   {
      HttpURLConnection connection = (HttpURLConnection) url.openConnection();

      /* Build the headers */
      // the verb first
      connection.setRequestMethod("POST");

      // Content type is a string message
      connection.setRequestProperty("content-type", "text/plain");

      // set the message priority to low
      connection.setRequestProperty("x-msg-priority", "LOW");

      // Ensure we can get the output stream from the connection
      connection.setDoOutput(true);

      OutputStream outputStream = connection.getOutputStream();
      // wrapper the outputstream in a writer
      BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(
            outputStream));

      // Now write the actual content.
      // Make sure the CRLF is there in case some HTTP servers don't understand
      // that it's the end of the message
      writer.write(message + CRLF + CRLF);
      writer.flush();

      // now actually send the message
      connection.connect();

      boolean response = true;
      // check the response for errors
      if (connection.getResponseCode() != 200)
      {
         printErrorMessage(connection.getErrorStream());
         connection.disconnect();
         response = false;
      }
      connection.disconnect();
      return response;
   }

   /**
    * Reads the response from the input stream and prints it to standard out.
    * 
    * @param inputStream the input stream from the HTTPUrlConnection
    * @throws IOException if there is a problem reading from the stream
    */
   private void printErrorMessage(InputStream inputStream) throws IOException
   {
      System.out
            .println("There has been an error trying to POST a message to '"
                  + url + "'. HTTP error response follows...");

      BufferedReader reader = new BufferedReader(new InputStreamReader(
            inputStream));
      String line = null;
      while ((line = reader.readLine()) != null)
      {
         System.out.println(line);
      }
   }

   /**
    * Reads in a line from standard in.
    * 
    * @return null, if EOF was read in.
    * @throws IOException if there is a problem reading in from the command line
    */
   private String readLine() throws IOException
   {
      BufferedReader reader = new BufferedReader(new InputStreamReader(
            System.in));
      return reader.readLine();
   }

   /**
    * Main method which creates the HTTPPOST object and then calls its
    * sendMessagesFromConsole() method.
    * 
    * @param args the command line arguments either zero, [queueName] or
    *           [queueName host:port]
    */
   public static void main(String[] args)
   {
      HTTPPOST postExample = null;
      try
      {
         // args are OK start the program
         postExample = new HTTPPOST(args);
      }
      catch (MalformedURLException e)
      {
         // thrown if the host name or port is not in a good format.
         printUsage();
         System.exit(MALFORMED_URL_EXCEPTION_RC);
      }
      try
      {
         postExample.sendMessagesFromConsole();
      }
      catch (IOException e1)
      {
         System.out.println("Failed to POST a message to "
               + postExample.url.toString());
         e1.printStackTrace();
         System.exit(END_IOEXCEPTION_RC);
      }
   }

   /**
    * Prints the usage to System.out
    */
   private static void printUsage()
   {
      System.out.println("Usage: java HTTPPOST <Queue Name> (default="
            + DEFAULT_QUEUE + ") <host:port> (default=" + DEFAULT_HOSTANDPORT
            + ")");
   }
}
