       IDENTIFICATION DIVISION.
      ****************************************************************
      *                                                              *
      * Program name: AMQ0SUB0                                       *
      *                                                              *
      * Description: Sample COBOL program that subscribes and gets   *
      *              messages from a topic (example using MQSUB)     *
      *              A managed destination is used                   *
      *   <copyright                                                 *
      *   notice="lm-source-program"                                 *
      *   pids="5724-H72,"                                           *
      *   years="2008,2012"                                          *
      *   crc="176281364" >                                          *
      *   Licensed Materials - Property of IBM                       *
      *                                                              *
      *   5724-H72,                                                  *
      *                                                              *
      *   (C) Copyright IBM Corp. 2008, 2012 All Rights Reserved.    *
      *                                                              *
      *   US Government Users Restricted Rights - Use, duplication or  *
      *   disclosure restricted by GSA ADP Schedule Contract with    *
      *   IBM Corp.                                                  *
      *   </copyright>                                               *
      *                                                              *
      ****************************************************************
      *                                                              *
      * Function:                                                    *
      *                                                              *
      *                                                              *
      *   AMQ0SUB0 is a sample COBOL program to subscribe and get    *
      *   messages from a topic.                                     *
      *   It is an example of the use of MQPUT                       *
      *                                                              *
      *      -- subscribe non-durably to the topic specified         *
      *         as the first input is using the ACCEPT               *
      *                                                              *
      *      -- calls MQGET repeatedly to get messages from the      *
      *         topic and displays the message.                      *
      *                                                              *
      *      -- writes a message for each MQI reason other than      *
      *         MQRC-NONE; stops if there is a MQI completion        *
      *         code of MQCC-FAILED                                  *
      *                                                              *
      *    Program logic:                                            *
      *         display prompt for queue name                        *
      *         ACCEPT the target topic name from the console        *
      *         MQSUB topic for input                                *
      *         while no MQI failures,                               *
      *         .  MQGET next message                                *
      *         .  print the result                                  *
      *         MQCLOSE the topic                                    *
      *                                                              *
      *                                                              *
      ****************************************************************
      *                                                              *
      *                                                              *
      *                                                              *
      *   Exceptions signaled:  none                                 *
      *   Exceptions monitored: none                                 *
      *                                                              *
      *   AMQ0SUB0 has no parameters                                 *
      *                                                              *
      ****************************************************************
       PROGRAM-ID. 'AMQ0SUB0'.

      ****************************************************************
       DATA DIVISION.
       WORKING-STORAGE SECTION.
      *
      **  Declare MQI structures needed
      * MQI named constants
       01 MY-MQ-CONSTANTS.
          COPY CMQV.
      * Subscription Descriptor
       01 SUB-DESCRIPTOR.
          COPY CMQSDV.
      * Message Descriptor
       01 MESSAGE-DESCRIPTOR.
          COPY CMQMDV.
      * Put message options
       01 PMOPTIONS.
          COPY CMQPMOV.
      * Get Message Options
       01 GMOOPTIONS.
          COPY CMQGMOV.

      ** note, sample uses defaults where it can
       01 QM-NAME                    PIC X(48) VALUE SPACES.
       01 HCONN                      PIC S9(9) BINARY.
       01 Q-HANDLE                   PIC S9(9) BINARY.
       01 SUB-HANDLE                 PIC S9(9) BINARY.
       01 OPTIONS                    PIC S9(9) BINARY.
       01 COMPLETION-CODE            PIC S9(9) BINARY.
       01 S-COMP-CODE                PIC S9(9) BINARY.
       01 CON-REASON                 PIC S9(9) BINARY.
       01 REASON                     PIC S9(9) BINARY.
       01 BUFFER                     PIC X(101).
       01 BUFFER-LENGTH              PIC S9(9) BINARY.
       01 MESS-LEN                   PIC S9(9) BINARY.
       01 TOPIC-LENGTH               PIC S9(9) BINARY VALUE 0.
       01 REAL-TOPIC-LENGTH          PIC S9(9) BINARY VALUE 0.
       01 WORK-FIELD                 PIC X(60).
       01 TARGET-TOPIC               PIC X(48).

      ****************************************************************
       PROCEDURE DIVISION.
       P0.
      ** indicate that sample program has started
           DISPLAY 'AMQ0SUB0 start'.

      ****************************************************************
      *                                                              *
      *    Display prompt for the topic string                       *
      *                                                              *
      ****************************************************************
           DISPLAY 'Please enter the topic string '

      ** get the target topic from StdIn.
           ACCEPT TARGET-TOPIC FROM CONSOLE.

      ****************************************************************
      *                                                              *
      *   Connect to default queue manager                           *
      *                                                              *
      ****************************************************************
           CALL 'MQCONN'
            USING QM-NAME, HCONN,
            COMPLETION-CODE, CON-REASON.

      *      report reason and stop if it failed
           IF COMPLETION-CODE IS EQUAL TO MQCC-FAILED
             DISPLAY 'MQCONN ended with reason code ' CON-REASON
             MOVE CON-REASON TO RETURN-CODE
             GOBACK
             END-IF.

      ****************************************************************
      *                                                              *
      *    Display name of target queue read from StdIn              *
      *                                                              *
      ****************************************************************
           DISPLAY 'target topic is ' TARGET-TOPIC.

      ****************************************************************
      *                                                              *
      *   Subscribe to the provided topic string (and fail if        *
      *   MQM is quiescing)                                          *
      *                                                              *
      ****************************************************************
       SUBSCRIBE.
      *     Work out how many trailing spaces we have to determine
      *     real length of TOPIC STRING. Need to reverse the string
      *     as there is no TRAILING keyword in some COBOL
      *     implementations.
           MOVE FUNCTION REVERSE (TARGET-TOPIC) TO WORK-FIELD.
           inspect WORK-FIELD Tallying TOPIC-LENGTH
               For LEADING Spaces.
           subtract TOPIC-LENGTH from LENGTH OF TARGET-TOPIC
               giving REAL-TOPIC-LENGTH.
           move REAL-TOPIC-LENGTH to MQSD-OBJECTSTRING-VSLENGTH.
           set MQSD-OBJECTSTRING-VSPTR to ADDRESS OF TARGET-TOPIC.


           MOVE MQCO-NONE TO OPTIONS.
           ADD MQSO-CREATE MQSO-NON-DURABLE MQSO-FAIL-IF-QUIESCING
               MQSO-MANAGED GIVING MQSD-OPTIONS.
           CALL 'MQSUB'
            USING HCONN, SUB-DESCRIPTOR,
            Q-HANDLE, SUB-HANDLE,
            S-COMP-CODE, REASON.

      *      report reason, if any; stop if failed
           IF REASON IS NOT EQUAL TO MQRC-NONE
             DISPLAY 'MQSUB ended with reason code ' REASON
             END-IF.

           IF S-COMP-CODE IS EQUAL TO MQCC-FAILED
             DISPLAY 'unable to subscribe to topic'
             MOVE REASON TO RETURN-CODE
             GOBACK
             END-IF.

      ****************************************************************
      *                                                              *
      *   Get messages that have been subscribed to                  *
      *                                                              *
      ****************************************************************
       GETS.
           add MQGMO-WAIT MQGMO-NO-SYNCPOINT MQGMO-CONVERT
               giving MQGMO-OPTIONS.
           move 30000 to MQGMO-WAITINTERVAL.

           MOVE S-COMP-CODE TO COMPLETION-CODE.
           PERFORM GET-SUB WITH TEST BEFORE
             UNTIL COMPLETION-CODE IS EQUAL TO MQCC-FAILED.

      ****************************************************************
      *                                                              *
      *   Close the subscription handle                              *
      *                                                              *
      ****************************************************************
       CLOSES.
           if S-COMP-CODE IS NOT EQUAL TO MQCC-FAILED
               MOVE MQCO-NONE TO OPTIONS
               CALL 'MQCLOSE'
                USING HCONN, SUB-HANDLE, OPTIONS,
                COMPLETION-CODE, REASON

      *          report reason, if any
               IF REASON IS NOT EQUAL TO MQRC-NONE
                 DISPLAY 'MQCLOSE ended with reason code ' REASON
               end-if
            END-IF.

      ****************************************************************
      *                                                              *
      *   Close the managed destination queue (if it was opened)     *
      *                                                              *
      ****************************************************************
           if S-COMP-CODE IS NOT EQUAL TO MQCC-FAILED
               MOVE MQCO-NONE TO OPTIONS
               CALL 'MQCLOSE'
                USING HCONN, Q-HANDLE, OPTIONS,
                COMPLETION-CODE, REASON

      *          report reason, if any
               IF REASON IS NOT EQUAL TO MQRC-NONE
                 DISPLAY 'MQCLOSE ended with reason code ' REASON
               end-if
             END-IF.



      ****************************************************************
      *                                                              *
      *  Disconnect from queue manager (if not previously connected) *
      *                                                              *
      ****************************************************************
       DISCS.
           IF CON-REASON IS NOT EQUAL TO MQRC-ALREADY-CONNECTED
             CALL 'MQDISC'
              USING HCONN, COMPLETION-CODE, REASON

      *      report reason, if any
             IF REASON IS NOT EQUAL TO MQRC-NONE
               DISPLAY 'MQDISC ended with reason code ' REASON
             END-IF
           END-IF.

       OVER.
      ** indicate that sample program has finished
           DISPLAY 'Sample AMQ0SUB0 end'.
           MOVE ZERO TO RETURN-CODE.
           GOBACK.

      ****************************************************************
      *                                                              *
      *   Get one request from subscription destination queue        *
      *                                                              *
      ****************************************************************
       GET-SUB.

      *       MQGET sets Encoding and CodedCharSet to values in the
      *       message returned, so these fields should be reset to
      *       the default values before every call, as MQGMO-CONVERT
      *       is specified
           MOVE MQENC-NATIVE to MQMD-ENCODING.
           MOVE MQCCSI-Q-MGR to MQMD-CODEDCHARSETID.

      *      In order to read the messages in sequence, MsgId and
      *      CorrelId must have the default value. MQGET sets them
      *      to the values in for message it returns, so re-iniialise
      *      them before every call
           move MQMI-NONE to MQMD-MSGID.
           move MQCI-NONE to MQMD-CORRELID.

           move 100 to BUFFER-LENGTH.

           display 'Calling MQGET : 30 seconds wait time'.

           CALL 'MQGET'
            USING HCONN, Q-HANDLE,
            MESSAGE-DESCRIPTOR, GMOOPTIONS,
            BUFFER-LENGTH, BUFFER, MESS-LEN,
            COMPLETION-CODE, REASON.

      *      report reason, if any
           IF REASON IS NOT EQUAL TO MQRC-NONE
               if reason IS EQUAL TO MQRC-NO-MSG-AVAILABLE
                   display 'No more messages available.'
               else
                  DISPLAY 'MQGET ended with reason code ' REASON
      *               treat truncated message as failure for this sample
                  if REASON is EQUAL TO MQRC-TRUNCATED-MSG-FAILED
                     move MQCC-FAILED to COMPLETION-CODE
                     display 'BUFFER LENGTH : ' BUFFER-LENGTH
                     display 'MESSAGE LENGTH :' MESS-LEN
                     end-if
               END-IF
           END-if.

      *       display each message received
           if COMPLETION-CODE IS NOT EQUAL TO MQCC-FAILED
               display 'message <' BUFFER
               end-if.


      ****************************************************************
      *                                                              *
      * END OF AMQ0SUB0                                              *
      *                                                              *
      ****************************************************************
