#-------------------------------------------------------------------------------
#  addAnnotations.pm:
#  Scripting Facility to add annotation to the page
#
#  Call:
#
#  On Windows:    afp2web.exe -q -c -doc_cold -sp:sfsamples\addAnnotations.pm samples\insure.afp
#
#  On Unix:    ./afp2web   -q -c -doc_cold -sp:sfsamples/addAnnotations.pm samples/insure.afp
#
#  Author  : Fa. OXSEED
#  Date    : 2010-01-04
#  Version : 1.0.2
#
#  $V100   2005-08-01    Initial Release
#
#  $V101   2008-07-10    Extended to add script file path as module search path
#                        on "BEGIN" block
#
#  $V102   2010-01-04    a. Extended to utilize
#                           "a2w::ConfigConstants"
#                           "a2w::DocumentConstants"
#                           "a2w::PageConstants"
#                           "a2w::FontConstants"
#                           packages
#                        b. Optimized constant declarations
#                        c. Modified PERL "@INC" extending to consider the
#                           Scripting Facility samples grouping under "sfsamples"
#                           directory
#-------------------------------------------------------------------------------

# $V101 Begin
#-----------------------------------------------------------------------
# BEGIN block of module
#
# Extends PERL module search path array (@INC) with new element having
# this script modules path in order to have better module portability
#-----------------------------------------------------------------------
BEGIN {
    #---- Fetch script filename
    my $sScriptFilenameTmp = $0;

    #---- Extract script file path from script filename
    my $sScriptFilePathTmp = "";
    if ( $sScriptFilenameTmp =~ /(.*)\/.*\.pm/ ){
        $sScriptFilePathTmp = $1;
    }

    #printf STDERR ( "Script filename: " . $0 . " Script filepath: " . $sScriptFilePathTmp . "\n" );
    # $V102 Begin
    if ( $sScriptFilePathTmp eq "" ){
        $sScriptFilePathTmp = ".";
    }
    else {
        my $sScriptFileParentPathTmp = "";
        if ( $sScriptFilePathTmp =~ /(.*)\/sfsamples/ ){
            $sScriptFileParentPathTmp = $1;
        }

        #---- Add script file parent path to module search path
        if ( $sScriptFileParentPathTmp ne "" ){
            unshift( @INC, $sScriptFileParentPathTmp );
        }
    }
    # $V102 End

    #---- Add script file path to module search path
    unshift( @INC, $sScriptFilePathTmp );
}
# $V101 End

use a2w::Config;
use a2w::Document;
use a2w::Font;
use a2w::Index;
use a2w::Kernel;
use a2w::Line;
use a2w::MediumMap;
use a2w::NOP;
use a2w::Overlay;
use a2w::Page;
use a2w::PSEG;
use a2w::Text;
# $V102 Begin
use a2w::ConfigConstants;
use a2w::DocumentConstants;
use a2w::PageConstants;
use a2w::FontConstants;
# $V102 End

#-----------------------------------------------------------------------
# Initialize once per process
#-----------------------------------------------------------------------
sub initialize(){

    #---- Get Parameter of initialize( Par: a2w::Config, a2w::Kernel )
    ( $a2wConfigPar, $a2wKernelPar ) = @_;

    #---- Define boolean values
    $TRUE  = 1;    # TRUE  boolean value
    $FALSE = 0;    # FALSE boolean value

    #---- Set/Reset Logging
    $bLog = $FALSE;
    if (index( lc($a2wConfigPar->getAttribute( $a2w::ConfigConstants::LOGGINGLEVEL )), "sf" ) >= 0 ){    # $V102 Change
        $bLog = $TRUE;
    }

    my $sScriptProcTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SCRIPTPROCEDURE );    # $V102 Change
    my $sScriptArgsTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SCRIPTARGUMENT );
    $sIndexFilePath    = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::INDEXPATH );
    $sOutputFilePath   = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::OUTPUTFILEPATH );
    $sSpoolFilename    = $a2wKernelPar->getSpoolFilename();

    if ( $bLog == $TRUE ){
        printf STDERR ( "Running $sScriptProcTmp...\n" );
        printf STDERR ( "initialize(): Processing $sSpoolFilename\n" );
        printf STDERR ( "initialize(): Args: $sScriptArgsTmp, IndexFilePath: $sIndexFilePath, OutputFilePath: $sOutputFilePath\n" );
    }    

    # $V102 Begin
    #---- Page process flags
    $APPEND = 0;    # append page to Current Document
    $SKIP   = 1;    # skip page
    $NEWDOC = 2;    # new document
    # $V102 End

    #---- Reset Page Id
    $iPageId = 0;

    #---- Set AutoSplit on
    $a2wConfigPar->setAttribute( $a2w::ConfigConstants::AUTOSPLIT, "on" );    # $V102 Change

    # -- Assuming width of a character as 2.53 millimeter
    # -- It is used to calculate Hyper Link Rectangle Width as below
    # -- Hyper Link Rect Width (in millimeters) = Hyper Link Text Length * $fCharWidth 
    $fCharWidth = 2.53;

    #--- Hyper Link Rectangle Height values in millimeters --- 
    $fHPLHeight    = 2.6;    
    
    #--- Hyper Link Rectangle Border Width value in millimeters (approximately 1.0 Point)
    #--- Set BorderWidth to zero if border is not needed.
    $fBorderWidth = 0.35;

    #--- Hyper Link Action URL
    $sUrlText = "http://afp2web.com";

    #--- Hyper Link Border color as RGB value (default color blue is assigned here)
    $cHPLColor=0x000000FF;        #---- Blue (0x00RRGGBB)
#    $cHPLColor=0x00FF0000;        #---- Red
#    $cHPLColor=0x0000FF00;        #---- Green
#    $cHPLColor=0x00FFFFFF;        #---- White
#    $cHPLColor=0x00000000;        #---- Black

    return 0;
}

#-----------------------------------------------------------------------
# InitializeDoc for each document
#-----------------------------------------------------------------------
sub initializeDoc(){

    #---- Get Parameter of initializeDoc( Par: a2w::Document )
    ($a2wDocumentPar) = @_;

    if ( $bLog == $TRUE ){
        printf STDERR ( "initializeDoc(): DocId " . $a2wDocumentPar->getId() . "\n" );
    }

    #---- Reset Page Id
    $iPageId = 0;

    return 0;
}

#-----------------------------------------------------------------------
# InitializePage for each page
#-----------------------------------------------------------------------
sub initializePage(){

    #---- Get Parameter of initializePage( Par: a2w::Page )
    ($a2wPagePar) = @_;

    if ( $bLog == $TRUE ){
        printf STDERR ( "initializePage()\n" );
    }
    return 0;
}

#-----------------------------------------------------------------------
# Main entry method
# Return values:
#        < 0:    error
#         0:    append page to Current Document
#         1:    skip page
#         2:    first page / new document
#-----------------------------------------------------------------------
sub afp2web(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "afp2web(): PageId " . $a2wPagePar->getParseId() . "\n" );
    }

    #---- Set default return value
    my $iRetTmp = $APPEND; # default: append page

    #---- Increment Page Id
    $iPageId++;

    #---- Add annotation to the first page of every document.
    if( $iPageId == 1 ) {

        #---- We are looking for a text object that has a insured customer name 
        #---- for example "Geoffrey R Stephens" in the first page of insure.afp
        #---- Area: 2438<= x <= 2458 and 5337 <= y <= 5357

        #---- Fetch first Text Object
        my $a2wTextTmp = $a2wPagePar->getFirstText();

        #---- Define temp variables
        my $sTextTmp        = "";
        my $iTextXPosTmp    = 0;
        my $iTextYPosTmp    = 0;
        my $iTextLenTmp        = 0 ;

        #--- Get Page Resolution 
        my $iPageResTmp    = $a2wPagePar->getResolution();

        #--- Convert Rectangle Height from millimeters to AFP units
        my $iHeightTmp    = $iPageResTmp * $fHPLHeight / 25.4;

        #--- Convert Rectangle Border Width from millimeters to AFP units
        my $fBorderWidthTmp = $iPageResTmp * $fBorderWidth / 25.4;

        #--- Border is drawn on top and bottom sides of a rectangle
        #--- So increase height to include border width
        $iHeightTmp    = $iHeightTmp + ( 2 * $fBorderWidthTmp );

        #---- Loop thru all the Text Objects
        while ( $a2wTextTmp != 0 ){
            
            $sTextTmp = $a2wTextTmp->getText();
            $iTextXPosTmp = $a2wTextTmp->getXPos();
            $iTextYPosTmp = $a2wTextTmp->getYPos();

            #---- If true ==> text object is a customer name 
            if ( $iTextXPosTmp >= 2438    
                 && $iTextXPosTmp <= 2458
                 && $iTextYPosTmp >= 5337
                 && $iTextYPosTmp <= 5357
               ){

                $iTextLenTmp = length( $sTextTmp );

                #--- Calcualte HyperLink Rectangle Width
                $iWidthTmp = $iTextLenTmp * $fCharWidth * $iPageResTmp / 25.4;

                #--- Border is drawn on both left and right sides
                #--- So increase width to include border width
                $iWidthTmp    = $iWidthTmp  + ( 2 * $fBorderWidthTmp );

                #--- CheckXPosTmp is X Position of lower left corner of a TextBox.
                #--- PDF Draws border inside the rectangle.
                #--- So move CheckXPosTmp position to the left by $fBorderWidthTmp 
                #--- so that it does not overlap the first character
                my $fCheckXPosTmp = $iTextXPosTmp  - $fBorderWidthTmp ;

                #--- CheckYPosTmp is Y Position of lower left corner of a TextBox.
                #--- PDF Draws border inside the rectangle, so move CheckYPosTmp position 
                #--- to the bottom by $fBorderWidthTmp so that it does not overlap the characters
                #--- Since addAnnotation API requires Upper left corner of rectangle,
                #--- subtract rectangle height from CheckYPosTmp 
                my $fCheckYPosTmp = $iTextYPosTmp + $fBorderWidthTmp - $iHeightTmp;

                $a2wPagePar->addAnnotation( $fCheckXPosTmp          #---- Upper left X position of annotation box (in AFP page units)
                                            , $fCheckYPosTmp        #---- Upper left Y position of annotation box (in AFP page units)
                                            , $iWidthTmp            #---- Width of annotation box (in AFP page units)
                                            , $iHeightTmp           #---- Height of annotation box (in AFP page units)
                                            , $sUrlText             #---- URL
                                            , $fBorderWidthTmp       #---- Annotation border width
                                            , $cHPLColor               #---- Annotation border color (Format=0x00BBGGRR)
                                         );
        
                if ( $bLog == $TRUE ){
                    printf STDERR ( "Annotation added @(" . $fCheckXPosTmp . "," . $fCheckYPosTmp . ")>" . $sTextTmp . "<\n" );
                }
                last; # leave while loop
            }

            #---- Fetch next Text Object
            $a2wTextTmp = $a2wPagePar->getNextText();
        }
    }
    return $iRetTmp;
}

#-----------------------------------------------------------------------
# FinalizePage for each page
#-----------------------------------------------------------------------
sub finalizePage(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "finalizePage()\n" );
    }
    return 0;
}

#-----------------------------------------------------------------------
# FinalizeDoc for each document
#-----------------------------------------------------------------------
sub finalizeDoc(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "finalizeDoc()\n" );
    }
    return 0;
}

#-----------------------------------------------------------------------
# Finalize once per process
#-----------------------------------------------------------------------
sub finalize(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "finalize()\n" );
    }
    return 0;
}

__END__
