#-------------------------------------------------------------------------------
#  a2w::core::dm::Block.pm
#  Data Mining Framework Block
#
#  Author  : AFP2web Team, Maas Holding GmbH
#
#  $V100   2014-05-08    Initial Release
#
#  $V101   2015-05-20    - Fixed minor bug in image positioning within block  (AFP-224)
#                        - Extended to dump block contents when log is set to debug
#
#  $V102   2015-09-28    Extended to group texts based on font (identifier) (AFP-297)
#
#  $V103   2015-10-15    Extended to pass in custom block info like HTML style (AFP-298)
#
#  $V104   2015-11-06    - Extended with "isParagraph", "isTable" and "isSkip" functions
#                        - Extended with "ContentHeight" attribute
#
#  $V105   2017-10-30    Extended with 'PrimaryColumns' attribute to determine multiline
#                        rows. i.e, when primary column cells are empty then row is considered
#                        as following line of current row (AFP-456)
#
#  $V106   2017-12-14    Extended to support json formatted (as occur in AFP TLE value) document block value (AFP-624)
#
#  $V107   2018-01-19    Extended to process illustration block type
#
#  $V108   2018-01-24    a. Extended to update block attributes
#                        b. Extended to force blocks as filled to have anchors less block definition
#                        c. Extended to have multiple constraints to have OR conditionals
#
#  $V109   2018-04-27    Extended to evaluate block width when not set and start anchor is detected
#                        as given below
#                        Block width = Page Width - StartX
#
#  $V110   2018-07-31    AFP-716: Modified to assert block info (json value) to be multiline
#
#  $V111   2018-09-03    a. OXS-8490: Extended with RAW block type
#                        b. OXS-8490: Extended to get block indexes
#                        c. OXS-8490: Extended with parser anchors to preprocess objects and to invoke callback
#                           when anchor matching object is found
#                        d. Made "_createAnchor" API as public "createAnchor" API
#                        e. OXS-8490: Fixed minor issue in evaluating block height
#
#  $V112   2018-10-03    a. AFP-743: Extended with composite block
#                        b. Extended to evaluate end Y based on configured block height
#                        c. Fixed minor bug (incorrect position info of object is used) in doesObjectFallIn function 
#                        d. Extended to determine block is filled along with chained blocks
#                        e. Extended with table caption support
#
#  $V113   2018-10-26    a. AFP-756: Extended with data mining APIs to find and fetch eyecatcher/value
#
#  $V114   2018-11-29    a. AFP-771: Extended with list block
#
#  $V115   2018-12-10    AFP-772: Extended to tag line objects under one <P> tag based on "TagLineAsParagraph" flag
#
#  $V116   2019-01-28    a. OXS-8853: Extended block with output context hash object (where output visitor decides the attributes of this context)
#
#-------------------------------------------------------------------------------
package a2w::core::dm::Block;

#-----------------------------------------------------------------------
# Include required modules
#-----------------------------------------------------------------------
use a2w::TypeConstants;
use a2w::core::log::Logger;
use a2w::core::dm::Constants;

use a2w::core::dm::Anchor;
use a2w::core::dm::ContentProcessor;
use a2w::core::dm::ParagraphContentDef;
use a2w::core::dm::SkipContentDef;
use a2w::core::dm::TableContentDef;
use a2w::core::dm::ListContentDef; # $V114 Change
use a2w::core::dm::IllustrationContentDef; # $V107 Change
use a2w::core::dm::RawContentDef; # $V111 Change
use a2w::core::dm::CompositeContentDef; # $V112 Change

use a2w::core::dm::ParagraphBlockFormatter;
use a2w::core::dm::SkipBlockFormatter;
use a2w::core::dm::TableBlockFormatter;
use a2w::core::dm::ListBlockFormatter; # $V114 Change
use a2w::core::dm::IllustrationBlockFormatter; # $V107 Change
use a2w::core::dm::RawBlockFormatter; # $V111 Change
use a2w::core::dm::CompositeBlockFormatter; # $V112 Change

use a2w::core::dm::MiningUtils; # $V113 Change

use a2w::core::visitor::Visitor;

use JSON::Tiny; # $V106 Change

#-----------------------------------------------------------------------
# Constructor
#-----------------------------------------------------------------------
sub new{
    my $proto = shift;
    my $class = ref( $proto ) || $proto;

    #---- Define boolean values
    $TRUE  = $a2w::TypeConstants::TRUE;     # TRUE  boolean value
    $FALSE = $a2w::TypeConstants::FALSE;    # FALSE boolean value

    my $this = {
        #---- Config attributes ----#
        # Identifier
          'Id' => ''

        # Start anchor
        , 'StartAnchor' => undef

        # End anchor
        , 'EndAnchor' => undef

        # Width of block
        , 'Width' => 0

        # Height of block
        , 'Height' => 0

        # Content definition
        , 'ContentDef' => undef

        # Id of next block in chain
        , 'Next' => undef

        # Quantifier - Block repetition quantity
        # Quantifier is optional
        # If defined, specifies repetition of block. Possible values are
        #
        # Quantifier Description
        #
        # 1          Block occurs only once (default)
        # ?          Block occurs zero or once
        # *          Block occurs zero or many
        # +          Block occurs minimum once and many (!!!???)
        # n          Block occurs exactly 'n' count
        , 'Quantifier' => '1'

        # Content Processor
        , 'ContentProcessor' => undef

        #---- Processing attributes ----#
        # Block formatter
        , 'Formatter' => undef

        # Reference to next block in chain
        , 'NextRef' => undef

        # List of objects belong to this block (array reference of objects)
        , 'ObjsList' => []

        # List of starting anchor objects of this block (array reference of objects)
        , 'StAncList' => []

        # List of ending anchor objects of this block (array reference of objects)
        , 'EdAncList' => []

        # Block starting X position (starting anchor's X position)
        , 'StartX' => 0

        # Block starting Y position (starting anchor's Y position)
        , 'StartY' => 0

        # Block starting Y position as on start page
        , 'BeginY' => 0

        # Block ending X position (starting anchor's X position + block width)
        , 'EndX' => 0

        # Block ending Y position (ending anchor's Y position)
        , 'EndY' => 0

        # Flag indicating whether start anchor found or not
        , 'StartFound' => $FALSE

        # Flag indicating whether end anchor found or not
        , 'EndFound' => $FALSE

        # Page id where block started
        , 'StartedOn' => 0

        # Flag indicating whether block is flushed or not
        , 'Flushed' => $FALSE

        # Block configuration reference (used to create repetitive blocks)
        , 'ConfigReference' => undef

        # Repeat count
        , 'RepeatCount' => 1

        # $V103 Begin
        , 'Info' => ''
        # $V103 End

        # $V104 Begin
        , 'ContentHeight' => 0
        # $V104 End

        # $V106 Begin
        , 'JSONInfo' => undef
        # $V106 End

        # $V108 Begin
        # Flag indicating whether block is like a region
        # - Auto detected as region if block defines attributes StartX, StartY, Width and Height
        # OR
        # - Auto detected as region if block defines attributes StartX, StartY, EndX and EndY
        # OR
        # - Set using 'Region=$TRUE' on block definition
        #
        , 'Region' => $FALSE

        # Flags indicating whether attributes are configured in block definition
        , 'StartXDefined' => $FALSE
        , 'StartYDefined' => $FALSE
        , 'EndXDefined' => $FALSE
        , 'EndYDefined' => $FALSE
        , 'WidthDefined' => $FALSE
        , 'HeightDefined' => $FALSE
        # $V108 End

        # $V112 Begin
        # Flag indicating whether current block is main or sub block
        , 'SubBlock' => $FALSE
        # $V112 End

        # $V116 Begin
        # Output context hash object (where output visitor decides the attributes of this context)
        , 'OutputContext' => undef
        # $V116 End
    };

    bless( $this, $class );

    #---- Evaluate unique id for this instance
    $this->{ 'UniqueId' } = "$this";

    #---- Get logger
    our $theLogger = a2w::core::log::Logger->getSingleton();
    our $bLog = $theLogger->isRegistered( __PACKAGE__ );

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "new()" );
    #}
    return $this;
}

#-----------------------------------------------------------------------
# Destructor
#-----------------------------------------------------------------------
sub DESTROY{
    my $this = shift;
}

#-----------------------------------------------------------------------
# Mutators
#-----------------------------------------------------------------------
sub setId{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setId()" );
    #}

    $this->{ 'Id' } = shift;
}

sub setStartAnchor{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setStartAnchor()" );
    #}

    $this->{ 'StartAnchor' } = shift;
}

sub setEndAnchor{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setEndAnchor()" );
    #}

    $this->{ 'EndAnchor' } = shift;
}

sub setWidth{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setWidth()" );
    #}

    $this->{ 'Width' } = shift;
}

sub setHeight{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setHeight()" );
    #}

    $this->{ 'Height' } = shift;
}

sub setContentDef{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setContentDef()" );
    #}

    $this->{ 'ContentDef' } = shift;
}

sub setNext{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setNext()" );
    #}

    $this->{ 'Next' } = shift;
}

sub setQuantifier{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setQuantifier()" );
    #}

    $this->{ 'Quantifier' } = shift;
}

sub setContentProcessor{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setContentProcessor()" );
    #}

    $this->{ 'ContentProcessor' } = shift;
}

sub setNextRef{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setNextRef()" );
    #}

    $this->{ 'NextRef' } = shift;
}

# V112 Begin
sub setObjsList{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setObjsList()" );
    #}

    $this->{ 'ObjsList' } = shift;
}
# V112 End

sub setStartX{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setStartX()" );
    #}

    $this->{ 'StartX' } = shift;
}

sub setStartY{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setStartY()" );
    #}

    $this->{ 'StartY' } = shift;
}

sub setEndX{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setEndX()" );
    #}

    $this->{ 'EndX' } = shift;
}

sub setEndY{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setEndY()" );
    #}

    $this->{ 'EndY' } = shift;
}

sub setStartedOn{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setStartedOn()" );
    #}

    $this->{ 'StartedOn' } = shift;
}

sub setFlushed{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setFlushed()" );
    #}

    $this->{ 'Flushed' } = shift;
}

sub setConfigReference{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setConfigReference()" );
    #}

    $this->{ 'ConfigReference' } = shift;
}

sub setRepeatCount{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setRepeatCount()" );
    #}

    $this->{ 'RepeatCount' } = shift;
}

# $V103 Begin
sub setInfo{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setInfo()" );
    #}

    $this->{ 'Info' } = shift;

    # $V106 Begin
    my $sBlockInfoTmp = $this->{ 'Info' };

    #---- Assert whether info is string or json and set appropriately on block
    #     NOTE: JSON value is determined by '{' and '}' characters at beginning and ending of info
    #if ( $sBlockInfoTmp =~ /^\s*{.*}\s*$/ ){ # info is json
    if ( $sBlockInfoTmp =~ /^\s*{.*}\s*$/s ){ # info is json # $V110 Change
        #if ( $bLog == $TRUE ){ $theLogger->logMessage( "Parsing JSON info (" . $sBlockInfoTmp . ") of block (" . $this->{ 'Id' } . ")" ); }

        #---- Parse json data
        my $hrefJSONTmp = JSON::Tiny::from_json( $sBlockInfoTmp );
        if ( ref( $hrefJSONTmp ) ne "HASH" ){
            if ( $bLog == $TRUE ){ $theLogger->logMessage( "Unable to parse block " . $this->{ 'Id' } . " JSON info ($sBlockInfoTmp). Reason: $hrefJSONTmp" ); }
            return ( -2, "Unable to parse block " . $this->{ 'Id' } . " JSON info ($sBlockInfoTmp). Reason: $hrefJSONTmp" );
        }
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Parsed JSON info (" . $sBlockInfoTmp . ") of block (" . $this->{ 'Id' } . ")" ); }

        $this->setJSONInfo( $hrefJSONTmp );

        # V112 Begin
        # Process and set sub block info in case of composite block
        my $contDefTmp = $this->getContentDef();
        if ( lc( $contDefTmp->getType() ) eq "composite" ){
            if ( defined( $hrefJSONTmp->{ 'subblock' } ) ){
                my @arrSubBlockIdsTmp = sort keys( %{ $hrefJSONTmp->{ 'subblock' } } );
                my @arrSubBlocksTmp = @{ $contDefTmp->getSubBlockObjects() };
                foreach my $id ( @arrSubBlockIdsTmp ){
                    my @arrMatchTmp = grep { $_->getId() eq $id } @arrSubBlocksTmp;
                    foreach my $blk ( @arrMatchTmp ){ $blk->setJSONInfo( $hrefJSONTmp->{ 'subblock' }{ $id } ); }
                }
            }
        }
        # V112 End
    }
    # $V106 End
}
# $V103 End

# $V104 Begin
sub setContentHeight{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setContentHeight()" );
    #}

    $this->{ 'ContentHeight' } = shift;
}
# $V104 End

# $V106 Begin
sub setJSONInfo{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setJSONInfo()" );
    #}

    $this->{ 'JSONInfo' } = shift;
}
# $V106 End

# $V108 Begin
sub setRegion{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setRegion()" );
    #}

    $this->{ 'Region' } = shift;
}
# $V108 End

# $V112 Begin
sub setSubBlock{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "setSubBlock()" ); }

    $this->{ 'SubBlock' } = shift;
}
# $V112 End

# $V116 Begin
sub setOutputContext{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "setOutputContext()" ); }

    $this->{ 'OutputContext' } = shift;
}
# $V116 End

#-----------------------------------------------------------------------
# Accessors
#-----------------------------------------------------------------------
sub getId{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getId()" );
    #}

    return $this->{ 'Id' };
}

sub getStartAnchor{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getStartAnchor()" );
    #}

    return $this->{ 'StartAnchor' };
}

sub getEndAnchor{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getEndAnchor()" );
    #}

    return $this->{ 'EndAnchor' };
}

sub getWidth{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getWidth()" );
    #}

    return $this->{ 'Width' };
}

sub getHeight{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getHeight()" );
    #}

    return $this->{ 'Height' };
}

sub getContentDef{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getContentDef()" );
    #}

    return $this->{ 'ContentDef' };
}

sub getNext{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getNext()" );
    #}

    return $this->{ 'Next' };
}

sub getQuantifier{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getQuantifier()" );
    #}

    return $this->{ 'Quantifier' };
}

sub getContentProcessor{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getContentProcessor()" );
    #}

    return $this->{ 'ContentProcessor' };
}

sub getNextRef{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getNextRef()" );
    #}

    return $this->{ 'NextRef' };
}

sub getObjsList{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getObjsList()" );
    #}

    return $this->{ 'ObjsList' };
}

sub getStartAnchorsList{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getStartAnchorsList()" );
    #}

    return $this->{ 'StAncList' };
}

sub getEndAnchorsList{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getEndAnchorsList()" );
    #}

    return $this->{ 'EdAncList' };
}

sub getStartX{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getStartX()" );
    #}

    return $this->{ 'StartX' };
}

sub getStartY{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getStartY()" );
    #}

    return $this->{ 'StartY' };
}

sub getBeginY{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getBeginY()" );
    #}

    return $this->{ 'BeginY' };
}

sub getEndX{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getEndX()" );
    #}

    return $this->{ 'EndX' };
}

sub getEndY{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getEndY()" );
    #}

    return $this->{ 'EndY' };
}

sub getFormatter{
    $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getFormatter()" );
    #}

    return $this->{ 'Formatter' };
}

sub hasStarted{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "hasStarted()" );
    #}

    return $this->{ 'StartFound' };
}

sub hasEnded{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "hasEnded()" );
    #}

    return $this->{ 'EndFound' };
}

sub getStartedOn{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getStartedOn()" );
    #}

    return $this->{ 'StartedOn' };
}

sub isFlushed{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "isFlushed()" );
    #}

    return $this->{ 'Flushed' };
}

sub getConfigReference{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getConfigReference()" );
    #}

    return $this->{ 'ConfigReference' };
}

sub getConfId{
    $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getConfId()" );
    #}

    return $this->{ 'ConfigReference' }{ 'Id' };
}

sub getUniqueId{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getUniqueId()" );
    #}

    return $this->{ 'UniqueId' };
}

sub getRepeatCount{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getRepeatCount()" );
    #}

    return $this->{ 'RepeatCount' };
}

# $V103 Begin
sub getInfo{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getInfo()" );
    #}

    return $this->{ 'Info' };
}
# $V103 End

# $V104 Begin
sub getContentHeight{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getContentHeight()" );
    #}

    #---- Evaluate if block is chained
    my $iContHtTmp = $this->{ 'ContentHeight' };
    if ( $this->isChained() == $TRUE ){
        #---- Get chained block
        $iContHtTmp += $this->{ 'NextRef' }->getContentHeight();
    }
    return $iContHtTmp;
}
# $V104 End

# $V106 Begin
sub getJSONInfo{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getJSONInfo()" );
    #}

    return $this->{ 'JSONInfo' };
}
# $V106 End

# $V108 Begin
sub isRegion{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getRegion()" );
    #}

    return $this->{ 'Region' };
}
# $V108 End

# $V112 Begin
sub isSubBlock{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "isSubBlock()" ); }

    return $this->{ 'SubBlock' };
}
# $V112 End

# $V116 Begin
sub getOutputContext{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "getOutputContext()" ); }

    return $this->{ 'OutputContext' };
}
# $V116 End

#-----------------------------------------------------------------------
# Workers
#-----------------------------------------------------------------------
#-----------------------------------------------------------------------
# updateBoundary
#-----------------------------------------------------------------------
sub updateBoundary{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "updateBoundary()" );
    #}

    #---- Get parameter
    #
    # 1. Kernel object
    # 2. POM object
    #
    my $a2wObjectPar = shift;
    my $pomObjectPar = shift;

    my $iXPosTmp = $pomObjectPar->{ $a2w::core::dm::Constants::AT_XPOS };
    my $iYPosTmp = $pomObjectPar->{ $a2w::core::dm::Constants::AT_YPOS };
    my $iTextHeightTmp = 0;
    if ( $pomObjectPar->{ $a2w::core::dm::Constants::AT_OBJTYPE } == $a2w::core::dm::Constants::OT_TEXT ){
	    $iTextHeightTmp = $pomObjectPar->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_FONTHT };
	    $iTextHeightTmp *= ( $pomObjectPar->{ $a2w::core::dm::Constants::AT_RESOLUTION } / 72 );
	    $iTextHeightTmp = int( $iTextHeightTmp + .5 * ( $iTextHeightTmp <=> 0 ) );
	    $iYPosTmp -= $iTextHeightTmp;
	}

    # $V108 Begin
    #---- Evaluate block top-left
    if ( $this->{ 'StartXDefined' } == $FALSE ){
        if (    $iXPosTmp < $this->{ 'StartX' }
             || $this->{ 'StartX' } == 0
           ){
            $this->{ 'StartX' } = $iXPosTmp;
        }
    }

    if ( $this->{ 'StartYDefined' } == $FALSE ){
        if (    $iYPosTmp < $this->{ 'StartY' }
             || $this->{ 'StartY' } == 0
           ){
            $this->{ 'StartY' } = $iYPosTmp;
        }
    }

    #---- Evaluate block bottom-right
    if ( $this->{ 'EndXDefined' } == $FALSE ){
        if (    $iXPosTmp > $this->{ 'EndX' }
             || $this->{ 'EndX' } == 0
           ){
            $this->{ 'EndX' } = $iXPosTmp;
        }
    }

    $iYPosTmp += $iTextHeightTmp;
    if ( $this->{ 'EndYDefined' } == $FALSE ){
        if (    $iYPosTmp > $this->{ 'EndY' }
             || $this->{ 'EndY' } == 0
           ){
            $this->{ 'EndY' } = $iYPosTmp;
        }
    }
    # $V108 End
}

# V111 Begin
#-----------------------------------------------------------------------
# createAnchor
#-----------------------------------------------------------------------
sub createAnchor{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "createAnchor()" );
    #}

    #---- Get parameter
    #
    # 1. Anchor config hash reference
    #
    my $hrefAnchorPar = shift;

    #---- Create anchor
    my $objAnchorTmp = new a2w::core::dm::Anchor();

    #---- Fill in anchor details
    $objAnchorTmp->setId( $hrefAnchorPar->{ 'Id' } ); # V111 Change

    # $V108 Begin
    #$objAnchorTmp->createAndAddConstraints( $hrefAnchorPar->{ 'Constraints' } );
    #
    my @arrConsTmp = grep { lc( $_ ) =~ /constraints*/ } keys( %{ $hrefAnchorPar } );
    foreach my $cons ( @arrConsTmp ){
        $objAnchorTmp->createAndAddConstraints( $cons, $hrefAnchorPar->{ $cons } );
    }
    #V108 End

    return $objAnchorTmp;
}
# V111 End

#-----------------------------------------------------------------------
# createAndSetStartAnchor
#-----------------------------------------------------------------------
sub createAndSetStartAnchor{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "createAndSetStartAnchor()" );
    #}

    #---- Get parameter
    #
    # 1. Anchor config hash reference
    #
    my $hrefAnchorPar = shift;

    #---- Create start anchor
    my $ancStartTmp = $this->createAnchor( $hrefAnchorPar ); # V111 Change

    $this->setStartAnchor( $ancStartTmp );
}

#-----------------------------------------------------------------------
# createAndSetEndAnchor
#-----------------------------------------------------------------------
sub createAndSetEndAnchor{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "createAndSetEndAnchor()" );
    #}

    #---- Get parameter
    #
    # 1. Anchor config hash reference
    #
    my $hrefAnchorPar = shift;

    #---- Create end anchor
    my $ancEndTmp = $this->createAnchor( $hrefAnchorPar ); # V111 Change

    $this->setEndAnchor( $ancEndTmp );
}

#-----------------------------------------------------------------------
# createAndSetContentDef
#-----------------------------------------------------------------------
sub createAndSetContentDef{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "createAndSetContentDef()" );
    #}

    #---- Get parameter
    #
    # 1. ContentDef config hash reference
    #
    my $hrefContentDefPar = shift;

    #---- Create content definition
    my $objContDefTmp = undef;
    my $sContDefTypeTmp = lc( $hrefContentDefPar->{ 'Type' } );
    if ( $sContDefTypeTmp eq "paragraph" ){
        $objContDefTmp = new a2w::core::dm::ParagraphContentDef();

        #---- Fill in content definition
        if ( lc( $hrefContentDefPar->{ 'Definition' }{ 'Group' } ) eq "true" ){
            $objContDefTmp->setGroup( $TRUE );
        }
        $objContDefTmp->setSeparator( $hrefContentDefPar->{ 'Definition' }{ 'Separator' } );
        # $V115 Begin
        if ( defined( $hrefContentDefPar->{ 'Definition' }{ 'TagLineAsParagraph' } ) ){
            $objContDefTmp->setTagLineAsParagraph( $hrefContentDefPar->{ 'Definition' }{ 'TagLineAsParagraph' } );
        }
        # $V115 End
        # $V102 Begin
        if ( lc( $hrefContentDefPar->{ 'Definition' }{ 'GroupBy' } ) eq "font" ){
            $objContDefTmp->setGroupBy( "font" );
        }
        else {
            if ( $bLog == $TRUE
                 && defined( $hrefContentDefPar->{ 'Definition' }{ 'GroupBy' } ) # V112 Change
                 # Since line is default, it is not asserted on above 'if' condition, so 'line' must be skipped while warning about invalid value
                 && lc( $hrefContentDefPar->{ 'Definition' }{ 'GroupBy' } ) ne "line"
               ){
                $theLogger->logMessage( "Warning! Invalid GroupBy (" . $hrefContentDefPar->{ 'Definition' }{ 'GroupBy' } . "), valid values are line and font" );
            }
        }
        # $V102 End
    }
    elsif ( $sContDefTypeTmp eq "skip" ){
        $objContDefTmp = new a2w::core::dm::SkipContentDef();
    }
    elsif ( $sContDefTypeTmp eq "table" ){
        $objContDefTmp = new a2w::core::dm::TableContentDef();

        #---- Fill in content definition
        unless (    $hrefContentDefPar->{ 'Definition' }{ 'Header' } == undef
                 || lc( $hrefContentDefPar->{ 'Definition' }{ 'Header' } ) eq "false"
               ){
            $objContDefTmp->setHeader( $hrefContentDefPar->{ 'Definition' }{ 'Header' } );
        }
        $objContDefTmp->setColumns( $hrefContentDefPar->{ 'Definition' }{ 'Columns' } );
        $objContDefTmp->setPrimaryColumns( $hrefContentDefPar->{ 'Definition' }{ 'PrimaryColumns' } ); # $V105 Change
        $objContDefTmp->setHeaderLineCount( $hrefContentDefPar->{ 'Definition' }{ 'HeaderLineCount' } ); # $V105 Change
        $objContDefTmp->setRowAdditionalRules( $hrefContentDefPar->{ 'Definition' }{ 'RowAdditionalRules' } ); # $V105 Change
        $objContDefTmp->setCaption( ( $hrefContentDefPar->{ 'Definition' }{ 'Caption' } eq "true" ) ? $TRUE : $FALSE ); # V112 Change
    }
    # $V114 Begin
    elsif ( $sContDefTypeTmp eq "list" ){
        $objContDefTmp = new a2w::core::dm::ListContentDef();

        #---- Fill in content definition
        $objContDefTmp->setOrdered( $hrefContentDefPar->{ 'Definition' }{ 'Ordered' } );
        $objContDefTmp->setBodyPosition( $hrefContentDefPar->{ 'Definition' }{ 'BodyPosition' } );
        $objContDefTmp->setCaption( ( $hrefContentDefPar->{ 'Definition' }{ 'Caption' } eq "true" ) ? $TRUE : $FALSE );
    }
    # $V114 End
    # $V107 Begin
    elsif ( $sContDefTypeTmp eq "illustration" ){
        $objContDefTmp = new a2w::core::dm::IllustrationContentDef();
    }
    # $V107 End
    # $V111 Begin
    elsif ( $sContDefTypeTmp eq "raw" ){
        $objContDefTmp = new a2w::core::dm::RawContentDef();
    }
    # $V111 End
    # $V112 Begin
    elsif ( $sContDefTypeTmp eq "composite" ){
        $objContDefTmp = new a2w::core::dm::CompositeContentDef();
        if (    $hrefContentDefPar->{ 'SubBlocks' } != undef
             && lc( ref( $hrefContentDefPar->{ 'SubBlocks' } ) ) eq "array"
           ){
            $objContDefTmp->setSubBlocks( $hrefContentDefPar->{ 'SubBlocks' } );
        }
    }
    # $V112 End
    else {
        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "Warning! Invalid content definition type (" . $hrefContentDefPar->{ 'Type' } . "), valid values are Paragraph, Table, Skip, Illustration, Raw and Composite" ); # V111 Change
        }
        return;
    }

    #---- Evaluate defaults for content definition
    $objContDefTmp->evaluateDefaults(); # $V105 Change

    #---- Create and add additional contents
    my @arrKeysTmp = sort keys( %{ $hrefContentDefPar->{ 'AddContent' } } );
    foreach my $sIdTmp ( @arrKeysTmp ){
        $objContDefTmp->createAndAddContent(   $sIdTmp
                                             , $hrefContentDefPar->{ 'AddContent' }{ $sIdTmp }
                                           );
    }

    $this->setContentDef( $objContDefTmp );
}

#-----------------------------------------------------------------------
# Create and set content processor
#-----------------------------------------------------------------------
sub createAndSetContentProcessor{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "createAndSetContentProcessor()" );
    #}

    #---- Get parameter
    #
    # 1. ContentProcessor config hash reference
    #
    my $hrefContentProcPar = shift;

    #---- Iterate through content processor and add appropriate objects ----#
    my $contProcessorTmp   = undef;
    my $sNameTmp           = '';
    my @arrContProcKeysTmp = sort keys( %{ $hrefContentProcPar } );
    foreach my $sKeyTmp ( @arrContProcKeysTmp ){
        if ( $hrefContentProcPar->{ $sKeyTmp } == undef ){
            next;
        }

        #---- Identify the content processor type (Find & Replace or Index)
        if ( $sKeyTmp =~ /^FNR\_(.*)/ ){
            $sNameTmp = $1;
            if ( $contProcessorTmp == undef ){
                $contProcessorTmp = new a2w::core::dm::ContentProcessor();
            }
            $contProcessorTmp->addFindAndReplace( $sNameTmp, $hrefContentProcPar->{ $sKeyTmp } );
        }
        elsif ( $sKeyTmp =~ /^IDX\_(.*)/ ){
            $sNameTmp = $1;
            if ( $contProcessorTmp == undef ){
                $contProcessorTmp = new a2w::core::dm::ContentProcessor();
            }
            $contProcessorTmp->addIndex( $sNameTmp, $hrefContentProcPar->{ $sKeyTmp } );
        }
        else {
            if ( $bLog == $TRUE ){
                $theLogger->logMessage( "Warning! Ignored unknown content processor object type (" . $sKeyTmp . ")" );
            }
        }
    }

    $this->setContentProcessor( $contProcessorTmp );
}

#-----------------------------------------------------------------------
# Add start anchor
#
# Adds given object to start anchor list
#
#-----------------------------------------------------------------------
sub addStartAnchor{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "addStartAnchor()" );
    #}

    #---- Get parameter
    #
    # 1. Object
    # 2. POM object wrapper
    #
    my $a2wObjectPar = shift;
    my $pomObjectPar = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logMessage(   "Object added to block (" . $this->getId() . ") start anchor list" );
    }

    #---- Evaluate existing object count
    my $iObjCountTmp = @{ $this->{ 'StAncList' } };

    #---- Add object at count index
    $this->{ 'StAncList' }[ $iObjCountTmp ] = {
          'A2WOBJ' => $a2wObjectPar
        , 'POMOBJ' => $pomObjectPar
    };

    #---- Collect index ----#
    if ( $this->{ 'StartAnchor' } != undef && $this->{ 'StartAnchor' }->isSkipped() == $FALSE ){
        #---- Get content processor
        my $contProcTmp = $this->getContentProcessor();

        if ( $contProcTmp != undef && $contProcTmp->doesAllIndexesCollected() == $FALSE ){
            #---- Collect index
            $contProcTmp->collectIndex( $a2wObjectPar, $pomObjectPar );
        }
    }
}

#-----------------------------------------------------------------------
# Add object
#
# Adds given object to list
#
#-----------------------------------------------------------------------
sub addObject{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "addObject()" );
    #}

    #---- Get parameter
    #
    # 1. Object
    # 2. POM object wrapper
    #
    my $a2wObjectPar = shift;
    my $pomObjectPar = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logMessage(   "Object added to block (" . $this->getId() . ")" );
    }

    #---- Evaluate existing object count
    my $iObjCountTmp = @{ $this->{ 'ObjsList' } };

    #---- Add object at count index
    $this->{ 'ObjsList' }[ $iObjCountTmp ] = {
          'A2WOBJ' => $a2wObjectPar
        , 'POMOBJ' => $pomObjectPar
    };

    #---- Collect index ----#
    #---- Get content processor
    my $contProcTmp = $this->getContentProcessor();

    if ( $contProcTmp != undef && $contProcTmp->doesAllIndexesCollected() == $FALSE ){
        #---- Collect index
        $contProcTmp->collectIndex( $a2wObjectPar, $pomObjectPar );
    }
}

#-----------------------------------------------------------------------
# Add end anchor
#
# Adds given object to end anchor list
#
#-----------------------------------------------------------------------
sub addEndAnchor{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "addEndAnchor()" );
    #}

    #---- Get parameter
    #
    # 1. Object
    # 2. POM object wrapper
    #
    my $a2wObjectPar = shift;
    my $pomObjectPar = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logMessage(   "Object added to block (" . $this->getId() . ") end anchor list" );
    }

    #---- Evaluate existing object count
    my $iObjCountTmp = @{ $this->{ 'EdAncList' } };

    #---- Add object at count index
    $this->{ 'EdAncList' }[ $iObjCountTmp ] = {
          'A2WOBJ' => $a2wObjectPar
        , 'POMOBJ' => $pomObjectPar
    };

    #---- Collect index ----#
    if ( $this->{ 'EndAnchor' } != undef && $this->{ 'EndAnchor' }->isSkipped() == $FALSE ){
        #---- Get content processor
        my $contProcTmp = $this->getContentProcessor();

        if ( $contProcTmp != undef && $contProcTmp->doesAllIndexesCollected() == $FALSE ){
            #---- Collect index
            $contProcTmp->collectIndex( $a2wObjectPar, $pomObjectPar );
        }
    }
}

#-----------------------------------------------------------------------
# doesStartedOn
#
# Returns true if block started on page (having given id)
#
#-----------------------------------------------------------------------
sub doesStartedOn{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "doesStartedOn()" );
    #}

    #---- Get parameter
    #
    # 1. Page id
    #
    my $iPageIdPar = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logMessage(   "Block " . $this->getId() . " started on " . $this->{ 'StartedOn' } . ", checking against " . $iPageIdPar );
    #}

    return ( $this->{ 'StartedOn' } == $iPageIdPar ) ? $TRUE : $FALSE;
}

#-----------------------------------------------------------------------
# getContentDefType
#
# Returns type of content definition
#
#-----------------------------------------------------------------------
sub getContentDefType{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getContentDefType()" );
    #}

    #---- Get content definition
    my $contDefTmp = $this->getContentDef();
    my $sRetTmp = "";
    if ( $contDefTmp != undef ){
        $sRetTmp = $contDefTmp->getType();
    }

    return $sRetTmp;
}

#-----------------------------------------------------------------------
# isRepetitive
#
# Returns true if block is repeatable
#
#-----------------------------------------------------------------------
sub isRepetitive{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "isRepetitive()" );
    #}

    my $bRetTmp = $FALSE;
    if ( $this->{ 'Quantifier' } ne "1" ){
        $bRetTmp = $TRUE;
    }

    return $bRetTmp;
}

#-----------------------------------------------------------------------
# hasMoreRepeatation
#
# Returns true if block configured to repeat more
#
#-----------------------------------------------------------------------
sub hasMoreRepeatation{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "hasMoreRepeatation()" );
    #}

    my $bRetTmp = $FALSE;
    my $sQuantifierTmp = $this->{ 'Quantifier' };

    # 1 - Block occurs only once (default)
    # n - Block occurs exactly 'n' count
    if ( $sQuantifierTmp =~ /^\d+$/ ){
        if ( $this->{ 'RepeatCount' } < $sQuantifierTmp ){
            $bRetTmp = $TRUE;
        }
    }
    # * - Block occurs zero or many
    # + - Block occurs minimum once and many (!!!???)
    elsif (    $sQuantifierTmp eq '*'
            || $sQuantifierTmp eq '+'
          ){
        $bRetTmp = $TRUE;
    }
    # ? - Block occurs zero or once
    elsif ( $sQuantifierTmp eq '?' ){
        # Already current instance exists, so nothing more to repeat
    }

    return $bRetTmp;
}

#-----------------------------------------------------------------------
# isChained
#
# Returns true if block is chained with next block
#
#-----------------------------------------------------------------------
sub isChained{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "isChained()" );
    #}

    return ( $this->{ 'Next' } ne "" ) ? $TRUE : $FALSE;
}

#-----------------------------------------------------------------------
# isFilled
#
# Asserts whether whole block content is collected or not
#
# Returns
# 1 in case of all content of block were collected
# 0 in case of none of block content is collected or still collecting block content
#
#-----------------------------------------------------------------------
sub isFilled{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "isFilled()" );
    #}

    # V112 Begin
    # Consider a block is filled only when the chained block is also filled
    my $bFilledTmp = (    ( $this->{ 'StartFound' } == $TRUE )
                       && ( $this->{ 'EndFound'   } == $TRUE )
                     );
    if ( $bFilledTmp == $TRUE ){
        #---- Get chained block
        my $chainedBlockTmp = $this->getNextRef();
        if ( $chainedBlockTmp != undef ){ $bFilledTmp &&= $chainedBlockTmp->isFilled(); }
    }

    return $bFilledTmp;
    # V112 End
}

#-----------------------------------------------------------------------
# setAsFilled
#-----------------------------------------------------------------------
sub setAsFilled{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setAsFilled()" );
    #}

    $this->{ 'StartFound' } = $TRUE;
    $this->{ 'EndFound'   } = $TRUE;

    # $V104 Begin
    #---- Evaluate collected content height
    $this->{ 'ContentHeight' } += $this->{ 'EndY' } - $this->{ 'StartY' };
    # $V104 End
}

#-----------------------------------------------------------------------
# hasToBeSkipped
#
# Returns whether whole block content has to be skipped from presentation
#
# Returns
# 1 in case of block has to be skipped from presentation
# 0 in case of block has to be presented
#
#-----------------------------------------------------------------------
sub hasToBeSkipped{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "hasToBeSkipped()" );
    #}

    my $bRetTmp = $FALSE;

    #---- Get content def
    my $contDefTmp = $this->getContentDef();
    if (    $contDefTmp != undef
         && $contDefTmp->getType() eq "skip"
       ){
        $bRetTmp = $TRUE;
    }
    return $bRetTmp;
}

#-----------------------------------------------------------------------
# Reset block for next document processing
#
# Returns
# >=0 in case of success
#  <0 in case of error
#
#-----------------------------------------------------------------------
sub reset{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "reset()" );
    }

    # $V108 Begin
    #---- Get block definition reference
    my $blkDefRefTmp = $this->{ 'ConfigReference' };

    #---- Reset all attributes
    $this->{ 'Id' }               = '';
    $this->{ 'StartAnchor' }      = undef;
    $this->{ 'EndAnchor' }        = undef;
    $this->{ 'Width' }            = 0;
    $this->{ 'Height' }           = 0;
    $this->{ 'ContentDef' }       = undef;
    $this->{ 'Next' }             = undef;
    $this->{ 'Quantifier' }       = '1';
    $this->{ 'ContentProcessor' } = undef;
    $this->{ 'Formatter' }        = undef;
    $this->{ 'NextRef' }          = undef;
    $this->{ 'ObjsList' }         = [];
    $this->{ 'StAncList' }        = [];
    $this->{ 'EdAncList' }        = [];
    $this->{ 'StartX' }           = 0;
    $this->{ 'StartY' }           = 0;
    $this->{ 'BeginY' }           = 0;
    $this->{ 'EndX' }             = 0;
    $this->{ 'EndY' }             = 0;
    $this->{ 'StartFound' }       = $FALSE;
    $this->{ 'EndFound' }         = $FALSE;
    $this->{ 'StartedOn' }        = 0;
    $this->{ 'Flushed' }          = $FALSE;
    $this->{ 'ConfigReference' }  = undef;
    $this->{ 'RepeatCount' }      = 1;
    $this->{ 'Info' }             = '';
    $this->{ 'ContentHeight' }    = 0;
    $this->{ 'JSONInfo' }         = undef;
    $this->{ 'Region' }           = $FALSE;
    $this->{ 'StartXDefined' }    = $FALSE;
    $this->{ 'StartYDefined' }    = $FALSE;
    $this->{ 'EndXDefined' }      = $FALSE;
    $this->{ 'EndYDefined' }      = $FALSE;
    $this->{ 'WidthDefined' }     = $FALSE;
    $this->{ 'HeightDefined' }    = $FALSE;
    $this->{ 'SubBlock' }         = $FALSE; # V112 Change

    #---- Fill in again from block definition
    $this->fillIn( $blkDefRefTmp );
    # $V108 End

    return 0;
}

#-----------------------------------------------------------------------
# Reset starting anchor (used when block content runs across pages)
#
# Returns
# >=0 in case of success
#  <0 in case of error
#
#-----------------------------------------------------------------------
sub resetStartAnchor{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "resetStartAnchor()" );
    }

    #---- Nothing to reset if start anchor is not defined in block definition (region case)
    if ( $this->{ 'StartAnchor' } == undef ){ return; } # $V108 Change

    # $V104 Begin
    #---- Evaluate collected content height
    $this->{ 'ContentHeight' } += $this->{ 'EndY' } - $this->{ 'StartY' };
    # $V104 End

    $this->{ 'StartFound' } = $FALSE;
    $this->{ 'StartX' }     = 0;
    $this->{ 'StartY' }     = 0;
    $this->{ 'EndX' }       = 0;

    return 0;
}

#-----------------------------------------------------------------------
# fillIn
#
# Fill in block with configured values
#
# Returns
# >=0 in case of success
# < 0 in case of error
#
#-----------------------------------------------------------------------
sub fillIn{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "fillIn()" );
    #}

    #---- Get parameter
    #
    # 1. Block config reference
    #
    my $blkConfigRefPar = shift;

    #---- Fill in configured values in block
    $this->setConfigReference( $blkConfigRefPar );
    $this->setId( $blkConfigRefPar->{ 'Id' } );
    if ( defined $blkConfigRefPar->{ 'StartAnchor' } ){ $this->createAndSetStartAnchor( $blkConfigRefPar->{ 'StartAnchor' } ); }
    if ( defined $blkConfigRefPar->{ 'EndAnchor' } ){ $this->createAndSetEndAnchor( $blkConfigRefPar->{ 'EndAnchor' } ); }

    # $V108 Begin
    $this->{ 'StartXDefined' } = $FALSE;
    $this->{ 'StartYDefined' } = $FALSE;
    $this->{ 'EndXDefined' }   = $FALSE;
    $this->{ 'EndYDefined' }   = $FALSE;
    $this->{ 'WidthDefined' }  = $FALSE;
    $this->{ 'HeightDefined' } = $FALSE;

    if ( defined $blkConfigRefPar->{ 'Width' } ){ $this->setWidth( $blkConfigRefPar->{ 'Width' } ); $this->{ 'WidthDefined' } = $TRUE; }
    if ( defined $blkConfigRefPar->{ 'Height' } ){ $this->setHeight( $blkConfigRefPar->{ 'Height' } ); $this->{ 'HeightDefined' } = $TRUE; }
    if ( defined $blkConfigRefPar->{ 'StartX' } ){ $this->setStartX( $blkConfigRefPar->{ 'StartX' } ); $this->{ 'StartXDefined' } = $TRUE; }
    if ( defined $blkConfigRefPar->{ 'StartY' } ){ $this->setStartY( $blkConfigRefPar->{ 'StartY' } ); $this->{ 'StartYDefined' } = $TRUE; }
    if ( defined $blkConfigRefPar->{ 'EndX' } ){ $this->setEndX( $blkConfigRefPar->{ 'EndX' } ); $this->{ 'EndXDefined' } = $TRUE; }
    if ( defined $blkConfigRefPar->{ 'EndY' } ){ $this->setEndY( $blkConfigRefPar->{ 'EndY' } ); $this->{ 'EndYDefined' } = $TRUE; }
    if ( defined $blkConfigRefPar->{ 'Region' } ){
        $this->setRegion( $blkConfigRefPar->{ 'Region' } );
    }
    else {
        #---- Evaluate region value based on given block attributes
        if (    ( $this->{ 'StartXDefined' } == $TRUE && $this->{ 'StartYDefined' } == $TRUE && $this->{ 'EndXDefined' } == $TRUE && $this->{ 'EndYDefined' } == $TRUE )
             || ( $this->{ 'StartXDefined' } == $TRUE && $this->{ 'StartYDefined' } == $TRUE && $this->{ 'WidthDefined' } == $TRUE && $this->{ 'HeightDefined' } == $TRUE )
           ){
            $this->setRegion( $TRUE );
        }
    }

    #---- Evaluate and set endx, endy (if not specified, but width/height specified along with startx, starty)
    if (    ( $this->{ 'StartXDefined' } == $TRUE && $this->{ 'StartYDefined' } == $TRUE )
         && ( $this->{ 'WidthDefined' } == $TRUE && $this->{ 'HeightDefined' } == $TRUE )
       ){
        if ( $this->{ 'EndXDefined' } == $FALSE ){ $this->setEndX( $blkConfigRefPar->{ 'StartX' } + $blkConfigRefPar->{ 'Width' } ); }
        if ( $this->{ 'EndYDefined' } == $FALSE ){ $this->setEndY( $blkConfigRefPar->{ 'StartY' } + $blkConfigRefPar->{ 'Height' } ); }
    }

    #---- Evaluate and set width, height (if not specified, but endx/endy specified along with startx, starty)
    if (    ( $this->{ 'StartXDefined' } == $TRUE && $this->{ 'StartYDefined' } == $TRUE )
         && ( $this->{ 'EndXDefined' } == $TRUE && $this->{ 'EndYDefined' } == $TRUE )
       ){
        if ( $this->{ 'WidthDefined' } == $FALSE ){ $this->setWidth( $blkConfigRefPar->{ 'StartX' } - $blkConfigRefPar->{ 'EndX' } ); }
        if ( $this->{ 'HeightDefined' } == $FALSE ){ $this->setHeight( $blkConfigRefPar->{ 'StartY' } - $blkConfigRefPar->{ 'EndY' } ); }
    }
    # $V108 End

    #---- Create and set content definition
    if ( $blkConfigRefPar->{ 'ContentDef' } == undef ){
        #---- Create default content definition of type paragraph
        $this->createAndSetContentDef( {   'Type' => 'Paragraph'
                                         , 'Definition' => {
                                                 'Group' => $FALSE
                                               , 'GroupBy' => "line"
                                               , 'Separator' => ""
                                           }
                                       }
                                     );
    }
    else {
        $this->createAndSetContentDef( $blkConfigRefPar->{ 'ContentDef' } );
    }
    # $V103 Begin
    if ( defined $blkConfigRefPar->{ 'Info' } ){
        $this->setInfo( $blkConfigRefPar->{ 'Info' } );
    }
    # $V103 End
    $this->setNext( $blkConfigRefPar->{ 'Next' } );
    if ( defined $blkConfigRefPar->{ 'Quantifier' } ){
        $this->setQuantifier( $blkConfigRefPar->{ 'Quantifier' } );
    }
    if ( defined $blkConfigRefPar->{ 'ContentProcessor' } ){
        $this->createAndSetContentProcessor( $blkConfigRefPar->{ 'ContentProcessor' } );
    }

    # V112 Begin
    #---- Assert and set objects list
    if ( defined $blkConfigRefPar->{ 'Objects' } ){
        $this->setObjsList( $blkConfigRefPar->{ 'Objects' } );
    }
    # V112 End

    return 0;
}

#-----------------------------------------------------------------------
# createRepetitiveBlock
#
# Creates repetitive block and fills in with configured values
#
# Returns
# Repetitive Block instance in case of success
# undefined in case of error
#
#-----------------------------------------------------------------------
sub createRepetitiveBlock{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "createRepetitiveBlock()" );
    #}

    #---- Create block
    my $blkRepetitiveTmp = new a2w::core::dm::Block();

    #---- Fill in block
    $blkRepetitiveTmp->fillIn( $this->getConfigReference() );
    $blkRepetitiveTmp->setRepeatCount( $this->getRepeatCount() + 1 );

    if ( $bLog == $TRUE ){
        $theLogger->logMessage( "Created repetitive block (" . $this->getId() . "-" . $blkRepetitiveTmp->getRepeatCount() . ")" );
    }

    return $blkRepetitiveTmp;
}

#-----------------------------------------------------------------------
# update
#
# Updates block
# - with actual id, if id has place holder from index value
#
# Returns
# >=0 in case of success
# < 0 in case of error
#
#-----------------------------------------------------------------------
sub update{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "update()" );
    #}

    #---- Assert whether block has id with place holder
    my $sIdTmp = $this->{ 'Id' };
    my $iSepIdxTmp = index( $sIdTmp, '' );
    if ( $iSepIdxTmp > 0 ){
        $sIdTmp = substr( $sIdTmp, $iSepIdxTmp );
    }
    my @arrIdxTmp = split( //, $sIdTmp );
    @arrIdxTmp = grep { $_ ne '' } @arrIdxTmp;
    foreach $sIdxTmp ( @arrIdxTmp ){
        #---- Get content processor
        my $sIdxValueTmp = "";
        my $contProcTmp = $this->getContentProcessor();
        if ( $contProcTmp != undef ){
		    #---- Get index with place holder name
		    my $idxPlaceHolderTmp = $contProcTmp->getIndex( $sIdxTmp );
		    if ( $idxPlaceHolderTmp != undef ){
		        $sIdxValueTmp = $idxPlaceHolderTmp->getResult();
		    }
        }

        #---- Update id with actual value
        my $sIdTmp = $this->{ 'Id' };
        $sIdTmp =~ s/(\Q$sIdxTmp\E)/$sIdxValueTmp/;
        $this->setId( $sIdTmp );
    }
    return 0;
}

#-----------------------------------------------------------------------
# write
#
# Write current block using formatter
#
#-----------------------------------------------------------------------
sub write{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "write(" . $this->getId() . ")" );
    }

    #---- Get parameter
    #
    # 1. Visitor
    #
    my $outVisitorPar = shift;

    #---- Set flushed flag
    $this->{ 'Flushed' } = $TRUE;

    #---- Write begin block
    my $iRetTmp = 0;
    my $bSkipWrapperTmp = $FALSE;
    my $contDefTmp = $this->getContentDef();
    my $hrefACTmp = $contDefTmp->getAddContent();
    my @arrACKeysTmp = sort keys( %{ $hrefACTmp } );
    if (    $contDefTmp->getType() eq "skip"
         && @arrACKeysTmp <= 0
       ){
        $bSkipWrapperTmp = $TRUE;
    }
    $iRetTmp = $outVisitorPar->beginBlock( $this, $bSkipWrapperTmp );
    if ( $iRetTmp < 0 ){ return $iRetTmp };

    #---- Write additional content to be added before block content
    foreach my $sIdTmp ( @arrACKeysTmp ){
        if ( lc( $hrefACTmp->{ $sIdTmp }->getInclude() ) eq "before" ){
            $outVisitorPar->writeRawContent( $hrefACTmp->{ $sIdTmp }->getContent() );
        }
    }

    #---- Write block content
    $iRetTmp = $this->writeContent( $outVisitorPar );
    if ( $iRetTmp < 0 ){ return $iRetTmp };

    #---- Write additional content to be added after block content
    foreach my $sIdTmp ( @arrACKeysTmp ){
        if ( lc( $hrefACTmp->{ $sIdTmp }->getInclude() ) eq "after" ){
            $outVisitorPar->writeRawContent( $hrefACTmp->{ $sIdTmp }->getContent() );
        }
    }

    #---- Write end block
    $iRetTmp = $outVisitorPar->endBlock( $this, $bSkipWrapperTmp );

    return $iRetTmp;
}

#-----------------------------------------------------------------------
# write block content
#
# Write block content using formatter
#
#-----------------------------------------------------------------------
sub writeContent{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeContent()" );
    }

    #---- Get parameter
    #
    # 1. Visitor
    #
    my $outVisitorPar = shift;

    # $V101 Begin
    #if ( $bLog == $TRUE && ( $theLogger->getLevel() & $a2w::core::log::Logger::LEVEL_DEBUG ) != 0 ){
    if ( $bLog == $TRUE ){
        #---- Collect block contents ----#
        my @arrObjsListTmp = @{ $this->getObjects() }; # V112 Change

        # Dump block objects
        foreach my $objdmp ( @arrObjsListTmp ){
            if ( $objdmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJTYPE } == $a2w::core::dm::Constants::OT_LINE ){
                $theLogger->logMessage(   "Line@"
                                        . "(" . $objdmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_XPOS }
                                        . "," . $objdmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_YPOS }
                                        . ") W=" . $objdmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_LINE_WIDTH }
                                        . " L=" . $objdmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_LINE_LENGTH }
                                      );

            }
            elsif ( $objdmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJTYPE } == $a2w::core::dm::Constants::OT_TEXT ){
                $theLogger->logMessage(   "Text@"
                                        . "(" . $objdmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_XPOS }
                                        . "," . $objdmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_YPOS }
                                        . ")>" . $objdmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_VALUE }
                                        . "<"
                                      );
            }
            elsif ( $objdmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJTYPE } == $a2w::core::dm::Constants::OT_IMAGE ){
                $theLogger->logMessage(   "Image@"
                                        . "(" . $objdmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_XPOS }
                                        . "," . $objdmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_YPOS }
                                        . ")>" . $objdmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_IMAGE_NAME }
                                        . "<"
                                      );
            }
            # $V107 Begin
            elsif ( $objdmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJTYPE } == $a2w::core::dm::Constants::OT_CONTAINER ){
                $theLogger->logMessage(   "Container@"
                                        . "(" . $objdmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_XPOS }
                                        . "," . $objdmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_YPOS }
                                        . ")>" . $objdmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_CONTAINER_NAME }
                                        . "<"
                                      );
            }
            # $V107 End
            # $V111 Begin
            elsif ( $objdmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJTYPE } == $a2w::core::dm::Constants::OT_VECTOR ){
                $theLogger->logMessage(   "Vector@"
                                        . "(" . $objdmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_XPOS }
                                        . "," . $objdmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_YPOS }
                                        . ") W=" . $objdmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_VECTOR_WIDTH }
                                        . " H=" . $objdmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_VECTOR_HEIGHT }
                                      );
            }
            # $V111 End
        }
    }
    # $V101 End

    #---- Check and create formatter
    $this->_createContentFormatter();
    my $fmtBlockTmp = $this->{ 'Formatter' };

    #---- Write block using formatter
    return $fmtBlockTmp->writeFormattedBlock( $this, $outVisitorPar );
}

#-----------------------------------------------------------------------
# Process content
#
# Process block content and do the following
# - Fetch indexes defined as specified content processor
# - Find and replace contents as specified content processor
# - Update block identifier with actual by replacing place holder
#
# Returns
# >=0 in case of success
# < 0 in case of error
#
#-----------------------------------------------------------------------
sub processContent{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "processContent()" );
    }

    #---- Get content processor ----#
    my $contProcTmp = $this->getContentProcessor();
    if (    $contProcTmp == undef
         || (    $contProcTmp->hasIndexes() == $FALSE
              && $contProcTmp->hasFNRs() == $FALSE
            )
       ){
        #---- No content processor, just return
        return 0;
    }

    #---- Get block objects ----#
    my @arrObjsListTmp = @{ $this->getObjects() }; # V112 Change
    if ( @arrObjsListTmp <= 0 ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Warning! No object found on block to process" ); }
        return -1;
    }

    #---- Iterate through the content and process them ----#
    my @arrIDXTmp = @{ $contProcTmp->getIndexList() };
    my @arrFNRTmp = @{ $contProcTmp->getFindAndReplaceList() };
    foreach my $objTmp ( @arrObjsListTmp ){
        #---- Process and fetch indexes ----#
        #---- Process find and replace ----#
    }

    return 0;
}

#-----------------------------------------------------------------------
# _createContentFormatter
#
# Creates content formatter
#
#-----------------------------------------------------------------------
sub _createContentFormatter{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "_createContentFormatter()" );
    }

    #---- Assert whether formatter is alreay created
    unless ( $this->{ 'Formatter' } eq undef ){
        return 0;
    }

    #---- Create formatter
    my $contDefTmp = $this->getContentDef();
    my $sContDefTypeTmp = lc( $contDefTmp->getType() ); # $V112 Change
    if ( $sContDefTypeTmp eq "paragraph" ){
        $this->{ 'Formatter' } = new a2w::core::dm::ParagraphBlockFormatter();
    }
    elsif ( $sContDefTypeTmp eq "skip" ){
        $this->{ 'Formatter' } = new a2w::core::dm::SkipBlockFormatter();
    }
    elsif ( $sContDefTypeTmp eq "table" ){
        $this->{ 'Formatter' } = new a2w::core::dm::TableBlockFormatter();
    }
    # $V114 Begin
    elsif ( $sContDefTypeTmp eq "list" ){
        $this->{ 'Formatter' } = new a2w::core::dm::ListBlockFormatter();
    }
    # $V114 End
    # $V107 Begin
    elsif ( $sContDefTypeTmp eq "illustration" ){
        $this->{ 'Formatter' } = new a2w::core::dm::IllustrationBlockFormatter();
    }
    # $V107 End
    # $V111 Begin
    elsif ( $sContDefTypeTmp eq "raw" ){
        $this->{ 'Formatter' } = new a2w::core::dm::RawBlockFormatter();
    }
    # $V111 End
    
    elsif ( $sContDefTypeTmp eq "composite" ){
        $this->{ 'Formatter' } = new a2w::core::dm::CompositeBlockFormatter();
    }
    # $V112 End
    else {
        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "  Warning! Unknown content type (" . $contDefTmp->getType() . ") to format the block " . $this->getId() );
        }
        return -1;
    }

    return 0;
}

#-----------------------------------------------------------------------
# isStartAnchor
#
# Asserts whether object is start anchor or not
#
# 1, if object is start anchor of a block
# 0, if object is not start anchor of a block
#
#-----------------------------------------------------------------------
sub isStartAnchor{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "isStartAnchor()" );
    #}

    #---- Parameter
    #
    # 1. POM page
    # 2. object
    # 3. POM object wrapper (optional)
    #
    my $pomPagePar   = shift;
    my $a2wObjectPar = shift;
    my $pomObjectPar = undef;
    if ( @_ > 0 ){
        $pomObjectPar = shift;
    }
    if (    $this->{ 'StartFound' } == $TRUE
         || $a2wObjectPar == undef
         || $this->{ 'Region' } == $TRUE # $V108 Change
       ){
        return $FALSE;
    }

    #---- Fetch starting anchor
    my $ancStartTmp = $this->getStartAnchor();
    if ( $ancStartTmp == undef ){ return $FALSE; } # $V108 Change

    #---- Assert whether object match start anchor
    my $bStartAnchorTmp = $ancStartTmp->doesConstraintsMatch( $a2wObjectPar, $pomObjectPar );
    if ( $bStartAnchorTmp == $TRUE ){
        if ( $this->{ 'StartXDefined' } == $FALSE ){ $this->{ 'StartX' } = $pomObjectPar->{ $a2w::core::dm::Constants::AT_XPOS }; } # $V108 Change
        if ( $this->{ 'StartYDefined' } == $FALSE ){ $this->{ 'StartY' } = $pomObjectPar->{ $a2w::core::dm::Constants::AT_YPOS }; } # $V108 Change
        # $V109 Begin
        # If width not defined, use (page width - startX) as width of block
        if ( $this->{ 'WidthDefined' } == $FALSE ){
            $this->{ 'WidthDefined' } = $TRUE;
            $this->{ 'Width' } = $pomPagePar->getPageWidth() - $this->{ 'StartX' };
        }
        # $V109 End

        if ( $a2wObjectPar->_getType() eq "text" ){
            my $iTextHeightTmp = $pomObjectPar->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_FONTHT };
            $iTextHeightTmp *= ( $pomPagePar->getPageRes() / 72 );
            $iTextHeightTmp = int( $iTextHeightTmp + .5 * ( $iTextHeightTmp <=> 0 ) );
            $this->{ 'StartY' } -= $iTextHeightTmp;
        }

        if ( $this->{ 'BeginY' } <= 0 ){ $this->{ 'BeginY' } = $this->{ 'StartY' }; }
        $this->{ 'EndX' } = $this->{ 'StartX' } + $this->{ 'Width' };
        $this->{ 'StartFound' } = $TRUE;

        # V112 Begin
        #---- Evaluate end Y based on detected start Y and configured block height
        if ( $this->{ 'HeightDefined' } == $TRUE ){ $this->{ 'EndY' } = $this->{ 'StartY' } + $this->{ 'Height' }; }

        #---- Evaluate whether block can be considered as region (region = start anchor + width + height)
        if ( $this->{ 'WidthDefined' } == $TRUE && $this->{ 'HeightDefined' } == $TRUE ){ $this->setRegion( $TRUE ); }
        # V112 End

        if ( $bLog == $TRUE ){
            if ( $this->isRepetitive() == $TRUE ){
                $theLogger->logMessage( "Object is starting anchor of block " . $this->getId() . "(" . $this->{ 'RepeatCount' } . ")" );
            }
            else {
                $theLogger->logMessage( "Object is starting anchor of block " . $this->getId() );
            }
        }
    }

    return $bStartAnchorTmp;
}

#-----------------------------------------------------------------------
# isEndAnchor
#
# Asserts whether object is end anchor or not
#
# 1, if object is end anchor of a block
# 0, if object is not end anchor of a block
#
#-----------------------------------------------------------------------
sub isEndAnchor{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "isEndAnchor()" );
    #}

    #---- Parameter
    #
    # 1. POM page
    # 2. object
    # 3. POM object wrapper (optional)
    #
    my $pomPagePar   = shift;
    my $a2wObjectPar = shift;
    my $pomObjectPar = undef;
    if ( @_ > 0 ){
        $pomObjectPar = shift;
    }
    if (    $this->isFilled() == $TRUE
         || $this->{ 'StartFound' } == $FALSE
         || $a2wObjectPar == undef
         || $this->{ 'Region' } == $TRUE # $V108 Change
       ){
        return $FALSE;
    }

    #---- Fetch ending anchor
    my $ancEndTmp = $this->getEndAnchor();
    if ( $ancEndTmp == undef ){ return $FALSE; } # $V108 Change

    #---- Assert whether object match end anchor
    my $bEndAnchorTmp = $ancEndTmp->doesConstraintsMatch( $a2wObjectPar, $pomObjectPar );
    if ( $bEndAnchorTmp == $TRUE ){
        $this->{ 'EndFound' } = $TRUE;
        if ( $bLog == $TRUE ){
            if ( $this->isRepetitive() == $TRUE ){
                $theLogger->logMessage( "Object is ending anchor of block " . $this->getId() . "(" . $this->{ 'RepeatCount' } . ")" );
            }
            else {
                $theLogger->logMessage( "Object is ending anchor of block " . $this->getId() );
            }
        }

        # V111 Begin
        if ( $pomObjectPar->{ $a2w::core::dm::Constants::AT_YPOS } > $this->{ 'EndY' } ){
            $this->{ 'EndY' } = $pomObjectPar->{ $a2w::core::dm::Constants::AT_YPOS };
        }
        # V111 End

        # $V104 Begin
        #---- Evaluate collected content height
        $this->{ 'ContentHeight' } += $this->{ 'EndY' } - $this->{ 'StartY' };
        # $V104 End
    }

    return $bEndAnchorTmp;
}

# $V101 Begin
#-----------------------------------------------------------------------
# doesObjectFallIn
#
# Asserts whether object fall in given block and returns following value
#
# 1, if object fall in given block defined region
# 0, if object does not fall in given block defined region
#
#-----------------------------------------------------------------------
sub doesObjectFallIn{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "doesObjectFallIn()" );
    #}

    #---- Parameter
    #
    # 1. POM page
    # 2. object
    # 3. POM object wrapper (optional)
    #
    my $pomPagePar   = shift;
    my $a2wObjectPar = shift;
    my $pomObjectPar = undef;
    if ( @_ > 0 ){
        $pomObjectPar = shift;
    }
    if ( $a2wObjectPar == undef ){ return $FALSE; }

    #---- Assert whether block starting found
    if (    $this->{ 'Region' } == $FALSE         # $V108 Change
         && (    $this->{ 'StartFound' } == $FALSE
              || $this->{ 'EndFound' } == $TRUE
            )
       ){
        return $FALSE;
    }
    my $bObjFallInTmp = $FALSE;

    #---- Fetch object position
    # V112 Begin
    #---- Get position from pom instead of core object
    my $iXPosTmp = $pomObjectPar->{ $a2w::core::dm::Constants::AT_XPOS };
    my $iYPosTmp = $pomObjectPar->{ $a2w::core::dm::Constants::AT_YPOS };
    # V112 End

    #---- Assert whether object fall beyond start anchor
    # $V108 Begin
    if ( $this->{ 'Region' } == $TRUE ){
        if (    ( $iXPosTmp >= $this->{ 'StartX' } && $iYPosTmp >= $this->{ 'StartY' } )
             && ( $iXPosTmp <= $this->{ 'EndX' }   && $iYPosTmp <= $this->{ 'EndY' } )
           ){
            $bObjFallInTmp = $TRUE;
            if ( $bLog == $TRUE ){ $theLogger->logMessage( "Object fall in block " . $this->getId() ); }
        }
    }
    else {
        if (    $iXPosTmp >= $this->{ 'StartX' }
             && $iYPosTmp >= $this->{ 'StartY' }
             && $iXPosTmp <= $this->{ 'EndX' }
           ){
            $bObjFallInTmp = $TRUE;
            $this->{ 'EndY' } = $iYPosTmp;
            if ( $bLog == $TRUE ){ $theLogger->logMessage( "Object fall in block " . $this->getId() ); }
        }
    }
    # $V108 End

    return $bObjFallInTmp;
}
# $V101 End

# $V104 Begin
#-----------------------------------------------------------------------
# isParagraph
#
# Asserts whether block type is paragraph or not
#
# 1, if block is paragraph
# 0, if block is NOT paragraph
#
#-----------------------------------------------------------------------
sub isParagraph{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "isParagraph()" );
    #}

    #---- Check type
    if ( $this->{ 'ContentDef' } == undef ){ return $FALSE; }

    my $bRetTmp = $FALSE;
    my $sContDefTypeTmp = lc( $this->{ 'ContentDef' }->getType() );
    if ( $sContDefTypeTmp eq "paragraph" ){
        $bRetTmp = $TRUE;
    }

    return $bRetTmp;
}

#-----------------------------------------------------------------------
# isTable
#
# Asserts whether block type is table or not
#
# 1, if block is table
# 0, if block is NOT table
#
#-----------------------------------------------------------------------
sub isTable{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "isTable()" );
    #}

    #---- Check type
    if ( $this->{ 'ContentDef' } == undef ){ return $FALSE; }

    my $bRetTmp = $FALSE;
    my $sContDefTypeTmp = lc( $this->{ 'ContentDef' }->getType() );
    if ( $sContDefTypeTmp eq "table" ){
        $bRetTmp = $TRUE;
    }

    return $bRetTmp;
}

#-----------------------------------------------------------------------
# isSkip
#
# Asserts whether block type is skip or not
#
# 1, if block is skip
# 0, if block is NOT skip
#
#-----------------------------------------------------------------------
sub isSkip{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "isSkip()" );
    #}

    #---- Check type
    if ( $this->{ 'ContentDef' } == undef ){ return $FALSE; }

    my $bRetTmp = $FALSE;
    my $sContDefTypeTmp = lc( $this->{ 'ContentDef' }->getType() );
    if ( $sContDefTypeTmp eq "skip" ){
        $bRetTmp = $TRUE;
    }

    return $bRetTmp;
}
# $V104 End

# $V114 Begin
#-----------------------------------------------------------------------
# isList
#
# Asserts whether block type is list or not
#
# 1, if block is list
# 0, if block is NOT list
#
#-----------------------------------------------------------------------
sub isList{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "isList()" );
    #}

    #---- Check type
    if ( $this->{ 'ContentDef' } == undef ){ return $FALSE; }

    my $bRetTmp = $FALSE;
    my $sContDefTypeTmp = lc( $this->{ 'ContentDef' }->getType() );
    if ( $sContDefTypeTmp eq "list" ){
        $bRetTmp = $TRUE;
    }

    return $bRetTmp;
}

#-----------------------------------------------------------------------
# isIllustration
#
# Asserts whether block type is illustration or not
#
# 1, if block is illustration
# 0, if block is NOT illustration
#
#-----------------------------------------------------------------------
sub isIllustration{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "isIllustration()" );
    #}

    #---- Check type
    if ( $this->{ 'ContentDef' } == undef ){ return $FALSE; }

    my $bRetTmp = $FALSE;
    my $sContDefTypeTmp = lc( $this->{ 'ContentDef' }->getType() );
    if ( $sContDefTypeTmp eq "illustration" ){
        $bRetTmp = $TRUE;
    }

    return $bRetTmp;
}

#-----------------------------------------------------------------------
# isComposite
#
# Asserts whether block type is composite or not
#
# 1, if block is composite
# 0, if block is NOT composite
#
#-----------------------------------------------------------------------
sub isComposite{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "isComposite()" );
    #}

    #---- Check type
    if ( $this->{ 'ContentDef' } == undef ){ return $FALSE; }

    my $bRetTmp = $FALSE;
    my $sContDefTypeTmp = lc( $this->{ 'ContentDef' }->getType() );
    if ( $sContDefTypeTmp eq "composite" ){
        $bRetTmp = $TRUE;
    }

    return $bRetTmp;
}

#-----------------------------------------------------------------------
# isRaw
#
# Asserts whether block type is raw or not
#
# 1, if block is raw
# 0, if block is NOT raw
#
#-----------------------------------------------------------------------
sub isRaw{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "isRaw()" );
    #}

    #---- Check type
    if ( $this->{ 'ContentDef' } == undef ){ return $FALSE; }

    my $bRetTmp = $FALSE;
    my $sContDefTypeTmp = lc( $this->{ 'ContentDef' }->getType() );
    if ( $sContDefTypeTmp eq "raw" ){
        $bRetTmp = $TRUE;
    }

    return $bRetTmp;
}
# $V114 End

# $V108 Begin
#-----------------------------------------------------------------------
# setAttribute
#
# Set attribute of block based on given attribute name/value
#
#-----------------------------------------------------------------------
sub setAttribute{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "setAttribute()" ); }

    #---- Get parameters
    #
	# 1. Attribute name
	# 2. Attribute value
    my $sNamePar  = shift;
    my $sValuePar = shift;
    if ( $bLog == $TRUE ){ $theLogger->logMessage( "Processing attribute ($sNamePar) with value ($sValuePar)" ); }

    #---- Assert parameters
    if ( $sNamePar eq "" || $sValuePar eq "" ){ return; }

    #---- Attribute name to mutator method mapping
    my $hrefMethodMapTmp = {
          'startanchor'      => 'setStartAnchor'
        , 'endanchor'        => 'setEndAnchor'
        , 'width'            => 'setWidth'
        , 'height'           => 'setHeight'
        , 'contentdef'       => 'setContentDef'
        , 'next'             => 'setNext'
        , 'quantifier'       => 'setQuantifier'
        , 'contentprocessor' => 'setContentProcessor'
        , 'info'             => 'setInfo'
    };

    my $sMethodNameTmp = $hrefMethodMapTmp->{ lc( $sNamePar ) };
    if ( $sMethodNameTmp eq "" ){ return; } # No matching attribute found, return

    #---- Set attribute
    $this->$sMethodNameTmp( $sValuePar );
}
# $V108 End

# $V111 Begin
#-----------------------------------------------------------------------
# getIndexes
#
# Returns array reference (having indexes) collected from block content
# or undef when no index is collected
#
# Array will have element as given below
# {
#     'name' => <index name>
#     'value' => <index value>
# }
#
#-----------------------------------------------------------------------
sub getIndexes{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "getIndexes()" ); }

    #---- Get content processor
    my $contProcTmp = $this->getContentProcessor();
    if ( $contProcTmp == undef ){ return undef; }

    #---- Get indexes list
    my $arefIndexesTmp = $contProcTmp->getIndexList();
    my @arrIndexesTmp = @{ $arefIndexesTmp };

    #---- Filter indexes detected alone
    @arrIndexesTmp = grep { $_->getResult() ne '' } @arrIndexesTmp;

    #---- Build return value
    my @arrResultTmp = ();
    foreach my $blkIndexTmp ( @arrIndexesTmp ){
        @arrResultTmp[ ( $#arrResultTmp + 1 ) ] = {
		      'name' => $blkIndexTmp->getName()
		    , 'value' => $blkIndexTmp->getResult()
		};
    }

    return \@arrResultTmp;
}
# $V111 End

# $V112 Begin
#-----------------------------------------------------------------------
# getObjects
#
# Get list of block objects
#
# Returns list of block objects (as array reference)
# a. Includes start anchor object on list, if start anchor is not skipped
# b. Includes all objects of current block
# c. Includes end anchor object on list, if end anchor is not skipped
#
#-----------------------------------------------------------------------
sub getObjects{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "getObjects()" ); }
    #if ( $bLog == $TRUE ){ $theLogger->logMessage( "Block: Id=>" . $this->getId() . "<" ); }

    # Check and add start anchor objects
    my @arrObjsListTmp = ();
    my $ancStTmp = $this->getStartAnchor();
    if ( $ancStTmp != undef && $ancStTmp->isSkipped() == $FALSE ){
        push( @arrObjsListTmp, @{ $this->getStartAnchorsList() } );
    }
	
    # Add Block objects
    push( @arrObjsListTmp, @{ $this->getObjsList() } );
	
    # Check and add end anchor objects
    my $ancEdTmp = $this->getEndAnchor();
    if ( $ancEdTmp != undef && $ancEdTmp->isSkipped() == $FALSE ){
        push( @arrObjsListTmp, @{ $this->getEndAnchorsList() } );
    }

    return \@arrObjsListTmp;
}

#-----------------------------------------------------------------------
# isEmpty
#
# Returns TRUE if block has no objects else returns FALSE
#
#-----------------------------------------------------------------------
sub isEmpty{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "isEmpty()" ); }

    #---- Get objects list
    my $arefObjsTmp = $this->getObjects();
    my @arrObjsTmp = @{ $arefObjsTmp };

    return ( @arrObjsTmp <= 0 ) ? $TRUE : $FALSE;
}
# $V112 End

# $V113 Begin
#-----------------------------------------------------------------------
# findEyecatcher
#
# Find eyecatcher on block content
#
# Parameters:
# options     options is an object, it should be defined as follow
#             "options":{
#               'xRange':        {'from':0,'to':100000}, // where xRange, from and to are optional
#               'yRange':        {'from':0,'to':100000}, // where yRange, from and to are optional
#               'reEC':          <regular expression>,   // eyecatcher regexpr
#               'start':         OPTIONAL: specify from which array index the search should start from, default is 0
#               'end':           OPTIONAL: specify up to which array index the search should go, default is maximum of content array
#               'objectType':    Type of eyecatcher object (text, line, vector, image, container). Default is text
#             }
#                  
# Prototypes:
#  ecIndexName = findEyecatcher(content, options)   where options is as defined above
# 
# Example:
#  ecIndexName = findEyecatcher(content,{'reEC' => qr/^Datum.*$/i})
#-----------------------------------------------------------------------
sub findEyecatcher{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "findEyecatcher()" ); }

    #---- Get objects list
    my $arefObjsTmp = $this->getObjects();
    my @arrObjsTmp = @{ $arefObjsTmp };
    if ( @arrObjsTmp <= 0 ){ return undef; }

    return a2w::core::dm::MiningUtils::findEyecatcher( $arefObjsTmp, @_ );
}

#-----------------------------------------------------------------------
# findEyecatcherValue
#
# Find eyecatcher value on block content
#
# Parameters:
# options     options is an object, it should be defined as follow
#             "options":{
#               'xRange':        {'from':0,'to':100000}, // where xRange, from and to are optional
#               'yRange':        {'from':0,'to':100000}, // where yRange, from and to are optional
#               'reEC':          <regular expression>,   // eyecatcher regexpr
#               'reECValue':     <regular expression>,   // eyecatcher value regexpr
#               'start':         OPTIONAL: specify from which array index the search should start from, default is 0
#               'end':           OPTIONAL: specify up to which array index the search should go, default is maximum of content array
#               'objectType':    Type of eyecatcher object (text, line, vector, image, container). Default is text
#               'direction':     Search direction for the eyecatcher value (right, left, top and bottom). Default is right
#               'xTolerance':    OPTIONAL: Specify adjustment in X range to search value. Default is 1
#               'yTolerance':    OPTIONAL: Specify adjustment in Y range to search value. Default is 1
#             }
#                  
# Prototypes:
#  ecIndexValue = findEyecatcherValue(content, options)   where options is as defined above
# 
# Example:
#  ecIndexValue = findEyecatcherValue(content,{'reEC' => qr/^Datum.*$/i, 'reECValue' => qr/^(\d{2}.\d{2}.\d{4})$/i})
#-----------------------------------------------------------------------
sub findEyecatcherValue{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "findEyecatcherValue()" ); }

    #---- Get objects list
    my $arefObjsTmp = $this->getObjects();
    my @arrObjsTmp = @{ $arefObjsTmp };
    if ( @arrObjsTmp <= 0 ){ return undef; }

    return a2w::core::dm::MiningUtils::findEyecatcherValue( $arefObjsTmp, @_ );
}
# $V113 End

#-----------------------------------------------------------------------
# Don't remove the following lines !!!
#-----------------------------------------------------------------------
1;
__END__
