#-------------------------------------------------------------------------------
#  a2w::core::dm::ParagraphBlockFormatter.pm:
#  Data Mining Framework Paragraph Block Formatter
#
#  Author  : AFP2web Team, Maas Holding GmbH
#
#  $V100   2014-05-08    Initial Release
#
#  $V101   2015-08-21    Extended to tolerate texts with minor Y position difference as one line (AFP-279)
#
#  $V102   2015-09-28    Extended to group texts based on font (identifier) (AFP-297)
#
#  $V103   2018-10-26    a. Optimized preparing list of block objects
#                        b. Extended to handle multi type chained blocks
#                        c. Sorted paragraph line objects (based on X position) to ensure proper line context
#                        d. Fixed minor bug in processing empty chained blocks
#                        e. AFP-756: Preprocessed page content to have lines with Y tolerance
#
#  $V104   2018-12-10    AFP-772: Extended to tag line objects under one <P> tag based on "TagLineAsParagraph" flag
#
#-------------------------------------------------------------------------------
package a2w::core::dm::ParagraphBlockFormatter;

#-----------------------------------------------------------------------
# Include required modules
#-----------------------------------------------------------------------
use a2w::TypeConstants;
use a2w::core::log::Logger;
use a2w::core::dm::Constants;
use a2w::core::dm::BlockFormatter;

#-----------------------------------------------------------------------
# Inherit from base class
#-----------------------------------------------------------------------
our @ISA = qw( a2w::core::dm::BlockFormatter );

#-----------------------------------------------------------------------
# Constructor
#-----------------------------------------------------------------------
sub new{
    my $proto = shift;
    my $class = ref( $proto ) || $proto;

    #---- Define boolean values
    $TRUE  = $a2w::TypeConstants::TRUE;     # TRUE  boolean value
    $FALSE = $a2w::TypeConstants::FALSE;    # FALSE boolean value

    #---- Instantiate from base class to inherit base class attributes
    my $this = a2w::core::dm::BlockFormatter->new( @_ );

    #---- Add this derived class specific attributes
    #$this->{ 'someattribute' } = 0;

    bless( $this, $class );

    #---- Get logger
    our $theLogger = a2w::core::log::Logger->getSingleton();
    our $bLog = $theLogger->isRegistered( __PACKAGE__ );

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "new()" );
    #}
    return $this;
}

#-----------------------------------------------------------------------
# Destructor
#-----------------------------------------------------------------------
sub DESTROY{
    my $this = shift;
}

#-----------------------------------------------------------------------
# Mutators
#-----------------------------------------------------------------------

#-----------------------------------------------------------------------
# Accessors
#-----------------------------------------------------------------------

#-----------------------------------------------------------------------
# Workers
#-----------------------------------------------------------------------
#-----------------------------------------------------------------------
# writeFormattedBlock
#
# Asserts whether object fall in given block and returns following value
#
#  0, in case of success
# <0, in case of error
#
#-----------------------------------------------------------------------
sub writeFormattedBlock{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeFormattedBlock()" );
    }

    #---- Parameter
    #
    # 1. Block
    # 2. Visitor
    #
    my $blkCurrentPar = shift;
    my $outVisitorPar = shift;

    #---- Fetch content definition of block
    my $contDefTmp = $blkCurrentPar->getContentDef();
    if ( lc( $contDefTmp->getType() ) ne "paragraph" ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Error! Paragraph block formatter can not process other type (" . $contDefTmp->getType() . ") of blocks<" ); }
        return -1;
    }

    # V103 Begin
    #---- Assert block has content or not ----#
    my $arefObjsListTmp = $blkCurrentPar->getObjects();
    my @arrObjsListTmp = @{ $arefObjsListTmp };

    #---- Get next block in chain
    my $blkNextTmp = $blkCurrentPar->getNextRef();
    if ( @arrObjsListTmp <= 0 ){
        # When current block (or chained block) is empty, ensure to continue with next block in
        # chain (if any exists) else return no error (so that formatting completed properly)
        $blkCurrentPar->setFlushed( $TRUE );
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Warning! Skipped writing empty block (" . $blkCurrentPar->getId() . ")" ); }

        #---- Write chained block contents ----#
        if ( $blkNextTmp != undef ){
            #---- Check and create formatter
            my $fmtNextTmp = $blkNextTmp->getFormatter();
            if ( $fmtNextTmp == undef ){
                $blkNextTmp->_createContentFormatter();
                $fmtNextTmp = $blkNextTmp->getFormatter();
            }

            #---- Write next block
            $iRetTmp = $fmtNextTmp->writeFormattedContent( # Block
                                                             $blkNextTmp
                                                           # Visitor
                                                           , $outVisitorPar
                                                           # Line count
                                                           , 0
                                                         );
            if ( $iRetTmp < 0 ){ return $iRetTmp; }
        }
        return 0;
    }
    # V103 End

    #---- Begin paragraph
    my $iRetTmp = $outVisitorPar->beginParagraph( $blkCurrentPar );
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    #---- Write formatted content of block ----#
    $iRetTmp = $this->_writeFormattedContent( # Block
                                               $blkCurrentPar
                                              # Visitor
                                              , $outVisitorPar
                                              # Line count
                                              , 0
                                            );
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    #---- End paragraph
    if ( $this->isFinalized() == $FALSE ){ $iRetTmp = $outVisitorPar->endParagraph( $blkCurrentPar ); } # V103 Change
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    return 0;
}

#-----------------------------------------------------------------------
# _writeFormattedContent
#
# Writes block content formatted as lines (low level implementation)
#
# >=0, in case of successful writing
# < 0, in case of error
#
#-----------------------------------------------------------------------
sub _writeFormattedContent{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "_writeFormattedContent()" );
    }

    #---- Parameter
    #
    # 1. Block
    # 2. Visitor
    # 3. Line count
    #
    my $blkCurrentPar = shift;
    my $outVisitorPar = shift;
    my $iLineCountPar = shift;
    if ( $bLog == $TRUE ){ $theLogger->logMessage( "Block: Id=>" . $blkCurrentPar->getId() . "<" ); }

    #---- Set flushed
    $blkCurrentPar->setFlushed( $TRUE );

    #---- Fetch content definition of block
    my $contDefTmp = $blkCurrentPar->getContentDef();
    if ( lc( $contDefTmp->getType() ) ne "paragraph" ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Error! Paragraph block formatter can not process other type (" . $contDefTmp->getType() . ") of blocks<" ); }
        return -1;
    }

    # V103 Begin
    #---- Assert block has content or not ----#
    my $arefObjsListTmp = $blkCurrentPar->getObjects();
    my @arrObjsListTmp = @{ $arefObjsListTmp };

    #---- Get next block in chain
    my $blkNextTmp = $blkCurrentPar->getNextRef();
    if ( @arrObjsListTmp <= 0 ){
        # When current block (or chained block) is empty, ensure to continue with next block in
        # chain (if any exists) else return no error (so that formatting completed properly)
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Warning! Skipped writing empty block (" . $blkCurrentPar->getId() . ")" ); }

        #---- Write chained block contents ----#
        if ( $blkNextTmp != undef ){
            #---- Check and create formatter
            my $fmtNextTmp = $blkNextTmp->getFormatter();
            if ( $fmtNextTmp == undef ){
                $blkNextTmp->_createContentFormatter();
                $fmtNextTmp = $blkNextTmp->getFormatter();
            }

            #---- Write next block
            $iRetTmp = $fmtNextTmp->_writeFormattedContent( # Block
                                                              $blkNextTmp
                                                            # Visitor
                                                            , $outVisitorPar
                                                            # Line count
                                                            , $iLineCountPar
                                                          );
            if ( $iRetTmp < 0 ){ return $iRetTmp; }
        }
        return 0;
    }
    # V103 End

    #---- Detect first row Y position
    my $iCurrentYTmp = 0;
    # $V101 Begin
    my $iObjResolutionTmp = 0;
    # $V101 End
    foreach my $elemFirstTmp ( @arrObjsListTmp ){
        if (    $elemFirstTmp != undef
             && $outVisitorPar->isWritable( $elemFirstTmp ) == $TRUE
           ){
            $iCurrentYTmp = $elemFirstTmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_YPOS };
            # $V101 Begin
            $iObjResolutionTmp = $elemFirstTmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_RESOLUTION };
            # $V101 End
            last;
        }
    }

    my $iRetTmp   = -1;
    my $a2wObjTmp = undef;
    my $pomObjTmp = undef;
    my $iLineCountTmp  = $iLineCountPar;
    my $iLineGapTmp    = 0;
    my @arrLineObjsTmp = ();

    # $V102 Begin
    # Texts are grouped
    if ( $contDefTmp->getGroup() == $TRUE ){
        # Group texts based on font (family & size) applied
        if ( $contDefTmp->getGroupBy() eq "font" ){
            my $iCurrentFontTmp = 0;
            my $hrefResTableTmp = '';
            foreach my $elemObjTmp ( @arrObjsListTmp ){
                $a2wObjTmp = $elemObjTmp->{ 'A2WOBJ' };
                $pomObjTmp = $elemObjTmp->{ 'POMOBJ' };

                #---- Assert whether visitor supports this object type ----#
                if ( $outVisitorPar->isWritable( $elemObjTmp ) == $FALSE ){
                    next;    # go to next object
                }

                #---- Write line, if collected ----#
                if (    $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJTYPE } == $a2w::core::dm::Constants::OT_TEXT
                     && $iCurrentFontTmp != $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_FONTID }
                   ){
                    $iCurrentFontTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_FONTID };
                    $iLineCountTmp++;

                    my $a2wFirstTextTmp = undef;
                    my $sTextTmp = "";
                    my $sSepTmp  = $contDefTmp->getSeparator();
                    foreach my $elemLineObjTmp ( @arrLineObjsTmp ){
                        if ( $elemLineObjTmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJTYPE } == $a2w::core::dm::Constants::OT_TEXT ){
                            if ( $sTextTmp eq "" ){
                                $a2wFirstTextTmp = $elemLineObjTmp;
                                $sTextTmp = $elemLineObjTmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_VALUE };
                            }
                            else {
                                $sTextTmp .= $sSepTmp . $elemLineObjTmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_VALUE };
                            }
                        }
                        else {
                            $outVisitorPar->writeObject( $elemLineObjTmp );
                       }
                    } # foreach my $elemLineObjTmp ( @arrLineObjsTmp )
                    if ( $sTextTmp ne "" ){
                        $outVisitorPar->writeGroupedText(   $a2wFirstTextTmp
                                                          , $sTextTmp
                                                          , 0
                                                        );
                    }

                    @arrLineObjsTmp = ();
                } # if ( $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJTYPE } == $a2w::core::dm::Constants::OT_TEXT && $iCurrentFontTmp != $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_FONTID } )

                #---- Collect objects on current line
                @arrLineObjsTmp[ ( $#arrLineObjsTmp + 1 ) ] = $elemObjTmp;
            } # foreach my $elemObjTmp ( @arrObjsListTmp )

            #---- Write last line, if exists ----#
            if ( @arrLineObjsTmp > 0 ){
                $iLineCountTmp++;

                my $a2wFirstTextTmp = undef;
                my $sTextTmp = "";
                my $sSepTmp  = $contDefTmp->getSeparator();
                foreach my $elemLineObjTmp ( @arrLineObjsTmp ){
                    if ( $elemLineObjTmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJTYPE } == $a2w::core::dm::Constants::OT_TEXT ){
                        if ( $sTextTmp eq "" ){
                            $a2wFirstTextTmp = $elemLineObjTmp;
                            $sTextTmp = $elemLineObjTmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_VALUE };
                        }
                        else {
                            $sTextTmp .= $sSepTmp . $elemLineObjTmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_VALUE };
                        }
                    }
                    else {
                        $outVisitorPar->writeObject( $elemLineObjTmp );
                   }
                } # foreach my $elemLineObjTmp ( @arrLineObjsTmp )
                if ( $sTextTmp ne "" ){
                    $outVisitorPar->writeGroupedText(   $a2wFirstTextTmp
                                                      , $sTextTmp
                                                      , $iLineGapTmp
                                                    );
                }
            } # if ( @arrLineObjsTmp > 0 )
        }
        # Group texts based on y positions
        else {
            foreach my $elemObjTmp ( @arrObjsListTmp ){
                $a2wObjTmp = $elemObjTmp->{ 'A2WOBJ' };
                $pomObjTmp = $elemObjTmp->{ 'POMOBJ' };

                #---- Assert whether visitor supports this object type ----#
                if ( $outVisitorPar->isWritable( $elemObjTmp ) == $FALSE ){
                    next;    # go to next object
                }

                #---- Write line, if collected ----#
                if ( $iCurrentYTmp != $pomObjTmp->{ $a2w::core::dm::Constants::AT_ADJ_YPOS } ){ # V103 Change
                    #---- Update row position
                    $iLineGapTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS } - $iCurrentYTmp;
                    $iCurrentYTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_ADJ_YPOS }; # V103 Change
                    $iLineCountTmp++;

                    my $a2wFirstTextTmp = undef;
                    my $sTextTmp = "";
                    my $sSepTmp  = $contDefTmp->getSeparator();
                    foreach my $elemLineObjTmp ( @arrLineObjsTmp ){
                        if ( $elemLineObjTmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJTYPE } == $a2w::core::dm::Constants::OT_TEXT ){
                            if ( $sTextTmp eq "" ){
                                $a2wFirstTextTmp = $elemLineObjTmp;
                                $sTextTmp = $elemLineObjTmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_VALUE };
                            }
                            else {
                                $sTextTmp .= $sSepTmp . $elemLineObjTmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_VALUE };
                            }
                        }
                        else {
                            $outVisitorPar->writeObject( $elemLineObjTmp );
                       }
                    } # foreach my $elemLineObjTmp ( @arrLineObjsTmp )
                    if ( $sTextTmp ne "" ){
                        $outVisitorPar->writeGroupedText(   $a2wFirstTextTmp
                                                          , $sTextTmp
                                                          , $iLineGapTmp
                                                        );
                    }

                    @arrLineObjsTmp = ();
                } # if ( $iCurrentYTmp != $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS } && && $iLineDiffTmp > $iToleranceTmp )

                #---- Collect objects on current line
                @arrLineObjsTmp[ ( $#arrLineObjsTmp + 1 ) ] = $elemObjTmp;
            } # foreach my $elemObjTmp ( @arrObjsListTmp )

            #---- Write last line, if exists ----#
            if ( @arrLineObjsTmp > 0 ){
                #---- Update row position
                $iLineGapTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS } - $iCurrentYTmp;
                $iCurrentYTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_ADJ_YPOS }; # V103 Change
                $iLineCountTmp++;

                my $a2wFirstTextTmp = undef;
                my $sTextTmp = "";
                my $sSepTmp  = $contDefTmp->getSeparator();
                foreach my $elemLineObjTmp ( @arrLineObjsTmp ){
                    if ( $elemLineObjTmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJTYPE } == $a2w::core::dm::Constants::OT_TEXT ){
                        if ( $sTextTmp eq "" ){
                            $a2wFirstTextTmp = $elemLineObjTmp;
                            $sTextTmp = $elemLineObjTmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_VALUE };
                        }
                        else {
                            $sTextTmp .= $sSepTmp . $elemLineObjTmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_VALUE };
                        }
                    }
                    else {
                        $outVisitorPar->writeObject( $elemLineObjTmp );
                   }
                } # foreach my $elemLineObjTmp ( @arrLineObjsTmp )
                if ( $sTextTmp ne "" ){
                    $outVisitorPar->writeGroupedText(   $a2wFirstTextTmp
                                                      , $sTextTmp
                                                      , $iLineGapTmp
                                                    );
                }
            } # if ( @arrLineObjsTmp > 0 )
        } # else if ( $contDefTmp->getGroupBy() eq "font" )
    } # if ( $contDefTmp->getGroup() == $TRUE )
    # Texts are not grouped
    else {
        foreach my $elemObjTmp ( @arrObjsListTmp ){
            $a2wObjTmp = $elemObjTmp->{ 'A2WOBJ' };
            $pomObjTmp = $elemObjTmp->{ 'POMOBJ' };

            #---- Assert whether visitor supports this object type ----#
            if ( $outVisitorPar->isWritable( $elemObjTmp ) == $FALSE ){
                next;    # go to next object
            }

            #---- Write line, if collected ----#
            if ( $iCurrentYTmp != $pomObjTmp->{ $a2w::core::dm::Constants::AT_ADJ_YPOS } ){ # V103 Change
                #---- Update row position
                $iLineGapTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS } - $iCurrentYTmp;
                $iCurrentYTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_ADJ_YPOS }; # V103 Change
                $iLineCountTmp++;

                # V103 Begin
                # When input text content is occur in below sequence
                # --------------------------------
                # ...
                # Text@(3168,8611)>09/04/2018<
                # Text@(1008,8628)>PAYMENT<
                # Text@(1950,8628)>DUE<
                # Text@(2382,8628)>DATE<
                # ...
                # --------------------------------
                # Here they are collected as one paragraph line (using Y position tolerance) but the objects
                # are not in readable sequence (since objects are not sorted based on X position)
                #
                # i,e. With above sequence, line would be read as "09/04/2018 PAYMENT DUE DATE"
                # but expected reading order is "PAYMENT DUE DATE 09/04/2018"
                #
                # So, ensure to sort the paragraph line based on X position before writing
                #
                #@arrLineObjsTmp = sort { $a->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_XPOS } <=> $b->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_XPOS } } @arrLineObjsTmp;
                # V103 End

                $outVisitorPar->beginParagraphLine( $contDefTmp->getTagLineAsParagraph() ); # V104 Change

                $outVisitorPar->writeParagraphLine(   \@arrLineObjsTmp
                                                    , $iLineCountTmp
                                                    , $iLineGapTmp
                                                  );

                $outVisitorPar->endParagraphLine(); # V104 Change
                @arrLineObjsTmp = ();
            } # if ( $iCurrentYTmp != $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS } && && $iLineDiffTmp > $iToleranceTmp )

            #---- Collect objects on current line
            @arrLineObjsTmp[ ( $#arrLineObjsTmp + 1 ) ] = $elemObjTmp;
        } # foreach my $elemObjTmp ( @arrObjsListTmp )

        #---- Write last line, if exists ----#
        if ( @arrLineObjsTmp > 0 ){
            #---- Update row position
            $iLineGapTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS } - $iCurrentYTmp;
            $iCurrentYTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_ADJ_YPOS }; # V103 Change
            $iLineCountTmp++;

            # V103 Begin
            # When input text content is occur in below sequence
            # --------------------------------
            # ...
            # Text@(3168,8611)>09/04/2018<
            # Text@(1008,8628)>PAYMENT<
            # Text@(1950,8628)>DUE<
            # Text@(2382,8628)>DATE<
            # ...
            # --------------------------------
            # Here they are collected as one paragraph line (using Y position tolerance) but the objects
            # are not in readable sequence (since objects are not sorted based on X position)
            #
            # i,e. With above sequence, line would be read as "09/04/2018 PAYMENT DUE DATE"
            # but expected reading order is "PAYMENT DUE DATE 09/04/2018"
            #
            # So, ensure to sort the paragraph line based on X position before writing
            #
            #@arrLineObjsTmp = sort { $a->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_XPOS } <=> $b->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_XPOS } } @arrLineObjsTmp;
            # V103 End

            $outVisitorPar->beginParagraphLine( $contDefTmp->getTagLineAsParagraph() ); # V104 Change

            $outVisitorPar->writeParagraphLine(   \@arrLineObjsTmp
                                                , $iLineCountTmp
                                                , $iLineGapTmp
                                              );

            $outVisitorPar->endParagraphLine(); # V104 Change
        }
    }
    # $V102 End

    #---- Write chained block contents ----#
    if ( $blkNextTmp != undef ){
        #---- Check and create formatter
        my $fmtNextTmp = $blkNextTmp->getFormatter();
        if ( $fmtNextTmp == undef ){
            $blkNextTmp->_createContentFormatter();
            $fmtNextTmp = $blkNextTmp->getFormatter();
        }

        # V103 Begin
        #---- Assert whether next block is of same type
        # a. If it is same, continue current block state on visitor
        # b. If it is NOT same, finalize current block state on visitor and let next block started properly
        if ( lc( $blkNextTmp->getContentDefType() ) eq "paragraph" ){
            #---- Continue next block writing
            $iRetTmp = $fmtNextTmp->_writeFormattedContent( # Block
                                                              $blkNextTmp
                                                            # Visitor
                                                            , $outVisitorPar
                                                            # Line count
                                                            , $iLineCountTmp
                                                          );
            $this->setFinalized( $fmtNextTmp->isFinalized() ); # Current block is also finalized, if block is finalized on next block
        }
        else {
            #---- Finalize current block
            $this->setFinalized( $TRUE );
            $iRetTmp = $outVisitorPar->endParagraph( $blkCurrentPar );
            if ( $iRetTmp < 0 ){ return $iRetTmp; }

            #---- Write next block
            $iRetTmp = $blkNextTmp->write( $outVisitorPar );
        }
        # V103 End
        if ( $iRetTmp < 0 ){ return $iRetTmp; }
    }

    return 0;
}

#-----------------------------------------------------------------------
# Don't remove the following lines !!!
#-----------------------------------------------------------------------
1;
__END__
