#-------------------------------------------------------------------------------
#  a2w/core/file/Utils.pm:
#  PERL module to define file handling utilities
#
#  Author : Paneer, Maas Holding GmbH
#
#  $V100   2016-09-08    Initial release
#
#-------------------------------------------------------------------------------
package a2w::core::file::Utils;

use a2w::TypeConstants;

use a2w::core::log::Logger;

#---- Declare constants
$TRUE  = $a2w::TypeConstants::TRUE;    # TRUE  boolean value
$FALSE = $a2w::TypeConstants::FALSE;   # FALSE boolean value

#-------------------------------------------------------------------------------
# Rename file
#
# Parameters:
# 1. Old filename
# 2. New filename
#
# Returns
# >= 0 in case of success
# < 0 in case of error
#
#-------------------------------------------------------------------------------
sub renameFile{
    #---- Get logger
    $theLogger  = a2w::core::log::Logger->getSingleton();
    my $bLogTmp = $theLogger->isRegistered( __PACKAGE__ );

    #---- Fetch parameters
    # 1. Old filename
    # 2. New filename
    #
    my $sOldNamePar = shift;
    my $sNewNamePar = shift;

    if ( $bLogTmp ){
        $theLogger->logFunctionName( __PACKAGE__, "renameFile( " . $sOldNamePar . ", " . $sNewNamePar . " )" );
    }

    #---- Rename file
    my $iRcTmp = rename( $sOldNamePar, $sNewNamePar );
    if ( $iRcTmp <= 0 ){
        if ( $bLogTmp ){
            $theLogger->logMessage( "Unable to rename file " . $sOldNamePar . " as " . $sNewNamePar . " rc=" . $iRcTmp . " reason=" . $! );
        }

        return ( -1, "Unable to rename file " . $sOldNamePar . " as " . $sNewNamePar . " rc=" . $iRcTmp . " reason=" . $! );
    }

    return 0;
}

#-------------------------------------------------------------------------------
# Merge files
#
# Parameters:
# 1. Target file name (to which the content of given files will be appended)
# 2. Files to be merged (Array reference of file names)
# 3. Delete inputs flag (optional, default is to delete input files after merge)
#
# Returns
# >= 0 in case of success
# < 0 in case of error
#
#-------------------------------------------------------------------------------
sub mergeFiles{
    #---- Get logger
    $theLogger  = a2w::core::log::Logger->getSingleton();
    my $bLogTmp = $theLogger->isRegistered( __PACKAGE__ );

    #---- Fetch parameters
    # 1. Target file name (to which the content of given files will be appended)
    # 2. Files to be merged (Array reference of file names)
    # 3. Delete inputs flag (optional, default is to delete input files after merge)
    #
    my $sTargetFNPar = shift;
    my $arefFilesPar = shift;
    my $bDeleteFilesPar = $TRUE;
    if ( @_ > 0 ){ $bDeleteFilesPar = shift; }

    if ( $bLogTmp ){
        $theLogger->logFunctionName( __PACKAGE__, "mergeFiles( " . $sTargetFNPar . ", " . $arefFilesPar . ", " . $bDeleteFilesPar . " )" );
    }

    #---- Assert parameters
    my @arrFilesTmp = @{ $arefFilesPar };
    if ( @arrFilesTmp <= 0 ){ return ( -1, "Empty files list to merge files" ); }

    #---- Assert whether target file exists, else move first file in list as target
    my $sFNTmp = "";
    my $iRcTmp = 0;
    if (    !(-e $sTargetFNPar)
         && $bDeleteFilesPar == $TRUE
       ){
        $sFNTmp = shift @arrFilesTmp;
        $iRcTmp = rename( $sFNTmp, $sTargetFNPar );
        if ( $iRcTmp <= 0 ){
            if ( $bLogTmp ){
                $theLogger->logMessage( "Unable to rename file $sFNTmp as $sTargetFNPar. rc=$iRcTmp reason=" . $! );
            }
            return ( -1, "Unable to rename file $sFNTmp as $sTargetFNPar in mergeFiles. rc=$iRcTmp reason=" . $! );
        }
    }

    #---- Initialize
    local $/ = \32768; # Set read block size to 32 Kb
    local *TARGET;
    $iRcTmp = open( TARGET, ">>$sTargetFNPar" );
    if ( $iRcTmp == 0 ){
        return ( -2, "Unable to open target file $sTargetFNPar in mergeFiles. rc=$iRcTmp reason=" . $! );
    }
    binmode( TARGET );

    #---- Iterate through files and merge them to target
    foreach $sFNTmp ( @arrFilesTmp ){
        local *MERGE;
        $iRcTmp = open( MERGE, "<$sFNTmp" );
        if ( $iRcTmp == 0 ){ close( TARGET ); return ( -3, "Unable to open merge file $sFNTmp in mergeFiles. rc=$iRcTmp reason=" . $! ); }
        binmode( MERGE );
        { local $_; print TARGET $_ while <MERGE>; }
        close( MERGE );

        if ( $bDeleteFilesPar == $TRUE ){
            #---- Delete current input file
            my ( $iDelRetTmp, $sMsgTmp ) = deleteFile( $sFNTmp );
            if ( $iDelRetTmp < 0 ){ close( TARGET ); return ( $iDelRetTmp, $sMsgTmp ); }
        }
    }

    #---- Finalize
    close( TARGET );

    return 0;
}

#-------------------------------------------------------------------------------
# Delete file
#
# Parameter:
# 1. Filename
#
# Returns
# >= 0 in case of success
# < 0 in case of error
#
#-------------------------------------------------------------------------------
sub deleteFile{
    #---- Get logger
    $theLogger  = a2w::core::log::Logger->getSingleton();
    my $bLogTmp = $theLogger->isRegistered( __PACKAGE__ );

    #---- Fetch parameter
    # 1. Filename
    #
    my $sFilenamePar = shift;

    if ( $bLogTmp ){
        $theLogger->logFunctionName( __PACKAGE__, "deleteFile( " . $sFilenamePar . " )" );
    }

    #---- Delete file
    my $iRcTmp = unlink( $sFilenamePar );
    if ( $iRcTmp <= 0 ){
        if ( $bLogTmp ){
            $theLogger->logMessage( "Unable to delete file " . $sFilenamePar . " rc=" . $iRcTmp . " reason=" . $! );
        }

        return ( -1, "Unable to delete file " . $sFilenamePar . " rc=" . $iRcTmp . " reason=" . $! );
    }

    return 0;
}

#-------------------------------------------------------------------------------
# Don't remove the following lines !!!
#-------------------------------------------------------------------------------
1;
__END__
