#-------------------------------------------------------------------------------
#  a2w/core/path/Utils.pm:
#
#  Path functionalities specific to OS and filesystem
#
#  Author    : Fa. Maas
#
#  $V100    2016-09-01    Initial Release
#
#-------------------------------------------------------------------------------
package a2w::core::path::Utils;

use a2w::TypeConstants;

use a2w::core::bo::BOPool;
use a2w::core::process::Logger;

#---- Declare constants
$TRUE  = $a2w::TypeConstants::TRUE;    # TRUE  boolean value
$FALSE = $a2w::TypeConstants::FALSE;   # FALSE boolean value

#-----------------------------------------------------------------------
# Worker
#-----------------------------------------------------------------------
#=======================================================================
# isAbsolute
#
# Asserts whether given path is absolute or not
#
# Returns
# $TRUE  (1) in case the given path is absolute
# $FALSE (0) in case the given path is relative
#
#=======================================================================
sub isAbsolute{
    #---- Get logger
    $theLoggerLocal = a2w::core::bo::BOPool::getBOPool()->getLogger();
    my $bLogTmp  = $theLoggerLocal->isRegistered( __PACKAGE__ );

    #---- Fetch parameter
    #
    # 1. Path
    #
    my $sPathPar = shift;

    if ( $bLogTmp ){
        $theLoggerLocal->logFunctionName( __PACKAGE__, "isAbsolute($sPathPar)" );
    }

    #---- Windows system
    my $bAbsPathTmp = $TRUE;
    if ( lc( $^O ) =~ /mswin.*/ ){
        if ( $sPathPar !~ /^?[\:]/ ){ $bAbsPathTmp = $FALSE; }
    }
    #---- *NIX system
    else {
        if ( $sPathPar !~ /^\// ){ $bAbsPathTmp = $FALSE; }
    }

    if ( $bLogTmp ){
        $theLoggerLocal->logMessage( $sPathPar . " is " . ( $bAbsPathTmp == $TRUE ) ? "absolute" : "relative" );
    }

    return $bAbsPathTmp;
}

#=======================================================================
# list
#
# Iterates and returns list of files/directories on given path
#
# Returns
# Array of names on success
# undef on error
#
#=======================================================================
sub list{
    #---- Get logger
    $theLoggerLocal = a2w::core::bo::BOPool::getBOPool()->getLogger();
    my $bLogTmp  = $theLoggerLocal->isRegistered( __PACKAGE__ );

    #---- Fetch parameter
    #
    # 1. Path
    # 2. Relative names flag (indicates whether return list must have 'path + name' or not, default is relative names with out path)
    # 3. Filter pattern (regular expression to filter out list elements, default is to list all except '.' and '..')
    #
    my $sPathPar = shift;
    my $bRelativePar = $TRUE;
    my $reFilterPar = qr/[^\.]/;
    if ( @_ > 0 ){ $bRelativePar = shift; }
    if ( @_ > 0 ){ $reFilterPar = shift; }

    if ( $bLogTmp ){
        $theLoggerLocal->logFunctionName( __PACKAGE__, "list($sPathPar, $bRelativePar, $reFilterPar)" );
    }

    #---- Assert parameters
    if ( !(-d $sPathPar) ){
        if ( $bLogTmp ){
            $theLoggerLocal->logMessage( "Invalid path (" . $sPathPar . ")" );
        }

        return undef;
    }

    #---- Open path
    if ( opendir( LISTPATH, $sPathPar ) <= 0 ){
        if ( $bLogTmp ){
            $theLoggerLocal->logMessage( "Unable to open path (" . $sPathPar . "). Reason: " . $! );
        }

        return undef;
    }

    #---- Fetch list from path
    my @arrListTmp = grep { $_ =~ /$reFilterPar/ } readdir( LISTPATH );

    #---- Close path
    closedir( LISTPATH );

    #---- Add path to list elements if absolute is requested
    if ( $bRelativePar == $FALSE ){
        #---- Check and add slash to the path at end
        if ( $sPathPar !~ /[\\\/]$/ ){ $sPathPar .= ( lc( $^O ) =~ /mswin.*/ ) ? "\\" : "/"; }

        #---- 'readdir' will return names without path, so add path to each entry
        for ( my $i = 0; $i < @arrListTmp; $i++ ){
            $arrListTmp[ $i ] = $sPathPar . $arrListTmp[ $i ];
        }
    }

    return @arrListTmp;
}

#-----------------------------------------------------------------------
# Don't remove the following lines !!!
#-----------------------------------------------------------------------
1;
__END__

