#-------------------------------------------------------------------------------
#  a2w/core/process/Logger.pm:
#
#  SINGLETON process logging engine
#
#  Author  : Fa. Maas
#  Copyright : (C) 2009-2016 by Maas Holding GmbH
#
#  $V100   2009-08-11    Initial Release
#
#  $V101   2010-05-26    Extended to register additional classes
#
#  $V102   2011-02-09    Added flush option to ensure all entries are logged proper
#                        in case of error
#
#  $V103   2015-02-19    Extended naming log file with random value and pid to make
#                        it unique for concurrent processes (as given below)
#                        <Log Prefix>_<Timestamp in format YYYYMMDD_hhmmss>_<Process ID>_<Random value of 3 digits>
#
#  $V104   2016-11-23    Extended to log hash structure
#
#-------------------------------------------------------------------------------
package a2w::core::process::Logger;

use a2w::TypeConstants;   # Global constants and variables

#-----------------------------------------------------------------------
# Constructor
#-----------------------------------------------------------------------
sub new{
    my $proto = shift;
    my $class = ref( $proto ) || $proto;

    #---- Define boolean values
    $TRUE  = $a2w::TypeConstants::TRUE;     # TRUE  boolean value
    $FALSE = $a2w::TypeConstants::FALSE;    # FALSE boolean value

    my $this = {
          'bLog'                  => $FALSE # Enable/Disable logging in modules
        , 'arefRegisteredClasses' => 0      # Array reference having list of registered class names
        , 'bFileOpen'             => $FALSE # Flag indicating whether log file is opened or not
        , 'sFilename'             => ""     # Log file name
        , 'fHandle'               => -100   # Log file handle
        , 'tStartTime'            => 0      # Start time
        , 'tEndTime'              => 0      # End time
    };

    bless( $this, $class );

    #---- Get Parameters
    #
    # 1. Log flag (of type boolean)
    #
    $this->{ bLog } = shift;

    return $this;
}

#-----------------------------------------------------------------------
# Destructor
#-----------------------------------------------------------------------
sub DESTROY{
    my $this = shift;
}

#-----------------------------------------------------------------------
# Mutator
#-----------------------------------------------------------------------
sub setLog{
    my $this = shift;

    #---- Get parameter
    #
    # 1. Log flag (of type boolean)
    #
    my $bLogPar = shift;

    #---- Set log flag
    $this->{ bLog } = $bLogPar;
}

sub setFilename{
    my $this = shift;

    #---- Get parameter
    #
    # 1. Log filename (of type string)
    #
    my $sLogFilenamePar = shift;

    #---- Set log filename
    $this->{ sFilename } = $sLogFilenamePar;
}

sub setStartTime{
    my $this = shift;

    #---- Get parameter
    #
    # 1. Start time (of type time)
    #
    $tStartTimePar = shift;

    #---- Set Start time
    $this->{ tStartTime } = $tStartTimePar;
}

sub setEndTime{
    my $this = shift;

    #---- Get parameter
    #
    # 1. End time (of type time)
    #
    $tEndTimePar = shift;

    #---- Set End time
    $this->{ tEndTime } = $tEndTimePar;
}

#-----------------------------------------------------------------------
# Accessor
#-----------------------------------------------------------------------
sub getLog{
    my $this = shift;

    #---- Return log flag
    return $this->{ bLog };
}

sub getStartTime{
    my $this = shift;

    #---- Get Start time
    return $this->{ tStartTime };
}

sub getEndTime{
    my $this = shift;

    #---- Get End time
    return $this->{ tEndTime };
}

sub getFilename{
    my $this = shift;

    #---- Get log filename
    return $this->{ sFilename };
}

#-----------------------------------------------------------------------
# Workers
#-----------------------------------------------------------------------
#-----------------------------------------------------------------------
# Initialize 
#-----------------------------------------------------------------------
sub initialize{
    my $this = shift;

    #---- Get parameter
    my $tStartTimePar = shift;
    my $sPathPar = shift;
    my $sFilePrefixPar = shift;
    my $bLogPar = shift;

    $this->setLog( $bLogPar );       
    $this->setStartTime( $tStartTimePar );

    #---- Open log file
    $this->open( $sPathPar, $sFilePrefixPar );
}

# $V101 Begin
#-----------------------------------------------------------------------
# Register a class additionally for logging
#-----------------------------------------------------------------------
sub registerAdditionalClass{
    my $this = shift;

    #---- Get parameter
    #
    # 1. Class name (of type string)
    #
    my $sClassnamePar = shift;

    #---- Get registered classes list
    $arrClassListTmp = $this->{ 'arefRegisteredClasses' };

    #---- Add given class
    my $iLenTmp = @{ $arrClassListTmp };
    $arrClassListTmp->[ $iLenTmp ] = lc( $sClassnamePar );
}
# $V101 End

#-----------------------------------------------------------------------
# Register collection of classes for logging
#-----------------------------------------------------------------------
sub registerClasses{
    my $this = shift;

    #---- Get parameter
    #
    # 1. List of comma separated class names (of type string)
    #
    my $sClassnameListPar = shift;

    #---- Split and create array
    my @arrClassListTmp = split( ',', $sClassnameListPar );

    #---- Convert all class names to lowercase for easy comparison on later
    for( my $i = 0; $i < @arrClassListTmp; $i++ ){
        @arrClassListTmp[ $i ] = lc( @arrClassListTmp[ $i ] );
    }

    #---- Set registered classes
    $this->{ 'arefRegisteredClasses' } = \@arrClassListTmp;
}


#-----------------------------------------------------------------------
# Is class registered for logging
#-----------------------------------------------------------------------
sub isRegistered{
    my $this = shift;

    #---- Get parameter
    #
    # 1. Class name (of type string)
    #
    my $sClassnamePar = shift;

    #---- Get registered classes
    my @arrClassListTmp = @{ $this->{ 'arefRegisteredClasses' } };
    my $iCountTmp = @arrClassListTmp;

    if ( $iCountTmp == 1
         && lc( @arrClassListTmp[ 0 ] ) eq "all" ){
        return $TRUE;
    }

    my $bRetTmp = $FALSE;
    my $sClassnameTmp = lc( $sClassnamePar );

    for( my $i = 0; $i < $iCountTmp; $i++ ){
        if ( $sClassnameTmp eq @arrClassListTmp[ $i ] ){
            $bRetTmp = $TRUE;

            last;   # break the loop
        }
    }

    #---- Return log flag
    return $bRetTmp;
}

#-----------------------------------------------------------------------
# Open file
#-----------------------------------------------------------------------
sub open{
    my $this = shift;

    #---- Get parameter
    #
    # 1. Log path (of type string)
    # 2. Log filename (of type string)
    #
    my $sPathPar = shift;
    my $sPrefixPar = shift;

    #---- Store start time
    if ( $this->{ tStartTime } == 0 ){
        $this->{ tStartTime } = time();
    }

    #---- Build filename
    #   0    1    2     3     4    5     6     7     8
    #($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime(time);

    my @arrCurrentTimeTmp = localtime( $this->{ tStartTime } );
    @arrCurrentTimeTmp[ 5 ] += 1900;        # Year is not complete 4 digit, so adding 1900
    @arrCurrentTimeTmp[ 4 ]++;              # Day of month starts from 0, so adding 1

    # $V103 Begin
    #---- Get process id
    my $iPIDTmp = $$;

    #---- Evaluate a 3 digit random value
    my $iRandValTmp = rand( 999 );

    #my $sFilenameTmp = sprintf(   "%s_%04d%02d%02d_%02d%02d%02d.log"
    my $sFilenameTmp = sprintf(   "%s_%04d%02d%02d_%02d%02d%02d_%06d_%03d.log"
                                , $sPrefixPar
                                , @arrCurrentTimeTmp[ 5 ]
                                , @arrCurrentTimeTmp[ 4 ]
                                , @arrCurrentTimeTmp[ 3 ]
                                , @arrCurrentTimeTmp[ 2 ]
                                , @arrCurrentTimeTmp[ 1 ]
                                , @arrCurrentTimeTmp[ 0 ]
                                , $iPIDTmp
                                , $iRandValTmp
                              );
    # $V103 End

    if ( $sPathPar ne "" ){
        $sFilenameTmp = "$sPathPar/$sFilenameTmp";
    }

    #---- Open file
    if ( open( $this->{ fHandle }, ">$sFilenameTmp" ) ){
        $this->{ bFileOpen } = $TRUE;
        $this->{ sFilename } = $sFilenameTmp;
        # $V102 Begin
        #---- Turn on auto flush for log file handle
        select( ( select( $this->{ fHandle } ), $| = 1 )[ 0 ] );
        # $V102 End
    }

    return $this->{ bFileOpen };
}

#-----------------------------------------------------------------------
# Close Log File
#-----------------------------------------------------------------------
sub close{
    my $this = shift;

    if ( $this->{ bFileOpen } == $FALSE ){
        return -1;
    }

    #---- Open file
    close( $this->{ fHandle } );
    $this->{ bFileOpen } = $FALSE;

    return 0;
}

#-----------------------------------------------------------------------
# Log function name
#-----------------------------------------------------------------------
sub logFunctionName{
    my $this = shift;

    if ( $this->{ bFileOpen } == $FALSE ){
        return -1;
    }

    #---- Get parameter
    #
    # 1. Module name (of type string)
    # 2. Function name (of type string)
    #
    my $sModuleNamePar = shift;
    my $sFuncNamePar = shift;

    my $hFileHandleTmp = $this->{ fHandle };
    printf $hFileHandleTmp ( "[%-54s] %s\n"
                             , $sModuleNamePar
                             , $sFuncNamePar
                           );

    return 0;
}

#-----------------------------------------------------------------------
# Log message
#-----------------------------------------------------------------------
sub logMessage{
    my $this = shift;

    if ( $this->{ bFileOpen } == $FALSE ){
        return -1;
    }

    #---- Get parameter
    #
    # 1. Message to log (of type string)
    #
    my $sMessagePar = shift;

    my $hFileHandleTmp = $this->{ fHandle };
    printf $hFileHandleTmp ( "%-54s   %s\n"
                             , " "
                             , $sMessagePar
                           );

    return 0;
}

# $V104 Begin
#-----------------------------------------------------------------------
# Log hash
#-----------------------------------------------------------------------
sub logHashMessage{
    my $this = shift;

    if ( $this->{ bFileOpen } == $FALSE ){
        return -1;
    }

    #---- Get parameter
    #
    # 1. Message to log (of type hash reference)
    # 2. Indent string (optional, of type string)
    #
    my $hrefMsgPar = shift;
    my $sIndentPar = "";
    if ( @_ > 0 ){ $sIndentPar = shift; }

    my $hFileHandleTmp = $this->{ fHandle };

    #---- Process and print hash structure
    my @arrKeysTmp = sort keys( %{ $hrefMsgPar } );
    foreach my $key ( @arrKeysTmp ){
        if ( ref( $hrefMsgPar->{ $key } ) eq "HASH" ){
            printf $hFileHandleTmp ( "%-54s   %s%s:\n", " ", $sIndentPar, $key );
            $this->logHashMessage( $hrefMsgPar->{ $key }, $sIndentPar . " " );
        }
        elsif ( ref( $hrefMsgPar->{ $key } ) eq "ARRAY" ){
            my @arrElemTmp = @{ $hrefMsgPar->{ $key } };
            printf $hFileHandleTmp ( "%-54s   %s%s:>%s<\n", " ", $sIndentPar, $key, "@arrElemTmp" );
        }
        else{
            printf $hFileHandleTmp ( "%-54s   %s%s:>%s<\n", " ", $sIndentPar, $key, $hrefMsgPar->{ $key } );
        }
    }

    return 0;
}
# $V104 End

#-----------------------------------------------------------------------
# Don't remove the following lines !!!
#-----------------------------------------------------------------------
1;
__END__

