#-------------------------------------------------------------------------------
#  AFP Spools Perl Processing Module
#
#  Author  : Fa. Maas
#
#  $V100   2003-05-02   Initial Release
#  $V101   2003-06-17   Modified suffix (ordering number) of protocol
#                       (SkippedPages and VSNRs) file names as prefix
#  $V102   2005-03-08   Extended to support CTVNEU spools
#  $V103   2005-03-10   Fixed bug (only one entry exist) with VSNR protocol for
#                       CTVNEU spools
#  $V104   2005-03-24   Fixed bug (VSNR of skipped page is written) with VSNR
#                       protocol for CTVNEU spools
#  $V105   2005-04-11   Fixed bug (VSNR is written for each page) with VSNR
#                       protocol for CTVNEU spools
#  $V106   2005-04-25   Fixed bug (empty) with VSNR protocol for CTVNEU spools
#  $V107   2005-04-26   Fixed bug with PERL error "Can't "last" outside a loop block at"
#  $V108   2005-04-29   Fixed bug with document level I_ARCHIV=N skipping process
#  $V109   2005-08-24   a. Extended to access page lines and to remove them from
#                          output presentation
#                       b. Removed "I_ARVHIV" index from "StoreProcessRequest" xml
#                          when it's value is "J"
#  $V110   2008-07-29   a. Modified module name as "a2w::process::afp"
#                       b. Modified index accessing based on A2W V3.2 new interfaces
#  $V111   2008-11-11   Fixed a bug when deleting AFP Index
#
#  $V112   2009-07-21   Extended to congifure skipped pages and VNR protocol using
#                       "LoggingLevel" values as given below
#                       "pspg" => Turn on skipped pages protocol (by default this is turned off)
#                       "pvnr" => Turn on VNR protocol (by default this is turned off)
#
#  $V113  2019-02-08  Migrated logging to Scripting Facility framework standard
#
#-------------------------------------------------------------------------------
# $V110 Begin
#package a2w::afpprocess::afp;
package a2w::process::afp;
# $V110 End

use a2w::Config;
use a2w::Document;
use a2w::Index;
use a2w::Kernel;
use a2w::Line;
use a2w::Page;
use a2w::Text;

use a2w::ConfigConstants; # V113 Change
use a2w::core::log::Logger; # V113 Change

#-----------------------------------------------------------------------
# Layout Type: all standard AFPs having following Page Index: ARCHIV='N'
#-----------------------------------------------------------------------
# NOTE :
#    To retrieve the passed arguments to any # subroutine PERL provides 
#    the user a special array variable "@_" where @ specifies the
#    variable is of type "array" and the name of the array is "_".
#    This array can be treated as normal array variable and could be 
#    used to retrieve the passed arguments.
#
#     @_         gives an array of parameters.
#     $_[idx] gives particular element from the
#             parameter array "@_"
#
# To assign to particular scalar or other type
# of variable we should do as below
#    
#     ($param1,$param2) = @_;
#
# is same as below
#
#     ($param1) = $_[0];
#    ($param2) = $_[1];
#
#-----------------------------------------------------------------------

$sVersion = "1.1.2";	# $V112

#-----------------------------------------------------------------------
# Initialize once per process
#-----------------------------------------------------------------------
sub initialize{

    #---- Get Parameters of initialize
    #
    # 1. AFP2web ini (a2w::Config)
    # 2. AFP2web kernel (a2w::Kernel)
    # 3. Process spool (a2w::request::ProcessSpool)
    # 4. Start time (time)
    $a2wConfigPar = shift;
    $a2wKernelPar = shift;
    $reqProcessSpoolPar = shift;
    $tStartTimePar = shift;

    #---- Define boolean values
    $TRUE  = 1;    # TRUE  boolean value
    $FALSE = 0;    # FALSE boolean value

    #---- Define afp2web return values
    $APPEND = 0;    # append page to Current Document
    $SKIP   = 1;    # skip page
    $NEWDOC = 2;    # new document

    #---- TRUE when all the pages of the current document must be skipped
    $bSkipDoc = $FALSE; 

    # V113 Begin
    #---- Get logger
    my $sLogLevelTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::LOGGINGLEVEL );

    our $theLogger = a2w::core::log::Logger->getSingleton();
    our $bLog = $theLogger->isRegistered( __PACKAGE__ );
    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "initialize()" ); }

    $sOutputFilePath   = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::OUTPUTFILEPATH );
    # V113 End

    #---- Request Id
    $sRequestId = $reqProcessSpoolPar->getRequestId();

    #---- Time stamp
    # (sec,min,hour,mday,mon,year,wday,yday,isdst)
    my @StartTimeTmp = localtime( $tStartTimePar );
    $sTimeStamp = sprintf( "%04d%02d%02d_%02d%02d%02d",
                           ( @StartTimeTmp[ 5 ] + 1900 ),
                           ( @StartTimeTmp[ 4 ] + 1 ),
                           @StartTimeTmp[ 3 ],
                           @StartTimeTmp[ 2 ],
                           @StartTimeTmp[ 1 ],
                           @StartTimeTmp[ 0 ]
                         );

    if ( $bLog == $TRUE ){ $theLogger->logMessage( "RequestId=" . $sRequestId . ", TimeStamp=" . $sTimeStamp ); } # V113 Change

    # $V112 Begin
    #---- Protocol skipped pages (by default this is turned off)
    $bProtocolSkippedPages = $FALSE;
    if ( index( lc( $sLogLevelTmp ), "pspg" ) >= 0 ){
        #---- Turned on, when logging level is set to "pspg"
        $bProtocolSkippedPages = $TRUE;
    }

    #---- Protocol VSNR (by default this is turned off)
    $bProtocolVNR = $FALSE;
    if ( index( lc( $sLogLevelTmp ), "pvnr" ) >= 0 ){
        #---- Turned on, when logging level is set to "pvnr"
        $bProtocolVNR = $TRUE;
    }
    # $V112 End

    #---- File Open status for VSNR protocol
    $bProtocolVNROpen = $FALSE;

    #---- File Open status for SkippedPages protocol
    $bProtocolSkippedPagesOpen = $FALSE;

    #---- Skipped pages protocol filename
    $sSkipPageProtocolFilename = "";

    #---- VSNR protocol filename
    $sVSNRProtocolFilename = "";

    #---- Document name
    $sDocumentName = "";

    #---- Reason for skipping page/document
    $sSkipReason = "";

    #-------------------------------------------------------------------
    #    Setting AutoSplit to true means that document splitting will
    #    done based on the standard AFP index details. In this case the 
    #    return value 2 of the afp2web() function is ignored.
    #
    #    Setting AutoSplit to false means that document splitting will
    #    be defined by the afp2web() function.
    #-------------------------------------------------------------------
    $a2wConfigPar->setAttribute( $a2w::ConfigConstants::AUTOSPLIT, "on" ); # V113 Change

    return 0;
}

#-----------------------------------------------------------------------
# InitializeDoc for each document
#-----------------------------------------------------------------------
sub initializeDoc{

    #---- Get Parameter of initializeDoc(Par:mhtDocument*)
    ($a2wDocumentPar) = @_;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "initializeDoc( " . $a2wDocumentPar->getId() . " )" ); } # V113 Change

    #---- Get Document name
    if ( $sDocumentName == "" ){ $sDocumentName = $a2wDocumentPar->getName(); }

    #---- Reset bSkipDoc
    $bSkipDoc = $FALSE;    # TRUE when all the pages of the current document must be skipped

    #---- Check and open SkippedPages protocol file
    if ( $bProtocolSkippedPages == $TRUE                # $V112 Change
         && $bProtocolSkippedPagesOpen == $FALSE ){
        #---- build file name
        #---- $V101 Begin
        #$svSkippedPagesProtocolFileNameTmp = sprintf( "%s%s_SkippedPages_%s_3.txt", $sOutputFilePath, $sRequestId, $sTimeStamp );
        $svSkippedPagesProtocolFileNameTmp = sprintf( "%s3_%s_SkippedPages_%s.txt", $sOutputFilePath, $sRequestId, $sTimeStamp );
        #---- $V101 End

        #---- Open Index file
        $svFileOpenSuccessTmp = open( SKIPPAGEPROTOCOLFILE, ">$svSkippedPagesProtocolFileNameTmp" );
        if ( $svFileOpenSuccessTmp ){
            $bProtocolSkippedPagesOpen = $TRUE;
            $sSkipPageProtocolFilename = $svSkippedPagesProtocolFileNameTmp;
        } else {
            return -1;
        }
    }

    #---- Check and open VSNR protocol file
    if ( $bProtocolVNR == $TRUE &&
         $bProtocolVNROpen == $FALSE ){
        #---- build file name
        #---- $V101 Begin
        #$svVNRProtocolFileNameTmp = sprintf( "%s%s_VSNRs_%s_4.txt", $sOutputFilePath, $sRequestId, $sTimeStamp );
        $svVNRProtocolFileNameTmp = sprintf( "%s4_%s_VSNRs_%s.txt", $sOutputFilePath, $sRequestId, $sTimeStamp );
        #---- $V101 End

        #---- Open Index file
        $svFileOpenSuccessTmp = open( VNRPROTOCOLFILE, ">$svVNRProtocolFileNameTmp" );
        if ( $svFileOpenSuccessTmp ){
            $bProtocolVNROpen = $TRUE;
            $sVSNRProtocolFilename = $svVNRProtocolFileNameTmp;
        } else {
            return -1;
        }
    }

    #---- Document page count                                                   #---- $V102 Change
    $svDocPageCount = 0;

    return 0;
}

#-----------------------------------------------------------------------
# InitializePage for each page
#-----------------------------------------------------------------------
sub initializePage{

    #---- Get Parameter of initializePage(Par:mhtPage*)
    ($a2wPagePar) = @_;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "initializePage()" ); } # V113 Change
    
    #---- $V102 Change Begin
    if ( $a2wDocumentPar != 0 ){
        #---- $V110 Change Begin
        #---- Fetch first DocumentIndex item
        my $a2wIndexTmp = $a2wDocumentPar->getFirstIndex();

        #---- Reset bSkipDoc
        # TRUE when all the pages of the current document must be skipped
        $bSkipDoc = $FALSE;                         #---- $V108 Change

        #---- $V105 Change Begin
        while ( $a2wIndexTmp != 0 ){
            if ( $bLog == $TRUE ){ $theLogger->logMessage( $a2wIndexTmp->getName() . "=" . $a2wIndexTmp->getValue() ); } # V113 Change

            #---- Check whether the page has to be archived
            #---- Watch out! Comparison is case sensitive !
            if ( $a2wIndexTmp->getName() eq "I_ARCHIV"
                 && $a2wIndexTmp->getValue() eq "N"
               ){
                $bSkipDoc = $TRUE; # skip all the pages of the current document
                $sSkipReason = "I_ARCHIV=N at doc level";
            }

            #---- Check whether we have I_VNR index only in case of VNR protocol
            if ( $bProtocolVNR == $TRUE
                 && $a2wIndexTmp->getName() eq "I_VNR"
               ){
                #---- Protocol the VNR
                $I_VNR = $a2wIndexTmp->getValue();
            }

            #---- If inner while-loop has been left, leave this while-loop too.
            if ( $bSkipDoc == $TRUE ){          #---- $V107 Change
                last; # leave while loop
            }                                   #---- $V107 Change

            #---- Fetch next PagePageGroupIndex item
            $a2wIndexTmp = $a2wDocumentPar->getNextIndex();
        }
        #---- $V105 Change End
        #---- $V110 Change End
    }
    #---- $V102 Change End

    return 0;
}

#-----------------------------------------------------------------------
# Main entry method
# Return values:
#        < 0:    error
#         0:    append page to Current Document
#         1:    skip page
#         2:    new document
#-----------------------------------------------------------------------
sub afp2web{

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "afp2web()" ); } # V113 Change

    #---- When bSkipDoc == $TRUE, skip all the pages of the current document
    if ( $bSkipDoc == $TRUE ){
        if ( $bProtocolSkippedPages == $TRUE ){    # $V112 Change
            protocolSkippedPage();
            $sSkipReason = "I_ARCHIV=N at doc level";
        }
        return $SKIP;
    }

    #---- Set default return value 
    $svRetTmp = $APPEND; # default: append page

    #---- Check whether the current page has to be skipped
    if ( $a2wPagePar != 0 ){
        #---- Fetch first PageIndex item
        my $a2wPageIndexTmp = $a2wPagePar->getFirstIndex();

        #---- Process index
        while ( $a2wPageIndexTmp != 0 ){
            if ( $a2wPageIndexTmp->getName()  eq "I_ARCHIV" ){
                #---- Skip if value is "N"
                my $svArchiveTmp = $a2wPageIndexTmp->getValue();

                if ( $svArchiveTmp eq "N" ){
                    $svRetTmp = $SKIP; # skip page
                    if ( $bProtocolSkippedPages == $TRUE ){    # $V112 Change
                        $sSkipReason = "I_ARCHIV=N at page level";
                        protocolSkippedPage();
                    }
                }
                elsif ( $svArchiveTmp eq "J" ){
                    $a2wPageIndexTmp->remove();	# $V111
                }
            }
            #---- $V109 End

            #---- Check whether we have I_VNR index only in case of VNR protocol
            if ( $bProtocolVNR == $TRUE &&
                 $a2wPageIndexTmp->getName()  eq "I_VNR" ){
                #---- Protocol the VNR
                $I_VNR = $a2wPageIndexTmp->getValue();
            }

            #---- If inner while-loop has been left, leave this while-loop too.
            if ( $svRetTmp == $SKIP ){
                last; # leave while loop
            }

            #---- Fetch next index
            $a2wPageIndexTmp = $a2wPagePar->getNextIndex();
        }

        #---- $V104 Change
        #---- Moved following VSNR protocol writing after looping
        #---- in order to protocol both Document & Page level
        #---- VSNR index
        #---- Protocol I_VNR of documents that have to be archived
        #if ( $svRetTmp != $SKIP &&
        #     $I_VNR ne "" ){
        #    protocolVNR( $I_VNR );
        #}
    }

    #---- $V109 Change
    if ( $svRetTmp != $SKIP ){
        #---- $V104 Change
        #---- Protocol I_VNR of documents that have to be archived
        if ( $I_VNR ne "" ){
            protocolVNR( $I_VNR );
        }

        #---- Remove OMR (Optical Mark Recognition) lines
        removeOMRLines( $a2wPagePar );
    }

    return $svRetTmp;
}

#-----------------------------------------------------------------------
# FinalizePage for each page
#-----------------------------------------------------------------------
sub finalizePage{

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "finalizePage()" ); } # V113 Change
    return 0;
}

#-----------------------------------------------------------------------
# FinalizeDoc for each document
#-----------------------------------------------------------------------
sub finalizeDoc{

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "finalizeDoc()" ); } # V113 Change

    #---- Reset document page count                                                   #---- $V103 Change
    $svDocPageCount = 0;

    return 0;
}

#-----------------------------------------------------------------------
# Finalize once per process
#-----------------------------------------------------------------------
sub finalize{

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "finalize()" ); } # V113 Change

    #---- Check and close Skipped Pages protocol file
    if ( $bProtocolSkippedPages == $TRUE              # $V112 Change
         && $bProtocolSkippedPagesOpen == $TRUE ){
        #---- Close skipped pages protocol file
        close( SKIPPAGEPROTOCOLFILE );
    }

    #---- Check and close VSNR protocol file
    if ( $bProtocolVNR == $TRUE &&
         $bProtocolVNROpen == $TRUE ){
        #---- Close VNR protocol file
        close( VNRPROTOCOLFILE );
    }

    return 0;
}

#-----------------------------------------------------------------------
# Get skipped pages protocol filename
#-----------------------------------------------------------------------
sub getSkippedPagesProtocolFilename{
    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "getSkippedPagesProtocolFilename()" ); } # V113 Change

	return $sSkipPageProtocolFilename;
}

#-----------------------------------------------------------------------
# Get VSNR protocol filename
#-----------------------------------------------------------------------
sub getVSNRProtocolFilename{
    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "getVSNRProtocolFilename()" ); } # V113 Change

	return $sVSNRProtocolFilename;
}

#-----------------------------------------------------------------------
# Protocol VNR Index
#-----------------------------------------------------------------------
sub protocolVNR{
    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "protocolVNR()" ); } # V113 Change

    #---- Get parameter of protocolVNR(Par:string VNR)
    ($svVNRPar) = @_;

    #---- Check and write on VSNR protocol file
    if ( $bProtocolVNR == $TRUE &&
         $bProtocolVNROpen == $TRUE ){
        #---- Write Versicherung Nummer
        print VNRPROTOCOLFILE ("VNR=$svVNRPar\n");
    }

    #---- Is set only when ProtocolVNR wanted and AFP index found
    $I_VNR = "";                            #---- $V106 Change
}

#-----------------------------------------------------------------------
# Called when $bProtocol is TRUE for protocoling content of skipped page
#-----------------------------------------------------------------------
sub protocolSkippedPage{
    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "protocolSkippedPage()" ); } # V113 Change

    #---- Check and write on Skipped Pages protocol file
    if ( $bProtocolSkippedPages == $TRUE              # $V112 Change
         && $bProtocolSkippedPagesOpen == $TRUE ){
        #---- Write skipped page content
        $svParseIdTmp = $a2wPagePar->getParseId();

        #---- Write header (reason for skipping)
        print SKIPPAGEPROTOCOLFILE ("Reason : $sSkipReason\n");
        print SKIPPAGEPROTOCOLFILE ("PageId : $svParseIdTmp\n");
        print SKIPPAGEPROTOCOLFILE ("PageContent :\n");

        #---- Write page content
        #---- Fetch 1st Text Object
        $a2wCurTextObjTmp = $a2wPagePar->getFirstText();

        #---- Loop thru all the Text Objects
        while ( $a2wCurTextObjTmp != 0 ){
            print SKIPPAGEPROTOCOLFILE "@(" . $a2wCurTextObjTmp->getXPos() . "," . $a2wCurTextObjTmp->getYPos() . ")==>" . $a2wCurTextObjTmp->getText() . "\n";

            #---- Get the next text object
            $a2wCurTextObjTmp = $a2wPagePar->getNextText();
        } # end-while
    }

    #---- Reset values
    $sSkipReason = "";
}

#-----------------------------------------------------------------------
# Remove OMR (Optical Mark Recognition) lines
#-----------------------------------------------------------------------
sub removeOMRLines{                                   #---- $V109 Change Begin
    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "removeOMRLines()" ); } # V113 Change

    #---- Get parameter(s) of removeOMRLines:
    #
    #     1. Page reference
    #
    my $a2wPageParRefPar = $_[ 0 ];

    #---- Loop thru page line objects ----#

    #---- Fetch first line of page
    my $a2wLineTmp = $a2wPageParRefPar->getFirstLine();

    my $svXPosTmp        = 0;
    my $svYPosTmp        = 0;
    my $svWidthTmp        = 0;
    my $svLengthTmp        = 0;
    my $bHorizontalTmp    = 0;

    my $svLineInfoTmp = "";
    my $bFoundOMRLineTmp = $FALSE;
    my $svPageParseIdTmp = $a2wPageParRefPar->getParseId();
    my $svOMRLineYLimitTmp = $a2wPageParRefPar->getHeight() / 2;

    while ( $a2wLineTmp != 0 ){
        $svLineInfoTmp = "";
        $svXPosTmp        = $a2wLineTmp->getXPos();
        $svYPosTmp        = $a2wLineTmp->getYPos();
        $svWidthTmp        = $a2wLineTmp->getWidth();
        $svLengthTmp    = $a2wLineTmp->getLength();
        $bHorizontalTmp    = $a2wLineTmp->isHorizontal();

        #---- Report line info
        if ( $bLog == $TRUE ){
            $svLineInfoTmp  = "XPos="            . $svXPosTmp;
            $svLineInfoTmp .= " YPos="           . $svYPosTmp;
            $svLineInfoTmp .= " Width="          . $svWidthTmp;
            $svLineInfoTmp .= " Length="         . $svLengthTmp;
            $svLineInfoTmp .= " Color="          . $a2wLineTmp->getColor();
            $svLineInfoTmp .= " Orientation="    . ( $a2wLineTmp->isHorizontal() ? "Horizontal" : "Vertical" );
            $svLineInfoTmp .= " NegativeLength=" . ( $a2wLineTmp->isNegative() ? "yes" : "no" );

            if ( $bLog == $TRUE ){ $theLogger->logMessage( $svLineInfoTmp ); } # V113 Change
        }

        #---- Check and skip OMR line
        #---- Partial OMR line info dump
        #
        #     ...
        #     XPos=28 YPos=527 Width=4 Length=56 Color=65535 Orientation=Horizontal NegativeLength=no
        #     XPos=28 YPos=567 Width=4 Length=56 Color=65535 Orientation=Horizontal NegativeLength=no
        #     XPos=28 YPos=727 Width=4 Length=56 Color=65535 Orientation=Horizontal NegativeLength=no
        #     ...
        #
        #---- Rule to recognize OMR lines
        #
        #     1. Starts at X position in between 23 to 33 (i.e 28-5 and 28+5) and
        #     2. Width is always 4
        #     3. Line is Horizontal
        #     4. Length in between 54 to 58 (i.e 56-2 and 56+2) and
        #     5. OMR line must be drawn within top half of the page (i.e within half page height from top)
        #
        if ( ( $svXPosTmp >= 23 && $svXPosTmp <= 33 )            # 1.
             && $svWidthTmp == 4                                # 2.
             && $bHorizontalTmp == $TRUE                        # 3.
             && ( $svLengthTmp >= 54 && $svLengthTmp <= 58 )    # 4.
             && $svYPosTmp < $svOMRLineYLimitTmp ){                # 5.

            #---- Remove OMR line from output presentation
            $a2wLineTmp->remove();

            $bFoundOMRLineTmp = $TRUE;
        }

        #---- Fetch next line of page
        $a2wLineTmp = $a2wPageParRefPar->getNextLine();
    }

    #---- Report whether we removed any OMR line or not on VNR protocol file
    if ( $bFoundOMRLineTmp == $FALSE
         && $bProtocolVNR == $TRUE
         && $bProtocolVNROpen == $TRUE ){
        print VNRPROTOCOLFILE ( "Warning! No OMR line found at page $svPageParseIdTmp\n" );
    }
}                                                       #---- $V109 Change End

#-----------------------------------------------------------------------
# Don't remove the following lines !!!
#-----------------------------------------------------------------------
1;
__END__
