#-------------------------------------------------------------------------------
# a2w::request::StoreProcess:
# Module for StoreProcess request
#
#  Author  : Fa. Maas
#
#  $V100  2008-07-29  Initial release
#
#  $V101  2008-11-14  Modified StoreProcess default encoding from "UTF-8" to "ISO-8859-1"
#
#  $V102  2009-01-19  Added missing "ProcessCount" element to "ImportProcess"
#
#  $V103  2019-02-08  Migrated logging to Scripting Facility framework standard
#
#-------------------------------------------------------------------------------
package a2w::request::StoreProcess;

#-------------------------------------------------------------------------------
# Include required modules
#-------------------------------------------------------------------------------
use IO::File;       # XML writer requirement
use XML::Writer;    # XML writer

use a2w::core::log::Logger; # V103 Change

#-------------------------------------------------------------------------------
# Constructor
#-------------------------------------------------------------------------------
sub new{

    my $proto = shift;
    my $class = ref( $proto ) || $proto;

    #---- Define boolean values
    $TRUE  = 1;    # TRUE  boolean value
    $FALSE = 0;    # FALSE boolean value

    my $this = {
        'sFilename'     => "",        # Filename to serialize Store Process
        'storeProcess'  => {}         # Store process BO
    };

    bless( $this, $class );

    #---- Get logger # V103 Change
    our $theLogger = a2w::core::log::Logger->getSingleton();
    our $bLog = $theLogger->isRegistered( __PACKAGE__ );

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "new()" ); }

    return $this;
}

#-------------------------------------------------------------------------------
# Destructor
#-------------------------------------------------------------------------------
sub DESTROY{

    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "DESTROY()" ); } # V103 Change
}

#-------------------------------------------------------------------------------
# Mutator(s)
#-------------------------------------------------------------------------------
#===============================================================================
# Set filename
#
# Parameter(s)
# 1. Filename
#
#===============================================================================
sub setFilename{

    my $this = shift;

    #---- Get parameter(s)
    #     1. Filename (string)
    #
    my $sFilenamePar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "setFilename( $sFilenamePar )" ); } # V103 Change

    $this->{ 'sFilename' } = $sFilenamePar;
}

#-------------------------------------------------------------------------------
# Accessor(s)
#-------------------------------------------------------------------------------
#===============================================================================
# Get filename
#===============================================================================
sub getFilename{

    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "getFilename()" ); } # V103 Change

    return $this->{ 'sFilename' };
}

#-------------------------------------------------------------------------------
# Worker(s)
#-------------------------------------------------------------------------------
#===============================================================================
# Set session context
#
# Parameter(s)
# 1. HttpSessionId
#    "Process Id" of current A2W process in HEX format, not prefixed with ZERO
#
# 2. SubSessionId
#    Always "1"
#
# 3. BackEndVersion (optional)
#    createSession->BeckEndVersion
#
#===============================================================================
sub setSessionContext{

    my $this = shift;

    # Fetch parameter(s)
    #
    # 1. HttpSessionId (string)
    # 2. SubSessionId (int)
    # 3. BackEndVersion (int)
    #
    my $sHttpSessionIdPar = shift;
    my $iSubSessionIdPar = shift;
    my $iBackEndVersionPar = undef;
    if ( @_ ){
        $iBackEndVersionPar = shift;
    }

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "setSessionContext( $sHttpSessionIdPar, $iSubSessionIdPar, $iBackEndVersionPar )" ); } # V103 Change

    # Fill in session context
    $this->{ 'storeProcess' }
          ->{ 'SessionContext' }
           ->{ 'HttpSessionId' } = $sHttpSessionIdPar;
    $this->{ 'storeProcess' }
          ->{ 'SessionContext' }
           ->{ 'SubSessionId' } = $iSubSessionIdPar;
    $this->{ 'storeProcess' }
          ->{ 'SessionContext' }
           ->{ 'BackEndVersion' } = $iBackEndVersionPar;
}

#===============================================================================
# Set request id
#
# Parameter
# 1. RequestId
#    Epoch time (time elapsed since 1.1.1970) in milliseconds, value rounded off to 13 digits and prefixed with ZERO
#
#===============================================================================
sub setRequestId{

    my $this = shift;

    # Fetch parameter
    #
    # 1. RequestId (string)
    #
    my $sRequestIdPar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "setRequestId( $sRequestIdPar )" ); } # V103 Change

    # Fill in request id
    $this->{ 'storeProcess' }
          ->{ 'RequestId' } = $sRequestIdPar;
}

#===============================================================================
# Set transaction
#
# Parameter(s)
# 1. TransactionID
#    processSpool->RequestId
#
# 2. TransactionSourceID
#    processSpool->TransactionSourceID, if this element is empty then
#    a value of "3" is considered,
#    where
#    1 mean SmartFix
#    2 mean Electronic documents
#    3 mean PrintSpool
#
# 3. Transaction progress flag
#    0 mean in transaction, default value
#    1 mean transaction begin
#    2 mean transaction end
#
#===============================================================================
sub setTransaction{

    my $this = shift;

    # Fetch parameter(s)
    #
    # 1. TransactionID (string)
    # 2. TransactionSourceID (string)
    # 3. Transaction progress flag (int)
    #
    my $sTransactionIDPar = shift;
    my $sTransactionSourceIDPar = shift;
    my $iTransactionProgressPar = 0;
    if ( @_ ){
        $iTransactionProgressPar = shift;
    }

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "setTransaction( $sTransactionIDPar, $sTransactionSourceIDPar, $iTransactionProgressPar )" ); } # V103 Change

    # Fill in transaction
    $this->{ 'storeProcess' }
          ->{ 'ImportProcess' }
           ->{ 'Transaction' }
            ->{ 'TransactionID' } = $sTransactionIDPar;
    $this->{ 'storeProcess' }
          ->{ 'ImportProcess' }
           ->{ 'Transaction' }
            ->{ 'TransactionSourceID' } = $sTransactionSourceIDPar;
    if ( $iTransactionProgressPar == 1 ){       # Transaction Begin
        $this->{ 'storeProcess' }
              ->{ 'ImportProcess' }
               ->{ 'Transaction' }
                ->{ 'TransactionBegin' } = "true";
    }
    elsif ( $iTransactionProgressPar == 2 ){    # Transaction End
        $this->{ 'storeProcess' }
              ->{ 'ImportProcess' }
               ->{ 'Transaction' }
                ->{ 'TransactionEnd' } = "true";
    }
}

#===============================================================================
# Set document consignment number
#
# Parameter
# 1. DocumentConsignmentNumber
#    Document id, value rounded off to 8 digits and prefixed with ZERO
#
#===============================================================================
sub setDocumentConsignmentNumber{

    my $this = shift;

    # Fetch parameter
    #
    # 1. DocumentConsignmentNumber (string)
    #
    my $sDocumentConsignmentNumberPar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "setDocumentConsignmentNumber( $sDocumentConsignmentNumberPar )" ); } # V103 Change

    # Fill in document consignment number
    $this->{ 'storeProcess' }
          ->{ 'ImportProcess' }
           ->{ 'DocumentConsignmentNumber' } = $sDocumentConsignmentNumberPar;
}

#===============================================================================
# Set skip process
#
# Parameter
# 1. SkipProcess
#    Always "false"
#
#===============================================================================
sub setSkipProcess{

    my $this = shift;

    # Fetch parameter
    #
    # 1. SkipProcess (string)
    #
    my $sSkipProcessPar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "setSkipProcess( $sSkipProcessPar )" ); } # V103 Change

    # Fill in skip process
    $this->{ 'storeProcess' }
          ->{ 'ImportProcess' }
           ->{ 'SkipProcess' } = $sSkipProcessPar;
}

#===============================================================================
# Set process type xml ref
#
# Parameter
# 1. ProcessTypeXmlRef
#    Value of I_PROCESS_TYPE_XML_REF index, if this doesn't exist, then value of
#    processSpool->ProcessTypeXmlRef is used
#
#===============================================================================
sub setProcessTypeXmlRef{

    my $this = shift;

    # Fetch parameter
    #
    # 1. ProcessTypeXmlRef (string)
    #
    my $sProcessTypeXmlRefPar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "setProcessTypeXmlRef( $sProcessTypeXmlRefPar )" ); } # V103 Change

    # Fill in process type xml ref
    $this->{ 'storeProcess' }
          ->{ 'ImportProcess' }
           ->{ 'ProcessTypeXmlRef' } = $sProcessTypeXmlRefPar;
}

#===============================================================================
# Set process type id
#
# Parameter
# 1. ProcessTypeID
#    Value of I_PROCESS_TYPE_ID index, if this doesn't exist, then value of
#    processSpool->ProcessTypeID is used
#
#===============================================================================
sub setProcessTypeID{

    my $this = shift;

    # Fetch parameter
    #
    # 1. ProcessTypeID (string)
    #
    my $sProcessTypeIDPar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "setProcessTypeID( $sProcessTypeIDPar )" ); } # V103 Change

    # Fill in process type id
    $this->{ 'storeProcess' }
          ->{ 'ImportProcess' }
           ->{ 'ProcessTypeID' } = $sProcessTypeIDPar;
}

#===============================================================================
# Set BoID
#
# Parameter
# 1. BoID
#    Value of I_VNR index
#
#===============================================================================
sub setBoID{

    my $this = shift;

    # Fetch parameter
    #
    # 1. BoID (string)
    #
    my $sBoIDPar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "setBoID( $sBoIDPar )" ); } # V103 Change

    # Fill in Bo Id
    $this->{ 'storeProcess' }
          ->{ 'ImportProcess' }
           ->{ 'BoID' } = $sBoIDPar;
}

#===============================================================================
# Set target work basket id
#
# Parameter
# 1. TargetWorkBasketID
#    processSpool->WorkBasketId
#
#===============================================================================
sub setTargetWorkBasketID{

    my $this = shift;

    # Fetch parameter
    #
    # 1. TargetWorkBasketID (string)
    #
    my $sTargetWorkBasketIDPar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "setTargetWorkBasketID( $sTargetWorkBasketIDPar )" ); } # V103 Change

    # Fill in target work basket id
    $this->{ 'storeProcess' }
          ->{ 'ImportProcess' }
           ->{ 'TargetWorkBasketID' } = $sTargetWorkBasketIDPar;
}

#===============================================================================
# Set priority
#
# Parameter
# 1. Priority
#    Always "4"
#
#===============================================================================
sub setPriority{

    my $this = shift;

    # Fetch parameter
    #
    # 1. Priority (int)
    #
    my $iPriorityPar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "setPriority( $iPriorityPar )" ); } # V103 Change

    # Fill in priority
    $this->{ 'storeProcess' }
          ->{ 'ImportProcess' }
           ->{ 'Priority' } = $iPriorityPar;
}

#===============================================================================
# Set section id
#
# Parameter
# 1. SectionID
#    Always "1"
#
#===============================================================================
sub setSectionID{

    my $this = shift;

    # Fetch parameter
    #
    # 1. SectionID (int)
    #
    my $iSectionIDPar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "setSectionID( $iSectionIDPar )" ); } # V103 Change

    # Fill in section id
    $this->{ 'storeProcess' }
          ->{ 'ImportProcess' }
           ->{ 'SectionID' } = $iSectionIDPar;
}

#===============================================================================
# Set in out date
#
# Parameter
# 1. InOutDate
#    Value of "I_DATUM" index, if this doesn't exist, then value of "0001-01-01" is used
#
#===============================================================================
sub setInOutDate{

    my $this = shift;

    # Fetch parameter
    #
    # 1. InOutDate (date)
    #
    my $dtInOutDatePar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "setInOutDate( $dtInOutDatePar )" ); } # V103 Change

    # Fill "in out" date
    $this->{ 'storeProcess' }
          ->{ 'ImportProcess' }
           ->{ 'InOutDate' } = $dtInOutDatePar;
}

#===============================================================================
# Add additional process attribute element
#
# "processSpool->IndexAliasList->IndexAlias->IndexLevel" with a value of "P" specifies indexes that
# has to be grouped under "storeProcess->ImportProcess->AdditionalProcessAttributeList"
#
# Parameter
# 1. DefID
#    Name of attribute
#
# 2. Value
#    Value of attribute
#
#===============================================================================
sub addAdditionalProcessAttributeElement{

    my $this = shift;

    # Fetch parameter
    #
    # 1. DefID (string)
    # 1. Value (string)
    #
    my $sDefIDPar = shift;
    my $sValuePar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "addAdditionalProcessAttributeElement( $sDefIDPar, $sValuePar )" ); } # V103 Change

    # Assert parameter(s)
    if ( $sDefIDPar eq "" && $sValuePar eq "" ){
        return;
    }

    # Fetch count of already added process attribute element
    my $iCountTmp = $this->{ 'storeProcess' }
                          ->{ 'ImportProcess' }
                           ->{ 'AdditionalProcessAttributeList' }
                            ->{ 'Count' };
    $iCountTmp++;   # increment for current element

    # Set new count on list
    $this->{ 'storeProcess' }
          ->{ 'ImportProcess' }
           ->{ 'AdditionalProcessAttributeList' }
           ->{ 'Count' } = $iCountTmp;

    # Add additional process attribute element
    $this->{ 'storeProcess' }
          ->{ 'ImportProcess' }
           ->{ 'AdditionalProcessAttributeList' }
            ->{ 'AdditionalAttribute' . $iCountTmp }
             ->{ 'AddAttribDefID' } = $sDefIDPar;
    $this->{ 'storeProcess' }
          ->{ 'ImportProcess' }
           ->{ 'AdditionalProcessAttributeList' }
            ->{ 'AdditionalAttribute' . $iCountTmp }
             ->{ 'AttribValue' } = $sValuePar;
}

#===============================================================================
# Set document count
#
# Parameter
# 1. DocumentCount
#    Always "1"
#
#===============================================================================
sub setDocumentCount{

    my $this = shift;

    # Fetch parameter
    #
    # 1. DocumentCount (int)
    #
    my $iDocumentCountPar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "setDocumentCount( $iDocumentCountPar )" ); } # V103 Change

    # Fill document
    $this->{ 'storeProcess' }
          ->{ 'ImportProcess' }
           ->{ 'DocumentCount' } = $iDocumentCountPar;
}

#===============================================================================
# Set document way
#
# Parameter
# 1. DocumentWay
#    Always "2"
#
#    where
#    1 mean incoming docs
#    2 mean outgoing docs (A2W case)
#    3 mean internal docs
#
#===============================================================================
sub setDocumentWay{

    my $this = shift;

    # Fetch parameter
    #
    # 1. DocumentWay (int)
    #
    my $iDocumentWayPar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "setDocumentWay( $iDocumentWayPar )" ); } # V103 Change

    # Fill document way
    $this->{ 'storeProcess' }
          ->{ 'ImportProcess' }
           ->{ 'ImportDocumentList' }
            ->{ 'ImportDocument' }
             ->{ 'DocumentWay' } = $iDocumentWayPar;
}

#===============================================================================
# Set document type xml ref
#
# Parameter
# 1. DocumentTypeXmlRef
#    Value of I_DOCUMENTTYPEXMLREF index, if this doesn't exist, then value of
#    processSpool->DocumentTypeXmlRef is used
#
#===============================================================================
sub setDocumentTypeXmlRef{

    my $this = shift;

    # Fetch parameter
    #
    # 1. DocumentTypeXmlRef (string)
    #
    my $sDocumentTypeXmlRefPar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "setDocumentTypeXmlRef( $sDocumentTypeXmlRefPar )" ); } # V103 Change

    # Fill document type xml ref
    $this->{ 'storeProcess' }
          ->{ 'ImportProcess' }
           ->{ 'ImportDocumentList' }
            ->{ 'ImportDocument' }
             ->{ 'DocumentTypeXmlRef' } = $sDocumentTypeXmlRefPar;
}

#===============================================================================
# Set document type id
#
# Parameter
# 1. DocumentTypeID
#    Value of I_DOCUMENTTYPEID index, if this doesn't exist, then value of
#    processSpool->DocumentTypeID is used
#
#===============================================================================
sub setDocumentTypeID{

    my $this = shift;

    # Fetch parameter
    #
    # 1. DocumentTypeID (string)
    #
    my $sDocumentTypeIDPar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "setDocumentTypeID( $sDocumentTypeIDPar )" ); } # V103 Change

    # Fill document type id
    $this->{ 'storeProcess' }
          ->{ 'ImportProcess' }
           ->{ 'ImportDocumentList' }
            ->{ 'ImportDocument' }
             ->{ 'DocumentTypeID' } = $sDocumentTypeIDPar;
}

#===============================================================================
# Set document content date
#
# Parameter
# 1. DocumentContentDate
#    Value of "I_DATUM" index
#
#===============================================================================
sub setDocumentContentDate{

    my $this = shift;

    # Fetch parameter
    #
    # 1. DocumentContentDate (date)
    #
    my $dtDocumentContentDatePar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "setDocumentContentDate( $dtDocumentContentDatePar )" ); } # V103 Change

    # Fill document content date
    $this->{ 'storeProcess' }
          ->{ 'ImportProcess' }
           ->{ 'ImportDocumentList' }
            ->{ 'ImportDocument' }
             ->{ 'DocumentContentDate' } = $dtDocumentContentDatePar;
}

#===============================================================================
# Set document subject
#
# Parameter
# 1. DocumentSubject
#    Value of "I_BETREFF" index
#
#===============================================================================
sub setDocumentSubject{

    my $this = shift;

    # Fetch parameter
    #
    # 1. DocumentSubject (string)
    #
    my $sDocumentSubjectPar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "setDocumentSubject( $sDocumentSubjectPar )" ); } # V103 Change

    # Fill document subject
    $this->{ 'storeProcess' }
          ->{ 'ImportProcess' }
           ->{ 'ImportDocumentList' }
            ->{ 'ImportDocument' }
             ->{ 'DocumentSubject' } = $sDocumentSubjectPar;
}

#===============================================================================
# Set document source xml ref
#
# Parameter
# 1. DocumentSourceXmlRef
#    processSpool->DocumentSourceXMLRef
#
#===============================================================================
sub setDocumentSourceXmlRef{

    my $this = shift;

    # Fetch parameter
    #
    # 1. DocumentSourceXmlRef (string)
    #
    my $sDocumentSourceXmlRefPar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "setDocumentSourceXmlRef( $sDocumentSourceXmlRefPar )" ); } # V103 Change

    # Fill document source xml ref
    $this->{ 'storeProcess' }
          ->{ 'ImportProcess' }
           ->{ 'ImportDocumentList' }
            ->{ 'ImportDocument' }
             ->{ 'DocumentSourceXmlRef' } = $sDocumentSourceXmlRefPar;
}

#===============================================================================
# Set document source id
#
# Parameter
# 1. DocumentSourceID
#    processSpool->DocumentSourceID
#
#===============================================================================
sub setDocumentSourceID{

    my $this = shift;

    # Fetch parameter
    #
    # 1. DocumentSourceID (string)
    #
    my $sDocumentSourceIDPar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "setDocumentSourceID( $sDocumentSourceIDPar )" ); } # V103 Change

    # Fill document source id
    $this->{ 'storeProcess' }
          ->{ 'ImportProcess' }
           ->{ 'ImportDocumentList' }
            ->{ 'ImportDocument' }
             ->{ 'DocumentSourceID' } = $sDocumentSourceIDPar;
}

#===============================================================================
# Set page count
#
# Parameter
# 1. PageCount
#    Number of pages in output document
#
#===============================================================================
sub setPageCount{

    my $this = shift;

    # Fetch parameter
    #
    # 1. PageCount (int)
    #
    my $iPageCountPar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "setPageCount( $iPageCountPar )" ); } # V103 Change

    # Fill page count
    $this->{ 'storeProcess' }
          ->{ 'ImportProcess' }
           ->{ 'ImportDocumentList' }
            ->{ 'ImportDocument' }
             ->{ 'PageCount' } = $iPageCountPar;
}

#===============================================================================
# Set preferred object file format
#
# Parameter
# 1. PreferredObjectFileFormat
#    Output document format as string like PDF, TIFF etc
#
#===============================================================================
sub setPreferredObjectFileFormat{

    my $this = shift;

    # Fetch parameter
    #
    # 1. PreferredObjectFileFormat (string)
    #
    my $sPreferredObjectFileFormatPar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "setPreferredObjectFileFormat( $sPreferredObjectFileFormatPar )" ); } # V103 Change

    # Fill preferred object file format
    $this->{ 'storeProcess' }
          ->{ 'ImportProcess' }
           ->{ 'ImportDocumentList' }
            ->{ 'ImportDocument' }
             ->{ 'ObjectContainerList' }
              ->{ 'ObjectContainer' }
               ->{ 'PreferredObjectFileFormat' } = $sPreferredObjectFileFormatPar;
}

#===============================================================================
# Set object id
#
# Parameter
# 1. ObjectID
#    Value filled based on "FilenamePattern" attribute of output document logical location.
#    For HN, "FilenamePattern" has been set with '"%s%s0008",MD5ID,TIMESTAMP' value, which mean
#    the object id value must be as given below
#    <MD5 Id of output document data><Current time in YYMMDDhhmmss format>0008
#
#===============================================================================
sub setObjectID{

    my $this = shift;

    # Fetch parameter
    #
    # 1. ObjectID (string)
    #
    my $sObjectIDPar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "setObjectID( $sObjectIDPar )" ); } # V103 Change

    # Fill object id
    $this->{ 'storeProcess' }
          ->{ 'ImportProcess' }
           ->{ 'ImportDocumentList' }
            ->{ 'ImportDocument' }
             ->{ 'ObjectContainerList' }
              ->{ 'ObjectContainer' }
               ->{ 'ObjectRepresentationInfoList' }
                ->{ 'ObjectRepresentationInfo' }
                 ->{ 'ObjectID' } = $sObjectIDPar;
}

#===============================================================================
# Set object representation format
#
# Parameter
# 1. ObjectRepresentationFormat
#    Output document format as string like PDF, TIFF
#
#===============================================================================
sub setObjectRepresentationFormat{

    my $this = shift;

    # Fetch parameter
    #
    # 1. ObjectRepresentationFormat (string)
    #
    my $sObjectRepresentationFormatPar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "setObjectRepresentationFormat( $sObjectRepresentationFormatPar )" ); } # V103 Change

    # Fill object representation format
    $this->{ 'storeProcess' }
          ->{ 'ImportProcess' }
           ->{ 'ImportDocumentList' }
            ->{ 'ImportDocument' }
             ->{ 'ObjectContainerList' }
              ->{ 'ObjectContainer' }
               ->{ 'ObjectRepresentationInfoList' }
                ->{ 'ObjectRepresentationInfo' }
                 ->{ 'ObjectRepresentationFormat' } = $sObjectRepresentationFormatPar;
}

#===============================================================================
# Set original format
#
# Parameter
# 1. OriginalFormat
#    Always "true"
#
#===============================================================================
sub setOriginalFormat{

    my $this = shift;

    # Fetch parameter
    #
    # 1. OriginalFormat (string)
    #
    my $sOriginalFormatPar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "setOriginalFormat( $sOriginalFormatPar )" ); } # V103 Change

    # Fill original format
    $this->{ 'storeProcess' }
          ->{ 'ImportProcess' }
           ->{ 'ImportDocumentList' }
            ->{ 'ImportDocument' }
             ->{ 'ObjectContainerList' }
              ->{ 'ObjectContainer' }
               ->{ 'ObjectRepresentationInfoList' }
                ->{ 'ObjectRepresentationInfo' }
                 ->{ 'OriginalFormat' } = $sOriginalFormatPar;
}

#===============================================================================
# Set sequence number
#
# Parameter
# 1. SequenceNumber
#    Always "1"
#
#===============================================================================
sub setSequenceNumber{

    my $this = shift;

    # Fetch parameter
    #
    # 1. SequenceNumber (int)
    #
    my $iSequenceNumberPar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "setSequenceNumber( $iSequenceNumberPar )" ); } # V103 Change

    # Fill sequence number
    $this->{ 'storeProcess' }
          ->{ 'ImportProcess' }
           ->{ 'ImportDocumentList' }
            ->{ 'ImportDocument' }
             ->{ 'ObjectContainerList' }
              ->{ 'ObjectContainer' }
               ->{ 'ObjectRepresentationInfoList' }
                ->{ 'ObjectRepresentationInfo' }
                 ->{ 'SequenceNumber' } = $iSequenceNumberPar;
}

#===============================================================================
# Add additional document attribute element
#
# a. "processSpool->IndexAliasList->IndexAlias->IndexLevel" with a value of "D" specifies indexes that has to be grouped under
#    "storeProcess->ImportProcess->ImportDocumentList->ImportDocument->AdditionalDocumentAttributeList" and also above elements
#    filling details
# b. All indexes that has not been grouped under "storeProcess->ImportProcess->AdditionalProcessAttributeList"
#
# Parameter
# 1. DefID
#    Name of attribute
#
# 2. Value
#    Value of attribute
#
#===============================================================================
sub addAdditionalDocumentAttributeElement{

    my $this = shift;

    # Fetch parameter
    #
    # 1. DefID (string)
    # 1. Value (string)
    #
    my $sDefIDPar = shift;
    my $sValuePar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "addAdditionalDocumentAttributeElement( $sDefIDPar, $sValuePar )" ); } # V103 Change

    # Assert parameter(s)
    if ( $sDefIDPar eq "" && $sValuePar eq "" ){
        return;
    }

    # Fetch count of already added document attribute element
    my $iCountTmp = $this->{ 'storeProcess' }
                          ->{ 'ImportProcess' }
                           ->{ 'ImportDocumentList' }
                            ->{ 'ImportDocument' }
                             ->{ 'AdditionalDocumentAttributeList' }
                              ->{ 'Count' };
    $iCountTmp++;   # increment for current element

    # Set new count on list
    $this->{ 'storeProcess' }
          ->{ 'ImportProcess' }
           ->{ 'ImportDocumentList' }
            ->{ 'ImportDocument' }
             ->{ 'AdditionalDocumentAttributeList' }
              ->{ 'Count' } = $iCountTmp;

    # Add additional document attribute element
    $this->{ 'storeProcess' }
          ->{ 'ImportProcess' }
           ->{ 'ImportDocumentList' }
            ->{ 'ImportDocument' }
             ->{ 'AdditionalDocumentAttributeList' }
              ->{ 'AdditionalAttribute' . $iCountTmp }
               ->{ 'AddAttribDefID' } = $sDefIDPar;
    $this->{ 'storeProcess' }
          ->{ 'ImportProcess' }
           ->{ 'ImportDocumentList' }
            ->{ 'ImportDocument' }
             ->{ 'AdditionalDocumentAttributeList' }
              ->{ 'AdditionalAttribute' . $iCountTmp }
               ->{ 'AttribValue' } = $sValuePar;
}

#===============================================================================
# Set spool filename
#
# Parameter
# 1. SpoolFileName
#    Input spool filename (without path)
#
#===============================================================================
sub setSpoolFileName{

    my $this = shift;

    # Fetch parameter
    #
    # 1. SpoolFileName (string)
    #
    my $sSpoolFileNamePar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "setSpoolFileName( $sSpoolFileNamePar )" ); } # V103 Change

    # Fill in spool filename
    $this->{ 'storeProcess' }
          ->{ 'ImportProcess' }
           ->{ 'PrintSpoolInfo' }
            ->{ 'SpoolFileName' } = $sSpoolFileNamePar;
}

#===============================================================================
# Set resource filename
#
# Parameter
# 1. ResourceFileName
#    Input resource group filename (without path)
#
#===============================================================================
sub setResourceFileName{

    my $this = shift;

    # Fetch parameter
    #
    # 1. ResourceFileName (string)
    #
    my $sResourceFileNamePar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "setResourceFileName( $sResourceFileNamePar )" ); } # V103 Change

    # Fill in resource filename
    $this->{ 'storeProcess' }
          ->{ 'ImportProcess' }
           ->{ 'PrintSpoolInfo' }
            ->{ 'ResourceFileName' } = $sResourceFileNamePar;
}

#===============================================================================
# Set page from
#
# Parameter
# 1. PageFrom
#    Page number as in input spool from which the output document begin
#
#===============================================================================
sub setPageFrom{

    my $this = shift;

    # Fetch parameter
    #
    # 1. PageFrom (int)
    #
    my $iPageFromPar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "setPageFrom( $iPageFromPar )" ); } # V103 Change

    # Fill in page from
    $this->{ 'storeProcess' }
          ->{ 'ImportProcess' }
           ->{ 'PrintSpoolInfo' }
            ->{ 'PageFrom' } = $iPageFromPar;
}

#===============================================================================
# Set page to
#
# Parameter
# 1. PageTo
#    Page number as in input spool at which the output document end
#
#===============================================================================
sub setPageTo{

    my $this = shift;

    # Fetch parameter
    #
    # 1. PageTo (int)
    #
    my $iPageToPar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "setPageTo( $iPageToPar )" ); } # V103 Change

    # Fill in page to
    $this->{ 'storeProcess' }
          ->{ 'ImportProcess' }
           ->{ 'PrintSpoolInfo' }
            ->{ 'PageTo' } = $iPageToPar;
}

# $V102 Begin
#===============================================================================
# Set process count
#
# Parameter
# 1. ProcessCount
#    Number of store process request generated for given spool
#    (aka Output document count). Set only on last "StoreProcess" request
#
#===============================================================================
sub setProcessCount{

    my $this = shift;

    # Fetch parameter
    #
    # 1. ProcessCount (int)
    #
    my $iProcessCountPar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "setProcessCount( $iProcessCountPar )" ); } # V103 Change

    # Fill in process count
    $this->{ 'storeProcess' }
          ->{ 'ImportProcess' }
           ->{ 'ProcessCount' } = $iProcessCountPar;
}
# $V102 End

#===============================================================================
# Serialize store process
#===============================================================================
sub serialize{
    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "serialize()" ); } # V103 Change

    # Assert filename
    if ( $this->{ 'sFilename' } eq "" ){
        return -1;
    }

    # Open file handle
    my $fhStoreProcessTmp = new IO::File( ">" . $this->{ 'sFilename' } );
    if ( $fhStoreProcessTmp == undef ){
        return -2;
    }

	# Create XML writer object
	my $xwStoreProcessTmp = new XML::Writer( OUTPUT        => $fhStoreProcessTmp
                                             , DATA_INDENT => 1
                                             #, DATA_MODE   => 1
                                             #, NAMESPACES  => 1
                                             #, UNSAFE      => 1
                                           );
    if ( $xwStoreProcessTmp == undef ){
        return -3;
    }

    # $V101 Begin
    #---- Modified encoding from UTF-8 to ISO-8859-1
    # Set xml encoding
    #$xwStoreProcessTmp->xmlDecl( "UTF-8" );
    $xwStoreProcessTmp->xmlDecl( "ISO-8859-1" );
    # $V101 End

	# Start "SOAP-ENV:Envelope" element
    $xwStoreProcessTmp->startTag( 'SOAP-ENV:Envelope',
                                  # Element attributes
                                  'xmlns:SOAP-ENV' => 'http://schemas.xmlsoap.org/soap/envelope/',
                                  'xmlns:SOAP-ENC' => 'http://schemas.xmlsoap.org/soap/encoding/',
                                  'xmlns:xsi' => 'http://www.w3.org/1999/XMLSchema-instance',
                                  'xmlns:xsd' => 'http://www.w3.org/1999/XMLSchema',
                                  'xmlns:ns' => 'http://maas.de/h2wdms/common/bo/',
                                  'SOAP-ENV:encodingStyle' => 'http://schemas.xmlsoap.org/soap/encoding/'
                                );

    # Turn on data mode after first element alone
    # else a empty line will be inserted between
    # xml header line and first element line
    $xwStoreProcessTmp->setDataMode( 1 );

	# Start "SOAP-ENV:Body" element
    $xwStoreProcessTmp->startTag( 'SOAP-ENV:Body' );

	# Start "ns:storeProcess" element
    $xwStoreProcessTmp->startTag( 'ns:storeProcess' );

    # Serialize session context
    $this->_serializeSessionContext( $xwStoreProcessTmp );

    # Serialize request id
    if ( $this->{ 'storeProcess' }->{ 'RequestId' } ne "" ){
        $xwStoreProcessTmp->dataElement( 'RequestId',
                                         $this->{ 'storeProcess' }->{ 'RequestId' }
                                       );
    }

	# Start "ImportProcess" element
    $xwStoreProcessTmp->startTag( 'ImportProcess' );

    # Serialize transaction
    $this->_serializeTransaction( $xwStoreProcessTmp );

    # Serialize document consignment number
    if ( $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'DocumentConsignmentNumber' } ne "" ){
        $xwStoreProcessTmp->dataElement( 'DocumentConsignmentNumber',
                                         $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'DocumentConsignmentNumber' }
                                       );
    }

    # Serialize skip process
    if ( $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'SkipProcess' } ne "" ){
        $xwStoreProcessTmp->dataElement( 'SkipProcess',
                                         $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'SkipProcess' }
                                       );
    }

    # Serialize process type xml ref
    if ( $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'ProcessTypeXmlRef' } ne "" ){
        $xwStoreProcessTmp->dataElement( 'ProcessTypeXmlRef',
                                         $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'ProcessTypeXmlRef' }
                                       );
    }

    # Serialize process type id
    if ( $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'ProcessTypeID' } ne "" ){
        $xwStoreProcessTmp->dataElement( 'ProcessTypeID',
                                         $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'ProcessTypeID' }
                                       );
    }

    # Serialize Bo id
    if ( $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'BoID' } ne "" ){
        $xwStoreProcessTmp->dataElement( 'BoID',
                                         $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'BoID' }
                                       );
    }

    # Serialize target work basket id
    if ( $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'TargetWorkBasketID' } ne "" ){
        $xwStoreProcessTmp->dataElement( 'TargetWorkBasketID',
                                         $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'TargetWorkBasketID' }
                                       );
    }

    # Serialize priority
    if ( $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'Priority' } ne "" ){
        $xwStoreProcessTmp->dataElement( 'Priority',
                                         $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'Priority' }
                                       );
    }

    # Serialize section id
    if ( $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'SectionID' } ne "" ){
        $xwStoreProcessTmp->dataElement( 'SectionID',
                                         $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'SectionID' }
                                       );
    }

    # Serialize in out date
    if ( $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'InOutDate' } ne "" ){
        $xwStoreProcessTmp->dataElement( 'InOutDate',
                                         $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'InOutDate' }
                                       );
    }

    # Serialize additional process attribute list
    $this->_serializeAdditionalProcessAttributeList( $xwStoreProcessTmp );

    # Serialize document count
    if ( $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'DocumentCount' } ne "" ){
        $xwStoreProcessTmp->dataElement( 'DocumentCount',
                                         $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'DocumentCount' }
                                       );
    }

	# Start "ImportDocumentList" element
    $xwStoreProcessTmp->startTag( 'ImportDocumentList' );

	# Start "ImportDocument" element
    $xwStoreProcessTmp->startTag( 'ImportDocument' );

    # Serialize document way
    if ( $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'ImportDocumentList' }->{ 'ImportDocument' }->{ 'DocumentWay' } ne "" ){
        $xwStoreProcessTmp->dataElement( 'DocumentWay',
                                         $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'ImportDocumentList' }->{ 'ImportDocument' }->{ 'DocumentWay' }
                                       );
    }

    # Serialize document type xml ref
    if ( $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'ImportDocumentList' }->{ 'ImportDocument' }->{ 'DocumentTypeXmlRef' } ne "" ){
        $xwStoreProcessTmp->dataElement( 'DocumentTypeXmlRef',
                                         $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'ImportDocumentList' }->{ 'ImportDocument' }->{ 'DocumentTypeXmlRef' }
                                       );
    }

    # Serialize document type id
    if ( $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'ImportDocumentList' }->{ 'ImportDocument' }->{ 'DocumentTypeID' } ne "" ){
        $xwStoreProcessTmp->dataElement( 'DocumentTypeID',
                                         $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'ImportDocumentList' }->{ 'ImportDocument' }->{ 'DocumentTypeID' }
                                       );
    }

    # Serialize document content date
    if ( $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'ImportDocumentList' }->{ 'ImportDocument' }->{ 'DocumentContentDate' } ne "" ){
        $xwStoreProcessTmp->dataElement( 'DocumentContentDate',
                                         $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'ImportDocumentList' }->{ 'ImportDocument' }->{ 'DocumentContentDate' }
                                       );
    }

    # Serialize document subject
    if ( $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'ImportDocumentList' }->{ 'ImportDocument' }->{ 'DocumentSubject' } ne "" ){
        $xwStoreProcessTmp->dataElement( 'DocumentSubject',
                                         $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'ImportDocumentList' }->{ 'ImportDocument' }->{ 'DocumentSubject' }
                                       );
    }

    # Serialize document source xml ref
    if ( $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'ImportDocumentList' }->{ 'ImportDocument' }->{ 'DocumentSourceXmlRef' } ne "" ){
        $xwStoreProcessTmp->dataElement( 'DocumentSourceXmlRef',
                                         $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'ImportDocumentList' }->{ 'ImportDocument' }->{ 'DocumentSourceXmlRef' }
                                       );
    }

    # Serialize document source id
    if ( $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'ImportDocumentList' }->{ 'ImportDocument' }->{ 'DocumentSourceID' } ne "" ){
        $xwStoreProcessTmp->dataElement( 'DocumentSourceID',
                                         $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'ImportDocumentList' }->{ 'ImportDocument' }->{ 'DocumentSourceID' }
                                       );
    }

    # Serialize page count
    if ( $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'ImportDocumentList' }->{ 'ImportDocument' }->{ 'PageCount' } ne "" ){
        $xwStoreProcessTmp->dataElement( 'PageCount',
                                         $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'ImportDocumentList' }->{ 'ImportDocument' }->{ 'PageCount' }
                                       );
    }

    # Serialize object container list
    $this->_serializeObjectContainerList( $xwStoreProcessTmp );

    # Serialize additional document attribute list
    $this->_serializeAdditionalDocumentAttributeList( $xwStoreProcessTmp );

	# End "ImportDocument" element
	$xwStoreProcessTmp->endTag();

	# End "ImportDocumentList" element
	$xwStoreProcessTmp->endTag();

    # Serialize print spool info
    $this->_serializePrintSpoolInfo( $xwStoreProcessTmp );

    # $V102 Begin
    # Serialize process count
    if ( $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'ProcessCount' } ne "" ){
        $xwStoreProcessTmp->dataElement( 'ProcessCount',
                                         $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'ProcessCount' }
                                       );
    }
    # $V102 End

	# End "ImportProcess" element
	$xwStoreProcessTmp->endTag();

	# End "ns:storeProcess" element
	$xwStoreProcessTmp->endTag();

	# End "SOAP-ENV:Body" element
	$xwStoreProcessTmp->endTag();

	# End "SOAP-ENV:Envelope" element
	$xwStoreProcessTmp->endTag();

	# Flust xml buffer into file
	$xwStoreProcessTmp->end();

	# Close file handle
	$fhStoreProcessTmp->close();

    # Reset references
    $xwStoreProcessTmp = undef;
    $fhStoreProcessTmp = undef;
}

#===============================================================================
# Serialize session context
#
# Parameter
# 1. XML Writer handle
#
#===============================================================================
sub _serializeSessionContext{
    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "_serializeSessionContext()" ); } # V103 Change

    # Fetch parameter
    #
    # 1. XML writer handle (XML::Writer)
    #
    my $xwSPPar = shift;

    if ( $xwSPPar == undef
         || ref( $xwSPPar ) ne "XML::Writer"
       ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Invalid XML::Writer handle to write session context" ); } # V103 Change
        return;
    }

    # Fetch  session context
    my $spSessionContextTmp = $this->{ 'storeProcess' }->{ 'SessionContext' };
    if ( $spSessionContextTmp == undef
         || $spSessionContextTmp eq ""
       ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Empty session context, nothing to write" ); } # V103 Change
        return;
    }

    #---- Write session context complex element ----#
	# Start "SessionContext" element
    $xwSPPar->startTag( 'SessionContext' );

    # Add "HttpSessionId" element
    if ( $spSessionContextTmp->{ 'HttpSessionId' } ne "" ){
        $xwSPPar->dataElement( 'HttpSessionId',
                               $spSessionContextTmp->{ 'HttpSessionId' }
                             );
    }

    # Add "SubSessionId" element
    if ( $spSessionContextTmp->{ 'SubSessionId' } ne "" ){
        $xwSPPar->dataElement( 'SubSessionId',
                               $spSessionContextTmp->{ 'SubSessionId' }
                             );
    }

    # Add "BackEndVersion" element
    if ( $spSessionContextTmp->{ 'BackEndVersion' } ne "" ){
        $xwSPPar->dataElement( 'BackEndVersion',
                               $spSessionContextTmp->{ 'BackEndVersion' }
                             );
    }

	# End "SessionContext" element
	$xwSPPar->endTag();
}

#===============================================================================
# Serialize transaction
#
# Parameter
# 1. XML Writer handle
#
#===============================================================================
sub _serializeTransaction{
    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "_serializeTransaction()" ); } # V103 Change

    # Fetch parameter
    #
    # 1. XML writer handle (XML::Writer)
    #
    my $xwSPPar = shift;

    if ( $xwSPPar == undef
         || ref( $xwSPPar ) ne "XML::Writer"
       ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Invalid XML::Writer handle to write transaction" ); } # V103 Change
        return;
    }

    # Fetch transaction
    my $spTransactionTmp = $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'Transaction' };
    if ( $spTransactionTmp == undef
         || $spTransactionTmp eq ""
       ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Empty transaction, nothing to write" ); } # V103 Change
        return;
    }

    #---- Write transaction complex element ----#
	# Start "Transaction" element
    $xwSPPar->startTag( 'Transaction' );

    # Add "TransactionID" element
    if ( $spTransactionTmp->{ 'TransactionID' } ne "" ){
        $xwSPPar->dataElement( 'TransactionID',
                               $spTransactionTmp->{ 'TransactionID' }
                             );
    }

    # Add "TransactionSourceID" element
    if ( $spTransactionTmp->{ 'TransactionSourceID' } ne "" ){
        $xwSPPar->dataElement( 'TransactionSourceID',
                               $spTransactionTmp->{ 'TransactionSourceID' }
                             );
    }

    # Add "TransactionBegin" element
    if ( $spTransactionTmp->{ 'TransactionBegin' } ne "" ){
        $xwSPPar->dataElement( 'TransactionBegin',
                               $spTransactionTmp->{ 'TransactionBegin' }
                             );
    }

    # Add "TransactionEnd" element
    if ( $spTransactionTmp->{ 'TransactionEnd' } ne "" ){
        $xwSPPar->dataElement( 'TransactionEnd',
                               $spTransactionTmp->{ 'TransactionEnd' }
                             );
    }

	# End "Transaction" element
	$xwSPPar->endTag();
}

#===============================================================================
# Serialize additional process attribute list
#
# Parameter
# 1. XML Writer handle
#
#===============================================================================
sub _serializeAdditionalProcessAttributeList{
    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "_serializeAdditionalProcessAttributeList()" ); } # V103 Change

    # Fetch parameter
    #
    # 1. XML writer handle (XML::Writer)
    #
    my $xwSPPar = shift;

    if ( $xwSPPar == undef
         || ref( $xwSPPar ) ne "XML::Writer"
       ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Invalid XML::Writer handle to write additional process attribute list" ); } # V103 Change
        return;
    }

    # Fetch additional process attribute list
    my $spAPALTmp = $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'AdditionalProcessAttributeList' };
    if ( $spAPALTmp == undef
         || $spAPALTmp eq ""
       ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Empty additional process attribute list, nothing to write" ); } # V103 Change
        return;
    }

    # Fetch additional process attribute elements count
    my $iElementCountTmp = $spAPALTmp->{ 'Count' };
    if ( $iElementCountTmp <= 0 ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "No additional process attribute element to write" ); } # V103 Change
        return;
    }

    #---- Write additional process attribute list ----#
	# Start "AdditionalProcessAttributeList" element
    $xwSPPar->startTag( 'AdditionalProcessAttributeList' );

    # Loop through and write elements
    for ( my $i = 1; $i <= $iElementCountTmp; $i++ ){
        # Start "AdditionalAttribute" element
        $xwSPPar->startTag( 'AdditionalAttribute' );

        # Add "AddAttribDefID" element
        $xwSPPar->dataElement( 'AddAttribDefID',
                               $spAPALTmp->{ 'AdditionalAttribute' . $i }->{ 'AddAttribDefID' }
                             );

        # Add "AttribValue" element
        $xwSPPar->dataElement( 'AttribValue',
                               $spAPALTmp->{ 'AdditionalAttribute' . $i }->{ 'AttribValue' }
                             );

        # End "AdditionalAttribute" element
        $xwSPPar->endTag();
    }

	# End "AdditionalProcessAttributeList" element
	$xwSPPar->endTag();
}

#===============================================================================
# Serialize object container list
#
# Parameter
# 1. XML Writer handle
#
#===============================================================================
sub _serializeObjectContainerList{
    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "_serializeObjectContainerList()" ); } # V103 Change

    # Fetch parameter
    #
    # 1. XML writer handle (XML::Writer)
    #
    my $xwSPPar = shift;

    if ( $xwSPPar == undef
         || ref( $xwSPPar ) ne "XML::Writer"
       ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Invalid XML::Writer handle to write object container list" ); } # V101 Change # V103 Change
        return;
    }

    # Fetch object container list
    my $spOCLTmp = $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'ImportDocumentList' }->{ 'ImportDocument' }->{ 'ObjectContainerList' };
    if ( $spOCLTmp == undef
         || $spOCLTmp eq ""
       ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Empty object container list, nothing to write" ); } # V103 Change
        return;
    }

    # Fetch object container
    my $spOCTmp = $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'ImportDocumentList' }->{ 'ImportDocument' }->{ 'ObjectContainerList' }->{ 'ObjectContainer' };
    if ( $spOCTmp == undef
         || $spOCTmp eq ""
       ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Empty object container, nothing to write" ); } # V103 Change
        return;
    }

    #---- Write object container list ----#
	# Start "ObjectContainerList" element
    $xwSPPar->startTag( 'ObjectContainerList' );

	# Start "ObjectContainer" element
    $xwSPPar->startTag( 'ObjectContainer' );

    # Add "PreferredObjectFileFormat" element
    if ( $spOCTmp->{ 'PreferredObjectFileFormat' } ne "" ){
        $xwSPPar->dataElement( 'PreferredObjectFileFormat',
                               $spOCTmp->{ 'PreferredObjectFileFormat' }
                             );
    }

    # Fetch object representation info list
    my $spORILTmp = $spOCTmp->{ 'ObjectRepresentationInfoList' };
    if ( $spORILTmp == undef
         || $spORILTmp eq ""
       ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Empty object representation info list, skipped writing it" ); } # V103 Change

    	# End "ObjectContainer" element
    	$xwSPPar->endTag();

    	# End "ObjectContainerList" element
    	$xwSPPar->endTag();

        return;
    }

    # Fetch object representation info
    my $spORITmp = $spORILTmp->{ 'ObjectRepresentationInfo' };
    if ( $spORITmp == undef
         || $spORITmp eq ""
       ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Empty object representation info, skipped writing object representation info list" ); } # V103 Change

    	# End "ObjectContainer" element
    	$xwSPPar->endTag();

    	# End "ObjectContainerList" element
    	$xwSPPar->endTag();

        return;
    }

	# Start "ObjectRepresentationInfoList" element
    $xwSPPar->startTag( 'ObjectRepresentationInfoList' );

	# Start "ObjectRepresentationInfo" element
    $xwSPPar->startTag( 'ObjectRepresentationInfo' );

    # Add "ObjectID" element
    if ( $spORITmp->{ 'ObjectID' } ne "" ){
        $xwSPPar->dataElement( 'ObjectID',
                               $spORITmp->{ 'ObjectID' }
                             );
    }

    # Add "ObjectRepresentationFormat" element
    if ( $spORITmp->{ 'ObjectRepresentationFormat' } ne "" ){
        $xwSPPar->dataElement( 'ObjectRepresentationFormat',
                               $spORITmp->{ 'ObjectRepresentationFormat' }
                             );
    }

    # Add "OriginalFormat" element
    if ( $spORITmp->{ 'OriginalFormat' } ne "" ){
        $xwSPPar->dataElement( 'OriginalFormat',
                               $spORITmp->{ 'OriginalFormat' }
                             );
    }

    # Add "SequenceNumber" element
    if ( $spORITmp->{ 'SequenceNumber' } ne "" ){
        $xwSPPar->dataElement( 'SequenceNumber',
                               $spORITmp->{ 'SequenceNumber' }
                             );
    }

	# End "ObjectRepresentationInfo" element
	$xwSPPar->endTag();

	# End "ObjectRepresentationInfoList" element
	$xwSPPar->endTag();

	# End "ObjectContainer" element
	$xwSPPar->endTag();

	# End "ObjectContainerList" element
	$xwSPPar->endTag();
}

#===============================================================================
# Serialize additional document attribute list
#
# Parameter
# 1. XML Writer handle
#
#===============================================================================
sub _serializeAdditionalDocumentAttributeList{
    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "_serializeAdditionalDocumentAttributeList()" ); } # V103 Change

    # Fetch parameter
    #
    # 1. XML writer handle (XML::Writer)
    #
    my $xwSPPar = shift;

    if ( $xwSPPar == undef
         || ref( $xwSPPar ) ne "XML::Writer"
       ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Invalid XML::Writer handle to write additional document attribute list" ); } # V103 Change
        return;
    }

    # Fetch additional document attribute list
    my $spADALTmp = $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'ImportDocumentList' }->{ 'ImportDocument' }->{ 'AdditionalDocumentAttributeList' };
    if ( $spADALTmp == undef
         || $spADALTmp eq ""
       ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Empty additional document attribute list, nothing to write" ); } # V103 Change
        return;
    }

    # Fetch additional document attribute elements count
    my $iElementCountTmp = $spADALTmp->{ 'Count' };
    if ( $iElementCountTmp <= 0 ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "No additional document attribute element to write" ); } # V103 Change
        return;
    }

    #---- Write additional document attribute list ----#
	# Start "AdditionalDocumentAttributeList" element
    $xwSPPar->startTag( 'AdditionalDocumentAttributeList' );

    # Loop through and write elements
    for ( my $i = 1; $i <= $iElementCountTmp; $i++ ){
        # Start "AdditionalAttribute" element
        $xwSPPar->startTag( 'AdditionalAttribute' );

        # Add "AddAttribDefID" element
        $xwSPPar->dataElement( 'AddAttribDefID',
                               $spADALTmp->{ 'AdditionalAttribute' . $i }->{ 'AddAttribDefID' }
                             );

        # Add "AttribValue" element
        $xwSPPar->dataElement( 'AttribValue',
                               $spADALTmp->{ 'AdditionalAttribute' . $i }->{ 'AttribValue' }
                             );

        # End "AdditionalAttribute" element
        $xwSPPar->endTag();
    }

	# End "AdditionalDocumentAttributeList" element
	$xwSPPar->endTag();
}

#===============================================================================
# Serialize print spool info
#
# Parameter
# 1. XML Writer handle
#
#===============================================================================
sub _serializePrintSpoolInfo{
    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "_serializePrintSpoolInfo()" ); } # V103 Change

    # Fetch parameter
    #
    # 1. XML writer handle (XML::Writer)
    #
    my $xwSPPar = shift;

    if ( $xwSPPar == undef
         || ref( $xwSPPar ) ne "XML::Writer"
       ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Invalid XML::Writer handle to write print spool info" ); } # V103 Change
        return;
    }

    # Fetch print spool info
    my $spPrintSpoolInfoTmp = $this->{ 'storeProcess' }->{ 'ImportProcess' }->{ 'PrintSpoolInfo' };
    if ( $spPrintSpoolInfoTmp == undef
         || $spPrintSpoolInfoTmp eq ""
       ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Empty print spool info, nothing to write" ); } # V103 Change
        return;
    }

    #---- Write print spool info complex element ----#
	# Start "PrintSpoolInfo" element
    $xwSPPar->startTag( 'PrintSpoolInfo' );

    # Add "SpoolFileName" element
    if ( $spPrintSpoolInfoTmp->{ 'SpoolFileName' } ne "" ){
        $xwSPPar->dataElement( 'SpoolFileName',
                               $spPrintSpoolInfoTmp->{ 'SpoolFileName' }
                             );
    }

    # Add "ResourceFileName" element
    if ( $spPrintSpoolInfoTmp->{ 'ResourceFileName' } ne "" ){
        $xwSPPar->dataElement( 'ResourceFileName',
                               $spPrintSpoolInfoTmp->{ 'ResourceFileName' }
                             );
    }

    # Add "PageFrom" element
    if ( $spPrintSpoolInfoTmp->{ 'PageFrom' } ne "" ){
        $xwSPPar->dataElement( 'PageFrom',
                               $spPrintSpoolInfoTmp->{ 'PageFrom' }
                             );
    }

    # Add "PageTo" element
    if ( $spPrintSpoolInfoTmp->{ 'PageTo' } ne "" ){
        $xwSPPar->dataElement( 'PageTo',
                               $spPrintSpoolInfoTmp->{ 'PageTo' }
                             );
    }

	# End "PrintSpoolInfo" element
	$xwSPPar->endTag();
}

#-------------------------------------------------------------------------------
# Don't remove the following lines !!!
#-------------------------------------------------------------------------------
1;
__END__
